"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _path = require("path");

var _arrify = _interopRequireDefault(require("arrify"));

var _globby = _interopRequireDefault(require("globby"));

var _micromatch = require("micromatch");

var _options = require("./options");

var _linter = _interopRequireDefault(require("./linter"));

var _utils = require("./utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// @ts-ignore
// @ts-ignore

/** @typedef {import('webpack').Compiler} Compiler */

/** @typedef {import('webpack').Module} Module */

/** @typedef {import('./options').Options} Options */

/** @typedef {Partial<{timestamp:number} | number>} FileSystemInfoEntry */
const STYLELINT_PLUGIN = 'StylelintWebpackPlugin';
let counter = 0;

class StylelintWebpackPlugin {
  /**
   * @param {Options} options
   */
  constructor(options = {}) {
    this.key = STYLELINT_PLUGIN;
    this.options = (0, _options.getOptions)(options);
    this.run = this.run.bind(this);
    this.startTime = Date.now();
    this.prevTimestamps = new Map();
  }
  /**
   * @param {Compiler} compiler
   * @returns {void}
   */


  apply(compiler) {
    // Generate key for each compilation,
    // this differentiates one from the other when being cached.
    this.key = compiler.name || `${this.key}_${counter += 1}`; // If `lintDirtyModulesOnly` is disabled,
    // execute the linter on the build

    if (!this.options.lintDirtyModulesOnly) {
      compiler.hooks.run.tapPromise(this.key, this.run);
    }

    let isFirstRun = this.options.lintDirtyModulesOnly;
    compiler.hooks.watchRun.tapPromise(this.key, c => {
      if (isFirstRun) {
        isFirstRun = false;
        return Promise.resolve();
      }

      return this.run(c);
    });
  }
  /**
   * @param {Compiler} compiler
   */


  async run(compiler) {
    // Do not re-hook

    /* istanbul ignore if */
    if ( // @ts-ignore
    compiler.hooks.thisCompilation.taps.find(({
      name
    }) => name === this.key)) {
      return;
    }

    const context = this.getContext(compiler);
    const options = { ...this.options,
      exclude: (0, _utils.parseFiles)(this.options.exclude || ['**/node_modules/**', compiler.options.output.path], context),
      extensions: (0, _arrify.default)(this.options.extensions),
      files: (0, _utils.parseFiles)(this.options.files || '', context)
    };
    const wanted = (0, _utils.parseFoldersToGlobs)(options.files, options.extensions);
    const exclude = (0, _utils.parseFoldersToGlobs)(options.exclude);
    compiler.hooks.thisCompilation.tap(this.key, compilation => {
      /** @type {import('./linter').Linter} */
      let lint;
      /** @type {import('./linter').Reporter} */

      let report;
      /** @type number */

      let threads;

      try {
        ({
          lint,
          report,
          threads
        } = (0, _linter.default)(this.key, options, compilation));
      } catch (e) {
        compilation.errors.push(e);
        return;
      }

      compilation.hooks.finishModules.tap(this.key, () => {
        const files = this.getFiles(compiler, wanted, exclude);

        if (threads > 1) {
          for (const file of files) {
            lint((0, _utils.parseFiles)(file, context));
          }
        } else if (files.length > 0) {
          lint((0, _utils.parseFiles)(files, context));
        }
      }); // await and interpret results

      compilation.hooks.additionalAssets.tapPromise(this.key, processResults);

      async function processResults() {
        const {
          errors,
          warnings,
          generateReportAsset
        } = await report();

        if (warnings && !options.failOnWarning) {
          // @ts-ignore
          compilation.warnings.push(warnings);
        } else if (warnings && options.failOnWarning) {
          // @ts-ignore
          compilation.errors.push(warnings);
        }

        if (errors && options.failOnError) {
          // @ts-ignore
          compilation.errors.push(errors);
        } else if (errors && !options.failOnError) {
          // @ts-ignore
          compilation.warnings.push(errors);
        }

        if (generateReportAsset) {
          await generateReportAsset(compilation);
        }
      }
    });
  }
  /**
   *
   * @param {Compiler} compiler
   * @returns {string}
   */


  getContext(compiler) {
    if (!this.options.context) {
      return String(compiler.options.context);
    }

    if (!(0, _path.isAbsolute)(this.options.context)) {
      return (0, _path.join)(String(compiler.options.context), this.options.context);
    }

    return this.options.context;
  }
  /**
   * @param {Compiler} compiler
   * @param {string[]} wanted
   * @param {string[]} exclude
   * @returns {string[]}
   */
  // eslint-disable-next-line no-unused-vars


  getFiles(compiler, wanted, exclude) {
    // webpack 5
    if (compiler.modifiedFiles) {
      return Array.from(compiler.modifiedFiles).filter(file => (0, _micromatch.isMatch)(file, wanted, {
        dot: true
      }) && !(0, _micromatch.isMatch)(file, exclude, {
        dot: true
      }));
    } // webpack 4

    /* istanbul ignore next */


    if (compiler.fileTimestamps && compiler.fileTimestamps.size > 0) {
      return this.getChangedFiles(compiler.fileTimestamps).filter(file => (0, _micromatch.isMatch)(file, wanted, {
        dot: true
      }) && !(0, _micromatch.isMatch)(file, exclude, {
        dot: true
      }));
    }

    return _globby.default.sync(wanted, {
      dot: true,
      ignore: exclude
    });
  }
  /**
   * @param {Map<string, null | FileSystemInfoEntry | "ignore">} fileTimestamps
   * @returns {string[]}
   */

  /* istanbul ignore next */


  getChangedFiles(fileTimestamps) {
    /**
     * @param {null | FileSystemInfoEntry | "ignore"} fileSystemInfoEntry
     * @returns {Partial<number>}
     */
    const getTimestamps = fileSystemInfoEntry => {
      // @ts-ignore
      if (fileSystemInfoEntry && fileSystemInfoEntry.timestamp) {
        // @ts-ignore
        return fileSystemInfoEntry.timestamp;
      } // @ts-ignore


      return fileSystemInfoEntry;
    };
    /**
     * @param {string} filename
     * @param {null | FileSystemInfoEntry | "ignore"} fileSystemInfoEntry
     * @returns {boolean}
     */


    const hasFileChanged = (filename, fileSystemInfoEntry) => {
      const prevTimestamp = getTimestamps(this.prevTimestamps.get(filename));
      const timestamp = getTimestamps(fileSystemInfoEntry);
      return (prevTimestamp || this.startTime) < (timestamp || Infinity);
    };

    const changedFiles = [];

    for (const [filename, timestamp] of fileTimestamps.entries()) {
      if (hasFileChanged(filename, timestamp)) {
        changedFiles.push(filename);
      }
    }

    this.prevTimestamps = fileTimestamps;
    return changedFiles;
  }

}

var _default = StylelintWebpackPlugin;
exports.default = _default;