package migpv

import (
	"encoding/json"

	"github.com/fusor/openshift-migration-plugin/velero-plugins/migcommon"
	"github.com/heptio/velero/pkg/plugin/velero"
	"github.com/sirupsen/logrus"
	corev1API "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/apis/meta/v1/unstructured"
)

// RestorePlugin is a restore item action plugin for Velero
type RestorePlugin struct {
	Log logrus.FieldLogger
}

// AppliesTo returns a velero.ResourceSelector that applies to PVs
func (p *RestorePlugin) AppliesTo() (velero.ResourceSelector, error) {
	return velero.ResourceSelector{
		IncludedResources: []string{"persistentvolumes"},
	}, nil
}

// Execute action for the restore plugin for the pv resource
func (p *RestorePlugin) Execute(input *velero.RestoreItemActionExecuteInput) (*velero.RestoreItemActionExecuteOutput, error) {
	p.Log.Info("[pv-restore] Entering Persistent Volume restore plugin")

	pv := corev1API.PersistentVolume{}
	itemMarshal, _ := json.Marshal(input.Item)
	json.Unmarshal(itemMarshal, &pv)
	p.Log.Infof("[pv-restore] pv: %s", pv.Name)

	if pv.Annotations[migcommon.MigrateTypeAnnotation] == "copy" {
		p.Log.Infof("[pv-restore] Setting storage class, %s.", pv.Name)
		storageClassName := pv.Annotations[migcommon.MigrateStorageClassAnnotation]
		pv.Spec.StorageClassName = storageClassName
		if pv.Annotations[corev1API.BetaStorageClassAnnotation] != "" {
			pv.Annotations[corev1API.BetaStorageClassAnnotation] = storageClassName
		}
	}

	var out map[string]interface{}
	objrec, _ := json.Marshal(pv)
	json.Unmarshal(objrec, &out)

	return velero.NewRestoreItemActionExecuteOutput(&unstructured.Unstructured{Object: out}), nil
}
