/*
Copyright 2018 the Velero contributors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package output

import (
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/kubernetes/pkg/printers"

	v1 "github.com/vmware-tanzu/velero/pkg/apis/velero/v1"
)

var (
	resticRepoColumns = []metav1.TableColumnDefinition{
		// name needs Type and Format defined for the decorator to identify it:
		// https://github.com/kubernetes/kubernetes/blob/v1.15.3/pkg/printers/tableprinter.go#L204
		{Name: "Name", Type: "string", Format: "name"},
		{Name: "Status"},
		{Name: "Last Maintenance"},
	}
)

func printResticRepoList(list *v1.ResticRepositoryList, options printers.PrintOptions) ([]metav1.TableRow, error) {
	rows := make([]metav1.TableRow, 0, len(list.Items))

	for i := range list.Items {
		r, err := printResticRepo(&list.Items[i], options)
		if err != nil {
			return nil, err
		}
		rows = append(rows, r...)
	}
	return rows, nil
}

func printResticRepo(repo *v1.ResticRepository, options printers.PrintOptions) ([]metav1.TableRow, error) {
	row := metav1.TableRow{
		Object: runtime.RawExtension{Object: repo},
	}

	status := repo.Status.Phase
	if status == "" {
		status = v1.ResticRepositoryPhaseNew
	}

	lastMaintenance := repo.Status.LastMaintenanceTime.String()
	if repo.Status.LastMaintenanceTime.IsZero() {
		lastMaintenance = "<never>"
	}

	row.Cells = append(row.Cells,
		repo.Name,
		status,
		lastMaintenance,
	)

	return []metav1.TableRow{row}, nil
}
