/*
 * Application Insights JavaScript SDK - Common, 2.8.4
 * Copyright (c) Microsoft and contributors. All rights reserved.
 */


import { dumpObj, getExceptionName, getGlobal, getGlobalInst, isNullOrUndefined, objForEachKey, _throwInternal } from "@microsoft/applicationinsights-core-js";
import { StorageType } from "./Enums";
var _canUseLocalStorage = undefined;
var _canUseSessionStorage = undefined;
/**
 * Gets the localStorage object if available
 * @return {Storage} - Returns the storage object if available else returns null
 */
function _getLocalStorageObject() {
    if (utlCanUseLocalStorage()) {
        return _getVerifiedStorageObject(StorageType.LocalStorage);
    }
    return null;
}
/**
 * Tests storage object (localStorage or sessionStorage) to verify that it is usable
 * More details here: https://mathiasbynens.be/notes/localstorage-pattern
 * @param storageType Type of storage
 * @return {Storage} Returns storage object verified that it is usable
 */
function _getVerifiedStorageObject(storageType) {
    try {
        if (isNullOrUndefined(getGlobal())) {
            return null;
        }
        var uid = (new Date).toString();
        var storage = getGlobalInst(storageType === StorageType.LocalStorage ? "localStorage" : "sessionStorage");
        storage.setItem(uid, uid);
        var fail = storage.getItem(uid) !== uid;
        storage.removeItem(uid);
        if (!fail) {
            return storage;
        }
    }
    catch (exception) {
        // eslint-disable-next-line no-empty
    }
    return null;
}
/**
 * Gets the sessionStorage object if available
 * @return {Storage} - Returns the storage object if available else returns null
 */
function _getSessionStorageObject() {
    if (utlCanUseSessionStorage()) {
        return _getVerifiedStorageObject(StorageType.SessionStorage);
    }
    return null;
}
/**
 * Disables the global SDK usage of local or session storage if available
 */
export function utlDisableStorage() {
    _canUseLocalStorage = false;
    _canUseSessionStorage = false;
}
/**
 * Re-enables the global SDK usage of local or session storage if available
 */
export function utlEnableStorage() {
    _canUseLocalStorage = utlCanUseLocalStorage(true);
    _canUseSessionStorage = utlCanUseSessionStorage(true);
}
/**
 * Returns whether LocalStorage can be used, if the reset parameter is passed a true this will override
 * any previous disable calls.
 * @param reset - Should the usage be reset and determined only based on whether LocalStorage is available
 */
export function utlCanUseLocalStorage(reset) {
    if (reset || _canUseLocalStorage === undefined) {
        _canUseLocalStorage = !!_getVerifiedStorageObject(StorageType.LocalStorage);
    }
    return _canUseLocalStorage;
}
export function utlGetLocalStorage(logger, name) {
    var storage = _getLocalStorageObject();
    if (storage !== null) {
        try {
            return storage.getItem(name);
        }
        catch (e) {
            _canUseLocalStorage = false;
            _throwInternal(logger, 2 /* eLoggingSeverity.WARNING */, 1 /* _eInternalMessageId.BrowserCannotReadLocalStorage */, "Browser failed read of local storage. " + getExceptionName(e), { exception: dumpObj(e) });
        }
    }
    return null;
}
export function utlSetLocalStorage(logger, name, data) {
    var storage = _getLocalStorageObject();
    if (storage !== null) {
        try {
            storage.setItem(name, data);
            return true;
        }
        catch (e) {
            _canUseLocalStorage = false;
            _throwInternal(logger, 2 /* eLoggingSeverity.WARNING */, 3 /* _eInternalMessageId.BrowserCannotWriteLocalStorage */, "Browser failed write to local storage. " + getExceptionName(e), { exception: dumpObj(e) });
        }
    }
    return false;
}
export function utlRemoveStorage(logger, name) {
    var storage = _getLocalStorageObject();
    if (storage !== null) {
        try {
            storage.removeItem(name);
            return true;
        }
        catch (e) {
            _canUseLocalStorage = false;
            _throwInternal(logger, 2 /* eLoggingSeverity.WARNING */, 5 /* _eInternalMessageId.BrowserFailedRemovalFromLocalStorage */, "Browser failed removal of local storage item. " + getExceptionName(e), { exception: dumpObj(e) });
        }
    }
    return false;
}
export function utlCanUseSessionStorage(reset) {
    if (reset || _canUseSessionStorage === undefined) {
        _canUseSessionStorage = !!_getVerifiedStorageObject(StorageType.SessionStorage);
    }
    return _canUseSessionStorage;
}
export function utlGetSessionStorageKeys() {
    var keys = [];
    if (utlCanUseSessionStorage()) {
        objForEachKey(getGlobalInst("sessionStorage"), function (key) {
            keys.push(key);
        });
    }
    return keys;
}
export function utlGetSessionStorage(logger, name) {
    var storage = _getSessionStorageObject();
    if (storage !== null) {
        try {
            return storage.getItem(name);
        }
        catch (e) {
            _canUseSessionStorage = false;
            _throwInternal(logger, 2 /* eLoggingSeverity.WARNING */, 2 /* _eInternalMessageId.BrowserCannotReadSessionStorage */, "Browser failed read of session storage. " + getExceptionName(e), { exception: dumpObj(e) });
        }
    }
    return null;
}
export function utlSetSessionStorage(logger, name, data) {
    var storage = _getSessionStorageObject();
    if (storage !== null) {
        try {
            storage.setItem(name, data);
            return true;
        }
        catch (e) {
            _canUseSessionStorage = false;
            _throwInternal(logger, 2 /* eLoggingSeverity.WARNING */, 4 /* _eInternalMessageId.BrowserCannotWriteSessionStorage */, "Browser failed write to session storage. " + getExceptionName(e), { exception: dumpObj(e) });
        }
    }
    return false;
}
export function utlRemoveSessionStorage(logger, name) {
    var storage = _getSessionStorageObject();
    if (storage !== null) {
        try {
            storage.removeItem(name);
            return true;
        }
        catch (e) {
            _canUseSessionStorage = false;
            _throwInternal(logger, 2 /* eLoggingSeverity.WARNING */, 6 /* _eInternalMessageId.BrowserFailedRemovalFromSessionStorage */, "Browser failed removal of session storage item. " + getExceptionName(e), { exception: dumpObj(e) });
        }
    }
    return false;
}
