"use strict";
/**********************************************************************
 * Copyright (c) 2023 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 ***********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.VSCodeLauncher = void 0;
const os_1 = require("os");
const process_1 = require("process");
const fs = require("./fs-extra");
const child_process = require("child_process");
const node_extra_certificate_1 = require("./node-extra-certificate");
const code_workspace_1 = require("./code-workspace");
class VSCodeLauncher {
    async launch() {
        console.log("# Launching VS Code...");
        if (!process_1.env.VSCODE_NODEJS_RUNTIME_DIR) {
            throw new Error("Failed to launch VS Code. VSCODE_NODEJS_RUNTIME_DIR environment variable is not set.");
        }
        if (!process_1.env.PROJECTS_ROOT) {
            throw new Error("Failed to launch VS Code. PROJECTS_ROOT environment variable is not set.");
        }
        const node = `${process_1.env.VSCODE_NODEJS_RUNTIME_DIR}/node`;
        const host = process_1.env.CODE_HOST || "127.0.0.1";
        const params = [
            "out/server-main.js",
            "--host",
            host,
            "--port",
            "3100",
            "--without-connection-token",
        ];
        if (process_1.env.VSCODE_DEFAULT_WORKSPACE &&
            (await fs.pathExists(process_1.env.VSCODE_DEFAULT_WORKSPACE))) {
            params.push("--default-workspace", process_1.env.VSCODE_DEFAULT_WORKSPACE);
        }
        else if (await fs.pathExists((0, code_workspace_1.workspaceFilePath)())) {
            params.push("--default-workspace", (0, code_workspace_1.workspaceFilePath)());
        }
        else {
            if (!process_1.env.PROJECT_SOURCE) {
                throw new Error("Failed to launch VS Code. PROJECT_SOURCE environment variable is not set.");
            }
            params.push("--default-folder", process_1.env.PROJECT_SOURCE);
        }
        if (await fs.pathExists(node_extra_certificate_1.NODE_EXTRA_CERTIFICATE)) {
            process_1.env.NODE_EXTRA_CA_CERTS = node_extra_certificate_1.NODE_EXTRA_CERTIFICATE;
        }
        if (!process_1.env.SHELL && (0, os_1.userInfo)().shell === "/sbin/nologin") {
            // The SHELL env var is not set. In this case, Code will attempt to read the appropriate shell from /etc/passwd,
            // which can cause issues when cri-o injects /sbin/nologin when starting containers. Instead, we'll check if bash
            // is installed, and use that.
            const shell = this.detectShell();
            console.log(`  > SHELL environment variable is not set. Setting it to ${shell}`);
            process_1.env.SHELL = shell;
        }
        console.log(`  > Running: ${node}`);
        console.log(`  > Params: ${params}`);
        const run = child_process.spawn(node, params);
        run.stdout.on("data", (data) => {
            console.log(`${data}`);
        });
        run.stderr.on("data", (data) => {
            console.error(`${data}`);
        });
        run.on("close", (code) => {
            console.log(`VS Code process exited with code ${code}`);
        });
    }
    detectShell() {
        try {
            // Check if bash is installed
            child_process.execSync("command -v /bin/bash", {
                timeout: 500,
            });
            return "/bin/bash";
        }
        catch (error) {
            // bash not installed, fallback blindly to sh since it's at least better than /sbin/nologin
            return "/bin/sh";
        }
    }
}
exports.VSCodeLauncher = VSCodeLauncher;
//# sourceMappingURL=vscode-launcher.js.map