/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/

import assert from 'assert';
import { $ } from 'vs/base/browser/dom';
import { unthemedMenuStyles } from 'vs/base/browser/ui/menu/menu';
import { MenuBar } from 'vs/base/browser/ui/menu/menubar';
import { ensureNoDisposablesAreLeakedInTestSuite } from 'vs/base/test/common/utils';

function getButtonElementByAriaLabel(menubarElement: HTMLElement, ariaLabel: string): HTMLElement | null {
	let i;
	for (i = 0; i < menubarElement.childElementCount; i++) {

		if (menubarElement.children[i].getAttribute('aria-label') === ariaLabel) {
			return menubarElement.children[i] as HTMLElement;
		}
	}

	return null;
}

function getTitleDivFromButtonDiv(menuButtonElement: HTMLElement): HTMLElement | null {
	let i;
	for (i = 0; i < menuButtonElement.childElementCount; i++) {
		if (menuButtonElement.children[i].classList.contains('menubar-menu-title')) {
			return menuButtonElement.children[i] as HTMLElement;
		}
	}

	return null;
}

function getMnemonicFromTitleDiv(menuTitleDiv: HTMLElement): string | null {
	let i;
	for (i = 0; i < menuTitleDiv.childElementCount; i++) {
		if (menuTitleDiv.children[i].tagName.toLocaleLowerCase() === 'mnemonic') {
			return menuTitleDiv.children[i].textContent;
		}
	}

	return null;
}

function validateMenuBarItem(menubar: MenuBar, menubarContainer: HTMLElement, label: string, readableLabel: string, mnemonic: string) {
	menubar.push([
		{
			actions: [],
			label: label
		}
	]);

	const buttonElement = getButtonElementByAriaLabel(menubarContainer, readableLabel);
	assert(buttonElement !== null, `Button element not found for ${readableLabel} button.`);

	const titleDiv = getTitleDivFromButtonDiv(buttonElement);
	assert(titleDiv !== null, `Title div not found for ${readableLabel} button.`);

	const mnem = getMnemonicFromTitleDiv(titleDiv);
	assert.strictEqual(mnem, mnemonic, 'Mnemonic not correct');
}

suite('Menubar', () => {
	ensureNoDisposablesAreLeakedInTestSuite();
	const container = $('.container');

	const menubar = new MenuBar(container, {
		enableMnemonics: true,
		visibility: 'visible'
	}, unthemedMenuStyles);

	test('English File menu renders mnemonics', function () {
		validateMenuBarItem(menubar, container, '&File', 'File', 'F');
	});

	test('Russian File menu renders mnemonics', function () {
		validateMenuBarItem(menubar, container, '&Файл', 'Файл', 'Ф');
	});

	test('Chinese File menu renders mnemonics', function () {
		validateMenuBarItem(menubar, container, '文件(&F)', '文件', 'F');
	});
});
