"use strict";
/**********************************************************************
 * Copyright (c) 2023 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 ***********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenVSIXRegistry = void 0;
const process_1 = require("process");
const files_1 = require("./files");
const fs = require("./fs-extra");
const product_json_1 = require("./product-json");
class OpenVSIXRegistry {
    /*****************************************************************************************************************
     *
     * Configures OpenVSIX registry.
     *  - if OPENVSX_REGISTRY_URL environment variable exists, applies it as a extension registry.
     *  - if not, then uses CHE_PLUGIN_REGISTRY_URL.
     *
     * It's a replacement of
     * https://github.com/che-incubator/che-code/blob/f9389060f4dd7a435b13c75b63f6f12ec41fbd8e/build/scripts/entrypoint-volume.sh#L47-L66
     *
     *****************************************************************************************************************/
    async configure() {
        console.log("# Configuring OpenVSIX registry...");
        if (process_1.env.OPENVSX_REGISTRY_URL === undefined) {
            console.log("  > env.OPENVSX_REGISTRY_URL is not set, skip this step");
            return;
        }
        try {
            let openvsxURL;
            if (process_1.env.OPENVSX_REGISTRY_URL) {
                console.log(`  > env.OPENVSX_REGISTRY_URL set to ${process_1.env.OPENVSX_REGISTRY_URL}`);
                openvsxURL = `${this.withoutTrailingSlash(process_1.env.OPENVSX_REGISTRY_URL)}/vscode`;
            }
            else if (process_1.env.CHE_PLUGIN_REGISTRY_URL) {
                console.log(`  > env.OPENVSX_REGISTRY_URL is empty, use env.CHE_PLUGIN_REGISTRY_URL ${process_1.env.CHE_PLUGIN_REGISTRY_URL}`);
                let registryURL = this.withoutTrailingSlash(process_1.env.CHE_PLUGIN_REGISTRY_URL);
                if (registryURL.endsWith("/v3")) {
                    registryURL = registryURL.substring(0, registryURL.length - 3);
                }
                openvsxURL = `${registryURL}/openvsx/vscode`;
            }
            else {
                console.error("  > CHE_PLUGIN_REGISTRY_URL environment variable is not set");
                return;
            }
            console.log(`  > apply OpenVSIX URL [${openvsxURL}]`);
            const productJSON = await new product_json_1.ProductJSON().load();
            const serviceURL = productJSON.getExtensionsGalleryServiceURL();
            const itemURL = productJSON.getExtensionsGalleryItemURL();
            const newServiceURL = `${openvsxURL}/gallery`;
            const newItemURL = `${openvsxURL}/item`;
            productJSON.setExtensionsGalleryServiceURL(newServiceURL);
            productJSON.setExtensionsGalleryItemURL(newItemURL);
            await productJSON.save();
            await this.update(files_1.FILE_WORKBENCH_WEB_MAIN, serviceURL, newServiceURL, itemURL, newItemURL);
        }
        catch (err) {
            console.error(`${err.message} Failure to configure OpenVSIX registry.`);
        }
    }
    async update(file, currentServiceURL, newServiceURL, currentItemURL, newItemURL) {
        const content = await fs.readFile(file);
        const newContent = content.replace(`extensionsGallery:{serviceUrl:"${currentServiceURL}",itemUrl:"${currentItemURL}"}`, `extensionsGallery:{serviceUrl:"${newServiceURL}",itemUrl:"${newItemURL}"}`);
        await fs.writeFile(file, newContent);
    }
    withoutTrailingSlash(url) {
        while (url.endsWith("/")) {
            url = url.substring(0, url.length - 1);
        }
        return url;
    }
}
exports.OpenVSIXRegistry = OpenVSIXRegistry;
//# sourceMappingURL=openvsix-registry.js.map