"use strict";
/**********************************************************************
 * Copyright (c) 2023 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 ***********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebviewResources = void 0;
const process_1 = require("process");
const fs = require("./fs-extra");
const flattened_devfile_1 = require("./flattened-devfile");
const product_json_1 = require("./product-json");
const files_1 = require("./files");
class WebviewResources {
    /*****************************************************************************************************************
     *
     * Configures WebView lo load static resources from local host.
     * To turn the feature on, the 'WEBVIEW_LOCAL_RESOURCES' environment variable should be initialized with 'true'.
     *
     * It's a replacement of
     * https://github.com/che-incubator/che-code/blob/f9389060f4dd7a435b13c75b63f6f12ec41fbd8e/build/scripts/entrypoint-volume.sh#L69-L117
     *
     *****************************************************************************************************************/
    async configure() {
        console.log("# Configuring Webview Resources location...");
        if ("true" !== process_1.env.WEBVIEW_LOCAL_RESOURCES) {
            console.log(`  > env.WEBVIEW_LOCAL_RESOURCES is not set to 'true', skip this step`);
            return;
        }
        try {
            const cheCodeEndpoint = await new flattened_devfile_1.FlattenedDevfile().getCheCodeEndpoint();
            const newURL = `${cheCodeEndpoint}oss-dev/static/out/vs/workbench/contrib/webview/browser/pre/`;
            const productJSON = await new product_json_1.ProductJSON().load();
            const currentURL = productJSON.getWebviewContentExternalBaseUrlTemplate();
            // update in product.json
            productJSON.setWebviewContentExternalBaseUrlTemplate(newURL);
            await productJSON.save();
            // update files
            await this.update(files_1.FILE_WORKBENCH_WEB_MAIN, currentURL, newURL);
            await this.update(files_1.FILE_EXTENSION_HOST_PROCESS, currentURL, newURL);
            console.log(`  > webview resources endpoint ${newURL}`);
        }
        catch (err) {
            console.error(`${err.message} Webviews will not work if CDN disabled.`);
        }
    }
    async update(file, currentURL, newURL) {
        const content = await fs.readFile(file);
        const newContent = content.replace(currentURL, newURL);
        await fs.writeFile(file, newContent);
    }
}
exports.WebviewResources = WebviewResources;
//# sourceMappingURL=webview-resources.js.map