/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.elasticsearch.core.convert;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.time.DayOfWeek;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Month;
import java.time.MonthDay;
import java.time.OffsetDateTime;
import java.time.OffsetTime;
import java.time.Year;
import java.time.YearMonth;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.chrono.ChronoLocalDate;
import java.time.chrono.ChronoLocalDateTime;
import java.time.chrono.ChronoZonedDateTime;
import java.time.chrono.HijrahDate;
import java.time.chrono.JapaneseDate;
import java.time.chrono.MinguoDate;
import java.time.chrono.ThaiBuddhistDate;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.TemporalQuery;
import java.util.Date;
import java.util.Locale;
import java.util.concurrent.ConcurrentHashMap;
import org.springframework.data.elasticsearch.annotations.DateFormat;
import org.springframework.data.elasticsearch.core.convert.ConversionException;
import org.springframework.data.elasticsearch.core.convert.DateFormatter;
import org.springframework.util.Assert;

public final class ElasticsearchDateConverter {
    private static final ConcurrentHashMap<String, ElasticsearchDateConverter> converters = new ConcurrentHashMap();
    private final DateFormatter dateFormatter;

    public static ElasticsearchDateConverter of(DateFormat dateFormat) {
        Assert.notNull((Object)((Object)dateFormat), (String)"dateFormat must not be null");
        return ElasticsearchDateConverter.of(dateFormat.name());
    }

    public static ElasticsearchDateConverter of(String pattern) {
        Assert.hasText((String)pattern, (String)"pattern must not be empty");
        String[] subPatterns = pattern.split("\\|\\|");
        return converters.computeIfAbsent(subPatterns[0].trim(), p -> new ElasticsearchDateConverter(ElasticsearchDateConverter.forPattern(p)));
    }

    private ElasticsearchDateConverter(DateFormatter dateFormatter) {
        this.dateFormatter = dateFormatter;
    }

    public String format(TemporalAccessor accessor) {
        Assert.notNull((Object)accessor, (String)"accessor must not be null");
        if (accessor instanceof Instant) {
            Instant instant = (Instant)accessor;
            ZonedDateTime zonedDateTime = ZonedDateTime.ofInstant(instant, ZoneId.of("UTC"));
            return this.dateFormatter.format(zonedDateTime);
        }
        return this.dateFormatter.format(accessor);
    }

    public String format(Date date) {
        Assert.notNull((Object)date, (String)"accessor must not be null");
        return this.dateFormatter.format(Instant.ofEpochMilli(date.getTime()));
    }

    public <T extends TemporalAccessor> T parse(String input, Class<T> type) {
        return this.dateFormatter.parse(input, type);
    }

    public Date parse(String input) {
        return new Date(this.dateFormatter.parse(input, Instant.class).toEpochMilli());
    }

    private static DateFormatter forPattern(String pattern) {
        String resolvedPattern = pattern;
        if (DateFormat.epoch_millis.getPattern().equals(pattern)) {
            return new EpochMillisDateFormatter();
        }
        if (DateFormat.epoch_second.getPattern().equals(pattern)) {
            return new EpochSecondDateFormatter();
        }
        block3: for (DateFormat dateFormat : DateFormat.values()) {
            switch (dateFormat) {
                case weekyear: 
                case weekyear_week: 
                case weekyear_week_day: {
                    continue block3;
                }
                default: {
                    if (!dateFormat.name().equals(pattern)) continue block3;
                    resolvedPattern = dateFormat.getPattern();
                    break block3;
                }
            }
        }
        DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern(resolvedPattern);
        return new PatternDateFormatter(dateTimeFormatter);
    }

    private static <T extends TemporalAccessor> TemporalQuery<T> getTemporalQuery(Class<T> type) {
        return temporal -> {
            if (type == HijrahDate.class) {
                return HijrahDate.from(temporal);
            }
            if (type == JapaneseDate.class) {
                return JapaneseDate.from(temporal);
            }
            if (type == ZonedDateTime.class) {
                return ZonedDateTime.from(temporal);
            }
            if (type == LocalDateTime.class) {
                return LocalDateTime.from(temporal);
            }
            if (type == ThaiBuddhistDate.class) {
                return ThaiBuddhistDate.from(temporal);
            }
            if (type == LocalTime.class) {
                return LocalTime.from(temporal);
            }
            if (type == ZoneOffset.class) {
                return ZoneOffset.from(temporal);
            }
            if (type == OffsetTime.class) {
                return OffsetTime.from(temporal);
            }
            if (type == ChronoLocalDate.class) {
                return ChronoLocalDate.from(temporal);
            }
            if (type == Month.class) {
                return Month.from(temporal);
            }
            if (type == ChronoLocalDateTime.class) {
                return ChronoLocalDateTime.from(temporal);
            }
            if (type == MonthDay.class) {
                return MonthDay.from(temporal);
            }
            if (type == Instant.class) {
                return Instant.from(temporal);
            }
            if (type == OffsetDateTime.class) {
                return OffsetDateTime.from(temporal);
            }
            if (type == ChronoZonedDateTime.class) {
                return ChronoZonedDateTime.from(temporal);
            }
            if (type == MinguoDate.class) {
                return MinguoDate.from(temporal);
            }
            if (type == Year.class) {
                return Year.from(temporal);
            }
            if (type == DayOfWeek.class) {
                return DayOfWeek.from(temporal);
            }
            if (type == LocalDate.class) {
                return LocalDate.from(temporal);
            }
            if (type == YearMonth.class) {
                return YearMonth.from(temporal);
            }
            try {
                Method method = type.getMethod("from", TemporalAccessor.class);
                Object o = method.invoke(null, temporal);
                return (TemporalAccessor)type.cast(o);
            }
            catch (NoSuchMethodException e) {
                throw new ConversionException("no 'from' factory method found in class " + type.getName());
            }
            catch (IllegalAccessException | InvocationTargetException e) {
                throw new ConversionException("could not create object of class " + type.getName(), e);
            }
        };
    }

    static class EpochMillisDateFormatter
    implements DateFormatter {
        EpochMillisDateFormatter() {
        }

        @Override
        public String format(TemporalAccessor accessor) {
            Assert.notNull((Object)accessor, (String)"accessor must not be null");
            return Long.toString(Instant.from(accessor).toEpochMilli());
        }

        @Override
        public <T extends TemporalAccessor> T parse(String input, Class<T> type) {
            Assert.notNull((Object)input, (String)"input must not be null");
            Assert.notNull(type, (String)"type must not be null");
            Instant instant = Instant.ofEpochMilli(Long.parseLong(input));
            TemporalQuery<T> query = ElasticsearchDateConverter.getTemporalQuery(type);
            return (T)((TemporalAccessor)query.queryFrom(instant));
        }
    }

    static class EpochSecondDateFormatter
    implements DateFormatter {
        EpochSecondDateFormatter() {
        }

        @Override
        public String format(TemporalAccessor accessor) {
            Assert.notNull((Object)accessor, (String)"accessor must not be null");
            long epochMilli = Instant.from(accessor).toEpochMilli();
            long fraction = epochMilli % 1000L;
            if (fraction == 0L) {
                return Long.toString(epochMilli / 1000L);
            }
            Double d = (double)epochMilli / 1000.0;
            return String.format(Locale.ROOT, "%.03f", d);
        }

        @Override
        public <T extends TemporalAccessor> T parse(String input, Class<T> type) {
            Assert.notNull((Object)input, (String)"input must not be null");
            Assert.notNull(type, (String)"type must not be null");
            Double epochMilli = Double.parseDouble(input) * 1000.0;
            Instant instant = Instant.ofEpochMilli(epochMilli.longValue());
            TemporalQuery<T> query = ElasticsearchDateConverter.getTemporalQuery(type);
            return (T)((TemporalAccessor)query.queryFrom(instant));
        }
    }

    static class PatternDateFormatter
    implements DateFormatter {
        private final DateTimeFormatter dateTimeFormatter;

        PatternDateFormatter(DateTimeFormatter dateTimeFormatter) {
            this.dateTimeFormatter = dateTimeFormatter;
        }

        @Override
        public String format(TemporalAccessor accessor) {
            Assert.notNull((Object)accessor, (String)"accessor must not be null");
            try {
                return this.dateTimeFormatter.format(accessor);
            }
            catch (Exception e) {
                if (accessor instanceof Instant) {
                    return this.dateTimeFormatter.format(ZonedDateTime.ofInstant((Instant)accessor, ZoneId.of("UTC")));
                }
                throw e;
            }
        }

        @Override
        public <T extends TemporalAccessor> T parse(String input, Class<T> type) {
            Assert.notNull((Object)input, (String)"input must not be null");
            Assert.notNull(type, (String)"type must not be null");
            try {
                return (T)((TemporalAccessor)this.dateTimeFormatter.parse((CharSequence)input, ElasticsearchDateConverter.getTemporalQuery(type)));
            }
            catch (Exception e) {
                if (type.equals(Instant.class)) {
                    try {
                        ZonedDateTime zonedDateTime = this.dateTimeFormatter.parse((CharSequence)input, ElasticsearchDateConverter.getTemporalQuery(ZonedDateTime.class));
                        return (T)zonedDateTime.toInstant();
                    }
                    catch (Exception exception) {
                        LocalDateTime localDateTime = this.dateTimeFormatter.parse((CharSequence)input, ElasticsearchDateConverter.getTemporalQuery(LocalDateTime.class));
                        return (T)localDateTime.toInstant(ZoneOffset.UTC);
                    }
                }
                throw e;
            }
        }
    }
}

