/*
 * Decompiled with CFR 0.152.
 */
package org.jobrunr.server;

import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.atomic.AtomicInteger;
import org.jobrunr.jobs.Job;
import org.jobrunr.jobs.context.JobRunrDashboardLogger;
import org.jobrunr.jobs.filters.JobPerformingFilters;
import org.jobrunr.jobs.mappers.MDCMapper;
import org.jobrunr.jobs.states.IllegalJobStateChangeException;
import org.jobrunr.jobs.states.StateName;
import org.jobrunr.scheduling.exceptions.JobNotFoundException;
import org.jobrunr.server.BackgroundJobServer;
import org.jobrunr.server.runner.BackgroundJobRunner;
import org.jobrunr.storage.ConcurrentJobModificationException;
import org.jobrunr.utils.annotations.VisibleFor;
import org.jobrunr.utils.exceptions.Exceptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

public class BackgroundJobPerformer
implements Runnable {
    private static final Logger LOGGER = LoggerFactory.getLogger(BackgroundJobPerformer.class);
    private static final AtomicInteger concurrentModificationExceptionCounter = new AtomicInteger();
    private final BackgroundJobServer backgroundJobServer;
    private final JobPerformingFilters jobPerformingFilters;
    private final Job job;

    public BackgroundJobPerformer(BackgroundJobServer backgroundJobServer, Job job) {
        this.backgroundJobServer = backgroundJobServer;
        this.jobPerformingFilters = new JobPerformingFilters(job, backgroundJobServer.getJobFilters());
        this.job = job;
    }

    @Override
    public void run() {
        try {
            this.backgroundJobServer.getJobZooKeeper().notifyThreadOccupied();
            MDCMapper.loadMDCContextFromJob(this.job);
            this.performJob();
        }
        catch (Exception e) {
            if (this.isJobDeletedWhileProcessing(e)) {
                return;
            }
            if (this.isJobServerStopped(e)) {
                this.updateJobStateToFailedAndRunJobFilters("Job processing was stopped as background job server has stopped", e);
                Thread.currentThread().interrupt();
            } else if (this.isJobNotFoundException(e)) {
                this.updateJobStateToFailedAndRunJobFilters("Job method not found", e);
            } else {
                this.updateJobStateToFailedAndRunJobFilters("An exception occurred during the performance of the job", e);
            }
        }
        finally {
            this.backgroundJobServer.getJobZooKeeper().notifyThreadIdle();
            MDC.clear();
        }
    }

    protected void performJob() throws Exception {
        boolean canProcess = this.updateJobStateToProcessingRunJobFiltersAndReturnIfProcessingCanStart();
        if (canProcess) {
            this.runActualJob();
            this.updateJobStateToSucceededAndRunJobFilters();
        }
    }

    private boolean updateJobStateToProcessingRunJobFiltersAndReturnIfProcessingCanStart() {
        try {
            this.job.startProcessingOn(this.backgroundJobServer);
            this.saveAndRunStateRelatedJobFilters(this.job);
            LOGGER.debug("Job(id={}, jobName='{}') processing started", (Object)this.job.getId(), (Object)this.job.getJobName());
            return this.job.hasState(StateName.PROCESSING);
        }
        catch (ConcurrentJobModificationException e) {
            LOGGER.trace("Could not start processing job {} - it is already in a newer state (collision {})", (Object)this.job.getId(), (Object)concurrentModificationExceptionCounter.incrementAndGet());
            return false;
        }
    }

    private void runActualJob() throws Exception {
        try {
            JobRunrDashboardLogger.setJob(this.job);
            this.backgroundJobServer.getJobZooKeeper().startProcessing(this.job, Thread.currentThread());
            LOGGER.trace("Job(id={}, jobName='{}') is running", (Object)this.job.getId(), (Object)this.job.getJobName());
            this.jobPerformingFilters.runOnJobProcessingFilters();
            BackgroundJobRunner backgroundJobRunner = this.backgroundJobServer.getBackgroundJobRunner(this.job);
            backgroundJobRunner.run(this.job);
            this.jobPerformingFilters.runOnJobProcessingSucceededFilters();
        }
        catch (Exception e) {
            this.jobPerformingFilters.runOnJobProcessingFailedFilters(e);
            throw e;
        }
        finally {
            this.backgroundJobServer.getJobZooKeeper().stopProcessing(this.job);
            JobRunrDashboardLogger.clearJob();
        }
    }

    private void updateJobStateToSucceededAndRunJobFilters() {
        try {
            LOGGER.debug("Job(id={}, jobName='{}') processing succeeded", (Object)this.job.getId(), (Object)this.job.getJobName());
            this.job.succeeded();
            this.saveAndRunStateRelatedJobFilters(this.job);
        }
        catch (IllegalJobStateChangeException ex) {
            if (ex.getFrom() == StateName.DELETED) {
                LOGGER.info("Job finished successfully but it was already deleted - ignoring illegal state change from {} to {}", (Object)ex.getFrom(), (Object)ex.getTo());
            }
            throw ex;
        }
        catch (Exception badException) {
            LOGGER.error("ERROR - could not update job(id={}, jobName='{}') to SUCCEEDED state", new Object[]{this.job.getId(), this.job.getJobName(), badException});
        }
    }

    private void updateJobStateToFailedAndRunJobFilters(String message, Exception e) {
        try {
            Exception actualException = BackgroundJobPerformer.unwrapException(e);
            this.job.failed(message, actualException);
            this.saveAndRunStateRelatedJobFilters(this.job);
            if (this.job.getState() == StateName.FAILED) {
                LOGGER.error("Job(id={}, jobName='{}') processing failed: {}", new Object[]{this.job.getId(), this.job.getJobName(), message, actualException});
            } else {
                LOGGER.warn("Job(id={}, jobName='{}') processing failed: {}", new Object[]{this.job.getId(), this.job.getJobName(), message, actualException});
            }
        }
        catch (IllegalJobStateChangeException ex) {
            if (ex.getFrom() == StateName.DELETED) {
                LOGGER.info("Job processing failed but it was already deleted - ignoring illegal state change from {} to {}", (Object)ex.getFrom(), (Object)ex.getTo());
            }
            throw ex;
        }
        catch (Exception badException) {
            LOGGER.error("ERROR - could not update job(id={}, jobName='{}') to FAILED state", new Object[]{this.job.getId(), this.job.getJobName(), badException});
        }
    }

    protected void saveAndRunStateRelatedJobFilters(Job job) {
        this.jobPerformingFilters.runOnStateAppliedFilters();
        StateName beforeStateElection = job.getState();
        this.jobPerformingFilters.runOnStateElectionFilter();
        StateName afterStateElection = job.getState();
        this.backgroundJobServer.getStorageProvider().save(job);
        if (beforeStateElection != afterStateElection) {
            this.jobPerformingFilters.runOnStateAppliedFilters();
        }
        if (afterStateElection == StateName.FAILED) {
            this.jobPerformingFilters.runOnJobFailedAfterRetriesFilters();
        }
    }

    private boolean isJobDeletedWhileProcessing(Exception e) {
        return Exceptions.hasCause(e, InterruptedException.class) && this.job.hasState(StateName.DELETED);
    }

    private boolean isJobServerStopped(Exception e) {
        return Exceptions.hasCause(e, InterruptedException.class) && !this.job.hasState(StateName.DELETED);
    }

    private boolean isJobNotFoundException(Exception e) {
        return e instanceof JobNotFoundException;
    }

    @VisibleFor(value="testing")
    static Exception unwrapException(Exception e) {
        if (e instanceof InvocationTargetException && e.getCause() instanceof Exception) {
            return (Exception)e.getCause();
        }
        return e;
    }
}

