/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.action.admin.indices.diskusage;

import java.io.Closeable;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.codecs.DocValuesProducer;
import org.apache.lucene.codecs.FieldsProducer;
import org.apache.lucene.codecs.NormsProducer;
import org.apache.lucene.codecs.PointsReader;
import org.apache.lucene.codecs.StoredFieldsReader;
import org.apache.lucene.codecs.TermVectorsReader;
import org.apache.lucene.codecs.lucene50.Lucene50PostingsFormat;
import org.apache.lucene.codecs.lucene84.Lucene84PostingsFormat;
import org.apache.lucene.index.BinaryDocValues;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.DocValuesType;
import org.apache.lucene.index.FieldInfo;
import org.apache.lucene.index.FieldInfos;
import org.apache.lucene.index.Fields;
import org.apache.lucene.index.IndexCommit;
import org.apache.lucene.index.IndexOptions;
import org.apache.lucene.index.LeafReaderContext;
import org.apache.lucene.index.NumericDocValues;
import org.apache.lucene.index.PointValues;
import org.apache.lucene.index.PostingsEnum;
import org.apache.lucene.index.SegmentReader;
import org.apache.lucene.index.SortedDocValues;
import org.apache.lucene.index.SortedSetDocValues;
import org.apache.lucene.index.StoredFieldVisitor;
import org.apache.lucene.index.TermState;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FilterDirectory;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.FutureArrays;
import org.elasticsearch.action.admin.indices.diskusage.IndexDiskUsageStats;
import org.elasticsearch.common.CheckedSupplier;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.lucene.FilterIndexCommit;
import org.elasticsearch.common.lucene.Lucene;
import org.elasticsearch.core.CheckedConsumer;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.store.LuceneFilesExtensions;

final class IndexDiskUsageAnalyzer {
    private final Logger logger;
    private final IndexCommit commit;
    private final TrackingReadBytesDirectory directory;
    private final CancellationChecker cancellationChecker;

    private IndexDiskUsageAnalyzer(ShardId shardId, IndexCommit commit, Runnable checkForCancellation) {
        this.logger = Loggers.getLogger(IndexDiskUsageAnalyzer.class, shardId, new String[0]);
        this.directory = new TrackingReadBytesDirectory(commit.getDirectory());
        this.commit = new FilterIndexCommit(commit){

            @Override
            public Directory getDirectory() {
                return IndexDiskUsageAnalyzer.this.directory;
            }
        };
        this.cancellationChecker = new CancellationChecker(checkForCancellation);
    }

    static IndexDiskUsageStats analyze(ShardId shardId, IndexCommit commit, Runnable checkForCancellation) throws IOException {
        IndexDiskUsageAnalyzer analyzer = new IndexDiskUsageAnalyzer(shardId, commit, checkForCancellation);
        IndexDiskUsageStats stats = new IndexDiskUsageStats(IndexDiskUsageAnalyzer.getIndexSize(commit));
        analyzer.doAnalyze(stats);
        return stats;
    }

    void doAnalyze(IndexDiskUsageStats stats) throws IOException {
        ExecutionTime executionTime = new ExecutionTime();
        try (DirectoryReader directoryReader = DirectoryReader.open((IndexCommit)this.commit);){
            this.directory.resetBytesRead();
            for (LeafReaderContext leaf : directoryReader.leaves()) {
                this.cancellationChecker.checkForCancellation();
                SegmentReader reader = Lucene.segmentReader(leaf.reader());
                long startTimeInNanos = System.nanoTime();
                this.analyzeInvertedIndex(reader, stats);
                executionTime.invertedIndexTimeInNanos += System.nanoTime() - startTimeInNanos;
                startTimeInNanos = System.nanoTime();
                this.analyzeStoredFields(reader, stats);
                executionTime.storedFieldsTimeInNanos += System.nanoTime() - startTimeInNanos;
                startTimeInNanos = System.nanoTime();
                this.analyzeDocValues(reader, stats);
                executionTime.docValuesTimeInNanos += System.nanoTime() - startTimeInNanos;
                startTimeInNanos = System.nanoTime();
                this.analyzePoints(reader, stats);
                executionTime.pointsTimeInNanos += System.nanoTime() - startTimeInNanos;
                startTimeInNanos = System.nanoTime();
                this.analyzeNorms(reader, stats);
                executionTime.normsTimeInNanos += System.nanoTime() - startTimeInNanos;
                startTimeInNanos = System.nanoTime();
                this.analyzeTermVectors(reader, stats);
                executionTime.termVectorsTimeInNanos += System.nanoTime() - startTimeInNanos;
            }
        }
        this.logger.debug("analyzing the disk usage took {} stats: {}", (Object)executionTime, (Object)stats);
    }

    void analyzeStoredFields(SegmentReader reader, IndexDiskUsageStats stats) throws IOException {
        StoredFieldsReader storedFieldsReader = reader.getFieldsReader().getMergeInstance();
        this.directory.resetBytesRead();
        TrackingSizeStoredFieldVisitor visitor = new TrackingSizeStoredFieldVisitor();
        int docID = 0;
        int skipMask = 511;
        while (docID < reader.maxDoc()) {
            this.cancellationChecker.logEvent();
            storedFieldsReader.visitDocument(docID, (StoredFieldVisitor)visitor);
            if ((docID & 0x1FF) == 511 && docID < reader.maxDoc() - 512) {
                docID = Math.toIntExact(Math.min((long)docID + 5120L, (long)reader.maxDoc() - 512L));
                continue;
            }
            ++docID;
        }
        if (!visitor.fields.isEmpty()) {
            long totalBytes = visitor.fields.values().stream().mapToLong(v -> v).sum();
            double ratio = (double)this.directory.getBytesRead() / (double)totalBytes;
            FieldInfos fieldInfos = reader.getFieldInfos();
            for (Map.Entry field : visitor.fields.entrySet()) {
                String fieldName = fieldInfos.fieldInfo((int)((Integer)field.getKey()).intValue()).name;
                long fieldSize = (long)Math.ceil((double)((Long)field.getValue()).longValue() * ratio);
                stats.addStoredField(fieldName, fieldSize);
            }
        }
    }

    private <DV extends DocIdSetIterator> DV iterateDocValues(int maxDocs, CheckedSupplier<DV, IOException> dvReader, CheckedConsumer<DV, IOException> valueAccessor) throws IOException {
        DocIdSetIterator dv = (DocIdSetIterator)dvReader.get();
        int docID = dv.nextDoc();
        if (docID != Integer.MAX_VALUE) {
            valueAccessor.accept((Object)dv);
            long left = docID;
            long right = 2L * ((long)maxDocs - 1L) - left;
            while (left < (long)maxDocs - 1L && left <= right) {
                this.cancellationChecker.logEvent();
                int mid = Math.toIntExact(left + right >>> 1);
                docID = dv.advance(mid);
                if (docID != Integer.MAX_VALUE) {
                    valueAccessor.accept((Object)dv);
                    left = docID + 1;
                    continue;
                }
                right = mid - 1;
                dv = (DocIdSetIterator)dvReader.get();
            }
            assert (dv.advance(Math.toIntExact(left + 1L)) == Integer.MAX_VALUE);
        }
        return (DV)dv;
    }

    void analyzeDocValues(SegmentReader reader, IndexDiskUsageStats stats) throws IOException {
        if (reader.getDocValuesReader() == null) {
            return;
        }
        DocValuesProducer docValuesReader = reader.getDocValuesReader().getMergeInstance();
        int maxDocs = reader.maxDoc();
        for (FieldInfo field : reader.getFieldInfos()) {
            DocValuesType dvType = field.getDocValuesType();
            if (dvType == DocValuesType.NONE) continue;
            this.cancellationChecker.checkForCancellation();
            this.directory.resetBytesRead();
            switch (dvType) {
                case NUMERIC: {
                    this.iterateDocValues(maxDocs, () -> docValuesReader.getNumeric(field), NumericDocValues::longValue);
                    break;
                }
                case SORTED_NUMERIC: {
                    this.iterateDocValues(maxDocs, () -> docValuesReader.getSortedNumeric(field), dv -> {
                        for (int i = 0; i < dv.docValueCount(); ++i) {
                            this.cancellationChecker.logEvent();
                            dv.nextValue();
                        }
                    });
                    break;
                }
                case BINARY: {
                    this.iterateDocValues(maxDocs, () -> docValuesReader.getBinary(field), BinaryDocValues::binaryValue);
                    break;
                }
                case SORTED: {
                    SortedDocValues sorted = this.iterateDocValues(maxDocs, () -> docValuesReader.getSorted(field), SortedDocValues::ordValue);
                    sorted.lookupOrd(0);
                    sorted.lookupOrd(sorted.getValueCount() - 1);
                    break;
                }
                case SORTED_SET: {
                    SortedSetDocValues sortedSet = this.iterateDocValues(maxDocs, () -> docValuesReader.getSortedSet(field), dv -> {
                        while (dv.nextOrd() != -1L) {
                            this.cancellationChecker.logEvent();
                        }
                    });
                    sortedSet.lookupOrd(0L);
                    sortedSet.lookupOrd(sortedSet.getValueCount() - 1L);
                    break;
                }
                default: {
                    assert (false) : "Unknown docValues type [" + dvType + "]";
                    throw new IllegalStateException("Unknown docValues type [" + dvType + "]");
                }
            }
            stats.addDocValues(field.name, this.directory.getBytesRead());
        }
    }

    private void readProximity(Terms terms, PostingsEnum postings) throws IOException {
        if (terms.hasPositions()) {
            for (int pos = 0; pos < postings.freq(); ++pos) {
                postings.nextPosition();
                postings.startOffset();
                postings.endOffset();
                postings.getPayload();
            }
        }
    }

    private BlockTermState getBlockTermState(TermsEnum termsEnum, BytesRef term) throws IOException {
        if (term != null && termsEnum.seekExact(term)) {
            TermState termState = termsEnum.termState();
            if (termState instanceof Lucene84PostingsFormat.IntBlockTermState) {
                Lucene84PostingsFormat.IntBlockTermState blockTermState = (Lucene84PostingsFormat.IntBlockTermState)termState;
                return new BlockTermState(blockTermState.docStartFP, blockTermState.posStartFP, blockTermState.payStartFP);
            }
            if (termState instanceof Lucene50PostingsFormat.IntBlockTermState) {
                Lucene50PostingsFormat.IntBlockTermState blockTermState = (Lucene50PostingsFormat.IntBlockTermState)termState;
                return new BlockTermState(blockTermState.docStartFP, blockTermState.posStartFP, blockTermState.payStartFP);
            }
        }
        return null;
    }

    void analyzeInvertedIndex(SegmentReader reader, IndexDiskUsageStats stats) throws IOException {
        FieldsProducer postingsReader = reader.getPostingsReader();
        if (postingsReader == null) {
            return;
        }
        postingsReader = postingsReader.getMergeInstance();
        PostingsEnum postings = null;
        for (FieldInfo field : reader.getFieldInfos()) {
            if (field.getIndexOptions() == IndexOptions.NONE) continue;
            this.cancellationChecker.checkForCancellation();
            this.directory.resetBytesRead();
            Terms terms = postingsReader.terms(field.name);
            if (terms == null) continue;
            TermsEnum termsEnum = terms.iterator();
            BlockTermState minState = this.getBlockTermState(termsEnum, terms.getMin());
            if (minState != null) {
                BlockTermState maxState = Objects.requireNonNull(this.getBlockTermState(termsEnum, terms.getMax()), "can't retrieve the block term state of the max term");
                long skippedBytes = maxState.distance(minState);
                stats.addInvertedIndex(field.name, skippedBytes);
                termsEnum.seekExact(terms.getMax());
                postings = termsEnum.postings(postings, 120);
                if (postings.advance(termsEnum.docFreq() - 1) != Integer.MAX_VALUE) {
                    postings.freq();
                    this.readProximity(terms, postings);
                }
                long bytesRead = this.directory.getBytesRead();
                int visitedTerms = 0;
                long totalTerms = terms.size();
                termsEnum = terms.iterator();
                while (termsEnum.next() != null) {
                    this.cancellationChecker.logEvent();
                    if (totalTerms <= 1000L || ++visitedTerms % 50 != 0 || this.directory.getBytesRead() <= bytesRead) continue;
                    break;
                }
            } else {
                while (termsEnum.next() != null) {
                    this.cancellationChecker.logEvent();
                    termsEnum.docFreq();
                    termsEnum.totalTermFreq();
                    postings = termsEnum.postings(postings, 120);
                    while (postings.nextDoc() != Integer.MAX_VALUE) {
                        this.cancellationChecker.logEvent();
                        postings.freq();
                        this.readProximity(terms, postings);
                    }
                }
            }
            stats.addInvertedIndex(field.name, this.directory.getBytesRead());
        }
    }

    void analyzePoints(SegmentReader reader, IndexDiskUsageStats stats) throws IOException {
        PointsReader pointsReader = reader.getPointsReader();
        if (pointsReader == null) {
            return;
        }
        pointsReader = pointsReader.getMergeInstance();
        for (FieldInfo field : reader.getFieldInfos()) {
            this.cancellationChecker.checkForCancellation();
            this.directory.resetBytesRead();
            if (field.getPointDimensionCount() <= 0) continue;
            PointValues values = pointsReader.getValues(field.name);
            values.intersect((PointValues.IntersectVisitor)new PointsVisitor(values.getMinPackedValue(), values.getNumIndexDimensions(), values.getBytesPerDimension()));
            values.intersect((PointValues.IntersectVisitor)new PointsVisitor(values.getMaxPackedValue(), values.getNumIndexDimensions(), values.getBytesPerDimension()));
            stats.addPoints(field.name, this.directory.getBytesRead());
        }
    }

    void analyzeNorms(SegmentReader reader, IndexDiskUsageStats stats) throws IOException {
        if (reader.getNormsReader() == null) {
            return;
        }
        NormsProducer normsReader = reader.getNormsReader().getMergeInstance();
        for (FieldInfo field : reader.getFieldInfos()) {
            if (!field.hasNorms()) continue;
            this.cancellationChecker.checkForCancellation();
            this.directory.resetBytesRead();
            this.iterateDocValues(reader.maxDoc(), () -> normsReader.getNorms(field), norms -> {
                this.cancellationChecker.logEvent();
                norms.longValue();
            });
            stats.addNorms(field.name, this.directory.getBytesRead());
        }
    }

    void analyzeTermVectors(SegmentReader reader, IndexDiskUsageStats stats) throws IOException {
        TermVectorsReader termVectorsReader = reader.getTermVectorsReader();
        if (termVectorsReader == null) {
            return;
        }
        termVectorsReader = termVectorsReader.getMergeInstance();
        this.directory.resetBytesRead();
        TermVectorsVisitor visitor = new TermVectorsVisitor();
        for (int docID = 0; docID < reader.numDocs(); ++docID) {
            this.cancellationChecker.logEvent();
            Fields vectors = termVectorsReader.get(docID);
            if (vectors == null) continue;
            for (String field : vectors) {
                this.cancellationChecker.logEvent();
                visitor.visitField(vectors, field);
            }
        }
        if (!visitor.fields.isEmpty()) {
            long totalBytes = visitor.fields.values().stream().mapToLong(v -> v).sum();
            double ratio = (double)this.directory.getBytesRead() / (double)totalBytes;
            for (Map.Entry<String, Long> field : visitor.fields.entrySet()) {
                long fieldBytes = (long)Math.ceil((double)field.getValue().longValue() * ratio);
                stats.addTermVectors(field.getKey(), fieldBytes);
            }
        }
    }

    static long getIndexSize(IndexCommit commit) throws IOException {
        long total = 0L;
        for (String file : commit.getFileNames()) {
            total += commit.getDirectory().fileLength(file);
        }
        return total;
    }

    private static class CancellationChecker {
        static final long THRESHOLD = 10000L;
        private long iterations;
        private final Runnable checkForCancellationRunner;

        CancellationChecker(Runnable checkForCancellationRunner) {
            this.checkForCancellationRunner = checkForCancellationRunner;
        }

        void logEvent() {
            if (this.iterations == 10000L) {
                this.checkForCancellation();
            } else {
                ++this.iterations;
            }
        }

        void checkForCancellation() {
            this.iterations = 0L;
            this.checkForCancellationRunner.run();
        }
    }

    private static class TrackingReadBytesDirectory
    extends FilterDirectory {
        private final Map<String, BytesReadTracker> trackers = new HashMap<String, BytesReadTracker>();

        TrackingReadBytesDirectory(Directory in) {
            super(in);
        }

        long getBytesRead() {
            return this.trackers.values().stream().mapToLong(BytesReadTracker::getBytesRead).sum();
        }

        void resetBytesRead() {
            this.trackers.values().forEach(BytesReadTracker::resetBytesRead);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public IndexInput openInput(String name, IOContext context) throws IOException {
            IndexInput in = super.openInput(name, context);
            try {
                BytesReadTracker tracker = this.trackers.computeIfAbsent(name, k -> {
                    if (LuceneFilesExtensions.fromFile(name) == LuceneFilesExtensions.CFS) {
                        return new CompoundFileBytesReaderTracker();
                    }
                    return new BytesReadTracker();
                });
                TrackingReadBytesIndexInput wrapped = new TrackingReadBytesIndexInput(in, 0L, tracker);
                in = null;
                TrackingReadBytesIndexInput trackingReadBytesIndexInput = wrapped;
                return trackingReadBytesIndexInput;
            }
            finally {
                IOUtils.close((Closeable)in);
            }
        }
    }

    private static class ExecutionTime {
        long invertedIndexTimeInNanos;
        long storedFieldsTimeInNanos;
        long docValuesTimeInNanos;
        long pointsTimeInNanos;
        long normsTimeInNanos;
        long termVectorsTimeInNanos;

        private ExecutionTime() {
        }

        long totalInNanos() {
            return this.invertedIndexTimeInNanos + this.storedFieldsTimeInNanos + this.docValuesTimeInNanos + this.pointsTimeInNanos + this.normsTimeInNanos + this.termVectorsTimeInNanos;
        }

        public String toString() {
            return "total: " + this.totalInNanos() / 1000000L + "ms, inverted index: " + this.invertedIndexTimeInNanos / 1000000L + "ms, stored fields: " + this.storedFieldsTimeInNanos / 1000000L + "ms, doc values: " + this.docValuesTimeInNanos / 1000000L + "ms, points: " + this.pointsTimeInNanos / 1000000L + "ms, norms: " + this.normsTimeInNanos / 1000000L + "ms, term vectors: " + this.termVectorsTimeInNanos / 1000000L + "ms";
        }
    }

    private static class TrackingSizeStoredFieldVisitor
    extends StoredFieldVisitor {
        private final Map<Integer, Long> fields = new HashMap<Integer, Long>();

        private TrackingSizeStoredFieldVisitor() {
        }

        private void trackField(FieldInfo fieldInfo, int fieldLength) {
            int totalBytes = fieldLength + 8;
            this.fields.compute(fieldInfo.number, (k, v) -> v == null ? (long)totalBytes : v + (long)totalBytes);
        }

        public void binaryField(FieldInfo fieldInfo, byte[] value) throws IOException {
            this.trackField(fieldInfo, 4 + value.length);
        }

        public void stringField(FieldInfo fieldInfo, byte[] value) throws IOException {
            this.trackField(fieldInfo, 4 + value.length);
        }

        public void intField(FieldInfo fieldInfo, int value) throws IOException {
            this.trackField(fieldInfo, 4);
        }

        public void longField(FieldInfo fieldInfo, long value) throws IOException {
            this.trackField(fieldInfo, 8);
        }

        public void floatField(FieldInfo fieldInfo, float value) throws IOException {
            this.trackField(fieldInfo, 4);
        }

        public void doubleField(FieldInfo fieldInfo, double value) throws IOException {
            this.trackField(fieldInfo, 8);
        }

        public StoredFieldVisitor.Status needsField(FieldInfo fieldInfo) throws IOException {
            return StoredFieldVisitor.Status.YES;
        }
    }

    private static class BlockTermState {
        final long docStartFP;
        final long posStartFP;
        final long payloadFP;

        BlockTermState(long docStartFP, long posStartFP, long payloadFP) {
            this.docStartFP = docStartFP;
            this.posStartFP = posStartFP;
            this.payloadFP = payloadFP;
        }

        long distance(BlockTermState other) {
            return this.docStartFP - other.docStartFP + this.posStartFP - other.posStartFP + this.payloadFP - other.payloadFP;
        }
    }

    private class PointsVisitor
    implements PointValues.IntersectVisitor {
        private final byte[] point;
        private final int numDims;
        private final int bytesPerDim;

        PointsVisitor(byte[] point, int numDims, int bytesPerDim) {
            this.point = point;
            this.numDims = numDims;
            this.bytesPerDim = bytesPerDim;
        }

        public void visit(int docID) throws IOException {
            IndexDiskUsageAnalyzer.this.cancellationChecker.logEvent();
        }

        public void visit(int docID, byte[] packedValue) throws IOException {
            IndexDiskUsageAnalyzer.this.cancellationChecker.logEvent();
        }

        public PointValues.Relation compare(byte[] minPackedValue, byte[] maxPackedValue) {
            for (int dim = 0; dim < this.numDims; ++dim) {
                int offset = dim * this.bytesPerDim;
                if (FutureArrays.compareUnsigned((byte[])minPackedValue, (int)offset, (int)(offset + this.bytesPerDim), (byte[])this.point, (int)offset, (int)(offset + this.bytesPerDim)) <= 0 && FutureArrays.compareUnsigned((byte[])maxPackedValue, (int)offset, (int)(offset + this.bytesPerDim), (byte[])this.point, (int)offset, (int)(offset + this.bytesPerDim)) >= 0) continue;
                return PointValues.Relation.CELL_OUTSIDE_QUERY;
            }
            return PointValues.Relation.CELL_CROSSES_QUERY;
        }
    }

    private class TermVectorsVisitor {
        final Map<String, Long> fields = new HashMap<String, Long>();
        private PostingsEnum docsAndPositions;

        private TermVectorsVisitor() {
        }

        void visitField(Fields vectors, String fieldName) throws IOException {
            BytesRef bytesRef;
            Terms terms = vectors.terms(fieldName);
            if (terms == null) {
                return;
            }
            boolean hasPositions = terms.hasPositions();
            boolean hasOffsets = terms.hasOffsets();
            boolean hasPayloads = terms.hasPayloads();
            assert (!hasPayloads || hasPositions);
            long fieldLength = 1L;
            TermsEnum termsEnum = terms.iterator();
            while ((bytesRef = termsEnum.next()) != null) {
                IndexDiskUsageAnalyzer.this.cancellationChecker.logEvent();
                fieldLength += (long)(4 + bytesRef.length);
                int freq = (int)termsEnum.totalTermFreq();
                fieldLength += 4L;
                if (!hasPositions && !hasOffsets) continue;
                this.docsAndPositions = termsEnum.postings(this.docsAndPositions, 120);
                assert (this.docsAndPositions != null);
                while (this.docsAndPositions.nextDoc() != Integer.MAX_VALUE) {
                    IndexDiskUsageAnalyzer.this.cancellationChecker.logEvent();
                    assert (this.docsAndPositions.freq() == freq);
                    for (int posUpTo = 0; posUpTo < freq; ++posUpTo) {
                        int pos = this.docsAndPositions.nextPosition();
                        fieldLength += 4L;
                        this.docsAndPositions.startOffset();
                        fieldLength += 4L;
                        this.docsAndPositions.endOffset();
                        fieldLength += 4L;
                        BytesRef payload = this.docsAndPositions.getPayload();
                        if (payload != null) {
                            fieldLength += (long)(4 + payload.length);
                        }
                        assert (!hasPositions || pos >= 0);
                    }
                }
            }
            long finalLength = fieldLength;
            this.fields.compute(fieldName, (k, v) -> v == null ? finalLength : v + finalLength);
        }
    }

    private static class CompoundFileBytesReaderTracker
    extends BytesReadTracker {
        private final Map<Long, BytesReadTracker> slicedTrackers = new HashMap<Long, BytesReadTracker>();

        private CompoundFileBytesReaderTracker() {
        }

        @Override
        BytesReadTracker createSliceTracker(long offset) {
            return this.slicedTrackers.computeIfAbsent(offset, k -> new BytesReadTracker());
        }

        @Override
        void trackPositions(long position, int length) {
        }

        @Override
        void resetBytesRead() {
            this.slicedTrackers.values().forEach(BytesReadTracker::resetBytesRead);
        }

        @Override
        long getBytesRead() {
            return this.slicedTrackers.values().stream().mapToLong(BytesReadTracker::getBytesRead).sum();
        }
    }

    private static class BytesReadTracker {
        private long minPosition = Long.MAX_VALUE;
        private long maxPosition = Long.MIN_VALUE;

        private BytesReadTracker() {
        }

        BytesReadTracker createSliceTracker(long offset) {
            return this;
        }

        void trackPositions(long position, int length) {
            this.minPosition = Math.min(this.minPosition, position);
            this.maxPosition = Math.max(this.maxPosition, position + (long)length - 1L);
        }

        void resetBytesRead() {
            this.minPosition = Long.MAX_VALUE;
            this.maxPosition = Long.MIN_VALUE;
        }

        long getBytesRead() {
            if (this.minPosition <= this.maxPosition) {
                return this.maxPosition - this.minPosition + 1L;
            }
            return 0L;
        }
    }

    private static class TrackingReadBytesIndexInput
    extends IndexInput {
        final IndexInput in;
        final BytesReadTracker bytesReadTracker;
        final long fileOffset;

        TrackingReadBytesIndexInput(IndexInput in, long fileOffset, BytesReadTracker bytesReadTracker) {
            super(in.toString());
            this.in = in;
            this.fileOffset = fileOffset;
            this.bytesReadTracker = bytesReadTracker;
        }

        public void close() throws IOException {
            this.in.close();
        }

        public long getFilePointer() {
            return this.in.getFilePointer();
        }

        public void seek(long pos) throws IOException {
            this.in.seek(pos);
        }

        public long length() {
            return this.in.length();
        }

        public IndexInput slice(String sliceDescription, long offset, long length) throws IOException {
            IndexInput slice = this.in.slice(sliceDescription, offset, length);
            return new TrackingReadBytesIndexInput(slice, this.fileOffset + offset, this.bytesReadTracker.createSliceTracker(offset));
        }

        public IndexInput clone() {
            return new TrackingReadBytesIndexInput(this.in.clone(), this.fileOffset, this.bytesReadTracker);
        }

        public byte readByte() throws IOException {
            this.bytesReadTracker.trackPositions(this.fileOffset + this.getFilePointer(), 1);
            return this.in.readByte();
        }

        public void readBytes(byte[] b, int offset, int len) throws IOException {
            this.bytesReadTracker.trackPositions(this.fileOffset + this.getFilePointer(), len);
            this.in.readBytes(b, offset, len);
        }
    }
}

