/*
 * Decompiled with CFR 0.152.
 */
package org.jobrunr.server;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.jobrunr.JobRunrException;
import org.jobrunr.SevereJobRunrException;
import org.jobrunr.jobs.Job;
import org.jobrunr.jobs.RecurringJob;
import org.jobrunr.jobs.filters.JobFilterUtils;
import org.jobrunr.jobs.states.StateName;
import org.jobrunr.server.BackgroundJobServer;
import org.jobrunr.server.concurrent.ConcurrentJobModificationResolver;
import org.jobrunr.server.concurrent.UnresolvableConcurrentJobModificationException;
import org.jobrunr.server.dashboard.DashboardNotificationManager;
import org.jobrunr.server.strategy.WorkDistributionStrategy;
import org.jobrunr.storage.BackgroundJobServerStatus;
import org.jobrunr.storage.ConcurrentJobModificationException;
import org.jobrunr.storage.PageRequest;
import org.jobrunr.storage.StorageProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JobZooKeeper
implements Runnable {
    static final Logger LOGGER = LoggerFactory.getLogger(JobZooKeeper.class);
    private final BackgroundJobServer backgroundJobServer;
    private final StorageProvider storageProvider;
    private final List<RecurringJob> recurringJobs;
    private final DashboardNotificationManager dashboardNotificationManager;
    private final JobFilterUtils jobFilterUtils;
    private final WorkDistributionStrategy workDistributionStrategy;
    private final ConcurrentJobModificationResolver concurrentJobModificationResolver;
    private final Map<Job, Thread> currentlyProcessedJobs;
    private final AtomicInteger exceptionCount;
    private final ReentrantLock reentrantLock;
    private final AtomicInteger occupiedWorkers;
    private final Duration durationPollIntervalTimeBox;
    private Instant runStartTime;

    public JobZooKeeper(BackgroundJobServer backgroundJobServer) {
        this.backgroundJobServer = backgroundJobServer;
        this.storageProvider = backgroundJobServer.getStorageProvider();
        this.recurringJobs = new ArrayList<RecurringJob>();
        this.workDistributionStrategy = backgroundJobServer.getWorkDistributionStrategy();
        this.dashboardNotificationManager = backgroundJobServer.getDashboardNotificationManager();
        this.jobFilterUtils = new JobFilterUtils(backgroundJobServer.getJobFilters());
        this.concurrentJobModificationResolver = this.createConcurrentJobModificationResolver();
        this.currentlyProcessedJobs = new ConcurrentHashMap<Job, Thread>();
        this.durationPollIntervalTimeBox = Duration.ofSeconds((long)((double)this.backgroundJobServerStatus().getPollIntervalInSeconds() - (double)this.backgroundJobServerStatus().getPollIntervalInSeconds() * 0.05));
        this.reentrantLock = new ReentrantLock();
        this.exceptionCount = new AtomicInteger();
        this.occupiedWorkers = new AtomicInteger();
    }

    @Override
    public void run() {
        try {
            this.runStartTime = Instant.now();
            if (this.backgroundJobServer.isUnAnnounced()) {
                return;
            }
            this.updateJobsThatAreBeingProcessed();
            this.runMasterTasksIfCurrentServerIsMaster();
            this.onboardNewWorkIfPossible();
        }
        catch (Exception e) {
            this.dashboardNotificationManager.handle(e);
            if (this.exceptionCount.getAndIncrement() < 5) {
                LOGGER.warn("JobRunr encountered a problematic exception. Please create a bug report (if possible, provide the code to reproduce this and the stacktrace) - Processing will continue.", (Throwable)e);
            }
            LOGGER.error("FATAL - JobRunr encountered too many processing exceptions. Shutting down.", (Throwable)JobRunrException.shouldNotHappenException(e));
            this.backgroundJobServer.stop();
        }
    }

    void updateJobsThatAreBeingProcessed() {
        LOGGER.debug("Updating currently processed jobs... ");
        this.processJobList(new ArrayList<Job>(this.currentlyProcessedJobs.keySet()), this::updateCurrentlyProcessingJob);
    }

    void runMasterTasksIfCurrentServerIsMaster() {
        if (this.backgroundJobServer.isMaster()) {
            this.checkForRecurringJobs();
            this.checkForScheduledJobs();
            this.checkForOrphanedJobs();
            this.checkForSucceededJobsThanCanGoToDeletedState();
            this.checkForJobsThatCanBeDeleted();
        }
    }

    boolean canOnboardNewWork() {
        return this.backgroundJobServerStatus().isRunning() && this.workDistributionStrategy.canOnboardNewWork();
    }

    void checkForRecurringJobs() {
        LOGGER.debug("Looking for recurring jobs... ");
        List<RecurringJob> recurringJobs = this.getRecurringJobs();
        this.processRecurringJobs(recurringJobs);
    }

    void checkForScheduledJobs() {
        LOGGER.debug("Looking for scheduled jobs... ");
        Supplier<List<Job>> scheduledJobsSupplier = () -> this.storageProvider.getScheduledJobs(Instant.now().plusSeconds(this.backgroundJobServerStatus().getPollIntervalInSeconds()), PageRequest.ascOnUpdatedAt(1000));
        this.processJobList(scheduledJobsSupplier, Job::enqueue);
    }

    void checkForOrphanedJobs() {
        LOGGER.debug("Looking for orphan jobs... ");
        Instant updatedBefore = this.runStartTime.minus(Duration.ofSeconds(this.backgroundJobServer.getServerStatus().getPollIntervalInSeconds()).multipliedBy(4L));
        Supplier<List<Job>> orphanedJobsSupplier = () -> this.storageProvider.getJobs(StateName.PROCESSING, updatedBefore, PageRequest.ascOnUpdatedAt(1000));
        this.processJobList(orphanedJobsSupplier, (Job job) -> job.failed("Orphaned job", new IllegalThreadStateException("Job was too long in PROCESSING state without being updated.")));
    }

    void checkForSucceededJobsThanCanGoToDeletedState() {
        LOGGER.debug("Looking for succeeded jobs that can go to the deleted state... ");
        AtomicInteger succeededJobsCounter = new AtomicInteger();
        Instant updatedBefore = Instant.now().minus(this.backgroundJobServer.getServerStatus().getDeleteSucceededJobsAfter());
        Supplier<List<Job>> succeededJobsSupplier = () -> this.storageProvider.getJobs(StateName.SUCCEEDED, updatedBefore, PageRequest.ascOnUpdatedAt(1000));
        this.processJobList(succeededJobsSupplier, (Job job) -> {
            succeededJobsCounter.incrementAndGet();
            job.delete("JobRunr maintenance - deleting succeeded job");
        });
        if (succeededJobsCounter.get() > 0) {
            this.storageProvider.publishTotalAmountOfSucceededJobs(succeededJobsCounter.get());
        }
    }

    void checkForJobsThatCanBeDeleted() {
        LOGGER.debug("Looking for deleted jobs that can be deleted permanently... ");
        this.storageProvider.deleteJobsPermanently(StateName.DELETED, Instant.now().minus(this.backgroundJobServer.getServerStatus().getPermanentlyDeleteDeletedJobsAfter()));
    }

    void onboardNewWorkIfPossible() {
        if (this.pollIntervalInSecondsTimeBoxIsAboutToPass()) {
            return;
        }
        if (this.canOnboardNewWork()) {
            this.checkForEnqueuedJobs();
        }
    }

    void checkForEnqueuedJobs() {
        try {
            if (this.reentrantLock.tryLock()) {
                LOGGER.debug("Looking for enqueued jobs... ");
                PageRequest workPageRequest = this.workDistributionStrategy.getWorkPageRequest();
                if (workPageRequest.getLimit() > 0) {
                    List<Job> enqueuedJobs = this.storageProvider.getJobs(StateName.ENQUEUED, workPageRequest);
                    enqueuedJobs.forEach(this.backgroundJobServer::processJob);
                }
            }
        }
        finally {
            if (this.reentrantLock.isHeldByCurrentThread()) {
                this.reentrantLock.unlock();
            }
        }
    }

    void processRecurringJobs(List<RecurringJob> recurringJobs) {
        LOGGER.debug("Found {} recurring jobs", (Object)recurringJobs.size());
        List<Job> jobsToSchedule = recurringJobs.stream().filter(this::mustSchedule).map(RecurringJob::toScheduledJob).collect(Collectors.toList());
        if (!jobsToSchedule.isEmpty()) {
            this.storageProvider.save(jobsToSchedule);
        }
    }

    boolean mustSchedule(RecurringJob recurringJob) {
        return recurringJob.getNextRun().isBefore(Instant.now().plus(this.durationPollIntervalTimeBox).plusSeconds(1L)) && !this.storageProvider.recurringJobExists(recurringJob.getId(), StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING);
    }

    void processJobList(Supplier<List<Job>> jobListSupplier, Consumer<Job> jobConsumer) {
        List<Job> jobs = this.getJobsToProcess(jobListSupplier);
        while (!jobs.isEmpty()) {
            this.processJobList(jobs, jobConsumer);
            jobs = this.getJobsToProcess(jobListSupplier);
        }
    }

    void processJobList(List<Job> jobs, Consumer<Job> jobConsumer) {
        if (!jobs.isEmpty()) {
            try {
                jobs.forEach(jobConsumer);
                this.jobFilterUtils.runOnStateElectionFilter(jobs);
                this.storageProvider.save(jobs);
                this.jobFilterUtils.runOnStateAppliedFilters(jobs);
            }
            catch (ConcurrentJobModificationException concurrentJobModificationException) {
                try {
                    this.concurrentJobModificationResolver.resolve(concurrentJobModificationException);
                }
                catch (UnresolvableConcurrentJobModificationException unresolvableConcurrentJobModificationException) {
                    throw new SevereJobRunrException("Could not resolve ConcurrentJobModificationException", unresolvableConcurrentJobModificationException);
                }
            }
        }
    }

    BackgroundJobServerStatus backgroundJobServerStatus() {
        return this.backgroundJobServer.getServerStatus();
    }

    public void startProcessing(Job job, Thread thread) {
        this.currentlyProcessedJobs.put(job, thread);
    }

    public void stopProcessing(Job job) {
        this.currentlyProcessedJobs.remove(job);
    }

    public Thread getThreadProcessingJob(Job job) {
        return this.currentlyProcessedJobs.get(job);
    }

    public int getOccupiedWorkerCount() {
        return this.occupiedWorkers.get();
    }

    public void notifyThreadOccupied() {
        this.occupiedWorkers.incrementAndGet();
    }

    public void notifyThreadIdle() {
        this.occupiedWorkers.decrementAndGet();
        if (this.workDistributionStrategy.canOnboardNewWork()) {
            this.checkForEnqueuedJobs();
        }
    }

    private List<Job> getJobsToProcess(Supplier<List<Job>> jobListSupplier) {
        if (this.pollIntervalInSecondsTimeBoxIsAboutToPass()) {
            return Collections.emptyList();
        }
        return jobListSupplier.get();
    }

    private void updateCurrentlyProcessingJob(Job job) {
        try {
            job.updateProcessing();
        }
        catch (ClassCastException classCastException) {
            // empty catch block
        }
    }

    private boolean pollIntervalInSecondsTimeBoxIsAboutToPass() {
        boolean runTimeBoxIsPassed;
        Duration durationRunTime = Duration.between(this.runStartTime, Instant.now());
        boolean bl = runTimeBoxIsPassed = durationRunTime.compareTo(this.durationPollIntervalTimeBox) >= 0;
        if (runTimeBoxIsPassed) {
            LOGGER.debug("JobRunr is passing the poll interval in seconds timebox because of too many tasks.");
        }
        return runTimeBoxIsPassed;
    }

    private List<RecurringJob> getRecurringJobs() {
        if ((long)this.recurringJobs.size() != this.storageProvider.countRecurringJobs()) {
            this.recurringJobs.clear();
            this.recurringJobs.addAll(this.storageProvider.getRecurringJobs());
        }
        return this.recurringJobs;
    }

    ConcurrentJobModificationResolver createConcurrentJobModificationResolver() {
        return this.backgroundJobServer.getConfiguration().concurrentJobModificationPolicy.toConcurrentJobModificationResolver(this.storageProvider, this);
    }
}

