"use strict";
/********************************************************************************
 * Copyright (c) 2020 TypeFox and others
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0
 ********************************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const isCI = require("is-ci");
const util_1 = require("./util");
function addLicense(packagePath, manifest) {
    return __awaiter(this, void 0, void 0, function* () {
        console.log('Extension ' + manifest.publisher + '.' + manifest.name + ' has no license. All Open VSX '
            + 'Registry Content Offerings must be licensed. You may choose to publish this extension under '
            + 'the MIT License (https://opensource.org/licenses/MIT). Please note you are responsible to '
            + 'ensure that you have the necessary rights to permit this extension to be made available under '
            + 'the MIT license and for compliance with that license.');
        let answer;
        do {
            console.log();
            answer = yield util_1.getUserChoice('Would you like to publish your extension '
                + manifest.publisher + '.' + manifest.name
                + ' under the MIT license?', ['yes', 'help', 'no'], 'no');
            switch (answer) {
                case 'yes':
                    yield useMITLicense(manifest, packagePath);
                    break;
                case 'help':
                    console.log('If you select "yes" your extension will be published under the MIT License. '
                        + 'You must enter the Copyright Year and Copyright Holder information. This information '
                        + 'along with the text of the MIT License will be written to a LICENSE file and '
                        + 'packaged with the uploaded extension.\n');
                    console.log(MIT_LICENSE_TEXT);
                    break;
                case 'no':
                    throw new Error('This extension cannot be accepted because it has no license.');
            }
        } while (answer === 'help');
    });
}
function isLicenseOk(packagePath, manifest) {
    return __awaiter(this, void 0, void 0, function* () {
        // manifest is optional in order to use isLicenseOk function
        // without need to read and parse manifest by the another step
        manifest = manifest !== null && manifest !== void 0 ? manifest : yield util_1.readManifest(packagePath);
        util_1.validateManifest(manifest);
        if (manifest.license) {
            // The extension has a license identifier or a pointer to an alternative LICENSE file
            return true;
        }
        if (yield hasLicenseFile(packagePath)) {
            // The extension has a LICENSE file that can be packaged
            return true;
        }
        return false;
    });
}
exports.isLicenseOk = isLicenseOk;
function checkLicense(packagePath) {
    return __awaiter(this, void 0, void 0, function* () {
        const manifest = yield util_1.readManifest(packagePath);
        if (!(yield isLicenseOk(packagePath, manifest)) && !isCI) {
            yield addLicense(packagePath, manifest);
        }
    });
}
exports.checkLicense = checkLicense;
function useMITLicense(manifest, packagePath) {
    return __awaiter(this, void 0, void 0, function* () {
        console.log('Please enter a value for Copyright Year and Copyright Holder.\n'
            + 'Example: "Copyright 2020 John Doe"\n');
        const copyright = yield util_1.getUserInput('Copyright ');
        if (!copyright) {
            throw new Error('A copyright declaration is necessary for the MIT license.');
        }
        manifest.license = 'MIT';
        yield util_1.writeManifest(manifest, packagePath);
        const license = MIT_LICENSE_TEXT.replace('<YEAR> <COPYRIGHT HOLDER>', copyright);
        yield util_1.writeFile('LICENSE', license, packagePath);
        console.log('LICENSE file has been written. Please commit it to the source repository.');
    });
}
const LICENSE_FILE_NAMES = ['license.md', 'license', 'license.txt', 'licence.md', 'licence', 'licence.txt'];
function hasLicenseFile(packagePath) {
    return __awaiter(this, void 0, void 0, function* () {
        const fileNames = yield util_1.promisify(fs.readdir)(packagePath !== null && packagePath !== void 0 ? packagePath : '.');
        for (const fileName of fileNames) {
            for (const licFileName of LICENSE_FILE_NAMES) {
                if (fileName.toLowerCase() === licFileName) {
                    return true;
                }
            }
        }
        return false;
    });
}
const MIT_LICENSE_TEXT = `Copyright <YEAR> <COPYRIGHT HOLDER>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.`;
//# sourceMappingURL=check-license.js.map