"use strict";
/********************************************************************************
 * Copyright (c) 2019 TypeFox and others
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0
 ********************************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const semver = require("semver");
const registry_1 = require("./registry");
const util_1 = require("./util");
/**
 * Downloads an extension or its metadata.
 */
function getExtension(options) {
    return __awaiter(this, void 0, void 0, function* () {
        util_1.addEnvOptions(options);
        if (!options.target) {
            options.target = 'universal';
        }
        const registry = new registry_1.Registry(options);
        const match = util_1.matchExtensionId(options.extensionId);
        if (!match) {
            throw new Error('The extension identifier must have the form `namespace.extension`.');
        }
        const extension = yield registry.getMetadata(match[1], match[2], options.target);
        if (extension.error) {
            throw new Error(extension.error);
        }
        const matchingVersion = yield findMatchingVersion(registry, extension, options.version);
        if (matchingVersion.error) {
            throw new Error(matchingVersion.error);
        }
        if (options.metadata) {
            yield printMetadata(registry, matchingVersion, options.output);
        }
        else {
            yield download(registry, matchingVersion, options.output);
        }
    });
}
exports.getExtension = getExtension;
function findMatchingVersion(registry, extension, constraint) {
    if (!constraint || semver.satisfies(extension.version, constraint)) {
        return Promise.resolve(extension);
    }
    for (const version of Object.keys(extension.allVersions)) {
        if (!isAlias(extension, version) && semver.satisfies(version, constraint)) {
            try {
                return registry.getJson(new URL(extension.allVersions[version]));
            }
            catch (err) {
                return Promise.reject(err);
            }
        }
    }
    return Promise.reject(`Extension ${extension.namespace}.${extension.name} has no published version matching '${constraint}'`);
}
function isAlias(extension, version) {
    return extension.versionAlias.includes(version);
}
function printMetadata(registry, extension, output) {
    return __awaiter(this, void 0, void 0, function* () {
        const metadata = JSON.stringify(extension, null, 4);
        if (!output) {
            console.log(metadata);
            return;
        }
        let filePath;
        const stats = yield util_1.optionalStat(output);
        if (stats && stats.isDirectory() || !stats && output.endsWith(path.sep)) {
            const fileName = `${extension.namespace}.${extension.name}-${extension.version}.json`;
            filePath = path.resolve(process.cwd(), output, fileName);
        }
        else {
            filePath = path.resolve(process.cwd(), output);
        }
        yield util_1.makeDirs(path.dirname(filePath));
        yield util_1.promisify(fs.writeFile)(filePath, metadata);
    });
}
function download(registry, extension, output) {
    return __awaiter(this, void 0, void 0, function* () {
        const downloadUrl = extension.files.download;
        if (!downloadUrl) {
            throw new Error(`Extension ${extension.namespace}.${extension.name} does not provide a download URL.`);
        }
        const fileNameIndex = downloadUrl.lastIndexOf('/');
        const fileName = decodeURIComponent(downloadUrl.substring(fileNameIndex + 1));
        let filePath;
        if (output) {
            const stats = yield util_1.optionalStat(output);
            if (stats && stats.isDirectory() || !stats && output.endsWith(path.sep)) {
                filePath = path.resolve(process.cwd(), output, fileName);
            }
            else {
                filePath = path.resolve(process.cwd(), output);
            }
        }
        else {
            filePath = path.resolve(process.cwd(), fileName);
        }
        yield util_1.makeDirs(path.dirname(filePath));
        const target = extension.targetPlatform !== 'universal' ? '@' + extension.targetPlatform : '';
        console.log(`Downloading ${extension.namespace}.${extension.name}-${extension.version}${target} to ${filePath}`);
        yield registry.download(filePath, new URL(downloadUrl));
    });
}
//# sourceMappingURL=get.js.map