"use strict";
/********************************************************************************
 * Copyright (c) 2019 TypeFox and others
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const querystring = require("querystring");
const followRedirects = require("follow-redirects");
const util_1 = require("./util");
exports.DEFAULT_URL = 'https://open-vsx.org';
exports.DEFAULT_NAMESPACE_SIZE = 1024;
exports.DEFAULT_PUBLISH_SIZE = 512 * 1024 * 1024;
class Registry {
    constructor(options = {}) {
        if (options.registryUrl && options.registryUrl.endsWith('/'))
            this.url = options.registryUrl.substring(0, options.registryUrl.length - 1);
        else if (options.registryUrl)
            this.url = options.registryUrl;
        else
            this.url = exports.DEFAULT_URL;
        this.maxNamespaceSize = options.maxNamespaceSize || exports.DEFAULT_NAMESPACE_SIZE;
        this.maxPublishSize = options.maxPublishSize || exports.DEFAULT_PUBLISH_SIZE;
        this.username = options.username;
        this.password = options.password;
    }
    get requiresLicense() {
        const url = new URL(this.url);
        return url.hostname === 'open-vsx.org' || url.hostname.endsWith('.open-vsx.org');
    }
    createNamespace(name, pat) {
        try {
            const query = { token: pat };
            const url = this.getUrl('api/-/namespace/create', query);
            const namespace = { name };
            return this.post(JSON.stringify(namespace), url, {
                'Content-Type': 'application/json'
            }, this.maxNamespaceSize);
        }
        catch (err) {
            return Promise.reject(err);
        }
    }
    publish(file, pat) {
        try {
            const query = { token: pat };
            const url = this.getUrl('api/-/publish', query);
            return this.postFile(file, url, {
                'Content-Type': 'application/octet-stream'
            }, this.maxPublishSize);
        }
        catch (err) {
            return Promise.reject(err);
        }
    }
    getMetadata(namespace, extension, target) {
        try {
            let path = `api/${encodeURIComponent(namespace)}/${encodeURIComponent(extension)}`;
            if (target) {
                path += `/${encodeURIComponent(target)}`;
            }
            return this.getJson(this.getUrl(path));
        }
        catch (err) {
            return Promise.reject(err);
        }
    }
    download(file, url) {
        return new Promise((resolve, reject) => {
            const stream = fs.createWriteStream(file);
            const requestOptions = this.getRequestOptions();
            const request = this.getProtocol(url)
                .request(url, requestOptions, response => {
                response.on('end', () => {
                    if (response.statusCode !== undefined && (response.statusCode < 200 || response.statusCode > 299)) {
                        reject(util_1.statusError(response));
                    }
                    else {
                        resolve();
                    }
                });
                response.pipe(stream);
            });
            stream.on('error', err => {
                request.abort();
                reject(err);
            });
            request.on('error', err => {
                stream.close();
                reject(err);
            });
            request.end();
        });
    }
    getJson(url) {
        return new Promise((resolve, reject) => {
            const requestOptions = this.getRequestOptions();
            const request = this.getProtocol(url)
                .request(url, requestOptions, this.getJsonResponse(resolve, reject));
            request.on('error', reject);
            request.end();
        });
    }
    post(content, url, headers, maxBodyLength) {
        return new Promise((resolve, reject) => {
            const requestOptions = this.getRequestOptions('POST', headers, maxBodyLength);
            const request = this.getProtocol(url)
                .request(url, requestOptions, this.getJsonResponse(resolve, reject));
            request.on('error', reject);
            request.write(content);
            request.end();
        });
    }
    postFile(file, url, headers, maxBodyLength) {
        return new Promise((resolve, reject) => {
            const stream = fs.createReadStream(file);
            const requestOptions = this.getRequestOptions('POST', headers, maxBodyLength);
            const request = this.getProtocol(url)
                .request(url, requestOptions, this.getJsonResponse(resolve, reject));
            stream.on('error', err => {
                request.abort();
                reject(err);
            });
            request.on('error', err => {
                stream.close();
                reject(err);
            });
            stream.on('open', () => stream.pipe(request));
        });
    }
    getUrl(path, query) {
        const url = new URL(this.url);
        url.pathname += path;
        if (query) {
            url.search = querystring.stringify(query);
        }
        return url;
    }
    getProtocol(url) {
        if (url.protocol === 'https:')
            return followRedirects.https;
        else
            return followRedirects.http;
    }
    getRequestOptions(method, headers, maxBodyLength) {
        if (this.username && this.password) {
            if (!headers) {
                headers = {};
            }
            const credentials = Buffer.from(this.username + ':' + this.password).toString('base64');
            headers['Authorization'] = 'Basic ' + credentials;
        }
        return {
            method,
            headers,
            maxBodyLength
        };
    }
    getJsonResponse(resolve, reject) {
        return response => {
            response.setEncoding('UTF-8');
            let json = '';
            response.on('data', chunk => json += chunk);
            response.on('end', () => {
                if (response.statusCode !== undefined && (response.statusCode < 200 || response.statusCode > 299)) {
                    if (json.startsWith('{')) {
                        try {
                            const parsed = JSON.parse(json);
                            const message = parsed.message || parsed.error;
                            if (message) {
                                reject(new Error(message));
                                return;
                            }
                        }
                        catch (err) {
                            // Ignore the error and reject with response status
                        }
                    }
                    reject(util_1.statusError(response));
                }
                else if (json.startsWith('<!DOCTYPE html>')) {
                    reject(json);
                }
                else {
                    try {
                        resolve(JSON.parse(json));
                    }
                    catch (err) {
                        reject(err);
                    }
                }
            });
        };
    }
}
exports.Registry = Registry;
//# sourceMappingURL=registry.js.map