"use strict";
/********************************************************************************
 * Copyright (C) 2017 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const temp = require("temp");
const yargs = require("yargs");
const yargsFactory = require("yargs/yargs");
const application_manager_1 = require("@theia/application-manager");
const application_package_1 = require("@theia/application-package");
const check_hoisting_1 = require("./check-hoisting");
const download_plugins_1 = require("./download-plugins");
const run_test_1 = require("./run-test");
process.on('unhandledRejection', (reason, promise) => {
    throw reason;
});
process.on('uncaughtException', error => {
    if (error) {
        console.error('Uncaught Exception: ', error.toString());
        if (error.stack) {
            console.error(error.stack);
        }
    }
    process.exit(1);
});
theiaCli();
function toStringArray(argv) {
    return argv === undefined
        ? undefined
        : argv.map(arg => String(arg));
}
function rebuildCommand(command, target) {
    return {
        command,
        describe: `Rebuild/revert native node modules for "${target}"`,
        builder: {
            'cacheRoot': {
                type: 'string',
                describe: 'Root folder where to store the .browser_modules cache'
            },
            'modules': {
                alias: 'm',
                array: true,
                describe: 'List of modules to rebuild/revert'
            },
            'force': {
                alias: 'f',
                boolean: true,
                describe: 'Rebuild modules for Electron anyway',
            }
        },
        handler: ({ cacheRoot, modules, force }) => {
            // Note: `modules` is actually `string[] | undefined`.
            if (modules) {
                // It is ergonomic to pass arguments as --modules="a,b,c,..."
                // but yargs doesn't parse it this way by default.
                const flattened = [];
                for (const value of modules) {
                    if (value.includes(',')) {
                        flattened.push(...value.split(',').map(mod => mod.trim()));
                    }
                    else {
                        flattened.push(value);
                    }
                }
                modules = flattened;
            }
            application_manager_1.rebuild(target, { cacheRoot, modules, force });
        }
    };
}
function defineCommonOptions(cli) {
    return cli
        .option('app-target', {
        description: 'The target application type. Overrides `theia.target` in the application\'s package.json',
        choices: ['browser', 'electron'],
    });
}
function theiaCli() {
    const projectPath = process.cwd();
    yargs.scriptName('theia').version(require('../package.json').version);
    // Create a sub `yargs` parser to read `app-target` without
    // affecting the global `yargs` instance used by the CLI.
    const { appTarget } = defineCommonOptions(yargsFactory()).help(false).parse();
    const manager = new application_manager_1.ApplicationPackageManager({ projectPath, appTarget });
    const { target } = manager.pck;
    defineCommonOptions(yargs)
        .command({
        command: 'start [theia-args...]',
        describe: `Start the ${target} backend`,
        // Disable this command's `--help` option so that it is forwarded to Theia's CLI
        builder: cli => cli.help(false),
        handler: async ({ theiaArgs }) => {
            manager.start(toStringArray(theiaArgs));
        }
    })
        .command({
        command: 'clean',
        describe: `Clean for the ${target} target`,
        handler: async () => {
            await manager.clean();
        }
    })
        .command({
        command: 'copy',
        describe: 'Copy various files from `src-gen` to `lib`',
        handler: async () => {
            await manager.copy();
        }
    })
        .command({
        command: 'generate',
        describe: `Generate various files for the ${target} target`,
        builder: cli => application_manager_1.ApplicationPackageManager.defineGeneratorOptions(cli),
        handler: async ({ mode, splitFrontend }) => {
            await manager.generate({ mode, splitFrontend });
        }
    })
        .command({
        command: 'build [webpack-args...]',
        describe: `Generate and bundle the ${target} frontend using webpack`,
        builder: cli => application_manager_1.ApplicationPackageManager.defineGeneratorOptions(cli)
            .option('webpack-help', {
            boolean: true,
            description: 'Display Webpack\'s help',
            default: false
        }),
        handler: async ({ mode, splitFrontend, webpackHelp, webpackArgs = [] }) => {
            await manager.build(webpackHelp
                ? ['--help']
                : [
                    // Forward the `mode` argument to Webpack too:
                    '--mode', mode,
                    ...toStringArray(webpackArgs)
                ], { mode, splitFrontend });
        }
    })
        .command(rebuildCommand('rebuild', target))
        .command(rebuildCommand('rebuild:browser', 'browser'))
        .command(rebuildCommand('rebuild:electron', 'electron'))
        .command({
        command: 'check:hoisted',
        describe: 'Check that all dependencies are hoisted',
        builder: {
            'suppress': {
                alias: 's',
                describe: 'Suppress exiting with failure code',
                boolean: true,
                default: false
            }
        },
        handler: ({ suppress }) => {
            check_hoisting_1.default({ suppress });
        }
    })
        .command({
        command: 'download:plugins',
        describe: 'Download defined external plugins',
        builder: {
            'packed': {
                alias: 'p',
                describe: 'Controls whether to pack or unpack plugins',
                boolean: true,
                default: false,
            },
            'ignore-errors': {
                alias: 'i',
                describe: 'Ignore errors while downloading plugins',
                boolean: true,
                default: false,
            },
            'api-version': {
                alias: 'v',
                describe: 'Supported API version for plugins',
                default: application_package_1.DEFAULT_SUPPORTED_API_VERSION
            },
            'api-url': {
                alias: 'u',
                describe: 'Open-VSX Registry API URL',
                default: 'https://open-vsx.org/api'
            }
        },
        handler: async ({ packed }) => {
            await download_plugins_1.default({ packed });
        },
    }).command({
        command: 'test [theia-args...]',
        builder: {
            'test-inspect': {
                describe: 'Whether to auto-open a DevTools panel for test page.',
                boolean: true,
                default: false
            },
            'test-extension': {
                describe: 'Test file extension(s) to load',
                array: true,
                default: ['js']
            },
            'test-file': {
                describe: 'Specify test file(s) to be loaded prior to root suite execution',
                array: true,
                default: []
            },
            'test-ignore': {
                describe: 'Ignore test file(s) or glob pattern(s)',
                array: true,
                default: []
            },
            'test-recursive': {
                describe: 'Look for tests in subdirectories',
                boolean: true,
                default: false
            },
            'test-sort': {
                describe: 'Sort test files',
                boolean: true,
                default: false
            },
            'test-spec': {
                describe: 'One or more test files, directories, or globs to test',
                array: true,
                default: ['test']
            },
            'test-coverage': {
                describe: 'Report test coverage consumable by istanbul',
                boolean: true,
                default: false
            }
        },
        handler: async ({ testInspect, testExtension, testFile, testIgnore, testRecursive, testSort, testSpec, testCoverage, theiaArgs }) => {
            if (!process.env.THEIA_CONFIG_DIR) {
                process.env.THEIA_CONFIG_DIR = temp.track().mkdirSync('theia-test-config-dir');
            }
            await run_test_1.default({
                start: () => new Promise((resolve, reject) => {
                    const serverProcess = manager.start(toStringArray(theiaArgs));
                    serverProcess.on('message', resolve);
                    serverProcess.on('error', reject);
                    serverProcess.on('close', (code, signal) => reject(`Server process exited unexpectedly: ${code !== null && code !== void 0 ? code : signal}`));
                }),
                launch: {
                    args: ['--no-sandbox'],
                    devtools: testInspect
                },
                files: {
                    extension: testExtension,
                    file: testFile,
                    ignore: testIgnore,
                    recursive: testRecursive,
                    sort: testSort,
                    spec: testSpec
                },
                coverage: testCoverage
            });
        }
    })
        .parserConfiguration({
        'unknown-options-as-args': true,
    })
        .strictCommands()
        .demandCommand(1, 'Please run a command')
        .fail((msg, err, cli) => {
        process.exitCode = 1;
        if (err) {
            // One of the handlers threw an error:
            console.error(err);
        }
        else {
            // Yargs detected a problem with commands and/or arguments while parsing:
            cli.showHelp();
            console.error(msg);
        }
    })
        .parse();
}
//# sourceMappingURL=theia.js.map