/********************************************************************************
 * Copyright (C) 2020 Ericsson and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
/// <reference types="node" />
import { BrowserWindow, BrowserWindowConstructorOptions, Event as ElectronEvent } from '../../shared/electron';
import { ForkOptions } from 'child_process';
import { FrontendApplicationConfig } from '@theia/application-package/lib/application-props';
import URI from '../common/uri';
import { Deferred } from '../common/promise-util';
import { MaybePromise } from '../common/types';
import { ContributionProvider } from '../common/contribution-provider';
import { ElectronSecurityTokenService } from './electron-security-token-service';
import { ElectronSecurityToken } from '../electron-common/electron-token';
import Storage = require('electron-store');
/**
 * Theia tracks the maximized state of Electron Browser Windows.
 */
export interface TheiaBrowserWindowOptions extends BrowserWindowConstructorOptions {
    isMaximized?: boolean;
    isFullScreen?: boolean;
}
/**
 * Options passed to the main/default command handler.
 */
export interface ElectronMainCommandOptions {
    /**
     * By default, the first positional argument. Should be either a relative or absolute file-system path pointing to a file or a folder.
     */
    readonly file?: string;
}
/**
 * Fields related to a launch event.
 *
 * This kind of event is triggered in two different contexts:
 *  1. The app is launched for the first time, `secondInstance` is false.
 *  2. The app is already running but user relaunches it, `secondInstance` is true.
 */
export interface ElectronMainExecutionParams {
    readonly secondInstance: boolean;
    readonly argv: string[];
    readonly cwd: string;
}
export declare const ElectronMainApplicationGlobals: unique symbol;
export interface ElectronMainApplicationGlobals {
    readonly THEIA_APP_PROJECT_PATH: string;
    readonly THEIA_BACKEND_MAIN_PATH: string;
    readonly THEIA_FRONTEND_HTML_PATH: string;
}
/**
 * The default entrypoint will handle a very rudimentary CLI to open workspaces by doing `app path/to/workspace`. To override this behavior, you can extend and rebind the
 * `ElectronMainApplication` class and overriding the `launch` method.
 * A JSON-RPC communication between the Electron Main Process and the Renderer Processes is available: You can bind services using the `ElectronConnectionHandler` and
 * `ElectronIpcConnectionProvider` APIs, example:
 *
 * From an `electron-main` module:
 *
 *     bind(ElectronConnectionHandler).toDynamicValue(context =>
 *          new JsonRpcConnectionHandler(electronMainWindowServicePath,
 *          () => context.container.get(ElectronMainWindowService))
 *     ).inSingletonScope();
 *
 * And from the `electron-browser` module:
 *
 *     bind(ElectronMainWindowService).toDynamicValue(context =>
 *          ElectronIpcConnectionProvider.createProxy(context.container, electronMainWindowServicePath)
 *     ).inSingletonScope();
 */
export declare const ElectronMainApplicationContribution: unique symbol;
export interface ElectronMainApplicationContribution {
    /**
     * The application is ready and is starting. This is the time to initialize
     * services global to this process.
     *
     * Invoked when the electron-main process starts for the first time.
     */
    onStart?(application: ElectronMainApplication): MaybePromise<void>;
    /**
     * The application is stopping. Contributions must perform only synchronous operations.
     */
    onStop?(application: ElectronMainApplication): void;
}
export declare class ElectronMainProcessArgv {
    protected get processArgvBinIndex(): number;
    protected get isBundledElectronApp(): boolean;
    protected get isElectronApp(): boolean;
    getProcessArgvWithoutBin(argv?: string[]): Array<string>;
    getProcessArgvBin(argv?: string[]): string;
}
export declare namespace ElectronMainProcessArgv {
    interface ElectronMainProcess extends NodeJS.Process {
        readonly defaultApp: boolean;
        readonly versions: NodeJS.ProcessVersions & {
            readonly electron: string;
        };
    }
}
export declare class ElectronMainApplication {
    protected readonly contributions: ContributionProvider<ElectronMainApplicationContribution>;
    protected readonly globals: ElectronMainApplicationGlobals;
    protected processArgv: ElectronMainProcessArgv;
    protected electronSecurityTokenService: ElectronSecurityTokenService;
    protected readonly electronSecurityToken: ElectronSecurityToken;
    protected readonly electronStore: Storage<any>;
    protected readonly _backendPort: Deferred<number>;
    readonly backendPort: Promise<number>;
    protected _config: FrontendApplicationConfig | undefined;
    protected useNativeWindowFrame: boolean;
    protected didUseNativeWindowFrameOnStart: Map<number, boolean>;
    protected restarting: boolean;
    get config(): FrontendApplicationConfig;
    start(config: FrontendApplicationConfig): Promise<void>;
    protected getTitleBarStyle(config: FrontendApplicationConfig): 'native' | 'custom';
    protected launch(params: ElectronMainExecutionParams): Promise<void>;
    /**
     * Use this rather than creating `BrowserWindow` instances from scratch, since some security parameters need to be set, this method will do it.
     *
     * @param options
     */
    createWindow(asyncOptions?: MaybePromise<TheiaBrowserWindowOptions>): Promise<BrowserWindow>;
    getLastWindowOptions(): Promise<TheiaBrowserWindowOptions>;
    protected avoidOverlap(options: TheiaBrowserWindowOptions): TheiaBrowserWindowOptions;
    protected getDefaultOptions(): TheiaBrowserWindowOptions;
    openDefaultWindow(): Promise<BrowserWindow>;
    protected openWindowWithWorkspace(workspacePath: string): Promise<BrowserWindow>;
    /**
     * "Gently" close all windows, application will not stop if a `beforeunload` handler returns `false`.
     */
    requestStop(): void;
    protected handleMainCommand(params: ElectronMainExecutionParams, options: ElectronMainCommandOptions): Promise<void>;
    protected createWindowUri(): Promise<URI>;
    protected getDefaultTheiaWindowOptions(): TheiaBrowserWindowOptions;
    /**
     * Only show the window when the content is ready.
     */
    protected attachReadyToShow(electronWindow: BrowserWindow): void;
    /**
     * Save the window geometry state on every change.
     */
    protected attachSaveWindowState(electronWindow: BrowserWindow): void;
    protected saveWindowState(electronWindow: BrowserWindow): void;
    /**
     * Catch certain keybindings to prevent reloading the window using keyboard shortcuts.
     */
    protected attachGlobalShortcuts(electronWindow: BrowserWindow): void;
    protected restoreMaximizedState(electronWindow: BrowserWindow, options: TheiaBrowserWindowOptions): void;
    /**
     * Start the NodeJS backend server.
     *
     * @return Running server's port promise.
     */
    protected startBackend(): Promise<number>;
    protected getForkOptions(): Promise<ForkOptions>;
    protected attachElectronSecurityToken(port: number): Promise<void>;
    protected hookApplicationEvents(): void;
    protected onWillQuit(event: ElectronEvent): void;
    protected onSecondInstance(event: ElectronEvent, argv: string[], cwd: string): Promise<void>;
    protected onWindowAllClosed(event: ElectronEvent): void;
    protected restart(id: number): void;
    protected startContributions(): Promise<void>;
    protected stopContributions(): void;
}
//# sourceMappingURL=electron-main-application.d.ts.map