"use strict";
/********************************************************************************
 * Copyright (C) 2018 Red Hat, Inc. and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PluginDebugService = void 0;
const debug_service_1 = require("@theia/debug/lib/common/debug-service");
const disposable_1 = require("@theia/core/lib/common/disposable");
const inversify_1 = require("@theia/core/shared/inversify");
const ws_connection_provider_1 = require("@theia/core/lib/browser/messaging/ws-connection-provider");
const browser_1 = require("@theia/workspace/lib/browser");
/**
 * Debug service to work with plugin and extension contributions.
 */
let PluginDebugService = class PluginDebugService {
    constructor() {
        this.debuggers = [];
        this.contributors = new Map();
        this.toDispose = new disposable_1.DisposableCollection();
        // maps session and contribution
        this.sessionId2contrib = new Map();
    }
    init() {
        this.delegated = this.connectionProvider.createProxy(debug_service_1.DebugPath);
        this.toDispose.pushAll([
            disposable_1.Disposable.create(() => this.delegated.dispose()),
            disposable_1.Disposable.create(() => {
                for (const sessionId of this.sessionId2contrib.keys()) {
                    const contrib = this.sessionId2contrib.get(sessionId);
                    contrib.terminateDebugSession(sessionId);
                }
                this.sessionId2contrib.clear();
            })
        ]);
    }
    registerDebugAdapterContribution(contrib) {
        const { type } = contrib;
        if (this.contributors.has(type)) {
            console.warn(`Debugger with type '${type}' already registered.`);
            return disposable_1.Disposable.NULL;
        }
        this.contributors.set(type, contrib);
        return disposable_1.Disposable.create(() => this.unregisterDebugAdapterContribution(type));
    }
    unregisterDebugAdapterContribution(debugType) {
        this.contributors.delete(debugType);
    }
    async debugTypes() {
        const debugTypes = new Set(await this.delegated.debugTypes());
        for (const contribution of this.debuggers) {
            debugTypes.add(contribution.type);
        }
        for (const debugType of this.contributors.keys()) {
            debugTypes.add(debugType);
        }
        return [...debugTypes];
    }
    async provideDebugConfigurations(debugType, workspaceFolderUri) {
        const contributor = this.contributors.get(debugType);
        if (contributor) {
            return contributor.provideDebugConfigurations && contributor.provideDebugConfigurations(workspaceFolderUri) || [];
        }
        else {
            return this.delegated.provideDebugConfigurations(debugType, workspaceFolderUri);
        }
    }
    async provideDynamicDebugConfigurations() {
        const result = [];
        for (const [type, contributor] of this.contributors.entries()) {
            const typeConfigurations = this.resolveDynamicConfigurationsForType(type, contributor);
            result.push(typeConfigurations);
        }
        return Promise.all(result);
    }
    async resolveDynamicConfigurationsForType(type, contributor) {
        const configurations = await contributor.provideDebugConfigurations(undefined, true);
        for (const configuration of configurations) {
            configuration.dynamic = true;
        }
        return { type, configurations };
    }
    async resolveDebugConfiguration(config, workspaceFolderUri) {
        let resolved = config;
        // we should iterate over all to handle configuration providers for `*`
        for (const contributor of this.contributors.values()) {
            if (contributor) {
                try {
                    const next = await contributor.resolveDebugConfiguration(resolved, workspaceFolderUri);
                    if (next) {
                        resolved = next;
                    }
                    else {
                        return resolved;
                    }
                }
                catch (e) {
                    console.error(e);
                }
            }
        }
        return this.delegated.resolveDebugConfiguration(resolved, workspaceFolderUri);
    }
    async resolveDebugConfigurationWithSubstitutedVariables(config, workspaceFolderUri) {
        let resolved = config;
        // we should iterate over all to handle configuration providers for `*`
        for (const contributor of this.contributors.values()) {
            if (contributor) {
                try {
                    const next = await contributor.resolveDebugConfigurationWithSubstitutedVariables(resolved, workspaceFolderUri);
                    if (next) {
                        resolved = next;
                    }
                    else {
                        return resolved;
                    }
                }
                catch (e) {
                    console.error(e);
                }
            }
        }
        return this.delegated.resolveDebugConfigurationWithSubstitutedVariables(resolved, workspaceFolderUri);
    }
    registerDebugger(contribution) {
        this.debuggers.push(contribution);
        return disposable_1.Disposable.create(() => {
            const index = this.debuggers.indexOf(contribution);
            if (index !== -1) {
                this.debuggers.splice(index, 1);
            }
        });
    }
    async getDebuggersForLanguage(language) {
        const debuggers = await this.delegated.getDebuggersForLanguage(language);
        for (const contributor of this.debuggers) {
            const languages = contributor.languages;
            if (languages && languages.indexOf(language) !== -1) {
                const { label, type } = contributor;
                debuggers.push({ type, label: label || type });
            }
        }
        return debuggers;
    }
    async getSchemaAttributes(debugType) {
        let schemas = await this.delegated.getSchemaAttributes(debugType);
        for (const contribution of this.debuggers) {
            if (contribution.configurationAttributes &&
                (contribution.type === debugType || contribution.type === '*' || debugType === '*')) {
                schemas = schemas.concat(contribution.configurationAttributes);
            }
        }
        return schemas;
    }
    async getConfigurationSnippets() {
        let snippets = await this.delegated.getConfigurationSnippets();
        for (const contribution of this.debuggers) {
            if (contribution.configurationSnippets) {
                snippets = snippets.concat(contribution.configurationSnippets);
            }
        }
        return snippets;
    }
    async createDebugSession(config) {
        const contributor = this.contributors.get(config.type);
        if (contributor) {
            const sessionId = await contributor.createDebugSession(config);
            this.sessionId2contrib.set(sessionId, contributor);
            return sessionId;
        }
        else {
            return this.delegated.createDebugSession(config);
        }
    }
    async terminateDebugSession(sessionId) {
        const contributor = this.sessionId2contrib.get(sessionId);
        if (contributor) {
            this.sessionId2contrib.delete(sessionId);
            return contributor.terminateDebugSession(sessionId);
        }
        else {
            return this.delegated.terminateDebugSession(sessionId);
        }
    }
    dispose() {
        this.toDispose.dispose();
    }
};
__decorate([
    inversify_1.inject(ws_connection_provider_1.WebSocketConnectionProvider),
    __metadata("design:type", ws_connection_provider_1.WebSocketConnectionProvider)
], PluginDebugService.prototype, "connectionProvider", void 0);
__decorate([
    inversify_1.inject(browser_1.WorkspaceService),
    __metadata("design:type", browser_1.WorkspaceService)
], PluginDebugService.prototype, "workspaceService", void 0);
__decorate([
    inversify_1.postConstruct(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", void 0)
], PluginDebugService.prototype, "init", null);
PluginDebugService = __decorate([
    inversify_1.injectable()
], PluginDebugService);
exports.PluginDebugService = PluginDebugService;
//# sourceMappingURL=plugin-debug-service.js.map