"use strict";
/********************************************************************************
 * Copyright (C) 2020 Red Hat, Inc. and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.CallHierarchyAdapter = void 0;
const vscode_uri_1 = require("@theia/core/shared/vscode-uri");
const Converter = require("../type-converters");
const types = require("../types-impl");
class CallHierarchyAdapter {
    constructor(provider, documents) {
        this.provider = provider;
        this.documents = documents;
    }
    async provideRootDefinition(resource, position, token) {
        const documentData = this.documents.getDocumentData(resource);
        if (!documentData) {
            return Promise.reject(new Error(`There is no document for ${resource}`));
        }
        const definition = await this.provider.prepareCallHierarchy(documentData.document, new types.Position(position.lineNumber, position.column), token);
        if (!definition) {
            return undefined;
        }
        return this.fromCallHierarchyItem(definition);
    }
    async provideCallers(definition, token) {
        const callers = await this.provider.provideCallHierarchyIncomingCalls(this.toCallHierarchyItem(definition), token);
        if (!callers) {
            return undefined;
        }
        return callers.map(item => this.fromCallHierarchyIncomingCall(item));
    }
    async provideCallees(definition, token) {
        const callees = await this.provider.provideCallHierarchyOutgoingCalls(this.toCallHierarchyItem(definition), token);
        if (!callees) {
            return undefined;
        }
        return callees.map(item => this.fromCallHierarchyOutgoingCall(item));
    }
    fromCallHierarchyItem(item) {
        return {
            uri: item.uri,
            range: this.fromRange(item.range),
            selectionRange: this.fromRange(item.selectionRange),
            name: item.name,
            kind: item.kind,
            tags: item.tags
        };
    }
    fromRange(range) {
        return {
            startLineNumber: range.start.line + 1,
            startColumn: range.start.character + 1,
            endLineNumber: range.end.line + 1,
            endColumn: range.end.character + 1,
        };
    }
    toRange(range) {
        return new types.Range(range.startLineNumber - 1, range.startColumn - 1, range.endLineNumber - 1, range.endColumn - 1);
    }
    toCallHierarchyItem(definition) {
        return new types.CallHierarchyItem(Converter.SymbolKind.toSymbolKind(definition.kind), definition.name, definition.detail ? definition.detail : '', vscode_uri_1.URI.revive(definition.uri), this.toRange(definition.range), this.toRange(definition.selectionRange));
    }
    fromCallHierarchyIncomingCall(caller) {
        return {
            callerDefinition: this.fromCallHierarchyItem(caller.from),
            references: caller.fromRanges.map(l => this.fromRange(l))
        };
    }
    fromCallHierarchyOutgoingCall(caller) {
        return {
            callerDefinition: this.fromCallHierarchyItem(caller.to),
            references: caller.fromRanges.map(this.fromRange.bind(this)),
        };
    }
}
exports.CallHierarchyAdapter = CallHierarchyAdapter;
//# sourceMappingURL=call-hierarchy.js.map