'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

function _interopDefault (ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

var StreamReader = _interopDefault(require('@emmetio/stream-reader'));
var _emmetio_streamReaderUtils = require('@emmetio/stream-reader-utils');

/**
 * Fixes StreamReader design flaw: check if stream is at the start-of-file
 * @param  {StreamReader}  stream
 * @return {Boolean}
 */
function isSoF(stream) {
	return stream.sof ? stream.sof() : stream.pos <= 0;
}

const DOT = 46; // .

/**
 * Consumes number from given stream, either in forward or backward direction
 * @param {StreamReader} stream
 * @param {Boolean}      backward Consume number in backward direction
 */
var consumeNumber = function(stream, backward) {
	return backward ? consumeBackward(stream) : consumeForward(stream);
};

/**
 * Consumes number in forward stream direction
 * @param  {StreamReader} stream
 * @return {Boolean}        Returns true if number was consumed
 */
function consumeForward(stream) {
	const start = stream.pos;
	if (stream.eat(DOT) && stream.eatWhile(_emmetio_streamReaderUtils.isNumber)) {
		// short decimal notation: .025
		return true;
	}

	if (stream.eatWhile(_emmetio_streamReaderUtils.isNumber) && (!stream.eat(DOT) || stream.eatWhile(_emmetio_streamReaderUtils.isNumber))) {
		// either integer or decimal: 10, 10.25
		return true;
	}

	stream.pos = start;
	return false;
}

/**
 * Consumes number in backward stream direction
 * @param  {StreamReader} stream
 * @return {Boolean}        Returns true if number was consumed
 */
function consumeBackward(stream) {
	const start = stream.pos;
	let ch, hadDot = false, hadNumber = false;
	// NB a StreamReader insance can be editor-specific and contain objects
	// as a position marker. Since we don’t know for sure how to compare editor
	// position markers, use consumed length instead to detect if number was consumed
	let len = 0;

	while (!isSoF(stream)) {
		stream.backUp(1);
		ch = stream.peek();

		if (ch === DOT && !hadDot && hadNumber) {
			hadDot = true;
		} else if (!_emmetio_streamReaderUtils.isNumber(ch)) {
			stream.next();
			break;
		}

		hadNumber = true;
		len++;
	}

	if (len) {
		const pos = stream.pos;
		stream.start = pos;
		stream.pos = start;
		return true;
	}

	stream.pos = start;
	return false;
}

/**
 * Expression parser and tokenizer
 */
// token types
const NUMBER = 'num';
const OP1    = 'op1';
const OP2    = 'op2';

// operators
const PLUS              = 43; // +
const MINUS             = 45; // -
const MULTIPLY          = 42; // *
const DIVIDE            = 47; // /
const INT_DIVIDE        = 92; // \
const LEFT_PARENTHESIS  = 40; // (
const RIGHT_PARENTHESIS = 41; // )

// parser states
const PRIMARY      = 1 << 0;
const OPERATOR     = 1 << 1;
const LPAREN       = 1 << 2;
const RPAREN       = 1 << 3;
const SIGN         = 1 << 4;
const NULLARY_CALL = 1 << 5;

class Token {
	constructor(type, value, priority) {
		this.type = type;
		this.value = value;
		this.priority = priority || 0;
	}
}

const nullary = new Token(NULLARY_CALL);

function parse(expr, backward) {
	return backward ? parseBackward(expr) : parseForward(expr);
}

/**
 * Parses given expression in forward direction
 * @param  {String|StreamReader} expr
 * @return {Token[]}
 */
function parseForward(expr) {
	const stream = typeof expr === 'object' ? expr : new StreamReader(expr);
	let ch, priority = 0;
	let expected = (PRIMARY | LPAREN | SIGN);
	const tokens = [];

	while (!stream.eof()) {
		stream.eatWhile(_emmetio_streamReaderUtils.isWhiteSpace);
		stream.start = stream.pos;

		if (consumeNumber(stream)) {
			if ((expected & PRIMARY) === 0) {
				error('Unexpected number', stream);
			}

			tokens.push( number(stream.current()) );
			expected = (OPERATOR | RPAREN);
		} else if (isOperator(stream.peek())) {
			ch = stream.next();
			if (isSign(ch) && (expected & SIGN)) {
				if (isNegativeSign(ch)) {
					tokens.push(op1(ch, priority));
				}
				expected = (PRIMARY | LPAREN | SIGN);
			} else {
				if ((expected & OPERATOR) === 0) {
					error('Unexpected operator', stream);
				}
				tokens.push(op2(ch, priority));
				expected = (PRIMARY | LPAREN | SIGN);
			}
		} else if (stream.eat(LEFT_PARENTHESIS)) {
			if ((expected & LPAREN) === 0) {
				error('Unexpected "("', stream);
			}

			priority += 10;
			expected = (PRIMARY | LPAREN | SIGN | NULLARY_CALL);
		} else if (stream.eat(RIGHT_PARENTHESIS)) {
			priority -= 10;

			if (expected & NULLARY_CALL) {
				tokens.push(nullary);
			} else if ((expected & RPAREN) === 0) {
				error('Unexpected ")"', stream);
			}

			expected = (OPERATOR | RPAREN | LPAREN);
		} else {
			error('Unknown character', stream);
		}
	}

	if (priority < 0 || priority >= 10) {
		error('Unmatched "()"', stream);
	}

	const result = orderTokens(tokens);

	if (result === null) {
		error('Parity', stream);
	}

	return result;
}

/**
 * Parses given exprssion in reverse order, e.g. from back to end, and stops when
 * first unknown character was found
 * @param  {String|StreamReader} expr
 * @return {Array}
 */
function parseBackward(expr) {
	let stream;
	if (typeof expr === 'object') {
		stream = expr;
	} else {
		stream = new StreamReader(expr);
		stream.start = stream.pos = expr.length;
	}

	let ch, priority = 0;
	let expected = (PRIMARY | RPAREN);
	const tokens = [];

	while (!isSoF(stream)) {
		if (consumeNumber(stream, true)) {
			if ((expected & PRIMARY) === 0) {
				error('Unexpected number', stream);
			}

			tokens.push( number(stream.current()) );
			expected = (OPERATOR | SIGN | LPAREN);

			// NB should explicitly update stream position for backward direction
			stream.pos = stream.start;
		} else {
			stream.backUp(1);
			ch = stream.peek();

			if (isOperator(ch)) {
				if (isSign(ch) && (expected & SIGN) && isReverseSignContext(stream)) {
					if (isNegativeSign(ch)) {
						tokens.push(op1(ch, priority));
					}
					expected = (LPAREN | RPAREN | OPERATOR | PRIMARY);
				} else {
					if ((expected & OPERATOR) === 0) {
						stream.next();
						break;
					}
					tokens.push(op2(ch, priority));
					expected = (PRIMARY | RPAREN);
				}
			} else if (ch === RIGHT_PARENTHESIS) {
				if ((expected & RPAREN) === 0) {
					stream.next();
					break;
				}

				priority += 10;
				expected = (PRIMARY | RPAREN | LPAREN);
			} else if (ch === LEFT_PARENTHESIS) {
				priority -= 10;

				if (expected & NULLARY_CALL) {
					tokens.push(nullary);
				} else if ((expected & LPAREN) === 0) {
					stream.next();
					break;
				}

				expected = (OPERATOR | SIGN | LPAREN | NULLARY_CALL);
			} else if (!_emmetio_streamReaderUtils.isWhiteSpace(ch)) {
				stream.next();
				break;
			}
		}
	}

	if (priority < 0 || priority >= 10) {
		error('Unmatched "()"', stream);
	}

	const result = orderTokens(tokens.reverse());
	if (result === null) {
		error('Parity', stream);
	}

	// edge case: expression is preceded by white-space;
	// move stream position back to expression start
	stream.eatWhile(_emmetio_streamReaderUtils.isWhiteSpace);

	return result;
}

/**
 * Orders parsed tokens (operands and operators) in given array so that they are
 * laid off in order of execution
 * @param  {Token[]} tokens
 * @return {Token[]}
 */
function orderTokens(tokens) {
	const operators = [], operands = [];
	let noperators = 0;

	for (let i = 0, token; i < tokens.length; i++) {
		token = tokens[i];

		if (token.type === NUMBER) {
			operands.push(token);
		} else {
			noperators += token.type === OP1 ? 1 : 2;

			while (operators.length) {
				if (token.priority <= operators[operators.length - 1].priority) {
					operands.push(operators.pop());
				} else {
					break;
				}
			}

			operators.push(token);
		}
	}

	return noperators + 1 === operands.length + operators.length
		? operands.concat(operators.reverse())
		: null /* parity */;
}

/**
 * Check if current stream state is in sign (e.g. positive or negative) context
 * for reverse parsing
 * @param  {StreamReader} stream
 * @return {Boolean}
 */
function isReverseSignContext(stream) {
	const start = stream.pos;
	let ch, inCtx = true;

	while (!isSoF(stream)) {
		stream.backUp(1);
		ch = stream.peek();

		if (_emmetio_streamReaderUtils.isWhiteSpace(ch)) {
			continue;
		}

		inCtx = ch === LEFT_PARENTHESIS || isOperator(ch);
		break;
	}

	stream.pos = start;
	return inCtx;
}

/**
 * Number token factory
 * @param  {String} value
 * @param  {Number} [priority ]
 * @return {Token}
 */
function number(value, priority) {
	return new Token(NUMBER, parseFloat(value), priority);
}

/**
 * Unary operator factory
 * @param  {Number} value    Operator  character code
 * @param  {Number} priority Operator execution priority
 * @return {Token[]}
 */
function op1(value, priority) {
	if (value === MINUS) {
		priority += 2;
	}
	return new Token(OP1, value, priority);
}

/**
 * Binary operator factory
 * @param  {Number} value    Operator  character code
 * @param  {Number} priority Operator execution priority
 * @return {Token[]}
 */
function op2(value, priority) {
	if (value === MULTIPLY) {
		priority += 1;
	} else if (value === DIVIDE || value === INT_DIVIDE) {
		priority += 2;
	}

	return new Token(OP2, value, priority);
}

function error(name, stream) {
	if (stream) {
		name += ` at column ${stream.start} of expression`;
	}
	throw new Error(name);
}

function isSign(ch) {
	return isPositiveSign(ch) || isNegativeSign(ch);
}

function isPositiveSign(ch) {
	return ch === PLUS;
}

function isNegativeSign(ch) {
	return ch === MINUS;
}

function isOperator(ch) {
	return ch === PLUS || ch === MINUS || ch === MULTIPLY
		|| ch === DIVIDE || ch === INT_DIVIDE;
}

const ops1 = {
	[MINUS]: num => -num
};

const ops2 = {
	[PLUS]:       (a, b) => a + b,
	[MINUS]:      (a, b) => a - b,
	[MULTIPLY]:   (a, b) => a * b,
	[DIVIDE]:     (a, b) => a / b,
	[INT_DIVIDE]: (a, b) => Math.floor(a / b)
};

/**
 * Evaluates given math expression
 * @param  {String|StreamReader|Array} expr Expression to evaluate
 * @param  {Boolean}                   [backward] Parses given expression (string
 *                                                or stream) in backward direction
 * @return {Number}
 */
var index = function(expr, backward) {
	if (!Array.isArray(expr)) {
		expr = parse(expr, backward);
	}

	if (!expr || !expr.length) {
		return null;
	}

	const nstack = [];
	let n1, n2, f;
	let item, value;

	for (let i = 0, il = expr.length, token; i < il; i++) {
		token = expr[i];
		if (token.type === NUMBER) {
			nstack.push(token.value);
		} else if (token.type === OP2) {
			n2 = nstack.pop();
			n1 = nstack.pop();
			f = ops2[token.value];
			nstack.push(f(n1, n2));
		} else if (token.type === OP1) {
			n1 = nstack.pop();
			f = ops1[token.value];
			nstack.push(f(n1));
		} else {
			throw new Error('Invalid expression');
		}
	}

	if (nstack.length > 1) {
		throw new Error('Invalid Expression (parity)');
	}

	return nstack[0];
};

exports['default'] = index;
exports.parse = parse;
