package cvo

import (
	"context"
	"fmt"
	"reflect"
	"strconv"
	"testing"
	"time"

	"github.com/davecgh/go-spew/spew"
	"github.com/google/uuid"

	"k8s.io/apimachinery/pkg/api/errors"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/apis/meta/v1/unstructured"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/runtime/schema"
	"k8s.io/apimachinery/pkg/util/wait"
	dynamicfake "k8s.io/client-go/dynamic/fake"
	clientgotesting "k8s.io/client-go/testing"
	"k8s.io/client-go/tools/record"
	"k8s.io/client-go/util/workqueue"

	configv1 "github.com/openshift/api/config/v1"
	"github.com/openshift/client-go/config/clientset/versioned/fake"

	"github.com/openshift/cluster-version-operator/pkg/payload"
	"github.com/openshift/cluster-version-operator/pkg/payload/precondition"
	"github.com/openshift/library-go/pkg/manifest"
)

func setupCVOTest(payloadDir string) (*Operator, map[string]runtime.Object, *fake.Clientset, *dynamicfake.FakeDynamicClient, func()) {
	client := &fake.Clientset{}
	client.AddReactor("*", "*", func(action clientgotesting.Action) (handled bool, ret runtime.Object, err error) {
		return false, nil, fmt.Errorf("unexpected client action: %#v", action)
	})
	cvs := make(map[string]runtime.Object)
	client.AddReactor("*", "clusterversions", func(action clientgotesting.Action) (handled bool, ret runtime.Object, err error) {
		switch a := action.(type) {
		case clientgotesting.GetActionImpl:
			obj, ok := cvs[a.GetName()]
			if !ok {
				return true, nil, errors.NewNotFound(schema.GroupResource{Resource: "clusterversions"}, a.GetName())
			}
			return true, obj.DeepCopyObject(), nil
		case clientgotesting.CreateActionImpl:
			obj := a.GetObject().DeepCopyObject().(*configv1.ClusterVersion)
			obj.Generation = 1
			cvs[obj.Name] = obj
			return true, obj, nil
		case clientgotesting.UpdateActionImpl:
			obj := a.GetObject().DeepCopyObject().(*configv1.ClusterVersion)
			existing := cvs[obj.Name].DeepCopyObject().(*configv1.ClusterVersion)
			rv, _ := strconv.Atoi(existing.ResourceVersion)
			nextRV := strconv.Itoa(rv + 1)
			if a.GetSubresource() == "status" {
				existing.Status = obj.Status
			} else {
				existing.Spec = obj.Spec
				existing.ObjectMeta = obj.ObjectMeta
				if existing.Generation > obj.Generation {
					existing.Generation = existing.Generation + 1
				} else {
					existing.Generation = obj.Generation + 1
				}
			}
			existing.ResourceVersion = nextRV
			cvs[existing.Name] = existing
			return true, existing, nil
		}
		return false, nil, fmt.Errorf("unexpected client action: %#v", action)
	})
	client.AddReactor("get", "featuregates", func(action clientgotesting.Action) (handled bool, ret runtime.Object, err error) {
		switch a := action.(type) {
		case clientgotesting.GetAction:
			return true, nil, errors.NewNotFound(schema.GroupResource{Resource: "clusterversions"}, a.GetName())
		}
		return false, nil, nil
	})

	o := &Operator{
		namespace:                   "test",
		name:                        "version",
		enableDefaultClusterVersion: true,
		queue:                       workqueue.NewNamedRateLimitingQueue(workqueue.DefaultControllerRateLimiter(), "cvo-loop-test"),
		client:                      client,
		cvLister:                    &clientCVLister{client: client},
		exclude:                     "exclude-test",
		eventRecorder:               record.NewFakeRecorder(100),
		clusterProfile:              payload.DefaultClusterProfile,
	}

	dynamicScheme := runtime.NewScheme()
	//dynamicScheme.AddKnownTypeWithName(schema.GroupVersionKind{Group: "test.cvo.io", Version: "v1", Kind: "TestA"}, &unstructured.Unstructured{})
	dynamicScheme.AddKnownTypeWithName(schema.GroupVersionKind{Group: "test.cvo.io", Version: "v1", Kind: "TestB"}, &unstructured.Unstructured{})
	dynamicClient := dynamicfake.NewSimpleDynamicClient(dynamicScheme)

	worker := NewSyncWorker(
		&fakeDirectoryRetriever{Info: PayloadInfo{Directory: payloadDir}},
		&testResourceBuilder{client: dynamicClient},
		time.Second/2,
		wait.Backoff{
			Steps: 1,
		},
		"exclude-test",
		false,
		record.NewFakeRecorder(100),
		o.clusterProfile,
	)
	o.configSync = worker

	return o, cvs, client, dynamicClient, func() { o.queue.ShutDown() }
}

func TestCVO_StartupAndSync(t *testing.T) {
	o, cvs, client, _, shutdownFn := setupCVOTest("testdata/payloadtest")

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	defer shutdownFn()
	worker := o.configSync.(*SyncWorker)
	go worker.Start(ctx, 1, o.name, o.cvLister)

	// Step 1: Verify the CVO creates the initial Cluster Version object
	//
	client.ClearActions()
	err := o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions := client.Actions()
	if len(actions) != 4 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	// read from lister
	expectGet(t, actions[0], "clusterversions", "", "version")
	// read before create
	expectGet(t, actions[1], "clusterversions", "", "version")
	// create initial version
	actual := cvs["version"].(*configv1.ClusterVersion)
	expectCreate(t, actions[2], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name: "version",
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: actual.Spec.ClusterID,
			Channel:   "fast",
		},
	})

	// read after create
	expectGet(t, actions[3], "clusterversions", "", "version")

	verifyAllStatus(t, worker.StatusCh())

	// Step 2: Ensure the CVO reports a status error if it has nothing to sync
	//
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	actual = cvs["version"].(*configv1.ClusterVersion)
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:       "version",
			Generation: 1,
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: actual.Spec.ClusterID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			History: []configv1.UpdateHistory{
				// empty because the operator release image is not set, so we have no input
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionFalse},
				// report back to the user that we don't have enough info to proceed
				{Type: ClusterStatusFailing, Status: configv1.ConditionTrue, Reason: "NoDesiredImage", Message: "No configured operator version, unable to update cluster"},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionTrue, Reason: "NoDesiredImage", Message: "Unable to apply <unknown>: an unknown error has occurred: NoDesiredImage"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
			},
		},
	})
	verifyAllStatus(t, worker.StatusCh())

	// Step 3: Given an operator image, begin synchronizing
	//
	o.release.Image = "image/image:1"
	o.release.Version = "1.0.0-abc"
	desired := configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"}
	//
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	actual = cvs["version"].(*configv1.ClusterVersion)
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			Generation:      1,
			ResourceVersion: "1",
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: actual.Spec.ClusterID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			ObservedGeneration: 1,
			Desired:            desired,
			History: []configv1.UpdateHistory{
				{State: configv1.PartialUpdate, Image: "image/image:1", Version: "1.0.0-abc", StartedTime: defaultStartedTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionFalse},
				// cleared failing status and set progressing
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionTrue, Message: "Working towards 1.0.0-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
				{Type: DesiredReleaseAccepted, Status: configv1.ConditionTrue, Reason: "PayloadLoaded",
					Message: "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\""},
			},
		},
	})
	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Generation: 1,
			Actual:     configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "RetrievePayload",
				Message:            "Retrieving and verifying payload version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Generation:   1,
			Actual:       configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			LastProgress: time.Unix(1, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(2, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Generation:  1,
			Total:       3,
			Initial:     true,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(2, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(3, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Generation:  1,
			Done:        1,
			Total:       3,
			Initial:     true,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(3, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(4, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Generation:  1,
			Done:        2,
			Total:       3,
			Initial:     true,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(4, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(5, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Generation:  1,
			Reconciling: true,
			Completed:   1,
			Done:        3,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(5, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(6, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
	)

	// Step 4: Now that sync is complete, verify status is updated to represent image contents
	//
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	// update the status to indicate we are synced, available, and report versions
	actual = cvs["version"].(*configv1.ClusterVersion)
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			Generation:      1,
			ResourceVersion: "2",
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: actual.Spec.ClusterID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			ObservedGeneration: 1,
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			Desired: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.CompletedUpdate, Image: "image/image:1", Version: "1.0.0-abc", StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.0-abc"},
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionFalse, Message: "Cluster version is 1.0.0-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
				{Type: DesiredReleaseAccepted, Status: configv1.ConditionTrue, Reason: "PayloadLoaded",
					Message: "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\""},
			},
		},
	})

	// Step 5: Wait for the SyncWorker to trigger a reconcile (500ms after the first)
	//
	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Generation:  1,
			Reconciling: true,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Generation:  1,
			Reconciling: true,
			Done:        1,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(1, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(2, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Generation:  1,
			Reconciling: true,
			Done:        2,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(2, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(3, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Generation:  1,
			Reconciling: true,
			Completed:   2,
			Done:        3,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(3, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(4, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
	)

	// Step 6: After a reconciliation, there should be no status change because the state is the same
	//
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 1 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
}

func TestCVO_StartupAndSyncUnverifiedPayload(t *testing.T) {
	o, cvs, client, _, shutdownFn := setupCVOTest("testdata/payloadtest")

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	defer shutdownFn()

	// make the image report unverified
	payloadErr := &payload.UpdateError{
		Reason:  "ImageVerificationFailed",
		Message: "The update cannot be verified: some random error",
		Nested:  fmt.Errorf("some random error"),
	}
	if !isImageVerificationError(payloadErr) {
		t.Fatal("not the correct error type")
	}
	worker := o.configSync.(*SyncWorker)
	worker.retriever.(*fakeDirectoryRetriever).Info = PayloadInfo{
		Directory: "testdata/payloadtest",
		Local:     true,

		VerificationError: payloadErr,
	}

	go worker.Start(ctx, 1, o.name, o.cvLister)

	// Step 1: Verify the CVO creates the initial Cluster Version object
	//
	client.ClearActions()
	err := o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions := client.Actions()
	if len(actions) != 4 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	// read from lister
	expectGet(t, actions[0], "clusterversions", "", "version")
	// read before create
	expectGet(t, actions[1], "clusterversions", "", "version")
	// create initial version
	actual := cvs["version"].(*configv1.ClusterVersion)
	expectCreate(t, actions[2], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name: "version",
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: actual.Spec.ClusterID,
			Channel:   "fast",
		},
	})
	// read after create
	expectGet(t, actions[3], "clusterversions", "", "version")

	verifyAllStatus(t, worker.StatusCh())

	// Step 2: Ensure the CVO reports a status error if it has nothing to sync
	//
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	actual = cvs["version"].(*configv1.ClusterVersion)
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:       "version",
			Generation: 1,
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: actual.Spec.ClusterID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			History: []configv1.UpdateHistory{
				// empty because the operator release image is not set, so we have no input
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionFalse},
				// report back to the user that we don't have enough info to proceed
				{Type: ClusterStatusFailing, Status: configv1.ConditionTrue, Reason: "NoDesiredImage", Message: "No configured operator version, unable to update cluster"},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionTrue, Reason: "NoDesiredImage", Message: "Unable to apply <unknown>: an unknown error has occurred: NoDesiredImage"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
			},
		},
	})
	verifyAllStatus(t, worker.StatusCh())

	// Step 3: Given an operator image, begin synchronizing
	//
	o.release.Image = "image/image:1"
	o.release.Version = "1.0.0-abc"
	desired := configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"}
	//
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	actual = cvs["version"].(*configv1.ClusterVersion)
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			Generation:      1,
			ResourceVersion: "1",
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: actual.Spec.ClusterID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			Desired:            desired,
			ObservedGeneration: 1,
			History: []configv1.UpdateHistory{
				{State: configv1.PartialUpdate, Image: "image/image:1", Version: "1.0.0-abc", StartedTime: defaultStartedTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionFalse},
				// cleared failing status and set progressing
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionTrue, Message: "Working towards 1.0.0-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
				{Type: DesiredReleaseAccepted, Status: configv1.ConditionTrue, Reason: "PayloadLoaded",
					Message: "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\""},
			},
		},
	})
	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Actual:     configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			Generation: 1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "RetrievePayload",
				Message:            "Retrieving and verifying payload version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Actual: configv1.Release{
				Version: "1.0.0-abc",
				Image:   "image/image:1",
			},
			LastProgress: time.Unix(1, 0),
			Generation:   1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(2, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Total:       3,
			Initial:     true,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(2, 0),
			Generation:   1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(3, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Done:        1,
			Total:       3,
			Initial:     true,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(3, 0),
			Generation:   1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(4, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Initial:     true,
			Done:        2,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(4, 0),
			Generation:   1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(5, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
	)

	// wait for status to reflect sync of new payload
	waitForStatusCompleted(t, worker)

	// Step 4: Now that sync is complete, verify status is updated to represent image contents
	//
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}

	actions = client.Actions()

	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	// update the status to indicate we are synced, available, and report versions
	actual = cvs["version"].(*configv1.ClusterVersion)
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			Generation:      1,
			ResourceVersion: "2",
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: actual.Spec.ClusterID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			ObservedGeneration: 1,
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			Desired: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.CompletedUpdate, Image: "image/image:1", Version: "1.0.0-abc", StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.0-abc"},
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionFalse, Message: "Cluster version is 1.0.0-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
				{Type: DesiredReleaseAccepted, Status: configv1.ConditionTrue, Reason: "PayloadLoaded",
					Message: "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\""},
			},
		},
	})

	// Step 5: Wait for the SyncWorker to trigger a reconcile (500ms after the first)
	//
	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Reconciling: true,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			Generation: 1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Done:        1,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			Generation:   1,
			LastProgress: time.Unix(1, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(2, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Done:        2,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			Generation:   1,
			LastProgress: time.Unix(2, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(3, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Completed:   2,
			Done:        3,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(3, 0),
			Generation:   1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(4, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
	)

	// Step 6: After a reconciliation, there should be no status change because the state is the same
	//
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 1 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
}

func TestCVO_StartupAndSyncPreconditionFailing(t *testing.T) {
	o, cvs, client, _, shutdownFn := setupCVOTest("testdata/payloadtest")

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	defer shutdownFn()

	worker := o.configSync.(*SyncWorker)
	// Need the precondition check to fail permanently, so setting failure until 100 attempt to simulate that.
	worker.preconditions = []precondition.Precondition{&testPrecondition{SuccessAfter: 100}}
	worker.retriever.(*fakeDirectoryRetriever).Info = PayloadInfo{
		Directory: "testdata/payloadtest",
		Local:     true,
	}
	go worker.Start(ctx, 1, o.name, o.cvLister)

	// Step 1: Verify the CVO creates the initial Cluster Version object
	//
	client.ClearActions()
	err := o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions := client.Actions()
	if len(actions) != 4 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	// read from lister
	expectGet(t, actions[0], "clusterversions", "", "version")
	// read before create
	expectGet(t, actions[1], "clusterversions", "", "version")
	// create initial version
	actual := cvs["version"].(*configv1.ClusterVersion)
	expectCreate(t, actions[2], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name: "version",
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: actual.Spec.ClusterID,
			Channel:   "fast",
		},
	})
	// read after create
	expectGet(t, actions[3], "clusterversions", "", "version")
	verifyAllStatus(t, worker.StatusCh())

	// Step 2: Ensure the CVO reports a status error if it has nothing to sync
	//
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	actual = cvs["version"].(*configv1.ClusterVersion)
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:       "version",
			Generation: 1,
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: actual.Spec.ClusterID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			History: []configv1.UpdateHistory{
				// empty because the operator release image is not set, so we have no input
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionFalse},
				// report back to the user that we don't have enough info to proceed
				{Type: ClusterStatusFailing, Status: configv1.ConditionTrue, Reason: "NoDesiredImage", Message: "No configured operator version, unable to update cluster"},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionTrue, Reason: "NoDesiredImage", Message: "Unable to apply <unknown>: an unknown error has occurred: NoDesiredImage"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
			},
		},
	})
	verifyAllStatus(t, worker.StatusCh())

	// Step 3: Given an operator image, begin synchronizing
	//
	o.release.Image = "image/image:1"
	o.release.Version = "1.0.0-abc"
	desired := configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"}
	//
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	actual = cvs["version"].(*configv1.ClusterVersion)
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			Generation:      1,
			ResourceVersion: "1",
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: actual.Spec.ClusterID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			Desired:            desired,
			ObservedGeneration: 1,
			History: []configv1.UpdateHistory{
				{State: configv1.PartialUpdate, Image: "image/image:1", Version: "1.0.0-abc", StartedTime: defaultStartedTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionFalse},
				// cleared failing status and set progressing
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionTrue, Message: "Working towards 1.0.0-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
				{Type: DesiredReleaseAccepted, Status: configv1.ConditionTrue, Reason: "PayloadLoaded",
					Message: "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\""},
			},
		},
	})
	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Actual:     configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			Generation: 1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "RetrievePayload",
				Message:            "Retrieving and verifying payload version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Actual: configv1.Release{
				Version: "1.0.0-abc",
				Image:   "image/image:1",
			},
			Generation:   1,
			LastProgress: time.Unix(1, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(2, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Total:       3,
			Initial:     true,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(2, 0),
			Generation:   1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(3, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Done:        1,
			Total:       3,
			Initial:     true,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(3, 0),
			Generation:   1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(4, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Done:        2,
			Total:       3,
			Initial:     true,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(4, 0),
			Generation:   1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(5, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
	)

	// wait for status to reflect sync of new payload
	waitForStatusCompleted(t, worker)

	// Step 4: Now that sync is complete, verify status is updated to represent image contents
	//
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}

	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	// update the status to indicate we are synced, available, and report versions
	actual = cvs["version"].(*configv1.ClusterVersion)
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			Generation:      1,
			ResourceVersion: "2",
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: actual.Spec.ClusterID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			ObservedGeneration: 1,
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			Desired: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.CompletedUpdate, Image: "image/image:1", Version: "1.0.0-abc", StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.0-abc"},
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionFalse, Message: "Cluster version is 1.0.0-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
				{Type: DesiredReleaseAccepted, Status: configv1.ConditionTrue, Reason: "PayloadLoaded",
					Message: "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\""},
			},
		},
	})

	// Step 5: Wait for the SyncWorker to trigger a reconcile (500ms after the first)
	//
	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Reconciling: true,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			Generation: 1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Done:        1,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(1, 0),
			Generation:   1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(2, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Done:        2,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(2, 0),
			Generation:   1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(3, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Completed:   2,
			Done:        3,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(3, 0),
			Generation:   1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(4, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
	)

	// Step 6: After a reconciliation, there should be no status change because the state is the same
	//
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 1 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
}

func TestCVO_UpgradeUnverifiedPayload(t *testing.T) {
	o, cvs, client, _, shutdownFn := setupCVOTest("testdata/payloadtest-2")

	// Setup: a successful sync from a previous run, and the operator at the same image as before
	//
	o.release.Image = "image/image:0"
	o.release.Version = "1.0.0-abc"
	desired := configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"}
	uid, _ := uuid.NewRandom()
	clusterUID := configv1.ClusterID(uid.String())
	cvs["version"] = &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "1",
			Generation:      1,
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID:     clusterUID,
			Channel:       "fast",
			DesiredUpdate: &configv1.Update{Version: desired.Version, Image: desired.Image},
		},
		Status: configv1.ClusterVersionStatus{
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			Desired:     desired,
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.CompletedUpdate, Image: "image/image:0", Version: "1.0.0-abc", Verified: true, StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
		},
	}

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	defer shutdownFn()

	// make the image report unverified
	payloadErr := &payload.UpdateError{
		Reason:  "ImageVerificationFailed",
		Message: "The update cannot be verified: some random error",
		Nested:  fmt.Errorf("some random error"),
	}
	if !isImageVerificationError(payloadErr) {
		t.Fatal("not the correct error type")
	}
	worker := o.configSync.(*SyncWorker)
	retriever := worker.retriever.(*fakeDirectoryRetriever)
	retriever.Set(PayloadInfo{}, payloadErr)

	go worker.Start(ctx, 1, o.name, o.cvLister)

	// Step 1: The operator should report that it is blocked on unverified content
	//
	client.ClearActions()
	err := o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions := client.Actions()
	verifyCVSingleUpdate(t, actions)

	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Actual:     configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			Generation: 1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "RetrievePayload",
				Message:            "Retrieving and verifying payload version=\"1.0.1-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Actual:       configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			Generation:   1,
			LastProgress: time.Unix(1, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "RetrievePayload",
				Message:            "Retrieving payload failed version=\"1.0.1-abc\" image=\"image/image:1\" failure=The update cannot be verified: some random error",
				LastTransitionTime: time.Unix(2, 0),
				Release:            configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
				Failure:            payloadErr,
			},
		},
	)
	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	actual := cvs["version"].(*configv1.ClusterVersion)
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "1",
			Generation:      1,
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID:     clusterUID,
			Channel:       "fast",
			DesiredUpdate: &configv1.Update{Version: desired.Version, Image: desired.Image},
		},
		Status: configv1.ClusterVersionStatus{
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			ObservedGeneration: 1,
			Desired:            desired,
			VersionHash:        "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.PartialUpdate, Image: "image/image:1", Version: "1.0.1-abc", StartedTime: defaultStartedTime},
				{State: configv1.CompletedUpdate, Image: "image/image:0", Version: "1.0.0-abc", Verified: true, StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: DesiredReleaseAccepted, Status: configv1.ConditionFalse, Reason: "RetrievePayload",
					Message: "Retrieving payload failed version=\"1.0.1-abc\" image=\"image/image:1\" failure=The update cannot be verified: some random error"},
				{Type: "Available", Status: "False"},
				{Type: "Failing", Status: "False"},
				{Type: "Progressing", Status: "True", Message: "Working towards 1.0.1-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
			},
		},
	})

	// Step 2: Set allowUnverifiedImages to true, trigger a sync and the operator should apply the payload
	//
	// set an update
	copied := configv1.Update{
		Version: desired.Version,
		Image:   desired.Image,
		Force:   true,
	}
	actual.Spec.DesiredUpdate = &copied
	retriever.Set(PayloadInfo{Directory: "testdata/payloadtest-2", VerificationError: payloadErr}, nil)
	//
	// ensure the sync worker tells the sync loop about it
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}

	// wait until we see the new payload show up
	count := 0
	for {
		var status SyncWorkerStatus
		select {
		case status = <-worker.StatusCh():
		case <-time.After(3 * time.Second):
			t.Fatalf("never saw expected retrieve payload event")
		}
		if reflect.DeepEqual(configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"}, status.Actual) {
			break
		}
		t.Logf("Unexpected status waiting to see first retrieve: %#v", status)
		count++
		if count > 8 {
			t.Fatalf("saw too many sync events of the wrong form")
		}
	}
	// wait until the new payload is applied
	count = 0
	for {
		var status SyncWorkerStatus
		select {
		case status = <-worker.StatusCh():
		case <-time.After(3 * time.Second):
			t.Fatalf("never saw expected apply event")
		}
		if status.loadPayloadStatus.Step == "PayloadLoaded" {
			break
		}
		t.Log("Waiting to see step PayloadLoaded")
		count++
		if count > 8 {
			t.Fatalf("saw too many sync events of the wrong form")
		}
	}
	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version: "1.0.1-abc",
				Image:   "image/image:1",
				URL:     configv1.URL("https://example.com/v1.0.1-abc"),
			},
			Generation: 1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.1-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			},
		},
	)

	// wait for status to reflect sync of new payload
	waitForStatusCompleted(t, worker)

	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "3",
			Generation:      1,
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID:     actual.Spec.ClusterID,
			Channel:       "fast",
			DesiredUpdate: &copied,
		},
		Status: configv1.ClusterVersionStatus{
			ObservedGeneration: 1,
			Desired: configv1.Release{
				Version: "1.0.1-abc",
				Image:   "image/image:1",
				URL:     configv1.URL("https://example.com/v1.0.1-abc"),
			},
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.CompletedUpdate, Image: "image/image:1", Version: "1.0.1-abc", StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
				{State: configv1.CompletedUpdate, Image: "image/image:0", Version: "1.0.0-abc", Verified: true, StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: DesiredReleaseAccepted, Status: configv1.ConditionTrue, Reason: "PayloadLoaded",
					Message: "Payload loaded version=\"1.0.1-abc\" image=\"image/image:1\""},
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.1-abc"},
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionFalse, Message: "Cluster version is 1.0.1-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
			},
		},
	})
}

func TestCVO_UpgradeUnverifiedPayloadRetrieveOnce(t *testing.T) {
	o, cvs, client, _, shutdownFn := setupCVOTest("testdata/payloadtest-2")

	// Setup: a successful sync from a previous run, and the operator at the same image as before
	//
	o.release.Image = "image/image:0"
	o.release.Version = "1.0.0-abc"
	desired := configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"}
	uid, _ := uuid.NewRandom()
	clusterUID := configv1.ClusterID(uid.String())
	cvs["version"] = &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "1",
			Generation:      1,
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID:     clusterUID,
			Channel:       "fast",
			DesiredUpdate: &configv1.Update{Version: desired.Version, Image: desired.Image},
		},
		Status: configv1.ClusterVersionStatus{
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			Desired:     desired,
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.CompletedUpdate, Image: "image/image:0", Version: "1.0.0-abc", Verified: true, StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.0-abc"},
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionFalse, Message: "Cluster version is 1.0.0-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
			},
		},
	}

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	defer shutdownFn()

	// make the image report unverified
	payloadErr := &payload.UpdateError{
		Reason:  "ImageVerificationFailed",
		Message: "The update cannot be verified: some random error",
		Nested:  fmt.Errorf("some random error"),
	}
	if !isImageVerificationError(payloadErr) {
		t.Fatal("not the correct error type")
	}
	worker := o.configSync.(*SyncWorker)
	retriever := worker.retriever.(*fakeDirectoryRetriever)
	retriever.Set(PayloadInfo{}, payloadErr)

	go worker.Start(ctx, 1, o.name, o.cvLister)

	// Step 1: The operator should report that it is blocked on unverified content
	//
	client.ClearActions()
	err := o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions := client.Actions()
	verifyCVSingleUpdate(t, actions)

	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Actual:     configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			Generation: 1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "RetrievePayload",
				Message:            "Retrieving and verifying payload version=\"1.0.1-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Actual:       configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			Generation:   1,
			LastProgress: time.Unix(1, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "RetrievePayload",
				Message:            "Retrieving payload failed version=\"1.0.1-abc\" image=\"image/image:1\" failure=The update cannot be verified: some random error",
				LastTransitionTime: time.Unix(2, 0),
				Release:            configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
				Failure:            payloadErr,
			},
		},
	)
	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	actual := cvs["version"].(*configv1.ClusterVersion)
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "1",
			Generation:      1,
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID:     clusterUID,
			Channel:       "fast",
			DesiredUpdate: &configv1.Update{Version: desired.Version, Image: desired.Image},
		},
		Status: configv1.ClusterVersionStatus{
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			ObservedGeneration: 1,
			Desired:            desired,
			VersionHash:        "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.PartialUpdate, Image: "image/image:1", Version: "1.0.1-abc", StartedTime: defaultStartedTime},
				{State: configv1.CompletedUpdate, Image: "image/image:0", Version: "1.0.0-abc", Verified: true, StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.0-abc"},
				// cleared failing status and set progressing
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionTrue, Message: "Working towards 1.0.1-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
				{Type: DesiredReleaseAccepted, Status: configv1.ConditionFalse, Reason: "RetrievePayload",
					Message: "Retrieving payload failed version=\"1.0.1-abc\" image=\"image/image:1\" failure=The update cannot be verified: some random error"},
			},
		},
	})

	// Step 2: Set allowUnverifiedImages to true, trigger a sync and the operator should apply the payload
	//
	// set an update
	copied := configv1.Update{
		Version: desired.Version,
		Image:   desired.Image,
		Force:   true,
	}
	actual.Spec.DesiredUpdate = &copied
	retriever.Set(PayloadInfo{Directory: "testdata/payloadtest-2", VerificationError: payloadErr}, nil)
	//
	// ensure the sync worker tells the sync loop about it
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}

	// wait until we see the new payload show up
	count := 0
	for {
		var status SyncWorkerStatus
		select {
		case status = <-worker.StatusCh():
		case <-time.After(3 * time.Second):
			t.Fatalf("never saw expected retrieve payload event")
		}
		if reflect.DeepEqual(configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"}, status.Actual) {
			break
		}
		t.Logf("Unexpected status waiting to see first retrieve: %#v", status)
		count++
		if count > 8 {
			t.Fatalf("saw too many sync events of the wrong form")
		}
	}
	// wait until the new payload is applied
	count = 0
	for {
		var status SyncWorkerStatus
		select {
		case status = <-worker.StatusCh():
		case <-time.After(3 * time.Second):
			t.Fatalf("never saw expected apply event")
		}
		if status.loadPayloadStatus.Step == "PayloadLoaded" {
			break
		}
		t.Log("Waiting to see step PayloadLoaded")
		count++
		if count > 8 {
			t.Fatalf("saw too many sync events of the wrong form")
		}
	}
	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version: "1.0.1-abc",
				Image:   "image/image:1",
				URL:     configv1.URL("https://example.com/v1.0.1-abc"),
			},
			Generation: 1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.1-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			},
		},
	)

	// wait for status to reflect sync of new payload
	waitForStatusCompleted(t, worker)

	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")

	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "3",
			Generation:      1,
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID:     actual.Spec.ClusterID,
			Channel:       "fast",
			DesiredUpdate: &copied,
		},
		Status: configv1.ClusterVersionStatus{
			ObservedGeneration: 1,
			Desired: configv1.Release{
				Version: "1.0.1-abc",
				Image:   "image/image:1",
				URL:     configv1.URL("https://example.com/v1.0.1-abc"),
			},
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.CompletedUpdate, Image: "image/image:1", Version: "1.0.1-abc", StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
				{State: configv1.CompletedUpdate, Image: "image/image:0", Version: "1.0.0-abc", Verified: true, StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.1-abc"},
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionFalse, Message: "Cluster version is 1.0.1-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
				{Type: DesiredReleaseAccepted, Status: configv1.ConditionTrue, Reason: "PayloadLoaded",
					Message: "Payload loaded version=\"1.0.1-abc\" image=\"image/image:1\""},
			},
		},
	})

	// Step 5: Wait for the SyncWorker to trigger a reconcile (500ms after the first)
	//
	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Reconciling: true,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version: "1.0.1-abc",
				Image:   "image/image:1",
				URL:     configv1.URL("https://example.com/v1.0.1-abc"),
			},
			Generation: 1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.1-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Done:        1,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version: "1.0.1-abc",
				Image:   "image/image:1",
				URL:     configv1.URL("https://example.com/v1.0.1-abc"),
			},
			Generation:   1,
			LastProgress: time.Unix(1, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.1-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(2, 0),
				Release:            configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Done:        2,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version: "1.0.1-abc",
				Image:   "image/image:1",
				URL:     configv1.URL("https://example.com/v1.0.1-abc"),
			},
			Generation:   1,
			LastProgress: time.Unix(2, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.1-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(3, 0),
				Release:            configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Completed:   2,
			Done:        3,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version: "1.0.1-abc",
				Image:   "image/image:1",
				URL:     configv1.URL("https://example.com/v1.0.1-abc"),
			},
			LastProgress: time.Unix(3, 0),
			Generation:   1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.1-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(4, 0),
				Release:            configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			},
		},
	)
}

func TestCVO_UpgradePreconditionFailing(t *testing.T) {
	o, cvs, client, _, shutdownFn := setupCVOTest("testdata/payloadtest-2")

	// Setup: a successful sync from a previous run, and the operator at the same image as before
	//
	o.release.Image = "image/image:0"
	o.release.Version = "1.0.0-abc"
	desired := configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"}
	uid, _ := uuid.NewRandom()
	clusterUID := configv1.ClusterID(uid.String())
	cvs["version"] = &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "1",
			Generation:      1,
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID:     clusterUID,
			Channel:       "fast",
			DesiredUpdate: &configv1.Update{Version: desired.Version, Image: desired.Image},
		},
		Status: configv1.ClusterVersionStatus{
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			Desired:     desired,
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.CompletedUpdate, Image: "image/image:0", Version: "1.0.0-abc", Verified: true, StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.0-abc"},
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionFalse, Message: "Cluster version is 1.0.0-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
			},
		},
	}

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	defer shutdownFn()

	worker := o.configSync.(*SyncWorker)
	worker.preconditions = []precondition.Precondition{&testPrecondition{SuccessAfter: 3}}

	go worker.Start(ctx, 1, o.name, o.cvLister)

	// Step 1: The operator should report that it is blocked on precondition checks failing
	//
	client.ClearActions()
	err := o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions := client.Actions()
	verifyCVSingleUpdate(t, actions)

	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Actual:     configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			Generation: 1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "RetrievePayload",
				Message:            "Retrieving and verifying payload version=\"1.0.1-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Actual:       configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			Generation:   1,
			LastProgress: time.Unix(1, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PreconditionChecks",
				Message:            "Preconditions failed for payload loaded version=\"1.0.1-abc\" image=\"image/image:1\": Precondition \"TestPrecondition SuccessAfter: 3\" failed because of \"CheckFailure\": failing, attempt: 1 will succeed after 3 attempt",
				LastTransitionTime: time.Unix(2, 0),
				Release:            configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
				Failure:            &payload.UpdateError{Reason: "UpgradePreconditionCheckFailed", Message: "Precondition \"TestPrecondition SuccessAfter: 3\" failed because of \"CheckFailure\": failing, attempt: 1 will succeed after 3 attempt", Name: "PreconditionCheck"},
			},
		},
	)

	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	actual := cvs["version"].(*configv1.ClusterVersion)

	// Step 2: Set allowUnverifiedImages to true, trigger a sync and the operator should apply the payload
	//
	// set an update
	copied := configv1.Update{
		Version: desired.Version,
		Image:   desired.Image,
		Force:   true,
	}
	actual.Spec.DesiredUpdate = &copied
	//
	// ensure the sync worker tells the sync loop about it
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}

	// wait until we see the new payload show up
	count := 0
	for {
		var status SyncWorkerStatus
		select {
		case status = <-worker.StatusCh():
		case <-time.After(3 * time.Second):
			t.Fatalf("never saw expected retrieve payload event")
		}
		if reflect.DeepEqual(configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"}, status.Actual) {
			break
		}
		t.Logf("Unexpected status waiting to see first retrieve: %#v", status)
		count++
		if count > 8 {
			t.Fatalf("saw too many sync events of the wrong form")
		}
	}
	// wait until the new payload is applied
	count = 0
	for {
		var status SyncWorkerStatus
		select {
		case status = <-worker.StatusCh():
		case <-time.After(3 * time.Second):
			t.Fatalf("never saw expected apply event")
		}
		if status.loadPayloadStatus.Step == "PayloadLoaded" {
			break
		}
		t.Log("Waiting to see step PayloadLoaded")
		count++
		if count > 8 {
			t.Fatalf("saw too many sync events of the wrong form")
		}
	}
	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version: "1.0.1-abc",
				Image:   "image/image:1",
				URL:     configv1.URL("https://example.com/v1.0.1-abc"),
			},
			Generation: 1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.1-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Done:        1,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version: "1.0.1-abc",
				Image:   "image/image:1",
				URL:     configv1.URL("https://example.com/v1.0.1-abc"),
			},
			LastProgress: time.Unix(1, 0),
			Generation:   1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.1-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(2, 0),
				Release:            configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Done:        2,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version: "1.0.1-abc",
				Image:   "image/image:1",
				URL:     configv1.URL("https://example.com/v1.0.1-abc"),
			},
			LastProgress: time.Unix(2, 0),
			Generation:   1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.1-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(3, 0),
				Release:            configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"},
			},
		},
	)

	// wait for status to reflect sync of new payload
	waitForStatusCompleted(t, worker)

	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}

	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "4",
			Generation:      1,
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID:     actual.Spec.ClusterID,
			Channel:       "fast",
			DesiredUpdate: &copied,
		},
		Status: configv1.ClusterVersionStatus{
			ObservedGeneration: 1,
			Desired: configv1.Release{
				Version: "1.0.1-abc",
				Image:   "image/image:1",
				URL:     configv1.URL("https://example.com/v1.0.1-abc"),
			},
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.CompletedUpdate, Image: "image/image:1", Version: "1.0.1-abc", StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
				{State: configv1.CompletedUpdate, Image: "image/image:0", Version: "1.0.0-abc", Verified: true, StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.1-abc"},
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionFalse, Message: "Cluster version is 1.0.1-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
				{Type: DesiredReleaseAccepted, Status: configv1.ConditionTrue, Reason: "PayloadLoaded",
					Message: "Payload loaded version=\"1.0.1-abc\" image=\"image/image:1\""},
			},
		},
	})
}

func TestCVO_UpgradeVerifiedPayload(t *testing.T) {
	o, cvs, client, _, shutdownFn := setupCVOTest("testdata/payloadtest-2")

	// Setup: a successful sync from a previous run, and the operator at the same image as before
	//
	o.release.Image = "image/image:0"
	o.release.Version = "1.0.0-abc"
	desired := configv1.Release{Version: "1.0.1-abc", Image: "image/image:1"}
	uid, _ := uuid.NewRandom()
	clusterUID := configv1.ClusterID(uid.String())
	cvs["version"] = &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "1",
			Generation:      1,
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID:     clusterUID,
			Channel:       "fast",
			DesiredUpdate: &configv1.Update{Version: desired.Version, Image: desired.Image},
		},
		Status: configv1.ClusterVersionStatus{
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			Desired:     desired,
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.CompletedUpdate, Image: "image/image:0", Version: "1.0.0-abc", Verified: true, StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.0-abc"},
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionFalse, Message: "Cluster version is 1.0.0-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
			},
		},
	}

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	defer shutdownFn()

	// make the image report unverified
	payloadErr := &payload.UpdateError{
		Reason:  "ImageVerificationFailed",
		Message: "The update cannot be verified: some random error",
		Nested:  fmt.Errorf("some random error"),
	}
	if !isImageVerificationError(payloadErr) {
		t.Fatal("not the correct error type")
	}
	worker := o.configSync.(*SyncWorker)
	retriever := worker.retriever.(*fakeDirectoryRetriever)
	retriever.Set(PayloadInfo{}, payloadErr)
	retriever.Set(PayloadInfo{Directory: "testdata/payloadtest-2", Verified: true}, nil)

	go worker.Start(ctx, 1, o.name, o.cvLister)

	// Step 1: Simulate a verified payload being retrieved and ensure the operator sets verified
	//
	client.ClearActions()
	err := o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions := client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "1",
			Generation:      1,
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID:     clusterUID,
			Channel:       "fast",
			DesiredUpdate: &configv1.Update{Version: desired.Version, Image: desired.Image},
		},
		Status: configv1.ClusterVersionStatus{
			ObservedGeneration: 1,
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			Desired:     desired,
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.PartialUpdate, Image: "image/image:1", Version: "1.0.1-abc", StartedTime: defaultStartedTime},
				{State: configv1.CompletedUpdate, Image: "image/image:0", Version: "1.0.0-abc", Verified: true, StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.0-abc"},
				// cleared failing status and set progressing
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionTrue, Message: "Working towards 1.0.1-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
				{Type: DesiredReleaseAccepted, Status: configv1.ConditionTrue, Reason: "PayloadLoaded",
					Message: "Payload loaded version=\"1.0.1-abc\" image=\"image/image:1\""},
			},
		},
	})
}

func TestCVO_RestartAndReconcile(t *testing.T) {
	o, cvs, client, _, shutdownFn := setupCVOTest("testdata/payloadtest")

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	defer shutdownFn()
	worker := o.configSync.(*SyncWorker)

	// Setup: a successful sync from a previous run, and the operator at the same image as before
	//
	o.release.Image = "image/image:1"
	o.release.Version = "1.0.0-abc"
	o.release.URL = configv1.URL("https://example.com/v1.0.0-abc")
	o.release.Channels = []string{"channel-a", "channel-b", "channel-c"}
	uid, _ := uuid.NewRandom()
	clusterUID := configv1.ClusterID(uid.String())
	cvs["version"] = &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "1",
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: clusterUID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			Desired: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				// TODO: this is wrong, should be single partial entry
				{State: configv1.CompletedUpdate, Image: "image/image:1", Version: "1.0.0-abc", Verified: true, StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
				{State: configv1.PartialUpdate, StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
				{State: configv1.PartialUpdate, StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.0-abc"},
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionFalse, Message: "Cluster version is 1.0.0-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
			},
		},
	}

	// Step 1: The sync loop starts and triggers a sync, but does not update status
	//
	client.ClearActions()
	err := o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions := client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")

	// check the worker status is initially set to reconciling
	if status := worker.Status(); !status.Reconciling || status.Completed != 0 {
		t.Fatalf("The worker should be reconciling from the beginning: %#v", status)
	}
	if worker.work.State != payload.ReconcilingPayload {
		t.Fatalf("The worker should be reconciling: %v", worker.work)
	}

	// Step 2: Start the sync worker and verify the sequence of events, and then verify
	//         the status does not change
	//
	go worker.Start(ctx, 1, o.name, o.cvLister)
	//
	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Reconciling: true,
			Actual:      configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "RetrievePayload",
				Message:            "Retrieving and verifying payload version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Actual: configv1.Release{
				Version: "1.0.0-abc",
				Image:   "image/image:1",
			},
			LastProgress: time.Unix(1, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(2, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(2, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(3, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Done:        1,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(3, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(4, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Done:        2,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(4, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(5, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
	)
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 1 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")

	// Step 3: Wait until the next resync is triggered, and then verify that status does
	//         not change
	//
	verifyAllStatus(t, worker.StatusCh(),
		// note that the image is not retrieved a second time
		SyncWorkerStatus{
			Reconciling: true,
			Completed:   1,
			Done:        3,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(1, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(2, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(2, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Done:        1,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(3, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(3, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Done:        2,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(4, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(4, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
	)
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 1 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
}

func TestCVO_ErrorDuringReconcile(t *testing.T) {
	o, cvs, client, _, shutdownFn := setupCVOTest("testdata/payloadtest")

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	defer shutdownFn()
	worker := o.configSync.(*SyncWorker)
	b := newBlockingResourceBuilder()
	worker.builder = b

	// Setup: a successful sync from a previous run, and the operator at the same image as before
	//
	o.release.Image = "image/image:1"
	o.release.Version = "1.0.0-abc"
	o.release.URL = configv1.URL("https://example.com/v1.0.0-abc")
	o.release.Channels = []string{"channel-a", "channel-b", "channel-c"}
	uid, _ := uuid.NewRandom()
	clusterUID := configv1.ClusterID(uid.String())
	cvs["version"] = &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "1",
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: clusterUID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			Desired: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.CompletedUpdate, Image: "image/image:1", Version: "1.0.0-abc", Verified: true, StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.0-abc"},
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionFalse, Message: "Cluster version is 1.0.0-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
			},
		},
	}

	// Step 1: The sync loop starts and triggers a sync, but does not update status
	//
	client.ClearActions()
	err := o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions := client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")

	// check the worker status is initially set to reconciling
	if status := worker.Status(); !status.Reconciling || status.Completed != 0 {
		t.Fatalf("The worker should be reconciling from the beginning: %#v", status)
	}
	if worker.work.State != payload.ReconcilingPayload {
		t.Fatalf("The worker should be reconciling: %v", worker.work)
	}
	//
	go worker.Start(ctx, 1, o.name, o.cvLister)
	//
	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Reconciling: true,
			Actual:      configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "RetrievePayload",
				Message:            "Retrieving and verifying payload version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
		SyncWorkerStatus{
			Reconciling: true,
			Actual: configv1.Release{
				Version: "1.0.0-abc",
				Image:   "image/image:1",
			},
			LastProgress: time.Unix(1, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(2, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
	)
	// verify we haven't observed any other events
	verifyAllStatus(t, worker.StatusCh())

	// Step 2: Simulate a sync being triggered while we are partway through our first
	//         reconcile sync and verify status is not updated
	//
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}

	actions = client.Actions()
	if len(actions) != 1 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")

	// Step 3: Unblock the first item from being applied
	//
	b.Send(nil)
	//
	// verify we observe the remaining changes in the first sync
	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Reconciling: true,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
	)
	clearAllStatus(t, worker.StatusCh())

	// Step 4: Unblock the first item from being applied
	//
	b.Send(nil)
	//
	// Verify we observe the remaining changes in the first sync. Since timing is
	// non-deterministic, use this instead of verifyAllStatus when don't know or
	// care how many are done.
	verifyAllStatusOptionalDone(t, true, worker.StatusCh(),
		SyncWorkerStatus{
			Reconciling: true,
			Done:        2,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(1, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
	)
	clearAllStatus(t, worker.StatusCh())

	// Step 5: Send an error, then verify it shows up in status
	//
	b.Send(fmt.Errorf("unable to proceed"))

	go func() {
		for len(b.ch) != 0 {
			time.Sleep(time.Millisecond)
		}
		cancel()
		for len(b.ch) == 0 || len(worker.StatusCh()) == 0 {
			time.Sleep(time.Millisecond)
		}
	}()

	//
	// verify we see the update after the context times out
	verifyAllStatus(t, worker.StatusCh(),
		SyncWorkerStatus{
			Reconciling: true,
			Done:        2,
			Total:       3,
			VersionHash: "DL-FFQ2Uem8=",
			Failure: &payload.UpdateError{
				Nested:  fmt.Errorf("unable to proceed"),
				Reason:  "UpdatePayloadFailed",
				Message: "Could not update test \"file-yml\" (3 of 3)",
				Task:    &payload.Task{Index: 3, Total: 3, Manifest: &worker.payload.Manifests[2]},
			},
			Actual: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			LastProgress: time.Unix(1, 0),
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: time.Unix(1, 0),
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		},
	)
	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "2",
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: clusterUID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			Desired: configv1.Release{
				Version:  "1.0.0-abc",
				Image:    "image/image:1",
				URL:      configv1.URL("https://example.com/v1.0.0-abc"),
				Channels: []string{"channel-a", "channel-b", "channel-c"},
			},
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.CompletedUpdate, Image: "image/image:1", Version: "1.0.0-abc", Verified: true, StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.0-abc"},
				{Type: ClusterStatusFailing, Status: configv1.ConditionTrue, Reason: "UpdatePayloadFailed", Message: "Could not update test \"file-yml\" (3 of 3)"},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionFalse, Reason: "UpdatePayloadFailed", Message: "Error while reconciling 1.0.0-abc: the update could not be applied"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
				{Type: DesiredReleaseAccepted, Status: configv1.ConditionTrue, Reason: "PayloadLoaded",
					Message: "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\""},
			},
		},
	})
}

func TestCVO_ParallelError(t *testing.T) {
	o, cvs, client, _, shutdownFn := setupCVOTest("testdata/paralleltest")

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	defer shutdownFn()
	worker := o.configSync.(*SyncWorker)
	b := &errorResourceBuilder{errors: map[string]error{
		"0000_10_a_file.yaml": &payload.UpdateError{
			Reason:       "ClusterOperatorNotAvailable",
			UpdateEffect: payload.UpdateEffectNone,
			Name:         "operator-1",
		},
		"0000_20_a_file.yaml": nil,
		"0000_20_b_file.yaml": &payload.UpdateError{
			Reason:       "ClusterOperatorNotAvailable",
			UpdateEffect: payload.UpdateEffectNone,
			Name:         "operator-2",
		},
	}}
	worker.builder = b

	// Setup: an initializing cluster version which will run in parallel
	//
	o.release.Image = "image/image:1"
	o.release.Version = "1.0.0-abc"
	desired := configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"}
	uid, _ := uuid.NewRandom()
	clusterUID := configv1.ClusterID(uid.String())
	cvs["version"] = &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "1",
			Generation:      1,
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: clusterUID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			Desired:    desired,
			History:    []configv1.UpdateHistory{},
			Conditions: []configv1.ClusterOperatorStatusCondition{},
		},
	}

	// Step 1: Write initial status
	//
	client.ClearActions()
	err := o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions := client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")

	// check the worker status is initially set to reconciling
	if status := worker.Status(); status.Reconciling || status.Completed != 0 {
		t.Fatalf("The worker should be reconciling from the beginning: %#v", status)
	}
	if worker.work.State != payload.InitializingPayload {
		t.Fatalf("The worker should be reconciling: %v", worker.work)
	}

	// Step 2: Start the sync worker and wait for the payload to be loaded
	//
	cancellable, cancel := context.WithCancel(ctx)
	defer cancel()
	go worker.Start(cancellable, 1, o.name, o.cvLister)

	// wait until the new payload is applied
	count := 0
	for {
		var status SyncWorkerStatus
		select {
		case status = <-worker.StatusCh():
		case <-time.After(3 * time.Second):
			t.Fatalf("never saw expected apply event")
		}
		if status.loadPayloadStatus.Step == "PayloadLoaded" {
			break
		}
		t.Log("Waiting to see step PayloadLoaded")
		count++
		if count > 8 {
			t.Fatalf("saw too many sync events of the wrong form")
		}
	}

	// Step 3: Cancel after we've accumulated 2/3 errors
	//
	time.Sleep(100 * time.Millisecond)
	cancel()
	//
	// verify we observe the remaining changes in the first sync
	for status := range worker.StatusCh() {
		if status.Failure == nil {
			if status.Done == 0 || (status.Done == 1 && status.Total == 3) {
				if !reflect.DeepEqual(status, SyncWorkerStatus{
					Initial:      true,
					Done:         status.Done,
					Total:        3,
					VersionHash:  "Gyh2W6qcDO4=",
					Actual:       configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
					LastProgress: status.LastProgress,
					Generation:   1,
					loadPayloadStatus: LoadPayloadStatus{
						Step:               "PayloadLoaded",
						Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
						LastTransitionTime: status.loadPayloadStatus.LastTransitionTime,
						Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
					},
				}) {
					t.Fatalf("unexpected status: %v", status)
				}
			}
			continue
		}
		err := status.Failure
		uErr, ok := err.(*payload.UpdateError)
		if !ok || uErr.Reason != "ClusterOperatorsNotAvailable" || uErr.Message != "Some cluster operators are still updating: operator-1, operator-2" {
			t.Fatalf("unexpected error: %v", err)
		}
		if status.LastProgress.IsZero() {
			t.Fatalf("unexpected last progress: %v", status.LastProgress)
		}
		if !reflect.DeepEqual(status, SyncWorkerStatus{
			Initial:      true,
			Failure:      err,
			Done:         1,
			Total:        3,
			VersionHash:  "Gyh2W6qcDO4=",
			Actual:       configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			LastProgress: status.LastProgress,
			Generation:   1,
			loadPayloadStatus: LoadPayloadStatus{
				Step:               "PayloadLoaded",
				Message:            "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\"",
				LastTransitionTime: status.loadPayloadStatus.LastTransitionTime,
				Release:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			},
		}) {
			t.Fatalf("unexpected final: %v", status)
		}
		break
	}
	verifyAllStatus(t, worker.StatusCh())

	client.ClearActions()
	err = o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}
	actions = client.Actions()
	if len(actions) != 2 {
		t.Fatalf("%s", spew.Sdump(actions))
	}
	expectGet(t, actions[0], "clusterversions", "", "version")
	expectUpdateStatus(t, actions[1], "clusterversions", "", &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			Generation:      1,
			ResourceVersion: "2",
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: clusterUID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			ObservedGeneration: 1,
			Desired:            configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"},
			VersionHash:        "Gyh2W6qcDO4=",
			History: []configv1.UpdateHistory{
				{State: configv1.PartialUpdate, Image: "image/image:1", Version: "1.0.0-abc", StartedTime: defaultStartedTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: DesiredReleaseAccepted, Status: configv1.ConditionTrue, Reason: "PayloadLoaded",
					Message: "Payload loaded version=\"1.0.0-abc\" image=\"image/image:1\""},
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionFalse},
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionTrue, Reason: "ClusterOperatorsNotAvailable", Message: "Working towards 1.0.0-abc: 1 of 3 done (33% complete), waiting on operator-1, operator-2"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
			},
		},
	})
}

func TestCVO_VerifyInitializingPayloadState(t *testing.T) {
	ctx := context.Background()
	o, cvs, client, _, shutdownFn := setupCVOTest("testdata/payloadtest")
	stopCh := make(chan struct{})
	defer close(stopCh)
	defer shutdownFn()
	worker := o.configSync.(*SyncWorker)
	b := newBlockingResourceBuilder()
	worker.builder = b

	// Setup: a successful sync from a previous run, and the operator at the same image as before
	//
	o.release.Image = "image/image:1"
	o.release.Version = "1.0.0-abc"
	desired := configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"}
	uid, _ := uuid.NewRandom()
	clusterUID := configv1.ClusterID(uid.String())
	cvs["version"] = &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "1",
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: clusterUID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			Desired:     desired,
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.PartialUpdate, Image: "image/image:1", Version: "1.0.0-abc", StartedTime: defaultStartedTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.0-abc"},
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionFalse, Message: "Cluster version is 1.0.0-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
			},
		},
	}

	// Step 1: The sync loop starts and triggers a sync, but does not update status
	//
	client.ClearActions()
	err := o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}

	// check the worker status is initially set to reconciling
	if status := worker.Status(); status.Reconciling || status.Completed != 0 {
		t.Fatalf("The worker should be initializing from the beginning: %#v", status)
	}
	if worker.work.State != payload.InitializingPayload {
		t.Fatalf("The worker should be initializing: %v", worker.work)
	}
}

func TestCVO_VerifyUpdatingPayloadState(t *testing.T) {
	ctx := context.Background()
	o, cvs, client, _, shutdownFn := setupCVOTest("testdata/payloadtest")
	stopCh := make(chan struct{})
	defer close(stopCh)
	defer shutdownFn()
	worker := o.configSync.(*SyncWorker)
	b := newBlockingResourceBuilder()
	worker.builder = b

	// Setup: a successful sync from a previous run, and the operator at the same image as before
	//
	o.release.Image = "image/image:1"
	o.release.Version = "1.0.0-abc"
	desired := configv1.Release{Version: "1.0.0-abc", Image: "image/image:1"}
	uid, _ := uuid.NewRandom()
	clusterUID := configv1.ClusterID(uid.String())
	cvs["version"] = &configv1.ClusterVersion{
		ObjectMeta: metav1.ObjectMeta{
			Name:            "version",
			ResourceVersion: "1",
		},
		Spec: configv1.ClusterVersionSpec{
			ClusterID: clusterUID,
			Channel:   "fast",
		},
		Status: configv1.ClusterVersionStatus{
			// Prefers the image version over the operator's version (although in general they will remain in sync)
			Desired:     desired,
			VersionHash: "DL-FFQ2Uem8=",
			History: []configv1.UpdateHistory{
				{State: configv1.PartialUpdate, Image: "image/image:1", Version: "1.0.0-abc", StartedTime: defaultStartedTime},
				{State: configv1.CompletedUpdate, Image: "image/image:0", Version: "1.0.0-abc.0", StartedTime: defaultStartedTime, CompletionTime: &defaultCompletionTime},
			},
			Conditions: []configv1.ClusterOperatorStatusCondition{
				{Type: configv1.OperatorAvailable, Status: configv1.ConditionTrue, Message: "Done applying 1.0.0-abc"},
				{Type: ClusterStatusFailing, Status: configv1.ConditionFalse},
				{Type: configv1.OperatorProgressing, Status: configv1.ConditionFalse, Message: "Cluster version is 1.0.0-abc"},
				{Type: configv1.RetrievedUpdates, Status: configv1.ConditionFalse},
			},
		},
	}

	// Step 1: The sync loop starts and triggers a sync, but does not update status
	//
	client.ClearActions()
	err := o.sync(ctx, o.queueKey())
	if err != nil {
		t.Fatal(err)
	}

	// check the worker status is initially set to reconciling
	if status := worker.Status(); status.Reconciling || status.Completed != 0 {
		t.Fatalf("The worker should be updating from the beginning: %#v", status)
	}
	if worker.work.State != payload.UpdatingPayload {
		t.Fatalf("The worker should be updating: %v", worker.work)
	}
}

// verifyCVSingleUpdate ensures that the only object to be updated is a ClusterVersion type and it is updated only once
func verifyCVSingleUpdate(t *testing.T, actions []clientgotesting.Action) {
	var count int
	for _, a := range actions {
		if a.GetResource().Resource != "clusterversions" {
			t.Fatalf("found an action which accesses/updates resource other than clusterversion: %#v", a)
		}
		if a.GetVerb() != "get" {
			count++
		}
	}
	if count != 1 {
		t.Fatalf("Expected only single update to clusterversion resource. Actual update count %d", count)
	}
}

func verifyAllStatus(t *testing.T, ch <-chan SyncWorkerStatus, items ...SyncWorkerStatus) {
	verifyAllStatusOptionalDone(t, false, ch, items...)
}

// Since timing can be non-deterministic, use this instead of verifyAllStatus when
// don't know or care how many are done.
func verifyAllStatusOptionalDone(t *testing.T, ignoreDone bool, ch <-chan SyncWorkerStatus, items ...SyncWorkerStatus) {
	t.Helper()
	if len(items) == 0 {
		if len(ch) > 0 {
			t.Fatalf("expected status to be empty, got %#v", <-ch)
		}
		return
	}
	var lastTime time.Time
	count := int64(1)
	count2 := int64(1)
	for i, expect := range items {
		actual, ok := <-ch
		if !ok {
			t.Fatalf("channel closed after reading only %d items", i)
		}

		if nextTime := actual.LastProgress; !nextTime.Equal(lastTime) {
			actual.LastProgress = time.Unix(count, 0)
			count++
		} else if !lastTime.IsZero() {
			actual.LastProgress = time.Unix(count, 0)
		}

		lastTime = time.Unix(0, 0)
		if nextTime := actual.loadPayloadStatus.LastTransitionTime; !nextTime.Equal(lastTime) {
			actual.loadPayloadStatus.LastTransitionTime = time.Unix(count2, 0)
			count2++
		} else if !lastTime.IsZero() {
			actual.loadPayloadStatus.LastTransitionTime = time.Unix(count2, 0)
		}
		if ignoreDone {
			expect.Done = actual.Done
		}

		if !reflect.DeepEqual(expect, actual) {
			t.Fatalf("unexpected status item %d\nExpected: %#v\nActual: %#v", i, expect, actual)
		}
	}
}

// blockingResourceBuilder controls how quickly Apply() is executed and allows error
// injection.
type blockingResourceBuilder struct {
	ch chan error
}

func newBlockingResourceBuilder() *blockingResourceBuilder {
	return &blockingResourceBuilder{
		ch: make(chan error),
	}
}

func (b *blockingResourceBuilder) Send(err error) {
	b.ch <- err
}

func (b *blockingResourceBuilder) Apply(ctx context.Context, m *manifest.Manifest, state payload.State) error {
	return <-b.ch
}

type errorResourceBuilder struct {
	errors map[string]error
}

func (b *errorResourceBuilder) Apply(ctx context.Context, m *manifest.Manifest, state payload.State) error {
	if err, ok := b.errors[m.OriginalFilename]; ok {
		return err
	}
	return fmt.Errorf("unknown file %s", m.OriginalFilename)
}

// wait for status completed
func waitForStatusCompleted(t *testing.T, worker *SyncWorker) {
	count := 0
	for {
		var status SyncWorkerStatus
		select {
		case status = <-worker.StatusCh():
		case <-time.After(3 * time.Second):
			t.Fatalf("never saw status Completed > 0")
		}
		if status.Completed > 0 {
			break
		}
		t.Log("Waiting for Completed > 0")
		count++
		if count > 8 {
			t.Fatalf("saw too many sync events of the wrong form")
		}
	}
}

func clearAllStatus(t *testing.T, ch <-chan SyncWorkerStatus) {
	count := 0
	for {
		if len(ch) <= 0 {
			break
		}
		<-ch
		t.Log("Waiting for SyncWorkerStatus to clear")
		count++
		if count > 8 {
			t.Fatalf("Waited too long for SyncWorkerStatus to clear")
		}
	}
}
