package xmlenc

import (
	"crypto/x509"
	"encoding/pem"
	"fmt"
	"math/rand"
	"strings"
	"testing"

	"github.com/beevik/etree"
	"github.com/kr/pretty"
	"github.com/stretchr/testify/assert"
)

const (
	testCertificate    = "-----BEGIN CERTIFICATE-----\nMIIB7zCCAVgCCQDFzbKIp7b3MTANBgkqhkiG9w0BAQUFADA8MQswCQYDVQQGEwJV\nUzELMAkGA1UECAwCR0ExDDAKBgNVBAoMA2ZvbzESMBAGA1UEAwwJbG9jYWxob3N0\nMB4XDTEzMTAwMjAwMDg1MVoXDTE0MTAwMjAwMDg1MVowPDELMAkGA1UEBhMCVVMx\nCzAJBgNVBAgMAkdBMQwwCgYDVQQKDANmb28xEjAQBgNVBAMMCWxvY2FsaG9zdDCB\nnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEA1PMHYmhZj308kWLhZVT4vOulqx/9\nibm5B86fPWwUKKQ2i12MYtz07tzukPymisTDhQaqyJ8Kqb/6JjhmeMnEOdTvSPmH\nO8m1ZVveJU6NoKRn/mP/BD7FW52WhbrUXLSeHVSKfWkNk6S4hk9MV9TswTvyRIKv\nRsw0X/gfnqkroJcCAwEAATANBgkqhkiG9w0BAQUFAAOBgQCMMlIO+GNcGekevKgk\nakpMdAqJfs24maGb90DvTLbRZRD7Xvn1MnVBBS9hzlXiFLYOInXACMW5gcoRFfeT\nQLSouMM8o57h0uKjfTmuoWHLQLi6hnF+cvCsEFiJZ4AbF+DgmO6TarJ8O05t8zvn\nOwJlNCASPZRH/JmF8tX0hoHuAQ==\n-----END CERTIFICATE-----\n"
	expectedCiphertext = `<xenc:EncryptedData xmlns:xenc="http://www.w3.org/2001/04/xmlenc#" Id="_e285ece1511455780875d64ee2d3d0d0">
	<xenc:EncryptionMethod Algorithm="http://www.w3.org/2001/04/xmlenc#aes128-cbc" xmlns:xenc="http://www.w3.org/2001/04/xmlenc#"/>
	<ds:KeyInfo xmlns:ds="http://www.w3.org/2000/09/xmldsig#">
		<xenc:EncryptedKey Id="_6e4ff95ff662a5eee82abdf44a2d0b75" xmlns:xenc="http://www.w3.org/2001/04/xmlenc#">
			<xenc:EncryptionMethod Algorithm="http://www.w3.org/2001/04/xmlenc#rsa-oaep-mgf1p" xmlns:xenc="http://www.w3.org/2001/04/xmlenc#">
				<ds:DigestMethod Algorithm="http://www.w3.org/2000/09/xmldsig#sha1" xmlns:ds="http://www.w3.org/2000/09/xmldsig#"/>
			</xenc:EncryptionMethod>
			<ds:KeyInfo>
				<ds:X509Data>
					<ds:X509Certificate>MIIB7zCCAVgCCQDFzbKIp7b3MTANBgkqhkiG9w0BAQUFADA8MQswCQYDVQQGEwJVUzELMAkGA1UECAwCR0ExDDAKBgNVBAoMA2ZvbzESMBAGA1UEAwwJbG9jYWxob3N0MB4XDTEzMTAwMjAwMDg1MVoXDTE0MTAwMjAwMDg1MVowPDELMAkGA1UEBhMCVVMxCzAJBgNVBAgMAkdBMQwwCgYDVQQKDANmb28xEjAQBgNVBAMMCWxvY2FsaG9zdDCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEA1PMHYmhZj308kWLhZVT4vOulqx/9ibm5B86fPWwUKKQ2i12MYtz07tzukPymisTDhQaqyJ8Kqb/6JjhmeMnEOdTvSPmHO8m1ZVveJU6NoKRn/mP/BD7FW52WhbrUXLSeHVSKfWkNk6S4hk9MV9TswTvyRIKvRsw0X/gfnqkroJcCAwEAATANBgkqhkiG9w0BAQUFAAOBgQCMMlIO+GNcGekevKgkakpMdAqJfs24maGb90DvTLbRZRD7Xvn1MnVBBS9hzlXiFLYOInXACMW5gcoRFfeTQLSouMM8o57h0uKjfTmuoWHLQLi6hnF+cvCsEFiJZ4AbF+DgmO6TarJ8O05t8zvnOwJlNCASPZRH/JmF8tX0hoHuAQ==</ds:X509Certificate>
				</ds:X509Data>
			</ds:KeyInfo>
			<xenc:CipherData xmlns:xenc="http://www.w3.org/2001/04/xmlenc#">
				<xenc:CipherValue>R9aHQv2U2ZZSuvRaL4/X8TXpm2/1so2IiOz/+NsAzEKoLAg8Sj87Nj5oMrYY2HF5DPQm/N/3+v6wOU9dX62spTzoSWocVzQU+GdTG2DiIIiAAvQwZo1FyUDKS1Fs5voWzgKvs8G43nj68147T96sXY9SyeUBBdhQtXRsEsmKiAs=</xenc:CipherValue>
			</xenc:CipherData>
		</xenc:EncryptedKey>
	</ds:KeyInfo>
	<xenc:CipherData xmlns:xenc="http://www.w3.org/2001/04/xmlenc#">
		<xenc:CipherValue>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</xenc:CipherValue>
	</xenc:CipherData>
</xenc:EncryptedData>
`
)

func TestCanEncryptOAEP(t *testing.T) {
	RandReader = rand.New(rand.NewSource(0)) // deterministic random numbers for tests

	pemBlock, _ := pem.Decode([]byte(testCertificate))
	certificate, err := x509.ParseCertificate(pemBlock.Bytes)
	assert.NoError(t, err)

	e := OAEP()
	e.BlockCipher = AES128CBC
	e.DigestMethod = &SHA1

	el, err := e.Encrypt(certificate, []byte(expectedPlaintext))
	assert.NoError(t, err)

	doc := etree.NewDocument()
	doc.SetRoot(el)
	doc.IndentTabs()
	ciphertext, _ := doc.WriteToString()

	//f, _ := os.Create("ciphertext.actual")
	//f.Write([]byte(ciphertext))

	diff := pretty.Diff(strings.Split(ciphertext, "\n"), strings.Split(expectedCiphertext, "\n"))
	for _, l := range diff {
		fmt.Println(l)
	}
	assert.Equal(t, expectedCiphertext, ciphertext)
}
