/*
Copyright 2016 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package logsgen

import (
	"fmt"
	"time"

	"github.com/spf13/cobra"

	"k8s.io/apimachinery/pkg/util/rand"
	"k8s.io/klog"
)

var (
	httpMethods = []string{
		"GET",
		"POST",
		"PUT",
	}
	namespaces = []string{
		"kube-system",
		"default",
		"ns",
	}
)

// CmdLogsGenerator is used by agnhost Cobra.
var CmdLogsGenerator = &cobra.Command{
	Use:   "logs-generator",
	Short: "Outputs lines of logs to stdout uniformly",
	Long:  "Outputs <linesTotal> lines of logs to stdout uniformly for <duration>",
	Args:  cobra.MaximumNArgs(0),
	Run:   generateLogs,
}

var (
	linesTotal int
	duration   time.Duration
)

func init() {
	CmdLogsGenerator.Flags().IntVarP(&linesTotal, "log-lines-total", "t", 0, "Total lines that should be generated by the end of the run")
	CmdLogsGenerator.Flags().DurationVarP(&duration, "run-duration", "d", 0, "Total duration of the run")
}

// Outputs linesTotal lines of logs to stdout uniformly for duration
func generateLogs(cmd *cobra.Command, args []string) {
	if linesTotal <= 0 {
		klog.Fatalf("Invalid total number of lines: %d", linesTotal)
	}

	if duration <= 0 {
		klog.Fatalf("Invalid duration: %v", duration)
	}

	delay := duration / time.Duration(linesTotal)

	ticker := time.NewTicker(delay)
	defer ticker.Stop()
	for id := 0; id < linesTotal; id++ {
		klog.Info(generateLogLine(id))
		<-ticker.C
	}
}

// Generates apiserver-like line with average length of 100 symbols
func generateLogLine(id int) string {
	method := httpMethods[rand.Intn(len(httpMethods))]
	namespace := namespaces[rand.Intn(len(namespaces))]

	podName := rand.String(rand.IntnRange(3, 5))
	url := fmt.Sprintf("/api/v1/namespaces/%s/pods/%s", namespace, podName)
	status := rand.IntnRange(200, 600)

	return fmt.Sprintf("%d %s %s %d", id, method, url, status)
}
