import * as React from 'react';
import * as _ from 'lodash';
import { useTranslation } from 'react-i18next';
import { Formik } from 'formik';
import { k8sUpdate, K8sResourceKind } from '@console/internal/module/k8s';
import { PipelineModel } from '../../../models';
import { removeEmptyDefaultFromPipelineParams } from './utils';

export interface PipelineFormProps {
  PipelineFormComponent: React.ComponentType<any>;
  formName: string;
  validationSchema: any;
  obj: K8sResourceKind;
}

const PipelineForm: React.FC<PipelineFormProps> = ({
  PipelineFormComponent,
  formName,
  validationSchema,
  obj,
}) => {
  const { t } = useTranslation();
  const initialValues = {
    parameters: _.get(obj.spec, 'params', []),
    resources: _.get(obj.spec, 'resources', []),
  };

  const handleSubmit = (values, actions) => {
    actions.setSubmitting(true);

    k8sUpdate(
      PipelineModel,
      {
        ...obj,
        spec: {
          ...obj.spec,
          params: removeEmptyDefaultFromPipelineParams(values.parameters),
          resources: values.resources,
        },
      },
      obj.metadata.namespace,
      obj.metadata.name,
    )
      .then((newObj) => {
        actions.setSubmitting(false);
        actions.resetForm({
          values: {
            parameters: _.get(newObj.spec, 'params', []),
            resources: _.get(newObj.spec, 'resources', []),
          },
          status: {
            success: t('pipelines-plugin~Successfully updated the pipeline {{formName}}.', {
              formName,
            }),
          },
        });
      })
      .catch((err) => {
        actions.setSubmitting(false);
        actions.setStatus({ submitError: err.message });
      });
  };

  const handleReset = (values, actions) => {
    actions.resetForm({ status: {} });
  };

  return (
    <div className="co-m-pane__body">
      <Formik
        initialValues={initialValues}
        onSubmit={handleSubmit}
        onReset={handleReset}
        validationSchema={validationSchema}
      >
        {(formikProps) => (
          <PipelineFormComponent namespace={obj.metadata.namespace} {...formikProps} />
        )}
      </Formik>
    </div>
  );
};

export default PipelineForm;
