package validation

import (
	"testing"

	"github.com/devfile/api/v2/pkg/attributes"

	"github.com/devfile/api/v2/pkg/apis/workspaces/v1alpha2"
	"github.com/hashicorp/go-multierror"
	"github.com/stretchr/testify/assert"
)

// generateDummyContainerComponent returns a dummy container component for testing
func generateDummyContainerComponent(name string, volMounts []v1alpha2.VolumeMount, endpoints []v1alpha2.Endpoint, envs []v1alpha2.EnvVar) v1alpha2.Component {
	image := "docker.io/maven:latest"
	mountSources := true

	return v1alpha2.Component{
		Name: name,
		ComponentUnion: v1alpha2.ComponentUnion{
			Container: &v1alpha2.ContainerComponent{
				Container: v1alpha2.Container{
					Image:        image,
					Env:          envs,
					VolumeMounts: volMounts,
					MountSources: &mountSources,
				},
				Endpoints: endpoints,
			}}}
}

// generateDummyVolumeComponent returns a dummy volume component for testing
func generateDummyVolumeComponent(name, size string) v1alpha2.Component {

	return v1alpha2.Component{

		Name: name,
		ComponentUnion: v1alpha2.ComponentUnion{
			Volume: &v1alpha2.VolumeComponent{
				Volume: v1alpha2.Volume{
					Size: size,
				},
			},
		},
	}
}

// generateDummyOpenshiftComponent returns a dummy Openshift component for testing
func generateDummyOpenshiftComponent(name string, endpoints []v1alpha2.Endpoint, uri string) v1alpha2.Component {

	return v1alpha2.Component{
		Name: name,
		ComponentUnion: v1alpha2.ComponentUnion{
			Openshift: &v1alpha2.OpenshiftComponent{
				K8sLikeComponent: v1alpha2.K8sLikeComponent{
					K8sLikeComponentLocation: v1alpha2.K8sLikeComponentLocation{
						Uri: uri,
					},
					Endpoints: endpoints,
				},
			},
		},
	}
}

// generateDummyKubernetesComponent returns a dummy Kubernetes component for testing
func generateDummyKubernetesComponent(name string, endpoints []v1alpha2.Endpoint, uri string) v1alpha2.Component {

	return v1alpha2.Component{
		Name: name,
		ComponentUnion: v1alpha2.ComponentUnion{
			Kubernetes: &v1alpha2.KubernetesComponent{
				K8sLikeComponent: v1alpha2.K8sLikeComponent{
					K8sLikeComponentLocation: v1alpha2.K8sLikeComponentLocation{
						Uri: uri,
					},
					Endpoints: endpoints,
				},
			},
		},
	}
}

// generateDummyImageComponent returns a dummy Image Dockerfile Component for testing
func generateDummyImageComponent(name string, src v1alpha2.DockerfileSrc) v1alpha2.Component {

	return v1alpha2.Component{
		Name: name,
		ComponentUnion: v1alpha2.ComponentUnion{
			Image: &v1alpha2.ImageComponent{
				Image: v1alpha2.Image{
					ImageName: "image:latest",
					ImageUnion: v1alpha2.ImageUnion{
						Dockerfile: &v1alpha2.DockerfileImage{
							DockerfileSrc: src,
							Dockerfile: v1alpha2.Dockerfile{
								BuildContext: "/path",
							},
						},
					},
				},
			},
		},
	}
}

// generateDummyPluginComponent returns a dummy Plugin component for testing
func generateDummyPluginComponent(name, url string, compAttribute attributes.Attributes) v1alpha2.Component {

	return v1alpha2.Component{
		Attributes: compAttribute,
		Name:       name,
		ComponentUnion: v1alpha2.ComponentUnion{
			Plugin: &v1alpha2.PluginComponent{
				ImportReference: v1alpha2.ImportReference{
					RegistryUrl: url,
				},
			},
		},
	}
}

func TestValidateComponents(t *testing.T) {

	volMounts := []v1alpha2.VolumeMount{
		{
			Name: "myvol",
			Path: "/some/path/",
		},
	}

	invalidVolMounts := []v1alpha2.VolumeMount{
		{
			Name: "myinvalidvol",
		},
		{
			Name: "myinvalidvol2",
		},
	}

	projectSourceEnv := []v1alpha2.EnvVar{
		{
			Name:  EnvProjectsSrc,
			Value: "/some/path/",
		},
	}

	projectsRootEnv := []v1alpha2.EnvVar{
		{
			Name:  EnvProjectsRoot,
			Value: "/some/path/",
		},
	}

	twoRemotesGitSrc := v1alpha2.DockerfileSrc{
		Git: &v1alpha2.DockerfileGitProjectSource{
			GitProjectSource: v1alpha2.GitProjectSource{
				GitLikeProjectSource: v1alpha2.GitLikeProjectSource{
					Remotes: map[string]string{
						"a": "abc",
						"x": "xyz",
					},
					CheckoutFrom: &v1alpha2.CheckoutFrom{
						Remote: "a",
					},
				},
			},
		},
	}

	zeroRemoteGitSrc := v1alpha2.DockerfileSrc{
		Git: &v1alpha2.DockerfileGitProjectSource{
			GitProjectSource: v1alpha2.GitProjectSource{
				GitLikeProjectSource: v1alpha2.GitLikeProjectSource{
					CheckoutFrom: &v1alpha2.CheckoutFrom{
						Remote: "a",
					},
				},
			},
		},
	}

	invalidRemoteGitSrc := v1alpha2.DockerfileSrc{
		Git: &v1alpha2.DockerfileGitProjectSource{
			GitProjectSource: v1alpha2.GitProjectSource{
				GitLikeProjectSource: v1alpha2.GitLikeProjectSource{
					Remotes: map[string]string{
						"a": "abc",
					},
					CheckoutFrom: &v1alpha2.CheckoutFrom{
						Remote: "b",
					},
				},
			},
		},
	}

	validRemoteGitSrc := v1alpha2.DockerfileSrc{
		Git: &v1alpha2.DockerfileGitProjectSource{
			GitProjectSource: v1alpha2.GitProjectSource{
				GitLikeProjectSource: v1alpha2.GitLikeProjectSource{
					Remotes: map[string]string{
						"a": "abc",
					},
					CheckoutFrom: &v1alpha2.CheckoutFrom{
						Remote: "a",
					},
				},
			},
		},
	}

	validUriSrc := v1alpha2.DockerfileSrc{
		Uri: "uri",
	}

	endpointUrl18080 := generateDummyEndpoint("url1", 8080)
	endpointUrl18081 := generateDummyEndpoint("url1", 8081)
	endpointUrl28080 := generateDummyEndpoint("url2", 8080)

	invalidVolMountErr := ".*\nvolume mount myinvalidvol belonging to the container component.*\nvolume mount myinvalidvol2 belonging to the container component.*"
	duplicateComponentErr := "duplicate key: component1"
	reservedEnvErr := "env variable .* is reserved and cannot be customized in component.*"
	invalidSizeErr := "size .* for volume component is invalid"
	sameEndpointNameErr := "devfile contains multiple endpoint entries with same name.*"
	sameTargetPortErr := "devfile contains multiple containers with same TargetPort.*"
	invalidURIErr := ".*invalid URI for request"
	imageCompTwoRemoteErr := "component .* should have one remote only"
	imageCompNoRemoteErr := "component .* should have at least one remote"
	imageCompInvalidRemoteErr := "unable to find the checkout remote .* in the remotes for component .*"

	pluginOverridesFromMainDevfile := attributes.Attributes{}.PutString(ImportSourceAttribute,
		"uri: http://127.0.0.1:8080").PutString(PluginOverrideAttribute, "main devfile")
	invalidURIErrWithImportAttributes := ".*invalid URI for request, imported from uri: http://127.0.0.1:8080, in plugin overrides from main devfile"

	tests := []struct {
		name       string
		components []v1alpha2.Component
		wantErr    []string
	}{
		{
			name: "Duplicate components present",
			components: []v1alpha2.Component{
				generateDummyVolumeComponent("component1", "1Gi"),
				generateDummyContainerComponent("component1", nil, nil, nil),
			},
			wantErr: []string{duplicateComponentErr},
		},
		{
			name: "Valid container and volume component",
			components: []v1alpha2.Component{
				generateDummyVolumeComponent("myvol", "1Gi"),
				generateDummyContainerComponent("container", volMounts, nil, nil),
				generateDummyContainerComponent("container2", volMounts, nil, nil),
			},
		},
		{
			name: "Invalid container using reserved env PROJECT_SOURCE",
			components: []v1alpha2.Component{
				generateDummyContainerComponent("container1", nil, nil, projectSourceEnv),
			},
			wantErr: []string{reservedEnvErr},
		},
		{
			name: "Invalid container using reserved env PROJECTS_ROOT",
			components: []v1alpha2.Component{
				generateDummyContainerComponent("container", nil, nil, projectsRootEnv),
			},
			wantErr: []string{reservedEnvErr},
		},
		{
			name: "Invalid volume component size",
			components: []v1alpha2.Component{
				generateDummyVolumeComponent("myvol", "invalid"),
				generateDummyContainerComponent("container", nil, nil, nil),
			},
			wantErr: []string{invalidSizeErr},
		},
		{
			name: "Invalid volume mount referencing a wrong volume component",
			components: []v1alpha2.Component{
				generateDummyVolumeComponent("myvol", "1Gi"),
				generateDummyContainerComponent("container1", invalidVolMounts, nil, nil),
			},
			wantErr: []string{invalidVolMountErr},
		},
		{
			name: "Invalid containers with the same endpoint names",
			components: []v1alpha2.Component{
				generateDummyContainerComponent("name1", nil, []v1alpha2.Endpoint{endpointUrl18080}, nil),
				generateDummyContainerComponent("name2", nil, []v1alpha2.Endpoint{endpointUrl18081}, nil),
			},
			wantErr: []string{sameEndpointNameErr},
		},
		{
			name: "Invalid containers with the same endpoint target ports",
			components: []v1alpha2.Component{
				generateDummyContainerComponent("name1", nil, []v1alpha2.Endpoint{endpointUrl18080}, nil),
				generateDummyContainerComponent("name2", nil, []v1alpha2.Endpoint{endpointUrl28080}, nil),
			},
			wantErr: []string{sameTargetPortErr},
		},
		{
			name: "Valid container with same target ports but different endpoint name",
			components: []v1alpha2.Component{
				generateDummyContainerComponent("name1", nil, []v1alpha2.Endpoint{endpointUrl18080, endpointUrl28080}, nil),
			},
		},
		{
			name: "Invalid Openshift Component with bad URI",
			components: []v1alpha2.Component{
				generateDummyOpenshiftComponent("name1", []v1alpha2.Endpoint{endpointUrl18080, endpointUrl28080}, "http//wronguri"),
			},
			wantErr: []string{invalidURIErr},
		},
		{
			name: "Valid Kubernetes Component",
			components: []v1alpha2.Component{
				generateDummyKubernetesComponent("name1", []v1alpha2.Endpoint{endpointUrl18080, endpointUrl28080}, "http://uri"),
			},
		},
		{
			name: "Invalid OpenShift Component with same endpoint names",
			components: []v1alpha2.Component{
				generateDummyOpenshiftComponent("name1", []v1alpha2.Endpoint{endpointUrl18080, endpointUrl18081}, "http://uri"),
			},
			wantErr: []string{sameEndpointNameErr},
		},
		{
			name: "Multiple errors: Duplicate component name, invalid plugin registry url, bad URI with import source attributes",
			components: []v1alpha2.Component{
				generateDummyVolumeComponent("component1", "1Gi"),
				generateDummyPluginComponent("component1", "http//invalidregistryurl", attributes.Attributes{}),
				generateDummyPluginComponent("abc", "http//invalidregistryurl", pluginOverridesFromMainDevfile),
			},
			wantErr: []string{duplicateComponentErr, invalidURIErr, invalidURIErrWithImportAttributes},
		},
		{
			name: "Invalid image dockerfile component with more than one remote",
			components: []v1alpha2.Component{
				generateDummyImageComponent("name1", twoRemotesGitSrc),
			},
			wantErr: []string{imageCompTwoRemoteErr},
		},
		{
			name: "Invalid image dockerfile component with zero remote",
			components: []v1alpha2.Component{
				generateDummyImageComponent("name1", zeroRemoteGitSrc),
			},
			wantErr: []string{imageCompNoRemoteErr},
		},
		{
			name: "Invalid image dockerfile component with wrong checkout",
			components: []v1alpha2.Component{
				generateDummyImageComponent("name1", invalidRemoteGitSrc),
			},
			wantErr: []string{imageCompInvalidRemoteErr},
		},
		{
			name: "Valid image dockerfile component with correct remote",
			components: []v1alpha2.Component{
				generateDummyImageComponent("name1", validRemoteGitSrc),
			},
		},
		{
			name: "Valid image dockerfile component with non git src",
			components: []v1alpha2.Component{
				generateDummyImageComponent("name1", validUriSrc),
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			err := ValidateComponents(tt.components)

			if merr, ok := err.(*multierror.Error); ok && tt.wantErr != nil {
				assert.Equal(t, len(tt.wantErr), len(merr.Errors), "Error list length should match")
				for i := 0; i < len(merr.Errors); i++ {
					assert.Regexp(t, tt.wantErr[i], merr.Errors[i].Error(), "Error message should match")
				}
			} else {
				assert.Equal(t, nil, err, "Error should be nil")
			}
		})
	}

}
