package compatibility

import (
	"fmt"
	"io"
	"path"
	"regexp"
	"strings"

	"k8s.io/gengo/generator"
	"k8s.io/gengo/namer"
	"k8s.io/gengo/types"
	"k8s.io/klog/v2"
)

// NewCompatibilityGen creates the openshift compatibility generator.
func NewCompatibilityGen(sanitizedName, targetPackage string) generator.Generator {
	return &genCompatibility{
		DefaultGen: generator.DefaultGen{
			OptionalName: sanitizedName,
		},
		targetPackage: targetPackage,
		imports:       generator.NewImportTracker(),
		typesForInit:  make([]*types.Type, 0),
	}
}

// genCompatibility produces a file with autogenerated compatibility functions.
type genCompatibility struct {
	generator.DefaultGen
	targetPackage string
	imports       namer.ImportTracker
	typesForInit  []*types.Type
}

func (g *genCompatibility) Namers(c *generator.Context) namer.NameSystems {
	return namer.NameSystems{
		"public":       namer.NewPublicNamer(1),
		"intrapackage": namer.NewPublicNamer(0),
		"raw":          namer.NewRawNamer("", nil),
	}
}

func (g *genCompatibility) Filter(c *generator.Context, t *types.Type) bool {
	// Filter out types not being processed or not copyable within the package.
	if !isAPIType(t) {
		klog.V(2).Infof("Type %v is not a valid target for compatibility", t)
		return false
	}
	g.typesForInit = append(g.typesForInit, t)
	return true
}

func (g *genCompatibility) Imports(c *generator.Context) (imports []string) {
	importLines := []string{}
	for _, singleImport := range g.imports.ImportLines() {
		if g.isOtherPackage(singleImport) {
			importLines = append(importLines, singleImport)
		}
	}
	return importLines
}

func (g *genCompatibility) isOtherPackage(pkg string) bool {
	if pkg == g.targetPackage {
		return false
	}
	if strings.HasSuffix(pkg, "\""+g.targetPackage+"\"") {
		return false
	}
	return true
}

func (g *genCompatibility) Init(c *generator.Context, w io.Writer) error {
	return nil
}

// GenerateType emits the compatibility code for a particular type.
func (g *genCompatibility) GenerateType(c *generator.Context, t *types.Type, w io.Writer) error {
	klog.V(3).Infof("Generating OpenShift compatibility for type %v", t)

	sw := generator.NewSnippetWriter(w, c, "$", "$")

	args, err := g.argsFromType(c, t)
	if err != nil {
		return err
	}

	if versionedMethodOrDie("CompatibilityLevel", t) == nil {
		sw.Do("// CompatibilityLevel is an autogenerated function, returning the OpenShift API compatibility level.\n", args)
		sw.Do("// It is controlled by the \""+levelTagName+"\" tag in types.go.\n", args)
		sw.Do("func (in *$.type|intrapackage$) CompatibilityLevel() (level int) {\n", args)
		sw.Do("    return $.level$\n", args)
		sw.Do("}\n\n", nil)
	}
	if versionedMethodOrDie("Internal", t) == nil {
		sw.Do("// Internal is an autogenerated function, returning the true if the type is internal to OpenShift and not exposed as a supported API .\n", args)
		sw.Do("// It is controlled by the \""+internalTagName+"\" tag in types.go.\n", args)
		sw.Do("func (in *$.type|intrapackage$) Internal() (bool) {\n", args)
		sw.Do("    return $.internal$\n", args)
		sw.Do("}\n\n", nil)
	}
	return sw.Error()
}

func (g *genCompatibility) argsFromType(c *generator.Context, t *types.Type) (generator.Args, error) {
	a := generator.Args{
		"type": t,
	}

	internal := isInternal(t)
	a = a.With("internal", internal)

	level, ok := extractOpenShiftCompatibilityLevelTag(t)
	if !internal && !ok {
		return a, fmt.Errorf("%s: level or internal must be specified", t.Name)
	}
	if !ok {
		// default level for internal types
		level = 4
	}
	a = a.With("level", level)

	ga := versionIsGenerallyAvailable(t)
	beta := versionIsPrerelease(t)
	alpha := versionIsExperimental(t)
	_ = c.Universe.Package(g.targetPackage).SourcePath

	switch {
	case internal && level != 4:
		return a, fmt.Errorf("%s: APIs that are not internal are only allowed to offer level 4 compatibility: long term support cannot be offered for the %s API", t.Name.Package, t.Name.Name)
	case internal:
	case !(ga || alpha || beta):
		return a, fmt.Errorf("%s: APIs whose versions do not conform to kube apiVersion format cannot be exposed: the %s API must be tagged with +%s", t.Name, t.Name.Name, internalTagName)
	case ga && level != 1:
		return a, fmt.Errorf("%s: generally available APIs must be supported for a minimum of 12 months", t.Name)
	case beta && level == 1:
		return a, fmt.Errorf("%s: pre-release (beta) APIs must offer level 2 compatibility: the %s API should be versioned as generally available if you with to offer level 1 compatibility", t.Name, t.Name.Name)
	case beta && level == 4:
		return a, fmt.Errorf("%s: pre-release (beta) APIs must offer level 2 compatibility: the %s API should be versioned as experimental (alpha) if you wish to offer level 4 compatibility", t.Name, t.Name.Name)
	case alpha && level != 4:
		return a, fmt.Errorf("%s: experimental (alpha) APIs are only allowed to offer level 4 compatibility: long term support cannot be offered for the %s API", t.Name, t.Name.Name)
	}

	return a, nil
}

func versionIsGenerallyAvailable(t *types.Type) bool {
	return regexp.MustCompile(`^v\d*$`).MatchString(path.Base(t.Name.Package))
}

func versionIsPrerelease(t *types.Type) bool {
	return regexp.MustCompile(`^v\d*beta\d*$`).MatchString(path.Base(t.Name.Package))
}

func versionIsExperimental(t *types.Type) bool {
	return regexp.MustCompile(`^v\d*alpha\d*$`).MatchString(path.Base(t.Name.Package))
}

// versionedMethodOrDie returns the signature of a <methodName>() method, nil or calls klog.Fatalf
// if the type is wrong.
func versionedMethodOrDie(methodName string, t *types.Type) *types.Signature {
	ret, err := versionMethod(methodName, t)
	if err != nil {
		klog.Fatal(err)
	}
	return ret
}

// versionMethod returns the signature of an <methodName>() method, nil or an error
// if the type is wrong. Introduced() allows more efficient deep copy
// implementations to be defined by the type's author.  The correct signature
//    func (t *T) <methodName>() string
func versionMethod(methodName string, t *types.Type) (*types.Signature, error) {
	f, found := t.Methods[methodName]
	if !found {
		return nil, nil
	}
	if len(f.Signature.Parameters) != 0 {
		return nil, fmt.Errorf("type %v: invalid  %v signature, expected no parameters", t, methodName)
	}
	if len(f.Signature.Results) != 2 {
		return nil, fmt.Errorf("type %v: invalid  %v signature, expected exactly two result types", t, methodName)
	}

	ptrRcvr := f.Signature.Receiver != nil && f.Signature.Receiver.Kind == types.Pointer && f.Signature.Receiver.Elem.Name == t.Name
	nonPtrRcvr := f.Signature.Receiver != nil && f.Signature.Receiver.Name == t.Name

	if !ptrRcvr && !nonPtrRcvr {
		// this should never happen
		return nil, fmt.Errorf("type %v: invalid %v signature, expected a receiver of type %s or *%s", t, methodName, t.Name.Name, t.Name.Name)
	}

	return f.Signature, nil
}
