package alicloud

import (
	"fmt"
	"log"
	"strings"
	"testing"
	"time"

	"github.com/PaesslerAG/jsonpath"
	util "github.com/alibabacloud-go/tea-utils/service"

	"github.com/aliyun/terraform-provider-alicloud/alicloud/connectivity"
	"github.com/hashicorp/terraform-plugin-sdk/helper/acctest"
	"github.com/hashicorp/terraform-plugin-sdk/helper/resource"
)

func init() {
	resource.AddTestSweepers(
		"alicloud_mhub_app",
		&resource.Sweeper{
			Name: "alicloud_mhub_app",
			F:    testSweepMHUBApp,
		})
}

func testSweepMHUBApp(region string) error {
	rawClient, err := sharedClientForRegion(region)
	if err != nil {
		return fmt.Errorf("error getting Alicloud client: %s", err)
	}
	client := rawClient.(*connectivity.AliyunClient)
	prefixes := []string{
		"tf-testAcc",
		"tf_testAcc",
	}

	action := "ListProducts"
	request := map[string]interface{}{}

	request["Offset"] = 0

	request["Simple"] = false

	request["Size"] = PageSizeMedium

	var response map[string]interface{}
	conn, err := client.NewMhubClient()
	if err != nil {
		log.Printf("[ERROR] %s get an error: %#v", action, err)
	}
	for {
		runtime := util.RuntimeOptions{}
		runtime.SetAutoretry(true)
		wait := incrementalWait(3*time.Second, 3*time.Second)
		err = resource.Retry(1*time.Minute, func() *resource.RetryError {
			response, err = conn.DoRequest(StringPointer(action), nil, StringPointer("POST"), StringPointer("2017-08-25"), StringPointer("AK"), nil, request, &runtime)
			if err != nil {
				if NeedRetry(err) {
					wait()
					return resource.RetryableError(err)
				}
				return resource.NonRetryableError(err)
			}
			return nil
		})
		addDebug(action, response, request)
		if err != nil {
			log.Printf("[ERROR] %s get an error: %#v", action, err)
			return nil
		}
		resp, err := jsonpath.Get("$.ProductInfos.ProductInfo", response)
		if err != nil {
			log.Printf("[ERROR] Getting resource %s attribute by path %s failed!!! Body: %v.", "$.ProductInfos.ProductInfo", action, err)
			return nil
		}
		result, _ := resp.([]interface{})
		for _, v := range result {
			item := v.(map[string]interface{})

			action := "ListApps"
			request := map[string]interface{}{}

			request["ProductId"] = item["ProductId"]
			request["PageSize"] = PageSizeLarge
			request["Page"] = 1

			var response map[string]interface{}
			conn, err := client.NewMhubClient()
			if err != nil {
				log.Printf("[ERROR] %s get an error: %#v", action, err)
			}
			for {
				runtime := util.RuntimeOptions{}
				runtime.SetAutoretry(true)
				wait := incrementalWait(3*time.Second, 3*time.Second)
				err = resource.Retry(1*time.Minute, func() *resource.RetryError {
					response, err = conn.DoRequest(StringPointer(action), nil, StringPointer("POST"), StringPointer("2017-08-25"), StringPointer("AK"), nil, request, &runtime)
					if err != nil {
						if NeedRetry(err) {
							wait()
							return resource.RetryableError(err)
						}
						return resource.NonRetryableError(err)
					}
					return nil
				})
				addDebug(action, response, request)
				if err != nil {
					log.Printf("[ERROR] %s get an error: %#v", action, err)
					return nil
				}
				resp, err := jsonpath.Get("$.AppInfos.AppInfo", response)
				if err != nil {
					log.Printf("[ERROR] Getting resource %s attribute by path %s failed!!! Body: %v.", "$.AppInfos.AppInfo", action, err)
					return nil
				}
				result, _ := resp.([]interface{})
				for _, v := range result {
					item := v.(map[string]interface{})

					if _, ok := item["Name"]; !ok {
						continue
					}
					skip := true
					for _, prefix := range prefixes {
						if strings.HasPrefix(strings.ToLower(item["Name"].(string)), strings.ToLower(prefix)) {
							skip = false
						}
					}
					if skip {
						log.Printf("[INFO] Skipping Alb Listener: %s", item["Name"].(string))
						continue
					}
					action := "DeleteApp"
					request := map[string]interface{}{
						"AppKey": item["AppKey"],
					}
					_, err = conn.DoRequest(StringPointer(action), nil, StringPointer("POST"), StringPointer("2017-08-25"), StringPointer("AK"), nil, request, &util.RuntimeOptions{})
					if err != nil {
						log.Printf("[ERROR] Failed to delete MHUB App (%s): %s", item["AppKey"], err)
					}
					log.Printf("[INFO] Delete MHUB App success: %s ", item["AppKey"])
				}
				if len(result) < PageSizeLarge {
					break
				}
				request["Page"] = request["Page"].(int) + 1
			}
		}
		if len(result) < PageSizeMedium {
			break
		}
		request["Offset"] = request["Offset"].(int) + 1
	}
	return nil
}

func TestAccAlicloudMHUBApp_basic0(t *testing.T) {
	var v map[string]interface{}
	resourceId := "alicloud_mhub_app.default"
	ra := resourceAttrInit(resourceId, AlicloudMHUBAppMap0)
	rc := resourceCheckInitWithDescribeMethod(resourceId, &v, func() interface{} {
		return &MhubService{testAccProvider.Meta().(*connectivity.AliyunClient)}
	}, "DescribeMhubApp")
	rac := resourceAttrCheckInit(rc, ra)
	testAccCheck := rac.resourceAttrMapUpdateSet()
	rand := acctest.RandIntRange(10000, 99999)
	name := fmt.Sprintf("tf_testaccmhubapp%d", rand)
	testAccConfig := resourceTestAccConfigFunc(resourceId, name, AlicloudMHUBAppBasicDependence0)
	resource.Test(t, resource.TestCase{
		PreCheck: func() {
			testAccPreCheck(t)
			testAccPreCheckWithRegions(t, true, connectivity.MHUBSupportRegions)
		},
		IDRefreshName: resourceId,
		Providers:     testAccProviders,
		CheckDestroy:  rac.checkResourceDestroy(),
		Steps: []resource.TestStep{
			{
				Config: testAccConfig(map[string]interface{}{
					"app_name":     name,
					"product_id":   "${alicloud_mhub_product.default.id}",
					"type":         "Android",
					"package_name": "com.test.android",
				}),
				Check: resource.ComposeTestCheckFunc(
					testAccCheck(map[string]string{
						"app_name":     name,
						"product_id":   CHECKSET,
						"type":         "Android",
						"package_name": "com.test.android",
					}),
				),
			},
			{
				Config: testAccConfig(map[string]interface{}{
					"app_name": "tf_testaccmhubapp",
				}),
				Check: resource.ComposeTestCheckFunc(
					testAccCheck(map[string]string{
						"app_name": "tf_testaccmhubapp",
					}),
				),
			},

			{
				Config: testAccConfig(map[string]interface{}{
					"industry_id": "1",
				}),
				Check: resource.ComposeTestCheckFunc(
					testAccCheck(map[string]string{
						"industry_id": "1",
					}),
				),
			},
			{
				Config: testAccConfig(map[string]interface{}{
					"encoded_icon": "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",
				}),
				Check: resource.ComposeTestCheckFunc(
					testAccCheck(map[string]string{
						"encoded_icon": "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",
					}),
				),
			},
			{
				Config: testAccConfig(map[string]interface{}{
					"app_name":     "tf_testaccmhubapp",
					"industry_id":  "1",
					"encoded_icon": "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",
				}),
				Check: resource.ComposeTestCheckFunc(
					testAccCheck(map[string]string{
						"app_name":     "tf_testaccmhubapp",
						"industry_id":  "1",
						"encoded_icon": CHECKSET,
					}),
				),
			},
			{
				ResourceName:            resourceId,
				ImportState:             true,
				ImportStateVerify:       true,
				ImportStateVerifyIgnore: []string{"product_id"},
			},
		},
	})
}

var AlicloudMHUBAppMap0 = map[string]string{}

func AlicloudMHUBAppBasicDependence0(name string) string {
	return fmt.Sprintf(` 
variable "name" {
  default = "%s"
}

resource "alicloud_mhub_product" "default"{
  product_name = var.name
}

`, name)
}
