// +build examples

/**
 * (C) Copyright IBM Corp. 2020, 2021.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package vulnerabilityadvisorv3_test

import (
	"encoding/json"
	"fmt"
	"os"

	"github.com/IBM/container-registry-go-sdk/vulnerabilityadvisorv3"
	"github.com/IBM/go-sdk-core/v5/core"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
)

const externalConfigFile = "../vulnerability_advisor_v3.env"

var (
	vulnerabilityAdvisorService *vulnerabilityadvisorv3.VulnerabilityAdvisorV3
	config                      map[string]string
	configLoaded                bool = false
)

func shouldSkipTest() {
	Skip("Container Registry examples are not intended to be runnable tests")
	if !configLoaded {
		Skip("External configuration is not available, skipping tests...")
	}
}

var _ = Describe(`VulnerabilityAdvisorV3 Examples Tests`, func() {
	Describe(`External configuration`, func() {
		It("Successfully load the configuration", func() {
			var err error
			_, err = os.Stat(externalConfigFile)
			if err != nil {
				Skip("External configuration file not found, skipping tests: " + err.Error())
			}
			/**
			Your configuration file (vulnerability_advisor_v3.env) should contain the following variables.
			VULNERABILITY_ADVISOR_URL=[Registry URL, eg https://uk.icr.io]
			VULNERABILITY_ADVISOR_AUTH_TYPE=iam
			VULNERABILITY_ADVISOR_AUTH_URL=https://iam.cloud.ibm.com/identity/token
			VULNERABILITY_ADVISOR_APIKEY=[An IAM Apikey]
			*/
			os.Setenv("IBM_CREDENTIALS_FILE", externalConfigFile)
			config, err = core.GetServiceProperties(vulnerabilityadvisorv3.DefaultServiceName)
			if err != nil {
				Skip("Error loading service properties, skipping tests: " + err.Error())
			}

			configLoaded = len(config) > 0
		})
	})

	Describe(`Client initialization`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It("Successfully construct the service client instance", func() {
			var err error

			// begin-common

			vulnerabilityAdvisorServiceOptions := &vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
				Account: core.StringPtr("accountID"),
			}

			vulnerabilityAdvisorService, err = vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3UsingExternalConfig(vulnerabilityAdvisorServiceOptions)

			if err != nil {
				panic(err)
			}

			// end-common

			Expect(vulnerabilityAdvisorService).ToNot(BeNil())
		})
	})

	Describe(`VulnerabilityAdvisorV3 request examples`, func() {
		BeforeEach(func() {
			shouldSkipTest()
		})
		It(`AccountReportQueryPath request example`, func() {
			// begin-accountReportQueryPath

			accountReportQueryPathOptions := vulnerabilityAdvisorService.NewAccountReportQueryPathOptions()

			accountReportQueryPathOptions.IncludePrivate = core.StringPtr("true")

			scanReportList, response, err := vulnerabilityAdvisorService.AccountReportQueryPath(accountReportQueryPathOptions)
			if err != nil {
				panic(err)
			}
			b, _ := json.MarshalIndent(scanReportList, "", "  ")
			fmt.Println(string(b))

			// end-accountReportQueryPath

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(scanReportList).ToNot(BeNil())

		})
		It(`AccountStatusQueryPath request example`, func() {
			// begin-accountStatusQueryPath

			accountStatusQueryPathOptions := vulnerabilityAdvisorService.NewAccountStatusQueryPathOptions()

			accountStatusQueryPathOptions.IncludePrivate = core.StringPtr("true")

			scanreportImageSummaryList, response, err := vulnerabilityAdvisorService.AccountStatusQueryPath(accountStatusQueryPathOptions)
			if err != nil {
				panic(err)
			}
			b, _ := json.MarshalIndent(scanreportImageSummaryList, "", "  ")
			fmt.Println(string(b))

			// end-accountStatusQueryPath

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(scanreportImageSummaryList).ToNot(BeNil())

		})
		It(`ImageStatusQueryPath request example`, func() {
			// begin-imageStatusQueryPath

			imageStatusQueryPathOptions := vulnerabilityAdvisorService.NewImageStatusQueryPathOptions(
				"image name",
			)

			scanreportSummary, response, err := vulnerabilityAdvisorService.ImageStatusQueryPath(imageStatusQueryPathOptions)
			if err != nil {
				panic(err)
			}
			b, _ := json.MarshalIndent(scanreportSummary, "", "  ")
			fmt.Println(string(b))

			// end-imageStatusQueryPath

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(scanreportSummary).ToNot(BeNil())

		})
		It(`ImageReportQueryPath request example`, func() {
			// begin-imageReportQueryPath

			imageReportQueryPathOptions := vulnerabilityAdvisorService.NewImageReportQueryPathOptions(
				"image name",
			)

			scanReport, response, err := vulnerabilityAdvisorService.ImageReportQueryPath(imageReportQueryPathOptions)
			if err != nil {
				panic(err)
			}
			b, _ := json.MarshalIndent(scanReport, "", "  ")
			fmt.Println(string(b))

			// end-imageReportQueryPath

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(202))
			Expect(scanReport).ToNot(BeNil())

		})
		It(`ListExemptionAccount request example`, func() {
			// begin-listExemptionAccount

			listExemptionAccountOptions := vulnerabilityAdvisorService.NewListExemptionAccountOptions()

			exemption, response, err := vulnerabilityAdvisorService.ListExemptionAccount(listExemptionAccountOptions)
			if err != nil {
				panic(err)
			}
			b, _ := json.MarshalIndent(exemption, "", "  ")
			fmt.Println(string(b))

			// end-listExemptionAccount

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemption).ToNot(BeNil())

		})
		It(`GetExemptionAccount request example`, func() {
			// begin-getExemptionAccount

			getExemptionAccountOptions := vulnerabilityAdvisorService.NewGetExemptionAccountOptions(
				"cve",
				"CVE-2020-0001",
			)

			exemption, response, err := vulnerabilityAdvisorService.GetExemptionAccount(getExemptionAccountOptions)
			if err != nil {
				panic(err)
			}
			b, _ := json.MarshalIndent(exemption, "", "  ")
			fmt.Println(string(b))

			// end-getExemptionAccount

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemption).ToNot(BeNil())

		})
		It(`CreateExemptionAccount request example`, func() {
			// begin-createExemptionAccount

			createExemptionAccountOptions := vulnerabilityAdvisorService.NewCreateExemptionAccountOptions(
				"cve",
				"CVE-2020-0001",
			)

			exemption, response, err := vulnerabilityAdvisorService.CreateExemptionAccount(createExemptionAccountOptions)
			if err != nil {
				panic(err)
			}
			b, _ := json.MarshalIndent(exemption, "", "  ")
			fmt.Println(string(b))

			// end-createExemptionAccount

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(201))
			Expect(exemption).ToNot(BeNil())

		})
		It(`ListExemptionResource request example`, func() {
			// begin-listExemptionResource

			listExemptionResourceOptions := vulnerabilityAdvisorService.NewListExemptionResourceOptions(
				"image name",
			)

			exemption, response, err := vulnerabilityAdvisorService.ListExemptionResource(listExemptionResourceOptions)
			if err != nil {
				panic(err)
			}
			b, _ := json.MarshalIndent(exemption, "", "  ")
			fmt.Println(string(b))

			// end-listExemptionResource

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemption).ToNot(BeNil())

		})
		It(`GetExemptionResource request example`, func() {
			// begin-getExemptionResource

			getExemptionResourceOptions := vulnerabilityAdvisorService.NewGetExemptionResourceOptions(
				"cve",
				"CVE-2020-0001",
				"image name",
			)

			exemption, response, err := vulnerabilityAdvisorService.GetExemptionResource(getExemptionResourceOptions)
			if err != nil {
				panic(err)
			}
			b, _ := json.MarshalIndent(exemption, "", "  ")
			fmt.Println(string(b))

			// end-getExemptionResource

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemption).ToNot(BeNil())

		})
		It(`CreateExemptionResource request example`, func() {
			// begin-createExemptionResource

			createExemptionResourceOptions := vulnerabilityAdvisorService.NewCreateExemptionResourceOptions(
				"cve",
				"CVE-2020-0001",
				"image name",
			)

			exemption, response, err := vulnerabilityAdvisorService.CreateExemptionResource(createExemptionResourceOptions)
			if err != nil {
				panic(err)
			}
			b, _ := json.MarshalIndent(exemption, "", "  ")
			fmt.Println(string(b))

			// end-createExemptionResource

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(201))
			Expect(exemption).ToNot(BeNil())

		})
		It(`ExemptHandler request example`, func() {
			// begin-exemptHandler

			exemptHandlerOptions := vulnerabilityAdvisorService.NewExemptHandlerOptions()

			exemptionTypeInfo, response, err := vulnerabilityAdvisorService.ExemptHandler(exemptHandlerOptions)
			if err != nil {
				panic(err)
			}
			b, _ := json.MarshalIndent(exemptionTypeInfo, "", "  ")
			fmt.Println(string(b))

			// end-exemptHandler

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemptionTypeInfo).ToNot(BeNil())

		})
		It(`ListAccountExemptions request example`, func() {
			// begin-listAccountExemptions

			listAccountExemptionsOptions := vulnerabilityAdvisorService.NewListAccountExemptionsOptions()

			exemption, response, err := vulnerabilityAdvisorService.ListAccountExemptions(listAccountExemptionsOptions)
			if err != nil {
				panic(err)
			}
			b, _ := json.MarshalIndent(exemption, "", "  ")
			fmt.Println(string(b))

			// end-listAccountExemptions

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemption).ToNot(BeNil())

		})
		It(`ExemptionsAccountDeleteHandler request example`, func() {
			// begin-exemptionsAccountDeleteHandler

			exemptionsAccountDeleteHandlerOptions := vulnerabilityAdvisorService.NewExemptionsAccountDeleteHandlerOptions()

			exemptionDeletionInfo, response, err := vulnerabilityAdvisorService.ExemptionsAccountDeleteHandler(exemptionsAccountDeleteHandlerOptions)
			if err != nil {
				panic(err)
			}
			b, _ := json.MarshalIndent(exemptionDeletionInfo, "", "  ")
			fmt.Println(string(b))

			// end-exemptionsAccountDeleteHandler

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemptionDeletionInfo).ToNot(BeNil())

		})
		It(`ListImageExemptions request example`, func() {
			// begin-listImageExemptions

			listImageExemptionsOptions := vulnerabilityAdvisorService.NewListImageExemptionsOptions(
				"image name",
			)

			exemption, response, err := vulnerabilityAdvisorService.ListImageExemptions(listImageExemptionsOptions)
			if err != nil {
				panic(err)
			}
			b, _ := json.MarshalIndent(exemption, "", "  ")
			fmt.Println(string(b))

			// end-listImageExemptions

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(exemption).ToNot(BeNil())

		})
		It(`ListBulkImageExemptions request example`, func() {
			// begin-listBulkImageExemptions

			listBulkImageExemptionsOptions := vulnerabilityAdvisorService.NewListBulkImageExemptionsOptions(
				[]string{"image name"},
			)

			mapStringExemption, response, err := vulnerabilityAdvisorService.ListBulkImageExemptions(listBulkImageExemptionsOptions)
			if err != nil {
				panic(err)
			}
			b, _ := json.MarshalIndent(mapStringExemption, "", "  ")
			fmt.Println(string(b))

			// end-listBulkImageExemptions

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))
			Expect(mapStringExemption).ToNot(BeNil())

		})
		It(`DeleteExemptionResource request example`, func() {
			// begin-deleteExemptionResource

			deleteExemptionResourceOptions := vulnerabilityAdvisorService.NewDeleteExemptionResourceOptions(
				"cve",
				"CVE-2020-0001",
				"image name",
			)

			response, err := vulnerabilityAdvisorService.DeleteExemptionResource(deleteExemptionResourceOptions)
			if err != nil {
				panic(err)
			}

			// end-deleteExemptionResource

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))

		})
		It(`DeleteExemptionAccount request example`, func() {
			// begin-deleteExemptionAccount

			deleteExemptionAccountOptions := vulnerabilityAdvisorService.NewDeleteExemptionAccountOptions(
				"cve",
				"CVE-2020-0001",
			)

			response, err := vulnerabilityAdvisorService.DeleteExemptionAccount(deleteExemptionAccountOptions)
			if err != nil {
				panic(err)
			}

			// end-deleteExemptionAccount

			Expect(err).To(BeNil())
			Expect(response.StatusCode).To(Equal(200))

		})
	})
})
