/**
 * (C) Copyright IBM Corp. 2020, 2021.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package vpcclassicv1_test

import (
	"bytes"
	"context"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"os"
	"time"

	"github.com/IBM/go-sdk-core/v5/core"
	"github.com/IBM/vpc-go-sdk/vpcclassicv1"
	"github.com/go-openapi/strfmt"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
)

var _ = Describe(`VpcClassicV1`, func() {
	var testServer *httptest.Server
	Describe(`Service constructor tests`, func() {
		version := "testString"
		It(`Instantiate service client`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				Authenticator: &core.NoAuthAuthenticator{},
				Version:       core.StringPtr(version),
			})
			Expect(vpcClassicService).ToNot(BeNil())
			Expect(serviceErr).To(BeNil())
		})
		It(`Instantiate service client with error: Invalid URL`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Invalid Auth`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "https://vpcclassicv1/api",
				Version: core.StringPtr(version),
				Authenticator: &core.BasicAuthenticator{
					Username: "",
					Password: "",
				},
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Validation Error`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
	})
	Describe(`Service constructor tests using external config`, func() {
		version := "testString"
		Context(`Using external config, construct service client instances`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "noauth",
			}

			It(`Create service client using external config successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url from constructor successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					URL:     "https://testService/api",
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url programatically successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				err := vpcClassicService.SetServiceURL("https://testService/api")
				Expect(err).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid Auth`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "someOtherAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid URL`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_AUTH_TYPE": "NOAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
	})
	Describe(`Regional endpoint tests`, func() {
		It(`GetServiceURLForRegion(region string)`, func() {
			var url string
			var err error
			url, err = vpcclassicv1.GetServiceURLForRegion("INVALID_REGION")
			Expect(url).To(BeEmpty())
			Expect(err).ToNot(BeNil())
			fmt.Fprintf(GinkgoWriter, "Expected error: %s\n", err.Error())
		})
	})
	Describe(`ListVpcs(listVpcsOptions *ListVpcsOptions) - Operation response error`, func() {
		version := "testString"
		listVpcsPath := "/vpcs"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVpcsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// TODO: Add check for classic_access query parameter

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListVpcs with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListVpcsOptions model
				listVpcsOptionsModel := new(vpcclassicv1.ListVpcsOptions)
				listVpcsOptionsModel.Start = core.StringPtr("testString")
				listVpcsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVpcsOptionsModel.ClassicAccess = core.BoolPtr(true)
				listVpcsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListVpcs(listVpcsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListVpcs(listVpcsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListVpcs(listVpcsOptions *ListVpcsOptions)`, func() {
		version := "testString"
		listVpcsPath := "/vpcs"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVpcsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// TODO: Add check for classic_access query parameter

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "vpcs": [{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available"}]}`)
				}))
			})
			It(`Invoke ListVpcs successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListVpcsOptions model
				listVpcsOptionsModel := new(vpcclassicv1.ListVpcsOptions)
				listVpcsOptionsModel.Start = core.StringPtr("testString")
				listVpcsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVpcsOptionsModel.ClassicAccess = core.BoolPtr(true)
				listVpcsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListVpcsWithContext(ctx, listVpcsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListVpcs(listVpcsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListVpcsWithContext(ctx, listVpcsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVpcsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// TODO: Add check for classic_access query parameter

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "vpcs": [{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available"}]}`)
				}))
			})
			It(`Invoke ListVpcs successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListVpcs(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListVpcsOptions model
				listVpcsOptionsModel := new(vpcclassicv1.ListVpcsOptions)
				listVpcsOptionsModel.Start = core.StringPtr("testString")
				listVpcsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVpcsOptionsModel.ClassicAccess = core.BoolPtr(true)
				listVpcsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListVpcs(listVpcsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListVpcs with error: Operation request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListVpcsOptions model
				listVpcsOptionsModel := new(vpcclassicv1.ListVpcsOptions)
				listVpcsOptionsModel.Start = core.StringPtr("testString")
				listVpcsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVpcsOptionsModel.ClassicAccess = core.BoolPtr(true)
				listVpcsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListVpcs(listVpcsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateVPC(createVPCOptions *CreateVPCOptions) - Operation response error`, func() {
		version := "testString"
		createVPCPath := "/vpcs"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateVPC with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateVPCOptions model
				createVPCOptionsModel := new(vpcclassicv1.CreateVPCOptions)
				createVPCOptionsModel.AddressPrefixManagement = core.StringPtr("manual")
				createVPCOptionsModel.ClassicAccess = core.BoolPtr(false)
				createVPCOptionsModel.Name = core.StringPtr("my-vpc")
				createVPCOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.CreateVPC(createVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.CreateVPC(createVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`CreateVPC(createVPCOptions *CreateVPCOptions)`, func() {
		version := "testString"
		createVPCPath := "/vpcs"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available"}`)
				}))
			})
			It(`Invoke CreateVPC successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateVPCOptions model
				createVPCOptionsModel := new(vpcclassicv1.CreateVPCOptions)
				createVPCOptionsModel.AddressPrefixManagement = core.StringPtr("manual")
				createVPCOptionsModel.ClassicAccess = core.BoolPtr(false)
				createVPCOptionsModel.Name = core.StringPtr("my-vpc")
				createVPCOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.CreateVPCWithContext(ctx, createVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.CreateVPC(createVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.CreateVPCWithContext(ctx, createVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available"}`)
				}))
			})
			It(`Invoke CreateVPC successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.CreateVPC(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateVPCOptions model
				createVPCOptionsModel := new(vpcclassicv1.CreateVPCOptions)
				createVPCOptionsModel.AddressPrefixManagement = core.StringPtr("manual")
				createVPCOptionsModel.ClassicAccess = core.BoolPtr(false)
				createVPCOptionsModel.Name = core.StringPtr("my-vpc")
				createVPCOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.CreateVPC(createVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateVPC with error: Operation request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateVPCOptions model
				createVPCOptionsModel := new(vpcclassicv1.CreateVPCOptions)
				createVPCOptionsModel.AddressPrefixManagement = core.StringPtr("manual")
				createVPCOptionsModel.ClassicAccess = core.BoolPtr(false)
				createVPCOptionsModel.Name = core.StringPtr("my-vpc")
				createVPCOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.CreateVPC(createVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`DeleteVPC(deleteVPCOptions *DeleteVPCOptions)`, func() {
		version := "testString"
		deleteVPCPath := "/vpcs/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteVPCPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteVPC successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcClassicService.DeleteVPC(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteVPCOptions model
				deleteVPCOptionsModel := new(vpcclassicv1.DeleteVPCOptions)
				deleteVPCOptionsModel.ID = core.StringPtr("testString")
				deleteVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcClassicService.DeleteVPC(deleteVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteVPC with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the DeleteVPCOptions model
				deleteVPCOptionsModel := new(vpcclassicv1.DeleteVPCOptions)
				deleteVPCOptionsModel.ID = core.StringPtr("testString")
				deleteVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcClassicService.DeleteVPC(deleteVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteVPCOptions model with no property values
				deleteVPCOptionsModelNew := new(vpcclassicv1.DeleteVPCOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcClassicService.DeleteVPC(deleteVPCOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPC(getVPCOptions *GetVPCOptions) - Operation response error`, func() {
		version := "testString"
		getVPCPath := "/vpcs/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetVPC with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetVPCOptions model
				getVPCOptionsModel := new(vpcclassicv1.GetVPCOptions)
				getVPCOptionsModel.ID = core.StringPtr("testString")
				getVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetVPC(getVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetVPC(getVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetVPC(getVPCOptions *GetVPCOptions)`, func() {
		version := "testString"
		getVPCPath := "/vpcs/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available"}`)
				}))
			})
			It(`Invoke GetVPC successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetVPCOptions model
				getVPCOptionsModel := new(vpcclassicv1.GetVPCOptions)
				getVPCOptionsModel.ID = core.StringPtr("testString")
				getVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetVPCWithContext(ctx, getVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetVPC(getVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetVPCWithContext(ctx, getVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available"}`)
				}))
			})
			It(`Invoke GetVPC successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetVPC(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetVPCOptions model
				getVPCOptionsModel := new(vpcclassicv1.GetVPCOptions)
				getVPCOptionsModel.ID = core.StringPtr("testString")
				getVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetVPC(getVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetVPC with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetVPCOptions model
				getVPCOptionsModel := new(vpcclassicv1.GetVPCOptions)
				getVPCOptionsModel.ID = core.StringPtr("testString")
				getVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetVPC(getVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetVPCOptions model with no property values
				getVPCOptionsModelNew := new(vpcclassicv1.GetVPCOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetVPC(getVPCOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateVPC(updateVPCOptions *UpdateVPCOptions) - Operation response error`, func() {
		version := "testString"
		updateVPCPath := "/vpcs/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCPath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateVPC with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the VPCPatch model
				vpcPatchModel := new(vpcclassicv1.VPCPatch)
				vpcPatchModel.Name = core.StringPtr("my-vpc")
				vpcPatchModelAsPatch, asPatchErr := vpcPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCOptions model
				updateVPCOptionsModel := new(vpcclassicv1.UpdateVPCOptions)
				updateVPCOptionsModel.ID = core.StringPtr("testString")
				updateVPCOptionsModel.VPCPatch = vpcPatchModelAsPatch
				updateVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.UpdateVPC(updateVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.UpdateVPC(updateVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`UpdateVPC(updateVPCOptions *UpdateVPCOptions)`, func() {
		version := "testString"
		updateVPCPath := "/vpcs/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available"}`)
				}))
			})
			It(`Invoke UpdateVPC successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the VPCPatch model
				vpcPatchModel := new(vpcclassicv1.VPCPatch)
				vpcPatchModel.Name = core.StringPtr("my-vpc")
				vpcPatchModelAsPatch, asPatchErr := vpcPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCOptions model
				updateVPCOptionsModel := new(vpcclassicv1.UpdateVPCOptions)
				updateVPCOptionsModel.ID = core.StringPtr("testString")
				updateVPCOptionsModel.VPCPatch = vpcPatchModelAsPatch
				updateVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.UpdateVPCWithContext(ctx, updateVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.UpdateVPC(updateVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.UpdateVPCWithContext(ctx, updateVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"classic_access": false, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "cse_source_ips": [{"ip": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "default_network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "default_security_group": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available"}`)
				}))
			})
			It(`Invoke UpdateVPC successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.UpdateVPC(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the VPCPatch model
				vpcPatchModel := new(vpcclassicv1.VPCPatch)
				vpcPatchModel.Name = core.StringPtr("my-vpc")
				vpcPatchModelAsPatch, asPatchErr := vpcPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCOptions model
				updateVPCOptionsModel := new(vpcclassicv1.UpdateVPCOptions)
				updateVPCOptionsModel.ID = core.StringPtr("testString")
				updateVPCOptionsModel.VPCPatch = vpcPatchModelAsPatch
				updateVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.UpdateVPC(updateVPCOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateVPC with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the VPCPatch model
				vpcPatchModel := new(vpcclassicv1.VPCPatch)
				vpcPatchModel.Name = core.StringPtr("my-vpc")
				vpcPatchModelAsPatch, asPatchErr := vpcPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCOptions model
				updateVPCOptionsModel := new(vpcclassicv1.UpdateVPCOptions)
				updateVPCOptionsModel.ID = core.StringPtr("testString")
				updateVPCOptionsModel.VPCPatch = vpcPatchModelAsPatch
				updateVPCOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.UpdateVPC(updateVPCOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateVPCOptions model with no property values
				updateVPCOptionsModelNew := new(vpcclassicv1.UpdateVPCOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.UpdateVPC(updateVPCOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptions *GetVPCDefaultSecurityGroupOptions) - Operation response error`, func() {
		version := "testString"
		getVPCDefaultSecurityGroupPath := "/vpcs/testString/default_security_group"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCDefaultSecurityGroupPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetVPCDefaultSecurityGroup with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetVPCDefaultSecurityGroupOptions model
				getVPCDefaultSecurityGroupOptionsModel := new(vpcclassicv1.GetVPCDefaultSecurityGroupOptions)
				getVPCDefaultSecurityGroupOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultSecurityGroupOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptions *GetVPCDefaultSecurityGroupOptions)`, func() {
		version := "testString"
		getVPCDefaultSecurityGroupPath := "/vpcs/testString/default_security_group"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCDefaultSecurityGroupPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "observant-chip-emphatic-engraver", "rules": [{"direction": "inbound", "id": "6f2a6efe-21e2-401c-b237-620aa26ba16a", "ip_version": "ipv4", "remote": {"address": "192.168.3.4"}, "protocol": "all"}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}}`)
				}))
			})
			It(`Invoke GetVPCDefaultSecurityGroup successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetVPCDefaultSecurityGroupOptions model
				getVPCDefaultSecurityGroupOptionsModel := new(vpcclassicv1.GetVPCDefaultSecurityGroupOptions)
				getVPCDefaultSecurityGroupOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultSecurityGroupOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetVPCDefaultSecurityGroupWithContext(ctx, getVPCDefaultSecurityGroupOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetVPCDefaultSecurityGroupWithContext(ctx, getVPCDefaultSecurityGroupOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCDefaultSecurityGroupPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "observant-chip-emphatic-engraver", "rules": [{"direction": "inbound", "id": "6f2a6efe-21e2-401c-b237-620aa26ba16a", "ip_version": "ipv4", "remote": {"address": "192.168.3.4"}, "protocol": "all"}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}}`)
				}))
			})
			It(`Invoke GetVPCDefaultSecurityGroup successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetVPCDefaultSecurityGroup(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetVPCDefaultSecurityGroupOptions model
				getVPCDefaultSecurityGroupOptionsModel := new(vpcclassicv1.GetVPCDefaultSecurityGroupOptions)
				getVPCDefaultSecurityGroupOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultSecurityGroupOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetVPCDefaultSecurityGroup with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetVPCDefaultSecurityGroupOptions model
				getVPCDefaultSecurityGroupOptionsModel := new(vpcclassicv1.GetVPCDefaultSecurityGroupOptions)
				getVPCDefaultSecurityGroupOptionsModel.ID = core.StringPtr("testString")
				getVPCDefaultSecurityGroupOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetVPCDefaultSecurityGroupOptions model with no property values
				getVPCDefaultSecurityGroupOptionsModelNew := new(vpcclassicv1.GetVPCDefaultSecurityGroupOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListVPCAddressPrefixes(listVPCAddressPrefixesOptions *ListVPCAddressPrefixesOptions) - Operation response error`, func() {
		version := "testString"
		listVPCAddressPrefixesPath := "/vpcs/testString/address_prefixes"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCAddressPrefixesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListVPCAddressPrefixes with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListVPCAddressPrefixesOptions model
				listVPCAddressPrefixesOptionsModel := new(vpcclassicv1.ListVPCAddressPrefixesOptions)
				listVPCAddressPrefixesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Start = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCAddressPrefixesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListVPCAddressPrefixes(listVPCAddressPrefixesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListVPCAddressPrefixes(listVPCAddressPrefixesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListVPCAddressPrefixes(listVPCAddressPrefixesOptions *ListVPCAddressPrefixesOptions)`, func() {
		version := "testString"
		listVPCAddressPrefixesPath := "/vpcs/testString/address_prefixes"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCAddressPrefixesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"address_prefixes": [{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/a4e28308-8ee7-46ab-8108-9f881f22bdbf/address_prefixes?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/a4e28308-8ee7-46ab-8108-9f881f22bdbf/address_prefixes?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}}`)
				}))
			})
			It(`Invoke ListVPCAddressPrefixes successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListVPCAddressPrefixesOptions model
				listVPCAddressPrefixesOptionsModel := new(vpcclassicv1.ListVPCAddressPrefixesOptions)
				listVPCAddressPrefixesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Start = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCAddressPrefixesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListVPCAddressPrefixesWithContext(ctx, listVPCAddressPrefixesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListVPCAddressPrefixes(listVPCAddressPrefixesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListVPCAddressPrefixesWithContext(ctx, listVPCAddressPrefixesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCAddressPrefixesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"address_prefixes": [{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/a4e28308-8ee7-46ab-8108-9f881f22bdbf/address_prefixes?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/a4e28308-8ee7-46ab-8108-9f881f22bdbf/address_prefixes?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}}`)
				}))
			})
			It(`Invoke ListVPCAddressPrefixes successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListVPCAddressPrefixes(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListVPCAddressPrefixesOptions model
				listVPCAddressPrefixesOptionsModel := new(vpcclassicv1.ListVPCAddressPrefixesOptions)
				listVPCAddressPrefixesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Start = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCAddressPrefixesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListVPCAddressPrefixes(listVPCAddressPrefixesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListVPCAddressPrefixes with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListVPCAddressPrefixesOptions model
				listVPCAddressPrefixesOptionsModel := new(vpcclassicv1.ListVPCAddressPrefixesOptions)
				listVPCAddressPrefixesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Start = core.StringPtr("testString")
				listVPCAddressPrefixesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVPCAddressPrefixesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListVPCAddressPrefixes(listVPCAddressPrefixesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListVPCAddressPrefixesOptions model with no property values
				listVPCAddressPrefixesOptionsModelNew := new(vpcclassicv1.ListVPCAddressPrefixesOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.ListVPCAddressPrefixes(listVPCAddressPrefixesOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateVPCAddressPrefix(createVPCAddressPrefixOptions *CreateVPCAddressPrefixOptions) - Operation response error`, func() {
		version := "testString"
		createVPCAddressPrefixPath := "/vpcs/testString/address_prefixes"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateVPCAddressPrefix with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the CreateVPCAddressPrefixOptions model
				createVPCAddressPrefixOptionsModel := new(vpcclassicv1.CreateVPCAddressPrefixOptions)
				createVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				createVPCAddressPrefixOptionsModel.CIDR = core.StringPtr("10.0.0.0/24")
				createVPCAddressPrefixOptionsModel.Zone = zoneIdentityModel
				createVPCAddressPrefixOptionsModel.IsDefault = core.BoolPtr(true)
				createVPCAddressPrefixOptionsModel.Name = core.StringPtr("my-address-prefix-2")
				createVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.CreateVPCAddressPrefix(createVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.CreateVPCAddressPrefix(createVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`CreateVPCAddressPrefix(createVPCAddressPrefixOptions *CreateVPCAddressPrefixOptions)`, func() {
		version := "testString"
		createVPCAddressPrefixPath := "/vpcs/testString/address_prefixes"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateVPCAddressPrefix successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the CreateVPCAddressPrefixOptions model
				createVPCAddressPrefixOptionsModel := new(vpcclassicv1.CreateVPCAddressPrefixOptions)
				createVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				createVPCAddressPrefixOptionsModel.CIDR = core.StringPtr("10.0.0.0/24")
				createVPCAddressPrefixOptionsModel.Zone = zoneIdentityModel
				createVPCAddressPrefixOptionsModel.IsDefault = core.BoolPtr(true)
				createVPCAddressPrefixOptionsModel.Name = core.StringPtr("my-address-prefix-2")
				createVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.CreateVPCAddressPrefixWithContext(ctx, createVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.CreateVPCAddressPrefix(createVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.CreateVPCAddressPrefixWithContext(ctx, createVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateVPCAddressPrefix successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.CreateVPCAddressPrefix(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the CreateVPCAddressPrefixOptions model
				createVPCAddressPrefixOptionsModel := new(vpcclassicv1.CreateVPCAddressPrefixOptions)
				createVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				createVPCAddressPrefixOptionsModel.CIDR = core.StringPtr("10.0.0.0/24")
				createVPCAddressPrefixOptionsModel.Zone = zoneIdentityModel
				createVPCAddressPrefixOptionsModel.IsDefault = core.BoolPtr(true)
				createVPCAddressPrefixOptionsModel.Name = core.StringPtr("my-address-prefix-2")
				createVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.CreateVPCAddressPrefix(createVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateVPCAddressPrefix with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the CreateVPCAddressPrefixOptions model
				createVPCAddressPrefixOptionsModel := new(vpcclassicv1.CreateVPCAddressPrefixOptions)
				createVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				createVPCAddressPrefixOptionsModel.CIDR = core.StringPtr("10.0.0.0/24")
				createVPCAddressPrefixOptionsModel.Zone = zoneIdentityModel
				createVPCAddressPrefixOptionsModel.IsDefault = core.BoolPtr(true)
				createVPCAddressPrefixOptionsModel.Name = core.StringPtr("my-address-prefix-2")
				createVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.CreateVPCAddressPrefix(createVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateVPCAddressPrefixOptions model with no property values
				createVPCAddressPrefixOptionsModelNew := new(vpcclassicv1.CreateVPCAddressPrefixOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.CreateVPCAddressPrefix(createVPCAddressPrefixOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`DeleteVPCAddressPrefix(deleteVPCAddressPrefixOptions *DeleteVPCAddressPrefixOptions)`, func() {
		version := "testString"
		deleteVPCAddressPrefixPath := "/vpcs/testString/address_prefixes/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteVPCAddressPrefix successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcClassicService.DeleteVPCAddressPrefix(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteVPCAddressPrefixOptions model
				deleteVPCAddressPrefixOptionsModel := new(vpcclassicv1.DeleteVPCAddressPrefixOptions)
				deleteVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				deleteVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				deleteVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcClassicService.DeleteVPCAddressPrefix(deleteVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteVPCAddressPrefix with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the DeleteVPCAddressPrefixOptions model
				deleteVPCAddressPrefixOptionsModel := new(vpcclassicv1.DeleteVPCAddressPrefixOptions)
				deleteVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				deleteVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				deleteVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcClassicService.DeleteVPCAddressPrefix(deleteVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteVPCAddressPrefixOptions model with no property values
				deleteVPCAddressPrefixOptionsModelNew := new(vpcclassicv1.DeleteVPCAddressPrefixOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcClassicService.DeleteVPCAddressPrefix(deleteVPCAddressPrefixOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCAddressPrefix(getVPCAddressPrefixOptions *GetVPCAddressPrefixOptions) - Operation response error`, func() {
		version := "testString"
		getVPCAddressPrefixPath := "/vpcs/testString/address_prefixes/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetVPCAddressPrefix with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetVPCAddressPrefixOptions model
				getVPCAddressPrefixOptionsModel := new(vpcclassicv1.GetVPCAddressPrefixOptions)
				getVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetVPCAddressPrefix(getVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetVPCAddressPrefix(getVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetVPCAddressPrefix(getVPCAddressPrefixOptions *GetVPCAddressPrefixOptions)`, func() {
		version := "testString"
		getVPCAddressPrefixPath := "/vpcs/testString/address_prefixes/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetVPCAddressPrefix successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetVPCAddressPrefixOptions model
				getVPCAddressPrefixOptionsModel := new(vpcclassicv1.GetVPCAddressPrefixOptions)
				getVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetVPCAddressPrefixWithContext(ctx, getVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetVPCAddressPrefix(getVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetVPCAddressPrefixWithContext(ctx, getVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetVPCAddressPrefix successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetVPCAddressPrefix(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetVPCAddressPrefixOptions model
				getVPCAddressPrefixOptionsModel := new(vpcclassicv1.GetVPCAddressPrefixOptions)
				getVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetVPCAddressPrefix(getVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetVPCAddressPrefix with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetVPCAddressPrefixOptions model
				getVPCAddressPrefixOptionsModel := new(vpcclassicv1.GetVPCAddressPrefixOptions)
				getVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				getVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetVPCAddressPrefix(getVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetVPCAddressPrefixOptions model with no property values
				getVPCAddressPrefixOptionsModelNew := new(vpcclassicv1.GetVPCAddressPrefixOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetVPCAddressPrefix(getVPCAddressPrefixOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateVPCAddressPrefix(updateVPCAddressPrefixOptions *UpdateVPCAddressPrefixOptions) - Operation response error`, func() {
		version := "testString"
		updateVPCAddressPrefixPath := "/vpcs/testString/address_prefixes/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateVPCAddressPrefix with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the AddressPrefixPatch model
				addressPrefixPatchModel := new(vpcclassicv1.AddressPrefixPatch)
				addressPrefixPatchModel.IsDefault = core.BoolPtr(false)
				addressPrefixPatchModel.Name = core.StringPtr("my-address-prefix-2")
				addressPrefixPatchModelAsPatch, asPatchErr := addressPrefixPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCAddressPrefixOptions model
				updateVPCAddressPrefixOptionsModel := new(vpcclassicv1.UpdateVPCAddressPrefixOptions)
				updateVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.AddressPrefixPatch = addressPrefixPatchModelAsPatch
				updateVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.UpdateVPCAddressPrefix(updateVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.UpdateVPCAddressPrefix(updateVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`UpdateVPCAddressPrefix(updateVPCAddressPrefixOptions *UpdateVPCAddressPrefixOptions)`, func() {
		version := "testString"
		updateVPCAddressPrefixPath := "/vpcs/testString/address_prefixes/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateVPCAddressPrefix successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the AddressPrefixPatch model
				addressPrefixPatchModel := new(vpcclassicv1.AddressPrefixPatch)
				addressPrefixPatchModel.IsDefault = core.BoolPtr(false)
				addressPrefixPatchModel.Name = core.StringPtr("my-address-prefix-2")
				addressPrefixPatchModelAsPatch, asPatchErr := addressPrefixPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCAddressPrefixOptions model
				updateVPCAddressPrefixOptionsModel := new(vpcclassicv1.UpdateVPCAddressPrefixOptions)
				updateVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.AddressPrefixPatch = addressPrefixPatchModelAsPatch
				updateVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.UpdateVPCAddressPrefixWithContext(ctx, updateVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.UpdateVPCAddressPrefix(updateVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.UpdateVPCAddressPrefixWithContext(ctx, updateVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCAddressPrefixPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"cidr": "192.168.3.0/24", "created_at": "2019-01-01T12:00:00.000Z", "has_subnets": true, "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/address_prefixes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "is_default": false, "name": "my-address-prefix-2", "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateVPCAddressPrefix successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.UpdateVPCAddressPrefix(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the AddressPrefixPatch model
				addressPrefixPatchModel := new(vpcclassicv1.AddressPrefixPatch)
				addressPrefixPatchModel.IsDefault = core.BoolPtr(false)
				addressPrefixPatchModel.Name = core.StringPtr("my-address-prefix-2")
				addressPrefixPatchModelAsPatch, asPatchErr := addressPrefixPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCAddressPrefixOptions model
				updateVPCAddressPrefixOptionsModel := new(vpcclassicv1.UpdateVPCAddressPrefixOptions)
				updateVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.AddressPrefixPatch = addressPrefixPatchModelAsPatch
				updateVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.UpdateVPCAddressPrefix(updateVPCAddressPrefixOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateVPCAddressPrefix with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the AddressPrefixPatch model
				addressPrefixPatchModel := new(vpcclassicv1.AddressPrefixPatch)
				addressPrefixPatchModel.IsDefault = core.BoolPtr(false)
				addressPrefixPatchModel.Name = core.StringPtr("my-address-prefix-2")
				addressPrefixPatchModelAsPatch, asPatchErr := addressPrefixPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCAddressPrefixOptions model
				updateVPCAddressPrefixOptionsModel := new(vpcclassicv1.UpdateVPCAddressPrefixOptions)
				updateVPCAddressPrefixOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.ID = core.StringPtr("testString")
				updateVPCAddressPrefixOptionsModel.AddressPrefixPatch = addressPrefixPatchModelAsPatch
				updateVPCAddressPrefixOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.UpdateVPCAddressPrefix(updateVPCAddressPrefixOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateVPCAddressPrefixOptions model with no property values
				updateVPCAddressPrefixOptionsModelNew := new(vpcclassicv1.UpdateVPCAddressPrefixOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.UpdateVPCAddressPrefix(updateVPCAddressPrefixOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListVPCRoutes(listVPCRoutesOptions *ListVPCRoutesOptions) - Operation response error`, func() {
		version := "testString"
		listVPCRoutesPath := "/vpcs/testString/routes"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCRoutesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["zone.name"]).To(Equal([]string{"testString"}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListVPCRoutes with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListVPCRoutesOptions model
				listVPCRoutesOptionsModel := new(vpcclassicv1.ListVPCRoutesOptions)
				listVPCRoutesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutesOptionsModel.ZoneName = core.StringPtr("testString")
				listVPCRoutesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListVPCRoutes(listVPCRoutesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListVPCRoutes(listVPCRoutesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListVPCRoutes(listVPCRoutesOptions *ListVPCRoutesOptions)`, func() {
		version := "testString"
		listVPCRoutesPath := "/vpcs/testString/routes"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCRoutesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["zone.name"]).To(Equal([]string{"testString"}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"routes": [{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}]}`)
				}))
			})
			It(`Invoke ListVPCRoutes successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListVPCRoutesOptions model
				listVPCRoutesOptionsModel := new(vpcclassicv1.ListVPCRoutesOptions)
				listVPCRoutesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutesOptionsModel.ZoneName = core.StringPtr("testString")
				listVPCRoutesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListVPCRoutesWithContext(ctx, listVPCRoutesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListVPCRoutes(listVPCRoutesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListVPCRoutesWithContext(ctx, listVPCRoutesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVPCRoutesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["zone.name"]).To(Equal([]string{"testString"}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"routes": [{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}]}`)
				}))
			})
			It(`Invoke ListVPCRoutes successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListVPCRoutes(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListVPCRoutesOptions model
				listVPCRoutesOptionsModel := new(vpcclassicv1.ListVPCRoutesOptions)
				listVPCRoutesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutesOptionsModel.ZoneName = core.StringPtr("testString")
				listVPCRoutesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListVPCRoutes(listVPCRoutesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListVPCRoutes with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListVPCRoutesOptions model
				listVPCRoutesOptionsModel := new(vpcclassicv1.ListVPCRoutesOptions)
				listVPCRoutesOptionsModel.VPCID = core.StringPtr("testString")
				listVPCRoutesOptionsModel.ZoneName = core.StringPtr("testString")
				listVPCRoutesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListVPCRoutes(listVPCRoutesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListVPCRoutesOptions model with no property values
				listVPCRoutesOptionsModelNew := new(vpcclassicv1.ListVPCRoutesOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.ListVPCRoutes(listVPCRoutesOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateVPCRoute(createVPCRouteOptions *CreateVPCRouteOptions) - Operation response error`, func() {
		version := "testString"
		createVPCRoutePath := "/vpcs/testString/routes"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCRoutePath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateVPCRoute with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcclassicv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the CreateVPCRouteOptions model
				createVPCRouteOptionsModel := new(vpcclassicv1.CreateVPCRouteOptions)
				createVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRouteOptionsModel.Destination = core.StringPtr("192.168.3.0/24")
				createVPCRouteOptionsModel.NextHop = routeNextHopPrototypeModel
				createVPCRouteOptionsModel.Zone = zoneIdentityModel
				createVPCRouteOptionsModel.Name = core.StringPtr("my-route-2")
				createVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.CreateVPCRoute(createVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.CreateVPCRoute(createVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`CreateVPCRoute(createVPCRouteOptions *CreateVPCRouteOptions)`, func() {
		version := "testString"
		createVPCRoutePath := "/vpcs/testString/routes"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCRoutePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateVPCRoute successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcclassicv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the CreateVPCRouteOptions model
				createVPCRouteOptionsModel := new(vpcclassicv1.CreateVPCRouteOptions)
				createVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRouteOptionsModel.Destination = core.StringPtr("192.168.3.0/24")
				createVPCRouteOptionsModel.NextHop = routeNextHopPrototypeModel
				createVPCRouteOptionsModel.Zone = zoneIdentityModel
				createVPCRouteOptionsModel.Name = core.StringPtr("my-route-2")
				createVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.CreateVPCRouteWithContext(ctx, createVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.CreateVPCRoute(createVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.CreateVPCRouteWithContext(ctx, createVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVPCRoutePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateVPCRoute successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.CreateVPCRoute(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcclassicv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the CreateVPCRouteOptions model
				createVPCRouteOptionsModel := new(vpcclassicv1.CreateVPCRouteOptions)
				createVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRouteOptionsModel.Destination = core.StringPtr("192.168.3.0/24")
				createVPCRouteOptionsModel.NextHop = routeNextHopPrototypeModel
				createVPCRouteOptionsModel.Zone = zoneIdentityModel
				createVPCRouteOptionsModel.Name = core.StringPtr("my-route-2")
				createVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.CreateVPCRoute(createVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateVPCRoute with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the RouteNextHopPrototypeRouteNextHopIP model
				routeNextHopPrototypeModel := new(vpcclassicv1.RouteNextHopPrototypeRouteNextHopIP)
				routeNextHopPrototypeModel.Address = core.StringPtr("192.168.3.4")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the CreateVPCRouteOptions model
				createVPCRouteOptionsModel := new(vpcclassicv1.CreateVPCRouteOptions)
				createVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				createVPCRouteOptionsModel.Destination = core.StringPtr("192.168.3.0/24")
				createVPCRouteOptionsModel.NextHop = routeNextHopPrototypeModel
				createVPCRouteOptionsModel.Zone = zoneIdentityModel
				createVPCRouteOptionsModel.Name = core.StringPtr("my-route-2")
				createVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.CreateVPCRoute(createVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateVPCRouteOptions model with no property values
				createVPCRouteOptionsModelNew := new(vpcclassicv1.CreateVPCRouteOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.CreateVPCRoute(createVPCRouteOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`DeleteVPCRoute(deleteVPCRouteOptions *DeleteVPCRouteOptions)`, func() {
		version := "testString"
		deleteVPCRoutePath := "/vpcs/testString/routes/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteVPCRoutePath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteVPCRoute successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcClassicService.DeleteVPCRoute(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteVPCRouteOptions model
				deleteVPCRouteOptionsModel := new(vpcclassicv1.DeleteVPCRouteOptions)
				deleteVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				deleteVPCRouteOptionsModel.ID = core.StringPtr("testString")
				deleteVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcClassicService.DeleteVPCRoute(deleteVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteVPCRoute with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the DeleteVPCRouteOptions model
				deleteVPCRouteOptionsModel := new(vpcclassicv1.DeleteVPCRouteOptions)
				deleteVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				deleteVPCRouteOptionsModel.ID = core.StringPtr("testString")
				deleteVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcClassicService.DeleteVPCRoute(deleteVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteVPCRouteOptions model with no property values
				deleteVPCRouteOptionsModelNew := new(vpcclassicv1.DeleteVPCRouteOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcClassicService.DeleteVPCRoute(deleteVPCRouteOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVPCRoute(getVPCRouteOptions *GetVPCRouteOptions) - Operation response error`, func() {
		version := "testString"
		getVPCRoutePath := "/vpcs/testString/routes/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCRoutePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetVPCRoute with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetVPCRouteOptions model
				getVPCRouteOptionsModel := new(vpcclassicv1.GetVPCRouteOptions)
				getVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRouteOptionsModel.ID = core.StringPtr("testString")
				getVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetVPCRoute(getVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetVPCRoute(getVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetVPCRoute(getVPCRouteOptions *GetVPCRouteOptions)`, func() {
		version := "testString"
		getVPCRoutePath := "/vpcs/testString/routes/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCRoutePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetVPCRoute successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetVPCRouteOptions model
				getVPCRouteOptionsModel := new(vpcclassicv1.GetVPCRouteOptions)
				getVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRouteOptionsModel.ID = core.StringPtr("testString")
				getVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetVPCRouteWithContext(ctx, getVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetVPCRoute(getVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetVPCRouteWithContext(ctx, getVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVPCRoutePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetVPCRoute successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetVPCRoute(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetVPCRouteOptions model
				getVPCRouteOptionsModel := new(vpcclassicv1.GetVPCRouteOptions)
				getVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRouteOptionsModel.ID = core.StringPtr("testString")
				getVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetVPCRoute(getVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetVPCRoute with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetVPCRouteOptions model
				getVPCRouteOptionsModel := new(vpcclassicv1.GetVPCRouteOptions)
				getVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				getVPCRouteOptionsModel.ID = core.StringPtr("testString")
				getVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetVPCRoute(getVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetVPCRouteOptions model with no property values
				getVPCRouteOptionsModelNew := new(vpcclassicv1.GetVPCRouteOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetVPCRoute(getVPCRouteOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateVPCRoute(updateVPCRouteOptions *UpdateVPCRouteOptions) - Operation response error`, func() {
		version := "testString"
		updateVPCRoutePath := "/vpcs/testString/routes/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCRoutePath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateVPCRoute with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the RoutePatch model
				routePatchModel := new(vpcclassicv1.RoutePatch)
				routePatchModel.Name = core.StringPtr("my-route-2")
				routePatchModelAsPatch, asPatchErr := routePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRouteOptions model
				updateVPCRouteOptionsModel := new(vpcclassicv1.UpdateVPCRouteOptions)
				updateVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.ID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.RoutePatch = routePatchModelAsPatch
				updateVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.UpdateVPCRoute(updateVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.UpdateVPCRoute(updateVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`UpdateVPCRoute(updateVPCRouteOptions *UpdateVPCRouteOptions)`, func() {
		version := "testString"
		updateVPCRoutePath := "/vpcs/testString/routes/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCRoutePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateVPCRoute successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the RoutePatch model
				routePatchModel := new(vpcclassicv1.RoutePatch)
				routePatchModel.Name = core.StringPtr("my-route-2")
				routePatchModelAsPatch, asPatchErr := routePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRouteOptions model
				updateVPCRouteOptionsModel := new(vpcclassicv1.UpdateVPCRouteOptions)
				updateVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.ID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.RoutePatch = routePatchModelAsPatch
				updateVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.UpdateVPCRouteWithContext(ctx, updateVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.UpdateVPCRoute(updateVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.UpdateVPCRouteWithContext(ctx, updateVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVPCRoutePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/8e454ead-0db7-48ac-9a8b-2698d8c470a7/routes/1a15dca5-7e33-45e1-b7c5-bc690e569531", "id": "1a15dca5-7e33-45e1-b7c5-bc690e569531", "lifecycle_state": "stable", "name": "my-route-1", "next_hop": {"address": "192.168.3.4"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateVPCRoute successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.UpdateVPCRoute(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the RoutePatch model
				routePatchModel := new(vpcclassicv1.RoutePatch)
				routePatchModel.Name = core.StringPtr("my-route-2")
				routePatchModelAsPatch, asPatchErr := routePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRouteOptions model
				updateVPCRouteOptionsModel := new(vpcclassicv1.UpdateVPCRouteOptions)
				updateVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.ID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.RoutePatch = routePatchModelAsPatch
				updateVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.UpdateVPCRoute(updateVPCRouteOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateVPCRoute with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the RoutePatch model
				routePatchModel := new(vpcclassicv1.RoutePatch)
				routePatchModel.Name = core.StringPtr("my-route-2")
				routePatchModelAsPatch, asPatchErr := routePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVPCRouteOptions model
				updateVPCRouteOptionsModel := new(vpcclassicv1.UpdateVPCRouteOptions)
				updateVPCRouteOptionsModel.VPCID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.ID = core.StringPtr("testString")
				updateVPCRouteOptionsModel.RoutePatch = routePatchModelAsPatch
				updateVPCRouteOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.UpdateVPCRoute(updateVPCRouteOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateVPCRouteOptions model with no property values
				updateVPCRouteOptionsModelNew := new(vpcclassicv1.UpdateVPCRouteOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.UpdateVPCRoute(updateVPCRouteOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`Service constructor tests`, func() {
		version := "testString"
		It(`Instantiate service client`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				Authenticator: &core.NoAuthAuthenticator{},
				Version:       core.StringPtr(version),
			})
			Expect(vpcClassicService).ToNot(BeNil())
			Expect(serviceErr).To(BeNil())
		})
		It(`Instantiate service client with error: Invalid URL`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Invalid Auth`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "https://vpcclassicv1/api",
				Version: core.StringPtr(version),
				Authenticator: &core.BasicAuthenticator{
					Username: "",
					Password: "",
				},
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Validation Error`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
	})
	Describe(`Service constructor tests using external config`, func() {
		version := "testString"
		Context(`Using external config, construct service client instances`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "noauth",
			}

			It(`Create service client using external config successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url from constructor successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					URL:     "https://testService/api",
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url programatically successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				err := vpcClassicService.SetServiceURL("https://testService/api")
				Expect(err).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid Auth`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "someOtherAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid URL`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_AUTH_TYPE": "NOAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
	})
	Describe(`Regional endpoint tests`, func() {
		It(`GetServiceURLForRegion(region string)`, func() {
			var url string
			var err error
			url, err = vpcclassicv1.GetServiceURLForRegion("INVALID_REGION")
			Expect(url).To(BeEmpty())
			Expect(err).ToNot(BeNil())
			fmt.Fprintf(GinkgoWriter, "Expected error: %s\n", err.Error())
		})
	})
	Describe(`ListSubnets(listSubnetsOptions *ListSubnetsOptions) - Operation response error`, func() {
		version := "testString"
		listSubnetsPath := "/subnets"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listSubnetsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListSubnets with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListSubnetsOptions model
				listSubnetsOptionsModel := new(vpcclassicv1.ListSubnetsOptions)
				listSubnetsOptionsModel.Start = core.StringPtr("testString")
				listSubnetsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listSubnetsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListSubnets(listSubnetsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListSubnets(listSubnetsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListSubnets(listSubnetsOptions *ListSubnetsOptions)`, func() {
		version := "testString"
		listSubnetsPath := "/subnets"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listSubnetsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/subnets?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/subnets?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "subnets": [{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}]}`)
				}))
			})
			It(`Invoke ListSubnets successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListSubnetsOptions model
				listSubnetsOptionsModel := new(vpcclassicv1.ListSubnetsOptions)
				listSubnetsOptionsModel.Start = core.StringPtr("testString")
				listSubnetsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listSubnetsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListSubnetsWithContext(ctx, listSubnetsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListSubnets(listSubnetsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListSubnetsWithContext(ctx, listSubnetsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listSubnetsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/subnets?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/subnets?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "subnets": [{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}]}`)
				}))
			})
			It(`Invoke ListSubnets successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListSubnets(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListSubnetsOptions model
				listSubnetsOptionsModel := new(vpcclassicv1.ListSubnetsOptions)
				listSubnetsOptionsModel.Start = core.StringPtr("testString")
				listSubnetsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listSubnetsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListSubnets(listSubnetsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListSubnets with error: Operation request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListSubnetsOptions model
				listSubnetsOptionsModel := new(vpcclassicv1.ListSubnetsOptions)
				listSubnetsOptionsModel.Start = core.StringPtr("testString")
				listSubnetsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listSubnetsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListSubnets(listSubnetsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateSubnet(createSubnetOptions *CreateSubnetOptions) - Operation response error`, func() {
		version := "testString"
		createSubnetPath := "/subnets"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createSubnetPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateSubnet with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcclassicv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcclassicv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcclassicv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("4727d842-f94f-4a2d-824a-9bc9b02c523b")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the SubnetPrototypeSubnetByTotalCount model
				subnetPrototypeModel := new(vpcclassicv1.SubnetPrototypeSubnetByTotalCount)
				subnetPrototypeModel.Name = core.StringPtr("my-subnet")
				subnetPrototypeModel.NetworkACL = networkACLIdentityModel
				subnetPrototypeModel.PublicGateway = publicGatewayIdentityModel
				subnetPrototypeModel.VPC = vpcIdentityModel
				subnetPrototypeModel.TotalIpv4AddressCount = core.Int64Ptr(int64(256))
				subnetPrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateSubnetOptions model
				createSubnetOptionsModel := new(vpcclassicv1.CreateSubnetOptions)
				createSubnetOptionsModel.SubnetPrototype = subnetPrototypeModel
				createSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.CreateSubnet(createSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.CreateSubnet(createSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`CreateSubnet(createSubnetOptions *CreateSubnetOptions)`, func() {
		version := "testString"
		createSubnetPath := "/subnets"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createSubnetPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateSubnet successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcclassicv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcclassicv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcclassicv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("4727d842-f94f-4a2d-824a-9bc9b02c523b")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the SubnetPrototypeSubnetByTotalCount model
				subnetPrototypeModel := new(vpcclassicv1.SubnetPrototypeSubnetByTotalCount)
				subnetPrototypeModel.Name = core.StringPtr("my-subnet")
				subnetPrototypeModel.NetworkACL = networkACLIdentityModel
				subnetPrototypeModel.PublicGateway = publicGatewayIdentityModel
				subnetPrototypeModel.VPC = vpcIdentityModel
				subnetPrototypeModel.TotalIpv4AddressCount = core.Int64Ptr(int64(256))
				subnetPrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateSubnetOptions model
				createSubnetOptionsModel := new(vpcclassicv1.CreateSubnetOptions)
				createSubnetOptionsModel.SubnetPrototype = subnetPrototypeModel
				createSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.CreateSubnetWithContext(ctx, createSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.CreateSubnet(createSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.CreateSubnetWithContext(ctx, createSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createSubnetPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateSubnet successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.CreateSubnet(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcclassicv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcclassicv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcclassicv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("4727d842-f94f-4a2d-824a-9bc9b02c523b")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the SubnetPrototypeSubnetByTotalCount model
				subnetPrototypeModel := new(vpcclassicv1.SubnetPrototypeSubnetByTotalCount)
				subnetPrototypeModel.Name = core.StringPtr("my-subnet")
				subnetPrototypeModel.NetworkACL = networkACLIdentityModel
				subnetPrototypeModel.PublicGateway = publicGatewayIdentityModel
				subnetPrototypeModel.VPC = vpcIdentityModel
				subnetPrototypeModel.TotalIpv4AddressCount = core.Int64Ptr(int64(256))
				subnetPrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateSubnetOptions model
				createSubnetOptionsModel := new(vpcclassicv1.CreateSubnetOptions)
				createSubnetOptionsModel.SubnetPrototype = subnetPrototypeModel
				createSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.CreateSubnet(createSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateSubnet with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcclassicv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcclassicv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcclassicv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("4727d842-f94f-4a2d-824a-9bc9b02c523b")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the SubnetPrototypeSubnetByTotalCount model
				subnetPrototypeModel := new(vpcclassicv1.SubnetPrototypeSubnetByTotalCount)
				subnetPrototypeModel.Name = core.StringPtr("my-subnet")
				subnetPrototypeModel.NetworkACL = networkACLIdentityModel
				subnetPrototypeModel.PublicGateway = publicGatewayIdentityModel
				subnetPrototypeModel.VPC = vpcIdentityModel
				subnetPrototypeModel.TotalIpv4AddressCount = core.Int64Ptr(int64(256))
				subnetPrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateSubnetOptions model
				createSubnetOptionsModel := new(vpcclassicv1.CreateSubnetOptions)
				createSubnetOptionsModel.SubnetPrototype = subnetPrototypeModel
				createSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.CreateSubnet(createSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateSubnetOptions model with no property values
				createSubnetOptionsModelNew := new(vpcclassicv1.CreateSubnetOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.CreateSubnet(createSubnetOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`DeleteSubnet(deleteSubnetOptions *DeleteSubnetOptions)`, func() {
		version := "testString"
		deleteSubnetPath := "/subnets/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteSubnetPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteSubnet successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcClassicService.DeleteSubnet(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteSubnetOptions model
				deleteSubnetOptionsModel := new(vpcclassicv1.DeleteSubnetOptions)
				deleteSubnetOptionsModel.ID = core.StringPtr("testString")
				deleteSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcClassicService.DeleteSubnet(deleteSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteSubnet with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the DeleteSubnetOptions model
				deleteSubnetOptionsModel := new(vpcclassicv1.DeleteSubnetOptions)
				deleteSubnetOptionsModel.ID = core.StringPtr("testString")
				deleteSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcClassicService.DeleteSubnet(deleteSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteSubnetOptions model with no property values
				deleteSubnetOptionsModelNew := new(vpcclassicv1.DeleteSubnetOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcClassicService.DeleteSubnet(deleteSubnetOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetSubnet(getSubnetOptions *GetSubnetOptions) - Operation response error`, func() {
		version := "testString"
		getSubnetPath := "/subnets/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetSubnet with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetSubnetOptions model
				getSubnetOptionsModel := new(vpcclassicv1.GetSubnetOptions)
				getSubnetOptionsModel.ID = core.StringPtr("testString")
				getSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetSubnet(getSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetSubnet(getSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetSubnet(getSubnetOptions *GetSubnetOptions)`, func() {
		version := "testString"
		getSubnetPath := "/subnets/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetSubnet successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetSubnetOptions model
				getSubnetOptionsModel := new(vpcclassicv1.GetSubnetOptions)
				getSubnetOptionsModel.ID = core.StringPtr("testString")
				getSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetSubnetWithContext(ctx, getSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetSubnet(getSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetSubnetWithContext(ctx, getSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetSubnet successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetSubnet(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetSubnetOptions model
				getSubnetOptionsModel := new(vpcclassicv1.GetSubnetOptions)
				getSubnetOptionsModel.ID = core.StringPtr("testString")
				getSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetSubnet(getSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetSubnet with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetSubnetOptions model
				getSubnetOptionsModel := new(vpcclassicv1.GetSubnetOptions)
				getSubnetOptionsModel.ID = core.StringPtr("testString")
				getSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetSubnet(getSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetSubnetOptions model with no property values
				getSubnetOptionsModelNew := new(vpcclassicv1.GetSubnetOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetSubnet(getSubnetOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateSubnet(updateSubnetOptions *UpdateSubnetOptions) - Operation response error`, func() {
		version := "testString"
		updateSubnetPath := "/subnets/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateSubnetPath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateSubnet with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcclassicv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcclassicv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the SubnetPatch model
				subnetPatchModel := new(vpcclassicv1.SubnetPatch)
				subnetPatchModel.Name = core.StringPtr("my-subnet")
				subnetPatchModel.NetworkACL = networkACLIdentityModel
				subnetPatchModel.PublicGateway = publicGatewayIdentityModel
				subnetPatchModelAsPatch, asPatchErr := subnetPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateSubnetOptions model
				updateSubnetOptionsModel := new(vpcclassicv1.UpdateSubnetOptions)
				updateSubnetOptionsModel.ID = core.StringPtr("testString")
				updateSubnetOptionsModel.SubnetPatch = subnetPatchModelAsPatch
				updateSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.UpdateSubnet(updateSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.UpdateSubnet(updateSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`UpdateSubnet(updateSubnetOptions *UpdateSubnetOptions)`, func() {
		version := "testString"
		updateSubnetPath := "/subnets/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateSubnetPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateSubnet successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcclassicv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcclassicv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the SubnetPatch model
				subnetPatchModel := new(vpcclassicv1.SubnetPatch)
				subnetPatchModel.Name = core.StringPtr("my-subnet")
				subnetPatchModel.NetworkACL = networkACLIdentityModel
				subnetPatchModel.PublicGateway = publicGatewayIdentityModel
				subnetPatchModelAsPatch, asPatchErr := subnetPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateSubnetOptions model
				updateSubnetOptionsModel := new(vpcclassicv1.UpdateSubnetOptions)
				updateSubnetOptionsModel.ID = core.StringPtr("testString")
				updateSubnetOptionsModel.SubnetPatch = subnetPatchModelAsPatch
				updateSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.UpdateSubnetWithContext(ctx, updateSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.UpdateSubnet(updateSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.UpdateSubnetWithContext(ctx, updateSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateSubnetPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"available_ipv4_address_count": 15, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "ipv4_cidr_block": "10.0.0.0/24", "name": "my-subnet", "network_acl": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl"}, "public_gateway": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway"}, "status": "available", "total_ipv4_address_count": 256, "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateSubnet successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.UpdateSubnet(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcclassicv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcclassicv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the SubnetPatch model
				subnetPatchModel := new(vpcclassicv1.SubnetPatch)
				subnetPatchModel.Name = core.StringPtr("my-subnet")
				subnetPatchModel.NetworkACL = networkACLIdentityModel
				subnetPatchModel.PublicGateway = publicGatewayIdentityModel
				subnetPatchModelAsPatch, asPatchErr := subnetPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateSubnetOptions model
				updateSubnetOptionsModel := new(vpcclassicv1.UpdateSubnetOptions)
				updateSubnetOptionsModel.ID = core.StringPtr("testString")
				updateSubnetOptionsModel.SubnetPatch = subnetPatchModelAsPatch
				updateSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.UpdateSubnet(updateSubnetOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateSubnet with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcclassicv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcclassicv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the SubnetPatch model
				subnetPatchModel := new(vpcclassicv1.SubnetPatch)
				subnetPatchModel.Name = core.StringPtr("my-subnet")
				subnetPatchModel.NetworkACL = networkACLIdentityModel
				subnetPatchModel.PublicGateway = publicGatewayIdentityModel
				subnetPatchModelAsPatch, asPatchErr := subnetPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateSubnetOptions model
				updateSubnetOptionsModel := new(vpcclassicv1.UpdateSubnetOptions)
				updateSubnetOptionsModel.ID = core.StringPtr("testString")
				updateSubnetOptionsModel.SubnetPatch = subnetPatchModelAsPatch
				updateSubnetOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.UpdateSubnet(updateSubnetOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateSubnetOptions model with no property values
				updateSubnetOptionsModelNew := new(vpcclassicv1.UpdateSubnetOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.UpdateSubnet(updateSubnetOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetSubnetNetworkACL(getSubnetNetworkACLOptions *GetSubnetNetworkACLOptions) - Operation response error`, func() {
		version := "testString"
		getSubnetNetworkACLPath := "/subnets/testString/network_acl"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetNetworkACLPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetSubnetNetworkACL with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetSubnetNetworkACLOptions model
				getSubnetNetworkACLOptionsModel := new(vpcclassicv1.GetSubnetNetworkACLOptions)
				getSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetSubnetNetworkACL(getSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetSubnetNetworkACL(getSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetSubnetNetworkACL(getSubnetNetworkACLOptions *GetSubnetNetworkACLOptions)`, func() {
		version := "testString"
		getSubnetNetworkACLPath := "/subnets/testString/network_acl"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetNetworkACLPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "rules": [{"action": "allow", "before": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "port_max": 22, "port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke GetSubnetNetworkACL successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetSubnetNetworkACLOptions model
				getSubnetNetworkACLOptionsModel := new(vpcclassicv1.GetSubnetNetworkACLOptions)
				getSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetSubnetNetworkACLWithContext(ctx, getSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetSubnetNetworkACL(getSubnetNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetSubnetNetworkACLWithContext(ctx, getSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetNetworkACLPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "rules": [{"action": "allow", "before": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "port_max": 22, "port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke GetSubnetNetworkACL successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetSubnetNetworkACL(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetSubnetNetworkACLOptions model
				getSubnetNetworkACLOptionsModel := new(vpcclassicv1.GetSubnetNetworkACLOptions)
				getSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetSubnetNetworkACL(getSubnetNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetSubnetNetworkACL with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetSubnetNetworkACLOptions model
				getSubnetNetworkACLOptionsModel := new(vpcclassicv1.GetSubnetNetworkACLOptions)
				getSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetSubnetNetworkACL(getSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetSubnetNetworkACLOptions model with no property values
				getSubnetNetworkACLOptionsModelNew := new(vpcclassicv1.GetSubnetNetworkACLOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetSubnetNetworkACL(getSubnetNetworkACLOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptions *ReplaceSubnetNetworkACLOptions) - Operation response error`, func() {
		version := "testString"
		replaceSubnetNetworkACLPath := "/subnets/testString/network_acl"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceSubnetNetworkACLPath))
					Expect(req.Method).To(Equal("PUT"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ReplaceSubnetNetworkACL with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcclassicv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the ReplaceSubnetNetworkACLOptions model
				replaceSubnetNetworkACLOptionsModel := new(vpcclassicv1.ReplaceSubnetNetworkACLOptions)
				replaceSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				replaceSubnetNetworkACLOptionsModel.NetworkACLIdentity = networkACLIdentityModel
				replaceSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptions *ReplaceSubnetNetworkACLOptions)`, func() {
		version := "testString"
		replaceSubnetNetworkACLPath := "/subnets/testString/network_acl"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceSubnetNetworkACLPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "rules": [{"action": "allow", "before": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "port_max": 22, "port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke ReplaceSubnetNetworkACL successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcclassicv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the ReplaceSubnetNetworkACLOptions model
				replaceSubnetNetworkACLOptionsModel := new(vpcclassicv1.ReplaceSubnetNetworkACLOptions)
				replaceSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				replaceSubnetNetworkACLOptionsModel.NetworkACLIdentity = networkACLIdentityModel
				replaceSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ReplaceSubnetNetworkACLWithContext(ctx, replaceSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ReplaceSubnetNetworkACLWithContext(ctx, replaceSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(replaceSubnetNetworkACLPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "rules": [{"action": "allow", "before": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "port_max": 22, "port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke ReplaceSubnetNetworkACL successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ReplaceSubnetNetworkACL(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcclassicv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the ReplaceSubnetNetworkACLOptions model
				replaceSubnetNetworkACLOptionsModel := new(vpcclassicv1.ReplaceSubnetNetworkACLOptions)
				replaceSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				replaceSubnetNetworkACLOptionsModel.NetworkACLIdentity = networkACLIdentityModel
				replaceSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ReplaceSubnetNetworkACL with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the NetworkACLIdentityByID model
				networkACLIdentityModel := new(vpcclassicv1.NetworkACLIdentityByID)
				networkACLIdentityModel.ID = core.StringPtr("a4e28308-8ee7-46ab-8108-9f881f22bdbf")

				// Construct an instance of the ReplaceSubnetNetworkACLOptions model
				replaceSubnetNetworkACLOptionsModel := new(vpcclassicv1.ReplaceSubnetNetworkACLOptions)
				replaceSubnetNetworkACLOptionsModel.ID = core.StringPtr("testString")
				replaceSubnetNetworkACLOptionsModel.NetworkACLIdentity = networkACLIdentityModel
				replaceSubnetNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ReplaceSubnetNetworkACLOptions model with no property values
				replaceSubnetNetworkACLOptionsModelNew := new(vpcclassicv1.ReplaceSubnetNetworkACLOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`UnsetSubnetPublicGateway(unsetSubnetPublicGatewayOptions *UnsetSubnetPublicGatewayOptions)`, func() {
		version := "testString"
		unsetSubnetPublicGatewayPath := "/subnets/testString/public_gateway"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(unsetSubnetPublicGatewayPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.WriteHeader(204)
				}))
			})
			It(`Invoke UnsetSubnetPublicGateway successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcClassicService.UnsetSubnetPublicGateway(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the UnsetSubnetPublicGatewayOptions model
				unsetSubnetPublicGatewayOptionsModel := new(vpcclassicv1.UnsetSubnetPublicGatewayOptions)
				unsetSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				unsetSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcClassicService.UnsetSubnetPublicGateway(unsetSubnetPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke UnsetSubnetPublicGateway with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the UnsetSubnetPublicGatewayOptions model
				unsetSubnetPublicGatewayOptionsModel := new(vpcclassicv1.UnsetSubnetPublicGatewayOptions)
				unsetSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				unsetSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcClassicService.UnsetSubnetPublicGateway(unsetSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the UnsetSubnetPublicGatewayOptions model with no property values
				unsetSubnetPublicGatewayOptionsModelNew := new(vpcclassicv1.UnsetSubnetPublicGatewayOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcClassicService.UnsetSubnetPublicGateway(unsetSubnetPublicGatewayOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetSubnetPublicGateway(getSubnetPublicGatewayOptions *GetSubnetPublicGatewayOptions) - Operation response error`, func() {
		version := "testString"
		getSubnetPublicGatewayPath := "/subnets/testString/public_gateway"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetPublicGatewayPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetSubnetPublicGateway with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetSubnetPublicGatewayOptions model
				getSubnetPublicGatewayOptionsModel := new(vpcclassicv1.GetSubnetPublicGatewayOptions)
				getSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				getSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetSubnetPublicGateway(getSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetSubnetPublicGateway(getSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetSubnetPublicGateway(getSubnetPublicGatewayOptions *GetSubnetPublicGatewayOptions)`, func() {
		version := "testString"
		getSubnetPublicGatewayPath := "/subnets/testString/public_gateway"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetPublicGatewayPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetSubnetPublicGateway successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetSubnetPublicGatewayOptions model
				getSubnetPublicGatewayOptionsModel := new(vpcclassicv1.GetSubnetPublicGatewayOptions)
				getSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				getSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetSubnetPublicGatewayWithContext(ctx, getSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetSubnetPublicGateway(getSubnetPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetSubnetPublicGatewayWithContext(ctx, getSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getSubnetPublicGatewayPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetSubnetPublicGateway successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetSubnetPublicGateway(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetSubnetPublicGatewayOptions model
				getSubnetPublicGatewayOptionsModel := new(vpcclassicv1.GetSubnetPublicGatewayOptions)
				getSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				getSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetSubnetPublicGateway(getSubnetPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetSubnetPublicGateway with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetSubnetPublicGatewayOptions model
				getSubnetPublicGatewayOptionsModel := new(vpcclassicv1.GetSubnetPublicGatewayOptions)
				getSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				getSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetSubnetPublicGateway(getSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetSubnetPublicGatewayOptions model with no property values
				getSubnetPublicGatewayOptionsModelNew := new(vpcclassicv1.GetSubnetPublicGatewayOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetSubnetPublicGateway(getSubnetPublicGatewayOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`SetSubnetPublicGateway(setSubnetPublicGatewayOptions *SetSubnetPublicGatewayOptions) - Operation response error`, func() {
		version := "testString"
		setSubnetPublicGatewayPath := "/subnets/testString/public_gateway"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(setSubnetPublicGatewayPath))
					Expect(req.Method).To(Equal("PUT"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke SetSubnetPublicGateway with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcclassicv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the SetSubnetPublicGatewayOptions model
				setSubnetPublicGatewayOptionsModel := new(vpcclassicv1.SetSubnetPublicGatewayOptions)
				setSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				setSubnetPublicGatewayOptionsModel.PublicGatewayIdentity = publicGatewayIdentityModel
				setSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.SetSubnetPublicGateway(setSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.SetSubnetPublicGateway(setSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`SetSubnetPublicGateway(setSubnetPublicGatewayOptions *SetSubnetPublicGatewayOptions)`, func() {
		version := "testString"
		setSubnetPublicGatewayPath := "/subnets/testString/public_gateway"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(setSubnetPublicGatewayPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke SetSubnetPublicGateway successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcclassicv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the SetSubnetPublicGatewayOptions model
				setSubnetPublicGatewayOptionsModel := new(vpcclassicv1.SetSubnetPublicGatewayOptions)
				setSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				setSubnetPublicGatewayOptionsModel.PublicGatewayIdentity = publicGatewayIdentityModel
				setSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.SetSubnetPublicGatewayWithContext(ctx, setSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.SetSubnetPublicGateway(setSubnetPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.SetSubnetPublicGatewayWithContext(ctx, setSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(setSubnetPublicGatewayPath))
					Expect(req.Method).To(Equal("PUT"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke SetSubnetPublicGateway successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.SetSubnetPublicGateway(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcclassicv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the SetSubnetPublicGatewayOptions model
				setSubnetPublicGatewayOptionsModel := new(vpcclassicv1.SetSubnetPublicGatewayOptions)
				setSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				setSubnetPublicGatewayOptionsModel.PublicGatewayIdentity = publicGatewayIdentityModel
				setSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.SetSubnetPublicGateway(setSubnetPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke SetSubnetPublicGateway with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the PublicGatewayIdentityByID model
				publicGatewayIdentityModel := new(vpcclassicv1.PublicGatewayIdentityByID)
				publicGatewayIdentityModel.ID = core.StringPtr("dc5431ef-1fc6-4861-adc9-a59d077d1241")

				// Construct an instance of the SetSubnetPublicGatewayOptions model
				setSubnetPublicGatewayOptionsModel := new(vpcclassicv1.SetSubnetPublicGatewayOptions)
				setSubnetPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				setSubnetPublicGatewayOptionsModel.PublicGatewayIdentity = publicGatewayIdentityModel
				setSubnetPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.SetSubnetPublicGateway(setSubnetPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the SetSubnetPublicGatewayOptions model with no property values
				setSubnetPublicGatewayOptionsModelNew := new(vpcclassicv1.SetSubnetPublicGatewayOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.SetSubnetPublicGateway(setSubnetPublicGatewayOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`Service constructor tests`, func() {
		version := "testString"
		It(`Instantiate service client`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				Authenticator: &core.NoAuthAuthenticator{},
				Version:       core.StringPtr(version),
			})
			Expect(vpcClassicService).ToNot(BeNil())
			Expect(serviceErr).To(BeNil())
		})
		It(`Instantiate service client with error: Invalid URL`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Invalid Auth`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "https://vpcclassicv1/api",
				Version: core.StringPtr(version),
				Authenticator: &core.BasicAuthenticator{
					Username: "",
					Password: "",
				},
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Validation Error`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
	})
	Describe(`Service constructor tests using external config`, func() {
		version := "testString"
		Context(`Using external config, construct service client instances`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "noauth",
			}

			It(`Create service client using external config successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url from constructor successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					URL:     "https://testService/api",
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url programatically successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				err := vpcClassicService.SetServiceURL("https://testService/api")
				Expect(err).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid Auth`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "someOtherAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid URL`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_AUTH_TYPE": "NOAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
	})
	Describe(`Regional endpoint tests`, func() {
		It(`GetServiceURLForRegion(region string)`, func() {
			var url string
			var err error
			url, err = vpcclassicv1.GetServiceURLForRegion("INVALID_REGION")
			Expect(url).To(BeEmpty())
			Expect(err).ToNot(BeNil())
			fmt.Fprintf(GinkgoWriter, "Expected error: %s\n", err.Error())
		})
	})
	Describe(`ListImages(listImagesOptions *ListImagesOptions) - Operation response error`, func() {
		version := "testString"
		listImagesPath := "/images"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listImagesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["visibility"]).To(Equal([]string{"private"}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListImages with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListImagesOptions model
				listImagesOptionsModel := new(vpcclassicv1.ListImagesOptions)
				listImagesOptionsModel.Start = core.StringPtr("testString")
				listImagesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listImagesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listImagesOptionsModel.Name = core.StringPtr("testString")
				listImagesOptionsModel.Visibility = core.StringPtr("private")
				listImagesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListImages(listImagesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListImages(listImagesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListImages(listImagesOptions *ListImagesOptions)`, func() {
		version := "testString"
		listImagesPath := "/images"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listImagesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["visibility"]).To(Equal([]string{"private"}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/images?limit=20"}, "images": [{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "file": {"size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "visibility": "private"}], "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/images?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}}`)
				}))
			})
			It(`Invoke ListImages successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListImagesOptions model
				listImagesOptionsModel := new(vpcclassicv1.ListImagesOptions)
				listImagesOptionsModel.Start = core.StringPtr("testString")
				listImagesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listImagesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listImagesOptionsModel.Name = core.StringPtr("testString")
				listImagesOptionsModel.Visibility = core.StringPtr("private")
				listImagesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListImagesWithContext(ctx, listImagesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListImages(listImagesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListImagesWithContext(ctx, listImagesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listImagesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["resource_group.id"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["visibility"]).To(Equal([]string{"private"}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/images?limit=20"}, "images": [{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "file": {"size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "visibility": "private"}], "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/images?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}}`)
				}))
			})
			It(`Invoke ListImages successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListImages(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListImagesOptions model
				listImagesOptionsModel := new(vpcclassicv1.ListImagesOptions)
				listImagesOptionsModel.Start = core.StringPtr("testString")
				listImagesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listImagesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listImagesOptionsModel.Name = core.StringPtr("testString")
				listImagesOptionsModel.Visibility = core.StringPtr("private")
				listImagesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListImages(listImagesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListImages with error: Operation request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListImagesOptions model
				listImagesOptionsModel := new(vpcclassicv1.ListImagesOptions)
				listImagesOptionsModel.Start = core.StringPtr("testString")
				listImagesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listImagesOptionsModel.ResourceGroupID = core.StringPtr("testString")
				listImagesOptionsModel.Name = core.StringPtr("testString")
				listImagesOptionsModel.Visibility = core.StringPtr("private")
				listImagesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListImages(listImagesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateImage(createImageOptions *CreateImageOptions) - Operation response error`, func() {
		version := "testString"
		createImagePath := "/images"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createImagePath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateImage with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the ImageFilePrototype model
				imageFilePrototypeModel := new(vpcclassicv1.ImageFilePrototype)
				imageFilePrototypeModel.Href = core.StringPtr("cos://us-south/my-bucket/my-image.qcow2")

				// Construct an instance of the OperatingSystemIdentityByName model
				operatingSystemIdentityModel := new(vpcclassicv1.OperatingSystemIdentityByName)
				operatingSystemIdentityModel.Name = core.StringPtr("debian-9-amd64")

				// Construct an instance of the ImagePrototypeImageByFile model
				imagePrototypeModel := new(vpcclassicv1.ImagePrototypeImageByFile)
				imagePrototypeModel.Name = core.StringPtr("my-image")
				imagePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				imagePrototypeModel.File = imageFilePrototypeModel
				imagePrototypeModel.OperatingSystem = operatingSystemIdentityModel

				// Construct an instance of the CreateImageOptions model
				createImageOptionsModel := new(vpcclassicv1.CreateImageOptions)
				createImageOptionsModel.ImagePrototype = imagePrototypeModel
				createImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.CreateImage(createImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.CreateImage(createImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`CreateImage(createImageOptions *CreateImageOptions)`, func() {
		version := "testString"
		createImagePath := "/images"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createImagePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "file": {"size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "visibility": "private"}`)
				}))
			})
			It(`Invoke CreateImage successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the ImageFilePrototype model
				imageFilePrototypeModel := new(vpcclassicv1.ImageFilePrototype)
				imageFilePrototypeModel.Href = core.StringPtr("cos://us-south/my-bucket/my-image.qcow2")

				// Construct an instance of the OperatingSystemIdentityByName model
				operatingSystemIdentityModel := new(vpcclassicv1.OperatingSystemIdentityByName)
				operatingSystemIdentityModel.Name = core.StringPtr("debian-9-amd64")

				// Construct an instance of the ImagePrototypeImageByFile model
				imagePrototypeModel := new(vpcclassicv1.ImagePrototypeImageByFile)
				imagePrototypeModel.Name = core.StringPtr("my-image")
				imagePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				imagePrototypeModel.File = imageFilePrototypeModel
				imagePrototypeModel.OperatingSystem = operatingSystemIdentityModel

				// Construct an instance of the CreateImageOptions model
				createImageOptionsModel := new(vpcclassicv1.CreateImageOptions)
				createImageOptionsModel.ImagePrototype = imagePrototypeModel
				createImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.CreateImageWithContext(ctx, createImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.CreateImage(createImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.CreateImageWithContext(ctx, createImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createImagePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "file": {"size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "visibility": "private"}`)
				}))
			})
			It(`Invoke CreateImage successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.CreateImage(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the ImageFilePrototype model
				imageFilePrototypeModel := new(vpcclassicv1.ImageFilePrototype)
				imageFilePrototypeModel.Href = core.StringPtr("cos://us-south/my-bucket/my-image.qcow2")

				// Construct an instance of the OperatingSystemIdentityByName model
				operatingSystemIdentityModel := new(vpcclassicv1.OperatingSystemIdentityByName)
				operatingSystemIdentityModel.Name = core.StringPtr("debian-9-amd64")

				// Construct an instance of the ImagePrototypeImageByFile model
				imagePrototypeModel := new(vpcclassicv1.ImagePrototypeImageByFile)
				imagePrototypeModel.Name = core.StringPtr("my-image")
				imagePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				imagePrototypeModel.File = imageFilePrototypeModel
				imagePrototypeModel.OperatingSystem = operatingSystemIdentityModel

				// Construct an instance of the CreateImageOptions model
				createImageOptionsModel := new(vpcclassicv1.CreateImageOptions)
				createImageOptionsModel.ImagePrototype = imagePrototypeModel
				createImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.CreateImage(createImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateImage with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the ImageFilePrototype model
				imageFilePrototypeModel := new(vpcclassicv1.ImageFilePrototype)
				imageFilePrototypeModel.Href = core.StringPtr("cos://us-south/my-bucket/my-image.qcow2")

				// Construct an instance of the OperatingSystemIdentityByName model
				operatingSystemIdentityModel := new(vpcclassicv1.OperatingSystemIdentityByName)
				operatingSystemIdentityModel.Name = core.StringPtr("debian-9-amd64")

				// Construct an instance of the ImagePrototypeImageByFile model
				imagePrototypeModel := new(vpcclassicv1.ImagePrototypeImageByFile)
				imagePrototypeModel.Name = core.StringPtr("my-image")
				imagePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				imagePrototypeModel.File = imageFilePrototypeModel
				imagePrototypeModel.OperatingSystem = operatingSystemIdentityModel

				// Construct an instance of the CreateImageOptions model
				createImageOptionsModel := new(vpcclassicv1.CreateImageOptions)
				createImageOptionsModel.ImagePrototype = imagePrototypeModel
				createImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.CreateImage(createImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateImageOptions model with no property values
				createImageOptionsModelNew := new(vpcclassicv1.CreateImageOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.CreateImage(createImageOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`DeleteImage(deleteImageOptions *DeleteImageOptions)`, func() {
		version := "testString"
		deleteImagePath := "/images/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteImagePath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.WriteHeader(202)
				}))
			})
			It(`Invoke DeleteImage successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcClassicService.DeleteImage(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteImageOptions model
				deleteImageOptionsModel := new(vpcclassicv1.DeleteImageOptions)
				deleteImageOptionsModel.ID = core.StringPtr("testString")
				deleteImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcClassicService.DeleteImage(deleteImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteImage with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the DeleteImageOptions model
				deleteImageOptionsModel := new(vpcclassicv1.DeleteImageOptions)
				deleteImageOptionsModel.ID = core.StringPtr("testString")
				deleteImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcClassicService.DeleteImage(deleteImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteImageOptions model with no property values
				deleteImageOptionsModelNew := new(vpcclassicv1.DeleteImageOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcClassicService.DeleteImage(deleteImageOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetImage(getImageOptions *GetImageOptions) - Operation response error`, func() {
		version := "testString"
		getImagePath := "/images/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getImagePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetImage with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetImageOptions model
				getImageOptionsModel := new(vpcclassicv1.GetImageOptions)
				getImageOptionsModel.ID = core.StringPtr("testString")
				getImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetImage(getImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetImage(getImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetImage(getImageOptions *GetImageOptions)`, func() {
		version := "testString"
		getImagePath := "/images/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getImagePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "file": {"size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "visibility": "private"}`)
				}))
			})
			It(`Invoke GetImage successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetImageOptions model
				getImageOptionsModel := new(vpcclassicv1.GetImageOptions)
				getImageOptionsModel.ID = core.StringPtr("testString")
				getImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetImageWithContext(ctx, getImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetImage(getImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetImageWithContext(ctx, getImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getImagePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "file": {"size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "visibility": "private"}`)
				}))
			})
			It(`Invoke GetImage successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetImage(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetImageOptions model
				getImageOptionsModel := new(vpcclassicv1.GetImageOptions)
				getImageOptionsModel.ID = core.StringPtr("testString")
				getImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetImage(getImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetImage with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetImageOptions model
				getImageOptionsModel := new(vpcclassicv1.GetImageOptions)
				getImageOptionsModel.ID = core.StringPtr("testString")
				getImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetImage(getImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetImageOptions model with no property values
				getImageOptionsModelNew := new(vpcclassicv1.GetImageOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetImage(getImageOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateImage(updateImageOptions *UpdateImageOptions) - Operation response error`, func() {
		version := "testString"
		updateImagePath := "/images/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateImagePath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateImage with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ImagePatch model
				imagePatchModel := new(vpcclassicv1.ImagePatch)
				imagePatchModel.Name = core.StringPtr("my-image")
				imagePatchModelAsPatch, asPatchErr := imagePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateImageOptions model
				updateImageOptionsModel := new(vpcclassicv1.UpdateImageOptions)
				updateImageOptionsModel.ID = core.StringPtr("testString")
				updateImageOptionsModel.ImagePatch = imagePatchModelAsPatch
				updateImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.UpdateImage(updateImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.UpdateImage(updateImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`UpdateImage(updateImageOptions *UpdateImageOptions)`, func() {
		version := "testString"
		updateImagePath := "/images/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateImagePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "file": {"size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "visibility": "private"}`)
				}))
			})
			It(`Invoke UpdateImage successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ImagePatch model
				imagePatchModel := new(vpcclassicv1.ImagePatch)
				imagePatchModel.Name = core.StringPtr("my-image")
				imagePatchModelAsPatch, asPatchErr := imagePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateImageOptions model
				updateImageOptionsModel := new(vpcclassicv1.UpdateImageOptions)
				updateImageOptionsModel.ID = core.StringPtr("testString")
				updateImageOptionsModel.ImagePatch = imagePatchModelAsPatch
				updateImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.UpdateImageWithContext(ctx, updateImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.UpdateImage(updateImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.UpdateImageWithContext(ctx, updateImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateImagePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "file": {"size": 1}, "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "minimum_provisioned_size": 22, "name": "my-image", "operating_system": {"architecture": "amd64", "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "visibility": "private"}`)
				}))
			})
			It(`Invoke UpdateImage successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.UpdateImage(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ImagePatch model
				imagePatchModel := new(vpcclassicv1.ImagePatch)
				imagePatchModel.Name = core.StringPtr("my-image")
				imagePatchModelAsPatch, asPatchErr := imagePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateImageOptions model
				updateImageOptionsModel := new(vpcclassicv1.UpdateImageOptions)
				updateImageOptionsModel.ID = core.StringPtr("testString")
				updateImageOptionsModel.ImagePatch = imagePatchModelAsPatch
				updateImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.UpdateImage(updateImageOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateImage with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ImagePatch model
				imagePatchModel := new(vpcclassicv1.ImagePatch)
				imagePatchModel.Name = core.StringPtr("my-image")
				imagePatchModelAsPatch, asPatchErr := imagePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateImageOptions model
				updateImageOptionsModel := new(vpcclassicv1.UpdateImageOptions)
				updateImageOptionsModel.ID = core.StringPtr("testString")
				updateImageOptionsModel.ImagePatch = imagePatchModelAsPatch
				updateImageOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.UpdateImage(updateImageOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateImageOptions model with no property values
				updateImageOptionsModelNew := new(vpcclassicv1.UpdateImageOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.UpdateImage(updateImageOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListOperatingSystems(listOperatingSystemsOptions *ListOperatingSystemsOptions) - Operation response error`, func() {
		version := "testString"
		listOperatingSystemsPath := "/operating_systems"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listOperatingSystemsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListOperatingSystems with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListOperatingSystemsOptions model
				listOperatingSystemsOptionsModel := new(vpcclassicv1.ListOperatingSystemsOptions)
				listOperatingSystemsOptionsModel.Start = core.StringPtr("testString")
				listOperatingSystemsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listOperatingSystemsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListOperatingSystems(listOperatingSystemsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListOperatingSystems(listOperatingSystemsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListOperatingSystems(listOperatingSystemsOptions *ListOperatingSystemsOptions)`, func() {
		version := "testString"
		listOperatingSystemsPath := "/operating_systems"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listOperatingSystemsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "operating_systems": [{"architecture": "amd64", "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}]}`)
				}))
			})
			It(`Invoke ListOperatingSystems successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListOperatingSystemsOptions model
				listOperatingSystemsOptionsModel := new(vpcclassicv1.ListOperatingSystemsOptions)
				listOperatingSystemsOptionsModel.Start = core.StringPtr("testString")
				listOperatingSystemsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listOperatingSystemsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListOperatingSystemsWithContext(ctx, listOperatingSystemsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListOperatingSystems(listOperatingSystemsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListOperatingSystemsWithContext(ctx, listOperatingSystemsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listOperatingSystemsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "operating_systems": [{"architecture": "amd64", "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}]}`)
				}))
			})
			It(`Invoke ListOperatingSystems successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListOperatingSystems(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListOperatingSystemsOptions model
				listOperatingSystemsOptionsModel := new(vpcclassicv1.ListOperatingSystemsOptions)
				listOperatingSystemsOptionsModel.Start = core.StringPtr("testString")
				listOperatingSystemsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listOperatingSystemsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListOperatingSystems(listOperatingSystemsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListOperatingSystems with error: Operation request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListOperatingSystemsOptions model
				listOperatingSystemsOptionsModel := new(vpcclassicv1.ListOperatingSystemsOptions)
				listOperatingSystemsOptionsModel.Start = core.StringPtr("testString")
				listOperatingSystemsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listOperatingSystemsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListOperatingSystems(listOperatingSystemsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetOperatingSystem(getOperatingSystemOptions *GetOperatingSystemOptions) - Operation response error`, func() {
		version := "testString"
		getOperatingSystemPath := "/operating_systems/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOperatingSystemPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetOperatingSystem with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetOperatingSystemOptions model
				getOperatingSystemOptionsModel := new(vpcclassicv1.GetOperatingSystemOptions)
				getOperatingSystemOptionsModel.Name = core.StringPtr("testString")
				getOperatingSystemOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetOperatingSystem(getOperatingSystemOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetOperatingSystem(getOperatingSystemOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetOperatingSystem(getOperatingSystemOptions *GetOperatingSystemOptions)`, func() {
		version := "testString"
		getOperatingSystemPath := "/operating_systems/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOperatingSystemPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"architecture": "amd64", "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}`)
				}))
			})
			It(`Invoke GetOperatingSystem successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetOperatingSystemOptions model
				getOperatingSystemOptionsModel := new(vpcclassicv1.GetOperatingSystemOptions)
				getOperatingSystemOptionsModel.Name = core.StringPtr("testString")
				getOperatingSystemOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetOperatingSystemWithContext(ctx, getOperatingSystemOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetOperatingSystem(getOperatingSystemOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetOperatingSystemWithContext(ctx, getOperatingSystemOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getOperatingSystemPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"architecture": "amd64", "display_name": "Ubuntu Server 16.04 LTS amd64", "family": "Ubuntu Server", "href": "https://us-south.iaas.cloud.ibm.com/v1/operating_systems/ubuntu-16-amd64", "name": "ubuntu-16-amd64", "vendor": "Canonical", "version": "16.04 LTS"}`)
				}))
			})
			It(`Invoke GetOperatingSystem successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetOperatingSystem(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetOperatingSystemOptions model
				getOperatingSystemOptionsModel := new(vpcclassicv1.GetOperatingSystemOptions)
				getOperatingSystemOptionsModel.Name = core.StringPtr("testString")
				getOperatingSystemOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetOperatingSystem(getOperatingSystemOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetOperatingSystem with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetOperatingSystemOptions model
				getOperatingSystemOptionsModel := new(vpcclassicv1.GetOperatingSystemOptions)
				getOperatingSystemOptionsModel.Name = core.StringPtr("testString")
				getOperatingSystemOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetOperatingSystem(getOperatingSystemOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetOperatingSystemOptions model with no property values
				getOperatingSystemOptionsModelNew := new(vpcclassicv1.GetOperatingSystemOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetOperatingSystem(getOperatingSystemOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`Service constructor tests`, func() {
		version := "testString"
		It(`Instantiate service client`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				Authenticator: &core.NoAuthAuthenticator{},
				Version:       core.StringPtr(version),
			})
			Expect(vpcClassicService).ToNot(BeNil())
			Expect(serviceErr).To(BeNil())
		})
		It(`Instantiate service client with error: Invalid URL`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Invalid Auth`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "https://vpcclassicv1/api",
				Version: core.StringPtr(version),
				Authenticator: &core.BasicAuthenticator{
					Username: "",
					Password: "",
				},
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Validation Error`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
	})
	Describe(`Service constructor tests using external config`, func() {
		version := "testString"
		Context(`Using external config, construct service client instances`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "noauth",
			}

			It(`Create service client using external config successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url from constructor successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					URL:     "https://testService/api",
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url programatically successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				err := vpcClassicService.SetServiceURL("https://testService/api")
				Expect(err).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid Auth`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "someOtherAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid URL`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_AUTH_TYPE": "NOAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
	})
	Describe(`Regional endpoint tests`, func() {
		It(`GetServiceURLForRegion(region string)`, func() {
			var url string
			var err error
			url, err = vpcclassicv1.GetServiceURLForRegion("INVALID_REGION")
			Expect(url).To(BeEmpty())
			Expect(err).ToNot(BeNil())
			fmt.Fprintf(GinkgoWriter, "Expected error: %s\n", err.Error())
		})
	})
	Describe(`ListKeys(listKeysOptions *ListKeysOptions) - Operation response error`, func() {
		version := "testString"
		listKeysPath := "/keys"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listKeysPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListKeys with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListKeysOptions model
				listKeysOptionsModel := new(vpcclassicv1.ListKeysOptions)
				listKeysOptionsModel.Start = core.StringPtr("testString")
				listKeysOptionsModel.Limit = core.Int64Ptr(int64(1))
				listKeysOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListKeys(listKeysOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListKeys(listKeysOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListKeys(listKeysOptions *ListKeysOptions)`, func() {
		version := "testString"
		listKeysPath := "/keys"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listKeysPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/keys?limit=20"}, "keys": [{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "type": "rsa"}], "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/keys?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "total_count": 132}`)
				}))
			})
			It(`Invoke ListKeys successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListKeysOptions model
				listKeysOptionsModel := new(vpcclassicv1.ListKeysOptions)
				listKeysOptionsModel.Start = core.StringPtr("testString")
				listKeysOptionsModel.Limit = core.Int64Ptr(int64(1))
				listKeysOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListKeysWithContext(ctx, listKeysOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListKeys(listKeysOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListKeysWithContext(ctx, listKeysOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listKeysPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/keys?limit=20"}, "keys": [{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "type": "rsa"}], "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/keys?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "total_count": 132}`)
				}))
			})
			It(`Invoke ListKeys successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListKeys(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListKeysOptions model
				listKeysOptionsModel := new(vpcclassicv1.ListKeysOptions)
				listKeysOptionsModel.Start = core.StringPtr("testString")
				listKeysOptionsModel.Limit = core.Int64Ptr(int64(1))
				listKeysOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListKeys(listKeysOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListKeys with error: Operation request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListKeysOptions model
				listKeysOptionsModel := new(vpcclassicv1.ListKeysOptions)
				listKeysOptionsModel.Start = core.StringPtr("testString")
				listKeysOptionsModel.Limit = core.Int64Ptr(int64(1))
				listKeysOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListKeys(listKeysOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateKey(createKeyOptions *CreateKeyOptions) - Operation response error`, func() {
		version := "testString"
		createKeyPath := "/keys"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createKeyPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateKey with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateKeyOptions model
				createKeyOptionsModel := new(vpcclassicv1.CreateKeyOptions)
				createKeyOptionsModel.PublicKey = core.StringPtr("AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En")
				createKeyOptionsModel.Name = core.StringPtr("my-key")
				createKeyOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createKeyOptionsModel.Type = core.StringPtr("rsa")
				createKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.CreateKey(createKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.CreateKey(createKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`CreateKey(createKeyOptions *CreateKeyOptions)`, func() {
		version := "testString"
		createKeyPath := "/keys"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createKeyPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "type": "rsa"}`)
				}))
			})
			It(`Invoke CreateKey successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateKeyOptions model
				createKeyOptionsModel := new(vpcclassicv1.CreateKeyOptions)
				createKeyOptionsModel.PublicKey = core.StringPtr("AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En")
				createKeyOptionsModel.Name = core.StringPtr("my-key")
				createKeyOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createKeyOptionsModel.Type = core.StringPtr("rsa")
				createKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.CreateKeyWithContext(ctx, createKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.CreateKey(createKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.CreateKeyWithContext(ctx, createKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createKeyPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "type": "rsa"}`)
				}))
			})
			It(`Invoke CreateKey successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.CreateKey(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateKeyOptions model
				createKeyOptionsModel := new(vpcclassicv1.CreateKeyOptions)
				createKeyOptionsModel.PublicKey = core.StringPtr("AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En")
				createKeyOptionsModel.Name = core.StringPtr("my-key")
				createKeyOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createKeyOptionsModel.Type = core.StringPtr("rsa")
				createKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.CreateKey(createKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateKey with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the CreateKeyOptions model
				createKeyOptionsModel := new(vpcclassicv1.CreateKeyOptions)
				createKeyOptionsModel.PublicKey = core.StringPtr("AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En")
				createKeyOptionsModel.Name = core.StringPtr("my-key")
				createKeyOptionsModel.ResourceGroup = resourceGroupIdentityModel
				createKeyOptionsModel.Type = core.StringPtr("rsa")
				createKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.CreateKey(createKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateKeyOptions model with no property values
				createKeyOptionsModelNew := new(vpcclassicv1.CreateKeyOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.CreateKey(createKeyOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`DeleteKey(deleteKeyOptions *DeleteKeyOptions)`, func() {
		version := "testString"
		deleteKeyPath := "/keys/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteKeyPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteKey successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcClassicService.DeleteKey(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteKeyOptions model
				deleteKeyOptionsModel := new(vpcclassicv1.DeleteKeyOptions)
				deleteKeyOptionsModel.ID = core.StringPtr("testString")
				deleteKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcClassicService.DeleteKey(deleteKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteKey with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the DeleteKeyOptions model
				deleteKeyOptionsModel := new(vpcclassicv1.DeleteKeyOptions)
				deleteKeyOptionsModel.ID = core.StringPtr("testString")
				deleteKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcClassicService.DeleteKey(deleteKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteKeyOptions model with no property values
				deleteKeyOptionsModelNew := new(vpcclassicv1.DeleteKeyOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcClassicService.DeleteKey(deleteKeyOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetKey(getKeyOptions *GetKeyOptions) - Operation response error`, func() {
		version := "testString"
		getKeyPath := "/keys/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getKeyPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetKey with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetKeyOptions model
				getKeyOptionsModel := new(vpcclassicv1.GetKeyOptions)
				getKeyOptionsModel.ID = core.StringPtr("testString")
				getKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetKey(getKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetKey(getKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetKey(getKeyOptions *GetKeyOptions)`, func() {
		version := "testString"
		getKeyPath := "/keys/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getKeyPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "type": "rsa"}`)
				}))
			})
			It(`Invoke GetKey successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetKeyOptions model
				getKeyOptionsModel := new(vpcclassicv1.GetKeyOptions)
				getKeyOptionsModel.ID = core.StringPtr("testString")
				getKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetKeyWithContext(ctx, getKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetKey(getKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetKeyWithContext(ctx, getKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getKeyPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "type": "rsa"}`)
				}))
			})
			It(`Invoke GetKey successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetKey(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetKeyOptions model
				getKeyOptionsModel := new(vpcclassicv1.GetKeyOptions)
				getKeyOptionsModel.ID = core.StringPtr("testString")
				getKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetKey(getKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetKey with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetKeyOptions model
				getKeyOptionsModel := new(vpcclassicv1.GetKeyOptions)
				getKeyOptionsModel.ID = core.StringPtr("testString")
				getKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetKey(getKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetKeyOptions model with no property values
				getKeyOptionsModelNew := new(vpcclassicv1.GetKeyOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetKey(getKeyOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateKey(updateKeyOptions *UpdateKeyOptions) - Operation response error`, func() {
		version := "testString"
		updateKeyPath := "/keys/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateKeyPath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateKey with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the KeyPatch model
				keyPatchModel := new(vpcclassicv1.KeyPatch)
				keyPatchModel.Name = core.StringPtr("my-key")
				keyPatchModelAsPatch, asPatchErr := keyPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateKeyOptions model
				updateKeyOptionsModel := new(vpcclassicv1.UpdateKeyOptions)
				updateKeyOptionsModel.ID = core.StringPtr("testString")
				updateKeyOptionsModel.KeyPatch = keyPatchModelAsPatch
				updateKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.UpdateKey(updateKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.UpdateKey(updateKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`UpdateKey(updateKeyOptions *UpdateKeyOptions)`, func() {
		version := "testString"
		updateKeyPath := "/keys/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateKeyPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "type": "rsa"}`)
				}))
			})
			It(`Invoke UpdateKey successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the KeyPatch model
				keyPatchModel := new(vpcclassicv1.KeyPatch)
				keyPatchModel.Name = core.StringPtr("my-key")
				keyPatchModelAsPatch, asPatchErr := keyPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateKeyOptions model
				updateKeyOptionsModel := new(vpcclassicv1.UpdateKeyOptions)
				updateKeyOptionsModel.ID = core.StringPtr("testString")
				updateKeyOptionsModel.KeyPatch = keyPatchModelAsPatch
				updateKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.UpdateKeyWithContext(ctx, updateKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.UpdateKey(updateKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.UpdateKeyWithContext(ctx, updateKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateKeyPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "length": 2048, "name": "my-key", "public_key": "AAAAB3NzaC1yc2EAAAADAQABAAABAQDDGe50Bxa5T5NDddrrtbx2Y4/VGbiCgXqnBsYToIUKoFSHTQl5IX3PasGnneKanhcLwWz5M5MoCRvhxTp66NKzIfAz7r+FX9rxgR+ZgcM253YAqOVeIpOU408simDZKriTlN8kYsXL7P34tsWuAJf4MgZtJAQxous/2byetpdCv8ddnT4X3ltOg9w+LqSCPYfNivqH00Eh7S1Ldz7I8aw5WOp5a+sQFP/RbwfpwHp+ny7DfeIOokcuI42tJkoBn7UsLTVpCSmXr2EDRlSWe/1M/iHNRBzaT3CK0+SwZWd2AEjePxSnWKNGIEUJDlUYp7hKhiQcgT5ZAnWU121oc5En", "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "type": "rsa"}`)
				}))
			})
			It(`Invoke UpdateKey successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.UpdateKey(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the KeyPatch model
				keyPatchModel := new(vpcclassicv1.KeyPatch)
				keyPatchModel.Name = core.StringPtr("my-key")
				keyPatchModelAsPatch, asPatchErr := keyPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateKeyOptions model
				updateKeyOptionsModel := new(vpcclassicv1.UpdateKeyOptions)
				updateKeyOptionsModel.ID = core.StringPtr("testString")
				updateKeyOptionsModel.KeyPatch = keyPatchModelAsPatch
				updateKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.UpdateKey(updateKeyOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateKey with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the KeyPatch model
				keyPatchModel := new(vpcclassicv1.KeyPatch)
				keyPatchModel.Name = core.StringPtr("my-key")
				keyPatchModelAsPatch, asPatchErr := keyPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateKeyOptions model
				updateKeyOptionsModel := new(vpcclassicv1.UpdateKeyOptions)
				updateKeyOptionsModel.ID = core.StringPtr("testString")
				updateKeyOptionsModel.KeyPatch = keyPatchModelAsPatch
				updateKeyOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.UpdateKey(updateKeyOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateKeyOptions model with no property values
				updateKeyOptionsModelNew := new(vpcclassicv1.UpdateKeyOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.UpdateKey(updateKeyOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`Service constructor tests`, func() {
		version := "testString"
		It(`Instantiate service client`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				Authenticator: &core.NoAuthAuthenticator{},
				Version:       core.StringPtr(version),
			})
			Expect(vpcClassicService).ToNot(BeNil())
			Expect(serviceErr).To(BeNil())
		})
		It(`Instantiate service client with error: Invalid URL`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Invalid Auth`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "https://vpcclassicv1/api",
				Version: core.StringPtr(version),
				Authenticator: &core.BasicAuthenticator{
					Username: "",
					Password: "",
				},
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Validation Error`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
	})
	Describe(`Service constructor tests using external config`, func() {
		version := "testString"
		Context(`Using external config, construct service client instances`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "noauth",
			}

			It(`Create service client using external config successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url from constructor successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					URL:     "https://testService/api",
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url programatically successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				err := vpcClassicService.SetServiceURL("https://testService/api")
				Expect(err).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid Auth`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "someOtherAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid URL`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_AUTH_TYPE": "NOAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
	})
	Describe(`Regional endpoint tests`, func() {
		It(`GetServiceURLForRegion(region string)`, func() {
			var url string
			var err error
			url, err = vpcclassicv1.GetServiceURLForRegion("INVALID_REGION")
			Expect(url).To(BeEmpty())
			Expect(err).ToNot(BeNil())
			fmt.Fprintf(GinkgoWriter, "Expected error: %s\n", err.Error())
		})
	})
	Describe(`ListInstanceProfiles(listInstanceProfilesOptions *ListInstanceProfilesOptions) - Operation response error`, func() {
		version := "testString"
		listInstanceProfilesPath := "/instance/profiles"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceProfilesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListInstanceProfiles with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListInstanceProfilesOptions model
				listInstanceProfilesOptionsModel := new(vpcclassicv1.ListInstanceProfilesOptions)
				listInstanceProfilesOptionsModel.Start = core.StringPtr("testString")
				listInstanceProfilesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listInstanceProfilesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListInstanceProfiles(listInstanceProfilesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListInstanceProfiles(listInstanceProfilesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListInstanceProfiles(listInstanceProfilesOptions *ListInstanceProfilesOptions)`, func() {
		version := "testString"
		listInstanceProfilesPath := "/instance/profiles"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceProfilesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "profiles": [{"bandwidth": {"type": "fixed", "value": 20000}, "crn": "crn:v1:bluemix:public:is:us-south-1:::instance-profile:bc1-4x16", "family": "balanced", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16", "port_speed": {"type": "fixed", "value": 1000}}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListInstanceProfiles successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListInstanceProfilesOptions model
				listInstanceProfilesOptionsModel := new(vpcclassicv1.ListInstanceProfilesOptions)
				listInstanceProfilesOptionsModel.Start = core.StringPtr("testString")
				listInstanceProfilesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listInstanceProfilesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListInstanceProfilesWithContext(ctx, listInstanceProfilesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListInstanceProfiles(listInstanceProfilesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListInstanceProfilesWithContext(ctx, listInstanceProfilesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceProfilesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "profiles": [{"bandwidth": {"type": "fixed", "value": 20000}, "crn": "crn:v1:bluemix:public:is:us-south-1:::instance-profile:bc1-4x16", "family": "balanced", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16", "port_speed": {"type": "fixed", "value": 1000}}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListInstanceProfiles successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListInstanceProfiles(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListInstanceProfilesOptions model
				listInstanceProfilesOptionsModel := new(vpcclassicv1.ListInstanceProfilesOptions)
				listInstanceProfilesOptionsModel.Start = core.StringPtr("testString")
				listInstanceProfilesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listInstanceProfilesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListInstanceProfiles(listInstanceProfilesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListInstanceProfiles with error: Operation request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListInstanceProfilesOptions model
				listInstanceProfilesOptionsModel := new(vpcclassicv1.ListInstanceProfilesOptions)
				listInstanceProfilesOptionsModel.Start = core.StringPtr("testString")
				listInstanceProfilesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listInstanceProfilesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListInstanceProfiles(listInstanceProfilesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetInstanceProfile(getInstanceProfileOptions *GetInstanceProfileOptions) - Operation response error`, func() {
		version := "testString"
		getInstanceProfilePath := "/instance/profiles/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceProfilePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetInstanceProfile with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetInstanceProfileOptions model
				getInstanceProfileOptionsModel := new(vpcclassicv1.GetInstanceProfileOptions)
				getInstanceProfileOptionsModel.Name = core.StringPtr("testString")
				getInstanceProfileOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetInstanceProfile(getInstanceProfileOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetInstanceProfile(getInstanceProfileOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetInstanceProfile(getInstanceProfileOptions *GetInstanceProfileOptions)`, func() {
		version := "testString"
		getInstanceProfilePath := "/instance/profiles/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceProfilePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"bandwidth": {"type": "fixed", "value": 20000}, "crn": "crn:v1:bluemix:public:is:us-south-1:::instance-profile:bc1-4x16", "family": "balanced", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16", "port_speed": {"type": "fixed", "value": 1000}}`)
				}))
			})
			It(`Invoke GetInstanceProfile successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetInstanceProfileOptions model
				getInstanceProfileOptionsModel := new(vpcclassicv1.GetInstanceProfileOptions)
				getInstanceProfileOptionsModel.Name = core.StringPtr("testString")
				getInstanceProfileOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetInstanceProfileWithContext(ctx, getInstanceProfileOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetInstanceProfile(getInstanceProfileOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetInstanceProfileWithContext(ctx, getInstanceProfileOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceProfilePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"bandwidth": {"type": "fixed", "value": 20000}, "crn": "crn:v1:bluemix:public:is:us-south-1:::instance-profile:bc1-4x16", "family": "balanced", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16", "port_speed": {"type": "fixed", "value": 1000}}`)
				}))
			})
			It(`Invoke GetInstanceProfile successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetInstanceProfile(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetInstanceProfileOptions model
				getInstanceProfileOptionsModel := new(vpcclassicv1.GetInstanceProfileOptions)
				getInstanceProfileOptionsModel.Name = core.StringPtr("testString")
				getInstanceProfileOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetInstanceProfile(getInstanceProfileOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetInstanceProfile with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetInstanceProfileOptions model
				getInstanceProfileOptionsModel := new(vpcclassicv1.GetInstanceProfileOptions)
				getInstanceProfileOptionsModel.Name = core.StringPtr("testString")
				getInstanceProfileOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetInstanceProfile(getInstanceProfileOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetInstanceProfileOptions model with no property values
				getInstanceProfileOptionsModelNew := new(vpcclassicv1.GetInstanceProfileOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetInstanceProfile(getInstanceProfileOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListInstances(listInstancesOptions *ListInstancesOptions) - Operation response error`, func() {
		version := "testString"
		listInstancesPath := "/instances"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstancesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["network_interfaces.subnet.id"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["network_interfaces.subnet.crn"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["network_interfaces.subnet.name"]).To(Equal([]string{"testString"}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListInstances with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListInstancesOptions model
				listInstancesOptionsModel := new(vpcclassicv1.ListInstancesOptions)
				listInstancesOptionsModel.Start = core.StringPtr("testString")
				listInstancesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listInstancesOptionsModel.NetworkInterfacesSubnetID = core.StringPtr("testString")
				listInstancesOptionsModel.NetworkInterfacesSubnetCRN = core.StringPtr("testString")
				listInstancesOptionsModel.NetworkInterfacesSubnetName = core.StringPtr("testString")
				listInstancesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListInstances(listInstancesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListInstances(listInstancesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListInstances(listInstancesOptions *ListInstancesOptions)`, func() {
		version := "testString"
		listInstancesPath := "/instances"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstancesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["network_interfaces.subnet.id"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["network_interfaces.subnet.crn"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["network_interfaces.subnet.name"]).To(Equal([]string{"testString"}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances?limit=20"}, "instances": [{"bandwidth": 1000, "boot_volume_attachment": {"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"crn": "crn:v1:bluemix:public:is:us-south-1:::instance-profile:bc1-4x16", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "deleting", "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "total_count": 132}`)
				}))
			})
			It(`Invoke ListInstances successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListInstancesOptions model
				listInstancesOptionsModel := new(vpcclassicv1.ListInstancesOptions)
				listInstancesOptionsModel.Start = core.StringPtr("testString")
				listInstancesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listInstancesOptionsModel.NetworkInterfacesSubnetID = core.StringPtr("testString")
				listInstancesOptionsModel.NetworkInterfacesSubnetCRN = core.StringPtr("testString")
				listInstancesOptionsModel.NetworkInterfacesSubnetName = core.StringPtr("testString")
				listInstancesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListInstancesWithContext(ctx, listInstancesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListInstances(listInstancesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListInstancesWithContext(ctx, listInstancesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstancesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["network_interfaces.subnet.id"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["network_interfaces.subnet.crn"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["network_interfaces.subnet.name"]).To(Equal([]string{"testString"}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances?limit=20"}, "instances": [{"bandwidth": 1000, "boot_volume_attachment": {"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"crn": "crn:v1:bluemix:public:is:us-south-1:::instance-profile:bc1-4x16", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "deleting", "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "total_count": 132}`)
				}))
			})
			It(`Invoke ListInstances successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListInstances(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListInstancesOptions model
				listInstancesOptionsModel := new(vpcclassicv1.ListInstancesOptions)
				listInstancesOptionsModel.Start = core.StringPtr("testString")
				listInstancesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listInstancesOptionsModel.NetworkInterfacesSubnetID = core.StringPtr("testString")
				listInstancesOptionsModel.NetworkInterfacesSubnetCRN = core.StringPtr("testString")
				listInstancesOptionsModel.NetworkInterfacesSubnetName = core.StringPtr("testString")
				listInstancesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListInstances(listInstancesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListInstances with error: Operation request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListInstancesOptions model
				listInstancesOptionsModel := new(vpcclassicv1.ListInstancesOptions)
				listInstancesOptionsModel.Start = core.StringPtr("testString")
				listInstancesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listInstancesOptionsModel.NetworkInterfacesSubnetID = core.StringPtr("testString")
				listInstancesOptionsModel.NetworkInterfacesSubnetCRN = core.StringPtr("testString")
				listInstancesOptionsModel.NetworkInterfacesSubnetName = core.StringPtr("testString")
				listInstancesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListInstances(listInstancesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateInstance(createInstanceOptions *CreateInstanceOptions) - Operation response error`, func() {
		version := "testString"
		createInstancePath := "/instances"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstancePath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateInstance with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the KeyIdentityByID model
				keyIdentityModel := new(vpcclassicv1.KeyIdentityByID)
				keyIdentityModel.ID = core.StringPtr("363f6d70-0000-0001-0000-00000013b96c")

				// Construct an instance of the SecurityGroupIdentityByID model
				securityGroupIdentityModel := new(vpcclassicv1.SecurityGroupIdentityByID)
				securityGroupIdentityModel.ID = core.StringPtr("be5df5ca-12a0-494b-907e-aa6ec2bfa271")

				// Construct an instance of the SubnetIdentityByID model
				subnetIdentityModel := new(vpcclassicv1.SubnetIdentityByID)
				subnetIdentityModel.ID = core.StringPtr("7ec86020-1c6e-4889-b3f0-a15f2e50f87e")

				// Construct an instance of the NetworkInterfacePrototype model
				networkInterfacePrototypeModel := new(vpcclassicv1.NetworkInterfacePrototype)
				networkInterfacePrototypeModel.Name = core.StringPtr("my-network-interface")
				networkInterfacePrototypeModel.PrimaryIpv4Address = core.StringPtr("10.0.0.5")
				networkInterfacePrototypeModel.SecurityGroups = []vpcclassicv1.SecurityGroupIdentityIntf{securityGroupIdentityModel}
				networkInterfacePrototypeModel.Subnet = subnetIdentityModel

				// Construct an instance of the InstanceProfileIdentityByName model
				instanceProfileIdentityModel := new(vpcclassicv1.InstanceProfileIdentityByName)
				instanceProfileIdentityModel.Name = core.StringPtr("bx2-2x8")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcclassicv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:[...]")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcclassicv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("5iops-tier")

				// Construct an instance of the VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity model
				volumeAttachmentVolumePrototypeInstanceContextModel := new(vpcclassicv1.VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity)
				volumeAttachmentVolumePrototypeInstanceContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumeAttachmentVolumePrototypeInstanceContextModel.Iops = core.Int64Ptr(int64(10000))
				volumeAttachmentVolumePrototypeInstanceContextModel.Name = core.StringPtr("my-data-volume")
				volumeAttachmentVolumePrototypeInstanceContextModel.Profile = volumeProfileIdentityModel
				volumeAttachmentVolumePrototypeInstanceContextModel.Capacity = core.Int64Ptr(int64(1000))

				// Construct an instance of the VolumeAttachmentPrototypeInstanceContext model
				volumeAttachmentPrototypeInstanceContextModel := new(vpcclassicv1.VolumeAttachmentPrototypeInstanceContext)
				volumeAttachmentPrototypeInstanceContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceContextModel.Volume = volumeAttachmentVolumePrototypeInstanceContextModel

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcclassicv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("f0aae929-7047-46d1-92e1-9102b07a7f6f")

				// Construct an instance of the VolumePrototypeInstanceByImageContext model
				volumePrototypeInstanceByImageContextModel := new(vpcclassicv1.VolumePrototypeInstanceByImageContext)
				volumePrototypeInstanceByImageContextModel.Capacity = core.Int64Ptr(int64(100))
				volumePrototypeInstanceByImageContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeInstanceByImageContextModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeInstanceByImageContextModel.Name = core.StringPtr("my-boot-volume")
				volumePrototypeInstanceByImageContextModel.Profile = volumeProfileIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceByImageContext model
				volumeAttachmentPrototypeInstanceByImageContextModel := new(vpcclassicv1.VolumeAttachmentPrototypeInstanceByImageContext)
				volumeAttachmentPrototypeInstanceByImageContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceByImageContextModel.Volume = volumePrototypeInstanceByImageContextModel

				// Construct an instance of the ImageIdentityByID model
				imageIdentityModel := new(vpcclassicv1.ImageIdentityByID)
				imageIdentityModel.ID = core.StringPtr("9aaf3bcb-dcd7-4de7-bb60-24e39ff9d366")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the InstancePrototypeInstanceByImage model
				instancePrototypeModel := new(vpcclassicv1.InstancePrototypeInstanceByImage)
				instancePrototypeModel.Keys = []vpcclassicv1.KeyIdentityIntf{keyIdentityModel}
				instancePrototypeModel.Name = core.StringPtr("my-instance")
				instancePrototypeModel.NetworkInterfaces = []vpcclassicv1.NetworkInterfacePrototype{*networkInterfacePrototypeModel}
				instancePrototypeModel.Profile = instanceProfileIdentityModel
				instancePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				instancePrototypeModel.UserData = core.StringPtr("testString")
				instancePrototypeModel.VolumeAttachments = []vpcclassicv1.VolumeAttachmentPrototypeInstanceContext{*volumeAttachmentPrototypeInstanceContextModel}
				instancePrototypeModel.VPC = vpcIdentityModel
				instancePrototypeModel.BootVolumeAttachment = volumeAttachmentPrototypeInstanceByImageContextModel
				instancePrototypeModel.Image = imageIdentityModel
				instancePrototypeModel.PrimaryNetworkInterface = networkInterfacePrototypeModel
				instancePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateInstanceOptions model
				createInstanceOptionsModel := new(vpcclassicv1.CreateInstanceOptions)
				createInstanceOptionsModel.InstancePrototype = instancePrototypeModel
				createInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.CreateInstance(createInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.CreateInstance(createInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`CreateInstance(createInstanceOptions *CreateInstanceOptions)`, func() {
		version := "testString"
		createInstancePath := "/instances"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstancePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"bandwidth": 1000, "boot_volume_attachment": {"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"crn": "crn:v1:bluemix:public:is:us-south-1:::instance-profile:bc1-4x16", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "deleting", "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateInstance successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the KeyIdentityByID model
				keyIdentityModel := new(vpcclassicv1.KeyIdentityByID)
				keyIdentityModel.ID = core.StringPtr("363f6d70-0000-0001-0000-00000013b96c")

				// Construct an instance of the SecurityGroupIdentityByID model
				securityGroupIdentityModel := new(vpcclassicv1.SecurityGroupIdentityByID)
				securityGroupIdentityModel.ID = core.StringPtr("be5df5ca-12a0-494b-907e-aa6ec2bfa271")

				// Construct an instance of the SubnetIdentityByID model
				subnetIdentityModel := new(vpcclassicv1.SubnetIdentityByID)
				subnetIdentityModel.ID = core.StringPtr("7ec86020-1c6e-4889-b3f0-a15f2e50f87e")

				// Construct an instance of the NetworkInterfacePrototype model
				networkInterfacePrototypeModel := new(vpcclassicv1.NetworkInterfacePrototype)
				networkInterfacePrototypeModel.Name = core.StringPtr("my-network-interface")
				networkInterfacePrototypeModel.PrimaryIpv4Address = core.StringPtr("10.0.0.5")
				networkInterfacePrototypeModel.SecurityGroups = []vpcclassicv1.SecurityGroupIdentityIntf{securityGroupIdentityModel}
				networkInterfacePrototypeModel.Subnet = subnetIdentityModel

				// Construct an instance of the InstanceProfileIdentityByName model
				instanceProfileIdentityModel := new(vpcclassicv1.InstanceProfileIdentityByName)
				instanceProfileIdentityModel.Name = core.StringPtr("bx2-2x8")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcclassicv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:[...]")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcclassicv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("5iops-tier")

				// Construct an instance of the VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity model
				volumeAttachmentVolumePrototypeInstanceContextModel := new(vpcclassicv1.VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity)
				volumeAttachmentVolumePrototypeInstanceContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumeAttachmentVolumePrototypeInstanceContextModel.Iops = core.Int64Ptr(int64(10000))
				volumeAttachmentVolumePrototypeInstanceContextModel.Name = core.StringPtr("my-data-volume")
				volumeAttachmentVolumePrototypeInstanceContextModel.Profile = volumeProfileIdentityModel
				volumeAttachmentVolumePrototypeInstanceContextModel.Capacity = core.Int64Ptr(int64(1000))

				// Construct an instance of the VolumeAttachmentPrototypeInstanceContext model
				volumeAttachmentPrototypeInstanceContextModel := new(vpcclassicv1.VolumeAttachmentPrototypeInstanceContext)
				volumeAttachmentPrototypeInstanceContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceContextModel.Volume = volumeAttachmentVolumePrototypeInstanceContextModel

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcclassicv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("f0aae929-7047-46d1-92e1-9102b07a7f6f")

				// Construct an instance of the VolumePrototypeInstanceByImageContext model
				volumePrototypeInstanceByImageContextModel := new(vpcclassicv1.VolumePrototypeInstanceByImageContext)
				volumePrototypeInstanceByImageContextModel.Capacity = core.Int64Ptr(int64(100))
				volumePrototypeInstanceByImageContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeInstanceByImageContextModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeInstanceByImageContextModel.Name = core.StringPtr("my-boot-volume")
				volumePrototypeInstanceByImageContextModel.Profile = volumeProfileIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceByImageContext model
				volumeAttachmentPrototypeInstanceByImageContextModel := new(vpcclassicv1.VolumeAttachmentPrototypeInstanceByImageContext)
				volumeAttachmentPrototypeInstanceByImageContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceByImageContextModel.Volume = volumePrototypeInstanceByImageContextModel

				// Construct an instance of the ImageIdentityByID model
				imageIdentityModel := new(vpcclassicv1.ImageIdentityByID)
				imageIdentityModel.ID = core.StringPtr("9aaf3bcb-dcd7-4de7-bb60-24e39ff9d366")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the InstancePrototypeInstanceByImage model
				instancePrototypeModel := new(vpcclassicv1.InstancePrototypeInstanceByImage)
				instancePrototypeModel.Keys = []vpcclassicv1.KeyIdentityIntf{keyIdentityModel}
				instancePrototypeModel.Name = core.StringPtr("my-instance")
				instancePrototypeModel.NetworkInterfaces = []vpcclassicv1.NetworkInterfacePrototype{*networkInterfacePrototypeModel}
				instancePrototypeModel.Profile = instanceProfileIdentityModel
				instancePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				instancePrototypeModel.UserData = core.StringPtr("testString")
				instancePrototypeModel.VolumeAttachments = []vpcclassicv1.VolumeAttachmentPrototypeInstanceContext{*volumeAttachmentPrototypeInstanceContextModel}
				instancePrototypeModel.VPC = vpcIdentityModel
				instancePrototypeModel.BootVolumeAttachment = volumeAttachmentPrototypeInstanceByImageContextModel
				instancePrototypeModel.Image = imageIdentityModel
				instancePrototypeModel.PrimaryNetworkInterface = networkInterfacePrototypeModel
				instancePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateInstanceOptions model
				createInstanceOptionsModel := new(vpcclassicv1.CreateInstanceOptions)
				createInstanceOptionsModel.InstancePrototype = instancePrototypeModel
				createInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.CreateInstanceWithContext(ctx, createInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.CreateInstance(createInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.CreateInstanceWithContext(ctx, createInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstancePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"bandwidth": 1000, "boot_volume_attachment": {"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"crn": "crn:v1:bluemix:public:is:us-south-1:::instance-profile:bc1-4x16", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "deleting", "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateInstance successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.CreateInstance(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the KeyIdentityByID model
				keyIdentityModel := new(vpcclassicv1.KeyIdentityByID)
				keyIdentityModel.ID = core.StringPtr("363f6d70-0000-0001-0000-00000013b96c")

				// Construct an instance of the SecurityGroupIdentityByID model
				securityGroupIdentityModel := new(vpcclassicv1.SecurityGroupIdentityByID)
				securityGroupIdentityModel.ID = core.StringPtr("be5df5ca-12a0-494b-907e-aa6ec2bfa271")

				// Construct an instance of the SubnetIdentityByID model
				subnetIdentityModel := new(vpcclassicv1.SubnetIdentityByID)
				subnetIdentityModel.ID = core.StringPtr("7ec86020-1c6e-4889-b3f0-a15f2e50f87e")

				// Construct an instance of the NetworkInterfacePrototype model
				networkInterfacePrototypeModel := new(vpcclassicv1.NetworkInterfacePrototype)
				networkInterfacePrototypeModel.Name = core.StringPtr("my-network-interface")
				networkInterfacePrototypeModel.PrimaryIpv4Address = core.StringPtr("10.0.0.5")
				networkInterfacePrototypeModel.SecurityGroups = []vpcclassicv1.SecurityGroupIdentityIntf{securityGroupIdentityModel}
				networkInterfacePrototypeModel.Subnet = subnetIdentityModel

				// Construct an instance of the InstanceProfileIdentityByName model
				instanceProfileIdentityModel := new(vpcclassicv1.InstanceProfileIdentityByName)
				instanceProfileIdentityModel.Name = core.StringPtr("bx2-2x8")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcclassicv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:[...]")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcclassicv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("5iops-tier")

				// Construct an instance of the VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity model
				volumeAttachmentVolumePrototypeInstanceContextModel := new(vpcclassicv1.VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity)
				volumeAttachmentVolumePrototypeInstanceContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumeAttachmentVolumePrototypeInstanceContextModel.Iops = core.Int64Ptr(int64(10000))
				volumeAttachmentVolumePrototypeInstanceContextModel.Name = core.StringPtr("my-data-volume")
				volumeAttachmentVolumePrototypeInstanceContextModel.Profile = volumeProfileIdentityModel
				volumeAttachmentVolumePrototypeInstanceContextModel.Capacity = core.Int64Ptr(int64(1000))

				// Construct an instance of the VolumeAttachmentPrototypeInstanceContext model
				volumeAttachmentPrototypeInstanceContextModel := new(vpcclassicv1.VolumeAttachmentPrototypeInstanceContext)
				volumeAttachmentPrototypeInstanceContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceContextModel.Volume = volumeAttachmentVolumePrototypeInstanceContextModel

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcclassicv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("f0aae929-7047-46d1-92e1-9102b07a7f6f")

				// Construct an instance of the VolumePrototypeInstanceByImageContext model
				volumePrototypeInstanceByImageContextModel := new(vpcclassicv1.VolumePrototypeInstanceByImageContext)
				volumePrototypeInstanceByImageContextModel.Capacity = core.Int64Ptr(int64(100))
				volumePrototypeInstanceByImageContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeInstanceByImageContextModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeInstanceByImageContextModel.Name = core.StringPtr("my-boot-volume")
				volumePrototypeInstanceByImageContextModel.Profile = volumeProfileIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceByImageContext model
				volumeAttachmentPrototypeInstanceByImageContextModel := new(vpcclassicv1.VolumeAttachmentPrototypeInstanceByImageContext)
				volumeAttachmentPrototypeInstanceByImageContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceByImageContextModel.Volume = volumePrototypeInstanceByImageContextModel

				// Construct an instance of the ImageIdentityByID model
				imageIdentityModel := new(vpcclassicv1.ImageIdentityByID)
				imageIdentityModel.ID = core.StringPtr("9aaf3bcb-dcd7-4de7-bb60-24e39ff9d366")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the InstancePrototypeInstanceByImage model
				instancePrototypeModel := new(vpcclassicv1.InstancePrototypeInstanceByImage)
				instancePrototypeModel.Keys = []vpcclassicv1.KeyIdentityIntf{keyIdentityModel}
				instancePrototypeModel.Name = core.StringPtr("my-instance")
				instancePrototypeModel.NetworkInterfaces = []vpcclassicv1.NetworkInterfacePrototype{*networkInterfacePrototypeModel}
				instancePrototypeModel.Profile = instanceProfileIdentityModel
				instancePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				instancePrototypeModel.UserData = core.StringPtr("testString")
				instancePrototypeModel.VolumeAttachments = []vpcclassicv1.VolumeAttachmentPrototypeInstanceContext{*volumeAttachmentPrototypeInstanceContextModel}
				instancePrototypeModel.VPC = vpcIdentityModel
				instancePrototypeModel.BootVolumeAttachment = volumeAttachmentPrototypeInstanceByImageContextModel
				instancePrototypeModel.Image = imageIdentityModel
				instancePrototypeModel.PrimaryNetworkInterface = networkInterfacePrototypeModel
				instancePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateInstanceOptions model
				createInstanceOptionsModel := new(vpcclassicv1.CreateInstanceOptions)
				createInstanceOptionsModel.InstancePrototype = instancePrototypeModel
				createInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.CreateInstance(createInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateInstance with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the KeyIdentityByID model
				keyIdentityModel := new(vpcclassicv1.KeyIdentityByID)
				keyIdentityModel.ID = core.StringPtr("363f6d70-0000-0001-0000-00000013b96c")

				// Construct an instance of the SecurityGroupIdentityByID model
				securityGroupIdentityModel := new(vpcclassicv1.SecurityGroupIdentityByID)
				securityGroupIdentityModel.ID = core.StringPtr("be5df5ca-12a0-494b-907e-aa6ec2bfa271")

				// Construct an instance of the SubnetIdentityByID model
				subnetIdentityModel := new(vpcclassicv1.SubnetIdentityByID)
				subnetIdentityModel.ID = core.StringPtr("7ec86020-1c6e-4889-b3f0-a15f2e50f87e")

				// Construct an instance of the NetworkInterfacePrototype model
				networkInterfacePrototypeModel := new(vpcclassicv1.NetworkInterfacePrototype)
				networkInterfacePrototypeModel.Name = core.StringPtr("my-network-interface")
				networkInterfacePrototypeModel.PrimaryIpv4Address = core.StringPtr("10.0.0.5")
				networkInterfacePrototypeModel.SecurityGroups = []vpcclassicv1.SecurityGroupIdentityIntf{securityGroupIdentityModel}
				networkInterfacePrototypeModel.Subnet = subnetIdentityModel

				// Construct an instance of the InstanceProfileIdentityByName model
				instanceProfileIdentityModel := new(vpcclassicv1.InstanceProfileIdentityByName)
				instanceProfileIdentityModel.Name = core.StringPtr("bx2-2x8")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcclassicv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:[...]")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcclassicv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("5iops-tier")

				// Construct an instance of the VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity model
				volumeAttachmentVolumePrototypeInstanceContextModel := new(vpcclassicv1.VolumeAttachmentVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumePrototypeInstanceContextVolumeByCapacity)
				volumeAttachmentVolumePrototypeInstanceContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumeAttachmentVolumePrototypeInstanceContextModel.Iops = core.Int64Ptr(int64(10000))
				volumeAttachmentVolumePrototypeInstanceContextModel.Name = core.StringPtr("my-data-volume")
				volumeAttachmentVolumePrototypeInstanceContextModel.Profile = volumeProfileIdentityModel
				volumeAttachmentVolumePrototypeInstanceContextModel.Capacity = core.Int64Ptr(int64(1000))

				// Construct an instance of the VolumeAttachmentPrototypeInstanceContext model
				volumeAttachmentPrototypeInstanceContextModel := new(vpcclassicv1.VolumeAttachmentPrototypeInstanceContext)
				volumeAttachmentPrototypeInstanceContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceContextModel.Volume = volumeAttachmentVolumePrototypeInstanceContextModel

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcclassicv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("f0aae929-7047-46d1-92e1-9102b07a7f6f")

				// Construct an instance of the VolumePrototypeInstanceByImageContext model
				volumePrototypeInstanceByImageContextModel := new(vpcclassicv1.VolumePrototypeInstanceByImageContext)
				volumePrototypeInstanceByImageContextModel.Capacity = core.Int64Ptr(int64(100))
				volumePrototypeInstanceByImageContextModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeInstanceByImageContextModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeInstanceByImageContextModel.Name = core.StringPtr("my-boot-volume")
				volumePrototypeInstanceByImageContextModel.Profile = volumeProfileIdentityModel

				// Construct an instance of the VolumeAttachmentPrototypeInstanceByImageContext model
				volumeAttachmentPrototypeInstanceByImageContextModel := new(vpcclassicv1.VolumeAttachmentPrototypeInstanceByImageContext)
				volumeAttachmentPrototypeInstanceByImageContextModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPrototypeInstanceByImageContextModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPrototypeInstanceByImageContextModel.Volume = volumePrototypeInstanceByImageContextModel

				// Construct an instance of the ImageIdentityByID model
				imageIdentityModel := new(vpcclassicv1.ImageIdentityByID)
				imageIdentityModel.ID = core.StringPtr("9aaf3bcb-dcd7-4de7-bb60-24e39ff9d366")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the InstancePrototypeInstanceByImage model
				instancePrototypeModel := new(vpcclassicv1.InstancePrototypeInstanceByImage)
				instancePrototypeModel.Keys = []vpcclassicv1.KeyIdentityIntf{keyIdentityModel}
				instancePrototypeModel.Name = core.StringPtr("my-instance")
				instancePrototypeModel.NetworkInterfaces = []vpcclassicv1.NetworkInterfacePrototype{*networkInterfacePrototypeModel}
				instancePrototypeModel.Profile = instanceProfileIdentityModel
				instancePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				instancePrototypeModel.UserData = core.StringPtr("testString")
				instancePrototypeModel.VolumeAttachments = []vpcclassicv1.VolumeAttachmentPrototypeInstanceContext{*volumeAttachmentPrototypeInstanceContextModel}
				instancePrototypeModel.VPC = vpcIdentityModel
				instancePrototypeModel.BootVolumeAttachment = volumeAttachmentPrototypeInstanceByImageContextModel
				instancePrototypeModel.Image = imageIdentityModel
				instancePrototypeModel.PrimaryNetworkInterface = networkInterfacePrototypeModel
				instancePrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateInstanceOptions model
				createInstanceOptionsModel := new(vpcclassicv1.CreateInstanceOptions)
				createInstanceOptionsModel.InstancePrototype = instancePrototypeModel
				createInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.CreateInstance(createInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateInstanceOptions model with no property values
				createInstanceOptionsModelNew := new(vpcclassicv1.CreateInstanceOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.CreateInstance(createInstanceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`DeleteInstance(deleteInstanceOptions *DeleteInstanceOptions)`, func() {
		version := "testString"
		deleteInstancePath := "/instances/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteInstancePath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteInstance successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcClassicService.DeleteInstance(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteInstanceOptions model
				deleteInstanceOptionsModel := new(vpcclassicv1.DeleteInstanceOptions)
				deleteInstanceOptionsModel.ID = core.StringPtr("testString")
				deleteInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcClassicService.DeleteInstance(deleteInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteInstance with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the DeleteInstanceOptions model
				deleteInstanceOptionsModel := new(vpcclassicv1.DeleteInstanceOptions)
				deleteInstanceOptionsModel.ID = core.StringPtr("testString")
				deleteInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcClassicService.DeleteInstance(deleteInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteInstanceOptions model with no property values
				deleteInstanceOptionsModelNew := new(vpcclassicv1.DeleteInstanceOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcClassicService.DeleteInstance(deleteInstanceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetInstance(getInstanceOptions *GetInstanceOptions) - Operation response error`, func() {
		version := "testString"
		getInstancePath := "/instances/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstancePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetInstance with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetInstanceOptions model
				getInstanceOptionsModel := new(vpcclassicv1.GetInstanceOptions)
				getInstanceOptionsModel.ID = core.StringPtr("testString")
				getInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetInstance(getInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetInstance(getInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetInstance(getInstanceOptions *GetInstanceOptions)`, func() {
		version := "testString"
		getInstancePath := "/instances/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstancePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"bandwidth": 1000, "boot_volume_attachment": {"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"crn": "crn:v1:bluemix:public:is:us-south-1:::instance-profile:bc1-4x16", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "deleting", "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetInstance successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetInstanceOptions model
				getInstanceOptionsModel := new(vpcclassicv1.GetInstanceOptions)
				getInstanceOptionsModel.ID = core.StringPtr("testString")
				getInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetInstanceWithContext(ctx, getInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetInstance(getInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetInstanceWithContext(ctx, getInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstancePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"bandwidth": 1000, "boot_volume_attachment": {"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"crn": "crn:v1:bluemix:public:is:us-south-1:::instance-profile:bc1-4x16", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "deleting", "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetInstance successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetInstance(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetInstanceOptions model
				getInstanceOptionsModel := new(vpcclassicv1.GetInstanceOptions)
				getInstanceOptionsModel.ID = core.StringPtr("testString")
				getInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetInstance(getInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetInstance with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetInstanceOptions model
				getInstanceOptionsModel := new(vpcclassicv1.GetInstanceOptions)
				getInstanceOptionsModel.ID = core.StringPtr("testString")
				getInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetInstance(getInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetInstanceOptions model with no property values
				getInstanceOptionsModelNew := new(vpcclassicv1.GetInstanceOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetInstance(getInstanceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateInstance(updateInstanceOptions *UpdateInstanceOptions) - Operation response error`, func() {
		version := "testString"
		updateInstancePath := "/instances/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateInstancePath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateInstance with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the InstancePatch model
				instancePatchModel := new(vpcclassicv1.InstancePatch)
				instancePatchModel.Name = core.StringPtr("my-instance")
				instancePatchModelAsPatch, asPatchErr := instancePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceOptions model
				updateInstanceOptionsModel := new(vpcclassicv1.UpdateInstanceOptions)
				updateInstanceOptionsModel.ID = core.StringPtr("testString")
				updateInstanceOptionsModel.InstancePatch = instancePatchModelAsPatch
				updateInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.UpdateInstance(updateInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.UpdateInstance(updateInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`UpdateInstance(updateInstanceOptions *UpdateInstanceOptions)`, func() {
		version := "testString"
		updateInstancePath := "/instances/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateInstancePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"bandwidth": 1000, "boot_volume_attachment": {"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"crn": "crn:v1:bluemix:public:is:us-south-1:::instance-profile:bc1-4x16", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "deleting", "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateInstance successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the InstancePatch model
				instancePatchModel := new(vpcclassicv1.InstancePatch)
				instancePatchModel.Name = core.StringPtr("my-instance")
				instancePatchModelAsPatch, asPatchErr := instancePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceOptions model
				updateInstanceOptionsModel := new(vpcclassicv1.UpdateInstanceOptions)
				updateInstanceOptionsModel.ID = core.StringPtr("testString")
				updateInstanceOptionsModel.InstancePatch = instancePatchModelAsPatch
				updateInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.UpdateInstanceWithContext(ctx, updateInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.UpdateInstance(updateInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.UpdateInstanceWithContext(ctx, updateInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateInstancePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"bandwidth": 1000, "boot_volume_attachment": {"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "image": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::image:72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "href": "https://us-south.iaas.cloud.ibm.com/v1/images/72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "id": "72b27b5c-f4b0-48bb-b954-5becc7c1dcb8", "name": "my-image"}, "memory": 8, "name": "my-instance", "network_interfaces": [{"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}], "primary_network_interface": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}}, "profile": {"crn": "crn:v1:bluemix:public:is:us-south-1:::instance-profile:bc1-4x16", "href": "https://us-south.iaas.cloud.ibm.com/v1/instance/profiles/bc1-4x16", "name": "bc1-4x16"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "deleting", "vcpu": {"architecture": "amd64", "count": 4}, "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}], "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateInstance successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.UpdateInstance(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the InstancePatch model
				instancePatchModel := new(vpcclassicv1.InstancePatch)
				instancePatchModel.Name = core.StringPtr("my-instance")
				instancePatchModelAsPatch, asPatchErr := instancePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceOptions model
				updateInstanceOptionsModel := new(vpcclassicv1.UpdateInstanceOptions)
				updateInstanceOptionsModel.ID = core.StringPtr("testString")
				updateInstanceOptionsModel.InstancePatch = instancePatchModelAsPatch
				updateInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.UpdateInstance(updateInstanceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateInstance with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the InstancePatch model
				instancePatchModel := new(vpcclassicv1.InstancePatch)
				instancePatchModel.Name = core.StringPtr("my-instance")
				instancePatchModelAsPatch, asPatchErr := instancePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceOptions model
				updateInstanceOptionsModel := new(vpcclassicv1.UpdateInstanceOptions)
				updateInstanceOptionsModel.ID = core.StringPtr("testString")
				updateInstanceOptionsModel.InstancePatch = instancePatchModelAsPatch
				updateInstanceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.UpdateInstance(updateInstanceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateInstanceOptions model with no property values
				updateInstanceOptionsModelNew := new(vpcclassicv1.UpdateInstanceOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.UpdateInstance(updateInstanceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetInstanceInitialization(getInstanceInitializationOptions *GetInstanceInitializationOptions) - Operation response error`, func() {
		version := "testString"
		getInstanceInitializationPath := "/instances/testString/initialization"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceInitializationPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetInstanceInitialization with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetInstanceInitializationOptions model
				getInstanceInitializationOptionsModel := new(vpcclassicv1.GetInstanceInitializationOptions)
				getInstanceInitializationOptionsModel.ID = core.StringPtr("testString")
				getInstanceInitializationOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetInstanceInitialization(getInstanceInitializationOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetInstanceInitialization(getInstanceInitializationOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetInstanceInitialization(getInstanceInitializationOptions *GetInstanceInitializationOptions)`, func() {
		version := "testString"
		getInstanceInitializationPath := "/instances/testString/initialization"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceInitializationPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"keys": [{"crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "name": "my-key"}], "password": {"encrypted_password": "VGhpcyBpcyBhbiBlbmNvZGVkIGJ5dGUgYXJyYXku", "encryption_key": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "name": "my-key"}}}`)
				}))
			})
			It(`Invoke GetInstanceInitialization successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetInstanceInitializationOptions model
				getInstanceInitializationOptionsModel := new(vpcclassicv1.GetInstanceInitializationOptions)
				getInstanceInitializationOptionsModel.ID = core.StringPtr("testString")
				getInstanceInitializationOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetInstanceInitializationWithContext(ctx, getInstanceInitializationOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetInstanceInitialization(getInstanceInitializationOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetInstanceInitializationWithContext(ctx, getInstanceInitializationOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceInitializationPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"keys": [{"crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "name": "my-key"}], "password": {"encrypted_password": "VGhpcyBpcyBhbiBlbmNvZGVkIGJ5dGUgYXJyYXku", "encryption_key": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::key:a6b1a881-2ce8-41a3-80fc-36316a73f803", "fingerprint": "SHA256:yxavE4CIOL2NlsqcurRO3xGjkP6m/0mp8ugojH5yxlY", "href": "https://us-south.iaas.cloud.ibm.com/v1/keys/a6b1a881-2ce8-41a3-80fc-36316a73f803", "id": "a6b1a881-2ce8-41a3-80fc-36316a73f803", "name": "my-key"}}}`)
				}))
			})
			It(`Invoke GetInstanceInitialization successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetInstanceInitialization(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetInstanceInitializationOptions model
				getInstanceInitializationOptionsModel := new(vpcclassicv1.GetInstanceInitializationOptions)
				getInstanceInitializationOptionsModel.ID = core.StringPtr("testString")
				getInstanceInitializationOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetInstanceInitialization(getInstanceInitializationOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetInstanceInitialization with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetInstanceInitializationOptions model
				getInstanceInitializationOptionsModel := new(vpcclassicv1.GetInstanceInitializationOptions)
				getInstanceInitializationOptionsModel.ID = core.StringPtr("testString")
				getInstanceInitializationOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetInstanceInitialization(getInstanceInitializationOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetInstanceInitializationOptions model with no property values
				getInstanceInitializationOptionsModelNew := new(vpcclassicv1.GetInstanceInitializationOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetInstanceInitialization(getInstanceInitializationOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateInstanceAction(createInstanceActionOptions *CreateInstanceActionOptions) - Operation response error`, func() {
		version := "testString"
		createInstanceActionPath := "/instances/testString/actions"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstanceActionPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateInstanceAction with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the CreateInstanceActionOptions model
				createInstanceActionOptionsModel := new(vpcclassicv1.CreateInstanceActionOptions)
				createInstanceActionOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceActionOptionsModel.Type = core.StringPtr("reboot")
				createInstanceActionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.CreateInstanceAction(createInstanceActionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.CreateInstanceAction(createInstanceActionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`CreateInstanceAction(createInstanceActionOptions *CreateInstanceActionOptions)`, func() {
		version := "testString"
		createInstanceActionPath := "/instances/testString/actions"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstanceActionPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"completed_at": "2019-01-01T12:00:00.000Z", "created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/actions/109a1b6e-1242-4de1-be44-38705e9474ed", "id": "109a1b6e-1242-4de1-be44-38705e9474ed", "started_at": "2019-01-01T12:00:00.000Z", "status": "completed", "type": "reboot"}`)
				}))
			})
			It(`Invoke CreateInstanceAction successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the CreateInstanceActionOptions model
				createInstanceActionOptionsModel := new(vpcclassicv1.CreateInstanceActionOptions)
				createInstanceActionOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceActionOptionsModel.Type = core.StringPtr("reboot")
				createInstanceActionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.CreateInstanceActionWithContext(ctx, createInstanceActionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.CreateInstanceAction(createInstanceActionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.CreateInstanceActionWithContext(ctx, createInstanceActionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstanceActionPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"completed_at": "2019-01-01T12:00:00.000Z", "created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/actions/109a1b6e-1242-4de1-be44-38705e9474ed", "id": "109a1b6e-1242-4de1-be44-38705e9474ed", "started_at": "2019-01-01T12:00:00.000Z", "status": "completed", "type": "reboot"}`)
				}))
			})
			It(`Invoke CreateInstanceAction successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.CreateInstanceAction(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the CreateInstanceActionOptions model
				createInstanceActionOptionsModel := new(vpcclassicv1.CreateInstanceActionOptions)
				createInstanceActionOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceActionOptionsModel.Type = core.StringPtr("reboot")
				createInstanceActionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.CreateInstanceAction(createInstanceActionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateInstanceAction with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the CreateInstanceActionOptions model
				createInstanceActionOptionsModel := new(vpcclassicv1.CreateInstanceActionOptions)
				createInstanceActionOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceActionOptionsModel.Type = core.StringPtr("reboot")
				createInstanceActionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.CreateInstanceAction(createInstanceActionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateInstanceActionOptions model with no property values
				createInstanceActionOptionsModelNew := new(vpcclassicv1.CreateInstanceActionOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.CreateInstanceAction(createInstanceActionOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListInstanceNetworkInterfaces(listInstanceNetworkInterfacesOptions *ListInstanceNetworkInterfacesOptions) - Operation response error`, func() {
		version := "testString"
		listInstanceNetworkInterfacesPath := "/instances/testString/network_interfaces"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceNetworkInterfacesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListInstanceNetworkInterfaces with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListInstanceNetworkInterfacesOptions model
				listInstanceNetworkInterfacesOptionsModel := new(vpcclassicv1.ListInstanceNetworkInterfacesOptions)
				listInstanceNetworkInterfacesOptionsModel.InstanceID = core.StringPtr("testString")
				listInstanceNetworkInterfacesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListInstanceNetworkInterfaces(listInstanceNetworkInterfacesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListInstanceNetworkInterfaces(listInstanceNetworkInterfacesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListInstanceNetworkInterfaces(listInstanceNetworkInterfacesOptions *ListInstanceNetworkInterfacesOptions)`, func() {
		version := "testString"
		listInstanceNetworkInterfacesPath := "/instances/testString/network_interfaces"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceNetworkInterfacesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"network_interfaces": [{"created_at": "2019-01-01T12:00:00.000Z", "floating_ips": [{"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}], "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "port_speed": 1000, "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "security_groups": [{"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}], "status": "available", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}, "type": "primary"}]}`)
				}))
			})
			It(`Invoke ListInstanceNetworkInterfaces successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListInstanceNetworkInterfacesOptions model
				listInstanceNetworkInterfacesOptionsModel := new(vpcclassicv1.ListInstanceNetworkInterfacesOptions)
				listInstanceNetworkInterfacesOptionsModel.InstanceID = core.StringPtr("testString")
				listInstanceNetworkInterfacesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListInstanceNetworkInterfacesWithContext(ctx, listInstanceNetworkInterfacesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListInstanceNetworkInterfaces(listInstanceNetworkInterfacesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListInstanceNetworkInterfacesWithContext(ctx, listInstanceNetworkInterfacesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceNetworkInterfacesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"network_interfaces": [{"created_at": "2019-01-01T12:00:00.000Z", "floating_ips": [{"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}], "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "port_speed": 1000, "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "security_groups": [{"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}], "status": "available", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}, "type": "primary"}]}`)
				}))
			})
			It(`Invoke ListInstanceNetworkInterfaces successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListInstanceNetworkInterfaces(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListInstanceNetworkInterfacesOptions model
				listInstanceNetworkInterfacesOptionsModel := new(vpcclassicv1.ListInstanceNetworkInterfacesOptions)
				listInstanceNetworkInterfacesOptionsModel.InstanceID = core.StringPtr("testString")
				listInstanceNetworkInterfacesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListInstanceNetworkInterfaces(listInstanceNetworkInterfacesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListInstanceNetworkInterfaces with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListInstanceNetworkInterfacesOptions model
				listInstanceNetworkInterfacesOptionsModel := new(vpcclassicv1.ListInstanceNetworkInterfacesOptions)
				listInstanceNetworkInterfacesOptionsModel.InstanceID = core.StringPtr("testString")
				listInstanceNetworkInterfacesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListInstanceNetworkInterfaces(listInstanceNetworkInterfacesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListInstanceNetworkInterfacesOptions model with no property values
				listInstanceNetworkInterfacesOptionsModelNew := new(vpcclassicv1.ListInstanceNetworkInterfacesOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.ListInstanceNetworkInterfaces(listInstanceNetworkInterfacesOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetInstanceNetworkInterface(getInstanceNetworkInterfaceOptions *GetInstanceNetworkInterfaceOptions) - Operation response error`, func() {
		version := "testString"
		getInstanceNetworkInterfacePath := "/instances/testString/network_interfaces/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceNetworkInterfacePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetInstanceNetworkInterface with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetInstanceNetworkInterfaceOptions model
				getInstanceNetworkInterfaceOptionsModel := new(vpcclassicv1.GetInstanceNetworkInterfaceOptions)
				getInstanceNetworkInterfaceOptionsModel.InstanceID = core.StringPtr("testString")
				getInstanceNetworkInterfaceOptionsModel.ID = core.StringPtr("testString")
				getInstanceNetworkInterfaceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetInstanceNetworkInterface(getInstanceNetworkInterfaceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetInstanceNetworkInterface(getInstanceNetworkInterfaceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetInstanceNetworkInterface(getInstanceNetworkInterfaceOptions *GetInstanceNetworkInterfaceOptions)`, func() {
		version := "testString"
		getInstanceNetworkInterfacePath := "/instances/testString/network_interfaces/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceNetworkInterfacePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "floating_ips": [{"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}], "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "port_speed": 1000, "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "security_groups": [{"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}], "status": "available", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}, "type": "primary"}`)
				}))
			})
			It(`Invoke GetInstanceNetworkInterface successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetInstanceNetworkInterfaceOptions model
				getInstanceNetworkInterfaceOptionsModel := new(vpcclassicv1.GetInstanceNetworkInterfaceOptions)
				getInstanceNetworkInterfaceOptionsModel.InstanceID = core.StringPtr("testString")
				getInstanceNetworkInterfaceOptionsModel.ID = core.StringPtr("testString")
				getInstanceNetworkInterfaceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetInstanceNetworkInterfaceWithContext(ctx, getInstanceNetworkInterfaceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetInstanceNetworkInterface(getInstanceNetworkInterfaceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetInstanceNetworkInterfaceWithContext(ctx, getInstanceNetworkInterfaceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceNetworkInterfacePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "floating_ips": [{"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}], "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "port_speed": 1000, "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface", "security_groups": [{"crn": "crn:v1:bluemix:public:is:us-south:a/123456::security-group:be5df5ca-12a0-494b-907e-aa6ec2bfa271", "href": "https://us-south.iaas.cloud.ibm.com/v1/security_groups/be5df5ca-12a0-494b-907e-aa6ec2bfa271", "id": "be5df5ca-12a0-494b-907e-aa6ec2bfa271", "name": "my-security-group"}], "status": "available", "subnet": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}, "type": "primary"}`)
				}))
			})
			It(`Invoke GetInstanceNetworkInterface successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetInstanceNetworkInterface(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetInstanceNetworkInterfaceOptions model
				getInstanceNetworkInterfaceOptionsModel := new(vpcclassicv1.GetInstanceNetworkInterfaceOptions)
				getInstanceNetworkInterfaceOptionsModel.InstanceID = core.StringPtr("testString")
				getInstanceNetworkInterfaceOptionsModel.ID = core.StringPtr("testString")
				getInstanceNetworkInterfaceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetInstanceNetworkInterface(getInstanceNetworkInterfaceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetInstanceNetworkInterface with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetInstanceNetworkInterfaceOptions model
				getInstanceNetworkInterfaceOptionsModel := new(vpcclassicv1.GetInstanceNetworkInterfaceOptions)
				getInstanceNetworkInterfaceOptionsModel.InstanceID = core.StringPtr("testString")
				getInstanceNetworkInterfaceOptionsModel.ID = core.StringPtr("testString")
				getInstanceNetworkInterfaceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetInstanceNetworkInterface(getInstanceNetworkInterfaceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetInstanceNetworkInterfaceOptions model with no property values
				getInstanceNetworkInterfaceOptionsModelNew := new(vpcclassicv1.GetInstanceNetworkInterfaceOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetInstanceNetworkInterface(getInstanceNetworkInterfaceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListInstanceNetworkInterfaceFloatingIps(listInstanceNetworkInterfaceFloatingIpsOptions *ListInstanceNetworkInterfaceFloatingIpsOptions) - Operation response error`, func() {
		version := "testString"
		listInstanceNetworkInterfaceFloatingIpsPath := "/instances/testString/network_interfaces/testString/floating_ips"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceNetworkInterfaceFloatingIpsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListInstanceNetworkInterfaceFloatingIps with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListInstanceNetworkInterfaceFloatingIpsOptions model
				listInstanceNetworkInterfaceFloatingIpsOptionsModel := new(vpcclassicv1.ListInstanceNetworkInterfaceFloatingIpsOptions)
				listInstanceNetworkInterfaceFloatingIpsOptionsModel.InstanceID = core.StringPtr("testString")
				listInstanceNetworkInterfaceFloatingIpsOptionsModel.NetworkInterfaceID = core.StringPtr("testString")
				listInstanceNetworkInterfaceFloatingIpsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListInstanceNetworkInterfaceFloatingIps(listInstanceNetworkInterfaceFloatingIpsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListInstanceNetworkInterfaceFloatingIps(listInstanceNetworkInterfaceFloatingIpsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListInstanceNetworkInterfaceFloatingIps(listInstanceNetworkInterfaceFloatingIpsOptions *ListInstanceNetworkInterfaceFloatingIpsOptions)`, func() {
		version := "testString"
		listInstanceNetworkInterfaceFloatingIpsPath := "/instances/testString/network_interfaces/testString/floating_ips"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceNetworkInterfaceFloatingIpsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"floating_ips": [{"address": "203.0.113.1", "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip", "status": "available", "target": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}]}`)
				}))
			})
			It(`Invoke ListInstanceNetworkInterfaceFloatingIps successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListInstanceNetworkInterfaceFloatingIpsOptions model
				listInstanceNetworkInterfaceFloatingIpsOptionsModel := new(vpcclassicv1.ListInstanceNetworkInterfaceFloatingIpsOptions)
				listInstanceNetworkInterfaceFloatingIpsOptionsModel.InstanceID = core.StringPtr("testString")
				listInstanceNetworkInterfaceFloatingIpsOptionsModel.NetworkInterfaceID = core.StringPtr("testString")
				listInstanceNetworkInterfaceFloatingIpsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListInstanceNetworkInterfaceFloatingIpsWithContext(ctx, listInstanceNetworkInterfaceFloatingIpsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListInstanceNetworkInterfaceFloatingIps(listInstanceNetworkInterfaceFloatingIpsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListInstanceNetworkInterfaceFloatingIpsWithContext(ctx, listInstanceNetworkInterfaceFloatingIpsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceNetworkInterfaceFloatingIpsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"floating_ips": [{"address": "203.0.113.1", "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip", "status": "available", "target": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}]}`)
				}))
			})
			It(`Invoke ListInstanceNetworkInterfaceFloatingIps successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListInstanceNetworkInterfaceFloatingIps(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListInstanceNetworkInterfaceFloatingIpsOptions model
				listInstanceNetworkInterfaceFloatingIpsOptionsModel := new(vpcclassicv1.ListInstanceNetworkInterfaceFloatingIpsOptions)
				listInstanceNetworkInterfaceFloatingIpsOptionsModel.InstanceID = core.StringPtr("testString")
				listInstanceNetworkInterfaceFloatingIpsOptionsModel.NetworkInterfaceID = core.StringPtr("testString")
				listInstanceNetworkInterfaceFloatingIpsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListInstanceNetworkInterfaceFloatingIps(listInstanceNetworkInterfaceFloatingIpsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListInstanceNetworkInterfaceFloatingIps with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListInstanceNetworkInterfaceFloatingIpsOptions model
				listInstanceNetworkInterfaceFloatingIpsOptionsModel := new(vpcclassicv1.ListInstanceNetworkInterfaceFloatingIpsOptions)
				listInstanceNetworkInterfaceFloatingIpsOptionsModel.InstanceID = core.StringPtr("testString")
				listInstanceNetworkInterfaceFloatingIpsOptionsModel.NetworkInterfaceID = core.StringPtr("testString")
				listInstanceNetworkInterfaceFloatingIpsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListInstanceNetworkInterfaceFloatingIps(listInstanceNetworkInterfaceFloatingIpsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListInstanceNetworkInterfaceFloatingIpsOptions model with no property values
				listInstanceNetworkInterfaceFloatingIpsOptionsModelNew := new(vpcclassicv1.ListInstanceNetworkInterfaceFloatingIpsOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.ListInstanceNetworkInterfaceFloatingIps(listInstanceNetworkInterfaceFloatingIpsOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`RemoveInstanceNetworkInterfaceFloatingIP(removeInstanceNetworkInterfaceFloatingIPOptions *RemoveInstanceNetworkInterfaceFloatingIPOptions)`, func() {
		version := "testString"
		removeInstanceNetworkInterfaceFloatingIPPath := "/instances/testString/network_interfaces/testString/floating_ips/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(removeInstanceNetworkInterfaceFloatingIPPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.WriteHeader(204)
				}))
			})
			It(`Invoke RemoveInstanceNetworkInterfaceFloatingIP successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcClassicService.RemoveInstanceNetworkInterfaceFloatingIP(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the RemoveInstanceNetworkInterfaceFloatingIPOptions model
				removeInstanceNetworkInterfaceFloatingIPOptionsModel := new(vpcclassicv1.RemoveInstanceNetworkInterfaceFloatingIPOptions)
				removeInstanceNetworkInterfaceFloatingIPOptionsModel.InstanceID = core.StringPtr("testString")
				removeInstanceNetworkInterfaceFloatingIPOptionsModel.NetworkInterfaceID = core.StringPtr("testString")
				removeInstanceNetworkInterfaceFloatingIPOptionsModel.ID = core.StringPtr("testString")
				removeInstanceNetworkInterfaceFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcClassicService.RemoveInstanceNetworkInterfaceFloatingIP(removeInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke RemoveInstanceNetworkInterfaceFloatingIP with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the RemoveInstanceNetworkInterfaceFloatingIPOptions model
				removeInstanceNetworkInterfaceFloatingIPOptionsModel := new(vpcclassicv1.RemoveInstanceNetworkInterfaceFloatingIPOptions)
				removeInstanceNetworkInterfaceFloatingIPOptionsModel.InstanceID = core.StringPtr("testString")
				removeInstanceNetworkInterfaceFloatingIPOptionsModel.NetworkInterfaceID = core.StringPtr("testString")
				removeInstanceNetworkInterfaceFloatingIPOptionsModel.ID = core.StringPtr("testString")
				removeInstanceNetworkInterfaceFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcClassicService.RemoveInstanceNetworkInterfaceFloatingIP(removeInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the RemoveInstanceNetworkInterfaceFloatingIPOptions model with no property values
				removeInstanceNetworkInterfaceFloatingIPOptionsModelNew := new(vpcclassicv1.RemoveInstanceNetworkInterfaceFloatingIPOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcClassicService.RemoveInstanceNetworkInterfaceFloatingIP(removeInstanceNetworkInterfaceFloatingIPOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetInstanceNetworkInterfaceFloatingIP(getInstanceNetworkInterfaceFloatingIPOptions *GetInstanceNetworkInterfaceFloatingIPOptions) - Operation response error`, func() {
		version := "testString"
		getInstanceNetworkInterfaceFloatingIPPath := "/instances/testString/network_interfaces/testString/floating_ips/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceNetworkInterfaceFloatingIPPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetInstanceNetworkInterfaceFloatingIP with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetInstanceNetworkInterfaceFloatingIPOptions model
				getInstanceNetworkInterfaceFloatingIPOptionsModel := new(vpcclassicv1.GetInstanceNetworkInterfaceFloatingIPOptions)
				getInstanceNetworkInterfaceFloatingIPOptionsModel.InstanceID = core.StringPtr("testString")
				getInstanceNetworkInterfaceFloatingIPOptionsModel.NetworkInterfaceID = core.StringPtr("testString")
				getInstanceNetworkInterfaceFloatingIPOptionsModel.ID = core.StringPtr("testString")
				getInstanceNetworkInterfaceFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetInstanceNetworkInterfaceFloatingIP(getInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetInstanceNetworkInterfaceFloatingIP(getInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetInstanceNetworkInterfaceFloatingIP(getInstanceNetworkInterfaceFloatingIPOptions *GetInstanceNetworkInterfaceFloatingIPOptions)`, func() {
		version := "testString"
		getInstanceNetworkInterfaceFloatingIPPath := "/instances/testString/network_interfaces/testString/floating_ips/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceNetworkInterfaceFloatingIPPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"address": "203.0.113.1", "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip", "status": "available", "target": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetInstanceNetworkInterfaceFloatingIP successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetInstanceNetworkInterfaceFloatingIPOptions model
				getInstanceNetworkInterfaceFloatingIPOptionsModel := new(vpcclassicv1.GetInstanceNetworkInterfaceFloatingIPOptions)
				getInstanceNetworkInterfaceFloatingIPOptionsModel.InstanceID = core.StringPtr("testString")
				getInstanceNetworkInterfaceFloatingIPOptionsModel.NetworkInterfaceID = core.StringPtr("testString")
				getInstanceNetworkInterfaceFloatingIPOptionsModel.ID = core.StringPtr("testString")
				getInstanceNetworkInterfaceFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetInstanceNetworkInterfaceFloatingIPWithContext(ctx, getInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetInstanceNetworkInterfaceFloatingIP(getInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetInstanceNetworkInterfaceFloatingIPWithContext(ctx, getInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceNetworkInterfaceFloatingIPPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"address": "203.0.113.1", "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip", "status": "available", "target": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetInstanceNetworkInterfaceFloatingIP successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetInstanceNetworkInterfaceFloatingIP(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetInstanceNetworkInterfaceFloatingIPOptions model
				getInstanceNetworkInterfaceFloatingIPOptionsModel := new(vpcclassicv1.GetInstanceNetworkInterfaceFloatingIPOptions)
				getInstanceNetworkInterfaceFloatingIPOptionsModel.InstanceID = core.StringPtr("testString")
				getInstanceNetworkInterfaceFloatingIPOptionsModel.NetworkInterfaceID = core.StringPtr("testString")
				getInstanceNetworkInterfaceFloatingIPOptionsModel.ID = core.StringPtr("testString")
				getInstanceNetworkInterfaceFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetInstanceNetworkInterfaceFloatingIP(getInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetInstanceNetworkInterfaceFloatingIP with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetInstanceNetworkInterfaceFloatingIPOptions model
				getInstanceNetworkInterfaceFloatingIPOptionsModel := new(vpcclassicv1.GetInstanceNetworkInterfaceFloatingIPOptions)
				getInstanceNetworkInterfaceFloatingIPOptionsModel.InstanceID = core.StringPtr("testString")
				getInstanceNetworkInterfaceFloatingIPOptionsModel.NetworkInterfaceID = core.StringPtr("testString")
				getInstanceNetworkInterfaceFloatingIPOptionsModel.ID = core.StringPtr("testString")
				getInstanceNetworkInterfaceFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetInstanceNetworkInterfaceFloatingIP(getInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetInstanceNetworkInterfaceFloatingIPOptions model with no property values
				getInstanceNetworkInterfaceFloatingIPOptionsModelNew := new(vpcclassicv1.GetInstanceNetworkInterfaceFloatingIPOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetInstanceNetworkInterfaceFloatingIP(getInstanceNetworkInterfaceFloatingIPOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`AddInstanceNetworkInterfaceFloatingIP(addInstanceNetworkInterfaceFloatingIPOptions *AddInstanceNetworkInterfaceFloatingIPOptions) - Operation response error`, func() {
		version := "testString"
		addInstanceNetworkInterfaceFloatingIPPath := "/instances/testString/network_interfaces/testString/floating_ips/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(addInstanceNetworkInterfaceFloatingIPPath))
					Expect(req.Method).To(Equal("PUT"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke AddInstanceNetworkInterfaceFloatingIP with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the AddInstanceNetworkInterfaceFloatingIPOptions model
				addInstanceNetworkInterfaceFloatingIPOptionsModel := new(vpcclassicv1.AddInstanceNetworkInterfaceFloatingIPOptions)
				addInstanceNetworkInterfaceFloatingIPOptionsModel.InstanceID = core.StringPtr("testString")
				addInstanceNetworkInterfaceFloatingIPOptionsModel.NetworkInterfaceID = core.StringPtr("testString")
				addInstanceNetworkInterfaceFloatingIPOptionsModel.ID = core.StringPtr("testString")
				addInstanceNetworkInterfaceFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.AddInstanceNetworkInterfaceFloatingIP(addInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.AddInstanceNetworkInterfaceFloatingIP(addInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`AddInstanceNetworkInterfaceFloatingIP(addInstanceNetworkInterfaceFloatingIPOptions *AddInstanceNetworkInterfaceFloatingIPOptions)`, func() {
		version := "testString"
		addInstanceNetworkInterfaceFloatingIPPath := "/instances/testString/network_interfaces/testString/floating_ips/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(addInstanceNetworkInterfaceFloatingIPPath))
					Expect(req.Method).To(Equal("PUT"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"address": "203.0.113.1", "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip", "status": "available", "target": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke AddInstanceNetworkInterfaceFloatingIP successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the AddInstanceNetworkInterfaceFloatingIPOptions model
				addInstanceNetworkInterfaceFloatingIPOptionsModel := new(vpcclassicv1.AddInstanceNetworkInterfaceFloatingIPOptions)
				addInstanceNetworkInterfaceFloatingIPOptionsModel.InstanceID = core.StringPtr("testString")
				addInstanceNetworkInterfaceFloatingIPOptionsModel.NetworkInterfaceID = core.StringPtr("testString")
				addInstanceNetworkInterfaceFloatingIPOptionsModel.ID = core.StringPtr("testString")
				addInstanceNetworkInterfaceFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.AddInstanceNetworkInterfaceFloatingIPWithContext(ctx, addInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.AddInstanceNetworkInterfaceFloatingIP(addInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.AddInstanceNetworkInterfaceFloatingIPWithContext(ctx, addInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(addInstanceNetworkInterfaceFloatingIPPath))
					Expect(req.Method).To(Equal("PUT"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"address": "203.0.113.1", "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip", "status": "available", "target": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke AddInstanceNetworkInterfaceFloatingIP successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.AddInstanceNetworkInterfaceFloatingIP(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the AddInstanceNetworkInterfaceFloatingIPOptions model
				addInstanceNetworkInterfaceFloatingIPOptionsModel := new(vpcclassicv1.AddInstanceNetworkInterfaceFloatingIPOptions)
				addInstanceNetworkInterfaceFloatingIPOptionsModel.InstanceID = core.StringPtr("testString")
				addInstanceNetworkInterfaceFloatingIPOptionsModel.NetworkInterfaceID = core.StringPtr("testString")
				addInstanceNetworkInterfaceFloatingIPOptionsModel.ID = core.StringPtr("testString")
				addInstanceNetworkInterfaceFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.AddInstanceNetworkInterfaceFloatingIP(addInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke AddInstanceNetworkInterfaceFloatingIP with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the AddInstanceNetworkInterfaceFloatingIPOptions model
				addInstanceNetworkInterfaceFloatingIPOptionsModel := new(vpcclassicv1.AddInstanceNetworkInterfaceFloatingIPOptions)
				addInstanceNetworkInterfaceFloatingIPOptionsModel.InstanceID = core.StringPtr("testString")
				addInstanceNetworkInterfaceFloatingIPOptionsModel.NetworkInterfaceID = core.StringPtr("testString")
				addInstanceNetworkInterfaceFloatingIPOptionsModel.ID = core.StringPtr("testString")
				addInstanceNetworkInterfaceFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.AddInstanceNetworkInterfaceFloatingIP(addInstanceNetworkInterfaceFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the AddInstanceNetworkInterfaceFloatingIPOptions model with no property values
				addInstanceNetworkInterfaceFloatingIPOptionsModelNew := new(vpcclassicv1.AddInstanceNetworkInterfaceFloatingIPOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.AddInstanceNetworkInterfaceFloatingIP(addInstanceNetworkInterfaceFloatingIPOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListInstanceVolumeAttachments(listInstanceVolumeAttachmentsOptions *ListInstanceVolumeAttachmentsOptions) - Operation response error`, func() {
		version := "testString"
		listInstanceVolumeAttachmentsPath := "/instances/testString/volume_attachments"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceVolumeAttachmentsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListInstanceVolumeAttachments with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListInstanceVolumeAttachmentsOptions model
				listInstanceVolumeAttachmentsOptionsModel := new(vpcclassicv1.ListInstanceVolumeAttachmentsOptions)
				listInstanceVolumeAttachmentsOptionsModel.InstanceID = core.StringPtr("testString")
				listInstanceVolumeAttachmentsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListInstanceVolumeAttachments(listInstanceVolumeAttachmentsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListInstanceVolumeAttachments(listInstanceVolumeAttachmentsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListInstanceVolumeAttachments(listInstanceVolumeAttachmentsOptions *ListInstanceVolumeAttachmentsOptions)`, func() {
		version := "testString"
		listInstanceVolumeAttachmentsPath := "/instances/testString/volume_attachments"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceVolumeAttachmentsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"volume_attachments": [{"created_at": "2019-01-01T12:00:00.000Z", "delete_volume_on_instance_delete": true, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "status": "attached", "type": "boot", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}]}`)
				}))
			})
			It(`Invoke ListInstanceVolumeAttachments successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListInstanceVolumeAttachmentsOptions model
				listInstanceVolumeAttachmentsOptionsModel := new(vpcclassicv1.ListInstanceVolumeAttachmentsOptions)
				listInstanceVolumeAttachmentsOptionsModel.InstanceID = core.StringPtr("testString")
				listInstanceVolumeAttachmentsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListInstanceVolumeAttachmentsWithContext(ctx, listInstanceVolumeAttachmentsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListInstanceVolumeAttachments(listInstanceVolumeAttachmentsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListInstanceVolumeAttachmentsWithContext(ctx, listInstanceVolumeAttachmentsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listInstanceVolumeAttachmentsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"volume_attachments": [{"created_at": "2019-01-01T12:00:00.000Z", "delete_volume_on_instance_delete": true, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "status": "attached", "type": "boot", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}]}`)
				}))
			})
			It(`Invoke ListInstanceVolumeAttachments successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListInstanceVolumeAttachments(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListInstanceVolumeAttachmentsOptions model
				listInstanceVolumeAttachmentsOptionsModel := new(vpcclassicv1.ListInstanceVolumeAttachmentsOptions)
				listInstanceVolumeAttachmentsOptionsModel.InstanceID = core.StringPtr("testString")
				listInstanceVolumeAttachmentsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListInstanceVolumeAttachments(listInstanceVolumeAttachmentsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListInstanceVolumeAttachments with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListInstanceVolumeAttachmentsOptions model
				listInstanceVolumeAttachmentsOptionsModel := new(vpcclassicv1.ListInstanceVolumeAttachmentsOptions)
				listInstanceVolumeAttachmentsOptionsModel.InstanceID = core.StringPtr("testString")
				listInstanceVolumeAttachmentsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListInstanceVolumeAttachments(listInstanceVolumeAttachmentsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListInstanceVolumeAttachmentsOptions model with no property values
				listInstanceVolumeAttachmentsOptionsModelNew := new(vpcclassicv1.ListInstanceVolumeAttachmentsOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.ListInstanceVolumeAttachments(listInstanceVolumeAttachmentsOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateInstanceVolumeAttachment(createInstanceVolumeAttachmentOptions *CreateInstanceVolumeAttachmentOptions) - Operation response error`, func() {
		version := "testString"
		createInstanceVolumeAttachmentPath := "/instances/testString/volume_attachments"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstanceVolumeAttachmentPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateInstanceVolumeAttachment with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the VolumeIdentityByID model
				volumeIdentityModel := new(vpcclassicv1.VolumeIdentityByID)
				volumeIdentityModel.ID = core.StringPtr("1a6b7274-678d-4dfb-8981-c71dd9d4daa5")

				// Construct an instance of the CreateInstanceVolumeAttachmentOptions model
				createInstanceVolumeAttachmentOptionsModel := new(vpcclassicv1.CreateInstanceVolumeAttachmentOptions)
				createInstanceVolumeAttachmentOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceVolumeAttachmentOptionsModel.Volume = volumeIdentityModel
				createInstanceVolumeAttachmentOptionsModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				createInstanceVolumeAttachmentOptionsModel.Name = core.StringPtr("my-volume-attachment")
				createInstanceVolumeAttachmentOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.CreateInstanceVolumeAttachment(createInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.CreateInstanceVolumeAttachment(createInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`CreateInstanceVolumeAttachment(createInstanceVolumeAttachmentOptions *CreateInstanceVolumeAttachmentOptions)`, func() {
		version := "testString"
		createInstanceVolumeAttachmentPath := "/instances/testString/volume_attachments"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstanceVolumeAttachmentPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "delete_volume_on_instance_delete": true, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "status": "attached", "type": "boot", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}`)
				}))
			})
			It(`Invoke CreateInstanceVolumeAttachment successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the VolumeIdentityByID model
				volumeIdentityModel := new(vpcclassicv1.VolumeIdentityByID)
				volumeIdentityModel.ID = core.StringPtr("1a6b7274-678d-4dfb-8981-c71dd9d4daa5")

				// Construct an instance of the CreateInstanceVolumeAttachmentOptions model
				createInstanceVolumeAttachmentOptionsModel := new(vpcclassicv1.CreateInstanceVolumeAttachmentOptions)
				createInstanceVolumeAttachmentOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceVolumeAttachmentOptionsModel.Volume = volumeIdentityModel
				createInstanceVolumeAttachmentOptionsModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				createInstanceVolumeAttachmentOptionsModel.Name = core.StringPtr("my-volume-attachment")
				createInstanceVolumeAttachmentOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.CreateInstanceVolumeAttachmentWithContext(ctx, createInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.CreateInstanceVolumeAttachment(createInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.CreateInstanceVolumeAttachmentWithContext(ctx, createInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createInstanceVolumeAttachmentPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "delete_volume_on_instance_delete": true, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "status": "attached", "type": "boot", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}`)
				}))
			})
			It(`Invoke CreateInstanceVolumeAttachment successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.CreateInstanceVolumeAttachment(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the VolumeIdentityByID model
				volumeIdentityModel := new(vpcclassicv1.VolumeIdentityByID)
				volumeIdentityModel.ID = core.StringPtr("1a6b7274-678d-4dfb-8981-c71dd9d4daa5")

				// Construct an instance of the CreateInstanceVolumeAttachmentOptions model
				createInstanceVolumeAttachmentOptionsModel := new(vpcclassicv1.CreateInstanceVolumeAttachmentOptions)
				createInstanceVolumeAttachmentOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceVolumeAttachmentOptionsModel.Volume = volumeIdentityModel
				createInstanceVolumeAttachmentOptionsModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				createInstanceVolumeAttachmentOptionsModel.Name = core.StringPtr("my-volume-attachment")
				createInstanceVolumeAttachmentOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.CreateInstanceVolumeAttachment(createInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateInstanceVolumeAttachment with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the VolumeIdentityByID model
				volumeIdentityModel := new(vpcclassicv1.VolumeIdentityByID)
				volumeIdentityModel.ID = core.StringPtr("1a6b7274-678d-4dfb-8981-c71dd9d4daa5")

				// Construct an instance of the CreateInstanceVolumeAttachmentOptions model
				createInstanceVolumeAttachmentOptionsModel := new(vpcclassicv1.CreateInstanceVolumeAttachmentOptions)
				createInstanceVolumeAttachmentOptionsModel.InstanceID = core.StringPtr("testString")
				createInstanceVolumeAttachmentOptionsModel.Volume = volumeIdentityModel
				createInstanceVolumeAttachmentOptionsModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				createInstanceVolumeAttachmentOptionsModel.Name = core.StringPtr("my-volume-attachment")
				createInstanceVolumeAttachmentOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.CreateInstanceVolumeAttachment(createInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateInstanceVolumeAttachmentOptions model with no property values
				createInstanceVolumeAttachmentOptionsModelNew := new(vpcclassicv1.CreateInstanceVolumeAttachmentOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.CreateInstanceVolumeAttachment(createInstanceVolumeAttachmentOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`DeleteInstanceVolumeAttachment(deleteInstanceVolumeAttachmentOptions *DeleteInstanceVolumeAttachmentOptions)`, func() {
		version := "testString"
		deleteInstanceVolumeAttachmentPath := "/instances/testString/volume_attachments/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteInstanceVolumeAttachmentPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteInstanceVolumeAttachment successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcClassicService.DeleteInstanceVolumeAttachment(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteInstanceVolumeAttachmentOptions model
				deleteInstanceVolumeAttachmentOptionsModel := new(vpcclassicv1.DeleteInstanceVolumeAttachmentOptions)
				deleteInstanceVolumeAttachmentOptionsModel.InstanceID = core.StringPtr("testString")
				deleteInstanceVolumeAttachmentOptionsModel.ID = core.StringPtr("testString")
				deleteInstanceVolumeAttachmentOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcClassicService.DeleteInstanceVolumeAttachment(deleteInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteInstanceVolumeAttachment with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the DeleteInstanceVolumeAttachmentOptions model
				deleteInstanceVolumeAttachmentOptionsModel := new(vpcclassicv1.DeleteInstanceVolumeAttachmentOptions)
				deleteInstanceVolumeAttachmentOptionsModel.InstanceID = core.StringPtr("testString")
				deleteInstanceVolumeAttachmentOptionsModel.ID = core.StringPtr("testString")
				deleteInstanceVolumeAttachmentOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcClassicService.DeleteInstanceVolumeAttachment(deleteInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteInstanceVolumeAttachmentOptions model with no property values
				deleteInstanceVolumeAttachmentOptionsModelNew := new(vpcclassicv1.DeleteInstanceVolumeAttachmentOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcClassicService.DeleteInstanceVolumeAttachment(deleteInstanceVolumeAttachmentOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetInstanceVolumeAttachment(getInstanceVolumeAttachmentOptions *GetInstanceVolumeAttachmentOptions) - Operation response error`, func() {
		version := "testString"
		getInstanceVolumeAttachmentPath := "/instances/testString/volume_attachments/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceVolumeAttachmentPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetInstanceVolumeAttachment with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetInstanceVolumeAttachmentOptions model
				getInstanceVolumeAttachmentOptionsModel := new(vpcclassicv1.GetInstanceVolumeAttachmentOptions)
				getInstanceVolumeAttachmentOptionsModel.InstanceID = core.StringPtr("testString")
				getInstanceVolumeAttachmentOptionsModel.ID = core.StringPtr("testString")
				getInstanceVolumeAttachmentOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetInstanceVolumeAttachment(getInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetInstanceVolumeAttachment(getInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetInstanceVolumeAttachment(getInstanceVolumeAttachmentOptions *GetInstanceVolumeAttachmentOptions)`, func() {
		version := "testString"
		getInstanceVolumeAttachmentPath := "/instances/testString/volume_attachments/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceVolumeAttachmentPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "delete_volume_on_instance_delete": true, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "status": "attached", "type": "boot", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}`)
				}))
			})
			It(`Invoke GetInstanceVolumeAttachment successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetInstanceVolumeAttachmentOptions model
				getInstanceVolumeAttachmentOptionsModel := new(vpcclassicv1.GetInstanceVolumeAttachmentOptions)
				getInstanceVolumeAttachmentOptionsModel.InstanceID = core.StringPtr("testString")
				getInstanceVolumeAttachmentOptionsModel.ID = core.StringPtr("testString")
				getInstanceVolumeAttachmentOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetInstanceVolumeAttachmentWithContext(ctx, getInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetInstanceVolumeAttachment(getInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetInstanceVolumeAttachmentWithContext(ctx, getInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getInstanceVolumeAttachmentPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "delete_volume_on_instance_delete": true, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "status": "attached", "type": "boot", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}`)
				}))
			})
			It(`Invoke GetInstanceVolumeAttachment successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetInstanceVolumeAttachment(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetInstanceVolumeAttachmentOptions model
				getInstanceVolumeAttachmentOptionsModel := new(vpcclassicv1.GetInstanceVolumeAttachmentOptions)
				getInstanceVolumeAttachmentOptionsModel.InstanceID = core.StringPtr("testString")
				getInstanceVolumeAttachmentOptionsModel.ID = core.StringPtr("testString")
				getInstanceVolumeAttachmentOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetInstanceVolumeAttachment(getInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetInstanceVolumeAttachment with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetInstanceVolumeAttachmentOptions model
				getInstanceVolumeAttachmentOptionsModel := new(vpcclassicv1.GetInstanceVolumeAttachmentOptions)
				getInstanceVolumeAttachmentOptionsModel.InstanceID = core.StringPtr("testString")
				getInstanceVolumeAttachmentOptionsModel.ID = core.StringPtr("testString")
				getInstanceVolumeAttachmentOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetInstanceVolumeAttachment(getInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetInstanceVolumeAttachmentOptions model with no property values
				getInstanceVolumeAttachmentOptionsModelNew := new(vpcclassicv1.GetInstanceVolumeAttachmentOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetInstanceVolumeAttachment(getInstanceVolumeAttachmentOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateInstanceVolumeAttachment(updateInstanceVolumeAttachmentOptions *UpdateInstanceVolumeAttachmentOptions) - Operation response error`, func() {
		version := "testString"
		updateInstanceVolumeAttachmentPath := "/instances/testString/volume_attachments/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateInstanceVolumeAttachmentPath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateInstanceVolumeAttachment with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the VolumeAttachmentPatch model
				volumeAttachmentPatchModel := new(vpcclassicv1.VolumeAttachmentPatch)
				volumeAttachmentPatchModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPatchModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPatchModelAsPatch, asPatchErr := volumeAttachmentPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceVolumeAttachmentOptions model
				updateInstanceVolumeAttachmentOptionsModel := new(vpcclassicv1.UpdateInstanceVolumeAttachmentOptions)
				updateInstanceVolumeAttachmentOptionsModel.InstanceID = core.StringPtr("testString")
				updateInstanceVolumeAttachmentOptionsModel.ID = core.StringPtr("testString")
				updateInstanceVolumeAttachmentOptionsModel.VolumeAttachmentPatch = volumeAttachmentPatchModelAsPatch
				updateInstanceVolumeAttachmentOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.UpdateInstanceVolumeAttachment(updateInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.UpdateInstanceVolumeAttachment(updateInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`UpdateInstanceVolumeAttachment(updateInstanceVolumeAttachmentOptions *UpdateInstanceVolumeAttachmentOptions)`, func() {
		version := "testString"
		updateInstanceVolumeAttachmentPath := "/instances/testString/volume_attachments/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateInstanceVolumeAttachmentPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "delete_volume_on_instance_delete": true, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "status": "attached", "type": "boot", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}`)
				}))
			})
			It(`Invoke UpdateInstanceVolumeAttachment successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the VolumeAttachmentPatch model
				volumeAttachmentPatchModel := new(vpcclassicv1.VolumeAttachmentPatch)
				volumeAttachmentPatchModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPatchModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPatchModelAsPatch, asPatchErr := volumeAttachmentPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceVolumeAttachmentOptions model
				updateInstanceVolumeAttachmentOptionsModel := new(vpcclassicv1.UpdateInstanceVolumeAttachmentOptions)
				updateInstanceVolumeAttachmentOptionsModel.InstanceID = core.StringPtr("testString")
				updateInstanceVolumeAttachmentOptionsModel.ID = core.StringPtr("testString")
				updateInstanceVolumeAttachmentOptionsModel.VolumeAttachmentPatch = volumeAttachmentPatchModelAsPatch
				updateInstanceVolumeAttachmentOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.UpdateInstanceVolumeAttachmentWithContext(ctx, updateInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.UpdateInstanceVolumeAttachment(updateInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.UpdateInstanceVolumeAttachmentWithContext(ctx, updateInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateInstanceVolumeAttachmentPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "delete_volume_on_instance_delete": true, "device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "name": "my-volume-attachment", "status": "attached", "type": "boot", "volume": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "name": "my-volume"}}`)
				}))
			})
			It(`Invoke UpdateInstanceVolumeAttachment successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.UpdateInstanceVolumeAttachment(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the VolumeAttachmentPatch model
				volumeAttachmentPatchModel := new(vpcclassicv1.VolumeAttachmentPatch)
				volumeAttachmentPatchModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPatchModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPatchModelAsPatch, asPatchErr := volumeAttachmentPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceVolumeAttachmentOptions model
				updateInstanceVolumeAttachmentOptionsModel := new(vpcclassicv1.UpdateInstanceVolumeAttachmentOptions)
				updateInstanceVolumeAttachmentOptionsModel.InstanceID = core.StringPtr("testString")
				updateInstanceVolumeAttachmentOptionsModel.ID = core.StringPtr("testString")
				updateInstanceVolumeAttachmentOptionsModel.VolumeAttachmentPatch = volumeAttachmentPatchModelAsPatch
				updateInstanceVolumeAttachmentOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.UpdateInstanceVolumeAttachment(updateInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateInstanceVolumeAttachment with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the VolumeAttachmentPatch model
				volumeAttachmentPatchModel := new(vpcclassicv1.VolumeAttachmentPatch)
				volumeAttachmentPatchModel.DeleteVolumeOnInstanceDelete = core.BoolPtr(true)
				volumeAttachmentPatchModel.Name = core.StringPtr("my-volume-attachment")
				volumeAttachmentPatchModelAsPatch, asPatchErr := volumeAttachmentPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateInstanceVolumeAttachmentOptions model
				updateInstanceVolumeAttachmentOptionsModel := new(vpcclassicv1.UpdateInstanceVolumeAttachmentOptions)
				updateInstanceVolumeAttachmentOptionsModel.InstanceID = core.StringPtr("testString")
				updateInstanceVolumeAttachmentOptionsModel.ID = core.StringPtr("testString")
				updateInstanceVolumeAttachmentOptionsModel.VolumeAttachmentPatch = volumeAttachmentPatchModelAsPatch
				updateInstanceVolumeAttachmentOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.UpdateInstanceVolumeAttachment(updateInstanceVolumeAttachmentOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateInstanceVolumeAttachmentOptions model with no property values
				updateInstanceVolumeAttachmentOptionsModelNew := new(vpcclassicv1.UpdateInstanceVolumeAttachmentOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.UpdateInstanceVolumeAttachment(updateInstanceVolumeAttachmentOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`Service constructor tests`, func() {
		version := "testString"
		It(`Instantiate service client`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				Authenticator: &core.NoAuthAuthenticator{},
				Version:       core.StringPtr(version),
			})
			Expect(vpcClassicService).ToNot(BeNil())
			Expect(serviceErr).To(BeNil())
		})
		It(`Instantiate service client with error: Invalid URL`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Invalid Auth`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "https://vpcclassicv1/api",
				Version: core.StringPtr(version),
				Authenticator: &core.BasicAuthenticator{
					Username: "",
					Password: "",
				},
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Validation Error`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
	})
	Describe(`Service constructor tests using external config`, func() {
		version := "testString"
		Context(`Using external config, construct service client instances`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "noauth",
			}

			It(`Create service client using external config successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url from constructor successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					URL:     "https://testService/api",
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url programatically successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				err := vpcClassicService.SetServiceURL("https://testService/api")
				Expect(err).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid Auth`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "someOtherAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid URL`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_AUTH_TYPE": "NOAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
	})
	Describe(`Regional endpoint tests`, func() {
		It(`GetServiceURLForRegion(region string)`, func() {
			var url string
			var err error
			url, err = vpcclassicv1.GetServiceURLForRegion("INVALID_REGION")
			Expect(url).To(BeEmpty())
			Expect(err).ToNot(BeNil())
			fmt.Fprintf(GinkgoWriter, "Expected error: %s\n", err.Error())
		})
	})
	Describe(`ListVolumeProfiles(listVolumeProfilesOptions *ListVolumeProfilesOptions) - Operation response error`, func() {
		version := "testString"
		listVolumeProfilesPath := "/volume/profiles"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVolumeProfilesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListVolumeProfiles with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListVolumeProfilesOptions model
				listVolumeProfilesOptionsModel := new(vpcclassicv1.ListVolumeProfilesOptions)
				listVolumeProfilesOptionsModel.Start = core.StringPtr("testString")
				listVolumeProfilesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVolumeProfilesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListVolumeProfiles(listVolumeProfilesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListVolumeProfiles(listVolumeProfilesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListVolumeProfiles(listVolumeProfilesOptions *ListVolumeProfilesOptions)`, func() {
		version := "testString"
		listVolumeProfilesPath := "/volume/profiles"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVolumeProfilesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "profiles": [{"family": "tiered", "href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles/general-purpose", "name": "general-purpose"}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListVolumeProfiles successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListVolumeProfilesOptions model
				listVolumeProfilesOptionsModel := new(vpcclassicv1.ListVolumeProfilesOptions)
				listVolumeProfilesOptionsModel.Start = core.StringPtr("testString")
				listVolumeProfilesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVolumeProfilesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListVolumeProfilesWithContext(ctx, listVolumeProfilesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListVolumeProfiles(listVolumeProfilesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListVolumeProfilesWithContext(ctx, listVolumeProfilesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVolumeProfilesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "profiles": [{"family": "tiered", "href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles/general-purpose", "name": "general-purpose"}], "total_count": 132}`)
				}))
			})
			It(`Invoke ListVolumeProfiles successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListVolumeProfiles(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListVolumeProfilesOptions model
				listVolumeProfilesOptionsModel := new(vpcclassicv1.ListVolumeProfilesOptions)
				listVolumeProfilesOptionsModel.Start = core.StringPtr("testString")
				listVolumeProfilesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVolumeProfilesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListVolumeProfiles(listVolumeProfilesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListVolumeProfiles with error: Operation request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListVolumeProfilesOptions model
				listVolumeProfilesOptionsModel := new(vpcclassicv1.ListVolumeProfilesOptions)
				listVolumeProfilesOptionsModel.Start = core.StringPtr("testString")
				listVolumeProfilesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVolumeProfilesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListVolumeProfiles(listVolumeProfilesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVolumeProfile(getVolumeProfileOptions *GetVolumeProfileOptions) - Operation response error`, func() {
		version := "testString"
		getVolumeProfilePath := "/volume/profiles/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVolumeProfilePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetVolumeProfile with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetVolumeProfileOptions model
				getVolumeProfileOptionsModel := new(vpcclassicv1.GetVolumeProfileOptions)
				getVolumeProfileOptionsModel.Name = core.StringPtr("testString")
				getVolumeProfileOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetVolumeProfile(getVolumeProfileOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetVolumeProfile(getVolumeProfileOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetVolumeProfile(getVolumeProfileOptions *GetVolumeProfileOptions)`, func() {
		version := "testString"
		getVolumeProfilePath := "/volume/profiles/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVolumeProfilePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"family": "tiered", "href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles/general-purpose", "name": "general-purpose"}`)
				}))
			})
			It(`Invoke GetVolumeProfile successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetVolumeProfileOptions model
				getVolumeProfileOptionsModel := new(vpcclassicv1.GetVolumeProfileOptions)
				getVolumeProfileOptionsModel.Name = core.StringPtr("testString")
				getVolumeProfileOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetVolumeProfileWithContext(ctx, getVolumeProfileOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetVolumeProfile(getVolumeProfileOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetVolumeProfileWithContext(ctx, getVolumeProfileOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVolumeProfilePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"family": "tiered", "href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles/general-purpose", "name": "general-purpose"}`)
				}))
			})
			It(`Invoke GetVolumeProfile successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetVolumeProfile(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetVolumeProfileOptions model
				getVolumeProfileOptionsModel := new(vpcclassicv1.GetVolumeProfileOptions)
				getVolumeProfileOptionsModel.Name = core.StringPtr("testString")
				getVolumeProfileOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetVolumeProfile(getVolumeProfileOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetVolumeProfile with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetVolumeProfileOptions model
				getVolumeProfileOptionsModel := new(vpcclassicv1.GetVolumeProfileOptions)
				getVolumeProfileOptionsModel.Name = core.StringPtr("testString")
				getVolumeProfileOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetVolumeProfile(getVolumeProfileOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetVolumeProfileOptions model with no property values
				getVolumeProfileOptionsModelNew := new(vpcclassicv1.GetVolumeProfileOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetVolumeProfile(getVolumeProfileOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListVolumes(listVolumesOptions *ListVolumesOptions) - Operation response error`, func() {
		version := "testString"
		listVolumesPath := "/volumes"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVolumesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["zone.name"]).To(Equal([]string{"testString"}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListVolumes with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListVolumesOptions model
				listVolumesOptionsModel := new(vpcclassicv1.ListVolumesOptions)
				listVolumesOptionsModel.Start = core.StringPtr("testString")
				listVolumesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVolumesOptionsModel.Name = core.StringPtr("testString")
				listVolumesOptionsModel.ZoneName = core.StringPtr("testString")
				listVolumesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListVolumes(listVolumesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListVolumes(listVolumesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListVolumes(listVolumesOptions *ListVolumesOptions)`, func() {
		version := "testString"
		listVolumesPath := "/volumes"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVolumesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["zone.name"]).To(Equal([]string{"testString"}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volumes?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volumes?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "volumes": [{"capacity": 100, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "encryption": "provider_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "iops": 10000, "name": "my-volume", "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles/general-purpose", "name": "general-purpose"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "instance": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "name": "my-instance"}, "name": "my-volume-attachment", "type": "boot"}], "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}]}`)
				}))
			})
			It(`Invoke ListVolumes successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListVolumesOptions model
				listVolumesOptionsModel := new(vpcclassicv1.ListVolumesOptions)
				listVolumesOptionsModel.Start = core.StringPtr("testString")
				listVolumesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVolumesOptionsModel.Name = core.StringPtr("testString")
				listVolumesOptionsModel.ZoneName = core.StringPtr("testString")
				listVolumesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListVolumesWithContext(ctx, listVolumesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListVolumes(listVolumesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListVolumesWithContext(ctx, listVolumesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listVolumesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["name"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["zone.name"]).To(Equal([]string{"testString"}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volumes?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volumes?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "volumes": [{"capacity": 100, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "encryption": "provider_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "iops": 10000, "name": "my-volume", "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles/general-purpose", "name": "general-purpose"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "instance": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "name": "my-instance"}, "name": "my-volume-attachment", "type": "boot"}], "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}]}`)
				}))
			})
			It(`Invoke ListVolumes successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListVolumes(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListVolumesOptions model
				listVolumesOptionsModel := new(vpcclassicv1.ListVolumesOptions)
				listVolumesOptionsModel.Start = core.StringPtr("testString")
				listVolumesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVolumesOptionsModel.Name = core.StringPtr("testString")
				listVolumesOptionsModel.ZoneName = core.StringPtr("testString")
				listVolumesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListVolumes(listVolumesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListVolumes with error: Operation request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListVolumesOptions model
				listVolumesOptionsModel := new(vpcclassicv1.ListVolumesOptions)
				listVolumesOptionsModel.Start = core.StringPtr("testString")
				listVolumesOptionsModel.Limit = core.Int64Ptr(int64(1))
				listVolumesOptionsModel.Name = core.StringPtr("testString")
				listVolumesOptionsModel.ZoneName = core.StringPtr("testString")
				listVolumesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListVolumes(listVolumesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateVolume(createVolumeOptions *CreateVolumeOptions) - Operation response error`, func() {
		version := "testString"
		createVolumePath := "/volumes"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVolumePath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateVolume with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcclassicv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcclassicv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("5iops-tier")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the VolumePrototypeVolumeByCapacity model
				volumePrototypeModel := new(vpcclassicv1.VolumePrototypeVolumeByCapacity)
				volumePrototypeModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeModel.Name = core.StringPtr("my-volume")
				volumePrototypeModel.Profile = volumeProfileIdentityModel
				volumePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				volumePrototypeModel.Zone = zoneIdentityModel
				volumePrototypeModel.Capacity = core.Int64Ptr(int64(100))

				// Construct an instance of the CreateVolumeOptions model
				createVolumeOptionsModel := new(vpcclassicv1.CreateVolumeOptions)
				createVolumeOptionsModel.VolumePrototype = volumePrototypeModel
				createVolumeOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.CreateVolume(createVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.CreateVolume(createVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`CreateVolume(createVolumeOptions *CreateVolumeOptions)`, func() {
		version := "testString"
		createVolumePath := "/volumes"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVolumePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"capacity": 100, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "encryption": "provider_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "iops": 10000, "name": "my-volume", "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles/general-purpose", "name": "general-purpose"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "instance": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "name": "my-instance"}, "name": "my-volume-attachment", "type": "boot"}], "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateVolume successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcclassicv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcclassicv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("5iops-tier")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the VolumePrototypeVolumeByCapacity model
				volumePrototypeModel := new(vpcclassicv1.VolumePrototypeVolumeByCapacity)
				volumePrototypeModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeModel.Name = core.StringPtr("my-volume")
				volumePrototypeModel.Profile = volumeProfileIdentityModel
				volumePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				volumePrototypeModel.Zone = zoneIdentityModel
				volumePrototypeModel.Capacity = core.Int64Ptr(int64(100))

				// Construct an instance of the CreateVolumeOptions model
				createVolumeOptionsModel := new(vpcclassicv1.CreateVolumeOptions)
				createVolumeOptionsModel.VolumePrototype = volumePrototypeModel
				createVolumeOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.CreateVolumeWithContext(ctx, createVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.CreateVolume(createVolumeOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.CreateVolumeWithContext(ctx, createVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createVolumePath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"capacity": 100, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "encryption": "provider_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "iops": 10000, "name": "my-volume", "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles/general-purpose", "name": "general-purpose"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "instance": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "name": "my-instance"}, "name": "my-volume-attachment", "type": "boot"}], "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateVolume successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.CreateVolume(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcclassicv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcclassicv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("5iops-tier")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the VolumePrototypeVolumeByCapacity model
				volumePrototypeModel := new(vpcclassicv1.VolumePrototypeVolumeByCapacity)
				volumePrototypeModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeModel.Name = core.StringPtr("my-volume")
				volumePrototypeModel.Profile = volumeProfileIdentityModel
				volumePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				volumePrototypeModel.Zone = zoneIdentityModel
				volumePrototypeModel.Capacity = core.Int64Ptr(int64(100))

				// Construct an instance of the CreateVolumeOptions model
				createVolumeOptionsModel := new(vpcclassicv1.CreateVolumeOptions)
				createVolumeOptionsModel.VolumePrototype = volumePrototypeModel
				createVolumeOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.CreateVolume(createVolumeOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateVolume with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the EncryptionKeyIdentityByCRN model
				encryptionKeyIdentityModel := new(vpcclassicv1.EncryptionKeyIdentityByCRN)
				encryptionKeyIdentityModel.CRN = core.StringPtr("crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179")

				// Construct an instance of the VolumeProfileIdentityByName model
				volumeProfileIdentityModel := new(vpcclassicv1.VolumeProfileIdentityByName)
				volumeProfileIdentityModel.Name = core.StringPtr("5iops-tier")

				// Construct an instance of the ResourceGroupIdentityByID model
				resourceGroupIdentityModel := new(vpcclassicv1.ResourceGroupIdentityByID)
				resourceGroupIdentityModel.ID = core.StringPtr("fee82deba12e4c0fb69c3b09d1f12345")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the VolumePrototypeVolumeByCapacity model
				volumePrototypeModel := new(vpcclassicv1.VolumePrototypeVolumeByCapacity)
				volumePrototypeModel.EncryptionKey = encryptionKeyIdentityModel
				volumePrototypeModel.Iops = core.Int64Ptr(int64(10000))
				volumePrototypeModel.Name = core.StringPtr("my-volume")
				volumePrototypeModel.Profile = volumeProfileIdentityModel
				volumePrototypeModel.ResourceGroup = resourceGroupIdentityModel
				volumePrototypeModel.Zone = zoneIdentityModel
				volumePrototypeModel.Capacity = core.Int64Ptr(int64(100))

				// Construct an instance of the CreateVolumeOptions model
				createVolumeOptionsModel := new(vpcclassicv1.CreateVolumeOptions)
				createVolumeOptionsModel.VolumePrototype = volumePrototypeModel
				createVolumeOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.CreateVolume(createVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateVolumeOptions model with no property values
				createVolumeOptionsModelNew := new(vpcclassicv1.CreateVolumeOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.CreateVolume(createVolumeOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`DeleteVolume(deleteVolumeOptions *DeleteVolumeOptions)`, func() {
		version := "testString"
		deleteVolumePath := "/volumes/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteVolumePath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteVolume successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcClassicService.DeleteVolume(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteVolumeOptions model
				deleteVolumeOptionsModel := new(vpcclassicv1.DeleteVolumeOptions)
				deleteVolumeOptionsModel.ID = core.StringPtr("testString")
				deleteVolumeOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcClassicService.DeleteVolume(deleteVolumeOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteVolume with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the DeleteVolumeOptions model
				deleteVolumeOptionsModel := new(vpcclassicv1.DeleteVolumeOptions)
				deleteVolumeOptionsModel.ID = core.StringPtr("testString")
				deleteVolumeOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcClassicService.DeleteVolume(deleteVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteVolumeOptions model with no property values
				deleteVolumeOptionsModelNew := new(vpcclassicv1.DeleteVolumeOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcClassicService.DeleteVolume(deleteVolumeOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetVolume(getVolumeOptions *GetVolumeOptions) - Operation response error`, func() {
		version := "testString"
		getVolumePath := "/volumes/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVolumePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetVolume with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetVolumeOptions model
				getVolumeOptionsModel := new(vpcclassicv1.GetVolumeOptions)
				getVolumeOptionsModel.ID = core.StringPtr("testString")
				getVolumeOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetVolume(getVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetVolume(getVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetVolume(getVolumeOptions *GetVolumeOptions)`, func() {
		version := "testString"
		getVolumePath := "/volumes/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVolumePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"capacity": 100, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "encryption": "provider_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "iops": 10000, "name": "my-volume", "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles/general-purpose", "name": "general-purpose"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "instance": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "name": "my-instance"}, "name": "my-volume-attachment", "type": "boot"}], "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetVolume successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetVolumeOptions model
				getVolumeOptionsModel := new(vpcclassicv1.GetVolumeOptions)
				getVolumeOptionsModel.ID = core.StringPtr("testString")
				getVolumeOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetVolumeWithContext(ctx, getVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetVolume(getVolumeOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetVolumeWithContext(ctx, getVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getVolumePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"capacity": 100, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "encryption": "provider_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "iops": 10000, "name": "my-volume", "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles/general-purpose", "name": "general-purpose"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "instance": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "name": "my-instance"}, "name": "my-volume-attachment", "type": "boot"}], "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetVolume successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetVolume(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetVolumeOptions model
				getVolumeOptionsModel := new(vpcclassicv1.GetVolumeOptions)
				getVolumeOptionsModel.ID = core.StringPtr("testString")
				getVolumeOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetVolume(getVolumeOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetVolume with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetVolumeOptions model
				getVolumeOptionsModel := new(vpcclassicv1.GetVolumeOptions)
				getVolumeOptionsModel.ID = core.StringPtr("testString")
				getVolumeOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetVolume(getVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetVolumeOptions model with no property values
				getVolumeOptionsModelNew := new(vpcclassicv1.GetVolumeOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetVolume(getVolumeOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateVolume(updateVolumeOptions *UpdateVolumeOptions) - Operation response error`, func() {
		version := "testString"
		updateVolumePath := "/volumes/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVolumePath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateVolume with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the VolumePatch model
				volumePatchModel := new(vpcclassicv1.VolumePatch)
				volumePatchModel.Name = core.StringPtr("my-volume")
				volumePatchModelAsPatch, asPatchErr := volumePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVolumeOptions model
				updateVolumeOptionsModel := new(vpcclassicv1.UpdateVolumeOptions)
				updateVolumeOptionsModel.ID = core.StringPtr("testString")
				updateVolumeOptionsModel.VolumePatch = volumePatchModelAsPatch
				updateVolumeOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.UpdateVolume(updateVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.UpdateVolume(updateVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`UpdateVolume(updateVolumeOptions *UpdateVolumeOptions)`, func() {
		version := "testString"
		updateVolumePath := "/volumes/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVolumePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"capacity": 100, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "encryption": "provider_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "iops": 10000, "name": "my-volume", "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles/general-purpose", "name": "general-purpose"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "instance": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "name": "my-instance"}, "name": "my-volume-attachment", "type": "boot"}], "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateVolume successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the VolumePatch model
				volumePatchModel := new(vpcclassicv1.VolumePatch)
				volumePatchModel.Name = core.StringPtr("my-volume")
				volumePatchModelAsPatch, asPatchErr := volumePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVolumeOptions model
				updateVolumeOptionsModel := new(vpcclassicv1.UpdateVolumeOptions)
				updateVolumeOptionsModel.ID = core.StringPtr("testString")
				updateVolumeOptionsModel.VolumePatch = volumePatchModelAsPatch
				updateVolumeOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.UpdateVolumeWithContext(ctx, updateVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.UpdateVolume(updateVolumeOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.UpdateVolumeWithContext(ctx, updateVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateVolumePath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"capacity": 100, "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::volume:1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "encryption": "provider_managed", "encryption_key": {"crn": "crn:v1:bluemix:public:kms:us-south:a/dffc98a0f1f0f95f6613b3b752286b87:e4a29d1a-2ef0-42a6-8fd2-350deb1c647e:key:5437653b-c4b1-447f-9646-b2a2a4cd6179"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/volumes/1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "id": "1a6b7274-678d-4dfb-8981-c71dd9d4daa5", "iops": 10000, "name": "my-volume", "profile": {"href": "https://us-south.iaas.cloud.ibm.com/v1/volume/profiles/general-purpose", "name": "general-purpose"}, "resource_group": {"href": "https://resource-controller.cloud.ibm.com/v2/resource_groups/fee82deba12e4c0fb69c3b09d1f12345", "id": "fee82deba12e4c0fb69c3b09d1f12345"}, "status": "available", "volume_attachments": [{"device": {"id": "80b3e36e-41f4-40e9-bd56-beae81792a68"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/volume_attachments/82cbf856-9cbb-45fb-b62f-d7bcef32399a", "id": "82cbf856-9cbb-45fb-b62f-d7bcef32399a", "instance": {"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::instance:1e09281b-f177-46fb-baf1-bc152b2e391a", "href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a", "id": "1e09281b-f177-46fb-baf1-bc152b2e391a", "name": "my-instance"}, "name": "my-volume-attachment", "type": "boot"}], "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateVolume successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.UpdateVolume(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the VolumePatch model
				volumePatchModel := new(vpcclassicv1.VolumePatch)
				volumePatchModel.Name = core.StringPtr("my-volume")
				volumePatchModelAsPatch, asPatchErr := volumePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVolumeOptions model
				updateVolumeOptionsModel := new(vpcclassicv1.UpdateVolumeOptions)
				updateVolumeOptionsModel.ID = core.StringPtr("testString")
				updateVolumeOptionsModel.VolumePatch = volumePatchModelAsPatch
				updateVolumeOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.UpdateVolume(updateVolumeOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateVolume with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the VolumePatch model
				volumePatchModel := new(vpcclassicv1.VolumePatch)
				volumePatchModel.Name = core.StringPtr("my-volume")
				volumePatchModelAsPatch, asPatchErr := volumePatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateVolumeOptions model
				updateVolumeOptionsModel := new(vpcclassicv1.UpdateVolumeOptions)
				updateVolumeOptionsModel.ID = core.StringPtr("testString")
				updateVolumeOptionsModel.VolumePatch = volumePatchModelAsPatch
				updateVolumeOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.UpdateVolume(updateVolumeOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateVolumeOptions model with no property values
				updateVolumeOptionsModelNew := new(vpcclassicv1.UpdateVolumeOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.UpdateVolume(updateVolumeOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`Service constructor tests`, func() {
		version := "testString"
		It(`Instantiate service client`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				Authenticator: &core.NoAuthAuthenticator{},
				Version:       core.StringPtr(version),
			})
			Expect(vpcClassicService).ToNot(BeNil())
			Expect(serviceErr).To(BeNil())
		})
		It(`Instantiate service client with error: Invalid URL`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Invalid Auth`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "https://vpcclassicv1/api",
				Version: core.StringPtr(version),
				Authenticator: &core.BasicAuthenticator{
					Username: "",
					Password: "",
				},
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Validation Error`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
	})
	Describe(`Service constructor tests using external config`, func() {
		version := "testString"
		Context(`Using external config, construct service client instances`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "noauth",
			}

			It(`Create service client using external config successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url from constructor successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					URL:     "https://testService/api",
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url programatically successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				err := vpcClassicService.SetServiceURL("https://testService/api")
				Expect(err).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid Auth`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "someOtherAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid URL`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_AUTH_TYPE": "NOAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
	})
	Describe(`Regional endpoint tests`, func() {
		It(`GetServiceURLForRegion(region string)`, func() {
			var url string
			var err error
			url, err = vpcclassicv1.GetServiceURLForRegion("INVALID_REGION")
			Expect(url).To(BeEmpty())
			Expect(err).ToNot(BeNil())
			fmt.Fprintf(GinkgoWriter, "Expected error: %s\n", err.Error())
		})
	})
	Describe(`ListRegions(listRegionsOptions *ListRegionsOptions) - Operation response error`, func() {
		version := "testString"
		listRegionsPath := "/regions"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listRegionsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListRegions with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListRegionsOptions model
				listRegionsOptionsModel := new(vpcclassicv1.ListRegionsOptions)
				listRegionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListRegions(listRegionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListRegions(listRegionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListRegions(listRegionsOptions *ListRegionsOptions)`, func() {
		version := "testString"
		listRegionsPath := "/regions"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listRegionsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"regions": [{"endpoint": "Endpoint", "href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south", "name": "us-south", "status": "available"}]}`)
				}))
			})
			It(`Invoke ListRegions successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListRegionsOptions model
				listRegionsOptionsModel := new(vpcclassicv1.ListRegionsOptions)
				listRegionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListRegionsWithContext(ctx, listRegionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListRegions(listRegionsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListRegionsWithContext(ctx, listRegionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listRegionsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"regions": [{"endpoint": "Endpoint", "href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south", "name": "us-south", "status": "available"}]}`)
				}))
			})
			It(`Invoke ListRegions successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListRegions(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListRegionsOptions model
				listRegionsOptionsModel := new(vpcclassicv1.ListRegionsOptions)
				listRegionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListRegions(listRegionsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListRegions with error: Operation request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListRegionsOptions model
				listRegionsOptionsModel := new(vpcclassicv1.ListRegionsOptions)
				listRegionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListRegions(listRegionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetRegion(getRegionOptions *GetRegionOptions) - Operation response error`, func() {
		version := "testString"
		getRegionPath := "/regions/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getRegionPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetRegion with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetRegionOptions model
				getRegionOptionsModel := new(vpcclassicv1.GetRegionOptions)
				getRegionOptionsModel.Name = core.StringPtr("testString")
				getRegionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetRegion(getRegionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetRegion(getRegionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetRegion(getRegionOptions *GetRegionOptions)`, func() {
		version := "testString"
		getRegionPath := "/regions/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getRegionPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"endpoint": "Endpoint", "href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south", "name": "us-south", "status": "available"}`)
				}))
			})
			It(`Invoke GetRegion successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetRegionOptions model
				getRegionOptionsModel := new(vpcclassicv1.GetRegionOptions)
				getRegionOptionsModel.Name = core.StringPtr("testString")
				getRegionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetRegionWithContext(ctx, getRegionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetRegion(getRegionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetRegionWithContext(ctx, getRegionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getRegionPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"endpoint": "Endpoint", "href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south", "name": "us-south", "status": "available"}`)
				}))
			})
			It(`Invoke GetRegion successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetRegion(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetRegionOptions model
				getRegionOptionsModel := new(vpcclassicv1.GetRegionOptions)
				getRegionOptionsModel.Name = core.StringPtr("testString")
				getRegionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetRegion(getRegionOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetRegion with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetRegionOptions model
				getRegionOptionsModel := new(vpcclassicv1.GetRegionOptions)
				getRegionOptionsModel.Name = core.StringPtr("testString")
				getRegionOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetRegion(getRegionOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetRegionOptions model with no property values
				getRegionOptionsModelNew := new(vpcclassicv1.GetRegionOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetRegion(getRegionOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListRegionZones(listRegionZonesOptions *ListRegionZonesOptions) - Operation response error`, func() {
		version := "testString"
		listRegionZonesPath := "/regions/testString/zones"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listRegionZonesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListRegionZones with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListRegionZonesOptions model
				listRegionZonesOptionsModel := new(vpcclassicv1.ListRegionZonesOptions)
				listRegionZonesOptionsModel.RegionName = core.StringPtr("testString")
				listRegionZonesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListRegionZones(listRegionZonesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListRegionZones(listRegionZonesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListRegionZones(listRegionZonesOptions *ListRegionZonesOptions)`, func() {
		version := "testString"
		listRegionZonesPath := "/regions/testString/zones"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listRegionZonesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"zones": [{"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1", "region": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south", "name": "us-south"}, "status": "available"}]}`)
				}))
			})
			It(`Invoke ListRegionZones successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListRegionZonesOptions model
				listRegionZonesOptionsModel := new(vpcclassicv1.ListRegionZonesOptions)
				listRegionZonesOptionsModel.RegionName = core.StringPtr("testString")
				listRegionZonesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListRegionZonesWithContext(ctx, listRegionZonesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListRegionZones(listRegionZonesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListRegionZonesWithContext(ctx, listRegionZonesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listRegionZonesPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"zones": [{"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1", "region": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south", "name": "us-south"}, "status": "available"}]}`)
				}))
			})
			It(`Invoke ListRegionZones successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListRegionZones(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListRegionZonesOptions model
				listRegionZonesOptionsModel := new(vpcclassicv1.ListRegionZonesOptions)
				listRegionZonesOptionsModel.RegionName = core.StringPtr("testString")
				listRegionZonesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListRegionZones(listRegionZonesOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListRegionZones with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListRegionZonesOptions model
				listRegionZonesOptionsModel := new(vpcclassicv1.ListRegionZonesOptions)
				listRegionZonesOptionsModel.RegionName = core.StringPtr("testString")
				listRegionZonesOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListRegionZones(listRegionZonesOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListRegionZonesOptions model with no property values
				listRegionZonesOptionsModelNew := new(vpcclassicv1.ListRegionZonesOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.ListRegionZones(listRegionZonesOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetRegionZone(getRegionZoneOptions *GetRegionZoneOptions) - Operation response error`, func() {
		version := "testString"
		getRegionZonePath := "/regions/testString/zones/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getRegionZonePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetRegionZone with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetRegionZoneOptions model
				getRegionZoneOptionsModel := new(vpcclassicv1.GetRegionZoneOptions)
				getRegionZoneOptionsModel.RegionName = core.StringPtr("testString")
				getRegionZoneOptionsModel.Name = core.StringPtr("testString")
				getRegionZoneOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetRegionZone(getRegionZoneOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetRegionZone(getRegionZoneOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetRegionZone(getRegionZoneOptions *GetRegionZoneOptions)`, func() {
		version := "testString"
		getRegionZonePath := "/regions/testString/zones/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getRegionZonePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1", "region": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south", "name": "us-south"}, "status": "available"}`)
				}))
			})
			It(`Invoke GetRegionZone successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetRegionZoneOptions model
				getRegionZoneOptionsModel := new(vpcclassicv1.GetRegionZoneOptions)
				getRegionZoneOptionsModel.RegionName = core.StringPtr("testString")
				getRegionZoneOptionsModel.Name = core.StringPtr("testString")
				getRegionZoneOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetRegionZoneWithContext(ctx, getRegionZoneOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetRegionZone(getRegionZoneOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetRegionZoneWithContext(ctx, getRegionZoneOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getRegionZonePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1", "region": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south", "name": "us-south"}, "status": "available"}`)
				}))
			})
			It(`Invoke GetRegionZone successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetRegionZone(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetRegionZoneOptions model
				getRegionZoneOptionsModel := new(vpcclassicv1.GetRegionZoneOptions)
				getRegionZoneOptionsModel.RegionName = core.StringPtr("testString")
				getRegionZoneOptionsModel.Name = core.StringPtr("testString")
				getRegionZoneOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetRegionZone(getRegionZoneOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetRegionZone with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetRegionZoneOptions model
				getRegionZoneOptionsModel := new(vpcclassicv1.GetRegionZoneOptions)
				getRegionZoneOptionsModel.RegionName = core.StringPtr("testString")
				getRegionZoneOptionsModel.Name = core.StringPtr("testString")
				getRegionZoneOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetRegionZone(getRegionZoneOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetRegionZoneOptions model with no property values
				getRegionZoneOptionsModelNew := new(vpcclassicv1.GetRegionZoneOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetRegionZone(getRegionZoneOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`Service constructor tests`, func() {
		version := "testString"
		It(`Instantiate service client`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				Authenticator: &core.NoAuthAuthenticator{},
				Version:       core.StringPtr(version),
			})
			Expect(vpcClassicService).ToNot(BeNil())
			Expect(serviceErr).To(BeNil())
		})
		It(`Instantiate service client with error: Invalid URL`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Invalid Auth`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "https://vpcclassicv1/api",
				Version: core.StringPtr(version),
				Authenticator: &core.BasicAuthenticator{
					Username: "",
					Password: "",
				},
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Validation Error`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
	})
	Describe(`Service constructor tests using external config`, func() {
		version := "testString"
		Context(`Using external config, construct service client instances`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "noauth",
			}

			It(`Create service client using external config successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url from constructor successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					URL:     "https://testService/api",
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url programatically successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				err := vpcClassicService.SetServiceURL("https://testService/api")
				Expect(err).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid Auth`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "someOtherAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid URL`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_AUTH_TYPE": "NOAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
	})
	Describe(`Regional endpoint tests`, func() {
		It(`GetServiceURLForRegion(region string)`, func() {
			var url string
			var err error
			url, err = vpcclassicv1.GetServiceURLForRegion("INVALID_REGION")
			Expect(url).To(BeEmpty())
			Expect(err).ToNot(BeNil())
			fmt.Fprintf(GinkgoWriter, "Expected error: %s\n", err.Error())
		})
	})
	Describe(`ListPublicGateways(listPublicGatewaysOptions *ListPublicGatewaysOptions) - Operation response error`, func() {
		version := "testString"
		listPublicGatewaysPath := "/public_gateways"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listPublicGatewaysPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListPublicGateways with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListPublicGatewaysOptions model
				listPublicGatewaysOptionsModel := new(vpcclassicv1.ListPublicGatewaysOptions)
				listPublicGatewaysOptionsModel.Start = core.StringPtr("testString")
				listPublicGatewaysOptionsModel.Limit = core.Int64Ptr(int64(1))
				listPublicGatewaysOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListPublicGateways(listPublicGatewaysOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListPublicGateways(listPublicGatewaysOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListPublicGateways(listPublicGatewaysOptions *ListPublicGatewaysOptions)`, func() {
		version := "testString"
		listPublicGatewaysPath := "/public_gateways"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listPublicGatewaysPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "public_gateways": [{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}]}`)
				}))
			})
			It(`Invoke ListPublicGateways successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListPublicGatewaysOptions model
				listPublicGatewaysOptionsModel := new(vpcclassicv1.ListPublicGatewaysOptions)
				listPublicGatewaysOptionsModel.Start = core.StringPtr("testString")
				listPublicGatewaysOptionsModel.Limit = core.Int64Ptr(int64(1))
				listPublicGatewaysOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListPublicGatewaysWithContext(ctx, listPublicGatewaysOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListPublicGateways(listPublicGatewaysOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListPublicGatewaysWithContext(ctx, listPublicGatewaysOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listPublicGatewaysPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways?limit=20"}, "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}, "public_gateways": [{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}]}`)
				}))
			})
			It(`Invoke ListPublicGateways successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListPublicGateways(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListPublicGatewaysOptions model
				listPublicGatewaysOptionsModel := new(vpcclassicv1.ListPublicGatewaysOptions)
				listPublicGatewaysOptionsModel.Start = core.StringPtr("testString")
				listPublicGatewaysOptionsModel.Limit = core.Int64Ptr(int64(1))
				listPublicGatewaysOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListPublicGateways(listPublicGatewaysOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListPublicGateways with error: Operation request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListPublicGatewaysOptions model
				listPublicGatewaysOptionsModel := new(vpcclassicv1.ListPublicGatewaysOptions)
				listPublicGatewaysOptionsModel.Start = core.StringPtr("testString")
				listPublicGatewaysOptionsModel.Limit = core.Int64Ptr(int64(1))
				listPublicGatewaysOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListPublicGateways(listPublicGatewaysOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreatePublicGateway(createPublicGatewayOptions *CreatePublicGatewayOptions) - Operation response error`, func() {
		version := "testString"
		createPublicGatewayPath := "/public_gateways"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createPublicGatewayPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreatePublicGateway with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcclassicv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("4727d842-f94f-4a2d-824a-9bc9b02c523b")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the PublicGatewayFloatingIPPrototypeFloatingIPIdentityFloatingIPIdentityByID model
				publicGatewayFloatingIPPrototypeModel := new(vpcclassicv1.PublicGatewayFloatingIPPrototypeFloatingIPIdentityFloatingIPIdentityByID)
				publicGatewayFloatingIPPrototypeModel.ID = core.StringPtr("39300233-9995-4806-89a5-3c1b6eb88689")

				// Construct an instance of the CreatePublicGatewayOptions model
				createPublicGatewayOptionsModel := new(vpcclassicv1.CreatePublicGatewayOptions)
				createPublicGatewayOptionsModel.VPC = vpcIdentityModel
				createPublicGatewayOptionsModel.Zone = zoneIdentityModel
				createPublicGatewayOptionsModel.FloatingIP = publicGatewayFloatingIPPrototypeModel
				createPublicGatewayOptionsModel.Name = core.StringPtr("my-public-gateway")
				createPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.CreatePublicGateway(createPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.CreatePublicGateway(createPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`CreatePublicGateway(createPublicGatewayOptions *CreatePublicGatewayOptions)`, func() {
		version := "testString"
		createPublicGatewayPath := "/public_gateways"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createPublicGatewayPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreatePublicGateway successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcclassicv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("4727d842-f94f-4a2d-824a-9bc9b02c523b")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the PublicGatewayFloatingIPPrototypeFloatingIPIdentityFloatingIPIdentityByID model
				publicGatewayFloatingIPPrototypeModel := new(vpcclassicv1.PublicGatewayFloatingIPPrototypeFloatingIPIdentityFloatingIPIdentityByID)
				publicGatewayFloatingIPPrototypeModel.ID = core.StringPtr("39300233-9995-4806-89a5-3c1b6eb88689")

				// Construct an instance of the CreatePublicGatewayOptions model
				createPublicGatewayOptionsModel := new(vpcclassicv1.CreatePublicGatewayOptions)
				createPublicGatewayOptionsModel.VPC = vpcIdentityModel
				createPublicGatewayOptionsModel.Zone = zoneIdentityModel
				createPublicGatewayOptionsModel.FloatingIP = publicGatewayFloatingIPPrototypeModel
				createPublicGatewayOptionsModel.Name = core.StringPtr("my-public-gateway")
				createPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.CreatePublicGatewayWithContext(ctx, createPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.CreatePublicGateway(createPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.CreatePublicGatewayWithContext(ctx, createPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createPublicGatewayPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreatePublicGateway successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.CreatePublicGateway(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcclassicv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("4727d842-f94f-4a2d-824a-9bc9b02c523b")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the PublicGatewayFloatingIPPrototypeFloatingIPIdentityFloatingIPIdentityByID model
				publicGatewayFloatingIPPrototypeModel := new(vpcclassicv1.PublicGatewayFloatingIPPrototypeFloatingIPIdentityFloatingIPIdentityByID)
				publicGatewayFloatingIPPrototypeModel.ID = core.StringPtr("39300233-9995-4806-89a5-3c1b6eb88689")

				// Construct an instance of the CreatePublicGatewayOptions model
				createPublicGatewayOptionsModel := new(vpcclassicv1.CreatePublicGatewayOptions)
				createPublicGatewayOptionsModel.VPC = vpcIdentityModel
				createPublicGatewayOptionsModel.Zone = zoneIdentityModel
				createPublicGatewayOptionsModel.FloatingIP = publicGatewayFloatingIPPrototypeModel
				createPublicGatewayOptionsModel.Name = core.StringPtr("my-public-gateway")
				createPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.CreatePublicGateway(createPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreatePublicGateway with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the VPCIdentityByID model
				vpcIdentityModel := new(vpcclassicv1.VPCIdentityByID)
				vpcIdentityModel.ID = core.StringPtr("4727d842-f94f-4a2d-824a-9bc9b02c523b")

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the PublicGatewayFloatingIPPrototypeFloatingIPIdentityFloatingIPIdentityByID model
				publicGatewayFloatingIPPrototypeModel := new(vpcclassicv1.PublicGatewayFloatingIPPrototypeFloatingIPIdentityFloatingIPIdentityByID)
				publicGatewayFloatingIPPrototypeModel.ID = core.StringPtr("39300233-9995-4806-89a5-3c1b6eb88689")

				// Construct an instance of the CreatePublicGatewayOptions model
				createPublicGatewayOptionsModel := new(vpcclassicv1.CreatePublicGatewayOptions)
				createPublicGatewayOptionsModel.VPC = vpcIdentityModel
				createPublicGatewayOptionsModel.Zone = zoneIdentityModel
				createPublicGatewayOptionsModel.FloatingIP = publicGatewayFloatingIPPrototypeModel
				createPublicGatewayOptionsModel.Name = core.StringPtr("my-public-gateway")
				createPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.CreatePublicGateway(createPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreatePublicGatewayOptions model with no property values
				createPublicGatewayOptionsModelNew := new(vpcclassicv1.CreatePublicGatewayOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.CreatePublicGateway(createPublicGatewayOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`DeletePublicGateway(deletePublicGatewayOptions *DeletePublicGatewayOptions)`, func() {
		version := "testString"
		deletePublicGatewayPath := "/public_gateways/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deletePublicGatewayPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeletePublicGateway successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcClassicService.DeletePublicGateway(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeletePublicGatewayOptions model
				deletePublicGatewayOptionsModel := new(vpcclassicv1.DeletePublicGatewayOptions)
				deletePublicGatewayOptionsModel.ID = core.StringPtr("testString")
				deletePublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcClassicService.DeletePublicGateway(deletePublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeletePublicGateway with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the DeletePublicGatewayOptions model
				deletePublicGatewayOptionsModel := new(vpcclassicv1.DeletePublicGatewayOptions)
				deletePublicGatewayOptionsModel.ID = core.StringPtr("testString")
				deletePublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcClassicService.DeletePublicGateway(deletePublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeletePublicGatewayOptions model with no property values
				deletePublicGatewayOptionsModelNew := new(vpcclassicv1.DeletePublicGatewayOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcClassicService.DeletePublicGateway(deletePublicGatewayOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetPublicGateway(getPublicGatewayOptions *GetPublicGatewayOptions) - Operation response error`, func() {
		version := "testString"
		getPublicGatewayPath := "/public_gateways/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getPublicGatewayPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetPublicGateway with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetPublicGatewayOptions model
				getPublicGatewayOptionsModel := new(vpcclassicv1.GetPublicGatewayOptions)
				getPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				getPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetPublicGateway(getPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetPublicGateway(getPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetPublicGateway(getPublicGatewayOptions *GetPublicGatewayOptions)`, func() {
		version := "testString"
		getPublicGatewayPath := "/public_gateways/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getPublicGatewayPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetPublicGateway successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetPublicGatewayOptions model
				getPublicGatewayOptionsModel := new(vpcclassicv1.GetPublicGatewayOptions)
				getPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				getPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetPublicGatewayWithContext(ctx, getPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetPublicGateway(getPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetPublicGatewayWithContext(ctx, getPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getPublicGatewayPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetPublicGateway successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetPublicGateway(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetPublicGatewayOptions model
				getPublicGatewayOptionsModel := new(vpcclassicv1.GetPublicGatewayOptions)
				getPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				getPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetPublicGateway(getPublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetPublicGateway with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetPublicGatewayOptions model
				getPublicGatewayOptionsModel := new(vpcclassicv1.GetPublicGatewayOptions)
				getPublicGatewayOptionsModel.ID = core.StringPtr("testString")
				getPublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetPublicGateway(getPublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetPublicGatewayOptions model with no property values
				getPublicGatewayOptionsModelNew := new(vpcclassicv1.GetPublicGatewayOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetPublicGateway(getPublicGatewayOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdatePublicGateway(updatePublicGatewayOptions *UpdatePublicGatewayOptions) - Operation response error`, func() {
		version := "testString"
		updatePublicGatewayPath := "/public_gateways/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updatePublicGatewayPath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdatePublicGateway with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the PublicGatewayPatch model
				publicGatewayPatchModel := new(vpcclassicv1.PublicGatewayPatch)
				publicGatewayPatchModel.Name = core.StringPtr("my-public-gateway")
				publicGatewayPatchModelAsPatch, asPatchErr := publicGatewayPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdatePublicGatewayOptions model
				updatePublicGatewayOptionsModel := new(vpcclassicv1.UpdatePublicGatewayOptions)
				updatePublicGatewayOptionsModel.ID = core.StringPtr("testString")
				updatePublicGatewayOptionsModel.PublicGatewayPatch = publicGatewayPatchModelAsPatch
				updatePublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.UpdatePublicGateway(updatePublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.UpdatePublicGateway(updatePublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`UpdatePublicGateway(updatePublicGatewayOptions *UpdatePublicGatewayOptions)`, func() {
		version := "testString"
		updatePublicGatewayPath := "/public_gateways/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updatePublicGatewayPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdatePublicGateway successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the PublicGatewayPatch model
				publicGatewayPatchModel := new(vpcclassicv1.PublicGatewayPatch)
				publicGatewayPatchModel.Name = core.StringPtr("my-public-gateway")
				publicGatewayPatchModelAsPatch, asPatchErr := publicGatewayPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdatePublicGatewayOptions model
				updatePublicGatewayOptionsModel := new(vpcclassicv1.UpdatePublicGatewayOptions)
				updatePublicGatewayOptionsModel.ID = core.StringPtr("testString")
				updatePublicGatewayOptionsModel.PublicGatewayPatch = publicGatewayPatchModelAsPatch
				updatePublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.UpdatePublicGatewayWithContext(ctx, updatePublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.UpdatePublicGateway(updatePublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.UpdatePublicGatewayWithContext(ctx, updatePublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updatePublicGatewayPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::public-gateway:dc5431ef-1fc6-4861-adc9-a59d077d1241", "floating_ip": {"address": "203.0.113.1", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip"}, "href": "https://us-south.iaas.cloud.ibm.com/v1/public_gateways/dc5431ef-1fc6-4861-adc9-a59d077d1241", "id": "dc5431ef-1fc6-4861-adc9-a59d077d1241", "name": "my-public-gateway", "resource_type": "public_gateway", "status": "available", "vpc": {"crn": "crn:v1:bluemix:public:is:us-south:a/123456::vpc:4727d842-f94f-4a2d-824a-9bc9b02c523b", "href": "https://us-south.iaas.cloud.ibm.com/v1/vpcs/4727d842-f94f-4a2d-824a-9bc9b02c523b", "id": "4727d842-f94f-4a2d-824a-9bc9b02c523b", "name": "my-vpc"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdatePublicGateway successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.UpdatePublicGateway(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the PublicGatewayPatch model
				publicGatewayPatchModel := new(vpcclassicv1.PublicGatewayPatch)
				publicGatewayPatchModel.Name = core.StringPtr("my-public-gateway")
				publicGatewayPatchModelAsPatch, asPatchErr := publicGatewayPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdatePublicGatewayOptions model
				updatePublicGatewayOptionsModel := new(vpcclassicv1.UpdatePublicGatewayOptions)
				updatePublicGatewayOptionsModel.ID = core.StringPtr("testString")
				updatePublicGatewayOptionsModel.PublicGatewayPatch = publicGatewayPatchModelAsPatch
				updatePublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.UpdatePublicGateway(updatePublicGatewayOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdatePublicGateway with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the PublicGatewayPatch model
				publicGatewayPatchModel := new(vpcclassicv1.PublicGatewayPatch)
				publicGatewayPatchModel.Name = core.StringPtr("my-public-gateway")
				publicGatewayPatchModelAsPatch, asPatchErr := publicGatewayPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdatePublicGatewayOptions model
				updatePublicGatewayOptionsModel := new(vpcclassicv1.UpdatePublicGatewayOptions)
				updatePublicGatewayOptionsModel.ID = core.StringPtr("testString")
				updatePublicGatewayOptionsModel.PublicGatewayPatch = publicGatewayPatchModelAsPatch
				updatePublicGatewayOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.UpdatePublicGateway(updatePublicGatewayOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdatePublicGatewayOptions model with no property values
				updatePublicGatewayOptionsModelNew := new(vpcclassicv1.UpdatePublicGatewayOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.UpdatePublicGateway(updatePublicGatewayOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`Service constructor tests`, func() {
		version := "testString"
		It(`Instantiate service client`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				Authenticator: &core.NoAuthAuthenticator{},
				Version:       core.StringPtr(version),
			})
			Expect(vpcClassicService).ToNot(BeNil())
			Expect(serviceErr).To(BeNil())
		})
		It(`Instantiate service client with error: Invalid URL`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Invalid Auth`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "https://vpcclassicv1/api",
				Version: core.StringPtr(version),
				Authenticator: &core.BasicAuthenticator{
					Username: "",
					Password: "",
				},
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Validation Error`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
	})
	Describe(`Service constructor tests using external config`, func() {
		version := "testString"
		Context(`Using external config, construct service client instances`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "noauth",
			}

			It(`Create service client using external config successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url from constructor successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					URL:     "https://testService/api",
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url programatically successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				err := vpcClassicService.SetServiceURL("https://testService/api")
				Expect(err).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid Auth`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "someOtherAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid URL`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_AUTH_TYPE": "NOAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
	})
	Describe(`Regional endpoint tests`, func() {
		It(`GetServiceURLForRegion(region string)`, func() {
			var url string
			var err error
			url, err = vpcclassicv1.GetServiceURLForRegion("INVALID_REGION")
			Expect(url).To(BeEmpty())
			Expect(err).ToNot(BeNil())
			fmt.Fprintf(GinkgoWriter, "Expected error: %s\n", err.Error())
		})
	})
	Describe(`ListFloatingIps(listFloatingIpsOptions *ListFloatingIpsOptions) - Operation response error`, func() {
		version := "testString"
		listFloatingIpsPath := "/floating_ips"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listFloatingIpsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListFloatingIps with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListFloatingIpsOptions model
				listFloatingIpsOptionsModel := new(vpcclassicv1.ListFloatingIpsOptions)
				listFloatingIpsOptionsModel.Start = core.StringPtr("testString")
				listFloatingIpsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listFloatingIpsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListFloatingIps(listFloatingIpsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListFloatingIps(listFloatingIpsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListFloatingIps(listFloatingIpsOptions *ListFloatingIpsOptions)`, func() {
		version := "testString"
		listFloatingIpsPath := "/floating_ips"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listFloatingIpsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips?limit=20"}, "floating_ips": [{"address": "203.0.113.1", "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip", "status": "available", "target": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}}`)
				}))
			})
			It(`Invoke ListFloatingIps successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListFloatingIpsOptions model
				listFloatingIpsOptionsModel := new(vpcclassicv1.ListFloatingIpsOptions)
				listFloatingIpsOptionsModel.Start = core.StringPtr("testString")
				listFloatingIpsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listFloatingIpsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListFloatingIpsWithContext(ctx, listFloatingIpsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListFloatingIps(listFloatingIpsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListFloatingIpsWithContext(ctx, listFloatingIpsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listFloatingIpsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips?limit=20"}, "floating_ips": [{"address": "203.0.113.1", "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip", "status": "available", "target": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}], "limit": 20, "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}}`)
				}))
			})
			It(`Invoke ListFloatingIps successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListFloatingIps(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListFloatingIpsOptions model
				listFloatingIpsOptionsModel := new(vpcclassicv1.ListFloatingIpsOptions)
				listFloatingIpsOptionsModel.Start = core.StringPtr("testString")
				listFloatingIpsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listFloatingIpsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListFloatingIps(listFloatingIpsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListFloatingIps with error: Operation request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListFloatingIpsOptions model
				listFloatingIpsOptionsModel := new(vpcclassicv1.ListFloatingIpsOptions)
				listFloatingIpsOptionsModel.Start = core.StringPtr("testString")
				listFloatingIpsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listFloatingIpsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListFloatingIps(listFloatingIpsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateFloatingIP(createFloatingIPOptions *CreateFloatingIPOptions) - Operation response error`, func() {
		version := "testString"
		createFloatingIPPath := "/floating_ips"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createFloatingIPPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateFloatingIP with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the FloatingIPPrototypeFloatingIPByZone model
				floatingIPPrototypeModel := new(vpcclassicv1.FloatingIPPrototypeFloatingIPByZone)
				floatingIPPrototypeModel.Name = core.StringPtr("my-floating-ip")
				floatingIPPrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateFloatingIPOptions model
				createFloatingIPOptionsModel := new(vpcclassicv1.CreateFloatingIPOptions)
				createFloatingIPOptionsModel.FloatingIPPrototype = floatingIPPrototypeModel
				createFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.CreateFloatingIP(createFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.CreateFloatingIP(createFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`CreateFloatingIP(createFloatingIPOptions *CreateFloatingIPOptions)`, func() {
		version := "testString"
		createFloatingIPPath := "/floating_ips"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createFloatingIPPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"address": "203.0.113.1", "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip", "status": "available", "target": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateFloatingIP successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the FloatingIPPrototypeFloatingIPByZone model
				floatingIPPrototypeModel := new(vpcclassicv1.FloatingIPPrototypeFloatingIPByZone)
				floatingIPPrototypeModel.Name = core.StringPtr("my-floating-ip")
				floatingIPPrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateFloatingIPOptions model
				createFloatingIPOptionsModel := new(vpcclassicv1.CreateFloatingIPOptions)
				createFloatingIPOptionsModel.FloatingIPPrototype = floatingIPPrototypeModel
				createFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.CreateFloatingIPWithContext(ctx, createFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.CreateFloatingIP(createFloatingIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.CreateFloatingIPWithContext(ctx, createFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createFloatingIPPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"address": "203.0.113.1", "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip", "status": "available", "target": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke CreateFloatingIP successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.CreateFloatingIP(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the FloatingIPPrototypeFloatingIPByZone model
				floatingIPPrototypeModel := new(vpcclassicv1.FloatingIPPrototypeFloatingIPByZone)
				floatingIPPrototypeModel.Name = core.StringPtr("my-floating-ip")
				floatingIPPrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateFloatingIPOptions model
				createFloatingIPOptionsModel := new(vpcclassicv1.CreateFloatingIPOptions)
				createFloatingIPOptionsModel.FloatingIPPrototype = floatingIPPrototypeModel
				createFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.CreateFloatingIP(createFloatingIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateFloatingIP with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ZoneIdentityByName model
				zoneIdentityModel := new(vpcclassicv1.ZoneIdentityByName)
				zoneIdentityModel.Name = core.StringPtr("us-south-1")

				// Construct an instance of the FloatingIPPrototypeFloatingIPByZone model
				floatingIPPrototypeModel := new(vpcclassicv1.FloatingIPPrototypeFloatingIPByZone)
				floatingIPPrototypeModel.Name = core.StringPtr("my-floating-ip")
				floatingIPPrototypeModel.Zone = zoneIdentityModel

				// Construct an instance of the CreateFloatingIPOptions model
				createFloatingIPOptionsModel := new(vpcclassicv1.CreateFloatingIPOptions)
				createFloatingIPOptionsModel.FloatingIPPrototype = floatingIPPrototypeModel
				createFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.CreateFloatingIP(createFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateFloatingIPOptions model with no property values
				createFloatingIPOptionsModelNew := new(vpcclassicv1.CreateFloatingIPOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.CreateFloatingIP(createFloatingIPOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`DeleteFloatingIP(deleteFloatingIPOptions *DeleteFloatingIPOptions)`, func() {
		version := "testString"
		deleteFloatingIPPath := "/floating_ips/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteFloatingIPPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteFloatingIP successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcClassicService.DeleteFloatingIP(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteFloatingIPOptions model
				deleteFloatingIPOptionsModel := new(vpcclassicv1.DeleteFloatingIPOptions)
				deleteFloatingIPOptionsModel.ID = core.StringPtr("testString")
				deleteFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcClassicService.DeleteFloatingIP(deleteFloatingIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteFloatingIP with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the DeleteFloatingIPOptions model
				deleteFloatingIPOptionsModel := new(vpcclassicv1.DeleteFloatingIPOptions)
				deleteFloatingIPOptionsModel.ID = core.StringPtr("testString")
				deleteFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcClassicService.DeleteFloatingIP(deleteFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteFloatingIPOptions model with no property values
				deleteFloatingIPOptionsModelNew := new(vpcclassicv1.DeleteFloatingIPOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcClassicService.DeleteFloatingIP(deleteFloatingIPOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetFloatingIP(getFloatingIPOptions *GetFloatingIPOptions) - Operation response error`, func() {
		version := "testString"
		getFloatingIPPath := "/floating_ips/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getFloatingIPPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetFloatingIP with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetFloatingIPOptions model
				getFloatingIPOptionsModel := new(vpcclassicv1.GetFloatingIPOptions)
				getFloatingIPOptionsModel.ID = core.StringPtr("testString")
				getFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetFloatingIP(getFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetFloatingIP(getFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetFloatingIP(getFloatingIPOptions *GetFloatingIPOptions)`, func() {
		version := "testString"
		getFloatingIPPath := "/floating_ips/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getFloatingIPPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"address": "203.0.113.1", "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip", "status": "available", "target": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetFloatingIP successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetFloatingIPOptions model
				getFloatingIPOptionsModel := new(vpcclassicv1.GetFloatingIPOptions)
				getFloatingIPOptionsModel.ID = core.StringPtr("testString")
				getFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetFloatingIPWithContext(ctx, getFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetFloatingIP(getFloatingIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetFloatingIPWithContext(ctx, getFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getFloatingIPPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"address": "203.0.113.1", "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip", "status": "available", "target": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke GetFloatingIP successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetFloatingIP(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetFloatingIPOptions model
				getFloatingIPOptionsModel := new(vpcclassicv1.GetFloatingIPOptions)
				getFloatingIPOptionsModel.ID = core.StringPtr("testString")
				getFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetFloatingIP(getFloatingIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetFloatingIP with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetFloatingIPOptions model
				getFloatingIPOptionsModel := new(vpcclassicv1.GetFloatingIPOptions)
				getFloatingIPOptionsModel.ID = core.StringPtr("testString")
				getFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetFloatingIP(getFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetFloatingIPOptions model with no property values
				getFloatingIPOptionsModelNew := new(vpcclassicv1.GetFloatingIPOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetFloatingIP(getFloatingIPOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateFloatingIP(updateFloatingIPOptions *UpdateFloatingIPOptions) - Operation response error`, func() {
		version := "testString"
		updateFloatingIPPath := "/floating_ips/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateFloatingIPPath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateFloatingIP with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the FloatingIPPatchTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByID model
				floatingIPPatchTargetNetworkInterfaceIdentityModel := new(vpcclassicv1.FloatingIPPatchTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByID)
				floatingIPPatchTargetNetworkInterfaceIdentityModel.ID = core.StringPtr("69e55145-cc7d-4d8e-9e1f-cc3fb60b1793")

				// Construct an instance of the FloatingIPPatch model
				floatingIPPatchModel := new(vpcclassicv1.FloatingIPPatch)
				floatingIPPatchModel.Name = core.StringPtr("my-floating-ip")
				floatingIPPatchModel.Target = floatingIPPatchTargetNetworkInterfaceIdentityModel
				floatingIPPatchModelAsPatch, asPatchErr := floatingIPPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateFloatingIPOptions model
				updateFloatingIPOptionsModel := new(vpcclassicv1.UpdateFloatingIPOptions)
				updateFloatingIPOptionsModel.ID = core.StringPtr("testString")
				updateFloatingIPOptionsModel.FloatingIPPatch = floatingIPPatchModelAsPatch
				updateFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.UpdateFloatingIP(updateFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.UpdateFloatingIP(updateFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`UpdateFloatingIP(updateFloatingIPOptions *UpdateFloatingIPOptions)`, func() {
		version := "testString"
		updateFloatingIPPath := "/floating_ips/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateFloatingIPPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"address": "203.0.113.1", "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip", "status": "available", "target": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateFloatingIP successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the FloatingIPPatchTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByID model
				floatingIPPatchTargetNetworkInterfaceIdentityModel := new(vpcclassicv1.FloatingIPPatchTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByID)
				floatingIPPatchTargetNetworkInterfaceIdentityModel.ID = core.StringPtr("69e55145-cc7d-4d8e-9e1f-cc3fb60b1793")

				// Construct an instance of the FloatingIPPatch model
				floatingIPPatchModel := new(vpcclassicv1.FloatingIPPatch)
				floatingIPPatchModel.Name = core.StringPtr("my-floating-ip")
				floatingIPPatchModel.Target = floatingIPPatchTargetNetworkInterfaceIdentityModel
				floatingIPPatchModelAsPatch, asPatchErr := floatingIPPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateFloatingIPOptions model
				updateFloatingIPOptionsModel := new(vpcclassicv1.UpdateFloatingIPOptions)
				updateFloatingIPOptionsModel.ID = core.StringPtr("testString")
				updateFloatingIPOptionsModel.FloatingIPPatch = floatingIPPatchModelAsPatch
				updateFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.UpdateFloatingIPWithContext(ctx, updateFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.UpdateFloatingIP(updateFloatingIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.UpdateFloatingIPWithContext(ctx, updateFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateFloatingIPPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"address": "203.0.113.1", "created_at": "2019-01-01T12:00:00.000Z", "crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::floating-ip:39300233-9995-4806-89a5-3c1b6eb88689", "href": "https://us-south.iaas.cloud.ibm.com/v1/floating_ips/39300233-9995-4806-89a5-3c1b6eb88689", "id": "39300233-9995-4806-89a5-3c1b6eb88689", "name": "my-floating-ip", "status": "available", "target": {"href": "https://us-south.iaas.cloud.ibm.com/v1/instances/1e09281b-f177-46fb-baf1-bc152b2e391a/network_interfaces/10c02d81-0ecb-4dc5-897d-28392913b81e", "id": "10c02d81-0ecb-4dc5-897d-28392913b81e", "name": "my-network-interface", "primary_ipv4_address": "192.168.3.4", "resource_type": "network_interface"}, "zone": {"href": "https://us-south.iaas.cloud.ibm.com/v1/regions/us-south/zones/us-south-1", "name": "us-south-1"}}`)
				}))
			})
			It(`Invoke UpdateFloatingIP successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.UpdateFloatingIP(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the FloatingIPPatchTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByID model
				floatingIPPatchTargetNetworkInterfaceIdentityModel := new(vpcclassicv1.FloatingIPPatchTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByID)
				floatingIPPatchTargetNetworkInterfaceIdentityModel.ID = core.StringPtr("69e55145-cc7d-4d8e-9e1f-cc3fb60b1793")

				// Construct an instance of the FloatingIPPatch model
				floatingIPPatchModel := new(vpcclassicv1.FloatingIPPatch)
				floatingIPPatchModel.Name = core.StringPtr("my-floating-ip")
				floatingIPPatchModel.Target = floatingIPPatchTargetNetworkInterfaceIdentityModel
				floatingIPPatchModelAsPatch, asPatchErr := floatingIPPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateFloatingIPOptions model
				updateFloatingIPOptionsModel := new(vpcclassicv1.UpdateFloatingIPOptions)
				updateFloatingIPOptionsModel.ID = core.StringPtr("testString")
				updateFloatingIPOptionsModel.FloatingIPPatch = floatingIPPatchModelAsPatch
				updateFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.UpdateFloatingIP(updateFloatingIPOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateFloatingIP with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the FloatingIPPatchTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByID model
				floatingIPPatchTargetNetworkInterfaceIdentityModel := new(vpcclassicv1.FloatingIPPatchTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByID)
				floatingIPPatchTargetNetworkInterfaceIdentityModel.ID = core.StringPtr("69e55145-cc7d-4d8e-9e1f-cc3fb60b1793")

				// Construct an instance of the FloatingIPPatch model
				floatingIPPatchModel := new(vpcclassicv1.FloatingIPPatch)
				floatingIPPatchModel.Name = core.StringPtr("my-floating-ip")
				floatingIPPatchModel.Target = floatingIPPatchTargetNetworkInterfaceIdentityModel
				floatingIPPatchModelAsPatch, asPatchErr := floatingIPPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateFloatingIPOptions model
				updateFloatingIPOptionsModel := new(vpcclassicv1.UpdateFloatingIPOptions)
				updateFloatingIPOptionsModel.ID = core.StringPtr("testString")
				updateFloatingIPOptionsModel.FloatingIPPatch = floatingIPPatchModelAsPatch
				updateFloatingIPOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.UpdateFloatingIP(updateFloatingIPOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateFloatingIPOptions model with no property values
				updateFloatingIPOptionsModelNew := new(vpcclassicv1.UpdateFloatingIPOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.UpdateFloatingIP(updateFloatingIPOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`Service constructor tests`, func() {
		version := "testString"
		It(`Instantiate service client`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				Authenticator: &core.NoAuthAuthenticator{},
				Version:       core.StringPtr(version),
			})
			Expect(vpcClassicService).ToNot(BeNil())
			Expect(serviceErr).To(BeNil())
		})
		It(`Instantiate service client with error: Invalid URL`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Invalid Auth`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
				URL:     "https://vpcclassicv1/api",
				Version: core.StringPtr(version),
				Authenticator: &core.BasicAuthenticator{
					Username: "",
					Password: "",
				},
			})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Validation Error`, func() {
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{})
			Expect(vpcClassicService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
	})
	Describe(`Service constructor tests using external config`, func() {
		version := "testString"
		Context(`Using external config, construct service client instances`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "noauth",
			}

			It(`Create service client using external config successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url from constructor successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					URL:     "https://testService/api",
					Version: core.StringPtr(version),
				})
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url programatically successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
					Version: core.StringPtr(version),
				})
				err := vpcClassicService.SetServiceURL("https://testService/api")
				Expect(err).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vpcClassicService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vpcClassicService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vpcClassicService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vpcClassicService.Service.Options.Authenticator))
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid Auth`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_URL":       "https://vpcclassicv1/api",
				"VPC_CLASSIC_AUTH_TYPE": "someOtherAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid URL`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VPC_CLASSIC_AUTH_TYPE": "NOAuth",
			}

			SetTestEnvironment(testEnvironment)
			vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1UsingExternalConfig(&vpcclassicv1.VpcClassicV1Options{
				URL:     "{BAD_URL_STRING",
				Version: core.StringPtr(version),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vpcClassicService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
	})
	Describe(`Regional endpoint tests`, func() {
		It(`GetServiceURLForRegion(region string)`, func() {
			var url string
			var err error
			url, err = vpcclassicv1.GetServiceURLForRegion("INVALID_REGION")
			Expect(url).To(BeEmpty())
			Expect(err).ToNot(BeNil())
			fmt.Fprintf(GinkgoWriter, "Expected error: %s\n", err.Error())
		})
	})
	Describe(`ListNetworkAcls(listNetworkAclsOptions *ListNetworkAclsOptions) - Operation response error`, func() {
		version := "testString"
		listNetworkAclsPath := "/network_acls"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listNetworkAclsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListNetworkAcls with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListNetworkAclsOptions model
				listNetworkAclsOptionsModel := new(vpcclassicv1.ListNetworkAclsOptions)
				listNetworkAclsOptionsModel.Start = core.StringPtr("testString")
				listNetworkAclsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listNetworkAclsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.ListNetworkAcls(listNetworkAclsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.ListNetworkAcls(listNetworkAclsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListNetworkAcls(listNetworkAclsOptions *ListNetworkAclsOptions)`, func() {
		version := "testString"
		listNetworkAclsPath := "/network_acls"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listNetworkAclsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls?limit=20"}, "limit": 20, "network_acls": [{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "rules": [{"action": "allow", "before": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "port_max": 22, "port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}], "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}}`)
				}))
			})
			It(`Invoke ListNetworkAcls successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the ListNetworkAclsOptions model
				listNetworkAclsOptionsModel := new(vpcclassicv1.ListNetworkAclsOptions)
				listNetworkAclsOptionsModel.Start = core.StringPtr("testString")
				listNetworkAclsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listNetworkAclsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.ListNetworkAclsWithContext(ctx, listNetworkAclsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.ListNetworkAcls(listNetworkAclsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.ListNetworkAclsWithContext(ctx, listNetworkAclsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listNetworkAclsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"first": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls?limit=20"}, "limit": 20, "network_acls": [{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "rules": [{"action": "allow", "before": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "port_max": 22, "port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}], "next": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls?start=9d5a91a3e2cbd233b5a5b33436855ed1&limit=20"}}`)
				}))
			})
			It(`Invoke ListNetworkAcls successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.ListNetworkAcls(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListNetworkAclsOptions model
				listNetworkAclsOptionsModel := new(vpcclassicv1.ListNetworkAclsOptions)
				listNetworkAclsOptionsModel.Start = core.StringPtr("testString")
				listNetworkAclsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listNetworkAclsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.ListNetworkAcls(listNetworkAclsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListNetworkAcls with error: Operation request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the ListNetworkAclsOptions model
				listNetworkAclsOptionsModel := new(vpcclassicv1.ListNetworkAclsOptions)
				listNetworkAclsOptionsModel.Start = core.StringPtr("testString")
				listNetworkAclsOptionsModel.Limit = core.Int64Ptr(int64(1))
				listNetworkAclsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.ListNetworkAcls(listNetworkAclsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateNetworkACL(createNetworkACLOptions *CreateNetworkACLOptions) - Operation response error`, func() {
		version := "testString"
		createNetworkACLPath := "/network_acls"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createNetworkACLPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateNetworkACL with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolAll model
				networkACLRulePrototypeNetworkACLContextModel := new(vpcclassicv1.NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolAll)
				networkACLRulePrototypeNetworkACLContextModel.Action = core.StringPtr("allow")
				networkACLRulePrototypeNetworkACLContextModel.Destination = core.StringPtr("192.168.3.2/32")
				networkACLRulePrototypeNetworkACLContextModel.Direction = core.StringPtr("inbound")
				networkACLRulePrototypeNetworkACLContextModel.Name = core.StringPtr("my-rule-2")
				networkACLRulePrototypeNetworkACLContextModel.Source = core.StringPtr("192.168.3.2/32")
				networkACLRulePrototypeNetworkACLContextModel.Protocol = core.StringPtr("all")

				// Construct an instance of the NetworkACLPrototypeNetworkACLByRules model
				networkACLPrototypeModel := new(vpcclassicv1.NetworkACLPrototypeNetworkACLByRules)
				networkACLPrototypeModel.Name = core.StringPtr("my-network-acl")
				networkACLPrototypeModel.Rules = []vpcclassicv1.NetworkACLRulePrototypeNetworkACLContextIntf{networkACLRulePrototypeNetworkACLContextModel}

				// Construct an instance of the CreateNetworkACLOptions model
				createNetworkACLOptionsModel := new(vpcclassicv1.CreateNetworkACLOptions)
				createNetworkACLOptionsModel.NetworkACLPrototype = networkACLPrototypeModel
				createNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.CreateNetworkACL(createNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.CreateNetworkACL(createNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`CreateNetworkACL(createNetworkACLOptions *CreateNetworkACLOptions)`, func() {
		version := "testString"
		createNetworkACLPath := "/network_acls"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createNetworkACLPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "rules": [{"action": "allow", "before": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "port_max": 22, "port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke CreateNetworkACL successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolAll model
				networkACLRulePrototypeNetworkACLContextModel := new(vpcclassicv1.NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolAll)
				networkACLRulePrototypeNetworkACLContextModel.Action = core.StringPtr("allow")
				networkACLRulePrototypeNetworkACLContextModel.Destination = core.StringPtr("192.168.3.2/32")
				networkACLRulePrototypeNetworkACLContextModel.Direction = core.StringPtr("inbound")
				networkACLRulePrototypeNetworkACLContextModel.Name = core.StringPtr("my-rule-2")
				networkACLRulePrototypeNetworkACLContextModel.Source = core.StringPtr("192.168.3.2/32")
				networkACLRulePrototypeNetworkACLContextModel.Protocol = core.StringPtr("all")

				// Construct an instance of the NetworkACLPrototypeNetworkACLByRules model
				networkACLPrototypeModel := new(vpcclassicv1.NetworkACLPrototypeNetworkACLByRules)
				networkACLPrototypeModel.Name = core.StringPtr("my-network-acl")
				networkACLPrototypeModel.Rules = []vpcclassicv1.NetworkACLRulePrototypeNetworkACLContextIntf{networkACLRulePrototypeNetworkACLContextModel}

				// Construct an instance of the CreateNetworkACLOptions model
				createNetworkACLOptionsModel := new(vpcclassicv1.CreateNetworkACLOptions)
				createNetworkACLOptionsModel.NetworkACLPrototype = networkACLPrototypeModel
				createNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.CreateNetworkACLWithContext(ctx, createNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.CreateNetworkACL(createNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.CreateNetworkACLWithContext(ctx, createNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createNetworkACLPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "rules": [{"action": "allow", "before": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "port_max": 22, "port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke CreateNetworkACL successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.CreateNetworkACL(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolAll model
				networkACLRulePrototypeNetworkACLContextModel := new(vpcclassicv1.NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolAll)
				networkACLRulePrototypeNetworkACLContextModel.Action = core.StringPtr("allow")
				networkACLRulePrototypeNetworkACLContextModel.Destination = core.StringPtr("192.168.3.2/32")
				networkACLRulePrototypeNetworkACLContextModel.Direction = core.StringPtr("inbound")
				networkACLRulePrototypeNetworkACLContextModel.Name = core.StringPtr("my-rule-2")
				networkACLRulePrototypeNetworkACLContextModel.Source = core.StringPtr("192.168.3.2/32")
				networkACLRulePrototypeNetworkACLContextModel.Protocol = core.StringPtr("all")

				// Construct an instance of the NetworkACLPrototypeNetworkACLByRules model
				networkACLPrototypeModel := new(vpcclassicv1.NetworkACLPrototypeNetworkACLByRules)
				networkACLPrototypeModel.Name = core.StringPtr("my-network-acl")
				networkACLPrototypeModel.Rules = []vpcclassicv1.NetworkACLRulePrototypeNetworkACLContextIntf{networkACLRulePrototypeNetworkACLContextModel}

				// Construct an instance of the CreateNetworkACLOptions model
				createNetworkACLOptionsModel := new(vpcclassicv1.CreateNetworkACLOptions)
				createNetworkACLOptionsModel.NetworkACLPrototype = networkACLPrototypeModel
				createNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.CreateNetworkACL(createNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateNetworkACL with error: Operation request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolAll model
				networkACLRulePrototypeNetworkACLContextModel := new(vpcclassicv1.NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolAll)
				networkACLRulePrototypeNetworkACLContextModel.Action = core.StringPtr("allow")
				networkACLRulePrototypeNetworkACLContextModel.Destination = core.StringPtr("192.168.3.2/32")
				networkACLRulePrototypeNetworkACLContextModel.Direction = core.StringPtr("inbound")
				networkACLRulePrototypeNetworkACLContextModel.Name = core.StringPtr("my-rule-2")
				networkACLRulePrototypeNetworkACLContextModel.Source = core.StringPtr("192.168.3.2/32")
				networkACLRulePrototypeNetworkACLContextModel.Protocol = core.StringPtr("all")

				// Construct an instance of the NetworkACLPrototypeNetworkACLByRules model
				networkACLPrototypeModel := new(vpcclassicv1.NetworkACLPrototypeNetworkACLByRules)
				networkACLPrototypeModel.Name = core.StringPtr("my-network-acl")
				networkACLPrototypeModel.Rules = []vpcclassicv1.NetworkACLRulePrototypeNetworkACLContextIntf{networkACLRulePrototypeNetworkACLContextModel}

				// Construct an instance of the CreateNetworkACLOptions model
				createNetworkACLOptionsModel := new(vpcclassicv1.CreateNetworkACLOptions)
				createNetworkACLOptionsModel.NetworkACLPrototype = networkACLPrototypeModel
				createNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.CreateNetworkACL(createNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`DeleteNetworkACL(deleteNetworkACLOptions *DeleteNetworkACLOptions)`, func() {
		version := "testString"
		deleteNetworkACLPath := "/network_acls/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteNetworkACLPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.WriteHeader(204)
				}))
			})
			It(`Invoke DeleteNetworkACL successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vpcClassicService.DeleteNetworkACL(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteNetworkACLOptions model
				deleteNetworkACLOptionsModel := new(vpcclassicv1.DeleteNetworkACLOptions)
				deleteNetworkACLOptionsModel.ID = core.StringPtr("testString")
				deleteNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vpcClassicService.DeleteNetworkACL(deleteNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteNetworkACL with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the DeleteNetworkACLOptions model
				deleteNetworkACLOptionsModel := new(vpcclassicv1.DeleteNetworkACLOptions)
				deleteNetworkACLOptionsModel.ID = core.StringPtr("testString")
				deleteNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vpcClassicService.DeleteNetworkACL(deleteNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteNetworkACLOptions model with no property values
				deleteNetworkACLOptionsModelNew := new(vpcclassicv1.DeleteNetworkACLOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vpcClassicService.DeleteNetworkACL(deleteNetworkACLOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetNetworkACL(getNetworkACLOptions *GetNetworkACLOptions) - Operation response error`, func() {
		version := "testString"
		getNetworkACLPath := "/network_acls/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getNetworkACLPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetNetworkACL with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetNetworkACLOptions model
				getNetworkACLOptionsModel := new(vpcclassicv1.GetNetworkACLOptions)
				getNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.GetNetworkACL(getNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.GetNetworkACL(getNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetNetworkACL(getNetworkACLOptions *GetNetworkACLOptions)`, func() {
		version := "testString"
		getNetworkACLPath := "/network_acls/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getNetworkACLPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "rules": [{"action": "allow", "before": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "port_max": 22, "port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke GetNetworkACL successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the GetNetworkACLOptions model
				getNetworkACLOptionsModel := new(vpcclassicv1.GetNetworkACLOptions)
				getNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.GetNetworkACLWithContext(ctx, getNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.GetNetworkACL(getNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.GetNetworkACLWithContext(ctx, getNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getNetworkACLPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "rules": [{"action": "allow", "before": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "port_max": 22, "port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke GetNetworkACL successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.GetNetworkACL(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetNetworkACLOptions model
				getNetworkACLOptionsModel := new(vpcclassicv1.GetNetworkACLOptions)
				getNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.GetNetworkACL(getNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetNetworkACL with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the GetNetworkACLOptions model
				getNetworkACLOptionsModel := new(vpcclassicv1.GetNetworkACLOptions)
				getNetworkACLOptionsModel.ID = core.StringPtr("testString")
				getNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.GetNetworkACL(getNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetNetworkACLOptions model with no property values
				getNetworkACLOptionsModelNew := new(vpcclassicv1.GetNetworkACLOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.GetNetworkACL(getNetworkACLOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`UpdateNetworkACL(updateNetworkACLOptions *UpdateNetworkACLOptions) - Operation response error`, func() {
		version := "testString"
		updateNetworkACLPath := "/network_acls/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateNetworkACLPath))
					Expect(req.Method).To(Equal("PATCH"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke UpdateNetworkACL with error: Operation response processing error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the NetworkACLPatch model
				networkACLPatchModel := new(vpcclassicv1.NetworkACLPatch)
				networkACLPatchModel.Name = core.StringPtr("my-network-acl")
				networkACLPatchModelAsPatch, asPatchErr := networkACLPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateNetworkACLOptions model
				updateNetworkACLOptionsModel := new(vpcclassicv1.UpdateNetworkACLOptions)
				updateNetworkACLOptionsModel.ID = core.StringPtr("testString")
				updateNetworkACLOptionsModel.NetworkACLPatch = networkACLPatchModelAsPatch
				updateNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vpcClassicService.UpdateNetworkACL(updateNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vpcClassicService.EnableRetries(0, 0)
				result, response, operationErr = vpcClassicService.UpdateNetworkACL(updateNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`UpdateNetworkACL(updateNetworkACLOptions *UpdateNetworkACLOptions)`, func() {
		version := "testString"
		updateNetworkACLPath := "/network_acls/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateNetworkACLPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "rules": [{"action": "allow", "before": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "port_max": 22, "port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke UpdateNetworkACL successfully with retries`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())
				vpcClassicService.EnableRetries(0, 0)

				// Construct an instance of the NetworkACLPatch model
				networkACLPatchModel := new(vpcclassicv1.NetworkACLPatch)
				networkACLPatchModel.Name = core.StringPtr("my-network-acl")
				networkACLPatchModelAsPatch, asPatchErr := networkACLPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateNetworkACLOptions model
				updateNetworkACLOptionsModel := new(vpcclassicv1.UpdateNetworkACLOptions)
				updateNetworkACLOptionsModel.ID = core.StringPtr("testString")
				updateNetworkACLOptionsModel.NetworkACLPatch = networkACLPatchModelAsPatch
				updateNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vpcClassicService.UpdateNetworkACLWithContext(ctx, updateNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vpcClassicService.DisableRetries()
				result, response, operationErr := vpcClassicService.UpdateNetworkACL(updateNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vpcClassicService.UpdateNetworkACLWithContext(ctx, updateNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(updateNetworkACLPath))
					Expect(req.Method).To(Equal("PATCH"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"created_at": "2019-01-01T12:00:00.000Z", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf", "id": "a4e28308-8ee7-46ab-8108-9f881f22bdbf", "name": "my-network-acl", "rules": [{"action": "allow", "before": {"href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "name": "my-rule-1"}, "created_at": "2019-01-01T12:00:00.000Z", "destination": "192.168.3.0/24", "direction": "inbound", "href": "https://us-south.iaas.cloud.ibm.com/v1/network_acls/a4e28308-8ee7-46ab-8108-9f881f22bdbf/rules/8daca77a-4980-4d33-8f3e-7038797be8f9", "id": "8daca77a-4980-4d33-8f3e-7038797be8f9", "ip_version": "ipv4", "name": "my-rule-2", "source": "192.168.3.0/24", "port_max": 22, "port_min": 22, "protocol": "udp", "source_port_max": 65535, "source_port_min": 49152}], "subnets": [{"crn": "crn:v1:bluemix:public:is:us-south-1:a/123456::subnet:7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "href": "https://us-south.iaas.cloud.ibm.com/v1/subnets/7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "id": "7ec86020-1c6e-4889-b3f0-a15f2e50f87e", "name": "my-subnet"}]}`)
				}))
			})
			It(`Invoke UpdateNetworkACL successfully`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vpcClassicService.UpdateNetworkACL(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the NetworkACLPatch model
				networkACLPatchModel := new(vpcclassicv1.NetworkACLPatch)
				networkACLPatchModel.Name = core.StringPtr("my-network-acl")
				networkACLPatchModelAsPatch, asPatchErr := networkACLPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateNetworkACLOptions model
				updateNetworkACLOptionsModel := new(vpcclassicv1.UpdateNetworkACLOptions)
				updateNetworkACLOptionsModel.ID = core.StringPtr("testString")
				updateNetworkACLOptionsModel.NetworkACLPatch = networkACLPatchModelAsPatch
				updateNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vpcClassicService.UpdateNetworkACL(updateNetworkACLOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke UpdateNetworkACL with error: Operation validation and request error`, func() {
				vpcClassicService, serviceErr := vpcclassicv1.NewVpcClassicV1(&vpcclassicv1.VpcClassicV1Options{
					URL:           testServer.URL,
					Authenticator: &core.NoAuthAuthenticator{},
					Version:       core.StringPtr(version),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vpcClassicService).ToNot(BeNil())

				// Construct an instance of the NetworkACLPatch model
				networkACLPatchModel := new(vpcclassicv1.NetworkACLPatch)
				networkACLPatchModel.Name = core.StringPtr("my-network-acl")
				networkACLPatchModelAsPatch, asPatchErr := networkACLPatchModel.AsPatch()
				Expect(asPatchErr).To(BeNil())

				// Construct an instance of the UpdateNetworkACLOptions model
				updateNetworkACLOptionsModel := new(vpcclassicv1.UpdateNetworkACLOptions)
				updateNetworkACLOptionsModel.ID = core.StringPtr("testString")
				updateNetworkACLOptionsModel.NetworkACLPatch = networkACLPatchModelAsPatch
				updateNetworkACLOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vpcClassicService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vpcClassicService.UpdateNetworkACL(updateNetworkACLOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the UpdateNetworkACLOptions model with no property values
				updateNetworkACLOptionsModelNew := new(vpcclassicv1.UpdateNetworkACLOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vpcClassicService.UpdateNetworkACL(updateNetworkACLOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListNetworkACLRules(listNetworkACLRulesOptions *ListNetworkACLRulesOptions) - Operation response error`, func() {
		version := "testString"
		listNetworkACLRulesPath := "/network_acls/testString/rules"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listNetworkACLRulesPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.URL.Query()["version"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["generation"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["start"]).To(Equal([]string{"testString"}))

					Expect(req.URL.Query()["limit"]).To(Equal([]string{fmt.Sprint(int64(1))}))

					Expect(req.URL.Query()["direction"]).To(Equal([]string{"inbound"}))

					res.Header().Set("Content-type", "application/json")
					res.