/**
 * (C) Copyright IBM Corp. 2020.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// Package directlinkproviderv2 : Operations and models for the DirectLinkProviderV2 service
package directlinkproviderv2

import (
	"encoding/json"
	"fmt"
	"reflect"

	"github.com/IBM/go-sdk-core/v4/core"
	common "github.com/IBM/networking-go-sdk/common"
	"github.com/go-openapi/strfmt"
)

// DirectLinkProviderV2 : No description provided (generated by Openapi Generator
// https://github.com/openapitools/openapi-generator)
//
// Version: __VERSION__
type DirectLinkProviderV2 struct {
	Service *core.BaseService

	// Requests the version of the API as a date in the format `YYYY-MM-DD`. Any date from 2020-04-28 up to the current
	// date may be provided. Specify the current date to request the latest version.
	Version *string
}

// DefaultServiceURL is the default URL to make service requests to.
const DefaultServiceURL = "https://directlink.cloud.ibm.com/provider/v2"

// DefaultServiceName is the default key used to find external configuration information.
const DefaultServiceName = "direct_link_provider"

// DirectLinkProviderV2Options : Service options
type DirectLinkProviderV2Options struct {
	ServiceName   string
	URL           string
	Authenticator core.Authenticator

	// Requests the version of the API as a date in the format `YYYY-MM-DD`. Any date from 2020-04-28 up to the current
	// date may be provided. Specify the current date to request the latest version.
	Version *string `validate:"required"`
}

// NewDirectLinkProviderV2UsingExternalConfig : constructs an instance of DirectLinkProviderV2 with passed in options and external configuration.
func NewDirectLinkProviderV2UsingExternalConfig(options *DirectLinkProviderV2Options) (directLinkProvider *DirectLinkProviderV2, err error) {
	if options.ServiceName == "" {
		options.ServiceName = DefaultServiceName
	}

	if options.Authenticator == nil {
		options.Authenticator, err = core.GetAuthenticatorFromEnvironment(options.ServiceName)
		if err != nil {
			return
		}
	}

	directLinkProvider, err = NewDirectLinkProviderV2(options)
	if err != nil {
		return
	}

	err = directLinkProvider.Service.ConfigureService(options.ServiceName)
	if err != nil {
		return
	}

	if options.URL != "" {
		err = directLinkProvider.Service.SetServiceURL(options.URL)
	}
	return
}

// NewDirectLinkProviderV2 : constructs an instance of DirectLinkProviderV2 with passed in options.
func NewDirectLinkProviderV2(options *DirectLinkProviderV2Options) (service *DirectLinkProviderV2, err error) {
	serviceOptions := &core.ServiceOptions{
		URL:           DefaultServiceURL,
		Authenticator: options.Authenticator,
	}

	err = core.ValidateStruct(options, "options")
	if err != nil {
		return
	}

	baseService, err := core.NewBaseService(serviceOptions)
	if err != nil {
		return
	}

	if options.URL != "" {
		err = baseService.SetServiceURL(options.URL)
		if err != nil {
			return
		}
	}

	service = &DirectLinkProviderV2{
		Service: baseService,
		Version: options.Version,
	}

	return
}

// SetServiceURL sets the service URL
func (directLinkProvider *DirectLinkProviderV2) SetServiceURL(url string) error {
	return directLinkProvider.Service.SetServiceURL(url)
}

// ListProviderGateways : List gateways
// List all Direct Link Connect gateways created by this provider.
func (directLinkProvider *DirectLinkProviderV2) ListProviderGateways(listProviderGatewaysOptions *ListProviderGatewaysOptions) (result *ProviderGatewayCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listProviderGatewaysOptions, "listProviderGatewaysOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways"}
	pathParameters := []string{}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLinkProvider.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range listProviderGatewaysOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link_provider", "V2", "ListProviderGateways")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLinkProvider.Version))
	if listProviderGatewaysOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listProviderGatewaysOptions.Start))
	}
	if listProviderGatewaysOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listProviderGatewaysOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLinkProvider.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalProviderGatewayCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateProviderGateway : Create gateway
// Create a Direct Link Connect gateway based on the supplied template in the specified customer account.
//
// The gateway will be 'provider_api_managed=true'.
func (directLinkProvider *DirectLinkProviderV2) CreateProviderGateway(createProviderGatewayOptions *CreateProviderGatewayOptions) (result *ProviderGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createProviderGatewayOptions, "createProviderGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createProviderGatewayOptions, "createProviderGatewayOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways"}
	pathParameters := []string{}

	builder := core.NewRequestBuilder(core.POST)
	_, err = builder.ConstructHTTPURL(directLinkProvider.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range createProviderGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link_provider", "V2", "CreateProviderGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLinkProvider.Version))
	if createProviderGatewayOptions.CheckOnly != nil {
		builder.AddQuery("check_only", fmt.Sprint(*createProviderGatewayOptions.CheckOnly))
	}

	body := make(map[string]interface{})
	if createProviderGatewayOptions.BgpAsn != nil {
		body["bgp_asn"] = createProviderGatewayOptions.BgpAsn
	}
	if createProviderGatewayOptions.CustomerAccountID != nil {
		body["customer_account_id"] = createProviderGatewayOptions.CustomerAccountID
	}
	if createProviderGatewayOptions.Name != nil {
		body["name"] = createProviderGatewayOptions.Name
	}
	if createProviderGatewayOptions.Port != nil {
		body["port"] = createProviderGatewayOptions.Port
	}
	if createProviderGatewayOptions.SpeedMbps != nil {
		body["speed_mbps"] = createProviderGatewayOptions.SpeedMbps
	}
	if createProviderGatewayOptions.BgpCerCidr != nil {
		body["bgp_cer_cidr"] = createProviderGatewayOptions.BgpCerCidr
	}
	if createProviderGatewayOptions.BgpIbmCidr != nil {
		body["bgp_ibm_cidr"] = createProviderGatewayOptions.BgpIbmCidr
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLinkProvider.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalProviderGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteProviderGateway : Delete gateway
// Delete a Direct Link Connect provider managed gateway.
func (directLinkProvider *DirectLinkProviderV2) DeleteProviderGateway(deleteProviderGatewayOptions *DeleteProviderGatewayOptions) (result *ProviderGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteProviderGatewayOptions, "deleteProviderGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteProviderGatewayOptions, "deleteProviderGatewayOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways"}
	pathParameters := []string{*deleteProviderGatewayOptions.ID}

	builder := core.NewRequestBuilder(core.DELETE)
	_, err = builder.ConstructHTTPURL(directLinkProvider.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteProviderGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link_provider", "V2", "DeleteProviderGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLinkProvider.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLinkProvider.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalProviderGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetProviderGateway : Get gateway
// Get a Direct Link Connect gateway.
//      Gateways with either `provider_api_managed=true` or `provider_api_managed=false` can be retrieved.
func (directLinkProvider *DirectLinkProviderV2) GetProviderGateway(getProviderGatewayOptions *GetProviderGatewayOptions) (result *ProviderGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getProviderGatewayOptions, "getProviderGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getProviderGatewayOptions, "getProviderGatewayOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways"}
	pathParameters := []string{*getProviderGatewayOptions.ID}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLinkProvider.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range getProviderGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link_provider", "V2", "GetProviderGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLinkProvider.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLinkProvider.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalProviderGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateProviderGateway : Update gateway
// Update a Direct Link Connect provider managed gateway.
//
// Name changes are applied immediately, other changes result in a gateway change_request and require approval from the
// client.
func (directLinkProvider *DirectLinkProviderV2) UpdateProviderGateway(updateProviderGatewayOptions *UpdateProviderGatewayOptions) (result *ProviderGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateProviderGatewayOptions, "updateProviderGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateProviderGatewayOptions, "updateProviderGatewayOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways"}
	pathParameters := []string{*updateProviderGatewayOptions.ID}

	builder := core.NewRequestBuilder(core.PATCH)
	_, err = builder.ConstructHTTPURL(directLinkProvider.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateProviderGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link_provider", "V2", "UpdateProviderGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLinkProvider.Version))

	body := make(map[string]interface{})
	if updateProviderGatewayOptions.Name != nil {
		body["name"] = updateProviderGatewayOptions.Name
	}
	if updateProviderGatewayOptions.SpeedMbps != nil {
		body["speed_mbps"] = updateProviderGatewayOptions.SpeedMbps
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLinkProvider.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalProviderGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListProviderPorts : List ports
// List all provider ports (associated with the caller).
func (directLinkProvider *DirectLinkProviderV2) ListProviderPorts(listProviderPortsOptions *ListProviderPortsOptions) (result *ProviderPortCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listProviderPortsOptions, "listProviderPortsOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"ports"}
	pathParameters := []string{}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLinkProvider.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range listProviderPortsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link_provider", "V2", "ListProviderPorts")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLinkProvider.Version))
	if listProviderPortsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listProviderPortsOptions.Start))
	}
	if listProviderPortsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listProviderPortsOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLinkProvider.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalProviderPortCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetProviderPort : Get port
// Get provider port information.
func (directLinkProvider *DirectLinkProviderV2) GetProviderPort(getProviderPortOptions *GetProviderPortOptions) (result *ProviderPort, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getProviderPortOptions, "getProviderPortOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getProviderPortOptions, "getProviderPortOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"ports"}
	pathParameters := []string{*getProviderPortOptions.ID}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLinkProvider.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range getProviderPortOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link_provider", "V2", "GetProviderPort")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLinkProvider.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLinkProvider.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalProviderPort)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateProviderGatewayOptions : The CreateProviderGateway options.
type CreateProviderGatewayOptions struct {
	// BGP ASN.
	//
	// For a 2-byte range, enter a value between 1-64495 or 64999. For a 2-byte or 4-byte range, enter a value between
	// 131072-4199999999. For a 4-byte range, enter a value between 4201000000-4201064511.
	BgpAsn *int64 `json:"bgp_asn" validate:"required"`

	// Customer IBM Cloud account ID for the new gateway.  A gateway object containing the pending create request will
	// become available in the specified account.
	CustomerAccountID *string `json:"customer_account_id" validate:"required"`

	// The unique user-defined name for this gateway.
	Name *string `json:"name" validate:"required"`

	// Select Port Label for the gateway.
	Port *ProviderGatewayPortIdentity `json:"port" validate:"required"`

	// Gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps" validate:"required"`

	// BGP customer edge router CIDR.
	//
	// For auto IP assignment, omit bgp_cer_cidr and bgp_ibm_cidr.  IBM will automatically select values for bgp_cer_cidr
	// and bgp_ibm_cidr.
	//
	// For manual IP assignment set a valid bgp_cer_cidr and bgp_ibm_cidr CIDR, the value must reside in one of
	// "10.254.0.0/16", "172.16.0.0/12", "192.168.0.0/16", "169.254.0.0/16" or an owned public CIDR.  bgp_cer_cidr and
	// bgp_ibm_cidr must have matching network and subnet mask values.
	BgpCerCidr *string `json:"bgp_cer_cidr,omitempty"`

	// BGP IBM CIDR.
	//
	// For auto IP assignment, omit bgp_cer_cidr and bgp_ibm_cidr.  IBM will automatically select values for bgp_cer_cidr
	// and bgp_ibm_cidr.
	//
	// For manual IP assignment set a valid bgp_cer_cidr and bgp_ibm_cidr CIDR, the value must reside in one of
	// "10.254.0.0/16", "172.16.0.0/12", "192.168.0.0/16", "169.254.0.0/16" or an owned public CIDR.  bgp_cer_cidr and
	// bgp_ibm_cidr must have matching network and subnet mask values.
	BgpIbmCidr *string `json:"bgp_ibm_cidr,omitempty"`

	// When true, perform request validation only and do not create a gateway.
	CheckOnly *string `json:"check_only,omitempty"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateProviderGatewayOptions : Instantiate CreateProviderGatewayOptions
func (*DirectLinkProviderV2) NewCreateProviderGatewayOptions(bgpAsn int64, customerAccountID string, name string, port *ProviderGatewayPortIdentity, speedMbps int64) *CreateProviderGatewayOptions {
	return &CreateProviderGatewayOptions{
		BgpAsn:            core.Int64Ptr(bgpAsn),
		CustomerAccountID: core.StringPtr(customerAccountID),
		Name:              core.StringPtr(name),
		Port:              port,
		SpeedMbps:         core.Int64Ptr(speedMbps),
	}
}

// SetBgpAsn : Allow user to set BgpAsn
func (options *CreateProviderGatewayOptions) SetBgpAsn(bgpAsn int64) *CreateProviderGatewayOptions {
	options.BgpAsn = core.Int64Ptr(bgpAsn)
	return options
}

// SetCustomerAccountID : Allow user to set CustomerAccountID
func (options *CreateProviderGatewayOptions) SetCustomerAccountID(customerAccountID string) *CreateProviderGatewayOptions {
	options.CustomerAccountID = core.StringPtr(customerAccountID)
	return options
}

// SetName : Allow user to set Name
func (options *CreateProviderGatewayOptions) SetName(name string) *CreateProviderGatewayOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetPort : Allow user to set Port
func (options *CreateProviderGatewayOptions) SetPort(port *ProviderGatewayPortIdentity) *CreateProviderGatewayOptions {
	options.Port = port
	return options
}

// SetSpeedMbps : Allow user to set SpeedMbps
func (options *CreateProviderGatewayOptions) SetSpeedMbps(speedMbps int64) *CreateProviderGatewayOptions {
	options.SpeedMbps = core.Int64Ptr(speedMbps)
	return options
}

// SetBgpCerCidr : Allow user to set BgpCerCidr
func (options *CreateProviderGatewayOptions) SetBgpCerCidr(bgpCerCidr string) *CreateProviderGatewayOptions {
	options.BgpCerCidr = core.StringPtr(bgpCerCidr)
	return options
}

// SetBgpIbmCidr : Allow user to set BgpIbmCidr
func (options *CreateProviderGatewayOptions) SetBgpIbmCidr(bgpIbmCidr string) *CreateProviderGatewayOptions {
	options.BgpIbmCidr = core.StringPtr(bgpIbmCidr)
	return options
}

// SetCheckOnly : Allow user to set CheckOnly
func (options *CreateProviderGatewayOptions) SetCheckOnly(checkOnly string) *CreateProviderGatewayOptions {
	options.CheckOnly = core.StringPtr(checkOnly)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateProviderGatewayOptions) SetHeaders(param map[string]string) *CreateProviderGatewayOptions {
	options.Headers = param
	return options
}

// DeleteProviderGatewayOptions : The DeleteProviderGateway options.
type DeleteProviderGatewayOptions struct {
	// Direct Link Connect gateway identifier.
	ID *string `json:"id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteProviderGatewayOptions : Instantiate DeleteProviderGatewayOptions
func (*DirectLinkProviderV2) NewDeleteProviderGatewayOptions(id string) *DeleteProviderGatewayOptions {
	return &DeleteProviderGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteProviderGatewayOptions) SetID(id string) *DeleteProviderGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteProviderGatewayOptions) SetHeaders(param map[string]string) *DeleteProviderGatewayOptions {
	options.Headers = param
	return options
}

// GetProviderGatewayOptions : The GetProviderGateway options.
type GetProviderGatewayOptions struct {
	// Direct Link Connect gateway identifier.
	ID *string `json:"id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetProviderGatewayOptions : Instantiate GetProviderGatewayOptions
func (*DirectLinkProviderV2) NewGetProviderGatewayOptions(id string) *GetProviderGatewayOptions {
	return &GetProviderGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetProviderGatewayOptions) SetID(id string) *GetProviderGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetProviderGatewayOptions) SetHeaders(param map[string]string) *GetProviderGatewayOptions {
	options.Headers = param
	return options
}

// GetProviderPortOptions : The GetProviderPort options.
type GetProviderPortOptions struct {
	// port identifier.
	ID *string `json:"id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetProviderPortOptions : Instantiate GetProviderPortOptions
func (*DirectLinkProviderV2) NewGetProviderPortOptions(id string) *GetProviderPortOptions {
	return &GetProviderPortOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetProviderPortOptions) SetID(id string) *GetProviderPortOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetProviderPortOptions) SetHeaders(param map[string]string) *GetProviderPortOptions {
	options.Headers = param
	return options
}

// ListProviderGatewaysOptions : The ListProviderGateways options.
type ListProviderGatewaysOptions struct {
	// A server-supplied token determining which resource to start the page on.
	Start *string `json:"start,omitempty"`

	// The number of resources to return on a page.
	Limit *int64 `json:"limit,omitempty"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListProviderGatewaysOptions : Instantiate ListProviderGatewaysOptions
func (*DirectLinkProviderV2) NewListProviderGatewaysOptions() *ListProviderGatewaysOptions {
	return &ListProviderGatewaysOptions{}
}

// SetStart : Allow user to set Start
func (options *ListProviderGatewaysOptions) SetStart(start string) *ListProviderGatewaysOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListProviderGatewaysOptions) SetLimit(limit int64) *ListProviderGatewaysOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListProviderGatewaysOptions) SetHeaders(param map[string]string) *ListProviderGatewaysOptions {
	options.Headers = param
	return options
}

// ListProviderPortsOptions : The ListProviderPorts options.
type ListProviderPortsOptions struct {
	// A server-supplied token determining which resource to start the page on.
	Start *string `json:"start,omitempty"`

	// The number of resources to return on a page.
	Limit *int64 `json:"limit,omitempty"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListProviderPortsOptions : Instantiate ListProviderPortsOptions
func (*DirectLinkProviderV2) NewListProviderPortsOptions() *ListProviderPortsOptions {
	return &ListProviderPortsOptions{}
}

// SetStart : Allow user to set Start
func (options *ListProviderPortsOptions) SetStart(start string) *ListProviderPortsOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListProviderPortsOptions) SetLimit(limit int64) *ListProviderPortsOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListProviderPortsOptions) SetHeaders(param map[string]string) *ListProviderPortsOptions {
	options.Headers = param
	return options
}

// ProviderGateway : gateway.
type ProviderGateway struct {
	// BGP ASN.
	//
	// For a 2-byte ASN values between 1-64495 or 64999. For a 2-byte or 4-byte range ASN values between 131072-4199999999.
	// For a 4-byte ASN values 4201000000-4201064511.
	BgpAsn *int64 `json:"bgp_asn" validate:"required"`

	// BGP customer edge router CIDR.
	BgpCerCidr *string `json:"bgp_cer_cidr,omitempty"`

	// IBM BGP ASN.
	BgpIbmAsn *int64 `json:"bgp_ibm_asn" validate:"required"`

	// BGP IBM CIDR.
	BgpIbmCidr *string `json:"bgp_ibm_cidr,omitempty"`

	// Gateway BGP status.
	//
	// The list of enumerated values for this property may expand in the future. Code and processes using this field  must
	// tolerate unexpected values.
	BgpStatus *string `json:"bgp_status,omitempty"`

	ChangeRequest ProviderGatewayChangeRequestIntf `json:"change_request,omitempty"`

	// The date and time resource was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN (Cloud Resource Name) of this gateway.
	Crn *string `json:"crn,omitempty"`

	// Customer IBM Cloud account ID.
	CustomerAccountID *string `json:"customer_account_id" validate:"required"`

	// The unique identifier of this gateway.
	ID *string `json:"id" validate:"required"`

	// The unique user-defined name for this gateway.
	Name *string `json:"name" validate:"required"`

	// Gateway operational status.
	//
	// The list of enumerated values for this property may expand in the future. Code and processes using this field  must
	// tolerate unexpected values.
	OperationalStatus *string `json:"operational_status" validate:"required"`

	// Port identifier for the gateway.
	Port *ProviderGatewayPortReference `json:"port" validate:"required"`

	// Set to `true` for gateways created through the Direct Link Provider APIs.
	//
	// Most Direct Link Provider APIs cannot interact with `provider_api_managed=false` gateways.
	ProviderApiManaged *bool `json:"provider_api_managed" validate:"required"`

	// Gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps" validate:"required"`

	// Gateway type.
	Type *string `json:"type" validate:"required"`

	// VLAN allocated for this gateway.
	Vlan *int64 `json:"vlan,omitempty"`
}

// Constants associated with the ProviderGateway.BgpStatus property.
// Gateway BGP status.
//
// The list of enumerated values for this property may expand in the future. Code and processes using this field  must
// tolerate unexpected values.
const (
	ProviderGateway_BgpStatus_Active      = "active"
	ProviderGateway_BgpStatus_Connect     = "connect"
	ProviderGateway_BgpStatus_Established = "established"
	ProviderGateway_BgpStatus_Idle        = "idle"
)

// Constants associated with the ProviderGateway.OperationalStatus property.
// Gateway operational status.
//
// The list of enumerated values for this property may expand in the future. Code and processes using this field  must
// tolerate unexpected values.
const (
	ProviderGateway_OperationalStatus_CreatePending  = "create_pending"
	ProviderGateway_OperationalStatus_CreateRejected = "create_rejected"
	ProviderGateway_OperationalStatus_DeletePending  = "delete_pending"
	ProviderGateway_OperationalStatus_Provisioned    = "provisioned"
)

// UnmarshalProviderGateway unmarshals an instance of ProviderGateway from the specified map of raw messages.
func UnmarshalProviderGateway(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderGateway)
	err = core.UnmarshalPrimitive(m, "bgp_asn", &obj.BgpAsn)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_cer_cidr", &obj.BgpCerCidr)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_ibm_asn", &obj.BgpIbmAsn)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_ibm_cidr", &obj.BgpIbmCidr)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_status", &obj.BgpStatus)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "change_request", &obj.ChangeRequest, UnmarshalProviderGatewayChangeRequest)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.Crn)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "customer_account_id", &obj.CustomerAccountID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "operational_status", &obj.OperationalStatus)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "port", &obj.Port, UnmarshalProviderGatewayPortReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "provider_api_managed", &obj.ProviderApiManaged)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "speed_mbps", &obj.SpeedMbps)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "vlan", &obj.Vlan)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ProviderGatewayChangeRequest : ProviderGatewayChangeRequest struct
// Models which "extend" this model:
// - ProviderGatewayChangeRequestProviderGatewayCreate
// - ProviderGatewayChangeRequestProviderGatewayDelete
// - ProviderGatewayChangeRequestProviderGatewayUpdateAttributes
type ProviderGatewayChangeRequest struct {
	// type of gateway change request.
	Type *string `json:"type,omitempty"`

	// array of pending updates.
	Updates []ProviderGatewayUpdateAttributesUpdatesItemIntf `json:"updates,omitempty"`
}

// Constants associated with the ProviderGatewayChangeRequest.Type property.
// type of gateway change request.
const (
	ProviderGatewayChangeRequest_Type_CreateGateway = "create_gateway"
)

func (*ProviderGatewayChangeRequest) isaProviderGatewayChangeRequest() bool {
	return true
}

type ProviderGatewayChangeRequestIntf interface {
	isaProviderGatewayChangeRequest() bool
}

// UnmarshalProviderGatewayChangeRequest unmarshals an instance of ProviderGatewayChangeRequest from the specified map of raw messages.
func UnmarshalProviderGatewayChangeRequest(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderGatewayChangeRequest)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "updates", &obj.Updates, UnmarshalProviderGatewayUpdateAttributesUpdatesItem)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ProviderGatewayCollection : A paginated collection of resources.
type ProviderGatewayCollection struct {
	// A reference to the first page of resources.
	First *ProviderGatewayCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A reference to the next page of resources; this reference is included for all pages except the last page.
	Next *ProviderGatewayCollectionNext `json:"next,omitempty"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`

	// Collection of Direct Link gateways.
	Gateways []ProviderGateway `json:"gateways" validate:"required"`
}

// UnmarshalProviderGatewayCollection unmarshals an instance of ProviderGatewayCollection from the specified map of raw messages.
func UnmarshalProviderGatewayCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderGatewayCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalProviderGatewayCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalProviderGatewayCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "gateways", &obj.Gateways, UnmarshalProviderGateway)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ProviderGatewayCollectionFirst : A reference to the first page of resources.
type ProviderGatewayCollectionFirst struct {
	// The URL for the first page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalProviderGatewayCollectionFirst unmarshals an instance of ProviderGatewayCollectionFirst from the specified map of raw messages.
func UnmarshalProviderGatewayCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderGatewayCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ProviderGatewayCollectionNext : A reference to the next page of resources; this reference is included for all pages except the last page.
type ProviderGatewayCollectionNext struct {
	// The URL for the next page of resources.
	Href *string `json:"href" validate:"required"`

	// start token for the next page of resources.
	Start *string `json:"start" validate:"required"`
}

// UnmarshalProviderGatewayCollectionNext unmarshals an instance of ProviderGatewayCollectionNext from the specified map of raw messages.
func UnmarshalProviderGatewayCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderGatewayCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "start", &obj.Start)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ProviderGatewayPortIdentity : Select Port Label for the gateway.
type ProviderGatewayPortIdentity struct {
	// Port identifier.
	ID *string `json:"id" validate:"required"`
}

// NewProviderGatewayPortIdentity : Instantiate ProviderGatewayPortIdentity (Generic Model Constructor)
func (*DirectLinkProviderV2) NewProviderGatewayPortIdentity(id string) (model *ProviderGatewayPortIdentity, err error) {
	model = &ProviderGatewayPortIdentity{
		ID: core.StringPtr(id),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalProviderGatewayPortIdentity unmarshals an instance of ProviderGatewayPortIdentity from the specified map of raw messages.
func UnmarshalProviderGatewayPortIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderGatewayPortIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ProviderGatewayPortReference : Port identifier for the gateway.
type ProviderGatewayPortReference struct {
	// Port identifier.
	ID *string `json:"id" validate:"required"`
}

// UnmarshalProviderGatewayPortReference unmarshals an instance of ProviderGatewayPortReference from the specified map of raw messages.
func UnmarshalProviderGatewayPortReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderGatewayPortReference)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ProviderGatewayUpdateAttributesUpdatesItem : ProviderGatewayUpdateAttributesUpdatesItem struct
// Models which "extend" this model:
// - ProviderGatewayUpdateAttributesUpdatesItemProviderGatewaySpeedUpdate
type ProviderGatewayUpdateAttributesUpdatesItem struct {
	// New gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps,omitempty"`
}

func (*ProviderGatewayUpdateAttributesUpdatesItem) isaProviderGatewayUpdateAttributesUpdatesItem() bool {
	return true
}

type ProviderGatewayUpdateAttributesUpdatesItemIntf interface {
	isaProviderGatewayUpdateAttributesUpdatesItem() bool
}

// UnmarshalProviderGatewayUpdateAttributesUpdatesItem unmarshals an instance of ProviderGatewayUpdateAttributesUpdatesItem from the specified map of raw messages.
func UnmarshalProviderGatewayUpdateAttributesUpdatesItem(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderGatewayUpdateAttributesUpdatesItem)
	err = core.UnmarshalPrimitive(m, "speed_mbps", &obj.SpeedMbps)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ProviderPort : Provider port details.
type ProviderPort struct {
	// Port identifier.
	ID *string `json:"id" validate:"required"`

	// Port Label.
	Label *string `json:"label" validate:"required"`

	// Port location long name.
	LocationDisplayName *string `json:"location_display_name" validate:"required"`

	// Port location name identifier.
	LocationName *string `json:"location_name" validate:"required"`

	// Port provider name.
	ProviderName *string `json:"provider_name" validate:"required"`

	// Port's supported speeds in megabits per second.
	SupportedLinkSpeeds []int64 `json:"supported_link_speeds" validate:"required"`
}

// UnmarshalProviderPort unmarshals an instance of ProviderPort from the specified map of raw messages.
func UnmarshalProviderPort(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderPort)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "label", &obj.Label)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "location_display_name", &obj.LocationDisplayName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "location_name", &obj.LocationName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "provider_name", &obj.ProviderName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "supported_link_speeds", &obj.SupportedLinkSpeeds)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ProviderPortCollection : List of port label details.
type ProviderPortCollection struct {
	// A reference to the first page of resources.
	First *ProviderPortCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A reference to the next page of resources; this reference is included for all pages except the last page.
	Next *ProviderPortCollectionNext `json:"next,omitempty"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`

	// Array of ports.
	Ports []ProviderPort `json:"ports,omitempty"`
}

// UnmarshalProviderPortCollection unmarshals an instance of ProviderPortCollection from the specified map of raw messages.
func UnmarshalProviderPortCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderPortCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalProviderPortCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalProviderPortCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "ports", &obj.Ports, UnmarshalProviderPort)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ProviderPortCollectionFirst : A reference to the first page of resources.
type ProviderPortCollectionFirst struct {
	// The URL for the first page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalProviderPortCollectionFirst unmarshals an instance of ProviderPortCollectionFirst from the specified map of raw messages.
func UnmarshalProviderPortCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderPortCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ProviderPortCollectionNext : A reference to the next page of resources; this reference is included for all pages except the last page.
type ProviderPortCollectionNext struct {
	// URL for the next page of resources.
	Href *string `json:"href" validate:"required"`

	// start token for the next page of resources.
	Start *string `json:"start" validate:"required"`
}

// UnmarshalProviderPortCollectionNext unmarshals an instance of ProviderPortCollectionNext from the specified map of raw messages.
func UnmarshalProviderPortCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderPortCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "start", &obj.Start)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// UpdateProviderGatewayOptions : The UpdateProviderGateway options.
type UpdateProviderGatewayOptions struct {
	// Direct Link Connect gateway identifier.
	ID *string `json:"id" validate:"required"`

	// The unique user-defined name for this gateway.
	Name *string `json:"name,omitempty"`

	// Gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps,omitempty"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateProviderGatewayOptions : Instantiate UpdateProviderGatewayOptions
func (*DirectLinkProviderV2) NewUpdateProviderGatewayOptions(id string) *UpdateProviderGatewayOptions {
	return &UpdateProviderGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *UpdateProviderGatewayOptions) SetID(id string) *UpdateProviderGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetName : Allow user to set Name
func (options *UpdateProviderGatewayOptions) SetName(name string) *UpdateProviderGatewayOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetSpeedMbps : Allow user to set SpeedMbps
func (options *UpdateProviderGatewayOptions) SetSpeedMbps(speedMbps int64) *UpdateProviderGatewayOptions {
	options.SpeedMbps = core.Int64Ptr(speedMbps)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateProviderGatewayOptions) SetHeaders(param map[string]string) *UpdateProviderGatewayOptions {
	options.Headers = param
	return options
}

// ProviderGatewayChangeRequestProviderGatewayCreate : gateway create.
// This model "extends" ProviderGatewayChangeRequest
type ProviderGatewayChangeRequestProviderGatewayCreate struct {
	// type of gateway change request.
	Type *string `json:"type" validate:"required"`
}

// Constants associated with the ProviderGatewayChangeRequestProviderGatewayCreate.Type property.
// type of gateway change request.
const (
	ProviderGatewayChangeRequestProviderGatewayCreate_Type_CreateGateway = "create_gateway"
)

func (*ProviderGatewayChangeRequestProviderGatewayCreate) isaProviderGatewayChangeRequest() bool {
	return true
}

// UnmarshalProviderGatewayChangeRequestProviderGatewayCreate unmarshals an instance of ProviderGatewayChangeRequestProviderGatewayCreate from the specified map of raw messages.
func UnmarshalProviderGatewayChangeRequestProviderGatewayCreate(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderGatewayChangeRequestProviderGatewayCreate)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ProviderGatewayChangeRequestProviderGatewayDelete : gateway delete.
// This model "extends" ProviderGatewayChangeRequest
type ProviderGatewayChangeRequestProviderGatewayDelete struct {
	// type of gateway change request.
	Type *string `json:"type" validate:"required"`
}

// Constants associated with the ProviderGatewayChangeRequestProviderGatewayDelete.Type property.
// type of gateway change request.
const (
	ProviderGatewayChangeRequestProviderGatewayDelete_Type_DeleteGateway = "delete_gateway"
)

func (*ProviderGatewayChangeRequestProviderGatewayDelete) isaProviderGatewayChangeRequest() bool {
	return true
}

// UnmarshalProviderGatewayChangeRequestProviderGatewayDelete unmarshals an instance of ProviderGatewayChangeRequestProviderGatewayDelete from the specified map of raw messages.
func UnmarshalProviderGatewayChangeRequestProviderGatewayDelete(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderGatewayChangeRequestProviderGatewayDelete)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ProviderGatewayChangeRequestProviderGatewayUpdateAttributes : gateway attributes update.
// This model "extends" ProviderGatewayChangeRequest
type ProviderGatewayChangeRequestProviderGatewayUpdateAttributes struct {
	// type of gateway change request.
	Type *string `json:"type" validate:"required"`

	// array of pending updates.
	Updates []ProviderGatewayUpdateAttributesUpdatesItemIntf `json:"updates" validate:"required"`
}

// Constants associated with the ProviderGatewayChangeRequestProviderGatewayUpdateAttributes.Type property.
// type of gateway change request.
const (
	ProviderGatewayChangeRequestProviderGatewayUpdateAttributes_Type_UpdateAttributes = "update_attributes"
)

func (*ProviderGatewayChangeRequestProviderGatewayUpdateAttributes) isaProviderGatewayChangeRequest() bool {
	return true
}

// UnmarshalProviderGatewayChangeRequestProviderGatewayUpdateAttributes unmarshals an instance of ProviderGatewayChangeRequestProviderGatewayUpdateAttributes from the specified map of raw messages.
func UnmarshalProviderGatewayChangeRequestProviderGatewayUpdateAttributes(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderGatewayChangeRequestProviderGatewayUpdateAttributes)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "updates", &obj.Updates, UnmarshalProviderGatewayUpdateAttributesUpdatesItem)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ProviderGatewayUpdateAttributesUpdatesItemProviderGatewaySpeedUpdate : gateway speed change.
// This model "extends" ProviderGatewayUpdateAttributesUpdatesItem
type ProviderGatewayUpdateAttributesUpdatesItemProviderGatewaySpeedUpdate struct {
	// New gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps,omitempty"`
}

func (*ProviderGatewayUpdateAttributesUpdatesItemProviderGatewaySpeedUpdate) isaProviderGatewayUpdateAttributesUpdatesItem() bool {
	return true
}

// UnmarshalProviderGatewayUpdateAttributesUpdatesItemProviderGatewaySpeedUpdate unmarshals an instance of ProviderGatewayUpdateAttributesUpdatesItemProviderGatewaySpeedUpdate from the specified map of raw messages.
func UnmarshalProviderGatewayUpdateAttributesUpdatesItemProviderGatewaySpeedUpdate(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ProviderGatewayUpdateAttributesUpdatesItemProviderGatewaySpeedUpdate)
	err = core.UnmarshalPrimitive(m, "speed_mbps", &obj.SpeedMbps)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}
