// Copyright 2019 The Cockroach Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
// implied. See the License for the specific language governing
// permissions and limitations under the License.

package markers

import (
	"context"
	"fmt"

	"github.com/cockroachdb/errors/errbase"
	"github.com/cockroachdb/errors/errorspb"
	"github.com/gogo/protobuf/proto"
)

// Is determines whether one of the causes of the given error or any
// of its causes is equivalent to some reference error.
//
// Note: if any of the error types has been migrated from a previous
// package location or a different type, ensure that
// RegisterTypeMigration() was called prior to Is().
func Is(err, reference error) bool {
	if err == nil {
		return err == reference
	}

	// Direct reference comparison is the fastest, and most
	// likely to be true, so do this first.
	for c := err; c != nil; c = errbase.UnwrapOnce(c) {
		if c == reference {
			return true
		}
	}

	// Not directly equal. Try harder, using error marks. We don't this
	// during the loop above as it may be more expensive.
	//
	// Note: there is a more effective recursive algorithm that ensures
	// that any pair of string only gets compared once. Should the
	// following code become a performance bottleneck, that algorithm
	// can be considered instead.
	refMark := getMark(reference)
	for c := err; c != nil; c = errbase.UnwrapOnce(c) {
		if equalMarks(getMark(c), refMark) {
			return true
		}
	}
	return false
}

// If returns a predicate's return value the first time the predicate returns true.
//
// Note: if any of the error types has been migrated from a previous
// package location or a different type, ensure that
// RegisterTypeMigration() was called prior to If().
func If(err error, pred func(err error) (interface{}, bool)) (interface{}, bool) {
	for c := err; c != nil; c = errbase.UnwrapOnce(c) {
		if v, ok := pred(c); ok {
			return v, ok
		}
	}
	return nil, false
}

// IsAny is like Is except that multiple references are compared.
//
// Note: if any of the error types has been migrated from a previous
// package location or a different type, ensure that
// RegisterTypeMigration() was called prior to IsAny().
func IsAny(err error, references ...error) bool {
	// First try using direct reference comparison.
	for c := err; c != nil; c = errbase.UnwrapOnce(c) {
		for _, refErr := range references {
			if err == refErr {
				return true
			}
		}
	}

	// Try harder with marks.
	// Note: there is a more effective recursive algorithm that ensures
	// that any pair of string only gets compared once. Should this
	// become a performance bottleneck, that algorithm can be considered
	// instead.
	refMarks := make([]errorMark, len(references))
	for i, refErr := range references {
		refMarks[i] = getMark(refErr)
	}
	for c := err; c != nil; c = errbase.UnwrapOnce(c) {
		errMark := getMark(err)
		for _, refMark := range refMarks {
			if equalMarks(errMark, refMark) {
				return true
			}
		}
	}
	return false
}

type errorMark struct {
	msg   string
	types []errorspb.ErrorTypeMark
}

// equalMarks compares two error markers.
func equalMarks(m1, m2 errorMark) bool {
	if m1.msg != m2.msg {
		return false
	}
	for i, t := range m1.types {
		if !t.Equals(m2.types[i]) {
			return false
		}
	}
	return true
}

// getMark computes a marker for the given error.
func getMark(err error) errorMark {
	if m, ok := err.(*withMark); ok {
		return m.mark
	}
	m := errorMark{msg: err.Error(), types: []errorspb.ErrorTypeMark{errbase.GetTypeMark(err)}}
	for c := errbase.UnwrapOnce(err); c != nil; c = errbase.UnwrapOnce(c) {
		m.types = append(m.types, errbase.GetTypeMark(c))
	}
	return m
}

// Mark creates an explicit mark for the given error, using
// the same mark as some reference error.
//
// Note: if any of the error types has been migrated from a previous
// package location or a different type, ensure that
// RegisterTypeMigration() was called prior to Mark().
func Mark(err error, reference error) error {
	if err == nil {
		return nil
	}
	refMark := getMark(reference)
	return &withMark{cause: err, mark: refMark}
}

// withMark carries an explicit mark.
type withMark struct {
	cause error
	mark  errorMark
}

var _ error = (*withMark)(nil)
var _ fmt.Formatter = (*withMark)(nil)
var _ errbase.Formatter = (*withMark)(nil)

func (m *withMark) Error() string { return m.cause.Error() }
func (m *withMark) Cause() error  { return m.cause }
func (m *withMark) Unwrap() error { return m.cause }

func (m *withMark) Format(s fmt.State, verb rune) { errbase.FormatError(m, s, verb) }

func (m *withMark) FormatError(p errbase.Printer) error {
	if p.Detail() {
		p.Printf("error with mark override:\n%q\n%s::%s",
			m.mark.msg,
			m.mark.types[0].FamilyName,
			m.mark.types[0].Extension,
		)
	}
	return m.cause
}

func encodeMark(_ context.Context, err error) (msg string, _ []string, payload proto.Message) {
	m := err.(*withMark)
	payload = &errorspb.MarkPayload{Msg: m.mark.msg, Types: m.mark.types}
	return "", nil, payload
}

func decodeMark(_ context.Context, cause error, _ string, _ []string, payload proto.Message) error {
	m, ok := payload.(*errorspb.MarkPayload)
	if !ok {
		// If this ever happens, this means some version of the library
		// (presumably future) changed the payload type, and we're
		// receiving this here. In this case, give up and let
		// DecodeError use the opaque type.
		return nil
	}
	return &withMark{cause: cause, mark: errorMark{msg: m.Msg, types: m.Types}}
}

func init() {
	errbase.RegisterWrapperEncoder(errbase.GetTypeKey((*withMark)(nil)), encodeMark)
	errbase.RegisterWrapperDecoder(errbase.GetTypeKey((*withMark)(nil)), decodeMark)
}
