package util

import (
	"fmt"
	"reflect"

	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/runtime/schema"
)

func BuildContainer(container *corev1.Container, buildFn func(*corev1.Container)) corev1.Container {
	buildFn(container)
	return *container
}

// AvailabilityProberImageName is the name under which components can find the availability prober
// image in the release image.
const AvailabilityProberImageName = "availability-prober"

func AvailabilityProber(target string, image string, spec *corev1.PodSpec, o ...AvailabilityProberOpt) {
	opts := AvailabilityProberOpts{}
	for _, opt := range o {
		opt(&opts)
	}
	availabilityProberContainer := corev1.Container{
		Name:            "availability-prober",
		Image:           image,
		ImagePullPolicy: corev1.PullAlways,
		Command: []string{
			"/usr/bin/availability-prober",
			"--target",
			target,
		},
	}
	if opts.KubeconfigVolumeName != "" {
		availabilityProberContainer.VolumeMounts = append(availabilityProberContainer.VolumeMounts, corev1.VolumeMount{
			Name:      opts.KubeconfigVolumeName,
			MountPath: "/var/kubeconfig",
		})
		availabilityProberContainer.Command = append(availabilityProberContainer.Command, "--kubeconfig=/var/kubeconfig/kubeconfig")
		for _, api := range opts.RequiredAPIs {
			availabilityProberContainer.Command = append(availabilityProberContainer.Command, fmt.Sprintf("--required-api=%s,%s,%s", api.Group, api.Version, api.Kind))
		}
	}
	if len(spec.InitContainers) == 0 || spec.InitContainers[0].Name != "availability-prober" {
		spec.InitContainers = append([]corev1.Container{{}}, spec.InitContainers...)
	}
	if !reflect.DeepEqual(spec.InitContainers[0], availabilityProberContainer) {
		spec.InitContainers[0] = availabilityProberContainer
	}
}

type AvailabilityProberOpts struct {
	KubeconfigVolumeName string
	RequiredAPIs         []schema.GroupVersionKind
}

type AvailabilityProberOpt func(*AvailabilityProberOpts)
