package manifests

import (
	"path/filepath"

	"github.com/ghodss/yaml"
	"github.com/pkg/errors"
	"github.com/sirupsen/logrus"

	configv1 "github.com/openshift/api/config/v1"
	"github.com/openshift/installer/pkg/asset"
	"github.com/openshift/installer/pkg/asset/installconfig"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

var (
	// SchedulerCfgFilename is the path of the Scheduler Config file
	SchedulerCfgFilename = filepath.Join(manifestDir, "cluster-scheduler-02-config.yml")
)

// Scheduler generates the cluster-scheduler-*.yml files.
type Scheduler struct {
	FileList []*asset.File
}

var _ asset.WritableAsset = (*Scheduler)(nil)

// Name returns a human friendly name for the asset.
func (*Scheduler) Name() string {
	return "Scheduler Config"
}

// Dependencies returns all of the dependencies directly needed to generate
// the asset.
func (*Scheduler) Dependencies() []asset.Asset {
	return []asset.Asset{
		&installconfig.InstallConfig{},
	}
}

// Generate generates the scheduler config and its CRD.
func (s *Scheduler) Generate(dependencies asset.Parents) error {
	config := &configv1.Scheduler{
		TypeMeta: metav1.TypeMeta{
			APIVersion: configv1.SchemeGroupVersion.String(),
			Kind:       "Scheduler",
		},
		ObjectMeta: metav1.ObjectMeta{
			Name: "cluster",
			// not namespaced
		},
		Spec: configv1.SchedulerSpec{
			MastersSchedulable: false,
		},
	}

	installConfig := &installconfig.InstallConfig{}
	dependencies.Get(installConfig)
	computeReplicas := int64(0)
	for _, pool := range installConfig.Config.Compute {
		if pool.Replicas != nil {
			computeReplicas += *pool.Replicas
		}
	}
	if computeReplicas == 0 {
		// A schedulable host is required for a successful install to complete.
		// If the install config has 0 replicas for compute hosts, it's one of two cases:
		//   1. An IPI deployment with no compute hosts.  The deployment can not succeed
		//      without MastersSchedulable = true.
		//   2. A UPI deployment.  The deployment may add compute hosts, but to ensure the
		//      the highest probability of a successful deployment, we default to
		//      schedulable masters.
		logrus.Warningf("Making control-plane schedulable by setting MastersSchedulable to true for Scheduler cluster settings")
		config.Spec.MastersSchedulable = true
	}

	configData, err := yaml.Marshal(config)
	if err != nil {
		return errors.Wrapf(err, "failed to create %s manifests from InstallConfig", s.Name())
	}

	s.FileList = []*asset.File{
		{
			Filename: SchedulerCfgFilename,
			Data:     configData,
		},
	}

	return nil
}

// Files returns the files generated by the asset.
func (s *Scheduler) Files() []*asset.File {
	return s.FileList
}

// Load returns false since this asset is not written to disk by the installer.
func (s *Scheduler) Load(f asset.FileFetcher) (bool, error) {
	return false, nil
}
