/**
 * (C) Copyright IBM Corp. 2021.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * IBM OpenAPI SDK Code Generator Version: 3.32.0-4c6a3129-20210514-210323
 */

// Package vpcv1 : Operations and models for the VpcV1 service
package vpcv1

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"reflect"
	"time"

	"github.com/IBM/go-sdk-core/v5/core"
	common "github.com/IBM/vpc-go-sdk/common"
	"github.com/go-openapi/strfmt"
)

// VpcV1 : The IBM Cloud Virtual Private Cloud (VPC) API can be used to programmatically provision and manage
// infrastructure resources, including virtual server instances, subnets, volumes, and load balancers.
//
// Version: 2021-06-08
type VpcV1 struct {
	Service *core.BaseService

	// Requests the version of the API as of a date in the format `YYYY-MM-DD`. Any date up to the current date may be
	// provided. Specify the current date to request the latest version.
	Version *string

	// The infrastructure generation for the request. For the API behavior documented here, use
	// `2`.
	generation *int64
}

// DefaultServiceURL is the default URL to make service requests to.
const DefaultServiceURL = "https://us-south.iaas.cloud.ibm.com/v1"

// DefaultServiceName is the default key used to find external configuration information.
const DefaultServiceName = "vpc"

// VpcV1Options : Service options
type VpcV1Options struct {
	ServiceName   string
	URL           string
	Authenticator core.Authenticator

	// Requests the version of the API as of a date in the format `YYYY-MM-DD`. Any date up to the current date may be
	// provided. Specify the current date to request the latest version.
	Version *string
}

// NewVpcV1UsingExternalConfig : constructs an instance of VpcV1 with passed in options and external configuration.
func NewVpcV1UsingExternalConfig(options *VpcV1Options) (vpc *VpcV1, err error) {
	if options.ServiceName == "" {
		options.ServiceName = DefaultServiceName
	}

	if options.Authenticator == nil {
		options.Authenticator, err = core.GetAuthenticatorFromEnvironment(options.ServiceName)
		if err != nil {
			return
		}
	}

	vpc, err = NewVpcV1(options)
	if err != nil {
		return
	}

	err = vpc.Service.ConfigureService(options.ServiceName)
	if err != nil {
		return
	}

	if options.URL != "" {
		err = vpc.Service.SetServiceURL(options.URL)
	}
	return
}

// NewVpcV1 : constructs an instance of VpcV1 with passed in options.
func NewVpcV1(options *VpcV1Options) (service *VpcV1, err error) {
	serviceOptions := &core.ServiceOptions{
		URL:           DefaultServiceURL,
		Authenticator: options.Authenticator,
	}

	err = core.ValidateStruct(options, "options")
	if err != nil {
		return
	}

	baseService, err := core.NewBaseService(serviceOptions)
	if err != nil {
		return
	}

	if options.URL != "" {
		err = baseService.SetServiceURL(options.URL)
		if err != nil {
			return
		}
	}

	if options.Version == nil {
		options.Version = core.StringPtr("2021-06-08")
	}

	service = &VpcV1{
		Service:    baseService,
		Version:    options.Version,
		generation: core.Int64Ptr(2),
	}

	return
}

// GetServiceURLForRegion returns the service URL to be used for the specified region
func GetServiceURLForRegion(region string) (string, error) {
	return "", fmt.Errorf("service does not support regional URLs")
}

// Clone makes a copy of "vpc" suitable for processing requests.
func (vpc *VpcV1) Clone() *VpcV1 {
	if core.IsNil(vpc) {
		return nil
	}
	clone := *vpc
	clone.Service = vpc.Service.Clone()
	return &clone
}

// SetServiceURL sets the service URL
func (vpc *VpcV1) SetServiceURL(url string) error {
	return vpc.Service.SetServiceURL(url)
}

// GetServiceURL returns the service URL
func (vpc *VpcV1) GetServiceURL() string {
	return vpc.Service.GetServiceURL()
}

// SetDefaultHeaders sets HTTP headers to be sent in every request
func (vpc *VpcV1) SetDefaultHeaders(headers http.Header) {
	vpc.Service.SetDefaultHeaders(headers)
}

// SetEnableGzipCompression sets the service's EnableGzipCompression field
func (vpc *VpcV1) SetEnableGzipCompression(enableGzip bool) {
	vpc.Service.SetEnableGzipCompression(enableGzip)
}

// GetEnableGzipCompression returns the service's EnableGzipCompression field
func (vpc *VpcV1) GetEnableGzipCompression() bool {
	return vpc.Service.GetEnableGzipCompression()
}

// EnableRetries enables automatic retries for requests invoked for this service instance.
// If either parameter is specified as 0, then a default value is used instead.
func (vpc *VpcV1) EnableRetries(maxRetries int, maxRetryInterval time.Duration) {
	vpc.Service.EnableRetries(maxRetries, maxRetryInterval)
}

// DisableRetries disables automatic retries for requests invoked for this service instance.
func (vpc *VpcV1) DisableRetries() {
	vpc.Service.DisableRetries()
}

// ListVpcs : List all VPCs
// This request lists all VPCs in the region. A VPC is a virtual network that belongs to an account and provides logical
// isolation from other networks. A VPC is made up of resources in one or more zones. VPCs are regional, and each VPC
// can contain resources in multiple zones in a region.
func (vpc *VpcV1) ListVpcs(listVpcsOptions *ListVpcsOptions) (result *VPCCollection, response *core.DetailedResponse, err error) {
	return vpc.ListVpcsWithContext(context.Background(), listVpcsOptions)
}

// ListVpcsWithContext is an alternate form of the ListVpcs method which supports a Context parameter
func (vpc *VpcV1) ListVpcsWithContext(ctx context.Context, listVpcsOptions *ListVpcsOptions) (result *VPCCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listVpcsOptions, "listVpcsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVpcsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListVpcs")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listVpcsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listVpcsOptions.Start))
	}
	if listVpcsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listVpcsOptions.Limit))
	}
	if listVpcsOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listVpcsOptions.ResourceGroupID))
	}
	if listVpcsOptions.ClassicAccess != nil {
		builder.AddQuery("classic_access", fmt.Sprint(*listVpcsOptions.ClassicAccess))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPCCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateVPC : Create a VPC
// This request creates a new VPC from a VPC prototype object. The prototype object is structured in the same way as a
// retrieved VPC, and contains the information necessary to create the new VPC.
func (vpc *VpcV1) CreateVPC(createVPCOptions *CreateVPCOptions) (result *VPC, response *core.DetailedResponse, err error) {
	return vpc.CreateVPCWithContext(context.Background(), createVPCOptions)
}

// CreateVPCWithContext is an alternate form of the CreateVPC method which supports a Context parameter
func (vpc *VpcV1) CreateVPCWithContext(ctx context.Context, createVPCOptions *CreateVPCOptions) (result *VPC, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(createVPCOptions, "createVPCOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createVPCOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateVPC")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createVPCOptions.AddressPrefixManagement != nil {
		body["address_prefix_management"] = createVPCOptions.AddressPrefixManagement
	}
	if createVPCOptions.ClassicAccess != nil {
		body["classic_access"] = createVPCOptions.ClassicAccess
	}
	if createVPCOptions.Name != nil {
		body["name"] = createVPCOptions.Name
	}
	if createVPCOptions.ResourceGroup != nil {
		body["resource_group"] = createVPCOptions.ResourceGroup
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPC)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteVPC : Delete a VPC
// This request deletes a VPC. This operation cannot be reversed. For this request to succeed, the VPC must not contain
// any instances, subnets, or public gateways. All security groups and network ACLs associated with the VPC are
// automatically deleted. All flow log collectors with `auto_delete` set to `true` targeting the VPC or any resource in
// the VPC are automatically deleted.
func (vpc *VpcV1) DeleteVPC(deleteVPCOptions *DeleteVPCOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteVPCWithContext(context.Background(), deleteVPCOptions)
}

// DeleteVPCWithContext is an alternate form of the DeleteVPC method which supports a Context parameter
func (vpc *VpcV1) DeleteVPCWithContext(ctx context.Context, deleteVPCOptions *DeleteVPCOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteVPCOptions, "deleteVPCOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteVPCOptions, "deleteVPCOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteVPCOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteVPCOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteVPC")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetVPC : Retrieve a VPC
// This request retrieves a single VPC specified by the identifier in the URL.
func (vpc *VpcV1) GetVPC(getVPCOptions *GetVPCOptions) (result *VPC, response *core.DetailedResponse, err error) {
	return vpc.GetVPCWithContext(context.Background(), getVPCOptions)
}

// GetVPCWithContext is an alternate form of the GetVPC method which supports a Context parameter
func (vpc *VpcV1) GetVPCWithContext(ctx context.Context, getVPCOptions *GetVPCOptions) (result *VPC, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPCOptions, "getVPCOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPCOptions, "getVPCOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getVPCOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPCOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetVPC")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPC)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateVPC : Update a VPC
// This request updates a VPC's name.
func (vpc *VpcV1) UpdateVPC(updateVPCOptions *UpdateVPCOptions) (result *VPC, response *core.DetailedResponse, err error) {
	return vpc.UpdateVPCWithContext(context.Background(), updateVPCOptions)
}

// UpdateVPCWithContext is an alternate form of the UpdateVPC method which supports a Context parameter
func (vpc *VpcV1) UpdateVPCWithContext(ctx context.Context, updateVPCOptions *UpdateVPCOptions) (result *VPC, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateVPCOptions, "updateVPCOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateVPCOptions, "updateVPCOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateVPCOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateVPCOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateVPC")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateVPCOptions.VPCPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPC)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetVPCDefaultNetworkACL : Retrieve a VPC's default network ACL
// This request retrieves the default network ACL for the VPC specified by the identifier in the URL. The default
// network ACL is applied to any new subnets in the VPC which do not specify a network ACL.
func (vpc *VpcV1) GetVPCDefaultNetworkACL(getVPCDefaultNetworkACLOptions *GetVPCDefaultNetworkACLOptions) (result *DefaultNetworkACL, response *core.DetailedResponse, err error) {
	return vpc.GetVPCDefaultNetworkACLWithContext(context.Background(), getVPCDefaultNetworkACLOptions)
}

// GetVPCDefaultNetworkACLWithContext is an alternate form of the GetVPCDefaultNetworkACL method which supports a Context parameter
func (vpc *VpcV1) GetVPCDefaultNetworkACLWithContext(ctx context.Context, getVPCDefaultNetworkACLOptions *GetVPCDefaultNetworkACLOptions) (result *DefaultNetworkACL, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPCDefaultNetworkACLOptions, "getVPCDefaultNetworkACLOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPCDefaultNetworkACLOptions, "getVPCDefaultNetworkACLOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getVPCDefaultNetworkACLOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{id}/default_network_acl`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPCDefaultNetworkACLOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetVPCDefaultNetworkACL")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDefaultNetworkACL)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetVPCDefaultRoutingTable : Retrieve a VPC's default routing table
// This request retrieves the default routing table for the VPC specified by the identifier in the URL. The default
// routing table is associated with any subnets in the VPC which have not been explicitly associated with a user-defined
// routing table.
func (vpc *VpcV1) GetVPCDefaultRoutingTable(getVPCDefaultRoutingTableOptions *GetVPCDefaultRoutingTableOptions) (result *DefaultRoutingTable, response *core.DetailedResponse, err error) {
	return vpc.GetVPCDefaultRoutingTableWithContext(context.Background(), getVPCDefaultRoutingTableOptions)
}

// GetVPCDefaultRoutingTableWithContext is an alternate form of the GetVPCDefaultRoutingTable method which supports a Context parameter
func (vpc *VpcV1) GetVPCDefaultRoutingTableWithContext(ctx context.Context, getVPCDefaultRoutingTableOptions *GetVPCDefaultRoutingTableOptions) (result *DefaultRoutingTable, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPCDefaultRoutingTableOptions, "getVPCDefaultRoutingTableOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPCDefaultRoutingTableOptions, "getVPCDefaultRoutingTableOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getVPCDefaultRoutingTableOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{id}/default_routing_table`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPCDefaultRoutingTableOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetVPCDefaultRoutingTable")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDefaultRoutingTable)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetVPCDefaultSecurityGroup : Retrieve a VPC's default security group
// This request retrieves the default security group for the VPC specified by the identifier in the URL. The default
// security group is applied to any new network interfaces in the VPC that do not specify a security group.
func (vpc *VpcV1) GetVPCDefaultSecurityGroup(getVPCDefaultSecurityGroupOptions *GetVPCDefaultSecurityGroupOptions) (result *DefaultSecurityGroup, response *core.DetailedResponse, err error) {
	return vpc.GetVPCDefaultSecurityGroupWithContext(context.Background(), getVPCDefaultSecurityGroupOptions)
}

// GetVPCDefaultSecurityGroupWithContext is an alternate form of the GetVPCDefaultSecurityGroup method which supports a Context parameter
func (vpc *VpcV1) GetVPCDefaultSecurityGroupWithContext(ctx context.Context, getVPCDefaultSecurityGroupOptions *GetVPCDefaultSecurityGroupOptions) (result *DefaultSecurityGroup, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPCDefaultSecurityGroupOptions, "getVPCDefaultSecurityGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPCDefaultSecurityGroupOptions, "getVPCDefaultSecurityGroupOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getVPCDefaultSecurityGroupOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{id}/default_security_group`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPCDefaultSecurityGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetVPCDefaultSecurityGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDefaultSecurityGroup)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListVPCAddressPrefixes : List all address prefixes for a VPC
// This request lists all address pool prefixes for a VPC.
func (vpc *VpcV1) ListVPCAddressPrefixes(listVPCAddressPrefixesOptions *ListVPCAddressPrefixesOptions) (result *AddressPrefixCollection, response *core.DetailedResponse, err error) {
	return vpc.ListVPCAddressPrefixesWithContext(context.Background(), listVPCAddressPrefixesOptions)
}

// ListVPCAddressPrefixesWithContext is an alternate form of the ListVPCAddressPrefixes method which supports a Context parameter
func (vpc *VpcV1) ListVPCAddressPrefixesWithContext(ctx context.Context, listVPCAddressPrefixesOptions *ListVPCAddressPrefixesOptions) (result *AddressPrefixCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listVPCAddressPrefixesOptions, "listVPCAddressPrefixesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listVPCAddressPrefixesOptions, "listVPCAddressPrefixesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *listVPCAddressPrefixesOptions.VPCID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/address_prefixes`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVPCAddressPrefixesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListVPCAddressPrefixes")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listVPCAddressPrefixesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listVPCAddressPrefixesOptions.Start))
	}
	if listVPCAddressPrefixesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listVPCAddressPrefixesOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalAddressPrefixCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateVPCAddressPrefix : Create an address prefix for a VPC
// This request creates a new prefix from a prefix prototype object. The prototype object is structured in the same way
// as a retrieved prefix, and contains the information necessary to create the new prefix.
func (vpc *VpcV1) CreateVPCAddressPrefix(createVPCAddressPrefixOptions *CreateVPCAddressPrefixOptions) (result *AddressPrefix, response *core.DetailedResponse, err error) {
	return vpc.CreateVPCAddressPrefixWithContext(context.Background(), createVPCAddressPrefixOptions)
}

// CreateVPCAddressPrefixWithContext is an alternate form of the CreateVPCAddressPrefix method which supports a Context parameter
func (vpc *VpcV1) CreateVPCAddressPrefixWithContext(ctx context.Context, createVPCAddressPrefixOptions *CreateVPCAddressPrefixOptions) (result *AddressPrefix, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createVPCAddressPrefixOptions, "createVPCAddressPrefixOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createVPCAddressPrefixOptions, "createVPCAddressPrefixOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *createVPCAddressPrefixOptions.VPCID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/address_prefixes`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createVPCAddressPrefixOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateVPCAddressPrefix")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createVPCAddressPrefixOptions.CIDR != nil {
		body["cidr"] = createVPCAddressPrefixOptions.CIDR
	}
	if createVPCAddressPrefixOptions.Zone != nil {
		body["zone"] = createVPCAddressPrefixOptions.Zone
	}
	if createVPCAddressPrefixOptions.IsDefault != nil {
		body["is_default"] = createVPCAddressPrefixOptions.IsDefault
	}
	if createVPCAddressPrefixOptions.Name != nil {
		body["name"] = createVPCAddressPrefixOptions.Name
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalAddressPrefix)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteVPCAddressPrefix : Delete an address prefix
// This request deletes a prefix. This operation cannot be reversed. The request will fail if any subnets use addresses
// from this prefix.
func (vpc *VpcV1) DeleteVPCAddressPrefix(deleteVPCAddressPrefixOptions *DeleteVPCAddressPrefixOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteVPCAddressPrefixWithContext(context.Background(), deleteVPCAddressPrefixOptions)
}

// DeleteVPCAddressPrefixWithContext is an alternate form of the DeleteVPCAddressPrefix method which supports a Context parameter
func (vpc *VpcV1) DeleteVPCAddressPrefixWithContext(ctx context.Context, deleteVPCAddressPrefixOptions *DeleteVPCAddressPrefixOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteVPCAddressPrefixOptions, "deleteVPCAddressPrefixOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteVPCAddressPrefixOptions, "deleteVPCAddressPrefixOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *deleteVPCAddressPrefixOptions.VPCID,
		"id":     *deleteVPCAddressPrefixOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/address_prefixes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteVPCAddressPrefixOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteVPCAddressPrefix")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetVPCAddressPrefix : Retrieve an address prefix
// This request retrieves a single prefix specified by the identifier in the URL.
func (vpc *VpcV1) GetVPCAddressPrefix(getVPCAddressPrefixOptions *GetVPCAddressPrefixOptions) (result *AddressPrefix, response *core.DetailedResponse, err error) {
	return vpc.GetVPCAddressPrefixWithContext(context.Background(), getVPCAddressPrefixOptions)
}

// GetVPCAddressPrefixWithContext is an alternate form of the GetVPCAddressPrefix method which supports a Context parameter
func (vpc *VpcV1) GetVPCAddressPrefixWithContext(ctx context.Context, getVPCAddressPrefixOptions *GetVPCAddressPrefixOptions) (result *AddressPrefix, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPCAddressPrefixOptions, "getVPCAddressPrefixOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPCAddressPrefixOptions, "getVPCAddressPrefixOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *getVPCAddressPrefixOptions.VPCID,
		"id":     *getVPCAddressPrefixOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/address_prefixes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPCAddressPrefixOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetVPCAddressPrefix")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalAddressPrefix)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateVPCAddressPrefix : Update an address prefix
// This request updates a prefix with the information in a provided prefix patch. The prefix patch object is structured
// in the same way as a retrieved prefix and contains only the information to be updated.
func (vpc *VpcV1) UpdateVPCAddressPrefix(updateVPCAddressPrefixOptions *UpdateVPCAddressPrefixOptions) (result *AddressPrefix, response *core.DetailedResponse, err error) {
	return vpc.UpdateVPCAddressPrefixWithContext(context.Background(), updateVPCAddressPrefixOptions)
}

// UpdateVPCAddressPrefixWithContext is an alternate form of the UpdateVPCAddressPrefix method which supports a Context parameter
func (vpc *VpcV1) UpdateVPCAddressPrefixWithContext(ctx context.Context, updateVPCAddressPrefixOptions *UpdateVPCAddressPrefixOptions) (result *AddressPrefix, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateVPCAddressPrefixOptions, "updateVPCAddressPrefixOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateVPCAddressPrefixOptions, "updateVPCAddressPrefixOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *updateVPCAddressPrefixOptions.VPCID,
		"id":     *updateVPCAddressPrefixOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/address_prefixes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateVPCAddressPrefixOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateVPCAddressPrefix")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateVPCAddressPrefixOptions.AddressPrefixPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalAddressPrefix)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListVPCRoutes : List all routes in a VPC's default routing table
// This request lists all routes in the VPC's default routing table. Each route is zone-specific and directs any packets
// matching its destination CIDR block to a `next_hop` IP address. The most specific route matching a packet's
// destination will be used. If multiple equally-specific routes exist, traffic will be distributed across them.
func (vpc *VpcV1) ListVPCRoutes(listVPCRoutesOptions *ListVPCRoutesOptions) (result *RouteCollection, response *core.DetailedResponse, err error) {
	return vpc.ListVPCRoutesWithContext(context.Background(), listVPCRoutesOptions)
}

// ListVPCRoutesWithContext is an alternate form of the ListVPCRoutes method which supports a Context parameter
func (vpc *VpcV1) ListVPCRoutesWithContext(ctx context.Context, listVPCRoutesOptions *ListVPCRoutesOptions) (result *RouteCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listVPCRoutesOptions, "listVPCRoutesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listVPCRoutesOptions, "listVPCRoutesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *listVPCRoutesOptions.VPCID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/routes`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVPCRoutesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListVPCRoutes")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listVPCRoutesOptions.ZoneName != nil {
		builder.AddQuery("zone.name", fmt.Sprint(*listVPCRoutesOptions.ZoneName))
	}
	if listVPCRoutesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listVPCRoutesOptions.Start))
	}
	if listVPCRoutesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listVPCRoutesOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRouteCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateVPCRoute : Create a route in a VPC's default routing table
// This request creates a new route in the VPC's default routing table. The route prototype object is structured in the
// same way as a retrieved route, and contains the information necessary to create the new route. The request will fail
// if the new route will cause a loop.
func (vpc *VpcV1) CreateVPCRoute(createVPCRouteOptions *CreateVPCRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	return vpc.CreateVPCRouteWithContext(context.Background(), createVPCRouteOptions)
}

// CreateVPCRouteWithContext is an alternate form of the CreateVPCRoute method which supports a Context parameter
func (vpc *VpcV1) CreateVPCRouteWithContext(ctx context.Context, createVPCRouteOptions *CreateVPCRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createVPCRouteOptions, "createVPCRouteOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createVPCRouteOptions, "createVPCRouteOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *createVPCRouteOptions.VPCID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/routes`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createVPCRouteOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateVPCRoute")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createVPCRouteOptions.Destination != nil {
		body["destination"] = createVPCRouteOptions.Destination
	}
	if createVPCRouteOptions.Zone != nil {
		body["zone"] = createVPCRouteOptions.Zone
	}
	if createVPCRouteOptions.Action != nil {
		body["action"] = createVPCRouteOptions.Action
	}
	if createVPCRouteOptions.Name != nil {
		body["name"] = createVPCRouteOptions.Name
	}
	if createVPCRouteOptions.NextHop != nil {
		body["next_hop"] = createVPCRouteOptions.NextHop
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRoute)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteVPCRoute : Delete a VPC route
// This request deletes a route. This operation cannot be reversed.
func (vpc *VpcV1) DeleteVPCRoute(deleteVPCRouteOptions *DeleteVPCRouteOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteVPCRouteWithContext(context.Background(), deleteVPCRouteOptions)
}

// DeleteVPCRouteWithContext is an alternate form of the DeleteVPCRoute method which supports a Context parameter
func (vpc *VpcV1) DeleteVPCRouteWithContext(ctx context.Context, deleteVPCRouteOptions *DeleteVPCRouteOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteVPCRouteOptions, "deleteVPCRouteOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteVPCRouteOptions, "deleteVPCRouteOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *deleteVPCRouteOptions.VPCID,
		"id":     *deleteVPCRouteOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/routes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteVPCRouteOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteVPCRoute")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetVPCRoute : Retrieve a VPC route
// This request retrieves a single route specified by the identifier in the URL.
func (vpc *VpcV1) GetVPCRoute(getVPCRouteOptions *GetVPCRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	return vpc.GetVPCRouteWithContext(context.Background(), getVPCRouteOptions)
}

// GetVPCRouteWithContext is an alternate form of the GetVPCRoute method which supports a Context parameter
func (vpc *VpcV1) GetVPCRouteWithContext(ctx context.Context, getVPCRouteOptions *GetVPCRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPCRouteOptions, "getVPCRouteOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPCRouteOptions, "getVPCRouteOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *getVPCRouteOptions.VPCID,
		"id":     *getVPCRouteOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/routes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPCRouteOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetVPCRoute")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRoute)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateVPCRoute : Update a VPC route
// This request updates a route with the information in a provided route patch. The route patch object is structured in
// the same way as a retrieved route and contains only the information to be updated.
func (vpc *VpcV1) UpdateVPCRoute(updateVPCRouteOptions *UpdateVPCRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	return vpc.UpdateVPCRouteWithContext(context.Background(), updateVPCRouteOptions)
}

// UpdateVPCRouteWithContext is an alternate form of the UpdateVPCRoute method which supports a Context parameter
func (vpc *VpcV1) UpdateVPCRouteWithContext(ctx context.Context, updateVPCRouteOptions *UpdateVPCRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateVPCRouteOptions, "updateVPCRouteOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateVPCRouteOptions, "updateVPCRouteOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *updateVPCRouteOptions.VPCID,
		"id":     *updateVPCRouteOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/routes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateVPCRouteOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateVPCRoute")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateVPCRouteOptions.RoutePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRoute)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListVPCRoutingTables : List all routing tables for a VPC
// This request lists all user-defined routing tables for a VPC.  Each subnet in a VPC is associated with a routing
// table, which controls delivery of packets sent on that subnet according to the action of the most specific matching
// route in the table.  If multiple equally-specific routes exist, traffic will be distributed across them.  If no
// routes match, delivery will be controlled by the system's built-in routes.
func (vpc *VpcV1) ListVPCRoutingTables(listVPCRoutingTablesOptions *ListVPCRoutingTablesOptions) (result *RoutingTableCollection, response *core.DetailedResponse, err error) {
	return vpc.ListVPCRoutingTablesWithContext(context.Background(), listVPCRoutingTablesOptions)
}

// ListVPCRoutingTablesWithContext is an alternate form of the ListVPCRoutingTables method which supports a Context parameter
func (vpc *VpcV1) ListVPCRoutingTablesWithContext(ctx context.Context, listVPCRoutingTablesOptions *ListVPCRoutingTablesOptions) (result *RoutingTableCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listVPCRoutingTablesOptions, "listVPCRoutingTablesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listVPCRoutingTablesOptions, "listVPCRoutingTablesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *listVPCRoutingTablesOptions.VPCID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/routing_tables`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVPCRoutingTablesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListVPCRoutingTables")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listVPCRoutingTablesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listVPCRoutingTablesOptions.Start))
	}
	if listVPCRoutingTablesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listVPCRoutingTablesOptions.Limit))
	}
	if listVPCRoutingTablesOptions.IsDefault != nil {
		builder.AddQuery("is_default", fmt.Sprint(*listVPCRoutingTablesOptions.IsDefault))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRoutingTableCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateVPCRoutingTable : Create a routing table for a VPC
// This request creates a user-defined routing table from a routing table prototype object. The prototype object is
// structured in the same way as a retrieved routing table, and contains the information necessary to create the new
// routing table.
func (vpc *VpcV1) CreateVPCRoutingTable(createVPCRoutingTableOptions *CreateVPCRoutingTableOptions) (result *RoutingTable, response *core.DetailedResponse, err error) {
	return vpc.CreateVPCRoutingTableWithContext(context.Background(), createVPCRoutingTableOptions)
}

// CreateVPCRoutingTableWithContext is an alternate form of the CreateVPCRoutingTable method which supports a Context parameter
func (vpc *VpcV1) CreateVPCRoutingTableWithContext(ctx context.Context, createVPCRoutingTableOptions *CreateVPCRoutingTableOptions) (result *RoutingTable, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createVPCRoutingTableOptions, "createVPCRoutingTableOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createVPCRoutingTableOptions, "createVPCRoutingTableOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *createVPCRoutingTableOptions.VPCID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/routing_tables`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createVPCRoutingTableOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateVPCRoutingTable")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createVPCRoutingTableOptions.Name != nil {
		body["name"] = createVPCRoutingTableOptions.Name
	}
	if createVPCRoutingTableOptions.RouteDirectLinkIngress != nil {
		body["route_direct_link_ingress"] = createVPCRoutingTableOptions.RouteDirectLinkIngress
	}
	if createVPCRoutingTableOptions.RouteTransitGatewayIngress != nil {
		body["route_transit_gateway_ingress"] = createVPCRoutingTableOptions.RouteTransitGatewayIngress
	}
	if createVPCRoutingTableOptions.RouteVPCZoneIngress != nil {
		body["route_vpc_zone_ingress"] = createVPCRoutingTableOptions.RouteVPCZoneIngress
	}
	if createVPCRoutingTableOptions.Routes != nil {
		body["routes"] = createVPCRoutingTableOptions.Routes
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRoutingTable)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteVPCRoutingTable : Delete a VPC routing table
// This request deletes a routing table.  A routing table cannot be deleted if it is associated with any subnets in the
// VPC. Additionally, a VPC's default routing table cannot be deleted. This operation cannot be reversed.
func (vpc *VpcV1) DeleteVPCRoutingTable(deleteVPCRoutingTableOptions *DeleteVPCRoutingTableOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteVPCRoutingTableWithContext(context.Background(), deleteVPCRoutingTableOptions)
}

// DeleteVPCRoutingTableWithContext is an alternate form of the DeleteVPCRoutingTable method which supports a Context parameter
func (vpc *VpcV1) DeleteVPCRoutingTableWithContext(ctx context.Context, deleteVPCRoutingTableOptions *DeleteVPCRoutingTableOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteVPCRoutingTableOptions, "deleteVPCRoutingTableOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteVPCRoutingTableOptions, "deleteVPCRoutingTableOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *deleteVPCRoutingTableOptions.VPCID,
		"id":     *deleteVPCRoutingTableOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/routing_tables/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteVPCRoutingTableOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteVPCRoutingTable")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetVPCRoutingTable : Retrieve a VPC routing table
// This request retrieves a single routing table specified by the identifier in the URL.
func (vpc *VpcV1) GetVPCRoutingTable(getVPCRoutingTableOptions *GetVPCRoutingTableOptions) (result *RoutingTable, response *core.DetailedResponse, err error) {
	return vpc.GetVPCRoutingTableWithContext(context.Background(), getVPCRoutingTableOptions)
}

// GetVPCRoutingTableWithContext is an alternate form of the GetVPCRoutingTable method which supports a Context parameter
func (vpc *VpcV1) GetVPCRoutingTableWithContext(ctx context.Context, getVPCRoutingTableOptions *GetVPCRoutingTableOptions) (result *RoutingTable, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPCRoutingTableOptions, "getVPCRoutingTableOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPCRoutingTableOptions, "getVPCRoutingTableOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *getVPCRoutingTableOptions.VPCID,
		"id":     *getVPCRoutingTableOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/routing_tables/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPCRoutingTableOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetVPCRoutingTable")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRoutingTable)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateVPCRoutingTable : Update a VPC routing table
// This request updates a routing table with the information in a provided routing table patch. The patch object is
// structured in the same way as a retrieved table and contains only the information to be updated.
func (vpc *VpcV1) UpdateVPCRoutingTable(updateVPCRoutingTableOptions *UpdateVPCRoutingTableOptions) (result *RoutingTable, response *core.DetailedResponse, err error) {
	return vpc.UpdateVPCRoutingTableWithContext(context.Background(), updateVPCRoutingTableOptions)
}

// UpdateVPCRoutingTableWithContext is an alternate form of the UpdateVPCRoutingTable method which supports a Context parameter
func (vpc *VpcV1) UpdateVPCRoutingTableWithContext(ctx context.Context, updateVPCRoutingTableOptions *UpdateVPCRoutingTableOptions) (result *RoutingTable, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateVPCRoutingTableOptions, "updateVPCRoutingTableOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateVPCRoutingTableOptions, "updateVPCRoutingTableOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id": *updateVPCRoutingTableOptions.VPCID,
		"id":     *updateVPCRoutingTableOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/routing_tables/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateVPCRoutingTableOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateVPCRoutingTable")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateVPCRoutingTableOptions.RoutingTablePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRoutingTable)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListVPCRoutingTableRoutes : List all routes in a VPC routing table
// This request lists all routes in a VPC routing table. If a subnet has been associated with this routing table,
// delivery of packets sent on a subnet is performed according to the action of the most specific matching route in the
// table (provided the subnet and route are in the same zone). If multiple equally-specific routes exist, traffic will
// be distributed across them. If no routes match, delivery will be controlled by the system's built-in routes.
func (vpc *VpcV1) ListVPCRoutingTableRoutes(listVPCRoutingTableRoutesOptions *ListVPCRoutingTableRoutesOptions) (result *RouteCollection, response *core.DetailedResponse, err error) {
	return vpc.ListVPCRoutingTableRoutesWithContext(context.Background(), listVPCRoutingTableRoutesOptions)
}

// ListVPCRoutingTableRoutesWithContext is an alternate form of the ListVPCRoutingTableRoutes method which supports a Context parameter
func (vpc *VpcV1) ListVPCRoutingTableRoutesWithContext(ctx context.Context, listVPCRoutingTableRoutesOptions *ListVPCRoutingTableRoutesOptions) (result *RouteCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listVPCRoutingTableRoutesOptions, "listVPCRoutingTableRoutesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listVPCRoutingTableRoutesOptions, "listVPCRoutingTableRoutesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id":           *listVPCRoutingTableRoutesOptions.VPCID,
		"routing_table_id": *listVPCRoutingTableRoutesOptions.RoutingTableID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/routing_tables/{routing_table_id}/routes`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVPCRoutingTableRoutesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListVPCRoutingTableRoutes")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listVPCRoutingTableRoutesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listVPCRoutingTableRoutesOptions.Start))
	}
	if listVPCRoutingTableRoutesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listVPCRoutingTableRoutesOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRouteCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateVPCRoutingTableRoute : Create a route in a VPC routing table
// This request creates a new VPC route from a VPC route prototype object. The prototype object is structured in the
// same way as a retrieved VPC route and contains the information necessary to create the route.
func (vpc *VpcV1) CreateVPCRoutingTableRoute(createVPCRoutingTableRouteOptions *CreateVPCRoutingTableRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	return vpc.CreateVPCRoutingTableRouteWithContext(context.Background(), createVPCRoutingTableRouteOptions)
}

// CreateVPCRoutingTableRouteWithContext is an alternate form of the CreateVPCRoutingTableRoute method which supports a Context parameter
func (vpc *VpcV1) CreateVPCRoutingTableRouteWithContext(ctx context.Context, createVPCRoutingTableRouteOptions *CreateVPCRoutingTableRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createVPCRoutingTableRouteOptions, "createVPCRoutingTableRouteOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createVPCRoutingTableRouteOptions, "createVPCRoutingTableRouteOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id":           *createVPCRoutingTableRouteOptions.VPCID,
		"routing_table_id": *createVPCRoutingTableRouteOptions.RoutingTableID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/routing_tables/{routing_table_id}/routes`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createVPCRoutingTableRouteOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateVPCRoutingTableRoute")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createVPCRoutingTableRouteOptions.Destination != nil {
		body["destination"] = createVPCRoutingTableRouteOptions.Destination
	}
	if createVPCRoutingTableRouteOptions.Zone != nil {
		body["zone"] = createVPCRoutingTableRouteOptions.Zone
	}
	if createVPCRoutingTableRouteOptions.Action != nil {
		body["action"] = createVPCRoutingTableRouteOptions.Action
	}
	if createVPCRoutingTableRouteOptions.Name != nil {
		body["name"] = createVPCRoutingTableRouteOptions.Name
	}
	if createVPCRoutingTableRouteOptions.NextHop != nil {
		body["next_hop"] = createVPCRoutingTableRouteOptions.NextHop
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRoute)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteVPCRoutingTableRoute : Delete a VPC routing table route
// This request deletes a VPC route. This operation cannot be reversed.
func (vpc *VpcV1) DeleteVPCRoutingTableRoute(deleteVPCRoutingTableRouteOptions *DeleteVPCRoutingTableRouteOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteVPCRoutingTableRouteWithContext(context.Background(), deleteVPCRoutingTableRouteOptions)
}

// DeleteVPCRoutingTableRouteWithContext is an alternate form of the DeleteVPCRoutingTableRoute method which supports a Context parameter
func (vpc *VpcV1) DeleteVPCRoutingTableRouteWithContext(ctx context.Context, deleteVPCRoutingTableRouteOptions *DeleteVPCRoutingTableRouteOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteVPCRoutingTableRouteOptions, "deleteVPCRoutingTableRouteOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteVPCRoutingTableRouteOptions, "deleteVPCRoutingTableRouteOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id":           *deleteVPCRoutingTableRouteOptions.VPCID,
		"routing_table_id": *deleteVPCRoutingTableRouteOptions.RoutingTableID,
		"id":               *deleteVPCRoutingTableRouteOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/routing_tables/{routing_table_id}/routes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteVPCRoutingTableRouteOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteVPCRoutingTableRoute")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetVPCRoutingTableRoute : Retrieve a VPC routing table route
// This request retrieves a single VPC route specified by the identifier in the URL path.
func (vpc *VpcV1) GetVPCRoutingTableRoute(getVPCRoutingTableRouteOptions *GetVPCRoutingTableRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	return vpc.GetVPCRoutingTableRouteWithContext(context.Background(), getVPCRoutingTableRouteOptions)
}

// GetVPCRoutingTableRouteWithContext is an alternate form of the GetVPCRoutingTableRoute method which supports a Context parameter
func (vpc *VpcV1) GetVPCRoutingTableRouteWithContext(ctx context.Context, getVPCRoutingTableRouteOptions *GetVPCRoutingTableRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPCRoutingTableRouteOptions, "getVPCRoutingTableRouteOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPCRoutingTableRouteOptions, "getVPCRoutingTableRouteOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id":           *getVPCRoutingTableRouteOptions.VPCID,
		"routing_table_id": *getVPCRoutingTableRouteOptions.RoutingTableID,
		"id":               *getVPCRoutingTableRouteOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/routing_tables/{routing_table_id}/routes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPCRoutingTableRouteOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetVPCRoutingTableRoute")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRoute)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateVPCRoutingTableRoute : Update a VPC routing table route
// This request updates a VPC route with the information provided in a route patch object. The patch object is
// structured in the same way as a retrieved VPC route and needs to contain only the information to be updated.
func (vpc *VpcV1) UpdateVPCRoutingTableRoute(updateVPCRoutingTableRouteOptions *UpdateVPCRoutingTableRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	return vpc.UpdateVPCRoutingTableRouteWithContext(context.Background(), updateVPCRoutingTableRouteOptions)
}

// UpdateVPCRoutingTableRouteWithContext is an alternate form of the UpdateVPCRoutingTableRoute method which supports a Context parameter
func (vpc *VpcV1) UpdateVPCRoutingTableRouteWithContext(ctx context.Context, updateVPCRoutingTableRouteOptions *UpdateVPCRoutingTableRouteOptions) (result *Route, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateVPCRoutingTableRouteOptions, "updateVPCRoutingTableRouteOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateVPCRoutingTableRouteOptions, "updateVPCRoutingTableRouteOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpc_id":           *updateVPCRoutingTableRouteOptions.VPCID,
		"routing_table_id": *updateVPCRoutingTableRouteOptions.RoutingTableID,
		"id":               *updateVPCRoutingTableRouteOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpcs/{vpc_id}/routing_tables/{routing_table_id}/routes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateVPCRoutingTableRouteOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateVPCRoutingTableRoute")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateVPCRoutingTableRouteOptions.RoutePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRoute)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListSubnets : List all subnets
// This request lists all subnets in the region. Subnets are contiguous ranges of IP addresses specified in CIDR block
// notation. Each subnet is within a particular zone and cannot span multiple zones or regions.
func (vpc *VpcV1) ListSubnets(listSubnetsOptions *ListSubnetsOptions) (result *SubnetCollection, response *core.DetailedResponse, err error) {
	return vpc.ListSubnetsWithContext(context.Background(), listSubnetsOptions)
}

// ListSubnetsWithContext is an alternate form of the ListSubnets method which supports a Context parameter
func (vpc *VpcV1) ListSubnetsWithContext(ctx context.Context, listSubnetsOptions *ListSubnetsOptions) (result *SubnetCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listSubnetsOptions, "listSubnetsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listSubnetsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListSubnets")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listSubnetsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listSubnetsOptions.Start))
	}
	if listSubnetsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listSubnetsOptions.Limit))
	}
	if listSubnetsOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listSubnetsOptions.ResourceGroupID))
	}
	if listSubnetsOptions.RoutingTableID != nil {
		builder.AddQuery("routing_table.id", fmt.Sprint(*listSubnetsOptions.RoutingTableID))
	}
	if listSubnetsOptions.RoutingTableName != nil {
		builder.AddQuery("routing_table.name", fmt.Sprint(*listSubnetsOptions.RoutingTableName))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSubnetCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateSubnet : Create a subnet
// This request creates a new subnet from a subnet prototype object. The prototype object is structured in the same way
// as a retrieved subnet, and contains the information necessary to create the new subnet. For this request to succeed,
// the prototype's CIDR block must not overlap with an existing subnet in the VPC.
func (vpc *VpcV1) CreateSubnet(createSubnetOptions *CreateSubnetOptions) (result *Subnet, response *core.DetailedResponse, err error) {
	return vpc.CreateSubnetWithContext(context.Background(), createSubnetOptions)
}

// CreateSubnetWithContext is an alternate form of the CreateSubnet method which supports a Context parameter
func (vpc *VpcV1) CreateSubnetWithContext(ctx context.Context, createSubnetOptions *CreateSubnetOptions) (result *Subnet, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createSubnetOptions, "createSubnetOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createSubnetOptions, "createSubnetOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createSubnetOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateSubnet")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(createSubnetOptions.SubnetPrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSubnet)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteSubnet : Delete a subnet
// This request deletes a subnet. This operation cannot be reversed. For this request to succeed, the subnet must not be
// referenced by any network interfaces, VPN gateways, or load balancers. A delete operation automatically detaches the
// subnet from any network ACLs, public gateways, or endpoint gateways. All flow log collectors with `auto_delete` set
// to `true` targeting the subnet or any resource in the subnet are automatically deleted.
func (vpc *VpcV1) DeleteSubnet(deleteSubnetOptions *DeleteSubnetOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteSubnetWithContext(context.Background(), deleteSubnetOptions)
}

// DeleteSubnetWithContext is an alternate form of the DeleteSubnet method which supports a Context parameter
func (vpc *VpcV1) DeleteSubnetWithContext(ctx context.Context, deleteSubnetOptions *DeleteSubnetOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteSubnetOptions, "deleteSubnetOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteSubnetOptions, "deleteSubnetOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteSubnetOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteSubnetOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteSubnet")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetSubnet : Retrieve a subnet
// This request retrieves a single subnet specified by the identifier in the URL.
func (vpc *VpcV1) GetSubnet(getSubnetOptions *GetSubnetOptions) (result *Subnet, response *core.DetailedResponse, err error) {
	return vpc.GetSubnetWithContext(context.Background(), getSubnetOptions)
}

// GetSubnetWithContext is an alternate form of the GetSubnet method which supports a Context parameter
func (vpc *VpcV1) GetSubnetWithContext(ctx context.Context, getSubnetOptions *GetSubnetOptions) (result *Subnet, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSubnetOptions, "getSubnetOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSubnetOptions, "getSubnetOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getSubnetOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSubnetOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetSubnet")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSubnet)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateSubnet : Update a subnet
// This request updates a subnet with the information in a provided subnet patch. The subnet patch object is structured
// in the same way as a retrieved subnet and contains only the information to be updated.
func (vpc *VpcV1) UpdateSubnet(updateSubnetOptions *UpdateSubnetOptions) (result *Subnet, response *core.DetailedResponse, err error) {
	return vpc.UpdateSubnetWithContext(context.Background(), updateSubnetOptions)
}

// UpdateSubnetWithContext is an alternate form of the UpdateSubnet method which supports a Context parameter
func (vpc *VpcV1) UpdateSubnetWithContext(ctx context.Context, updateSubnetOptions *UpdateSubnetOptions) (result *Subnet, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateSubnetOptions, "updateSubnetOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateSubnetOptions, "updateSubnetOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateSubnetOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateSubnetOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateSubnet")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateSubnetOptions.SubnetPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSubnet)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetSubnetNetworkACL : Retrieve a subnet's attached network ACL
// This request retrieves the network ACL attached to the subnet specified by the identifier in the URL.
func (vpc *VpcV1) GetSubnetNetworkACL(getSubnetNetworkACLOptions *GetSubnetNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	return vpc.GetSubnetNetworkACLWithContext(context.Background(), getSubnetNetworkACLOptions)
}

// GetSubnetNetworkACLWithContext is an alternate form of the GetSubnetNetworkACL method which supports a Context parameter
func (vpc *VpcV1) GetSubnetNetworkACLWithContext(ctx context.Context, getSubnetNetworkACLOptions *GetSubnetNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSubnetNetworkACLOptions, "getSubnetNetworkACLOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSubnetNetworkACLOptions, "getSubnetNetworkACLOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getSubnetNetworkACLOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets/{id}/network_acl`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSubnetNetworkACLOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetSubnetNetworkACL")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACL)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ReplaceSubnetNetworkACL : Attach a network ACL to a subnet
// This request attaches the network ACL, specified in the request body, to the subnet specified by the subnet
// identifier in the URL. This replaces the existing network ACL on the subnet.
func (vpc *VpcV1) ReplaceSubnetNetworkACL(replaceSubnetNetworkACLOptions *ReplaceSubnetNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	return vpc.ReplaceSubnetNetworkACLWithContext(context.Background(), replaceSubnetNetworkACLOptions)
}

// ReplaceSubnetNetworkACLWithContext is an alternate form of the ReplaceSubnetNetworkACL method which supports a Context parameter
func (vpc *VpcV1) ReplaceSubnetNetworkACLWithContext(ctx context.Context, replaceSubnetNetworkACLOptions *ReplaceSubnetNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(replaceSubnetNetworkACLOptions, "replaceSubnetNetworkACLOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(replaceSubnetNetworkACLOptions, "replaceSubnetNetworkACLOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *replaceSubnetNetworkACLOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets/{id}/network_acl`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range replaceSubnetNetworkACLOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ReplaceSubnetNetworkACL")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(replaceSubnetNetworkACLOptions.NetworkACLIdentity)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACL)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UnsetSubnetPublicGateway : Detach a public gateway from a subnet
// This request detaches the public gateway from the subnet specified by the subnet identifier in the URL.
func (vpc *VpcV1) UnsetSubnetPublicGateway(unsetSubnetPublicGatewayOptions *UnsetSubnetPublicGatewayOptions) (response *core.DetailedResponse, err error) {
	return vpc.UnsetSubnetPublicGatewayWithContext(context.Background(), unsetSubnetPublicGatewayOptions)
}

// UnsetSubnetPublicGatewayWithContext is an alternate form of the UnsetSubnetPublicGateway method which supports a Context parameter
func (vpc *VpcV1) UnsetSubnetPublicGatewayWithContext(ctx context.Context, unsetSubnetPublicGatewayOptions *UnsetSubnetPublicGatewayOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(unsetSubnetPublicGatewayOptions, "unsetSubnetPublicGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(unsetSubnetPublicGatewayOptions, "unsetSubnetPublicGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *unsetSubnetPublicGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets/{id}/public_gateway`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range unsetSubnetPublicGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UnsetSubnetPublicGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetSubnetPublicGateway : Retrieve a subnet's attached public gateway
// This request retrieves the public gateway attached to the subnet specified by the identifier in the URL.
func (vpc *VpcV1) GetSubnetPublicGateway(getSubnetPublicGatewayOptions *GetSubnetPublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	return vpc.GetSubnetPublicGatewayWithContext(context.Background(), getSubnetPublicGatewayOptions)
}

// GetSubnetPublicGatewayWithContext is an alternate form of the GetSubnetPublicGateway method which supports a Context parameter
func (vpc *VpcV1) GetSubnetPublicGatewayWithContext(ctx context.Context, getSubnetPublicGatewayOptions *GetSubnetPublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSubnetPublicGatewayOptions, "getSubnetPublicGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSubnetPublicGatewayOptions, "getSubnetPublicGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getSubnetPublicGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets/{id}/public_gateway`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSubnetPublicGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetSubnetPublicGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalPublicGateway)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// SetSubnetPublicGateway : Attach a public gateway to a subnet
// This request attaches the public gateway, specified in the request body, to the subnet specified by the subnet
// identifier in the URL. The public gateway must have the same VPC and zone as the subnet.
func (vpc *VpcV1) SetSubnetPublicGateway(setSubnetPublicGatewayOptions *SetSubnetPublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	return vpc.SetSubnetPublicGatewayWithContext(context.Background(), setSubnetPublicGatewayOptions)
}

// SetSubnetPublicGatewayWithContext is an alternate form of the SetSubnetPublicGateway method which supports a Context parameter
func (vpc *VpcV1) SetSubnetPublicGatewayWithContext(ctx context.Context, setSubnetPublicGatewayOptions *SetSubnetPublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(setSubnetPublicGatewayOptions, "setSubnetPublicGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(setSubnetPublicGatewayOptions, "setSubnetPublicGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *setSubnetPublicGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets/{id}/public_gateway`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range setSubnetPublicGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "SetSubnetPublicGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(setSubnetPublicGatewayOptions.PublicGatewayIdentity)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalPublicGateway)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetSubnetRoutingTable : Retrieve a subnet's attached routing table
// This request retrieves the routing table attached to the subnet specified by the identifier in the URL.
func (vpc *VpcV1) GetSubnetRoutingTable(getSubnetRoutingTableOptions *GetSubnetRoutingTableOptions) (result *RoutingTable, response *core.DetailedResponse, err error) {
	return vpc.GetSubnetRoutingTableWithContext(context.Background(), getSubnetRoutingTableOptions)
}

// GetSubnetRoutingTableWithContext is an alternate form of the GetSubnetRoutingTable method which supports a Context parameter
func (vpc *VpcV1) GetSubnetRoutingTableWithContext(ctx context.Context, getSubnetRoutingTableOptions *GetSubnetRoutingTableOptions) (result *RoutingTable, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSubnetRoutingTableOptions, "getSubnetRoutingTableOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSubnetRoutingTableOptions, "getSubnetRoutingTableOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getSubnetRoutingTableOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets/{id}/routing_table`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSubnetRoutingTableOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetSubnetRoutingTable")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRoutingTable)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ReplaceSubnetRoutingTable : Attach a routing table to a subnet
// This request attaches the routing table, specified in the request body, to the subnet specified by the subnet
// identifier in the URL. This replaces the existing routing table on the subnet.
//
// For this request to succeed, the routing table `route_direct_link_ingress`,
// `route_transit_gateway_ingress`, and `route_vpc_zone_ingress` properties must be `false`.
func (vpc *VpcV1) ReplaceSubnetRoutingTable(replaceSubnetRoutingTableOptions *ReplaceSubnetRoutingTableOptions) (result *RoutingTable, response *core.DetailedResponse, err error) {
	return vpc.ReplaceSubnetRoutingTableWithContext(context.Background(), replaceSubnetRoutingTableOptions)
}

// ReplaceSubnetRoutingTableWithContext is an alternate form of the ReplaceSubnetRoutingTable method which supports a Context parameter
func (vpc *VpcV1) ReplaceSubnetRoutingTableWithContext(ctx context.Context, replaceSubnetRoutingTableOptions *ReplaceSubnetRoutingTableOptions) (result *RoutingTable, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(replaceSubnetRoutingTableOptions, "replaceSubnetRoutingTableOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(replaceSubnetRoutingTableOptions, "replaceSubnetRoutingTableOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *replaceSubnetRoutingTableOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets/{id}/routing_table`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range replaceSubnetRoutingTableOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ReplaceSubnetRoutingTable")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(replaceSubnetRoutingTableOptions.RoutingTableIdentity)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRoutingTable)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListSubnetReservedIps : List all reserved IPs in a subnet
// This request lists reserved IPs in a subnet that are unbound or bound to an endpoint gateway.
func (vpc *VpcV1) ListSubnetReservedIps(listSubnetReservedIpsOptions *ListSubnetReservedIpsOptions) (result *ReservedIPCollection, response *core.DetailedResponse, err error) {
	return vpc.ListSubnetReservedIpsWithContext(context.Background(), listSubnetReservedIpsOptions)
}

// ListSubnetReservedIpsWithContext is an alternate form of the ListSubnetReservedIps method which supports a Context parameter
func (vpc *VpcV1) ListSubnetReservedIpsWithContext(ctx context.Context, listSubnetReservedIpsOptions *ListSubnetReservedIpsOptions) (result *ReservedIPCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listSubnetReservedIpsOptions, "listSubnetReservedIpsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listSubnetReservedIpsOptions, "listSubnetReservedIpsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"subnet_id": *listSubnetReservedIpsOptions.SubnetID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets/{subnet_id}/reserved_ips`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listSubnetReservedIpsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListSubnetReservedIps")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listSubnetReservedIpsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listSubnetReservedIpsOptions.Start))
	}
	if listSubnetReservedIpsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listSubnetReservedIpsOptions.Limit))
	}
	if listSubnetReservedIpsOptions.Sort != nil {
		builder.AddQuery("sort", fmt.Sprint(*listSubnetReservedIpsOptions.Sort))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalReservedIPCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateSubnetReservedIP : Reserve an IP in a subnet
// This request reserves a system-selected IP address in a subnet.
func (vpc *VpcV1) CreateSubnetReservedIP(createSubnetReservedIPOptions *CreateSubnetReservedIPOptions) (result *ReservedIP, response *core.DetailedResponse, err error) {
	return vpc.CreateSubnetReservedIPWithContext(context.Background(), createSubnetReservedIPOptions)
}

// CreateSubnetReservedIPWithContext is an alternate form of the CreateSubnetReservedIP method which supports a Context parameter
func (vpc *VpcV1) CreateSubnetReservedIPWithContext(ctx context.Context, createSubnetReservedIPOptions *CreateSubnetReservedIPOptions) (result *ReservedIP, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createSubnetReservedIPOptions, "createSubnetReservedIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createSubnetReservedIPOptions, "createSubnetReservedIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"subnet_id": *createSubnetReservedIPOptions.SubnetID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets/{subnet_id}/reserved_ips`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createSubnetReservedIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateSubnetReservedIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createSubnetReservedIPOptions.AutoDelete != nil {
		body["auto_delete"] = createSubnetReservedIPOptions.AutoDelete
	}
	if createSubnetReservedIPOptions.Name != nil {
		body["name"] = createSubnetReservedIPOptions.Name
	}
	if createSubnetReservedIPOptions.Target != nil {
		body["target"] = createSubnetReservedIPOptions.Target
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalReservedIP)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteSubnetReservedIP : Release a reserved IP
// This request releases a reserved IP. This operation cannot be reversed.
func (vpc *VpcV1) DeleteSubnetReservedIP(deleteSubnetReservedIPOptions *DeleteSubnetReservedIPOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteSubnetReservedIPWithContext(context.Background(), deleteSubnetReservedIPOptions)
}

// DeleteSubnetReservedIPWithContext is an alternate form of the DeleteSubnetReservedIP method which supports a Context parameter
func (vpc *VpcV1) DeleteSubnetReservedIPWithContext(ctx context.Context, deleteSubnetReservedIPOptions *DeleteSubnetReservedIPOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteSubnetReservedIPOptions, "deleteSubnetReservedIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteSubnetReservedIPOptions, "deleteSubnetReservedIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"subnet_id": *deleteSubnetReservedIPOptions.SubnetID,
		"id":        *deleteSubnetReservedIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets/{subnet_id}/reserved_ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteSubnetReservedIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteSubnetReservedIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetSubnetReservedIP : Retrieve a reserved IP
// This request retrieves a single reserved IP specified by the identifier in the URL.
func (vpc *VpcV1) GetSubnetReservedIP(getSubnetReservedIPOptions *GetSubnetReservedIPOptions) (result *ReservedIP, response *core.DetailedResponse, err error) {
	return vpc.GetSubnetReservedIPWithContext(context.Background(), getSubnetReservedIPOptions)
}

// GetSubnetReservedIPWithContext is an alternate form of the GetSubnetReservedIP method which supports a Context parameter
func (vpc *VpcV1) GetSubnetReservedIPWithContext(ctx context.Context, getSubnetReservedIPOptions *GetSubnetReservedIPOptions) (result *ReservedIP, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSubnetReservedIPOptions, "getSubnetReservedIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSubnetReservedIPOptions, "getSubnetReservedIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"subnet_id": *getSubnetReservedIPOptions.SubnetID,
		"id":        *getSubnetReservedIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets/{subnet_id}/reserved_ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSubnetReservedIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetSubnetReservedIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalReservedIP)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateSubnetReservedIP : Update a reserved IP
// This request updates a reserved IP with the information in a provided reserved IP patch. The reserved IP patch object
// is structured in the same way as a retrieved reserved IP and contains only the information to be updated.
func (vpc *VpcV1) UpdateSubnetReservedIP(updateSubnetReservedIPOptions *UpdateSubnetReservedIPOptions) (result *ReservedIP, response *core.DetailedResponse, err error) {
	return vpc.UpdateSubnetReservedIPWithContext(context.Background(), updateSubnetReservedIPOptions)
}

// UpdateSubnetReservedIPWithContext is an alternate form of the UpdateSubnetReservedIP method which supports a Context parameter
func (vpc *VpcV1) UpdateSubnetReservedIPWithContext(ctx context.Context, updateSubnetReservedIPOptions *UpdateSubnetReservedIPOptions) (result *ReservedIP, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateSubnetReservedIPOptions, "updateSubnetReservedIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateSubnetReservedIPOptions, "updateSubnetReservedIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"subnet_id": *updateSubnetReservedIPOptions.SubnetID,
		"id":        *updateSubnetReservedIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/subnets/{subnet_id}/reserved_ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateSubnetReservedIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateSubnetReservedIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateSubnetReservedIPOptions.ReservedIPPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalReservedIP)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListImages : List all images
// This request lists all provisionable images available in the region. An image provides source data for a volume.
// Images are either system-provided, or created from another source, such as importing from object storage.
//
// The images will be sorted by their `created_at` property values, with the newest first. Images with identical
// `created_at` values will be secondarily sorted by ascending `id` property values.
func (vpc *VpcV1) ListImages(listImagesOptions *ListImagesOptions) (result *ImageCollection, response *core.DetailedResponse, err error) {
	return vpc.ListImagesWithContext(context.Background(), listImagesOptions)
}

// ListImagesWithContext is an alternate form of the ListImages method which supports a Context parameter
func (vpc *VpcV1) ListImagesWithContext(ctx context.Context, listImagesOptions *ListImagesOptions) (result *ImageCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listImagesOptions, "listImagesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/images`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listImagesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListImages")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listImagesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listImagesOptions.Start))
	}
	if listImagesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listImagesOptions.Limit))
	}
	if listImagesOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listImagesOptions.ResourceGroupID))
	}
	if listImagesOptions.Name != nil {
		builder.AddQuery("name", fmt.Sprint(*listImagesOptions.Name))
	}
	if listImagesOptions.Visibility != nil {
		builder.AddQuery("visibility", fmt.Sprint(*listImagesOptions.Visibility))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalImageCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateImage : Create an image
// This request creates a new image from an image prototype object. The prototype object is structured in the same way
// as a retrieved image, and contains the information necessary to create the new image. If an image is being imported,
// a URL to the image file on object storage must be specified. If an image is being created from an existing volume,
// that volume must be specified.
func (vpc *VpcV1) CreateImage(createImageOptions *CreateImageOptions) (result *Image, response *core.DetailedResponse, err error) {
	return vpc.CreateImageWithContext(context.Background(), createImageOptions)
}

// CreateImageWithContext is an alternate form of the CreateImage method which supports a Context parameter
func (vpc *VpcV1) CreateImageWithContext(ctx context.Context, createImageOptions *CreateImageOptions) (result *Image, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createImageOptions, "createImageOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createImageOptions, "createImageOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/images`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createImageOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateImage")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(createImageOptions.ImagePrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalImage)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteImage : Delete an image
// This request deletes an image. This operation cannot be reversed. A system-provided image is not allowed to be
// deleted. Additionally, an image cannot be deleted if it has a
// `status` of `pending`, `tentative`, or `deleting`.
func (vpc *VpcV1) DeleteImage(deleteImageOptions *DeleteImageOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteImageWithContext(context.Background(), deleteImageOptions)
}

// DeleteImageWithContext is an alternate form of the DeleteImage method which supports a Context parameter
func (vpc *VpcV1) DeleteImageWithContext(ctx context.Context, deleteImageOptions *DeleteImageOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteImageOptions, "deleteImageOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteImageOptions, "deleteImageOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteImageOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/images/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteImageOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteImage")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetImage : Retrieve an image
// This request retrieves a single image specified by the identifier in the URL.
func (vpc *VpcV1) GetImage(getImageOptions *GetImageOptions) (result *Image, response *core.DetailedResponse, err error) {
	return vpc.GetImageWithContext(context.Background(), getImageOptions)
}

// GetImageWithContext is an alternate form of the GetImage method which supports a Context parameter
func (vpc *VpcV1) GetImageWithContext(ctx context.Context, getImageOptions *GetImageOptions) (result *Image, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getImageOptions, "getImageOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getImageOptions, "getImageOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getImageOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/images/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getImageOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetImage")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalImage)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateImage : Update an image
// This request updates an image with the information in a provided image patch. The image patch object is structured in
// the same way as a retrieved image and contains only the information to be updated. A system-provided image is not
// allowed to be updated. An image with a `status` of `deleting` cannot be updated.
func (vpc *VpcV1) UpdateImage(updateImageOptions *UpdateImageOptions) (result *Image, response *core.DetailedResponse, err error) {
	return vpc.UpdateImageWithContext(context.Background(), updateImageOptions)
}

// UpdateImageWithContext is an alternate form of the UpdateImage method which supports a Context parameter
func (vpc *VpcV1) UpdateImageWithContext(ctx context.Context, updateImageOptions *UpdateImageOptions) (result *Image, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateImageOptions, "updateImageOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateImageOptions, "updateImageOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateImageOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/images/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateImageOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateImage")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateImageOptions.ImagePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalImage)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListOperatingSystems : List all operating systems
// This request lists all operating systems in the region.
func (vpc *VpcV1) ListOperatingSystems(listOperatingSystemsOptions *ListOperatingSystemsOptions) (result *OperatingSystemCollection, response *core.DetailedResponse, err error) {
	return vpc.ListOperatingSystemsWithContext(context.Background(), listOperatingSystemsOptions)
}

// ListOperatingSystemsWithContext is an alternate form of the ListOperatingSystems method which supports a Context parameter
func (vpc *VpcV1) ListOperatingSystemsWithContext(ctx context.Context, listOperatingSystemsOptions *ListOperatingSystemsOptions) (result *OperatingSystemCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listOperatingSystemsOptions, "listOperatingSystemsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/operating_systems`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listOperatingSystemsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListOperatingSystems")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listOperatingSystemsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listOperatingSystemsOptions.Start))
	}
	if listOperatingSystemsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listOperatingSystemsOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalOperatingSystemCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetOperatingSystem : Retrieve an operating system
// This request retrieves a single operating system specified by the name in the URL.
func (vpc *VpcV1) GetOperatingSystem(getOperatingSystemOptions *GetOperatingSystemOptions) (result *OperatingSystem, response *core.DetailedResponse, err error) {
	return vpc.GetOperatingSystemWithContext(context.Background(), getOperatingSystemOptions)
}

// GetOperatingSystemWithContext is an alternate form of the GetOperatingSystem method which supports a Context parameter
func (vpc *VpcV1) GetOperatingSystemWithContext(ctx context.Context, getOperatingSystemOptions *GetOperatingSystemOptions) (result *OperatingSystem, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getOperatingSystemOptions, "getOperatingSystemOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getOperatingSystemOptions, "getOperatingSystemOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"name": *getOperatingSystemOptions.Name,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/operating_systems/{name}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getOperatingSystemOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetOperatingSystem")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalOperatingSystem)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListKeys : List all keys
// This request lists all keys in the region. A key contains a public SSH key which may be installed on instances when
// they are created. Private keys are not stored.
func (vpc *VpcV1) ListKeys(listKeysOptions *ListKeysOptions) (result *KeyCollection, response *core.DetailedResponse, err error) {
	return vpc.ListKeysWithContext(context.Background(), listKeysOptions)
}

// ListKeysWithContext is an alternate form of the ListKeys method which supports a Context parameter
func (vpc *VpcV1) ListKeysWithContext(ctx context.Context, listKeysOptions *ListKeysOptions) (result *KeyCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listKeysOptions, "listKeysOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/keys`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listKeysOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListKeys")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listKeysOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listKeysOptions.ResourceGroupID))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalKeyCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateKey : Create a key
// This request creates a new SSH key from an key prototype object. The prototype object is structured in the same way
// as a retrieved key, and contains the information necessary to create the new key. The public key value must be
// provided.
func (vpc *VpcV1) CreateKey(createKeyOptions *CreateKeyOptions) (result *Key, response *core.DetailedResponse, err error) {
	return vpc.CreateKeyWithContext(context.Background(), createKeyOptions)
}

// CreateKeyWithContext is an alternate form of the CreateKey method which supports a Context parameter
func (vpc *VpcV1) CreateKeyWithContext(ctx context.Context, createKeyOptions *CreateKeyOptions) (result *Key, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createKeyOptions, "createKeyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createKeyOptions, "createKeyOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/keys`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createKeyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateKey")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createKeyOptions.PublicKey != nil {
		body["public_key"] = createKeyOptions.PublicKey
	}
	if createKeyOptions.Name != nil {
		body["name"] = createKeyOptions.Name
	}
	if createKeyOptions.ResourceGroup != nil {
		body["resource_group"] = createKeyOptions.ResourceGroup
	}
	if createKeyOptions.Type != nil {
		body["type"] = createKeyOptions.Type
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalKey)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteKey : Delete a key
// This request deletes a key. This operation cannot be reversed.
func (vpc *VpcV1) DeleteKey(deleteKeyOptions *DeleteKeyOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteKeyWithContext(context.Background(), deleteKeyOptions)
}

// DeleteKeyWithContext is an alternate form of the DeleteKey method which supports a Context parameter
func (vpc *VpcV1) DeleteKeyWithContext(ctx context.Context, deleteKeyOptions *DeleteKeyOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteKeyOptions, "deleteKeyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteKeyOptions, "deleteKeyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteKeyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/keys/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteKeyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteKey")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetKey : Retrieve a key
// This request retrieves a single key specified by the identifier in the URL.
func (vpc *VpcV1) GetKey(getKeyOptions *GetKeyOptions) (result *Key, response *core.DetailedResponse, err error) {
	return vpc.GetKeyWithContext(context.Background(), getKeyOptions)
}

// GetKeyWithContext is an alternate form of the GetKey method which supports a Context parameter
func (vpc *VpcV1) GetKeyWithContext(ctx context.Context, getKeyOptions *GetKeyOptions) (result *Key, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getKeyOptions, "getKeyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getKeyOptions, "getKeyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getKeyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/keys/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getKeyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetKey")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalKey)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateKey : Update a key
// This request updates a key's name.
func (vpc *VpcV1) UpdateKey(updateKeyOptions *UpdateKeyOptions) (result *Key, response *core.DetailedResponse, err error) {
	return vpc.UpdateKeyWithContext(context.Background(), updateKeyOptions)
}

// UpdateKeyWithContext is an alternate form of the UpdateKey method which supports a Context parameter
func (vpc *VpcV1) UpdateKeyWithContext(ctx context.Context, updateKeyOptions *UpdateKeyOptions) (result *Key, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateKeyOptions, "updateKeyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateKeyOptions, "updateKeyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateKeyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/keys/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateKeyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateKey")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateKeyOptions.KeyPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalKey)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListInstanceProfiles : List all instance profiles
// This request lists provisionable instance profiles in the region. An instance profile specifies the performance
// characteristics and pricing model for an instance.
func (vpc *VpcV1) ListInstanceProfiles(listInstanceProfilesOptions *ListInstanceProfilesOptions) (result *InstanceProfileCollection, response *core.DetailedResponse, err error) {
	return vpc.ListInstanceProfilesWithContext(context.Background(), listInstanceProfilesOptions)
}

// ListInstanceProfilesWithContext is an alternate form of the ListInstanceProfiles method which supports a Context parameter
func (vpc *VpcV1) ListInstanceProfilesWithContext(ctx context.Context, listInstanceProfilesOptions *ListInstanceProfilesOptions) (result *InstanceProfileCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listInstanceProfilesOptions, "listInstanceProfilesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance/profiles`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstanceProfilesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListInstanceProfiles")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceProfileCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetInstanceProfile : Retrieve an instance profile
// This request retrieves a single instance profile specified by the name in the URL.
func (vpc *VpcV1) GetInstanceProfile(getInstanceProfileOptions *GetInstanceProfileOptions) (result *InstanceProfile, response *core.DetailedResponse, err error) {
	return vpc.GetInstanceProfileWithContext(context.Background(), getInstanceProfileOptions)
}

// GetInstanceProfileWithContext is an alternate form of the GetInstanceProfile method which supports a Context parameter
func (vpc *VpcV1) GetInstanceProfileWithContext(ctx context.Context, getInstanceProfileOptions *GetInstanceProfileOptions) (result *InstanceProfile, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceProfileOptions, "getInstanceProfileOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceProfileOptions, "getInstanceProfileOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"name": *getInstanceProfileOptions.Name,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance/profiles/{name}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceProfileOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetInstanceProfile")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceProfile)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListInstanceTemplates : List all instance templates
// This request lists all instance templates in the region.
func (vpc *VpcV1) ListInstanceTemplates(listInstanceTemplatesOptions *ListInstanceTemplatesOptions) (result *InstanceTemplateCollection, response *core.DetailedResponse, err error) {
	return vpc.ListInstanceTemplatesWithContext(context.Background(), listInstanceTemplatesOptions)
}

// ListInstanceTemplatesWithContext is an alternate form of the ListInstanceTemplates method which supports a Context parameter
func (vpc *VpcV1) ListInstanceTemplatesWithContext(ctx context.Context, listInstanceTemplatesOptions *ListInstanceTemplatesOptions) (result *InstanceTemplateCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listInstanceTemplatesOptions, "listInstanceTemplatesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance/templates`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstanceTemplatesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListInstanceTemplates")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceTemplateCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateInstanceTemplate : Create an instance template
// This request creates a new instance template.
func (vpc *VpcV1) CreateInstanceTemplate(createInstanceTemplateOptions *CreateInstanceTemplateOptions) (result InstanceTemplateIntf, response *core.DetailedResponse, err error) {
	return vpc.CreateInstanceTemplateWithContext(context.Background(), createInstanceTemplateOptions)
}

// CreateInstanceTemplateWithContext is an alternate form of the CreateInstanceTemplate method which supports a Context parameter
func (vpc *VpcV1) CreateInstanceTemplateWithContext(ctx context.Context, createInstanceTemplateOptions *CreateInstanceTemplateOptions) (result InstanceTemplateIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createInstanceTemplateOptions, "createInstanceTemplateOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createInstanceTemplateOptions, "createInstanceTemplateOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance/templates`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createInstanceTemplateOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateInstanceTemplate")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(createInstanceTemplateOptions.InstanceTemplatePrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceTemplate)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteInstanceTemplate : Delete an instance template
// This request deletes the instance template. This operation cannot be reversed.
func (vpc *VpcV1) DeleteInstanceTemplate(deleteInstanceTemplateOptions *DeleteInstanceTemplateOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteInstanceTemplateWithContext(context.Background(), deleteInstanceTemplateOptions)
}

// DeleteInstanceTemplateWithContext is an alternate form of the DeleteInstanceTemplate method which supports a Context parameter
func (vpc *VpcV1) DeleteInstanceTemplateWithContext(ctx context.Context, deleteInstanceTemplateOptions *DeleteInstanceTemplateOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteInstanceTemplateOptions, "deleteInstanceTemplateOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteInstanceTemplateOptions, "deleteInstanceTemplateOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteInstanceTemplateOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance/templates/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteInstanceTemplateOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteInstanceTemplate")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetInstanceTemplate : Retrieve an instance template
// This request retrieves a single instance template specified by the identifier in the URL.
func (vpc *VpcV1) GetInstanceTemplate(getInstanceTemplateOptions *GetInstanceTemplateOptions) (result InstanceTemplateIntf, response *core.DetailedResponse, err error) {
	return vpc.GetInstanceTemplateWithContext(context.Background(), getInstanceTemplateOptions)
}

// GetInstanceTemplateWithContext is an alternate form of the GetInstanceTemplate method which supports a Context parameter
func (vpc *VpcV1) GetInstanceTemplateWithContext(ctx context.Context, getInstanceTemplateOptions *GetInstanceTemplateOptions) (result InstanceTemplateIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceTemplateOptions, "getInstanceTemplateOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceTemplateOptions, "getInstanceTemplateOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getInstanceTemplateOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance/templates/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceTemplateOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetInstanceTemplate")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceTemplate)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateInstanceTemplate : Update an instance template
// This request updates an instance template with the information provided in the instance template patch. The instance
// template patch object is structured in the same way as a retrieved instance template and contains only the
// information to be updated.
func (vpc *VpcV1) UpdateInstanceTemplate(updateInstanceTemplateOptions *UpdateInstanceTemplateOptions) (result InstanceTemplateIntf, response *core.DetailedResponse, err error) {
	return vpc.UpdateInstanceTemplateWithContext(context.Background(), updateInstanceTemplateOptions)
}

// UpdateInstanceTemplateWithContext is an alternate form of the UpdateInstanceTemplate method which supports a Context parameter
func (vpc *VpcV1) UpdateInstanceTemplateWithContext(ctx context.Context, updateInstanceTemplateOptions *UpdateInstanceTemplateOptions) (result InstanceTemplateIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateInstanceTemplateOptions, "updateInstanceTemplateOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateInstanceTemplateOptions, "updateInstanceTemplateOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateInstanceTemplateOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance/templates/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateInstanceTemplateOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateInstanceTemplate")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateInstanceTemplateOptions.InstanceTemplatePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceTemplate)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListInstances : List all instances
// This request lists all instances in the region.
func (vpc *VpcV1) ListInstances(listInstancesOptions *ListInstancesOptions) (result *InstanceCollection, response *core.DetailedResponse, err error) {
	return vpc.ListInstancesWithContext(context.Background(), listInstancesOptions)
}

// ListInstancesWithContext is an alternate form of the ListInstances method which supports a Context parameter
func (vpc *VpcV1) ListInstancesWithContext(ctx context.Context, listInstancesOptions *ListInstancesOptions) (result *InstanceCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listInstancesOptions, "listInstancesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstancesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListInstances")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listInstancesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listInstancesOptions.Start))
	}
	if listInstancesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listInstancesOptions.Limit))
	}
	if listInstancesOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listInstancesOptions.ResourceGroupID))
	}
	if listInstancesOptions.Name != nil {
		builder.AddQuery("name", fmt.Sprint(*listInstancesOptions.Name))
	}
	if listInstancesOptions.VPCID != nil {
		builder.AddQuery("vpc.id", fmt.Sprint(*listInstancesOptions.VPCID))
	}
	if listInstancesOptions.VPCCRN != nil {
		builder.AddQuery("vpc.crn", fmt.Sprint(*listInstancesOptions.VPCCRN))
	}
	if listInstancesOptions.VPCName != nil {
		builder.AddQuery("vpc.name", fmt.Sprint(*listInstancesOptions.VPCName))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateInstance : Create an instance
// This request provisions a new instance from an instance prototype object. The prototype object is structured in the
// same way as a retrieved instance, and contains the information necessary to provision the new instance. The instance
// is automatically started.
func (vpc *VpcV1) CreateInstance(createInstanceOptions *CreateInstanceOptions) (result *Instance, response *core.DetailedResponse, err error) {
	return vpc.CreateInstanceWithContext(context.Background(), createInstanceOptions)
}

// CreateInstanceWithContext is an alternate form of the CreateInstance method which supports a Context parameter
func (vpc *VpcV1) CreateInstanceWithContext(ctx context.Context, createInstanceOptions *CreateInstanceOptions) (result *Instance, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createInstanceOptions, "createInstanceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createInstanceOptions, "createInstanceOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createInstanceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateInstance")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(createInstanceOptions.InstancePrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstance)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteInstance : Delete an instance
// This request deletes an instance. This operation cannot be reversed. Any floating IPs associated with the instance's
// network interfaces are implicitly disassociated. All flow log collectors with `auto_delete` set to `true` targeting
// the instance and/or the instance's network interfaces are automatically deleted.
func (vpc *VpcV1) DeleteInstance(deleteInstanceOptions *DeleteInstanceOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteInstanceWithContext(context.Background(), deleteInstanceOptions)
}

// DeleteInstanceWithContext is an alternate form of the DeleteInstance method which supports a Context parameter
func (vpc *VpcV1) DeleteInstanceWithContext(ctx context.Context, deleteInstanceOptions *DeleteInstanceOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteInstanceOptions, "deleteInstanceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteInstanceOptions, "deleteInstanceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteInstanceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteInstanceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteInstance")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetInstance : Retrieve an instance
// This request retrieves a single instance specified by the identifier in the URL.
func (vpc *VpcV1) GetInstance(getInstanceOptions *GetInstanceOptions) (result *Instance, response *core.DetailedResponse, err error) {
	return vpc.GetInstanceWithContext(context.Background(), getInstanceOptions)
}

// GetInstanceWithContext is an alternate form of the GetInstance method which supports a Context parameter
func (vpc *VpcV1) GetInstanceWithContext(ctx context.Context, getInstanceOptions *GetInstanceOptions) (result *Instance, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceOptions, "getInstanceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceOptions, "getInstanceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getInstanceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetInstance")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstance)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateInstance : Update an instance
// This request updates an instance with the information in a provided instance patch. The instance patch object is
// structured in the same way as a retrieved instance and contains only the information to be updated.
func (vpc *VpcV1) UpdateInstance(updateInstanceOptions *UpdateInstanceOptions) (result *Instance, response *core.DetailedResponse, err error) {
	return vpc.UpdateInstanceWithContext(context.Background(), updateInstanceOptions)
}

// UpdateInstanceWithContext is an alternate form of the UpdateInstance method which supports a Context parameter
func (vpc *VpcV1) UpdateInstanceWithContext(ctx context.Context, updateInstanceOptions *UpdateInstanceOptions) (result *Instance, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateInstanceOptions, "updateInstanceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateInstanceOptions, "updateInstanceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateInstanceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateInstanceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateInstance")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateInstanceOptions.InstancePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstance)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetInstanceInitialization : Retrieve initialization configuration for an instance
// This request retrieves configuration variables used to initialize the instance, such as SSH keys and the Windows
// administrator password.
func (vpc *VpcV1) GetInstanceInitialization(getInstanceInitializationOptions *GetInstanceInitializationOptions) (result *InstanceInitialization, response *core.DetailedResponse, err error) {
	return vpc.GetInstanceInitializationWithContext(context.Background(), getInstanceInitializationOptions)
}

// GetInstanceInitializationWithContext is an alternate form of the GetInstanceInitialization method which supports a Context parameter
func (vpc *VpcV1) GetInstanceInitializationWithContext(ctx context.Context, getInstanceInitializationOptions *GetInstanceInitializationOptions) (result *InstanceInitialization, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceInitializationOptions, "getInstanceInitializationOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceInitializationOptions, "getInstanceInitializationOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getInstanceInitializationOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{id}/initialization`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceInitializationOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetInstanceInitialization")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceInitialization)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateInstanceAction : Create an instance action
// This request creates a new action which will be queued up to run as soon as any pending or running actions have
// completed.
func (vpc *VpcV1) CreateInstanceAction(createInstanceActionOptions *CreateInstanceActionOptions) (result *InstanceAction, response *core.DetailedResponse, err error) {
	return vpc.CreateInstanceActionWithContext(context.Background(), createInstanceActionOptions)
}

// CreateInstanceActionWithContext is an alternate form of the CreateInstanceAction method which supports a Context parameter
func (vpc *VpcV1) CreateInstanceActionWithContext(ctx context.Context, createInstanceActionOptions *CreateInstanceActionOptions) (result *InstanceAction, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createInstanceActionOptions, "createInstanceActionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createInstanceActionOptions, "createInstanceActionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *createInstanceActionOptions.InstanceID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/actions`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createInstanceActionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateInstanceAction")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createInstanceActionOptions.Type != nil {
		body["type"] = createInstanceActionOptions.Type
	}
	if createInstanceActionOptions.Force != nil {
		body["force"] = createInstanceActionOptions.Force
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceAction)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateInstanceConsoleAccessToken : Create a console access token for an instance
// This request creates a new single-use console access token for an instance. All console configuration is provided at
// token create time, and the token is subsequently used in the `access_token` query parameter for the WebSocket
// request.  The access token is only valid for a short period of time, and a maximum of one token is valid for a given
// instance at a time.
func (vpc *VpcV1) CreateInstanceConsoleAccessToken(createInstanceConsoleAccessTokenOptions *CreateInstanceConsoleAccessTokenOptions) (result *InstanceConsoleAccessToken, response *core.DetailedResponse, err error) {
	return vpc.CreateInstanceConsoleAccessTokenWithContext(context.Background(), createInstanceConsoleAccessTokenOptions)
}

// CreateInstanceConsoleAccessTokenWithContext is an alternate form of the CreateInstanceConsoleAccessToken method which supports a Context parameter
func (vpc *VpcV1) CreateInstanceConsoleAccessTokenWithContext(ctx context.Context, createInstanceConsoleAccessTokenOptions *CreateInstanceConsoleAccessTokenOptions) (result *InstanceConsoleAccessToken, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createInstanceConsoleAccessTokenOptions, "createInstanceConsoleAccessTokenOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createInstanceConsoleAccessTokenOptions, "createInstanceConsoleAccessTokenOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *createInstanceConsoleAccessTokenOptions.InstanceID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/console_access_token`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createInstanceConsoleAccessTokenOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateInstanceConsoleAccessToken")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createInstanceConsoleAccessTokenOptions.ConsoleType != nil {
		body["console_type"] = createInstanceConsoleAccessTokenOptions.ConsoleType
	}
	if createInstanceConsoleAccessTokenOptions.Force != nil {
		body["force"] = createInstanceConsoleAccessTokenOptions.Force
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceConsoleAccessToken)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListInstanceDisks : List all disks on an instance
// This request lists all disks on an instance.  A disk is a block device that is locally attached to the instance's
// physical host and is also referred to as instance storage. By default, the listed disks are sorted by their
// `created_at` property values, with the newest disk first.
func (vpc *VpcV1) ListInstanceDisks(listInstanceDisksOptions *ListInstanceDisksOptions) (result *InstanceDiskCollection, response *core.DetailedResponse, err error) {
	return vpc.ListInstanceDisksWithContext(context.Background(), listInstanceDisksOptions)
}

// ListInstanceDisksWithContext is an alternate form of the ListInstanceDisks method which supports a Context parameter
func (vpc *VpcV1) ListInstanceDisksWithContext(ctx context.Context, listInstanceDisksOptions *ListInstanceDisksOptions) (result *InstanceDiskCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listInstanceDisksOptions, "listInstanceDisksOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listInstanceDisksOptions, "listInstanceDisksOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *listInstanceDisksOptions.InstanceID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/disks`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstanceDisksOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListInstanceDisks")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceDiskCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetInstanceDisk : Retrieve an instance disk
// This request retrieves a single instance disk specified by the identifier in the URL.
func (vpc *VpcV1) GetInstanceDisk(getInstanceDiskOptions *GetInstanceDiskOptions) (result *InstanceDisk, response *core.DetailedResponse, err error) {
	return vpc.GetInstanceDiskWithContext(context.Background(), getInstanceDiskOptions)
}

// GetInstanceDiskWithContext is an alternate form of the GetInstanceDisk method which supports a Context parameter
func (vpc *VpcV1) GetInstanceDiskWithContext(ctx context.Context, getInstanceDiskOptions *GetInstanceDiskOptions) (result *InstanceDisk, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceDiskOptions, "getInstanceDiskOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceDiskOptions, "getInstanceDiskOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *getInstanceDiskOptions.InstanceID,
		"id":          *getInstanceDiskOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/disks/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceDiskOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetInstanceDisk")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceDisk)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateInstanceDisk : Update an instance disk
// This request updates the instance disk with the information in a provided patch.
func (vpc *VpcV1) UpdateInstanceDisk(updateInstanceDiskOptions *UpdateInstanceDiskOptions) (result *InstanceDisk, response *core.DetailedResponse, err error) {
	return vpc.UpdateInstanceDiskWithContext(context.Background(), updateInstanceDiskOptions)
}

// UpdateInstanceDiskWithContext is an alternate form of the UpdateInstanceDisk method which supports a Context parameter
func (vpc *VpcV1) UpdateInstanceDiskWithContext(ctx context.Context, updateInstanceDiskOptions *UpdateInstanceDiskOptions) (result *InstanceDisk, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateInstanceDiskOptions, "updateInstanceDiskOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateInstanceDiskOptions, "updateInstanceDiskOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *updateInstanceDiskOptions.InstanceID,
		"id":          *updateInstanceDiskOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/disks/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateInstanceDiskOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateInstanceDisk")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateInstanceDiskOptions.InstanceDiskPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceDisk)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListInstanceNetworkInterfaces : List all network interfaces on an instance
// This request lists all network interfaces on an instance. A network interface is an abstract representation of a
// network interface card and connects an instance to a subnet. While each network interface can attach to only one
// subnet, multiple network interfaces can be created to attach to multiple subnets. Multiple interfaces may also attach
// to the same subnet.
func (vpc *VpcV1) ListInstanceNetworkInterfaces(listInstanceNetworkInterfacesOptions *ListInstanceNetworkInterfacesOptions) (result *NetworkInterfaceUnpaginatedCollection, response *core.DetailedResponse, err error) {
	return vpc.ListInstanceNetworkInterfacesWithContext(context.Background(), listInstanceNetworkInterfacesOptions)
}

// ListInstanceNetworkInterfacesWithContext is an alternate form of the ListInstanceNetworkInterfaces method which supports a Context parameter
func (vpc *VpcV1) ListInstanceNetworkInterfacesWithContext(ctx context.Context, listInstanceNetworkInterfacesOptions *ListInstanceNetworkInterfacesOptions) (result *NetworkInterfaceUnpaginatedCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listInstanceNetworkInterfacesOptions, "listInstanceNetworkInterfacesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listInstanceNetworkInterfacesOptions, "listInstanceNetworkInterfacesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *listInstanceNetworkInterfacesOptions.InstanceID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/network_interfaces`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstanceNetworkInterfacesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListInstanceNetworkInterfaces")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkInterfaceUnpaginatedCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateInstanceNetworkInterface : Create a network interface on an instance
// This request creates a new network interface from a network interface prototype object. The prototype object is
// structured in the same way as a retrieved network interface, and contains the information necessary to create the new
// network interface. Any subnet in the instance's VPC may be specified, even if it is already attached to another
// network interface. Addresses on the network interface must be within the specified subnet's CIDR blocks.
func (vpc *VpcV1) CreateInstanceNetworkInterface(createInstanceNetworkInterfaceOptions *CreateInstanceNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	return vpc.CreateInstanceNetworkInterfaceWithContext(context.Background(), createInstanceNetworkInterfaceOptions)
}

// CreateInstanceNetworkInterfaceWithContext is an alternate form of the CreateInstanceNetworkInterface method which supports a Context parameter
func (vpc *VpcV1) CreateInstanceNetworkInterfaceWithContext(ctx context.Context, createInstanceNetworkInterfaceOptions *CreateInstanceNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createInstanceNetworkInterfaceOptions, "createInstanceNetworkInterfaceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createInstanceNetworkInterfaceOptions, "createInstanceNetworkInterfaceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *createInstanceNetworkInterfaceOptions.InstanceID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/network_interfaces`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createInstanceNetworkInterfaceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateInstanceNetworkInterface")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createInstanceNetworkInterfaceOptions.Subnet != nil {
		body["subnet"] = createInstanceNetworkInterfaceOptions.Subnet
	}
	if createInstanceNetworkInterfaceOptions.AllowIPSpoofing != nil {
		body["allow_ip_spoofing"] = createInstanceNetworkInterfaceOptions.AllowIPSpoofing
	}
	if createInstanceNetworkInterfaceOptions.Name != nil {
		body["name"] = createInstanceNetworkInterfaceOptions.Name
	}
	if createInstanceNetworkInterfaceOptions.PrimaryIpv4Address != nil {
		body["primary_ipv4_address"] = createInstanceNetworkInterfaceOptions.PrimaryIpv4Address
	}
	if createInstanceNetworkInterfaceOptions.SecurityGroups != nil {
		body["security_groups"] = createInstanceNetworkInterfaceOptions.SecurityGroups
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkInterface)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteInstanceNetworkInterface : Delete a network interface
// This request deletes a network interface. This operation cannot be reversed. Any floating IPs associated with the
// network interface are implicitly disassociated. All flow log collectors with `auto_delete` set to `true` targeting
// the network interface are automatically deleted. The primary network interface is not allowed to be deleted.
func (vpc *VpcV1) DeleteInstanceNetworkInterface(deleteInstanceNetworkInterfaceOptions *DeleteInstanceNetworkInterfaceOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteInstanceNetworkInterfaceWithContext(context.Background(), deleteInstanceNetworkInterfaceOptions)
}

// DeleteInstanceNetworkInterfaceWithContext is an alternate form of the DeleteInstanceNetworkInterface method which supports a Context parameter
func (vpc *VpcV1) DeleteInstanceNetworkInterfaceWithContext(ctx context.Context, deleteInstanceNetworkInterfaceOptions *DeleteInstanceNetworkInterfaceOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteInstanceNetworkInterfaceOptions, "deleteInstanceNetworkInterfaceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteInstanceNetworkInterfaceOptions, "deleteInstanceNetworkInterfaceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *deleteInstanceNetworkInterfaceOptions.InstanceID,
		"id":          *deleteInstanceNetworkInterfaceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/network_interfaces/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteInstanceNetworkInterfaceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteInstanceNetworkInterface")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetInstanceNetworkInterface : Retrieve a network interface
// This request retrieves a single network interface specified by the identifier in the URL.
func (vpc *VpcV1) GetInstanceNetworkInterface(getInstanceNetworkInterfaceOptions *GetInstanceNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	return vpc.GetInstanceNetworkInterfaceWithContext(context.Background(), getInstanceNetworkInterfaceOptions)
}

// GetInstanceNetworkInterfaceWithContext is an alternate form of the GetInstanceNetworkInterface method which supports a Context parameter
func (vpc *VpcV1) GetInstanceNetworkInterfaceWithContext(ctx context.Context, getInstanceNetworkInterfaceOptions *GetInstanceNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceNetworkInterfaceOptions, "getInstanceNetworkInterfaceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceNetworkInterfaceOptions, "getInstanceNetworkInterfaceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *getInstanceNetworkInterfaceOptions.InstanceID,
		"id":          *getInstanceNetworkInterfaceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/network_interfaces/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceNetworkInterfaceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetInstanceNetworkInterface")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkInterface)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateInstanceNetworkInterface : Update a network interface
// This request updates a network interface with the information provided in a network interface patch object. The
// network interface patch object is structured in the same way as a retrieved network interface and needs to contain
// only the information to be updated.
func (vpc *VpcV1) UpdateInstanceNetworkInterface(updateInstanceNetworkInterfaceOptions *UpdateInstanceNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	return vpc.UpdateInstanceNetworkInterfaceWithContext(context.Background(), updateInstanceNetworkInterfaceOptions)
}

// UpdateInstanceNetworkInterfaceWithContext is an alternate form of the UpdateInstanceNetworkInterface method which supports a Context parameter
func (vpc *VpcV1) UpdateInstanceNetworkInterfaceWithContext(ctx context.Context, updateInstanceNetworkInterfaceOptions *UpdateInstanceNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateInstanceNetworkInterfaceOptions, "updateInstanceNetworkInterfaceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateInstanceNetworkInterfaceOptions, "updateInstanceNetworkInterfaceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *updateInstanceNetworkInterfaceOptions.InstanceID,
		"id":          *updateInstanceNetworkInterfaceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/network_interfaces/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateInstanceNetworkInterfaceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateInstanceNetworkInterface")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateInstanceNetworkInterfaceOptions.NetworkInterfacePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkInterface)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListInstanceNetworkInterfaceFloatingIps : List all floating IPs associated with a network interface
// This request lists all floating IPs associated with a network interface.
func (vpc *VpcV1) ListInstanceNetworkInterfaceFloatingIps(listInstanceNetworkInterfaceFloatingIpsOptions *ListInstanceNetworkInterfaceFloatingIpsOptions) (result *FloatingIPUnpaginatedCollection, response *core.DetailedResponse, err error) {
	return vpc.ListInstanceNetworkInterfaceFloatingIpsWithContext(context.Background(), listInstanceNetworkInterfaceFloatingIpsOptions)
}

// ListInstanceNetworkInterfaceFloatingIpsWithContext is an alternate form of the ListInstanceNetworkInterfaceFloatingIps method which supports a Context parameter
func (vpc *VpcV1) ListInstanceNetworkInterfaceFloatingIpsWithContext(ctx context.Context, listInstanceNetworkInterfaceFloatingIpsOptions *ListInstanceNetworkInterfaceFloatingIpsOptions) (result *FloatingIPUnpaginatedCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listInstanceNetworkInterfaceFloatingIpsOptions, "listInstanceNetworkInterfaceFloatingIpsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listInstanceNetworkInterfaceFloatingIpsOptions, "listInstanceNetworkInterfaceFloatingIpsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id":          *listInstanceNetworkInterfaceFloatingIpsOptions.InstanceID,
		"network_interface_id": *listInstanceNetworkInterfaceFloatingIpsOptions.NetworkInterfaceID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/network_interfaces/{network_interface_id}/floating_ips`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstanceNetworkInterfaceFloatingIpsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListInstanceNetworkInterfaceFloatingIps")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFloatingIPUnpaginatedCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// RemoveInstanceNetworkInterfaceFloatingIP : Disassociate a floating IP from a network interface
// This request disassociates the specified floating IP from the specified network interface.
func (vpc *VpcV1) RemoveInstanceNetworkInterfaceFloatingIP(removeInstanceNetworkInterfaceFloatingIPOptions *RemoveInstanceNetworkInterfaceFloatingIPOptions) (response *core.DetailedResponse, err error) {
	return vpc.RemoveInstanceNetworkInterfaceFloatingIPWithContext(context.Background(), removeInstanceNetworkInterfaceFloatingIPOptions)
}

// RemoveInstanceNetworkInterfaceFloatingIPWithContext is an alternate form of the RemoveInstanceNetworkInterfaceFloatingIP method which supports a Context parameter
func (vpc *VpcV1) RemoveInstanceNetworkInterfaceFloatingIPWithContext(ctx context.Context, removeInstanceNetworkInterfaceFloatingIPOptions *RemoveInstanceNetworkInterfaceFloatingIPOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(removeInstanceNetworkInterfaceFloatingIPOptions, "removeInstanceNetworkInterfaceFloatingIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(removeInstanceNetworkInterfaceFloatingIPOptions, "removeInstanceNetworkInterfaceFloatingIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id":          *removeInstanceNetworkInterfaceFloatingIPOptions.InstanceID,
		"network_interface_id": *removeInstanceNetworkInterfaceFloatingIPOptions.NetworkInterfaceID,
		"id":                   *removeInstanceNetworkInterfaceFloatingIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/network_interfaces/{network_interface_id}/floating_ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range removeInstanceNetworkInterfaceFloatingIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "RemoveInstanceNetworkInterfaceFloatingIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetInstanceNetworkInterfaceFloatingIP : Retrieve associated floating IP
// This request a retrieves a specified floating IP address if it is associated with the network interface and instance
// specified in the URL.
func (vpc *VpcV1) GetInstanceNetworkInterfaceFloatingIP(getInstanceNetworkInterfaceFloatingIPOptions *GetInstanceNetworkInterfaceFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	return vpc.GetInstanceNetworkInterfaceFloatingIPWithContext(context.Background(), getInstanceNetworkInterfaceFloatingIPOptions)
}

// GetInstanceNetworkInterfaceFloatingIPWithContext is an alternate form of the GetInstanceNetworkInterfaceFloatingIP method which supports a Context parameter
func (vpc *VpcV1) GetInstanceNetworkInterfaceFloatingIPWithContext(ctx context.Context, getInstanceNetworkInterfaceFloatingIPOptions *GetInstanceNetworkInterfaceFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceNetworkInterfaceFloatingIPOptions, "getInstanceNetworkInterfaceFloatingIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceNetworkInterfaceFloatingIPOptions, "getInstanceNetworkInterfaceFloatingIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id":          *getInstanceNetworkInterfaceFloatingIPOptions.InstanceID,
		"network_interface_id": *getInstanceNetworkInterfaceFloatingIPOptions.NetworkInterfaceID,
		"id":                   *getInstanceNetworkInterfaceFloatingIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/network_interfaces/{network_interface_id}/floating_ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceNetworkInterfaceFloatingIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetInstanceNetworkInterfaceFloatingIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFloatingIP)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// AddInstanceNetworkInterfaceFloatingIP : Associate a floating IP with a network interface
// This request associates the specified floating IP with the specified network interface, replacing any existing
// association. For this request to succeed, the existing floating IP must not be required by another resource, such as
// a public gateway. A request body is not required, and if supplied, is ignored.
func (vpc *VpcV1) AddInstanceNetworkInterfaceFloatingIP(addInstanceNetworkInterfaceFloatingIPOptions *AddInstanceNetworkInterfaceFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	return vpc.AddInstanceNetworkInterfaceFloatingIPWithContext(context.Background(), addInstanceNetworkInterfaceFloatingIPOptions)
}

// AddInstanceNetworkInterfaceFloatingIPWithContext is an alternate form of the AddInstanceNetworkInterfaceFloatingIP method which supports a Context parameter
func (vpc *VpcV1) AddInstanceNetworkInterfaceFloatingIPWithContext(ctx context.Context, addInstanceNetworkInterfaceFloatingIPOptions *AddInstanceNetworkInterfaceFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(addInstanceNetworkInterfaceFloatingIPOptions, "addInstanceNetworkInterfaceFloatingIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(addInstanceNetworkInterfaceFloatingIPOptions, "addInstanceNetworkInterfaceFloatingIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id":          *addInstanceNetworkInterfaceFloatingIPOptions.InstanceID,
		"network_interface_id": *addInstanceNetworkInterfaceFloatingIPOptions.NetworkInterfaceID,
		"id":                   *addInstanceNetworkInterfaceFloatingIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/network_interfaces/{network_interface_id}/floating_ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range addInstanceNetworkInterfaceFloatingIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "AddInstanceNetworkInterfaceFloatingIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFloatingIP)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListInstanceVolumeAttachments : List all volumes attachments on an instance
// This request lists all volume attachments on an instance. A volume attachment connects a volume to an instance. Each
// instance may have many volume attachments but each volume attachment connects exactly one instance to exactly one
// volume.
func (vpc *VpcV1) ListInstanceVolumeAttachments(listInstanceVolumeAttachmentsOptions *ListInstanceVolumeAttachmentsOptions) (result *VolumeAttachmentCollection, response *core.DetailedResponse, err error) {
	return vpc.ListInstanceVolumeAttachmentsWithContext(context.Background(), listInstanceVolumeAttachmentsOptions)
}

// ListInstanceVolumeAttachmentsWithContext is an alternate form of the ListInstanceVolumeAttachments method which supports a Context parameter
func (vpc *VpcV1) ListInstanceVolumeAttachmentsWithContext(ctx context.Context, listInstanceVolumeAttachmentsOptions *ListInstanceVolumeAttachmentsOptions) (result *VolumeAttachmentCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listInstanceVolumeAttachmentsOptions, "listInstanceVolumeAttachmentsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listInstanceVolumeAttachmentsOptions, "listInstanceVolumeAttachmentsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *listInstanceVolumeAttachmentsOptions.InstanceID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/volume_attachments`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstanceVolumeAttachmentsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListInstanceVolumeAttachments")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolumeAttachmentCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateInstanceVolumeAttachment : Create a volume attachment on an instance
// This request creates a new volume attachment from a volume attachment prototype object. The prototype object is
// structured in the same way as a retrieved volume attachment, and contains the information necessary to create the new
// volume attachment. The creation of a new volume attachment connects a volume to an instance.
func (vpc *VpcV1) CreateInstanceVolumeAttachment(createInstanceVolumeAttachmentOptions *CreateInstanceVolumeAttachmentOptions) (result *VolumeAttachment, response *core.DetailedResponse, err error) {
	return vpc.CreateInstanceVolumeAttachmentWithContext(context.Background(), createInstanceVolumeAttachmentOptions)
}

// CreateInstanceVolumeAttachmentWithContext is an alternate form of the CreateInstanceVolumeAttachment method which supports a Context parameter
func (vpc *VpcV1) CreateInstanceVolumeAttachmentWithContext(ctx context.Context, createInstanceVolumeAttachmentOptions *CreateInstanceVolumeAttachmentOptions) (result *VolumeAttachment, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createInstanceVolumeAttachmentOptions, "createInstanceVolumeAttachmentOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createInstanceVolumeAttachmentOptions, "createInstanceVolumeAttachmentOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *createInstanceVolumeAttachmentOptions.InstanceID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/volume_attachments`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createInstanceVolumeAttachmentOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateInstanceVolumeAttachment")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createInstanceVolumeAttachmentOptions.Volume != nil {
		body["volume"] = createInstanceVolumeAttachmentOptions.Volume
	}
	if createInstanceVolumeAttachmentOptions.DeleteVolumeOnInstanceDelete != nil {
		body["delete_volume_on_instance_delete"] = createInstanceVolumeAttachmentOptions.DeleteVolumeOnInstanceDelete
	}
	if createInstanceVolumeAttachmentOptions.Name != nil {
		body["name"] = createInstanceVolumeAttachmentOptions.Name
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolumeAttachment)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteInstanceVolumeAttachment : Delete a volume attachment
// This request deletes a volume attachment. The deletion of a volume attachment detaches a volume from an instance.
func (vpc *VpcV1) DeleteInstanceVolumeAttachment(deleteInstanceVolumeAttachmentOptions *DeleteInstanceVolumeAttachmentOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteInstanceVolumeAttachmentWithContext(context.Background(), deleteInstanceVolumeAttachmentOptions)
}

// DeleteInstanceVolumeAttachmentWithContext is an alternate form of the DeleteInstanceVolumeAttachment method which supports a Context parameter
func (vpc *VpcV1) DeleteInstanceVolumeAttachmentWithContext(ctx context.Context, deleteInstanceVolumeAttachmentOptions *DeleteInstanceVolumeAttachmentOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteInstanceVolumeAttachmentOptions, "deleteInstanceVolumeAttachmentOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteInstanceVolumeAttachmentOptions, "deleteInstanceVolumeAttachmentOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *deleteInstanceVolumeAttachmentOptions.InstanceID,
		"id":          *deleteInstanceVolumeAttachmentOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/volume_attachments/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteInstanceVolumeAttachmentOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteInstanceVolumeAttachment")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetInstanceVolumeAttachment : Retrieve a volume attachment
// This request retrieves a single volume attachment specified by the identifier in the URL.
func (vpc *VpcV1) GetInstanceVolumeAttachment(getInstanceVolumeAttachmentOptions *GetInstanceVolumeAttachmentOptions) (result *VolumeAttachment, response *core.DetailedResponse, err error) {
	return vpc.GetInstanceVolumeAttachmentWithContext(context.Background(), getInstanceVolumeAttachmentOptions)
}

// GetInstanceVolumeAttachmentWithContext is an alternate form of the GetInstanceVolumeAttachment method which supports a Context parameter
func (vpc *VpcV1) GetInstanceVolumeAttachmentWithContext(ctx context.Context, getInstanceVolumeAttachmentOptions *GetInstanceVolumeAttachmentOptions) (result *VolumeAttachment, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceVolumeAttachmentOptions, "getInstanceVolumeAttachmentOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceVolumeAttachmentOptions, "getInstanceVolumeAttachmentOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *getInstanceVolumeAttachmentOptions.InstanceID,
		"id":          *getInstanceVolumeAttachmentOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/volume_attachments/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceVolumeAttachmentOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetInstanceVolumeAttachment")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolumeAttachment)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateInstanceVolumeAttachment : Update a volume attachment
// This request updates a volume attachment with the information provided in a volume attachment patch object. The
// volume attachment patch object is structured in the same way as a retrieved volume attachment and needs to contain
// only the information to be updated.
func (vpc *VpcV1) UpdateInstanceVolumeAttachment(updateInstanceVolumeAttachmentOptions *UpdateInstanceVolumeAttachmentOptions) (result *VolumeAttachment, response *core.DetailedResponse, err error) {
	return vpc.UpdateInstanceVolumeAttachmentWithContext(context.Background(), updateInstanceVolumeAttachmentOptions)
}

// UpdateInstanceVolumeAttachmentWithContext is an alternate form of the UpdateInstanceVolumeAttachment method which supports a Context parameter
func (vpc *VpcV1) UpdateInstanceVolumeAttachmentWithContext(ctx context.Context, updateInstanceVolumeAttachmentOptions *UpdateInstanceVolumeAttachmentOptions) (result *VolumeAttachment, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateInstanceVolumeAttachmentOptions, "updateInstanceVolumeAttachmentOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateInstanceVolumeAttachmentOptions, "updateInstanceVolumeAttachmentOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_id": *updateInstanceVolumeAttachmentOptions.InstanceID,
		"id":          *updateInstanceVolumeAttachmentOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instances/{instance_id}/volume_attachments/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateInstanceVolumeAttachmentOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateInstanceVolumeAttachment")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateInstanceVolumeAttachmentOptions.VolumeAttachmentPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolumeAttachment)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListInstanceGroups : List all instance groups
// This request lists all instance groups in the region.
func (vpc *VpcV1) ListInstanceGroups(listInstanceGroupsOptions *ListInstanceGroupsOptions) (result *InstanceGroupCollection, response *core.DetailedResponse, err error) {
	return vpc.ListInstanceGroupsWithContext(context.Background(), listInstanceGroupsOptions)
}

// ListInstanceGroupsWithContext is an alternate form of the ListInstanceGroups method which supports a Context parameter
func (vpc *VpcV1) ListInstanceGroupsWithContext(ctx context.Context, listInstanceGroupsOptions *ListInstanceGroupsOptions) (result *InstanceGroupCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listInstanceGroupsOptions, "listInstanceGroupsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstanceGroupsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListInstanceGroups")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listInstanceGroupsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listInstanceGroupsOptions.Start))
	}
	if listInstanceGroupsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listInstanceGroupsOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateInstanceGroup : Create an instance group
// This request creates a new instance group.
func (vpc *VpcV1) CreateInstanceGroup(createInstanceGroupOptions *CreateInstanceGroupOptions) (result *InstanceGroup, response *core.DetailedResponse, err error) {
	return vpc.CreateInstanceGroupWithContext(context.Background(), createInstanceGroupOptions)
}

// CreateInstanceGroupWithContext is an alternate form of the CreateInstanceGroup method which supports a Context parameter
func (vpc *VpcV1) CreateInstanceGroupWithContext(ctx context.Context, createInstanceGroupOptions *CreateInstanceGroupOptions) (result *InstanceGroup, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createInstanceGroupOptions, "createInstanceGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createInstanceGroupOptions, "createInstanceGroupOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createInstanceGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateInstanceGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createInstanceGroupOptions.InstanceTemplate != nil {
		body["instance_template"] = createInstanceGroupOptions.InstanceTemplate
	}
	if createInstanceGroupOptions.Subnets != nil {
		body["subnets"] = createInstanceGroupOptions.Subnets
	}
	if createInstanceGroupOptions.ApplicationPort != nil {
		body["application_port"] = createInstanceGroupOptions.ApplicationPort
	}
	if createInstanceGroupOptions.LoadBalancer != nil {
		body["load_balancer"] = createInstanceGroupOptions.LoadBalancer
	}
	if createInstanceGroupOptions.LoadBalancerPool != nil {
		body["load_balancer_pool"] = createInstanceGroupOptions.LoadBalancerPool
	}
	if createInstanceGroupOptions.MembershipCount != nil {
		body["membership_count"] = createInstanceGroupOptions.MembershipCount
	}
	if createInstanceGroupOptions.Name != nil {
		body["name"] = createInstanceGroupOptions.Name
	}
	if createInstanceGroupOptions.ResourceGroup != nil {
		body["resource_group"] = createInstanceGroupOptions.ResourceGroup
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroup)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteInstanceGroup : Delete an instance group
// This request deletes an instance group. This operation cannot be reversed. Any instances associated with the group
// will be deleted.
func (vpc *VpcV1) DeleteInstanceGroup(deleteInstanceGroupOptions *DeleteInstanceGroupOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteInstanceGroupWithContext(context.Background(), deleteInstanceGroupOptions)
}

// DeleteInstanceGroupWithContext is an alternate form of the DeleteInstanceGroup method which supports a Context parameter
func (vpc *VpcV1) DeleteInstanceGroupWithContext(ctx context.Context, deleteInstanceGroupOptions *DeleteInstanceGroupOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteInstanceGroupOptions, "deleteInstanceGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteInstanceGroupOptions, "deleteInstanceGroupOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteInstanceGroupOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteInstanceGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteInstanceGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetInstanceGroup : Retrieve an instance group
// This request retrieves a single instance group specified by identifier in the URL.
func (vpc *VpcV1) GetInstanceGroup(getInstanceGroupOptions *GetInstanceGroupOptions) (result *InstanceGroup, response *core.DetailedResponse, err error) {
	return vpc.GetInstanceGroupWithContext(context.Background(), getInstanceGroupOptions)
}

// GetInstanceGroupWithContext is an alternate form of the GetInstanceGroup method which supports a Context parameter
func (vpc *VpcV1) GetInstanceGroupWithContext(ctx context.Context, getInstanceGroupOptions *GetInstanceGroupOptions) (result *InstanceGroup, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceGroupOptions, "getInstanceGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceGroupOptions, "getInstanceGroupOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getInstanceGroupOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetInstanceGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroup)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateInstanceGroup : Update an instance group
// This request updates an instance group with the information provided instance group patch. The instance group patch
// object is structured in the same way as a retrieved instance group and contains only the information to be updated.
func (vpc *VpcV1) UpdateInstanceGroup(updateInstanceGroupOptions *UpdateInstanceGroupOptions) (result *InstanceGroup, response *core.DetailedResponse, err error) {
	return vpc.UpdateInstanceGroupWithContext(context.Background(), updateInstanceGroupOptions)
}

// UpdateInstanceGroupWithContext is an alternate form of the UpdateInstanceGroup method which supports a Context parameter
func (vpc *VpcV1) UpdateInstanceGroupWithContext(ctx context.Context, updateInstanceGroupOptions *UpdateInstanceGroupOptions) (result *InstanceGroup, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateInstanceGroupOptions, "updateInstanceGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateInstanceGroupOptions, "updateInstanceGroupOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateInstanceGroupOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateInstanceGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateInstanceGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateInstanceGroupOptions.InstanceGroupPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroup)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteInstanceGroupLoadBalancer : Delete an instance group load balancer
// This request unbinds the instance group from the load balancer pool, and deletes the load balancer pool members.
func (vpc *VpcV1) DeleteInstanceGroupLoadBalancer(deleteInstanceGroupLoadBalancerOptions *DeleteInstanceGroupLoadBalancerOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteInstanceGroupLoadBalancerWithContext(context.Background(), deleteInstanceGroupLoadBalancerOptions)
}

// DeleteInstanceGroupLoadBalancerWithContext is an alternate form of the DeleteInstanceGroupLoadBalancer method which supports a Context parameter
func (vpc *VpcV1) DeleteInstanceGroupLoadBalancerWithContext(ctx context.Context, deleteInstanceGroupLoadBalancerOptions *DeleteInstanceGroupLoadBalancerOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteInstanceGroupLoadBalancerOptions, "deleteInstanceGroupLoadBalancerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteInstanceGroupLoadBalancerOptions, "deleteInstanceGroupLoadBalancerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id": *deleteInstanceGroupLoadBalancerOptions.InstanceGroupID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/load_balancer`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteInstanceGroupLoadBalancerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteInstanceGroupLoadBalancer")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// ListInstanceGroupManagers : List all managers for an instance group
// This request lists all managers for an instance group.
func (vpc *VpcV1) ListInstanceGroupManagers(listInstanceGroupManagersOptions *ListInstanceGroupManagersOptions) (result *InstanceGroupManagerCollection, response *core.DetailedResponse, err error) {
	return vpc.ListInstanceGroupManagersWithContext(context.Background(), listInstanceGroupManagersOptions)
}

// ListInstanceGroupManagersWithContext is an alternate form of the ListInstanceGroupManagers method which supports a Context parameter
func (vpc *VpcV1) ListInstanceGroupManagersWithContext(ctx context.Context, listInstanceGroupManagersOptions *ListInstanceGroupManagersOptions) (result *InstanceGroupManagerCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listInstanceGroupManagersOptions, "listInstanceGroupManagersOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listInstanceGroupManagersOptions, "listInstanceGroupManagersOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id": *listInstanceGroupManagersOptions.InstanceGroupID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/managers`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstanceGroupManagersOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListInstanceGroupManagers")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listInstanceGroupManagersOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listInstanceGroupManagersOptions.Start))
	}
	if listInstanceGroupManagersOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listInstanceGroupManagersOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupManagerCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateInstanceGroupManager : Create a manager for an instance group
// This request creates a new instance group manager.
func (vpc *VpcV1) CreateInstanceGroupManager(createInstanceGroupManagerOptions *CreateInstanceGroupManagerOptions) (result InstanceGroupManagerIntf, response *core.DetailedResponse, err error) {
	return vpc.CreateInstanceGroupManagerWithContext(context.Background(), createInstanceGroupManagerOptions)
}

// CreateInstanceGroupManagerWithContext is an alternate form of the CreateInstanceGroupManager method which supports a Context parameter
func (vpc *VpcV1) CreateInstanceGroupManagerWithContext(ctx context.Context, createInstanceGroupManagerOptions *CreateInstanceGroupManagerOptions) (result InstanceGroupManagerIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createInstanceGroupManagerOptions, "createInstanceGroupManagerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createInstanceGroupManagerOptions, "createInstanceGroupManagerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id": *createInstanceGroupManagerOptions.InstanceGroupID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/managers`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createInstanceGroupManagerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateInstanceGroupManager")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(createInstanceGroupManagerOptions.InstanceGroupManagerPrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupManager)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteInstanceGroupManager : Delete an instance group manager
// This request deletes an instance group manager. This operation cannot be reversed.
func (vpc *VpcV1) DeleteInstanceGroupManager(deleteInstanceGroupManagerOptions *DeleteInstanceGroupManagerOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteInstanceGroupManagerWithContext(context.Background(), deleteInstanceGroupManagerOptions)
}

// DeleteInstanceGroupManagerWithContext is an alternate form of the DeleteInstanceGroupManager method which supports a Context parameter
func (vpc *VpcV1) DeleteInstanceGroupManagerWithContext(ctx context.Context, deleteInstanceGroupManagerOptions *DeleteInstanceGroupManagerOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteInstanceGroupManagerOptions, "deleteInstanceGroupManagerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteInstanceGroupManagerOptions, "deleteInstanceGroupManagerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id": *deleteInstanceGroupManagerOptions.InstanceGroupID,
		"id":                *deleteInstanceGroupManagerOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/managers/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteInstanceGroupManagerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteInstanceGroupManager")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetInstanceGroupManager : Retrieve an instance group manager
// This request retrieves a single instance group manager specified by identifier in the URL.
func (vpc *VpcV1) GetInstanceGroupManager(getInstanceGroupManagerOptions *GetInstanceGroupManagerOptions) (result InstanceGroupManagerIntf, response *core.DetailedResponse, err error) {
	return vpc.GetInstanceGroupManagerWithContext(context.Background(), getInstanceGroupManagerOptions)
}

// GetInstanceGroupManagerWithContext is an alternate form of the GetInstanceGroupManager method which supports a Context parameter
func (vpc *VpcV1) GetInstanceGroupManagerWithContext(ctx context.Context, getInstanceGroupManagerOptions *GetInstanceGroupManagerOptions) (result InstanceGroupManagerIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceGroupManagerOptions, "getInstanceGroupManagerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceGroupManagerOptions, "getInstanceGroupManagerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id": *getInstanceGroupManagerOptions.InstanceGroupID,
		"id":                *getInstanceGroupManagerOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/managers/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceGroupManagerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetInstanceGroupManager")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupManager)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateInstanceGroupManager : Update an instance group manager
// This request updates an instance group manager with the information provided instance group manager patch.
func (vpc *VpcV1) UpdateInstanceGroupManager(updateInstanceGroupManagerOptions *UpdateInstanceGroupManagerOptions) (result InstanceGroupManagerIntf, response *core.DetailedResponse, err error) {
	return vpc.UpdateInstanceGroupManagerWithContext(context.Background(), updateInstanceGroupManagerOptions)
}

// UpdateInstanceGroupManagerWithContext is an alternate form of the UpdateInstanceGroupManager method which supports a Context parameter
func (vpc *VpcV1) UpdateInstanceGroupManagerWithContext(ctx context.Context, updateInstanceGroupManagerOptions *UpdateInstanceGroupManagerOptions) (result InstanceGroupManagerIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateInstanceGroupManagerOptions, "updateInstanceGroupManagerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateInstanceGroupManagerOptions, "updateInstanceGroupManagerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id": *updateInstanceGroupManagerOptions.InstanceGroupID,
		"id":                *updateInstanceGroupManagerOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/managers/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateInstanceGroupManagerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateInstanceGroupManager")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateInstanceGroupManagerOptions.InstanceGroupManagerPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupManager)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListInstanceGroupManagerActions : List all actions for an instance group manager
// This request lists all instance group actions for an instance group manager.
func (vpc *VpcV1) ListInstanceGroupManagerActions(listInstanceGroupManagerActionsOptions *ListInstanceGroupManagerActionsOptions) (result *InstanceGroupManagerActionsCollection, response *core.DetailedResponse, err error) {
	return vpc.ListInstanceGroupManagerActionsWithContext(context.Background(), listInstanceGroupManagerActionsOptions)
}

// ListInstanceGroupManagerActionsWithContext is an alternate form of the ListInstanceGroupManagerActions method which supports a Context parameter
func (vpc *VpcV1) ListInstanceGroupManagerActionsWithContext(ctx context.Context, listInstanceGroupManagerActionsOptions *ListInstanceGroupManagerActionsOptions) (result *InstanceGroupManagerActionsCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listInstanceGroupManagerActionsOptions, "listInstanceGroupManagerActionsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listInstanceGroupManagerActionsOptions, "listInstanceGroupManagerActionsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id":         *listInstanceGroupManagerActionsOptions.InstanceGroupID,
		"instance_group_manager_id": *listInstanceGroupManagerActionsOptions.InstanceGroupManagerID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/actions`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstanceGroupManagerActionsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListInstanceGroupManagerActions")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listInstanceGroupManagerActionsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listInstanceGroupManagerActionsOptions.Start))
	}
	if listInstanceGroupManagerActionsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listInstanceGroupManagerActionsOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupManagerActionsCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateInstanceGroupManagerAction : Create an instance group manager action
// This request creates a new instance group manager action.
func (vpc *VpcV1) CreateInstanceGroupManagerAction(createInstanceGroupManagerActionOptions *CreateInstanceGroupManagerActionOptions) (result InstanceGroupManagerActionIntf, response *core.DetailedResponse, err error) {
	return vpc.CreateInstanceGroupManagerActionWithContext(context.Background(), createInstanceGroupManagerActionOptions)
}

// CreateInstanceGroupManagerActionWithContext is an alternate form of the CreateInstanceGroupManagerAction method which supports a Context parameter
func (vpc *VpcV1) CreateInstanceGroupManagerActionWithContext(ctx context.Context, createInstanceGroupManagerActionOptions *CreateInstanceGroupManagerActionOptions) (result InstanceGroupManagerActionIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createInstanceGroupManagerActionOptions, "createInstanceGroupManagerActionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createInstanceGroupManagerActionOptions, "createInstanceGroupManagerActionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id":         *createInstanceGroupManagerActionOptions.InstanceGroupID,
		"instance_group_manager_id": *createInstanceGroupManagerActionOptions.InstanceGroupManagerID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/actions`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createInstanceGroupManagerActionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateInstanceGroupManagerAction")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(createInstanceGroupManagerActionOptions.InstanceGroupManagerActionPrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupManagerAction)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteInstanceGroupManagerAction : Delete specified instance group manager action
// This request deletes an instance group manager action. This operation cannot be reversed.
func (vpc *VpcV1) DeleteInstanceGroupManagerAction(deleteInstanceGroupManagerActionOptions *DeleteInstanceGroupManagerActionOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteInstanceGroupManagerActionWithContext(context.Background(), deleteInstanceGroupManagerActionOptions)
}

// DeleteInstanceGroupManagerActionWithContext is an alternate form of the DeleteInstanceGroupManagerAction method which supports a Context parameter
func (vpc *VpcV1) DeleteInstanceGroupManagerActionWithContext(ctx context.Context, deleteInstanceGroupManagerActionOptions *DeleteInstanceGroupManagerActionOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteInstanceGroupManagerActionOptions, "deleteInstanceGroupManagerActionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteInstanceGroupManagerActionOptions, "deleteInstanceGroupManagerActionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id":         *deleteInstanceGroupManagerActionOptions.InstanceGroupID,
		"instance_group_manager_id": *deleteInstanceGroupManagerActionOptions.InstanceGroupManagerID,
		"id":                        *deleteInstanceGroupManagerActionOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/actions/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteInstanceGroupManagerActionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteInstanceGroupManagerAction")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetInstanceGroupManagerAction : Retrieve specified instance group manager action
// This request retrieves a single instance group manager action specified by identifier in the URL.
func (vpc *VpcV1) GetInstanceGroupManagerAction(getInstanceGroupManagerActionOptions *GetInstanceGroupManagerActionOptions) (result InstanceGroupManagerActionIntf, response *core.DetailedResponse, err error) {
	return vpc.GetInstanceGroupManagerActionWithContext(context.Background(), getInstanceGroupManagerActionOptions)
}

// GetInstanceGroupManagerActionWithContext is an alternate form of the GetInstanceGroupManagerAction method which supports a Context parameter
func (vpc *VpcV1) GetInstanceGroupManagerActionWithContext(ctx context.Context, getInstanceGroupManagerActionOptions *GetInstanceGroupManagerActionOptions) (result InstanceGroupManagerActionIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceGroupManagerActionOptions, "getInstanceGroupManagerActionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceGroupManagerActionOptions, "getInstanceGroupManagerActionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id":         *getInstanceGroupManagerActionOptions.InstanceGroupID,
		"instance_group_manager_id": *getInstanceGroupManagerActionOptions.InstanceGroupManagerID,
		"id":                        *getInstanceGroupManagerActionOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/actions/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceGroupManagerActionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetInstanceGroupManagerAction")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupManagerAction)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateInstanceGroupManagerAction : Update specified instance group manager action
// This request updates an instance group manager action.
func (vpc *VpcV1) UpdateInstanceGroupManagerAction(updateInstanceGroupManagerActionOptions *UpdateInstanceGroupManagerActionOptions) (result InstanceGroupManagerActionIntf, response *core.DetailedResponse, err error) {
	return vpc.UpdateInstanceGroupManagerActionWithContext(context.Background(), updateInstanceGroupManagerActionOptions)
}

// UpdateInstanceGroupManagerActionWithContext is an alternate form of the UpdateInstanceGroupManagerAction method which supports a Context parameter
func (vpc *VpcV1) UpdateInstanceGroupManagerActionWithContext(ctx context.Context, updateInstanceGroupManagerActionOptions *UpdateInstanceGroupManagerActionOptions) (result InstanceGroupManagerActionIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateInstanceGroupManagerActionOptions, "updateInstanceGroupManagerActionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateInstanceGroupManagerActionOptions, "updateInstanceGroupManagerActionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id":         *updateInstanceGroupManagerActionOptions.InstanceGroupID,
		"instance_group_manager_id": *updateInstanceGroupManagerActionOptions.InstanceGroupManagerID,
		"id":                        *updateInstanceGroupManagerActionOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/actions/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateInstanceGroupManagerActionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateInstanceGroupManagerAction")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateInstanceGroupManagerActionOptions.InstanceGroupManagerActionPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupManagerAction)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListInstanceGroupManagerPolicies : List all policies for an instance group manager
// This request lists all policies for an instance group manager.
func (vpc *VpcV1) ListInstanceGroupManagerPolicies(listInstanceGroupManagerPoliciesOptions *ListInstanceGroupManagerPoliciesOptions) (result *InstanceGroupManagerPolicyCollection, response *core.DetailedResponse, err error) {
	return vpc.ListInstanceGroupManagerPoliciesWithContext(context.Background(), listInstanceGroupManagerPoliciesOptions)
}

// ListInstanceGroupManagerPoliciesWithContext is an alternate form of the ListInstanceGroupManagerPolicies method which supports a Context parameter
func (vpc *VpcV1) ListInstanceGroupManagerPoliciesWithContext(ctx context.Context, listInstanceGroupManagerPoliciesOptions *ListInstanceGroupManagerPoliciesOptions) (result *InstanceGroupManagerPolicyCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listInstanceGroupManagerPoliciesOptions, "listInstanceGroupManagerPoliciesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listInstanceGroupManagerPoliciesOptions, "listInstanceGroupManagerPoliciesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id":         *listInstanceGroupManagerPoliciesOptions.InstanceGroupID,
		"instance_group_manager_id": *listInstanceGroupManagerPoliciesOptions.InstanceGroupManagerID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/policies`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstanceGroupManagerPoliciesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListInstanceGroupManagerPolicies")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listInstanceGroupManagerPoliciesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listInstanceGroupManagerPoliciesOptions.Start))
	}
	if listInstanceGroupManagerPoliciesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listInstanceGroupManagerPoliciesOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupManagerPolicyCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateInstanceGroupManagerPolicy : Create a policy for an instance group manager
// This request creates a new instance group manager policy.
func (vpc *VpcV1) CreateInstanceGroupManagerPolicy(createInstanceGroupManagerPolicyOptions *CreateInstanceGroupManagerPolicyOptions) (result InstanceGroupManagerPolicyIntf, response *core.DetailedResponse, err error) {
	return vpc.CreateInstanceGroupManagerPolicyWithContext(context.Background(), createInstanceGroupManagerPolicyOptions)
}

// CreateInstanceGroupManagerPolicyWithContext is an alternate form of the CreateInstanceGroupManagerPolicy method which supports a Context parameter
func (vpc *VpcV1) CreateInstanceGroupManagerPolicyWithContext(ctx context.Context, createInstanceGroupManagerPolicyOptions *CreateInstanceGroupManagerPolicyOptions) (result InstanceGroupManagerPolicyIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createInstanceGroupManagerPolicyOptions, "createInstanceGroupManagerPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createInstanceGroupManagerPolicyOptions, "createInstanceGroupManagerPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id":         *createInstanceGroupManagerPolicyOptions.InstanceGroupID,
		"instance_group_manager_id": *createInstanceGroupManagerPolicyOptions.InstanceGroupManagerID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/policies`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createInstanceGroupManagerPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateInstanceGroupManagerPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(createInstanceGroupManagerPolicyOptions.InstanceGroupManagerPolicyPrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupManagerPolicy)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteInstanceGroupManagerPolicy : Delete an instance group manager policy
// This request deletes an instance group manager policy. This operation cannot be reversed.
func (vpc *VpcV1) DeleteInstanceGroupManagerPolicy(deleteInstanceGroupManagerPolicyOptions *DeleteInstanceGroupManagerPolicyOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteInstanceGroupManagerPolicyWithContext(context.Background(), deleteInstanceGroupManagerPolicyOptions)
}

// DeleteInstanceGroupManagerPolicyWithContext is an alternate form of the DeleteInstanceGroupManagerPolicy method which supports a Context parameter
func (vpc *VpcV1) DeleteInstanceGroupManagerPolicyWithContext(ctx context.Context, deleteInstanceGroupManagerPolicyOptions *DeleteInstanceGroupManagerPolicyOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteInstanceGroupManagerPolicyOptions, "deleteInstanceGroupManagerPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteInstanceGroupManagerPolicyOptions, "deleteInstanceGroupManagerPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id":         *deleteInstanceGroupManagerPolicyOptions.InstanceGroupID,
		"instance_group_manager_id": *deleteInstanceGroupManagerPolicyOptions.InstanceGroupManagerID,
		"id":                        *deleteInstanceGroupManagerPolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteInstanceGroupManagerPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteInstanceGroupManagerPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetInstanceGroupManagerPolicy : Retrieve an instance group manager policy
// This request retrieves a single instance group manager policy specified by identifier in the URL.
func (vpc *VpcV1) GetInstanceGroupManagerPolicy(getInstanceGroupManagerPolicyOptions *GetInstanceGroupManagerPolicyOptions) (result InstanceGroupManagerPolicyIntf, response *core.DetailedResponse, err error) {
	return vpc.GetInstanceGroupManagerPolicyWithContext(context.Background(), getInstanceGroupManagerPolicyOptions)
}

// GetInstanceGroupManagerPolicyWithContext is an alternate form of the GetInstanceGroupManagerPolicy method which supports a Context parameter
func (vpc *VpcV1) GetInstanceGroupManagerPolicyWithContext(ctx context.Context, getInstanceGroupManagerPolicyOptions *GetInstanceGroupManagerPolicyOptions) (result InstanceGroupManagerPolicyIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceGroupManagerPolicyOptions, "getInstanceGroupManagerPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceGroupManagerPolicyOptions, "getInstanceGroupManagerPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id":         *getInstanceGroupManagerPolicyOptions.InstanceGroupID,
		"instance_group_manager_id": *getInstanceGroupManagerPolicyOptions.InstanceGroupManagerID,
		"id":                        *getInstanceGroupManagerPolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceGroupManagerPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetInstanceGroupManagerPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupManagerPolicy)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateInstanceGroupManagerPolicy : Update an instance group manager policy
// This request updates an instance group manager policy.
func (vpc *VpcV1) UpdateInstanceGroupManagerPolicy(updateInstanceGroupManagerPolicyOptions *UpdateInstanceGroupManagerPolicyOptions) (result InstanceGroupManagerPolicyIntf, response *core.DetailedResponse, err error) {
	return vpc.UpdateInstanceGroupManagerPolicyWithContext(context.Background(), updateInstanceGroupManagerPolicyOptions)
}

// UpdateInstanceGroupManagerPolicyWithContext is an alternate form of the UpdateInstanceGroupManagerPolicy method which supports a Context parameter
func (vpc *VpcV1) UpdateInstanceGroupManagerPolicyWithContext(ctx context.Context, updateInstanceGroupManagerPolicyOptions *UpdateInstanceGroupManagerPolicyOptions) (result InstanceGroupManagerPolicyIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateInstanceGroupManagerPolicyOptions, "updateInstanceGroupManagerPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateInstanceGroupManagerPolicyOptions, "updateInstanceGroupManagerPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id":         *updateInstanceGroupManagerPolicyOptions.InstanceGroupID,
		"instance_group_manager_id": *updateInstanceGroupManagerPolicyOptions.InstanceGroupManagerID,
		"id":                        *updateInstanceGroupManagerPolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateInstanceGroupManagerPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateInstanceGroupManagerPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateInstanceGroupManagerPolicyOptions.InstanceGroupManagerPolicyPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupManagerPolicy)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteInstanceGroupMemberships : Delete all memberships from an instance group
// This request deletes all memberships of an instance group. This operation cannot be reversed. reversed. Any
// memberships that have `delete_instance_on_membership_delete` set to `true` will also have their instances deleted.
func (vpc *VpcV1) DeleteInstanceGroupMemberships(deleteInstanceGroupMembershipsOptions *DeleteInstanceGroupMembershipsOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteInstanceGroupMembershipsWithContext(context.Background(), deleteInstanceGroupMembershipsOptions)
}

// DeleteInstanceGroupMembershipsWithContext is an alternate form of the DeleteInstanceGroupMemberships method which supports a Context parameter
func (vpc *VpcV1) DeleteInstanceGroupMembershipsWithContext(ctx context.Context, deleteInstanceGroupMembershipsOptions *DeleteInstanceGroupMembershipsOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteInstanceGroupMembershipsOptions, "deleteInstanceGroupMembershipsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteInstanceGroupMembershipsOptions, "deleteInstanceGroupMembershipsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id": *deleteInstanceGroupMembershipsOptions.InstanceGroupID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/memberships`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteInstanceGroupMembershipsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteInstanceGroupMemberships")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// ListInstanceGroupMemberships : List all memberships for an instance group
// This request lists all instance group memberships for an instance group.
func (vpc *VpcV1) ListInstanceGroupMemberships(listInstanceGroupMembershipsOptions *ListInstanceGroupMembershipsOptions) (result *InstanceGroupMembershipCollection, response *core.DetailedResponse, err error) {
	return vpc.ListInstanceGroupMembershipsWithContext(context.Background(), listInstanceGroupMembershipsOptions)
}

// ListInstanceGroupMembershipsWithContext is an alternate form of the ListInstanceGroupMemberships method which supports a Context parameter
func (vpc *VpcV1) ListInstanceGroupMembershipsWithContext(ctx context.Context, listInstanceGroupMembershipsOptions *ListInstanceGroupMembershipsOptions) (result *InstanceGroupMembershipCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listInstanceGroupMembershipsOptions, "listInstanceGroupMembershipsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listInstanceGroupMembershipsOptions, "listInstanceGroupMembershipsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id": *listInstanceGroupMembershipsOptions.InstanceGroupID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/memberships`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listInstanceGroupMembershipsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListInstanceGroupMemberships")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listInstanceGroupMembershipsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listInstanceGroupMembershipsOptions.Start))
	}
	if listInstanceGroupMembershipsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listInstanceGroupMembershipsOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupMembershipCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteInstanceGroupMembership : Delete an instance group membership
// This request deletes a memberships of an instance group. This operation cannot be reversed. reversed. If the
// membership has `delete_instance_on_membership_delete` set to `true`, the instance will also be deleted.
func (vpc *VpcV1) DeleteInstanceGroupMembership(deleteInstanceGroupMembershipOptions *DeleteInstanceGroupMembershipOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteInstanceGroupMembershipWithContext(context.Background(), deleteInstanceGroupMembershipOptions)
}

// DeleteInstanceGroupMembershipWithContext is an alternate form of the DeleteInstanceGroupMembership method which supports a Context parameter
func (vpc *VpcV1) DeleteInstanceGroupMembershipWithContext(ctx context.Context, deleteInstanceGroupMembershipOptions *DeleteInstanceGroupMembershipOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteInstanceGroupMembershipOptions, "deleteInstanceGroupMembershipOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteInstanceGroupMembershipOptions, "deleteInstanceGroupMembershipOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id": *deleteInstanceGroupMembershipOptions.InstanceGroupID,
		"id":                *deleteInstanceGroupMembershipOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/memberships/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteInstanceGroupMembershipOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteInstanceGroupMembership")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetInstanceGroupMembership : Retrieve an instance group membership
// This request retrieves a single instance group membership specified by identifier in the URL.
func (vpc *VpcV1) GetInstanceGroupMembership(getInstanceGroupMembershipOptions *GetInstanceGroupMembershipOptions) (result *InstanceGroupMembership, response *core.DetailedResponse, err error) {
	return vpc.GetInstanceGroupMembershipWithContext(context.Background(), getInstanceGroupMembershipOptions)
}

// GetInstanceGroupMembershipWithContext is an alternate form of the GetInstanceGroupMembership method which supports a Context parameter
func (vpc *VpcV1) GetInstanceGroupMembershipWithContext(ctx context.Context, getInstanceGroupMembershipOptions *GetInstanceGroupMembershipOptions) (result *InstanceGroupMembership, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getInstanceGroupMembershipOptions, "getInstanceGroupMembershipOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getInstanceGroupMembershipOptions, "getInstanceGroupMembershipOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id": *getInstanceGroupMembershipOptions.InstanceGroupID,
		"id":                *getInstanceGroupMembershipOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/memberships/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getInstanceGroupMembershipOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetInstanceGroupMembership")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupMembership)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateInstanceGroupMembership : Update an instance group membership
// This request updates an instance group membership with the information provided instance group membership patch.
func (vpc *VpcV1) UpdateInstanceGroupMembership(updateInstanceGroupMembershipOptions *UpdateInstanceGroupMembershipOptions) (result *InstanceGroupMembership, response *core.DetailedResponse, err error) {
	return vpc.UpdateInstanceGroupMembershipWithContext(context.Background(), updateInstanceGroupMembershipOptions)
}

// UpdateInstanceGroupMembershipWithContext is an alternate form of the UpdateInstanceGroupMembership method which supports a Context parameter
func (vpc *VpcV1) UpdateInstanceGroupMembershipWithContext(ctx context.Context, updateInstanceGroupMembershipOptions *UpdateInstanceGroupMembershipOptions) (result *InstanceGroupMembership, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateInstanceGroupMembershipOptions, "updateInstanceGroupMembershipOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateInstanceGroupMembershipOptions, "updateInstanceGroupMembershipOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"instance_group_id": *updateInstanceGroupMembershipOptions.InstanceGroupID,
		"id":                *updateInstanceGroupMembershipOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/instance_groups/{instance_group_id}/memberships/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateInstanceGroupMembershipOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateInstanceGroupMembership")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateInstanceGroupMembershipOptions.InstanceGroupMembershipPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalInstanceGroupMembership)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListDedicatedHostGroups : List all dedicated host groups
// This request lists all dedicated host groups in the region. Host groups are a collection of dedicated hosts for
// placement of instances. Each dedicated host must belong to one and only one group. Host groups do not span zones.
func (vpc *VpcV1) ListDedicatedHostGroups(listDedicatedHostGroupsOptions *ListDedicatedHostGroupsOptions) (result *DedicatedHostGroupCollection, response *core.DetailedResponse, err error) {
	return vpc.ListDedicatedHostGroupsWithContext(context.Background(), listDedicatedHostGroupsOptions)
}

// ListDedicatedHostGroupsWithContext is an alternate form of the ListDedicatedHostGroups method which supports a Context parameter
func (vpc *VpcV1) ListDedicatedHostGroupsWithContext(ctx context.Context, listDedicatedHostGroupsOptions *ListDedicatedHostGroupsOptions) (result *DedicatedHostGroupCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listDedicatedHostGroupsOptions, "listDedicatedHostGroupsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/dedicated_host/groups`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listDedicatedHostGroupsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListDedicatedHostGroups")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listDedicatedHostGroupsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listDedicatedHostGroupsOptions.Start))
	}
	if listDedicatedHostGroupsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listDedicatedHostGroupsOptions.Limit))
	}
	if listDedicatedHostGroupsOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listDedicatedHostGroupsOptions.ResourceGroupID))
	}
	if listDedicatedHostGroupsOptions.ZoneName != nil {
		builder.AddQuery("zone.name", fmt.Sprint(*listDedicatedHostGroupsOptions.ZoneName))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDedicatedHostGroupCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateDedicatedHostGroup : Create a dedicated host group
// This request creates a new dedicated host group.
func (vpc *VpcV1) CreateDedicatedHostGroup(createDedicatedHostGroupOptions *CreateDedicatedHostGroupOptions) (result *DedicatedHostGroup, response *core.DetailedResponse, err error) {
	return vpc.CreateDedicatedHostGroupWithContext(context.Background(), createDedicatedHostGroupOptions)
}

// CreateDedicatedHostGroupWithContext is an alternate form of the CreateDedicatedHostGroup method which supports a Context parameter
func (vpc *VpcV1) CreateDedicatedHostGroupWithContext(ctx context.Context, createDedicatedHostGroupOptions *CreateDedicatedHostGroupOptions) (result *DedicatedHostGroup, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(createDedicatedHostGroupOptions, "createDedicatedHostGroupOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/dedicated_host/groups`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createDedicatedHostGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateDedicatedHostGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createDedicatedHostGroupOptions.Class != nil {
		body["class"] = createDedicatedHostGroupOptions.Class
	}
	if createDedicatedHostGroupOptions.Family != nil {
		body["family"] = createDedicatedHostGroupOptions.Family
	}
	if createDedicatedHostGroupOptions.Name != nil {
		body["name"] = createDedicatedHostGroupOptions.Name
	}
	if createDedicatedHostGroupOptions.ResourceGroup != nil {
		body["resource_group"] = createDedicatedHostGroupOptions.ResourceGroup
	}
	if createDedicatedHostGroupOptions.Zone != nil {
		body["zone"] = createDedicatedHostGroupOptions.Zone
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDedicatedHostGroup)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteDedicatedHostGroup : Delete a dedicated host group
// This request deletes a dedicated host group.
func (vpc *VpcV1) DeleteDedicatedHostGroup(deleteDedicatedHostGroupOptions *DeleteDedicatedHostGroupOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteDedicatedHostGroupWithContext(context.Background(), deleteDedicatedHostGroupOptions)
}

// DeleteDedicatedHostGroupWithContext is an alternate form of the DeleteDedicatedHostGroup method which supports a Context parameter
func (vpc *VpcV1) DeleteDedicatedHostGroupWithContext(ctx context.Context, deleteDedicatedHostGroupOptions *DeleteDedicatedHostGroupOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteDedicatedHostGroupOptions, "deleteDedicatedHostGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteDedicatedHostGroupOptions, "deleteDedicatedHostGroupOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteDedicatedHostGroupOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/dedicated_host/groups/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteDedicatedHostGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteDedicatedHostGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetDedicatedHostGroup : Retrieve a dedicated host group
// This request retrieves a single dedicated host group specified by the identifier in the URL.
func (vpc *VpcV1) GetDedicatedHostGroup(getDedicatedHostGroupOptions *GetDedicatedHostGroupOptions) (result *DedicatedHostGroup, response *core.DetailedResponse, err error) {
	return vpc.GetDedicatedHostGroupWithContext(context.Background(), getDedicatedHostGroupOptions)
}

// GetDedicatedHostGroupWithContext is an alternate form of the GetDedicatedHostGroup method which supports a Context parameter
func (vpc *VpcV1) GetDedicatedHostGroupWithContext(ctx context.Context, getDedicatedHostGroupOptions *GetDedicatedHostGroupOptions) (result *DedicatedHostGroup, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getDedicatedHostGroupOptions, "getDedicatedHostGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getDedicatedHostGroupOptions, "getDedicatedHostGroupOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getDedicatedHostGroupOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/dedicated_host/groups/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getDedicatedHostGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetDedicatedHostGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDedicatedHostGroup)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateDedicatedHostGroup : Update a dedicated host group
// This request updates a dedicated host group with the information in a provided dedicated host group patch. The
// dedicated host group patch object is structured in the same way as a retrieved dedicated host group and contains only
// the information to be updated.
func (vpc *VpcV1) UpdateDedicatedHostGroup(updateDedicatedHostGroupOptions *UpdateDedicatedHostGroupOptions) (result *DedicatedHostGroup, response *core.DetailedResponse, err error) {
	return vpc.UpdateDedicatedHostGroupWithContext(context.Background(), updateDedicatedHostGroupOptions)
}

// UpdateDedicatedHostGroupWithContext is an alternate form of the UpdateDedicatedHostGroup method which supports a Context parameter
func (vpc *VpcV1) UpdateDedicatedHostGroupWithContext(ctx context.Context, updateDedicatedHostGroupOptions *UpdateDedicatedHostGroupOptions) (result *DedicatedHostGroup, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateDedicatedHostGroupOptions, "updateDedicatedHostGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateDedicatedHostGroupOptions, "updateDedicatedHostGroupOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateDedicatedHostGroupOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/dedicated_host/groups/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateDedicatedHostGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateDedicatedHostGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateDedicatedHostGroupOptions.DedicatedHostGroupPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDedicatedHostGroup)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListDedicatedHostProfiles : List all dedicated host profiles
// This request lists all provisionable dedicated host profiles in the region. A dedicated host profile specifies the
// hardware characteristics for a dedicated host.
func (vpc *VpcV1) ListDedicatedHostProfiles(listDedicatedHostProfilesOptions *ListDedicatedHostProfilesOptions) (result *DedicatedHostProfileCollection, response *core.DetailedResponse, err error) {
	return vpc.ListDedicatedHostProfilesWithContext(context.Background(), listDedicatedHostProfilesOptions)
}

// ListDedicatedHostProfilesWithContext is an alternate form of the ListDedicatedHostProfiles method which supports a Context parameter
func (vpc *VpcV1) ListDedicatedHostProfilesWithContext(ctx context.Context, listDedicatedHostProfilesOptions *ListDedicatedHostProfilesOptions) (result *DedicatedHostProfileCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listDedicatedHostProfilesOptions, "listDedicatedHostProfilesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/dedicated_host/profiles`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listDedicatedHostProfilesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListDedicatedHostProfiles")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listDedicatedHostProfilesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listDedicatedHostProfilesOptions.Start))
	}
	if listDedicatedHostProfilesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listDedicatedHostProfilesOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDedicatedHostProfileCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetDedicatedHostProfile : Retrieve a dedicated host profile
// This request retrieves a single dedicated host profile specified by the name in the URL.
func (vpc *VpcV1) GetDedicatedHostProfile(getDedicatedHostProfileOptions *GetDedicatedHostProfileOptions) (result *DedicatedHostProfile, response *core.DetailedResponse, err error) {
	return vpc.GetDedicatedHostProfileWithContext(context.Background(), getDedicatedHostProfileOptions)
}

// GetDedicatedHostProfileWithContext is an alternate form of the GetDedicatedHostProfile method which supports a Context parameter
func (vpc *VpcV1) GetDedicatedHostProfileWithContext(ctx context.Context, getDedicatedHostProfileOptions *GetDedicatedHostProfileOptions) (result *DedicatedHostProfile, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getDedicatedHostProfileOptions, "getDedicatedHostProfileOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getDedicatedHostProfileOptions, "getDedicatedHostProfileOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"name": *getDedicatedHostProfileOptions.Name,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/dedicated_host/profiles/{name}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getDedicatedHostProfileOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetDedicatedHostProfile")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDedicatedHostProfile)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListDedicatedHosts : List all dedicated hosts
// This request lists all dedicated hosts in the region.
func (vpc *VpcV1) ListDedicatedHosts(listDedicatedHostsOptions *ListDedicatedHostsOptions) (result *DedicatedHostCollection, response *core.DetailedResponse, err error) {
	return vpc.ListDedicatedHostsWithContext(context.Background(), listDedicatedHostsOptions)
}

// ListDedicatedHostsWithContext is an alternate form of the ListDedicatedHosts method which supports a Context parameter
func (vpc *VpcV1) ListDedicatedHostsWithContext(ctx context.Context, listDedicatedHostsOptions *ListDedicatedHostsOptions) (result *DedicatedHostCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listDedicatedHostsOptions, "listDedicatedHostsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/dedicated_hosts`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listDedicatedHostsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListDedicatedHosts")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listDedicatedHostsOptions.DedicatedHostGroupID != nil {
		builder.AddQuery("dedicated_host_group.id", fmt.Sprint(*listDedicatedHostsOptions.DedicatedHostGroupID))
	}
	if listDedicatedHostsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listDedicatedHostsOptions.Start))
	}
	if listDedicatedHostsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listDedicatedHostsOptions.Limit))
	}
	if listDedicatedHostsOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listDedicatedHostsOptions.ResourceGroupID))
	}
	if listDedicatedHostsOptions.ZoneName != nil {
		builder.AddQuery("zone.name", fmt.Sprint(*listDedicatedHostsOptions.ZoneName))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDedicatedHostCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateDedicatedHost : Create a dedicated host
// This request creates a new dedicated host.
func (vpc *VpcV1) CreateDedicatedHost(createDedicatedHostOptions *CreateDedicatedHostOptions) (result *DedicatedHost, response *core.DetailedResponse, err error) {
	return vpc.CreateDedicatedHostWithContext(context.Background(), createDedicatedHostOptions)
}

// CreateDedicatedHostWithContext is an alternate form of the CreateDedicatedHost method which supports a Context parameter
func (vpc *VpcV1) CreateDedicatedHostWithContext(ctx context.Context, createDedicatedHostOptions *CreateDedicatedHostOptions) (result *DedicatedHost, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createDedicatedHostOptions, "createDedicatedHostOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createDedicatedHostOptions, "createDedicatedHostOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/dedicated_hosts`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createDedicatedHostOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateDedicatedHost")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(createDedicatedHostOptions.DedicatedHostPrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDedicatedHost)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListDedicatedHostDisks : List all disks on a dedicated host
// This request lists all disks on a dedicated host.  A disk is a physical device that is locally attached to the
// compute node. By default, the listed disks are sorted by their
// `created_at` property values, with the newest disk first.
func (vpc *VpcV1) ListDedicatedHostDisks(listDedicatedHostDisksOptions *ListDedicatedHostDisksOptions) (result *DedicatedHostDiskCollection, response *core.DetailedResponse, err error) {
	return vpc.ListDedicatedHostDisksWithContext(context.Background(), listDedicatedHostDisksOptions)
}

// ListDedicatedHostDisksWithContext is an alternate form of the ListDedicatedHostDisks method which supports a Context parameter
func (vpc *VpcV1) ListDedicatedHostDisksWithContext(ctx context.Context, listDedicatedHostDisksOptions *ListDedicatedHostDisksOptions) (result *DedicatedHostDiskCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listDedicatedHostDisksOptions, "listDedicatedHostDisksOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listDedicatedHostDisksOptions, "listDedicatedHostDisksOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"dedicated_host_id": *listDedicatedHostDisksOptions.DedicatedHostID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/dedicated_hosts/{dedicated_host_id}/disks`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listDedicatedHostDisksOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListDedicatedHostDisks")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDedicatedHostDiskCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetDedicatedHostDisk : Retrieve a dedicated host disk
// This request retrieves a single dedicated host disk specified by the identifier in the URL.
func (vpc *VpcV1) GetDedicatedHostDisk(getDedicatedHostDiskOptions *GetDedicatedHostDiskOptions) (result *DedicatedHostDisk, response *core.DetailedResponse, err error) {
	return vpc.GetDedicatedHostDiskWithContext(context.Background(), getDedicatedHostDiskOptions)
}

// GetDedicatedHostDiskWithContext is an alternate form of the GetDedicatedHostDisk method which supports a Context parameter
func (vpc *VpcV1) GetDedicatedHostDiskWithContext(ctx context.Context, getDedicatedHostDiskOptions *GetDedicatedHostDiskOptions) (result *DedicatedHostDisk, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getDedicatedHostDiskOptions, "getDedicatedHostDiskOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getDedicatedHostDiskOptions, "getDedicatedHostDiskOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"dedicated_host_id": *getDedicatedHostDiskOptions.DedicatedHostID,
		"id":                *getDedicatedHostDiskOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/dedicated_hosts/{dedicated_host_id}/disks/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getDedicatedHostDiskOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetDedicatedHostDisk")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDedicatedHostDisk)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateDedicatedHostDisk : Update a dedicated host disk
// This request updates the dedicated host disk with the information in a provided patch.
func (vpc *VpcV1) UpdateDedicatedHostDisk(updateDedicatedHostDiskOptions *UpdateDedicatedHostDiskOptions) (result *DedicatedHostDisk, response *core.DetailedResponse, err error) {
	return vpc.UpdateDedicatedHostDiskWithContext(context.Background(), updateDedicatedHostDiskOptions)
}

// UpdateDedicatedHostDiskWithContext is an alternate form of the UpdateDedicatedHostDisk method which supports a Context parameter
func (vpc *VpcV1) UpdateDedicatedHostDiskWithContext(ctx context.Context, updateDedicatedHostDiskOptions *UpdateDedicatedHostDiskOptions) (result *DedicatedHostDisk, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateDedicatedHostDiskOptions, "updateDedicatedHostDiskOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateDedicatedHostDiskOptions, "updateDedicatedHostDiskOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"dedicated_host_id": *updateDedicatedHostDiskOptions.DedicatedHostID,
		"id":                *updateDedicatedHostDiskOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/dedicated_hosts/{dedicated_host_id}/disks/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateDedicatedHostDiskOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateDedicatedHostDisk")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateDedicatedHostDiskOptions.DedicatedHostDiskPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDedicatedHostDisk)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteDedicatedHost : Delete a dedicated host
// This request deletes a dedicated host.
func (vpc *VpcV1) DeleteDedicatedHost(deleteDedicatedHostOptions *DeleteDedicatedHostOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteDedicatedHostWithContext(context.Background(), deleteDedicatedHostOptions)
}

// DeleteDedicatedHostWithContext is an alternate form of the DeleteDedicatedHost method which supports a Context parameter
func (vpc *VpcV1) DeleteDedicatedHostWithContext(ctx context.Context, deleteDedicatedHostOptions *DeleteDedicatedHostOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteDedicatedHostOptions, "deleteDedicatedHostOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteDedicatedHostOptions, "deleteDedicatedHostOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteDedicatedHostOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/dedicated_hosts/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteDedicatedHostOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteDedicatedHost")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetDedicatedHost : Retrieve a dedicated host
// This request retrieves a single dedicated host specified by the identifiers in the URL.
func (vpc *VpcV1) GetDedicatedHost(getDedicatedHostOptions *GetDedicatedHostOptions) (result *DedicatedHost, response *core.DetailedResponse, err error) {
	return vpc.GetDedicatedHostWithContext(context.Background(), getDedicatedHostOptions)
}

// GetDedicatedHostWithContext is an alternate form of the GetDedicatedHost method which supports a Context parameter
func (vpc *VpcV1) GetDedicatedHostWithContext(ctx context.Context, getDedicatedHostOptions *GetDedicatedHostOptions) (result *DedicatedHost, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getDedicatedHostOptions, "getDedicatedHostOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getDedicatedHostOptions, "getDedicatedHostOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getDedicatedHostOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/dedicated_hosts/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getDedicatedHostOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetDedicatedHost")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDedicatedHost)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateDedicatedHost : Update a dedicated host
// This request updates a dedicated host with the information in a provided dedicated host patch. The dedicated host
// patch object is structured in the same way as a retrieved dedicated host and contains only the information to be
// updated.
func (vpc *VpcV1) UpdateDedicatedHost(updateDedicatedHostOptions *UpdateDedicatedHostOptions) (result *DedicatedHost, response *core.DetailedResponse, err error) {
	return vpc.UpdateDedicatedHostWithContext(context.Background(), updateDedicatedHostOptions)
}

// UpdateDedicatedHostWithContext is an alternate form of the UpdateDedicatedHost method which supports a Context parameter
func (vpc *VpcV1) UpdateDedicatedHostWithContext(ctx context.Context, updateDedicatedHostOptions *UpdateDedicatedHostOptions) (result *DedicatedHost, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateDedicatedHostOptions, "updateDedicatedHostOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateDedicatedHostOptions, "updateDedicatedHostOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateDedicatedHostOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/dedicated_hosts/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateDedicatedHostOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateDedicatedHost")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateDedicatedHostOptions.DedicatedHostPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalDedicatedHost)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListVolumeProfiles : List all volume profiles
// This request lists all volume profiles available in the region. A volume profile specifies the performance
// characteristics and pricing model for a volume.
func (vpc *VpcV1) ListVolumeProfiles(listVolumeProfilesOptions *ListVolumeProfilesOptions) (result *VolumeProfileCollection, response *core.DetailedResponse, err error) {
	return vpc.ListVolumeProfilesWithContext(context.Background(), listVolumeProfilesOptions)
}

// ListVolumeProfilesWithContext is an alternate form of the ListVolumeProfiles method which supports a Context parameter
func (vpc *VpcV1) ListVolumeProfilesWithContext(ctx context.Context, listVolumeProfilesOptions *ListVolumeProfilesOptions) (result *VolumeProfileCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listVolumeProfilesOptions, "listVolumeProfilesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/volume/profiles`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVolumeProfilesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListVolumeProfiles")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listVolumeProfilesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listVolumeProfilesOptions.Start))
	}
	if listVolumeProfilesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listVolumeProfilesOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolumeProfileCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetVolumeProfile : Retrieve a volume profile
// This request retrieves a single volume profile specified by the name in the URL.
func (vpc *VpcV1) GetVolumeProfile(getVolumeProfileOptions *GetVolumeProfileOptions) (result *VolumeProfile, response *core.DetailedResponse, err error) {
	return vpc.GetVolumeProfileWithContext(context.Background(), getVolumeProfileOptions)
}

// GetVolumeProfileWithContext is an alternate form of the GetVolumeProfile method which supports a Context parameter
func (vpc *VpcV1) GetVolumeProfileWithContext(ctx context.Context, getVolumeProfileOptions *GetVolumeProfileOptions) (result *VolumeProfile, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVolumeProfileOptions, "getVolumeProfileOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVolumeProfileOptions, "getVolumeProfileOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"name": *getVolumeProfileOptions.Name,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/volume/profiles/{name}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVolumeProfileOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetVolumeProfile")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolumeProfile)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListVolumes : List all volumes
// This request lists all volumes in the region. Volumes are network-connected block storage devices that may be
// attached to one or more instances in the same region.
func (vpc *VpcV1) ListVolumes(listVolumesOptions *ListVolumesOptions) (result *VolumeCollection, response *core.DetailedResponse, err error) {
	return vpc.ListVolumesWithContext(context.Background(), listVolumesOptions)
}

// ListVolumesWithContext is an alternate form of the ListVolumes method which supports a Context parameter
func (vpc *VpcV1) ListVolumesWithContext(ctx context.Context, listVolumesOptions *ListVolumesOptions) (result *VolumeCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listVolumesOptions, "listVolumesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/volumes`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVolumesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListVolumes")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listVolumesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listVolumesOptions.Start))
	}
	if listVolumesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listVolumesOptions.Limit))
	}
	if listVolumesOptions.Name != nil {
		builder.AddQuery("name", fmt.Sprint(*listVolumesOptions.Name))
	}
	if listVolumesOptions.ZoneName != nil {
		builder.AddQuery("zone.name", fmt.Sprint(*listVolumesOptions.ZoneName))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolumeCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateVolume : Create a volume
// This request creates a new volume from a volume prototype object. The prototype object is structured in the same way
// as a retrieved volume, and contains the information necessary to create the new volume.
func (vpc *VpcV1) CreateVolume(createVolumeOptions *CreateVolumeOptions) (result *Volume, response *core.DetailedResponse, err error) {
	return vpc.CreateVolumeWithContext(context.Background(), createVolumeOptions)
}

// CreateVolumeWithContext is an alternate form of the CreateVolume method which supports a Context parameter
func (vpc *VpcV1) CreateVolumeWithContext(ctx context.Context, createVolumeOptions *CreateVolumeOptions) (result *Volume, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createVolumeOptions, "createVolumeOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createVolumeOptions, "createVolumeOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/volumes`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createVolumeOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateVolume")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(createVolumeOptions.VolumePrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolume)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteVolume : Delete a volume
// This request deletes a volume. This operation cannot be reversed. For this request to succeed, the volume must not be
// attached to any instances.
func (vpc *VpcV1) DeleteVolume(deleteVolumeOptions *DeleteVolumeOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteVolumeWithContext(context.Background(), deleteVolumeOptions)
}

// DeleteVolumeWithContext is an alternate form of the DeleteVolume method which supports a Context parameter
func (vpc *VpcV1) DeleteVolumeWithContext(ctx context.Context, deleteVolumeOptions *DeleteVolumeOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteVolumeOptions, "deleteVolumeOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteVolumeOptions, "deleteVolumeOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteVolumeOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/volumes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteVolumeOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteVolume")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetVolume : Retrieve a volume
// This request retrieves a single volume specified by the identifier in the URL.
func (vpc *VpcV1) GetVolume(getVolumeOptions *GetVolumeOptions) (result *Volume, response *core.DetailedResponse, err error) {
	return vpc.GetVolumeWithContext(context.Background(), getVolumeOptions)
}

// GetVolumeWithContext is an alternate form of the GetVolume method which supports a Context parameter
func (vpc *VpcV1) GetVolumeWithContext(ctx context.Context, getVolumeOptions *GetVolumeOptions) (result *Volume, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVolumeOptions, "getVolumeOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVolumeOptions, "getVolumeOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getVolumeOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/volumes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVolumeOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetVolume")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolume)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateVolume : Update a volume
// This request updates a volume with the information in a provided volume patch. The volume patch object is structured
// in the same way as a retrieved volume and contains only the information to be updated.
func (vpc *VpcV1) UpdateVolume(updateVolumeOptions *UpdateVolumeOptions) (result *Volume, response *core.DetailedResponse, err error) {
	return vpc.UpdateVolumeWithContext(context.Background(), updateVolumeOptions)
}

// UpdateVolumeWithContext is an alternate form of the UpdateVolume method which supports a Context parameter
func (vpc *VpcV1) UpdateVolumeWithContext(ctx context.Context, updateVolumeOptions *UpdateVolumeOptions) (result *Volume, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateVolumeOptions, "updateVolumeOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateVolumeOptions, "updateVolumeOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateVolumeOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/volumes/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateVolumeOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateVolume")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateVolumeOptions.VolumePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVolume)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteSnapshots : Delete a filtered collection of snapshots
// This request deletes all snapshots created from a specific source volume.
func (vpc *VpcV1) DeleteSnapshots(deleteSnapshotsOptions *DeleteSnapshotsOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteSnapshotsWithContext(context.Background(), deleteSnapshotsOptions)
}

// DeleteSnapshotsWithContext is an alternate form of the DeleteSnapshots method which supports a Context parameter
func (vpc *VpcV1) DeleteSnapshotsWithContext(ctx context.Context, deleteSnapshotsOptions *DeleteSnapshotsOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteSnapshotsOptions, "deleteSnapshotsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteSnapshotsOptions, "deleteSnapshotsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/snapshots`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteSnapshotsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteSnapshots")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	builder.AddQuery("source_volume.id", fmt.Sprint(*deleteSnapshotsOptions.SourceVolumeID))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// ListSnapshots : List all snapshots
// This request lists all snapshots in the region. A snapshot preserves the data of a volume at the time the snapshot is
// created.
func (vpc *VpcV1) ListSnapshots(listSnapshotsOptions *ListSnapshotsOptions) (result *SnapshotCollection, response *core.DetailedResponse, err error) {
	return vpc.ListSnapshotsWithContext(context.Background(), listSnapshotsOptions)
}

// ListSnapshotsWithContext is an alternate form of the ListSnapshots method which supports a Context parameter
func (vpc *VpcV1) ListSnapshotsWithContext(ctx context.Context, listSnapshotsOptions *ListSnapshotsOptions) (result *SnapshotCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listSnapshotsOptions, "listSnapshotsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/snapshots`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listSnapshotsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListSnapshots")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listSnapshotsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listSnapshotsOptions.Start))
	}
	if listSnapshotsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listSnapshotsOptions.Limit))
	}
	if listSnapshotsOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listSnapshotsOptions.ResourceGroupID))
	}
	if listSnapshotsOptions.Name != nil {
		builder.AddQuery("name", fmt.Sprint(*listSnapshotsOptions.Name))
	}
	if listSnapshotsOptions.SourceVolumeID != nil {
		builder.AddQuery("source_volume.id", fmt.Sprint(*listSnapshotsOptions.SourceVolumeID))
	}
	if listSnapshotsOptions.SourceVolumeCRN != nil {
		builder.AddQuery("source_volume.crn", fmt.Sprint(*listSnapshotsOptions.SourceVolumeCRN))
	}
	if listSnapshotsOptions.SourceImageID != nil {
		builder.AddQuery("source_image.id", fmt.Sprint(*listSnapshotsOptions.SourceImageID))
	}
	if listSnapshotsOptions.SourceImageCRN != nil {
		builder.AddQuery("source_image.crn", fmt.Sprint(*listSnapshotsOptions.SourceImageCRN))
	}
	if listSnapshotsOptions.Sort != nil {
		builder.AddQuery("sort", fmt.Sprint(*listSnapshotsOptions.Sort))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSnapshotCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateSnapshot : Create a snapshot
// This request creates a new snapshot from a snapshot prototype object.  The prototype object is structured in the same
// way as a retrieved snapshot, and contains the information necessary to provision the new snapshot.
func (vpc *VpcV1) CreateSnapshot(createSnapshotOptions *CreateSnapshotOptions) (result *Snapshot, response *core.DetailedResponse, err error) {
	return vpc.CreateSnapshotWithContext(context.Background(), createSnapshotOptions)
}

// CreateSnapshotWithContext is an alternate form of the CreateSnapshot method which supports a Context parameter
func (vpc *VpcV1) CreateSnapshotWithContext(ctx context.Context, createSnapshotOptions *CreateSnapshotOptions) (result *Snapshot, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createSnapshotOptions, "createSnapshotOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createSnapshotOptions, "createSnapshotOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/snapshots`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createSnapshotOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateSnapshot")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createSnapshotOptions.SourceVolume != nil {
		body["source_volume"] = createSnapshotOptions.SourceVolume
	}
	if createSnapshotOptions.Name != nil {
		body["name"] = createSnapshotOptions.Name
	}
	if createSnapshotOptions.ResourceGroup != nil {
		body["resource_group"] = createSnapshotOptions.ResourceGroup
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSnapshot)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteSnapshot : Delete a snapshot
// This request deletes a snapshot. This operation cannot be reversed.
func (vpc *VpcV1) DeleteSnapshot(deleteSnapshotOptions *DeleteSnapshotOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteSnapshotWithContext(context.Background(), deleteSnapshotOptions)
}

// DeleteSnapshotWithContext is an alternate form of the DeleteSnapshot method which supports a Context parameter
func (vpc *VpcV1) DeleteSnapshotWithContext(ctx context.Context, deleteSnapshotOptions *DeleteSnapshotOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteSnapshotOptions, "deleteSnapshotOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteSnapshotOptions, "deleteSnapshotOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteSnapshotOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/snapshots/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteSnapshotOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteSnapshot")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetSnapshot : Retrieve a snapshot
// This request retrieves a single snapshot specified by the identifier in the URL.
func (vpc *VpcV1) GetSnapshot(getSnapshotOptions *GetSnapshotOptions) (result *Snapshot, response *core.DetailedResponse, err error) {
	return vpc.GetSnapshotWithContext(context.Background(), getSnapshotOptions)
}

// GetSnapshotWithContext is an alternate form of the GetSnapshot method which supports a Context parameter
func (vpc *VpcV1) GetSnapshotWithContext(ctx context.Context, getSnapshotOptions *GetSnapshotOptions) (result *Snapshot, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSnapshotOptions, "getSnapshotOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSnapshotOptions, "getSnapshotOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getSnapshotOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/snapshots/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSnapshotOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetSnapshot")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSnapshot)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateSnapshot : Update a snapshot
// This request updates a snapshot's name.
func (vpc *VpcV1) UpdateSnapshot(updateSnapshotOptions *UpdateSnapshotOptions) (result *Snapshot, response *core.DetailedResponse, err error) {
	return vpc.UpdateSnapshotWithContext(context.Background(), updateSnapshotOptions)
}

// UpdateSnapshotWithContext is an alternate form of the UpdateSnapshot method which supports a Context parameter
func (vpc *VpcV1) UpdateSnapshotWithContext(ctx context.Context, updateSnapshotOptions *UpdateSnapshotOptions) (result *Snapshot, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateSnapshotOptions, "updateSnapshotOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateSnapshotOptions, "updateSnapshotOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateSnapshotOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/snapshots/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateSnapshotOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateSnapshot")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateSnapshotOptions.SnapshotPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSnapshot)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListRegions : List all regions
// This request lists all regions. Each region is a separate geographic area that contains multiple isolated zones.
// Resources can be provisioned into one or more zones in a region. Each zone is isolated, but connected to other zones
// in the same region with low-latency and high-bandwidth links. Regions represent the top-level of fault isolation
// available. Resources deployed within a single region also benefit from the low latency afforded by geographic
// proximity.
func (vpc *VpcV1) ListRegions(listRegionsOptions *ListRegionsOptions) (result *RegionCollection, response *core.DetailedResponse, err error) {
	return vpc.ListRegionsWithContext(context.Background(), listRegionsOptions)
}

// ListRegionsWithContext is an alternate form of the ListRegions method which supports a Context parameter
func (vpc *VpcV1) ListRegionsWithContext(ctx context.Context, listRegionsOptions *ListRegionsOptions) (result *RegionCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listRegionsOptions, "listRegionsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/regions`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listRegionsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListRegions")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRegionCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetRegion : Retrieve a region
// This request retrieves a single region specified by the name in the URL.
func (vpc *VpcV1) GetRegion(getRegionOptions *GetRegionOptions) (result *Region, response *core.DetailedResponse, err error) {
	return vpc.GetRegionWithContext(context.Background(), getRegionOptions)
}

// GetRegionWithContext is an alternate form of the GetRegion method which supports a Context parameter
func (vpc *VpcV1) GetRegionWithContext(ctx context.Context, getRegionOptions *GetRegionOptions) (result *Region, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getRegionOptions, "getRegionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getRegionOptions, "getRegionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"name": *getRegionOptions.Name,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/regions/{name}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getRegionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetRegion")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalRegion)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListRegionZones : List all zones in a region
// This request lists all zones in a region. Zones represent logically-isolated data centers with high-bandwidth and
// low-latency interconnects to other zones in the same region. Faults in a zone do not affect other zones.
func (vpc *VpcV1) ListRegionZones(listRegionZonesOptions *ListRegionZonesOptions) (result *ZoneCollection, response *core.DetailedResponse, err error) {
	return vpc.ListRegionZonesWithContext(context.Background(), listRegionZonesOptions)
}

// ListRegionZonesWithContext is an alternate form of the ListRegionZones method which supports a Context parameter
func (vpc *VpcV1) ListRegionZonesWithContext(ctx context.Context, listRegionZonesOptions *ListRegionZonesOptions) (result *ZoneCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listRegionZonesOptions, "listRegionZonesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listRegionZonesOptions, "listRegionZonesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"region_name": *listRegionZonesOptions.RegionName,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/regions/{region_name}/zones`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listRegionZonesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListRegionZones")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalZoneCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetRegionZone : Retrieve a zone
// This request retrieves a single zone specified by the region and zone names in the URL.
func (vpc *VpcV1) GetRegionZone(getRegionZoneOptions *GetRegionZoneOptions) (result *Zone, response *core.DetailedResponse, err error) {
	return vpc.GetRegionZoneWithContext(context.Background(), getRegionZoneOptions)
}

// GetRegionZoneWithContext is an alternate form of the GetRegionZone method which supports a Context parameter
func (vpc *VpcV1) GetRegionZoneWithContext(ctx context.Context, getRegionZoneOptions *GetRegionZoneOptions) (result *Zone, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getRegionZoneOptions, "getRegionZoneOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getRegionZoneOptions, "getRegionZoneOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"region_name": *getRegionZoneOptions.RegionName,
		"name":        *getRegionZoneOptions.Name,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/regions/{region_name}/zones/{name}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getRegionZoneOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetRegionZone")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalZone)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListPublicGateways : List all public gateways
// This request lists all public gateways in the region. A public gateway is a virtual network device associated with a
// VPC, which allows access to the Internet. A public gateway resides in a zone and can be connected to subnets in the
// same zone only.
func (vpc *VpcV1) ListPublicGateways(listPublicGatewaysOptions *ListPublicGatewaysOptions) (result *PublicGatewayCollection, response *core.DetailedResponse, err error) {
	return vpc.ListPublicGatewaysWithContext(context.Background(), listPublicGatewaysOptions)
}

// ListPublicGatewaysWithContext is an alternate form of the ListPublicGateways method which supports a Context parameter
func (vpc *VpcV1) ListPublicGatewaysWithContext(ctx context.Context, listPublicGatewaysOptions *ListPublicGatewaysOptions) (result *PublicGatewayCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listPublicGatewaysOptions, "listPublicGatewaysOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/public_gateways`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listPublicGatewaysOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListPublicGateways")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listPublicGatewaysOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listPublicGatewaysOptions.Start))
	}
	if listPublicGatewaysOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listPublicGatewaysOptions.Limit))
	}
	if listPublicGatewaysOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listPublicGatewaysOptions.ResourceGroupID))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalPublicGatewayCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreatePublicGateway : Create a public gateway
// This request creates a new public gateway from a public gateway prototype object. For this to succeed, the VPC must
// not already have a public gateway in the specified zone.
//
// If a floating IP is provided, it must be unbound. If a floating IP is not provided, one will be created and bound to
// the public gateway. Once a public gateway has been created, its floating IP cannot be unbound. A public gateway must
// be explicitly attached to each subnet it will provide connectivity for.
func (vpc *VpcV1) CreatePublicGateway(createPublicGatewayOptions *CreatePublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	return vpc.CreatePublicGatewayWithContext(context.Background(), createPublicGatewayOptions)
}

// CreatePublicGatewayWithContext is an alternate form of the CreatePublicGateway method which supports a Context parameter
func (vpc *VpcV1) CreatePublicGatewayWithContext(ctx context.Context, createPublicGatewayOptions *CreatePublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createPublicGatewayOptions, "createPublicGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createPublicGatewayOptions, "createPublicGatewayOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/public_gateways`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createPublicGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreatePublicGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createPublicGatewayOptions.VPC != nil {
		body["vpc"] = createPublicGatewayOptions.VPC
	}
	if createPublicGatewayOptions.Zone != nil {
		body["zone"] = createPublicGatewayOptions.Zone
	}
	if createPublicGatewayOptions.FloatingIP != nil {
		body["floating_ip"] = createPublicGatewayOptions.FloatingIP
	}
	if createPublicGatewayOptions.Name != nil {
		body["name"] = createPublicGatewayOptions.Name
	}
	if createPublicGatewayOptions.ResourceGroup != nil {
		body["resource_group"] = createPublicGatewayOptions.ResourceGroup
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalPublicGateway)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeletePublicGateway : Delete a public gateway
// This request deletes a public gateway. This operation cannot be reversed. For this request to succeed, the public
// gateway must not be attached to any subnets. The public gateway's floating IP will be automatically unbound. If the
// floating IP was created when the public gateway was created, it will be deleted.
func (vpc *VpcV1) DeletePublicGateway(deletePublicGatewayOptions *DeletePublicGatewayOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeletePublicGatewayWithContext(context.Background(), deletePublicGatewayOptions)
}

// DeletePublicGatewayWithContext is an alternate form of the DeletePublicGateway method which supports a Context parameter
func (vpc *VpcV1) DeletePublicGatewayWithContext(ctx context.Context, deletePublicGatewayOptions *DeletePublicGatewayOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deletePublicGatewayOptions, "deletePublicGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deletePublicGatewayOptions, "deletePublicGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deletePublicGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/public_gateways/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deletePublicGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeletePublicGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetPublicGateway : Retrieve a public gateway
// This request retrieves a single public gateway specified by the identifier in the URL.
func (vpc *VpcV1) GetPublicGateway(getPublicGatewayOptions *GetPublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	return vpc.GetPublicGatewayWithContext(context.Background(), getPublicGatewayOptions)
}

// GetPublicGatewayWithContext is an alternate form of the GetPublicGateway method which supports a Context parameter
func (vpc *VpcV1) GetPublicGatewayWithContext(ctx context.Context, getPublicGatewayOptions *GetPublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getPublicGatewayOptions, "getPublicGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getPublicGatewayOptions, "getPublicGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getPublicGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/public_gateways/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getPublicGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetPublicGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalPublicGateway)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdatePublicGateway : Update a public gateway
// This request updates a public gateway's name.
func (vpc *VpcV1) UpdatePublicGateway(updatePublicGatewayOptions *UpdatePublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	return vpc.UpdatePublicGatewayWithContext(context.Background(), updatePublicGatewayOptions)
}

// UpdatePublicGatewayWithContext is an alternate form of the UpdatePublicGateway method which supports a Context parameter
func (vpc *VpcV1) UpdatePublicGatewayWithContext(ctx context.Context, updatePublicGatewayOptions *UpdatePublicGatewayOptions) (result *PublicGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updatePublicGatewayOptions, "updatePublicGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updatePublicGatewayOptions, "updatePublicGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updatePublicGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/public_gateways/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updatePublicGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdatePublicGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updatePublicGatewayOptions.PublicGatewayPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalPublicGateway)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListFloatingIps : List all floating IPs
// This request lists all floating IPs in the region. Floating IPs allow inbound and outbound traffic from the Internet
// to an instance.
func (vpc *VpcV1) ListFloatingIps(listFloatingIpsOptions *ListFloatingIpsOptions) (result *FloatingIPCollection, response *core.DetailedResponse, err error) {
	return vpc.ListFloatingIpsWithContext(context.Background(), listFloatingIpsOptions)
}

// ListFloatingIpsWithContext is an alternate form of the ListFloatingIps method which supports a Context parameter
func (vpc *VpcV1) ListFloatingIpsWithContext(ctx context.Context, listFloatingIpsOptions *ListFloatingIpsOptions) (result *FloatingIPCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listFloatingIpsOptions, "listFloatingIpsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/floating_ips`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listFloatingIpsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListFloatingIps")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listFloatingIpsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listFloatingIpsOptions.Start))
	}
	if listFloatingIpsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listFloatingIpsOptions.Limit))
	}
	if listFloatingIpsOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listFloatingIpsOptions.ResourceGroupID))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFloatingIPCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateFloatingIP : Reserve a floating IP
// This request reserves a new floating IP.
func (vpc *VpcV1) CreateFloatingIP(createFloatingIPOptions *CreateFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	return vpc.CreateFloatingIPWithContext(context.Background(), createFloatingIPOptions)
}

// CreateFloatingIPWithContext is an alternate form of the CreateFloatingIP method which supports a Context parameter
func (vpc *VpcV1) CreateFloatingIPWithContext(ctx context.Context, createFloatingIPOptions *CreateFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createFloatingIPOptions, "createFloatingIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createFloatingIPOptions, "createFloatingIPOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/floating_ips`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createFloatingIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateFloatingIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(createFloatingIPOptions.FloatingIPPrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFloatingIP)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteFloatingIP : Release a floating IP
// This request disassociates (if associated) and releases a floating IP. This operation cannot be reversed. For this
// request to succeed, the floating IP must not be required by another resource, such as a public gateway.
func (vpc *VpcV1) DeleteFloatingIP(deleteFloatingIPOptions *DeleteFloatingIPOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteFloatingIPWithContext(context.Background(), deleteFloatingIPOptions)
}

// DeleteFloatingIPWithContext is an alternate form of the DeleteFloatingIP method which supports a Context parameter
func (vpc *VpcV1) DeleteFloatingIPWithContext(ctx context.Context, deleteFloatingIPOptions *DeleteFloatingIPOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteFloatingIPOptions, "deleteFloatingIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteFloatingIPOptions, "deleteFloatingIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteFloatingIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/floating_ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteFloatingIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteFloatingIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetFloatingIP : Retrieve a floating IP
// This request retrieves a single floating IP specified by the identifier in the URL.
func (vpc *VpcV1) GetFloatingIP(getFloatingIPOptions *GetFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	return vpc.GetFloatingIPWithContext(context.Background(), getFloatingIPOptions)
}

// GetFloatingIPWithContext is an alternate form of the GetFloatingIP method which supports a Context parameter
func (vpc *VpcV1) GetFloatingIPWithContext(ctx context.Context, getFloatingIPOptions *GetFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getFloatingIPOptions, "getFloatingIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getFloatingIPOptions, "getFloatingIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getFloatingIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/floating_ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getFloatingIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetFloatingIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFloatingIP)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateFloatingIP : Update a floating IP
// This request updates a floating IP's name and/or target.
func (vpc *VpcV1) UpdateFloatingIP(updateFloatingIPOptions *UpdateFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	return vpc.UpdateFloatingIPWithContext(context.Background(), updateFloatingIPOptions)
}

// UpdateFloatingIPWithContext is an alternate form of the UpdateFloatingIP method which supports a Context parameter
func (vpc *VpcV1) UpdateFloatingIPWithContext(ctx context.Context, updateFloatingIPOptions *UpdateFloatingIPOptions) (result *FloatingIP, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateFloatingIPOptions, "updateFloatingIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateFloatingIPOptions, "updateFloatingIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateFloatingIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/floating_ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateFloatingIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateFloatingIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateFloatingIPOptions.FloatingIPPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFloatingIP)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListNetworkAcls : List all network ACLs
// This request lists all network ACLs in the region. A network ACL defines a set of packet filtering (5-tuple) rules
// for all traffic in and out of a subnet. Both allow and deny rules can be defined, and rules are stateless such that
// reverse traffic in response to allowed traffic is not automatically permitted.
func (vpc *VpcV1) ListNetworkAcls(listNetworkAclsOptions *ListNetworkAclsOptions) (result *NetworkACLCollection, response *core.DetailedResponse, err error) {
	return vpc.ListNetworkAclsWithContext(context.Background(), listNetworkAclsOptions)
}

// ListNetworkAclsWithContext is an alternate form of the ListNetworkAcls method which supports a Context parameter
func (vpc *VpcV1) ListNetworkAclsWithContext(ctx context.Context, listNetworkAclsOptions *ListNetworkAclsOptions) (result *NetworkACLCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listNetworkAclsOptions, "listNetworkAclsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/network_acls`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listNetworkAclsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListNetworkAcls")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listNetworkAclsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listNetworkAclsOptions.Start))
	}
	if listNetworkAclsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listNetworkAclsOptions.Limit))
	}
	if listNetworkAclsOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listNetworkAclsOptions.ResourceGroupID))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACLCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateNetworkACL : Create a network ACL
// This request creates a new network ACL from a network ACL prototype object. The prototype object is structured in the
// same way as a retrieved network ACL, and contains the information necessary to create the new network ACL.
func (vpc *VpcV1) CreateNetworkACL(createNetworkACLOptions *CreateNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	return vpc.CreateNetworkACLWithContext(context.Background(), createNetworkACLOptions)
}

// CreateNetworkACLWithContext is an alternate form of the CreateNetworkACL method which supports a Context parameter
func (vpc *VpcV1) CreateNetworkACLWithContext(ctx context.Context, createNetworkACLOptions *CreateNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(createNetworkACLOptions, "createNetworkACLOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/network_acls`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createNetworkACLOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateNetworkACL")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	if createNetworkACLOptions.NetworkACLPrototype != nil {
		_, err = builder.SetBodyContentJSON(createNetworkACLOptions.NetworkACLPrototype)
		if err != nil {
			return
		}
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACL)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteNetworkACL : Delete a network ACL
// This request deletes a network ACL. This operation cannot be reversed. For this request to succeed, the network ACL
// must not be the default network ACL for any VPCs, and the network ACL must not be attached to any subnets.
func (vpc *VpcV1) DeleteNetworkACL(deleteNetworkACLOptions *DeleteNetworkACLOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteNetworkACLWithContext(context.Background(), deleteNetworkACLOptions)
}

// DeleteNetworkACLWithContext is an alternate form of the DeleteNetworkACL method which supports a Context parameter
func (vpc *VpcV1) DeleteNetworkACLWithContext(ctx context.Context, deleteNetworkACLOptions *DeleteNetworkACLOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteNetworkACLOptions, "deleteNetworkACLOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteNetworkACLOptions, "deleteNetworkACLOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteNetworkACLOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/network_acls/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteNetworkACLOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteNetworkACL")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetNetworkACL : Retrieve a network ACL
// This request retrieves a single network ACL specified by the identifier in the URL.
func (vpc *VpcV1) GetNetworkACL(getNetworkACLOptions *GetNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	return vpc.GetNetworkACLWithContext(context.Background(), getNetworkACLOptions)
}

// GetNetworkACLWithContext is an alternate form of the GetNetworkACL method which supports a Context parameter
func (vpc *VpcV1) GetNetworkACLWithContext(ctx context.Context, getNetworkACLOptions *GetNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getNetworkACLOptions, "getNetworkACLOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getNetworkACLOptions, "getNetworkACLOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getNetworkACLOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/network_acls/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getNetworkACLOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetNetworkACL")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACL)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateNetworkACL : Update a network ACL
// This request updates a network ACL's name.
func (vpc *VpcV1) UpdateNetworkACL(updateNetworkACLOptions *UpdateNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	return vpc.UpdateNetworkACLWithContext(context.Background(), updateNetworkACLOptions)
}

// UpdateNetworkACLWithContext is an alternate form of the UpdateNetworkACL method which supports a Context parameter
func (vpc *VpcV1) UpdateNetworkACLWithContext(ctx context.Context, updateNetworkACLOptions *UpdateNetworkACLOptions) (result *NetworkACL, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateNetworkACLOptions, "updateNetworkACLOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateNetworkACLOptions, "updateNetworkACLOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateNetworkACLOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/network_acls/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateNetworkACLOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateNetworkACL")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateNetworkACLOptions.NetworkACLPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACL)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListNetworkACLRules : List all rules for a network ACL
// This request lists all rules for a network ACL. These rules can allow or deny traffic between a source CIDR block and
// a destination CIDR block over a particular protocol and port range.
func (vpc *VpcV1) ListNetworkACLRules(listNetworkACLRulesOptions *ListNetworkACLRulesOptions) (result *NetworkACLRuleCollection, response *core.DetailedResponse, err error) {
	return vpc.ListNetworkACLRulesWithContext(context.Background(), listNetworkACLRulesOptions)
}

// ListNetworkACLRulesWithContext is an alternate form of the ListNetworkACLRules method which supports a Context parameter
func (vpc *VpcV1) ListNetworkACLRulesWithContext(ctx context.Context, listNetworkACLRulesOptions *ListNetworkACLRulesOptions) (result *NetworkACLRuleCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listNetworkACLRulesOptions, "listNetworkACLRulesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listNetworkACLRulesOptions, "listNetworkACLRulesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"network_acl_id": *listNetworkACLRulesOptions.NetworkACLID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/network_acls/{network_acl_id}/rules`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listNetworkACLRulesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListNetworkACLRules")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listNetworkACLRulesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listNetworkACLRulesOptions.Start))
	}
	if listNetworkACLRulesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listNetworkACLRulesOptions.Limit))
	}
	if listNetworkACLRulesOptions.Direction != nil {
		builder.AddQuery("direction", fmt.Sprint(*listNetworkACLRulesOptions.Direction))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACLRuleCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateNetworkACLRule : Create a rule for a network ACL
// This request creates a new rule from a network ACL rule prototype object. The prototype object is structured in the
// same way as a retrieved rule, and contains the information necessary to create the new rule.
func (vpc *VpcV1) CreateNetworkACLRule(createNetworkACLRuleOptions *CreateNetworkACLRuleOptions) (result NetworkACLRuleIntf, response *core.DetailedResponse, err error) {
	return vpc.CreateNetworkACLRuleWithContext(context.Background(), createNetworkACLRuleOptions)
}

// CreateNetworkACLRuleWithContext is an alternate form of the CreateNetworkACLRule method which supports a Context parameter
func (vpc *VpcV1) CreateNetworkACLRuleWithContext(ctx context.Context, createNetworkACLRuleOptions *CreateNetworkACLRuleOptions) (result NetworkACLRuleIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createNetworkACLRuleOptions, "createNetworkACLRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createNetworkACLRuleOptions, "createNetworkACLRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"network_acl_id": *createNetworkACLRuleOptions.NetworkACLID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/network_acls/{network_acl_id}/rules`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createNetworkACLRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateNetworkACLRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(createNetworkACLRuleOptions.NetworkACLRulePrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACLRule)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteNetworkACLRule : Delete a network ACL rule
// This request deletes a rule. This operation cannot be reversed.
func (vpc *VpcV1) DeleteNetworkACLRule(deleteNetworkACLRuleOptions *DeleteNetworkACLRuleOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteNetworkACLRuleWithContext(context.Background(), deleteNetworkACLRuleOptions)
}

// DeleteNetworkACLRuleWithContext is an alternate form of the DeleteNetworkACLRule method which supports a Context parameter
func (vpc *VpcV1) DeleteNetworkACLRuleWithContext(ctx context.Context, deleteNetworkACLRuleOptions *DeleteNetworkACLRuleOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteNetworkACLRuleOptions, "deleteNetworkACLRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteNetworkACLRuleOptions, "deleteNetworkACLRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"network_acl_id": *deleteNetworkACLRuleOptions.NetworkACLID,
		"id":             *deleteNetworkACLRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/network_acls/{network_acl_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteNetworkACLRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteNetworkACLRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetNetworkACLRule : Retrieve a network ACL rule
// This request retrieves a single rule specified by the identifier in the URL.
func (vpc *VpcV1) GetNetworkACLRule(getNetworkACLRuleOptions *GetNetworkACLRuleOptions) (result NetworkACLRuleIntf, response *core.DetailedResponse, err error) {
	return vpc.GetNetworkACLRuleWithContext(context.Background(), getNetworkACLRuleOptions)
}

// GetNetworkACLRuleWithContext is an alternate form of the GetNetworkACLRule method which supports a Context parameter
func (vpc *VpcV1) GetNetworkACLRuleWithContext(ctx context.Context, getNetworkACLRuleOptions *GetNetworkACLRuleOptions) (result NetworkACLRuleIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getNetworkACLRuleOptions, "getNetworkACLRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getNetworkACLRuleOptions, "getNetworkACLRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"network_acl_id": *getNetworkACLRuleOptions.NetworkACLID,
		"id":             *getNetworkACLRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/network_acls/{network_acl_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getNetworkACLRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetNetworkACLRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACLRule)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateNetworkACLRule : Update a network ACL rule
// This request updates a rule with the information in a provided rule patch. The rule patch object contains only the
// information to be updated. The request will fail if the information is not applicable to the rule's protocol.
func (vpc *VpcV1) UpdateNetworkACLRule(updateNetworkACLRuleOptions *UpdateNetworkACLRuleOptions) (result NetworkACLRuleIntf, response *core.DetailedResponse, err error) {
	return vpc.UpdateNetworkACLRuleWithContext(context.Background(), updateNetworkACLRuleOptions)
}

// UpdateNetworkACLRuleWithContext is an alternate form of the UpdateNetworkACLRule method which supports a Context parameter
func (vpc *VpcV1) UpdateNetworkACLRuleWithContext(ctx context.Context, updateNetworkACLRuleOptions *UpdateNetworkACLRuleOptions) (result NetworkACLRuleIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateNetworkACLRuleOptions, "updateNetworkACLRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateNetworkACLRuleOptions, "updateNetworkACLRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"network_acl_id": *updateNetworkACLRuleOptions.NetworkACLID,
		"id":             *updateNetworkACLRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/network_acls/{network_acl_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateNetworkACLRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateNetworkACLRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateNetworkACLRuleOptions.NetworkACLRulePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkACLRule)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListSecurityGroups : List all security groups
// This request lists all security groups in the region. Security groups provide a way to apply IP filtering rules to
// instances in the associated VPC. With security groups, all traffic is denied by default, and rules added to security
// groups define which traffic the security group permits. Security group rules are stateful such that reverse traffic
// in response to allowed traffic is automatically permitted.
func (vpc *VpcV1) ListSecurityGroups(listSecurityGroupsOptions *ListSecurityGroupsOptions) (result *SecurityGroupCollection, response *core.DetailedResponse, err error) {
	return vpc.ListSecurityGroupsWithContext(context.Background(), listSecurityGroupsOptions)
}

// ListSecurityGroupsWithContext is an alternate form of the ListSecurityGroups method which supports a Context parameter
func (vpc *VpcV1) ListSecurityGroupsWithContext(ctx context.Context, listSecurityGroupsOptions *ListSecurityGroupsOptions) (result *SecurityGroupCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listSecurityGroupsOptions, "listSecurityGroupsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listSecurityGroupsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListSecurityGroups")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listSecurityGroupsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listSecurityGroupsOptions.Start))
	}
	if listSecurityGroupsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listSecurityGroupsOptions.Limit))
	}
	if listSecurityGroupsOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listSecurityGroupsOptions.ResourceGroupID))
	}
	if listSecurityGroupsOptions.VPCID != nil {
		builder.AddQuery("vpc.id", fmt.Sprint(*listSecurityGroupsOptions.VPCID))
	}
	if listSecurityGroupsOptions.VPCCRN != nil {
		builder.AddQuery("vpc.crn", fmt.Sprint(*listSecurityGroupsOptions.VPCCRN))
	}
	if listSecurityGroupsOptions.VPCName != nil {
		builder.AddQuery("vpc.name", fmt.Sprint(*listSecurityGroupsOptions.VPCName))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroupCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateSecurityGroup : Create a security group
// This request creates a new security group from a security group prototype object. The prototype object is structured
// in the same way as a retrieved security group, and contains the information necessary to create the new security
// group. If security group rules are included in the prototype object, those rules will be added to the security group.
// Each security group is scoped to one VPC. Only network interfaces on instances in that VPC can be added to the
// security group.
func (vpc *VpcV1) CreateSecurityGroup(createSecurityGroupOptions *CreateSecurityGroupOptions) (result *SecurityGroup, response *core.DetailedResponse, err error) {
	return vpc.CreateSecurityGroupWithContext(context.Background(), createSecurityGroupOptions)
}

// CreateSecurityGroupWithContext is an alternate form of the CreateSecurityGroup method which supports a Context parameter
func (vpc *VpcV1) CreateSecurityGroupWithContext(ctx context.Context, createSecurityGroupOptions *CreateSecurityGroupOptions) (result *SecurityGroup, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createSecurityGroupOptions, "createSecurityGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createSecurityGroupOptions, "createSecurityGroupOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createSecurityGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateSecurityGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createSecurityGroupOptions.VPC != nil {
		body["vpc"] = createSecurityGroupOptions.VPC
	}
	if createSecurityGroupOptions.Name != nil {
		body["name"] = createSecurityGroupOptions.Name
	}
	if createSecurityGroupOptions.ResourceGroup != nil {
		body["resource_group"] = createSecurityGroupOptions.ResourceGroup
	}
	if createSecurityGroupOptions.Rules != nil {
		body["rules"] = createSecurityGroupOptions.Rules
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroup)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteSecurityGroup : Delete a security group
// This request deletes a security group. A security group cannot be deleted if it is referenced by any network
// interfaces or other security group rules. Additionally, a VPC's default security group cannot be deleted. This
// operation cannot be reversed.
func (vpc *VpcV1) DeleteSecurityGroup(deleteSecurityGroupOptions *DeleteSecurityGroupOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteSecurityGroupWithContext(context.Background(), deleteSecurityGroupOptions)
}

// DeleteSecurityGroupWithContext is an alternate form of the DeleteSecurityGroup method which supports a Context parameter
func (vpc *VpcV1) DeleteSecurityGroupWithContext(ctx context.Context, deleteSecurityGroupOptions *DeleteSecurityGroupOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteSecurityGroupOptions, "deleteSecurityGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteSecurityGroupOptions, "deleteSecurityGroupOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteSecurityGroupOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteSecurityGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteSecurityGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetSecurityGroup : Retrieve a security group
// This request retrieves a single security group specified by the identifier in the URL path.
func (vpc *VpcV1) GetSecurityGroup(getSecurityGroupOptions *GetSecurityGroupOptions) (result *SecurityGroup, response *core.DetailedResponse, err error) {
	return vpc.GetSecurityGroupWithContext(context.Background(), getSecurityGroupOptions)
}

// GetSecurityGroupWithContext is an alternate form of the GetSecurityGroup method which supports a Context parameter
func (vpc *VpcV1) GetSecurityGroupWithContext(ctx context.Context, getSecurityGroupOptions *GetSecurityGroupOptions) (result *SecurityGroup, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSecurityGroupOptions, "getSecurityGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSecurityGroupOptions, "getSecurityGroupOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getSecurityGroupOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSecurityGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetSecurityGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroup)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateSecurityGroup : Update a security group
// This request updates a security group with the information provided in a security group patch object. The security
// group patch object is structured in the same way as a retrieved security group and contains only the information to
// be updated.
func (vpc *VpcV1) UpdateSecurityGroup(updateSecurityGroupOptions *UpdateSecurityGroupOptions) (result *SecurityGroup, response *core.DetailedResponse, err error) {
	return vpc.UpdateSecurityGroupWithContext(context.Background(), updateSecurityGroupOptions)
}

// UpdateSecurityGroupWithContext is an alternate form of the UpdateSecurityGroup method which supports a Context parameter
func (vpc *VpcV1) UpdateSecurityGroupWithContext(ctx context.Context, updateSecurityGroupOptions *UpdateSecurityGroupOptions) (result *SecurityGroup, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateSecurityGroupOptions, "updateSecurityGroupOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateSecurityGroupOptions, "updateSecurityGroupOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateSecurityGroupOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateSecurityGroupOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateSecurityGroup")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateSecurityGroupOptions.SecurityGroupPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroup)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListSecurityGroupNetworkInterfaces : List all network interfaces associated with a security group
// This request lists all network interfaces associated with a security group, to which the rules in the security group
// are applied.
func (vpc *VpcV1) ListSecurityGroupNetworkInterfaces(listSecurityGroupNetworkInterfacesOptions *ListSecurityGroupNetworkInterfacesOptions) (result *NetworkInterfaceCollection, response *core.DetailedResponse, err error) {
	return vpc.ListSecurityGroupNetworkInterfacesWithContext(context.Background(), listSecurityGroupNetworkInterfacesOptions)
}

// ListSecurityGroupNetworkInterfacesWithContext is an alternate form of the ListSecurityGroupNetworkInterfaces method which supports a Context parameter
func (vpc *VpcV1) ListSecurityGroupNetworkInterfacesWithContext(ctx context.Context, listSecurityGroupNetworkInterfacesOptions *ListSecurityGroupNetworkInterfacesOptions) (result *NetworkInterfaceCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listSecurityGroupNetworkInterfacesOptions, "listSecurityGroupNetworkInterfacesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listSecurityGroupNetworkInterfacesOptions, "listSecurityGroupNetworkInterfacesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *listSecurityGroupNetworkInterfacesOptions.SecurityGroupID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{security_group_id}/network_interfaces`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listSecurityGroupNetworkInterfacesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListSecurityGroupNetworkInterfaces")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listSecurityGroupNetworkInterfacesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listSecurityGroupNetworkInterfacesOptions.Start))
	}
	if listSecurityGroupNetworkInterfacesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listSecurityGroupNetworkInterfacesOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkInterfaceCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// RemoveSecurityGroupNetworkInterface : Remove a network interface from a security group
// This request removes a network interface from a security group. Security groups are stateful, so any changes to a
// network interface's security groups are applied to new connections. Existing connections are not affected. If the
// network interface being removed has no other security groups, it will be attached to the VPC's default security
// group.
func (vpc *VpcV1) RemoveSecurityGroupNetworkInterface(removeSecurityGroupNetworkInterfaceOptions *RemoveSecurityGroupNetworkInterfaceOptions) (response *core.DetailedResponse, err error) {
	return vpc.RemoveSecurityGroupNetworkInterfaceWithContext(context.Background(), removeSecurityGroupNetworkInterfaceOptions)
}

// RemoveSecurityGroupNetworkInterfaceWithContext is an alternate form of the RemoveSecurityGroupNetworkInterface method which supports a Context parameter
func (vpc *VpcV1) RemoveSecurityGroupNetworkInterfaceWithContext(ctx context.Context, removeSecurityGroupNetworkInterfaceOptions *RemoveSecurityGroupNetworkInterfaceOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(removeSecurityGroupNetworkInterfaceOptions, "removeSecurityGroupNetworkInterfaceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(removeSecurityGroupNetworkInterfaceOptions, "removeSecurityGroupNetworkInterfaceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *removeSecurityGroupNetworkInterfaceOptions.SecurityGroupID,
		"id":                *removeSecurityGroupNetworkInterfaceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{security_group_id}/network_interfaces/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range removeSecurityGroupNetworkInterfaceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "RemoveSecurityGroupNetworkInterface")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetSecurityGroupNetworkInterface : Retrieve a network interface in a security group
// This request retrieves a single network interface specified by the identifier in the URL path. The network interface
// must be an existing member of the security group.
func (vpc *VpcV1) GetSecurityGroupNetworkInterface(getSecurityGroupNetworkInterfaceOptions *GetSecurityGroupNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	return vpc.GetSecurityGroupNetworkInterfaceWithContext(context.Background(), getSecurityGroupNetworkInterfaceOptions)
}

// GetSecurityGroupNetworkInterfaceWithContext is an alternate form of the GetSecurityGroupNetworkInterface method which supports a Context parameter
func (vpc *VpcV1) GetSecurityGroupNetworkInterfaceWithContext(ctx context.Context, getSecurityGroupNetworkInterfaceOptions *GetSecurityGroupNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSecurityGroupNetworkInterfaceOptions, "getSecurityGroupNetworkInterfaceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSecurityGroupNetworkInterfaceOptions, "getSecurityGroupNetworkInterfaceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *getSecurityGroupNetworkInterfaceOptions.SecurityGroupID,
		"id":                *getSecurityGroupNetworkInterfaceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{security_group_id}/network_interfaces/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSecurityGroupNetworkInterfaceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetSecurityGroupNetworkInterface")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkInterface)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// AddSecurityGroupNetworkInterface : Add a network interface to a security group
// This request adds an existing network interface to an existing security group. When a network interface is added to a
// security group, the security group rules are applied to the network interface. A request body is not required, and if
// supplied, is ignored.
func (vpc *VpcV1) AddSecurityGroupNetworkInterface(addSecurityGroupNetworkInterfaceOptions *AddSecurityGroupNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	return vpc.AddSecurityGroupNetworkInterfaceWithContext(context.Background(), addSecurityGroupNetworkInterfaceOptions)
}

// AddSecurityGroupNetworkInterfaceWithContext is an alternate form of the AddSecurityGroupNetworkInterface method which supports a Context parameter
func (vpc *VpcV1) AddSecurityGroupNetworkInterfaceWithContext(ctx context.Context, addSecurityGroupNetworkInterfaceOptions *AddSecurityGroupNetworkInterfaceOptions) (result *NetworkInterface, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(addSecurityGroupNetworkInterfaceOptions, "addSecurityGroupNetworkInterfaceOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(addSecurityGroupNetworkInterfaceOptions, "addSecurityGroupNetworkInterfaceOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *addSecurityGroupNetworkInterfaceOptions.SecurityGroupID,
		"id":                *addSecurityGroupNetworkInterfaceOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{security_group_id}/network_interfaces/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range addSecurityGroupNetworkInterfaceOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "AddSecurityGroupNetworkInterface")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalNetworkInterface)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListSecurityGroupRules : List all rules in a security group
// This request lists all rules in a security group. These rules define what traffic the security group permits.
// Security group rules are stateful, such that reverse traffic in response to allowed traffic is automatically
// permitted.
func (vpc *VpcV1) ListSecurityGroupRules(listSecurityGroupRulesOptions *ListSecurityGroupRulesOptions) (result *SecurityGroupRuleCollection, response *core.DetailedResponse, err error) {
	return vpc.ListSecurityGroupRulesWithContext(context.Background(), listSecurityGroupRulesOptions)
}

// ListSecurityGroupRulesWithContext is an alternate form of the ListSecurityGroupRules method which supports a Context parameter
func (vpc *VpcV1) ListSecurityGroupRulesWithContext(ctx context.Context, listSecurityGroupRulesOptions *ListSecurityGroupRulesOptions) (result *SecurityGroupRuleCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listSecurityGroupRulesOptions, "listSecurityGroupRulesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listSecurityGroupRulesOptions, "listSecurityGroupRulesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *listSecurityGroupRulesOptions.SecurityGroupID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{security_group_id}/rules`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listSecurityGroupRulesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListSecurityGroupRules")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroupRuleCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateSecurityGroupRule : Create a rule for a security group
// This request creates a new security group rule from a security group rule prototype object. The prototype object is
// structured in the same way as a retrieved security group rule and contains the information necessary to create the
// rule. As part of creating a new rule in a security group, the rule is applied to all the networking interfaces in the
// security group. Rules specify which IP traffic a security group should allow. Security group rules are stateful, such
// that reverse traffic in response to allowed traffic is automatically permitted. A rule allowing inbound TCP traffic
// on port 80 also allows outbound TCP traffic on port 80 without the need for an additional rule.
func (vpc *VpcV1) CreateSecurityGroupRule(createSecurityGroupRuleOptions *CreateSecurityGroupRuleOptions) (result SecurityGroupRuleIntf, response *core.DetailedResponse, err error) {
	return vpc.CreateSecurityGroupRuleWithContext(context.Background(), createSecurityGroupRuleOptions)
}

// CreateSecurityGroupRuleWithContext is an alternate form of the CreateSecurityGroupRule method which supports a Context parameter
func (vpc *VpcV1) CreateSecurityGroupRuleWithContext(ctx context.Context, createSecurityGroupRuleOptions *CreateSecurityGroupRuleOptions) (result SecurityGroupRuleIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createSecurityGroupRuleOptions, "createSecurityGroupRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createSecurityGroupRuleOptions, "createSecurityGroupRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *createSecurityGroupRuleOptions.SecurityGroupID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{security_group_id}/rules`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createSecurityGroupRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateSecurityGroupRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(createSecurityGroupRuleOptions.SecurityGroupRulePrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroupRule)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteSecurityGroupRule : Delete a security group rule
// This request deletes a security group rule. This operation cannot be reversed. Removing a security group rule will
// not end existing connections allowed by that rule.
func (vpc *VpcV1) DeleteSecurityGroupRule(deleteSecurityGroupRuleOptions *DeleteSecurityGroupRuleOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteSecurityGroupRuleWithContext(context.Background(), deleteSecurityGroupRuleOptions)
}

// DeleteSecurityGroupRuleWithContext is an alternate form of the DeleteSecurityGroupRule method which supports a Context parameter
func (vpc *VpcV1) DeleteSecurityGroupRuleWithContext(ctx context.Context, deleteSecurityGroupRuleOptions *DeleteSecurityGroupRuleOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteSecurityGroupRuleOptions, "deleteSecurityGroupRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteSecurityGroupRuleOptions, "deleteSecurityGroupRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *deleteSecurityGroupRuleOptions.SecurityGroupID,
		"id":                *deleteSecurityGroupRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{security_group_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteSecurityGroupRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteSecurityGroupRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetSecurityGroupRule : Retrieve a security group rule
// This request retrieves a single security group rule specified by the identifier in the URL path.
func (vpc *VpcV1) GetSecurityGroupRule(getSecurityGroupRuleOptions *GetSecurityGroupRuleOptions) (result SecurityGroupRuleIntf, response *core.DetailedResponse, err error) {
	return vpc.GetSecurityGroupRuleWithContext(context.Background(), getSecurityGroupRuleOptions)
}

// GetSecurityGroupRuleWithContext is an alternate form of the GetSecurityGroupRule method which supports a Context parameter
func (vpc *VpcV1) GetSecurityGroupRuleWithContext(ctx context.Context, getSecurityGroupRuleOptions *GetSecurityGroupRuleOptions) (result SecurityGroupRuleIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSecurityGroupRuleOptions, "getSecurityGroupRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSecurityGroupRuleOptions, "getSecurityGroupRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *getSecurityGroupRuleOptions.SecurityGroupID,
		"id":                *getSecurityGroupRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{security_group_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSecurityGroupRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetSecurityGroupRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroupRule)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateSecurityGroupRule : Update a security group rule
// This request updates a security group rule with the information in a provided rule patch object. The rule patch
// object contains only the information to be updated. The request will fail if the information is not applicable to the
// rule's protocol.
func (vpc *VpcV1) UpdateSecurityGroupRule(updateSecurityGroupRuleOptions *UpdateSecurityGroupRuleOptions) (result SecurityGroupRuleIntf, response *core.DetailedResponse, err error) {
	return vpc.UpdateSecurityGroupRuleWithContext(context.Background(), updateSecurityGroupRuleOptions)
}

// UpdateSecurityGroupRuleWithContext is an alternate form of the UpdateSecurityGroupRule method which supports a Context parameter
func (vpc *VpcV1) UpdateSecurityGroupRuleWithContext(ctx context.Context, updateSecurityGroupRuleOptions *UpdateSecurityGroupRuleOptions) (result SecurityGroupRuleIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateSecurityGroupRuleOptions, "updateSecurityGroupRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateSecurityGroupRuleOptions, "updateSecurityGroupRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *updateSecurityGroupRuleOptions.SecurityGroupID,
		"id":                *updateSecurityGroupRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{security_group_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateSecurityGroupRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateSecurityGroupRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateSecurityGroupRuleOptions.SecurityGroupRulePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroupRule)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListSecurityGroupTargets : List all targets associated with a security group
// This request lists all targets associated with a security group, to which the rules in the security group are
// applied.
func (vpc *VpcV1) ListSecurityGroupTargets(listSecurityGroupTargetsOptions *ListSecurityGroupTargetsOptions) (result *SecurityGroupTargetCollection, response *core.DetailedResponse, err error) {
	return vpc.ListSecurityGroupTargetsWithContext(context.Background(), listSecurityGroupTargetsOptions)
}

// ListSecurityGroupTargetsWithContext is an alternate form of the ListSecurityGroupTargets method which supports a Context parameter
func (vpc *VpcV1) ListSecurityGroupTargetsWithContext(ctx context.Context, listSecurityGroupTargetsOptions *ListSecurityGroupTargetsOptions) (result *SecurityGroupTargetCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listSecurityGroupTargetsOptions, "listSecurityGroupTargetsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listSecurityGroupTargetsOptions, "listSecurityGroupTargetsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *listSecurityGroupTargetsOptions.SecurityGroupID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{security_group_id}/targets`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listSecurityGroupTargetsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListSecurityGroupTargets")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listSecurityGroupTargetsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listSecurityGroupTargetsOptions.Start))
	}
	if listSecurityGroupTargetsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listSecurityGroupTargetsOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroupTargetCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteSecurityGroupTargetBinding : Remove a target from a security group
// This request removes a target from a security group. For this request to succeed, the target must be attached to at
// least one other security group.  The supplied target identifier can be:
//
// - A network interface identifier
// - An application load balancer identifier
//
// Security groups are stateful, so any changes to a target's security groups are applied to new connections. Existing
// connections are not affected.
func (vpc *VpcV1) DeleteSecurityGroupTargetBinding(deleteSecurityGroupTargetBindingOptions *DeleteSecurityGroupTargetBindingOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteSecurityGroupTargetBindingWithContext(context.Background(), deleteSecurityGroupTargetBindingOptions)
}

// DeleteSecurityGroupTargetBindingWithContext is an alternate form of the DeleteSecurityGroupTargetBinding method which supports a Context parameter
func (vpc *VpcV1) DeleteSecurityGroupTargetBindingWithContext(ctx context.Context, deleteSecurityGroupTargetBindingOptions *DeleteSecurityGroupTargetBindingOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteSecurityGroupTargetBindingOptions, "deleteSecurityGroupTargetBindingOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteSecurityGroupTargetBindingOptions, "deleteSecurityGroupTargetBindingOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *deleteSecurityGroupTargetBindingOptions.SecurityGroupID,
		"id":                *deleteSecurityGroupTargetBindingOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{security_group_id}/targets/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteSecurityGroupTargetBindingOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteSecurityGroupTargetBinding")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetSecurityGroupTarget : Retrieve a security group target
// This request retrieves a single target specified by the identifier in the URL path. The target must be an existing
// target of the security group.
func (vpc *VpcV1) GetSecurityGroupTarget(getSecurityGroupTargetOptions *GetSecurityGroupTargetOptions) (result SecurityGroupTargetReferenceIntf, response *core.DetailedResponse, err error) {
	return vpc.GetSecurityGroupTargetWithContext(context.Background(), getSecurityGroupTargetOptions)
}

// GetSecurityGroupTargetWithContext is an alternate form of the GetSecurityGroupTarget method which supports a Context parameter
func (vpc *VpcV1) GetSecurityGroupTargetWithContext(ctx context.Context, getSecurityGroupTargetOptions *GetSecurityGroupTargetOptions) (result SecurityGroupTargetReferenceIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getSecurityGroupTargetOptions, "getSecurityGroupTargetOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getSecurityGroupTargetOptions, "getSecurityGroupTargetOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *getSecurityGroupTargetOptions.SecurityGroupID,
		"id":                *getSecurityGroupTargetOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{security_group_id}/targets/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getSecurityGroupTargetOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetSecurityGroupTarget")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroupTargetReference)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateSecurityGroupTargetBinding : Add a target to a security group
// This request adds a resource to an existing security group. The supplied target identifier can be:
//
// - A network interface identifier
// - An application load balancer identifier
//
// When a target is added to a security group, the security group rules are applied to the target. A request body is not
// required, and if supplied, is ignored.
func (vpc *VpcV1) CreateSecurityGroupTargetBinding(createSecurityGroupTargetBindingOptions *CreateSecurityGroupTargetBindingOptions) (result SecurityGroupTargetReferenceIntf, response *core.DetailedResponse, err error) {
	return vpc.CreateSecurityGroupTargetBindingWithContext(context.Background(), createSecurityGroupTargetBindingOptions)
}

// CreateSecurityGroupTargetBindingWithContext is an alternate form of the CreateSecurityGroupTargetBinding method which supports a Context parameter
func (vpc *VpcV1) CreateSecurityGroupTargetBindingWithContext(ctx context.Context, createSecurityGroupTargetBindingOptions *CreateSecurityGroupTargetBindingOptions) (result SecurityGroupTargetReferenceIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createSecurityGroupTargetBindingOptions, "createSecurityGroupTargetBindingOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createSecurityGroupTargetBindingOptions, "createSecurityGroupTargetBindingOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"security_group_id": *createSecurityGroupTargetBindingOptions.SecurityGroupID,
		"id":                *createSecurityGroupTargetBindingOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/security_groups/{security_group_id}/targets/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createSecurityGroupTargetBindingOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateSecurityGroupTargetBinding")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalSecurityGroupTargetReference)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListIkePolicies : List all IKE policies
// This request lists all IKE policies in the region.
func (vpc *VpcV1) ListIkePolicies(listIkePoliciesOptions *ListIkePoliciesOptions) (result *IkePolicyCollection, response *core.DetailedResponse, err error) {
	return vpc.ListIkePoliciesWithContext(context.Background(), listIkePoliciesOptions)
}

// ListIkePoliciesWithContext is an alternate form of the ListIkePolicies method which supports a Context parameter
func (vpc *VpcV1) ListIkePoliciesWithContext(ctx context.Context, listIkePoliciesOptions *ListIkePoliciesOptions) (result *IkePolicyCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listIkePoliciesOptions, "listIkePoliciesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/ike_policies`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listIkePoliciesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListIkePolicies")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listIkePoliciesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listIkePoliciesOptions.Start))
	}
	if listIkePoliciesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listIkePoliciesOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIkePolicyCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateIkePolicy : Create an IKE policy
// This request creates a new IKE policy.
func (vpc *VpcV1) CreateIkePolicy(createIkePolicyOptions *CreateIkePolicyOptions) (result *IkePolicy, response *core.DetailedResponse, err error) {
	return vpc.CreateIkePolicyWithContext(context.Background(), createIkePolicyOptions)
}

// CreateIkePolicyWithContext is an alternate form of the CreateIkePolicy method which supports a Context parameter
func (vpc *VpcV1) CreateIkePolicyWithContext(ctx context.Context, createIkePolicyOptions *CreateIkePolicyOptions) (result *IkePolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createIkePolicyOptions, "createIkePolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createIkePolicyOptions, "createIkePolicyOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/ike_policies`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createIkePolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateIkePolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createIkePolicyOptions.AuthenticationAlgorithm != nil {
		body["authentication_algorithm"] = createIkePolicyOptions.AuthenticationAlgorithm
	}
	if createIkePolicyOptions.DhGroup != nil {
		body["dh_group"] = createIkePolicyOptions.DhGroup
	}
	if createIkePolicyOptions.EncryptionAlgorithm != nil {
		body["encryption_algorithm"] = createIkePolicyOptions.EncryptionAlgorithm
	}
	if createIkePolicyOptions.IkeVersion != nil {
		body["ike_version"] = createIkePolicyOptions.IkeVersion
	}
	if createIkePolicyOptions.KeyLifetime != nil {
		body["key_lifetime"] = createIkePolicyOptions.KeyLifetime
	}
	if createIkePolicyOptions.Name != nil {
		body["name"] = createIkePolicyOptions.Name
	}
	if createIkePolicyOptions.ResourceGroup != nil {
		body["resource_group"] = createIkePolicyOptions.ResourceGroup
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIkePolicy)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteIkePolicy : Delete an IKE policy
// This request deletes an IKE policy. This operation cannot be reversed.
func (vpc *VpcV1) DeleteIkePolicy(deleteIkePolicyOptions *DeleteIkePolicyOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteIkePolicyWithContext(context.Background(), deleteIkePolicyOptions)
}

// DeleteIkePolicyWithContext is an alternate form of the DeleteIkePolicy method which supports a Context parameter
func (vpc *VpcV1) DeleteIkePolicyWithContext(ctx context.Context, deleteIkePolicyOptions *DeleteIkePolicyOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteIkePolicyOptions, "deleteIkePolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteIkePolicyOptions, "deleteIkePolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteIkePolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/ike_policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteIkePolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteIkePolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetIkePolicy : Retrieve an IKE policy
// This request retrieves a single IKE policy specified by the identifier in the URL.
func (vpc *VpcV1) GetIkePolicy(getIkePolicyOptions *GetIkePolicyOptions) (result *IkePolicy, response *core.DetailedResponse, err error) {
	return vpc.GetIkePolicyWithContext(context.Background(), getIkePolicyOptions)
}

// GetIkePolicyWithContext is an alternate form of the GetIkePolicy method which supports a Context parameter
func (vpc *VpcV1) GetIkePolicyWithContext(ctx context.Context, getIkePolicyOptions *GetIkePolicyOptions) (result *IkePolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getIkePolicyOptions, "getIkePolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getIkePolicyOptions, "getIkePolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getIkePolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/ike_policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getIkePolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetIkePolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIkePolicy)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateIkePolicy : Update an IKE policy
// This request updates the properties of an existing IKE policy.
func (vpc *VpcV1) UpdateIkePolicy(updateIkePolicyOptions *UpdateIkePolicyOptions) (result *IkePolicy, response *core.DetailedResponse, err error) {
	return vpc.UpdateIkePolicyWithContext(context.Background(), updateIkePolicyOptions)
}

// UpdateIkePolicyWithContext is an alternate form of the UpdateIkePolicy method which supports a Context parameter
func (vpc *VpcV1) UpdateIkePolicyWithContext(ctx context.Context, updateIkePolicyOptions *UpdateIkePolicyOptions) (result *IkePolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateIkePolicyOptions, "updateIkePolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateIkePolicyOptions, "updateIkePolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateIkePolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/ike_policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateIkePolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateIkePolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateIkePolicyOptions.IkePolicyPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIkePolicy)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListIkePolicyConnections : List all VPN gateway connections that use a specified IKE policy
// This request lists all VPN gateway connections that use a policy.
func (vpc *VpcV1) ListIkePolicyConnections(listIkePolicyConnectionsOptions *ListIkePolicyConnectionsOptions) (result *VPNGatewayConnectionCollection, response *core.DetailedResponse, err error) {
	return vpc.ListIkePolicyConnectionsWithContext(context.Background(), listIkePolicyConnectionsOptions)
}

// ListIkePolicyConnectionsWithContext is an alternate form of the ListIkePolicyConnections method which supports a Context parameter
func (vpc *VpcV1) ListIkePolicyConnectionsWithContext(ctx context.Context, listIkePolicyConnectionsOptions *ListIkePolicyConnectionsOptions) (result *VPNGatewayConnectionCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listIkePolicyConnectionsOptions, "listIkePolicyConnectionsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listIkePolicyConnectionsOptions, "listIkePolicyConnectionsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *listIkePolicyConnectionsOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/ike_policies/{id}/connections`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listIkePolicyConnectionsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListIkePolicyConnections")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnectionCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListIpsecPolicies : List all IPsec policies
// This request lists all IPsec policies in the region.
func (vpc *VpcV1) ListIpsecPolicies(listIpsecPoliciesOptions *ListIpsecPoliciesOptions) (result *IPsecPolicyCollection, response *core.DetailedResponse, err error) {
	return vpc.ListIpsecPoliciesWithContext(context.Background(), listIpsecPoliciesOptions)
}

// ListIpsecPoliciesWithContext is an alternate form of the ListIpsecPolicies method which supports a Context parameter
func (vpc *VpcV1) ListIpsecPoliciesWithContext(ctx context.Context, listIpsecPoliciesOptions *ListIpsecPoliciesOptions) (result *IPsecPolicyCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listIpsecPoliciesOptions, "listIpsecPoliciesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/ipsec_policies`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listIpsecPoliciesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListIpsecPolicies")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listIpsecPoliciesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listIpsecPoliciesOptions.Start))
	}
	if listIpsecPoliciesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listIpsecPoliciesOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIPsecPolicyCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateIpsecPolicy : Create an IPsec policy
// This request creates a new IPsec policy.
func (vpc *VpcV1) CreateIpsecPolicy(createIpsecPolicyOptions *CreateIpsecPolicyOptions) (result *IPsecPolicy, response *core.DetailedResponse, err error) {
	return vpc.CreateIpsecPolicyWithContext(context.Background(), createIpsecPolicyOptions)
}

// CreateIpsecPolicyWithContext is an alternate form of the CreateIpsecPolicy method which supports a Context parameter
func (vpc *VpcV1) CreateIpsecPolicyWithContext(ctx context.Context, createIpsecPolicyOptions *CreateIpsecPolicyOptions) (result *IPsecPolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createIpsecPolicyOptions, "createIpsecPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createIpsecPolicyOptions, "createIpsecPolicyOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/ipsec_policies`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createIpsecPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateIpsecPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createIpsecPolicyOptions.AuthenticationAlgorithm != nil {
		body["authentication_algorithm"] = createIpsecPolicyOptions.AuthenticationAlgorithm
	}
	if createIpsecPolicyOptions.EncryptionAlgorithm != nil {
		body["encryption_algorithm"] = createIpsecPolicyOptions.EncryptionAlgorithm
	}
	if createIpsecPolicyOptions.Pfs != nil {
		body["pfs"] = createIpsecPolicyOptions.Pfs
	}
	if createIpsecPolicyOptions.KeyLifetime != nil {
		body["key_lifetime"] = createIpsecPolicyOptions.KeyLifetime
	}
	if createIpsecPolicyOptions.Name != nil {
		body["name"] = createIpsecPolicyOptions.Name
	}
	if createIpsecPolicyOptions.ResourceGroup != nil {
		body["resource_group"] = createIpsecPolicyOptions.ResourceGroup
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIPsecPolicy)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteIpsecPolicy : Delete an IPsec policy
// This request deletes an IPsec policy. This operation cannot be reversed.
func (vpc *VpcV1) DeleteIpsecPolicy(deleteIpsecPolicyOptions *DeleteIpsecPolicyOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteIpsecPolicyWithContext(context.Background(), deleteIpsecPolicyOptions)
}

// DeleteIpsecPolicyWithContext is an alternate form of the DeleteIpsecPolicy method which supports a Context parameter
func (vpc *VpcV1) DeleteIpsecPolicyWithContext(ctx context.Context, deleteIpsecPolicyOptions *DeleteIpsecPolicyOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteIpsecPolicyOptions, "deleteIpsecPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteIpsecPolicyOptions, "deleteIpsecPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteIpsecPolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/ipsec_policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteIpsecPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteIpsecPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetIpsecPolicy : Retrieve an IPsec policy
// This request retrieves a single IPsec policy specified by the identifier in the URL.
func (vpc *VpcV1) GetIpsecPolicy(getIpsecPolicyOptions *GetIpsecPolicyOptions) (result *IPsecPolicy, response *core.DetailedResponse, err error) {
	return vpc.GetIpsecPolicyWithContext(context.Background(), getIpsecPolicyOptions)
}

// GetIpsecPolicyWithContext is an alternate form of the GetIpsecPolicy method which supports a Context parameter
func (vpc *VpcV1) GetIpsecPolicyWithContext(ctx context.Context, getIpsecPolicyOptions *GetIpsecPolicyOptions) (result *IPsecPolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getIpsecPolicyOptions, "getIpsecPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getIpsecPolicyOptions, "getIpsecPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getIpsecPolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/ipsec_policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getIpsecPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetIpsecPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIPsecPolicy)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateIpsecPolicy : Update an IPsec policy
// This request updates the properties of an existing IPsec policy.
func (vpc *VpcV1) UpdateIpsecPolicy(updateIpsecPolicyOptions *UpdateIpsecPolicyOptions) (result *IPsecPolicy, response *core.DetailedResponse, err error) {
	return vpc.UpdateIpsecPolicyWithContext(context.Background(), updateIpsecPolicyOptions)
}

// UpdateIpsecPolicyWithContext is an alternate form of the UpdateIpsecPolicy method which supports a Context parameter
func (vpc *VpcV1) UpdateIpsecPolicyWithContext(ctx context.Context, updateIpsecPolicyOptions *UpdateIpsecPolicyOptions) (result *IPsecPolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateIpsecPolicyOptions, "updateIpsecPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateIpsecPolicyOptions, "updateIpsecPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateIpsecPolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/ipsec_policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateIpsecPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateIpsecPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateIpsecPolicyOptions.IPsecPolicyPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalIPsecPolicy)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListIpsecPolicyConnections : List all VPN gateway connections that use a specified IPsec policy
// This request lists all VPN gateway connections that use a policy.
func (vpc *VpcV1) ListIpsecPolicyConnections(listIpsecPolicyConnectionsOptions *ListIpsecPolicyConnectionsOptions) (result *VPNGatewayConnectionCollection, response *core.DetailedResponse, err error) {
	return vpc.ListIpsecPolicyConnectionsWithContext(context.Background(), listIpsecPolicyConnectionsOptions)
}

// ListIpsecPolicyConnectionsWithContext is an alternate form of the ListIpsecPolicyConnections method which supports a Context parameter
func (vpc *VpcV1) ListIpsecPolicyConnectionsWithContext(ctx context.Context, listIpsecPolicyConnectionsOptions *ListIpsecPolicyConnectionsOptions) (result *VPNGatewayConnectionCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listIpsecPolicyConnectionsOptions, "listIpsecPolicyConnectionsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listIpsecPolicyConnectionsOptions, "listIpsecPolicyConnectionsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *listIpsecPolicyConnectionsOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/ipsec_policies/{id}/connections`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listIpsecPolicyConnectionsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListIpsecPolicyConnections")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnectionCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListVPNGateways : List all VPN gateways
// This request lists all VPN gateways in the region.
func (vpc *VpcV1) ListVPNGateways(listVPNGatewaysOptions *ListVPNGatewaysOptions) (result *VPNGatewayCollection, response *core.DetailedResponse, err error) {
	return vpc.ListVPNGatewaysWithContext(context.Background(), listVPNGatewaysOptions)
}

// ListVPNGatewaysWithContext is an alternate form of the ListVPNGateways method which supports a Context parameter
func (vpc *VpcV1) ListVPNGatewaysWithContext(ctx context.Context, listVPNGatewaysOptions *ListVPNGatewaysOptions) (result *VPNGatewayCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listVPNGatewaysOptions, "listVPNGatewaysOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVPNGatewaysOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListVPNGateways")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listVPNGatewaysOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listVPNGatewaysOptions.Start))
	}
	if listVPNGatewaysOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listVPNGatewaysOptions.Limit))
	}
	if listVPNGatewaysOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listVPNGatewaysOptions.ResourceGroupID))
	}
	if listVPNGatewaysOptions.Mode != nil {
		builder.AddQuery("mode", fmt.Sprint(*listVPNGatewaysOptions.Mode))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateVPNGateway : Create a VPN gateway
// This request creates a new VPN gateway.
func (vpc *VpcV1) CreateVPNGateway(createVPNGatewayOptions *CreateVPNGatewayOptions) (result VPNGatewayIntf, response *core.DetailedResponse, err error) {
	return vpc.CreateVPNGatewayWithContext(context.Background(), createVPNGatewayOptions)
}

// CreateVPNGatewayWithContext is an alternate form of the CreateVPNGateway method which supports a Context parameter
func (vpc *VpcV1) CreateVPNGatewayWithContext(ctx context.Context, createVPNGatewayOptions *CreateVPNGatewayOptions) (result VPNGatewayIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createVPNGatewayOptions, "createVPNGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createVPNGatewayOptions, "createVPNGatewayOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createVPNGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateVPNGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(createVPNGatewayOptions.VPNGatewayPrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGateway)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteVPNGateway : Delete a VPN gateway
// This request deletes a VPN gateway. A VPN gateway with a `status` of `pending` cannot be deleted. This operation
// deletes all VPN gateway connections associated with this VPN gateway.  This operation cannot be reversed.
func (vpc *VpcV1) DeleteVPNGateway(deleteVPNGatewayOptions *DeleteVPNGatewayOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteVPNGatewayWithContext(context.Background(), deleteVPNGatewayOptions)
}

// DeleteVPNGatewayWithContext is an alternate form of the DeleteVPNGateway method which supports a Context parameter
func (vpc *VpcV1) DeleteVPNGatewayWithContext(ctx context.Context, deleteVPNGatewayOptions *DeleteVPNGatewayOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteVPNGatewayOptions, "deleteVPNGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteVPNGatewayOptions, "deleteVPNGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteVPNGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteVPNGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteVPNGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetVPNGateway : Retrieve a VPN gateway
// This request retrieves a single VPN gateway specified by the identifier in the URL.
func (vpc *VpcV1) GetVPNGateway(getVPNGatewayOptions *GetVPNGatewayOptions) (result VPNGatewayIntf, response *core.DetailedResponse, err error) {
	return vpc.GetVPNGatewayWithContext(context.Background(), getVPNGatewayOptions)
}

// GetVPNGatewayWithContext is an alternate form of the GetVPNGateway method which supports a Context parameter
func (vpc *VpcV1) GetVPNGatewayWithContext(ctx context.Context, getVPNGatewayOptions *GetVPNGatewayOptions) (result VPNGatewayIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPNGatewayOptions, "getVPNGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPNGatewayOptions, "getVPNGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getVPNGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPNGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetVPNGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGateway)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateVPNGateway : Update a VPN gateway
// This request updates the properties of an existing VPN gateway.
func (vpc *VpcV1) UpdateVPNGateway(updateVPNGatewayOptions *UpdateVPNGatewayOptions) (result VPNGatewayIntf, response *core.DetailedResponse, err error) {
	return vpc.UpdateVPNGatewayWithContext(context.Background(), updateVPNGatewayOptions)
}

// UpdateVPNGatewayWithContext is an alternate form of the UpdateVPNGateway method which supports a Context parameter
func (vpc *VpcV1) UpdateVPNGatewayWithContext(ctx context.Context, updateVPNGatewayOptions *UpdateVPNGatewayOptions) (result VPNGatewayIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateVPNGatewayOptions, "updateVPNGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateVPNGatewayOptions, "updateVPNGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateVPNGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateVPNGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateVPNGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateVPNGatewayOptions.VPNGatewayPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGateway)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListVPNGatewayConnections : List all connections of a VPN gateway
// This request lists all connections of a VPN gateway.
func (vpc *VpcV1) ListVPNGatewayConnections(listVPNGatewayConnectionsOptions *ListVPNGatewayConnectionsOptions) (result *VPNGatewayConnectionCollection, response *core.DetailedResponse, err error) {
	return vpc.ListVPNGatewayConnectionsWithContext(context.Background(), listVPNGatewayConnectionsOptions)
}

// ListVPNGatewayConnectionsWithContext is an alternate form of the ListVPNGatewayConnections method which supports a Context parameter
func (vpc *VpcV1) ListVPNGatewayConnectionsWithContext(ctx context.Context, listVPNGatewayConnectionsOptions *ListVPNGatewayConnectionsOptions) (result *VPNGatewayConnectionCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listVPNGatewayConnectionsOptions, "listVPNGatewayConnectionsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listVPNGatewayConnectionsOptions, "listVPNGatewayConnectionsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *listVPNGatewayConnectionsOptions.VPNGatewayID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVPNGatewayConnectionsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListVPNGatewayConnections")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listVPNGatewayConnectionsOptions.Status != nil {
		builder.AddQuery("status", fmt.Sprint(*listVPNGatewayConnectionsOptions.Status))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnectionCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateVPNGatewayConnection : Create a connection for a VPN gateway
// This request creates a new VPN gateway connection.
func (vpc *VpcV1) CreateVPNGatewayConnection(createVPNGatewayConnectionOptions *CreateVPNGatewayConnectionOptions) (result VPNGatewayConnectionIntf, response *core.DetailedResponse, err error) {
	return vpc.CreateVPNGatewayConnectionWithContext(context.Background(), createVPNGatewayConnectionOptions)
}

// CreateVPNGatewayConnectionWithContext is an alternate form of the CreateVPNGatewayConnection method which supports a Context parameter
func (vpc *VpcV1) CreateVPNGatewayConnectionWithContext(ctx context.Context, createVPNGatewayConnectionOptions *CreateVPNGatewayConnectionOptions) (result VPNGatewayConnectionIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createVPNGatewayConnectionOptions, "createVPNGatewayConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createVPNGatewayConnectionOptions, "createVPNGatewayConnectionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *createVPNGatewayConnectionOptions.VPNGatewayID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createVPNGatewayConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateVPNGatewayConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(createVPNGatewayConnectionOptions.VPNGatewayConnectionPrototype)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteVPNGatewayConnection : Delete a VPN gateway connection
// This request deletes a VPN gateway connection. This operation cannot be reversed.
func (vpc *VpcV1) DeleteVPNGatewayConnection(deleteVPNGatewayConnectionOptions *DeleteVPNGatewayConnectionOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteVPNGatewayConnectionWithContext(context.Background(), deleteVPNGatewayConnectionOptions)
}

// DeleteVPNGatewayConnectionWithContext is an alternate form of the DeleteVPNGatewayConnection method which supports a Context parameter
func (vpc *VpcV1) DeleteVPNGatewayConnectionWithContext(ctx context.Context, deleteVPNGatewayConnectionOptions *DeleteVPNGatewayConnectionOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteVPNGatewayConnectionOptions, "deleteVPNGatewayConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteVPNGatewayConnectionOptions, "deleteVPNGatewayConnectionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *deleteVPNGatewayConnectionOptions.VPNGatewayID,
		"id":             *deleteVPNGatewayConnectionOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteVPNGatewayConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteVPNGatewayConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetVPNGatewayConnection : Retrieve a VPN gateway connection
// This request retrieves a single VPN gateway connection specified by the identifier in the URL.
func (vpc *VpcV1) GetVPNGatewayConnection(getVPNGatewayConnectionOptions *GetVPNGatewayConnectionOptions) (result VPNGatewayConnectionIntf, response *core.DetailedResponse, err error) {
	return vpc.GetVPNGatewayConnectionWithContext(context.Background(), getVPNGatewayConnectionOptions)
}

// GetVPNGatewayConnectionWithContext is an alternate form of the GetVPNGatewayConnection method which supports a Context parameter
func (vpc *VpcV1) GetVPNGatewayConnectionWithContext(ctx context.Context, getVPNGatewayConnectionOptions *GetVPNGatewayConnectionOptions) (result VPNGatewayConnectionIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getVPNGatewayConnectionOptions, "getVPNGatewayConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getVPNGatewayConnectionOptions, "getVPNGatewayConnectionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *getVPNGatewayConnectionOptions.VPNGatewayID,
		"id":             *getVPNGatewayConnectionOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getVPNGatewayConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetVPNGatewayConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateVPNGatewayConnection : Update a VPN gateway connection
// This request updates the properties of an existing VPN gateway connection.
func (vpc *VpcV1) UpdateVPNGatewayConnection(updateVPNGatewayConnectionOptions *UpdateVPNGatewayConnectionOptions) (result VPNGatewayConnectionIntf, response *core.DetailedResponse, err error) {
	return vpc.UpdateVPNGatewayConnectionWithContext(context.Background(), updateVPNGatewayConnectionOptions)
}

// UpdateVPNGatewayConnectionWithContext is an alternate form of the UpdateVPNGatewayConnection method which supports a Context parameter
func (vpc *VpcV1) UpdateVPNGatewayConnectionWithContext(ctx context.Context, updateVPNGatewayConnectionOptions *UpdateVPNGatewayConnectionOptions) (result VPNGatewayConnectionIntf, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateVPNGatewayConnectionOptions, "updateVPNGatewayConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateVPNGatewayConnectionOptions, "updateVPNGatewayConnectionOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *updateVPNGatewayConnectionOptions.VPNGatewayID,
		"id":             *updateVPNGatewayConnectionOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateVPNGatewayConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateVPNGatewayConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateVPNGatewayConnectionOptions.VPNGatewayConnectionPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListVPNGatewayConnectionLocalCIDRs : List all local CIDRs for a VPN gateway connection
// This request lists all local CIDRs for a VPN gateway connection.
func (vpc *VpcV1) ListVPNGatewayConnectionLocalCIDRs(listVPNGatewayConnectionLocalCIDRsOptions *ListVPNGatewayConnectionLocalCIDRsOptions) (result *VPNGatewayConnectionLocalCIDRs, response *core.DetailedResponse, err error) {
	return vpc.ListVPNGatewayConnectionLocalCIDRsWithContext(context.Background(), listVPNGatewayConnectionLocalCIDRsOptions)
}

// ListVPNGatewayConnectionLocalCIDRsWithContext is an alternate form of the ListVPNGatewayConnectionLocalCIDRs method which supports a Context parameter
func (vpc *VpcV1) ListVPNGatewayConnectionLocalCIDRsWithContext(ctx context.Context, listVPNGatewayConnectionLocalCIDRsOptions *ListVPNGatewayConnectionLocalCIDRsOptions) (result *VPNGatewayConnectionLocalCIDRs, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listVPNGatewayConnectionLocalCIDRsOptions, "listVPNGatewayConnectionLocalCIDRsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listVPNGatewayConnectionLocalCIDRsOptions, "listVPNGatewayConnectionLocalCIDRsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *listVPNGatewayConnectionLocalCIDRsOptions.VPNGatewayID,
		"id":             *listVPNGatewayConnectionLocalCIDRsOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/local_cidrs`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVPNGatewayConnectionLocalCIDRsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListVPNGatewayConnectionLocalCIDRs")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnectionLocalCIDRs)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// RemoveVPNGatewayConnectionLocalCIDR : Remove a local CIDR from a VPN gateway connection
// This request removes a CIDR from a VPN gateway connection.
func (vpc *VpcV1) RemoveVPNGatewayConnectionLocalCIDR(removeVPNGatewayConnectionLocalCIDROptions *RemoveVPNGatewayConnectionLocalCIDROptions) (response *core.DetailedResponse, err error) {
	return vpc.RemoveVPNGatewayConnectionLocalCIDRWithContext(context.Background(), removeVPNGatewayConnectionLocalCIDROptions)
}

// RemoveVPNGatewayConnectionLocalCIDRWithContext is an alternate form of the RemoveVPNGatewayConnectionLocalCIDR method which supports a Context parameter
func (vpc *VpcV1) RemoveVPNGatewayConnectionLocalCIDRWithContext(ctx context.Context, removeVPNGatewayConnectionLocalCIDROptions *RemoveVPNGatewayConnectionLocalCIDROptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(removeVPNGatewayConnectionLocalCIDROptions, "removeVPNGatewayConnectionLocalCIDROptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(removeVPNGatewayConnectionLocalCIDROptions, "removeVPNGatewayConnectionLocalCIDROptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *removeVPNGatewayConnectionLocalCIDROptions.VPNGatewayID,
		"id":             *removeVPNGatewayConnectionLocalCIDROptions.ID,
		"cidr_prefix":    *removeVPNGatewayConnectionLocalCIDROptions.CIDRPrefix,
		"prefix_length":  *removeVPNGatewayConnectionLocalCIDROptions.PrefixLength,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/local_cidrs/{cidr_prefix}/{prefix_length}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range removeVPNGatewayConnectionLocalCIDROptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "RemoveVPNGatewayConnectionLocalCIDR")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// CheckVPNGatewayConnectionLocalCIDR : Check if the specified local CIDR exists on a VPN gateway connection
// This request succeeds if a CIDR exists on the specified VPN gateway connection and fails otherwise.
func (vpc *VpcV1) CheckVPNGatewayConnectionLocalCIDR(checkVPNGatewayConnectionLocalCIDROptions *CheckVPNGatewayConnectionLocalCIDROptions) (response *core.DetailedResponse, err error) {
	return vpc.CheckVPNGatewayConnectionLocalCIDRWithContext(context.Background(), checkVPNGatewayConnectionLocalCIDROptions)
}

// CheckVPNGatewayConnectionLocalCIDRWithContext is an alternate form of the CheckVPNGatewayConnectionLocalCIDR method which supports a Context parameter
func (vpc *VpcV1) CheckVPNGatewayConnectionLocalCIDRWithContext(ctx context.Context, checkVPNGatewayConnectionLocalCIDROptions *CheckVPNGatewayConnectionLocalCIDROptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(checkVPNGatewayConnectionLocalCIDROptions, "checkVPNGatewayConnectionLocalCIDROptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(checkVPNGatewayConnectionLocalCIDROptions, "checkVPNGatewayConnectionLocalCIDROptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *checkVPNGatewayConnectionLocalCIDROptions.VPNGatewayID,
		"id":             *checkVPNGatewayConnectionLocalCIDROptions.ID,
		"cidr_prefix":    *checkVPNGatewayConnectionLocalCIDROptions.CIDRPrefix,
		"prefix_length":  *checkVPNGatewayConnectionLocalCIDROptions.PrefixLength,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/local_cidrs/{cidr_prefix}/{prefix_length}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range checkVPNGatewayConnectionLocalCIDROptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CheckVPNGatewayConnectionLocalCIDR")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// AddVPNGatewayConnectionLocalCIDR : Set a local CIDR on a VPN gateway connection
// This request adds the specified CIDR to the specified VPN gateway connection. A request body is not required, and if
// supplied, is ignored. This request succeeds if the CIDR already exists on the specified VPN gateway connection.
func (vpc *VpcV1) AddVPNGatewayConnectionLocalCIDR(addVPNGatewayConnectionLocalCIDROptions *AddVPNGatewayConnectionLocalCIDROptions) (response *core.DetailedResponse, err error) {
	return vpc.AddVPNGatewayConnectionLocalCIDRWithContext(context.Background(), addVPNGatewayConnectionLocalCIDROptions)
}

// AddVPNGatewayConnectionLocalCIDRWithContext is an alternate form of the AddVPNGatewayConnectionLocalCIDR method which supports a Context parameter
func (vpc *VpcV1) AddVPNGatewayConnectionLocalCIDRWithContext(ctx context.Context, addVPNGatewayConnectionLocalCIDROptions *AddVPNGatewayConnectionLocalCIDROptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(addVPNGatewayConnectionLocalCIDROptions, "addVPNGatewayConnectionLocalCIDROptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(addVPNGatewayConnectionLocalCIDROptions, "addVPNGatewayConnectionLocalCIDROptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *addVPNGatewayConnectionLocalCIDROptions.VPNGatewayID,
		"id":             *addVPNGatewayConnectionLocalCIDROptions.ID,
		"cidr_prefix":    *addVPNGatewayConnectionLocalCIDROptions.CIDRPrefix,
		"prefix_length":  *addVPNGatewayConnectionLocalCIDROptions.PrefixLength,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/local_cidrs/{cidr_prefix}/{prefix_length}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range addVPNGatewayConnectionLocalCIDROptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "AddVPNGatewayConnectionLocalCIDR")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// ListVPNGatewayConnectionPeerCIDRs : List all peer CIDRs for a VPN gateway connection
// This request lists all peer CIDRs for a VPN gateway connection.
func (vpc *VpcV1) ListVPNGatewayConnectionPeerCIDRs(listVPNGatewayConnectionPeerCIDRsOptions *ListVPNGatewayConnectionPeerCIDRsOptions) (result *VPNGatewayConnectionPeerCIDRs, response *core.DetailedResponse, err error) {
	return vpc.ListVPNGatewayConnectionPeerCIDRsWithContext(context.Background(), listVPNGatewayConnectionPeerCIDRsOptions)
}

// ListVPNGatewayConnectionPeerCIDRsWithContext is an alternate form of the ListVPNGatewayConnectionPeerCIDRs method which supports a Context parameter
func (vpc *VpcV1) ListVPNGatewayConnectionPeerCIDRsWithContext(ctx context.Context, listVPNGatewayConnectionPeerCIDRsOptions *ListVPNGatewayConnectionPeerCIDRsOptions) (result *VPNGatewayConnectionPeerCIDRs, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listVPNGatewayConnectionPeerCIDRsOptions, "listVPNGatewayConnectionPeerCIDRsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listVPNGatewayConnectionPeerCIDRsOptions, "listVPNGatewayConnectionPeerCIDRsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *listVPNGatewayConnectionPeerCIDRsOptions.VPNGatewayID,
		"id":             *listVPNGatewayConnectionPeerCIDRsOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/peer_cidrs`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listVPNGatewayConnectionPeerCIDRsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListVPNGatewayConnectionPeerCIDRs")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalVPNGatewayConnectionPeerCIDRs)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// RemoveVPNGatewayConnectionPeerCIDR : Remove a peer CIDR from a VPN gateway connection
// This request removes a CIDR from a VPN gateway connection.
func (vpc *VpcV1) RemoveVPNGatewayConnectionPeerCIDR(removeVPNGatewayConnectionPeerCIDROptions *RemoveVPNGatewayConnectionPeerCIDROptions) (response *core.DetailedResponse, err error) {
	return vpc.RemoveVPNGatewayConnectionPeerCIDRWithContext(context.Background(), removeVPNGatewayConnectionPeerCIDROptions)
}

// RemoveVPNGatewayConnectionPeerCIDRWithContext is an alternate form of the RemoveVPNGatewayConnectionPeerCIDR method which supports a Context parameter
func (vpc *VpcV1) RemoveVPNGatewayConnectionPeerCIDRWithContext(ctx context.Context, removeVPNGatewayConnectionPeerCIDROptions *RemoveVPNGatewayConnectionPeerCIDROptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(removeVPNGatewayConnectionPeerCIDROptions, "removeVPNGatewayConnectionPeerCIDROptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(removeVPNGatewayConnectionPeerCIDROptions, "removeVPNGatewayConnectionPeerCIDROptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *removeVPNGatewayConnectionPeerCIDROptions.VPNGatewayID,
		"id":             *removeVPNGatewayConnectionPeerCIDROptions.ID,
		"cidr_prefix":    *removeVPNGatewayConnectionPeerCIDROptions.CIDRPrefix,
		"prefix_length":  *removeVPNGatewayConnectionPeerCIDROptions.PrefixLength,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/peer_cidrs/{cidr_prefix}/{prefix_length}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range removeVPNGatewayConnectionPeerCIDROptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "RemoveVPNGatewayConnectionPeerCIDR")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// CheckVPNGatewayConnectionPeerCIDR : Check if the specified peer CIDR exists on a VPN gateway connection
// This request succeeds if a CIDR exists on the specified VPN gateway connection and fails otherwise.
func (vpc *VpcV1) CheckVPNGatewayConnectionPeerCIDR(checkVPNGatewayConnectionPeerCIDROptions *CheckVPNGatewayConnectionPeerCIDROptions) (response *core.DetailedResponse, err error) {
	return vpc.CheckVPNGatewayConnectionPeerCIDRWithContext(context.Background(), checkVPNGatewayConnectionPeerCIDROptions)
}

// CheckVPNGatewayConnectionPeerCIDRWithContext is an alternate form of the CheckVPNGatewayConnectionPeerCIDR method which supports a Context parameter
func (vpc *VpcV1) CheckVPNGatewayConnectionPeerCIDRWithContext(ctx context.Context, checkVPNGatewayConnectionPeerCIDROptions *CheckVPNGatewayConnectionPeerCIDROptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(checkVPNGatewayConnectionPeerCIDROptions, "checkVPNGatewayConnectionPeerCIDROptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(checkVPNGatewayConnectionPeerCIDROptions, "checkVPNGatewayConnectionPeerCIDROptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *checkVPNGatewayConnectionPeerCIDROptions.VPNGatewayID,
		"id":             *checkVPNGatewayConnectionPeerCIDROptions.ID,
		"cidr_prefix":    *checkVPNGatewayConnectionPeerCIDROptions.CIDRPrefix,
		"prefix_length":  *checkVPNGatewayConnectionPeerCIDROptions.PrefixLength,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/peer_cidrs/{cidr_prefix}/{prefix_length}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range checkVPNGatewayConnectionPeerCIDROptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CheckVPNGatewayConnectionPeerCIDR")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// AddVPNGatewayConnectionPeerCIDR : Set a peer CIDR on a VPN gateway connection
// This request adds the specified CIDR to the specified VPN gateway connection. A request body is not required, and if
// supplied, is ignored. This request succeeds if the CIDR already exists on the specified VPN gateway connection.
func (vpc *VpcV1) AddVPNGatewayConnectionPeerCIDR(addVPNGatewayConnectionPeerCIDROptions *AddVPNGatewayConnectionPeerCIDROptions) (response *core.DetailedResponse, err error) {
	return vpc.AddVPNGatewayConnectionPeerCIDRWithContext(context.Background(), addVPNGatewayConnectionPeerCIDROptions)
}

// AddVPNGatewayConnectionPeerCIDRWithContext is an alternate form of the AddVPNGatewayConnectionPeerCIDR method which supports a Context parameter
func (vpc *VpcV1) AddVPNGatewayConnectionPeerCIDRWithContext(ctx context.Context, addVPNGatewayConnectionPeerCIDROptions *AddVPNGatewayConnectionPeerCIDROptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(addVPNGatewayConnectionPeerCIDROptions, "addVPNGatewayConnectionPeerCIDROptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(addVPNGatewayConnectionPeerCIDROptions, "addVPNGatewayConnectionPeerCIDROptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"vpn_gateway_id": *addVPNGatewayConnectionPeerCIDROptions.VPNGatewayID,
		"id":             *addVPNGatewayConnectionPeerCIDROptions.ID,
		"cidr_prefix":    *addVPNGatewayConnectionPeerCIDROptions.CIDRPrefix,
		"prefix_length":  *addVPNGatewayConnectionPeerCIDROptions.PrefixLength,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/vpn_gateways/{vpn_gateway_id}/connections/{id}/peer_cidrs/{cidr_prefix}/{prefix_length}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range addVPNGatewayConnectionPeerCIDROptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "AddVPNGatewayConnectionPeerCIDR")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// ListLoadBalancerProfiles : List all load balancer profiles
// This request lists all load balancer profiles available in the region. A load balancer profile specifies the
// performance characteristics and pricing model for a load balancer.
func (vpc *VpcV1) ListLoadBalancerProfiles(listLoadBalancerProfilesOptions *ListLoadBalancerProfilesOptions) (result *LoadBalancerProfileCollection, response *core.DetailedResponse, err error) {
	return vpc.ListLoadBalancerProfilesWithContext(context.Background(), listLoadBalancerProfilesOptions)
}

// ListLoadBalancerProfilesWithContext is an alternate form of the ListLoadBalancerProfiles method which supports a Context parameter
func (vpc *VpcV1) ListLoadBalancerProfilesWithContext(ctx context.Context, listLoadBalancerProfilesOptions *ListLoadBalancerProfilesOptions) (result *LoadBalancerProfileCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listLoadBalancerProfilesOptions, "listLoadBalancerProfilesOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancer/profiles`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listLoadBalancerProfilesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListLoadBalancerProfiles")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listLoadBalancerProfilesOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listLoadBalancerProfilesOptions.Start))
	}
	if listLoadBalancerProfilesOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listLoadBalancerProfilesOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerProfileCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetLoadBalancerProfile : Retrieve a load balancer profile
// This request retrieves a load balancer profile specified by the name in the URL.
func (vpc *VpcV1) GetLoadBalancerProfile(getLoadBalancerProfileOptions *GetLoadBalancerProfileOptions) (result *LoadBalancerProfile, response *core.DetailedResponse, err error) {
	return vpc.GetLoadBalancerProfileWithContext(context.Background(), getLoadBalancerProfileOptions)
}

// GetLoadBalancerProfileWithContext is an alternate form of the GetLoadBalancerProfile method which supports a Context parameter
func (vpc *VpcV1) GetLoadBalancerProfileWithContext(ctx context.Context, getLoadBalancerProfileOptions *GetLoadBalancerProfileOptions) (result *LoadBalancerProfile, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getLoadBalancerProfileOptions, "getLoadBalancerProfileOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getLoadBalancerProfileOptions, "getLoadBalancerProfileOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"name": *getLoadBalancerProfileOptions.Name,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancer/profiles/{name}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getLoadBalancerProfileOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetLoadBalancerProfile")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerProfile)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListLoadBalancers : List all load balancers
// This request lists all load balancers in the region.
func (vpc *VpcV1) ListLoadBalancers(listLoadBalancersOptions *ListLoadBalancersOptions) (result *LoadBalancerCollection, response *core.DetailedResponse, err error) {
	return vpc.ListLoadBalancersWithContext(context.Background(), listLoadBalancersOptions)
}

// ListLoadBalancersWithContext is an alternate form of the ListLoadBalancers method which supports a Context parameter
func (vpc *VpcV1) ListLoadBalancersWithContext(ctx context.Context, listLoadBalancersOptions *ListLoadBalancersOptions) (result *LoadBalancerCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listLoadBalancersOptions, "listLoadBalancersOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listLoadBalancersOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListLoadBalancers")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listLoadBalancersOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listLoadBalancersOptions.Start))
	}
	if listLoadBalancersOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listLoadBalancersOptions.Limit))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateLoadBalancer : Create a load balancer
// This request creates and provisions a new load balancer.
func (vpc *VpcV1) CreateLoadBalancer(createLoadBalancerOptions *CreateLoadBalancerOptions) (result *LoadBalancer, response *core.DetailedResponse, err error) {
	return vpc.CreateLoadBalancerWithContext(context.Background(), createLoadBalancerOptions)
}

// CreateLoadBalancerWithContext is an alternate form of the CreateLoadBalancer method which supports a Context parameter
func (vpc *VpcV1) CreateLoadBalancerWithContext(ctx context.Context, createLoadBalancerOptions *CreateLoadBalancerOptions) (result *LoadBalancer, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createLoadBalancerOptions, "createLoadBalancerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createLoadBalancerOptions, "createLoadBalancerOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createLoadBalancerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateLoadBalancer")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createLoadBalancerOptions.IsPublic != nil {
		body["is_public"] = createLoadBalancerOptions.IsPublic
	}
	if createLoadBalancerOptions.Subnets != nil {
		body["subnets"] = createLoadBalancerOptions.Subnets
	}
	if createLoadBalancerOptions.Listeners != nil {
		body["listeners"] = createLoadBalancerOptions.Listeners
	}
	if createLoadBalancerOptions.Logging != nil {
		body["logging"] = createLoadBalancerOptions.Logging
	}
	if createLoadBalancerOptions.Name != nil {
		body["name"] = createLoadBalancerOptions.Name
	}
	if createLoadBalancerOptions.Pools != nil {
		body["pools"] = createLoadBalancerOptions.Pools
	}
	if createLoadBalancerOptions.Profile != nil {
		body["profile"] = createLoadBalancerOptions.Profile
	}
	if createLoadBalancerOptions.ResourceGroup != nil {
		body["resource_group"] = createLoadBalancerOptions.ResourceGroup
	}
	if createLoadBalancerOptions.SecurityGroups != nil {
		body["security_groups"] = createLoadBalancerOptions.SecurityGroups
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancer)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteLoadBalancer : Delete a load balancer
// This request deletes a load balancer. This operation cannot be reversed.
func (vpc *VpcV1) DeleteLoadBalancer(deleteLoadBalancerOptions *DeleteLoadBalancerOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteLoadBalancerWithContext(context.Background(), deleteLoadBalancerOptions)
}

// DeleteLoadBalancerWithContext is an alternate form of the DeleteLoadBalancer method which supports a Context parameter
func (vpc *VpcV1) DeleteLoadBalancerWithContext(ctx context.Context, deleteLoadBalancerOptions *DeleteLoadBalancerOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteLoadBalancerOptions, "deleteLoadBalancerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteLoadBalancerOptions, "deleteLoadBalancerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteLoadBalancerOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteLoadBalancerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteLoadBalancer")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetLoadBalancer : Retrieve a load balancer
// This request retrieves a single load balancer specified by the identifier in the URL path.
func (vpc *VpcV1) GetLoadBalancer(getLoadBalancerOptions *GetLoadBalancerOptions) (result *LoadBalancer, response *core.DetailedResponse, err error) {
	return vpc.GetLoadBalancerWithContext(context.Background(), getLoadBalancerOptions)
}

// GetLoadBalancerWithContext is an alternate form of the GetLoadBalancer method which supports a Context parameter
func (vpc *VpcV1) GetLoadBalancerWithContext(ctx context.Context, getLoadBalancerOptions *GetLoadBalancerOptions) (result *LoadBalancer, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getLoadBalancerOptions, "getLoadBalancerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getLoadBalancerOptions, "getLoadBalancerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getLoadBalancerOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getLoadBalancerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetLoadBalancer")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancer)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateLoadBalancer : Update a load balancer
// This request updates a load balancer.
func (vpc *VpcV1) UpdateLoadBalancer(updateLoadBalancerOptions *UpdateLoadBalancerOptions) (result *LoadBalancer, response *core.DetailedResponse, err error) {
	return vpc.UpdateLoadBalancerWithContext(context.Background(), updateLoadBalancerOptions)
}

// UpdateLoadBalancerWithContext is an alternate form of the UpdateLoadBalancer method which supports a Context parameter
func (vpc *VpcV1) UpdateLoadBalancerWithContext(ctx context.Context, updateLoadBalancerOptions *UpdateLoadBalancerOptions) (result *LoadBalancer, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateLoadBalancerOptions, "updateLoadBalancerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateLoadBalancerOptions, "updateLoadBalancerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateLoadBalancerOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateLoadBalancerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateLoadBalancer")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateLoadBalancerOptions.LoadBalancerPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancer)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// GetLoadBalancerStatistics : List all statistics of a load balancer
// This request lists statistics of a load balancer.
func (vpc *VpcV1) GetLoadBalancerStatistics(getLoadBalancerStatisticsOptions *GetLoadBalancerStatisticsOptions) (result *LoadBalancerStatistics, response *core.DetailedResponse, err error) {
	return vpc.GetLoadBalancerStatisticsWithContext(context.Background(), getLoadBalancerStatisticsOptions)
}

// GetLoadBalancerStatisticsWithContext is an alternate form of the GetLoadBalancerStatistics method which supports a Context parameter
func (vpc *VpcV1) GetLoadBalancerStatisticsWithContext(ctx context.Context, getLoadBalancerStatisticsOptions *GetLoadBalancerStatisticsOptions) (result *LoadBalancerStatistics, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getLoadBalancerStatisticsOptions, "getLoadBalancerStatisticsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getLoadBalancerStatisticsOptions, "getLoadBalancerStatisticsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getLoadBalancerStatisticsOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{id}/statistics`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getLoadBalancerStatisticsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetLoadBalancerStatistics")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerStatistics)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListLoadBalancerListeners : List all listeners for a load balancer
// This request lists all listeners for a load balancer.
func (vpc *VpcV1) ListLoadBalancerListeners(listLoadBalancerListenersOptions *ListLoadBalancerListenersOptions) (result *LoadBalancerListenerCollection, response *core.DetailedResponse, err error) {
	return vpc.ListLoadBalancerListenersWithContext(context.Background(), listLoadBalancerListenersOptions)
}

// ListLoadBalancerListenersWithContext is an alternate form of the ListLoadBalancerListeners method which supports a Context parameter
func (vpc *VpcV1) ListLoadBalancerListenersWithContext(ctx context.Context, listLoadBalancerListenersOptions *ListLoadBalancerListenersOptions) (result *LoadBalancerListenerCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listLoadBalancerListenersOptions, "listLoadBalancerListenersOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listLoadBalancerListenersOptions, "listLoadBalancerListenersOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *listLoadBalancerListenersOptions.LoadBalancerID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listLoadBalancerListenersOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListLoadBalancerListeners")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateLoadBalancerListener : Create a listener for a load balancer
// This request creates a new listener for a load balancer.
func (vpc *VpcV1) CreateLoadBalancerListener(createLoadBalancerListenerOptions *CreateLoadBalancerListenerOptions) (result *LoadBalancerListener, response *core.DetailedResponse, err error) {
	return vpc.CreateLoadBalancerListenerWithContext(context.Background(), createLoadBalancerListenerOptions)
}

// CreateLoadBalancerListenerWithContext is an alternate form of the CreateLoadBalancerListener method which supports a Context parameter
func (vpc *VpcV1) CreateLoadBalancerListenerWithContext(ctx context.Context, createLoadBalancerListenerOptions *CreateLoadBalancerListenerOptions) (result *LoadBalancerListener, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createLoadBalancerListenerOptions, "createLoadBalancerListenerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createLoadBalancerListenerOptions, "createLoadBalancerListenerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *createLoadBalancerListenerOptions.LoadBalancerID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createLoadBalancerListenerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateLoadBalancerListener")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createLoadBalancerListenerOptions.Port != nil {
		body["port"] = createLoadBalancerListenerOptions.Port
	}
	if createLoadBalancerListenerOptions.Protocol != nil {
		body["protocol"] = createLoadBalancerListenerOptions.Protocol
	}
	if createLoadBalancerListenerOptions.AcceptProxyProtocol != nil {
		body["accept_proxy_protocol"] = createLoadBalancerListenerOptions.AcceptProxyProtocol
	}
	if createLoadBalancerListenerOptions.CertificateInstance != nil {
		body["certificate_instance"] = createLoadBalancerListenerOptions.CertificateInstance
	}
	if createLoadBalancerListenerOptions.ConnectionLimit != nil {
		body["connection_limit"] = createLoadBalancerListenerOptions.ConnectionLimit
	}
	if createLoadBalancerListenerOptions.DefaultPool != nil {
		body["default_pool"] = createLoadBalancerListenerOptions.DefaultPool
	}
	if createLoadBalancerListenerOptions.Policies != nil {
		body["policies"] = createLoadBalancerListenerOptions.Policies
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListener)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteLoadBalancerListener : Delete a load balancer listener
// This request deletes a load balancer listener. This operation cannot be reversed.
func (vpc *VpcV1) DeleteLoadBalancerListener(deleteLoadBalancerListenerOptions *DeleteLoadBalancerListenerOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteLoadBalancerListenerWithContext(context.Background(), deleteLoadBalancerListenerOptions)
}

// DeleteLoadBalancerListenerWithContext is an alternate form of the DeleteLoadBalancerListener method which supports a Context parameter
func (vpc *VpcV1) DeleteLoadBalancerListenerWithContext(ctx context.Context, deleteLoadBalancerListenerOptions *DeleteLoadBalancerListenerOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteLoadBalancerListenerOptions, "deleteLoadBalancerListenerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteLoadBalancerListenerOptions, "deleteLoadBalancerListenerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *deleteLoadBalancerListenerOptions.LoadBalancerID,
		"id":               *deleteLoadBalancerListenerOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteLoadBalancerListenerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteLoadBalancerListener")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetLoadBalancerListener : Retrieve a load balancer listener
// This request retrieves a single listener specified by the identifier in the URL path.
func (vpc *VpcV1) GetLoadBalancerListener(getLoadBalancerListenerOptions *GetLoadBalancerListenerOptions) (result *LoadBalancerListener, response *core.DetailedResponse, err error) {
	return vpc.GetLoadBalancerListenerWithContext(context.Background(), getLoadBalancerListenerOptions)
}

// GetLoadBalancerListenerWithContext is an alternate form of the GetLoadBalancerListener method which supports a Context parameter
func (vpc *VpcV1) GetLoadBalancerListenerWithContext(ctx context.Context, getLoadBalancerListenerOptions *GetLoadBalancerListenerOptions) (result *LoadBalancerListener, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getLoadBalancerListenerOptions, "getLoadBalancerListenerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getLoadBalancerListenerOptions, "getLoadBalancerListenerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *getLoadBalancerListenerOptions.LoadBalancerID,
		"id":               *getLoadBalancerListenerOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getLoadBalancerListenerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetLoadBalancerListener")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListener)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateLoadBalancerListener : Update a load balancer listener
// This request updates a load balancer listener from a listener patch.
func (vpc *VpcV1) UpdateLoadBalancerListener(updateLoadBalancerListenerOptions *UpdateLoadBalancerListenerOptions) (result *LoadBalancerListener, response *core.DetailedResponse, err error) {
	return vpc.UpdateLoadBalancerListenerWithContext(context.Background(), updateLoadBalancerListenerOptions)
}

// UpdateLoadBalancerListenerWithContext is an alternate form of the UpdateLoadBalancerListener method which supports a Context parameter
func (vpc *VpcV1) UpdateLoadBalancerListenerWithContext(ctx context.Context, updateLoadBalancerListenerOptions *UpdateLoadBalancerListenerOptions) (result *LoadBalancerListener, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateLoadBalancerListenerOptions, "updateLoadBalancerListenerOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateLoadBalancerListenerOptions, "updateLoadBalancerListenerOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *updateLoadBalancerListenerOptions.LoadBalancerID,
		"id":               *updateLoadBalancerListenerOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateLoadBalancerListenerOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateLoadBalancerListener")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateLoadBalancerListenerOptions.LoadBalancerListenerPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListener)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListLoadBalancerListenerPolicies : List all policies for a load balancer listener
// This request lists all policies for a load balancer listener.
func (vpc *VpcV1) ListLoadBalancerListenerPolicies(listLoadBalancerListenerPoliciesOptions *ListLoadBalancerListenerPoliciesOptions) (result *LoadBalancerListenerPolicyCollection, response *core.DetailedResponse, err error) {
	return vpc.ListLoadBalancerListenerPoliciesWithContext(context.Background(), listLoadBalancerListenerPoliciesOptions)
}

// ListLoadBalancerListenerPoliciesWithContext is an alternate form of the ListLoadBalancerListenerPolicies method which supports a Context parameter
func (vpc *VpcV1) ListLoadBalancerListenerPoliciesWithContext(ctx context.Context, listLoadBalancerListenerPoliciesOptions *ListLoadBalancerListenerPoliciesOptions) (result *LoadBalancerListenerPolicyCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listLoadBalancerListenerPoliciesOptions, "listLoadBalancerListenerPoliciesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listLoadBalancerListenerPoliciesOptions, "listLoadBalancerListenerPoliciesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *listLoadBalancerListenerPoliciesOptions.LoadBalancerID,
		"listener_id":      *listLoadBalancerListenerPoliciesOptions.ListenerID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listLoadBalancerListenerPoliciesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListLoadBalancerListenerPolicies")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicyCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateLoadBalancerListenerPolicy : Create a policy for a load balancer listener
// Creates a new policy for a load balancer listener.
func (vpc *VpcV1) CreateLoadBalancerListenerPolicy(createLoadBalancerListenerPolicyOptions *CreateLoadBalancerListenerPolicyOptions) (result *LoadBalancerListenerPolicy, response *core.DetailedResponse, err error) {
	return vpc.CreateLoadBalancerListenerPolicyWithContext(context.Background(), createLoadBalancerListenerPolicyOptions)
}

// CreateLoadBalancerListenerPolicyWithContext is an alternate form of the CreateLoadBalancerListenerPolicy method which supports a Context parameter
func (vpc *VpcV1) CreateLoadBalancerListenerPolicyWithContext(ctx context.Context, createLoadBalancerListenerPolicyOptions *CreateLoadBalancerListenerPolicyOptions) (result *LoadBalancerListenerPolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createLoadBalancerListenerPolicyOptions, "createLoadBalancerListenerPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createLoadBalancerListenerPolicyOptions, "createLoadBalancerListenerPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *createLoadBalancerListenerPolicyOptions.LoadBalancerID,
		"listener_id":      *createLoadBalancerListenerPolicyOptions.ListenerID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createLoadBalancerListenerPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateLoadBalancerListenerPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createLoadBalancerListenerPolicyOptions.Action != nil {
		body["action"] = createLoadBalancerListenerPolicyOptions.Action
	}
	if createLoadBalancerListenerPolicyOptions.Priority != nil {
		body["priority"] = createLoadBalancerListenerPolicyOptions.Priority
	}
	if createLoadBalancerListenerPolicyOptions.Name != nil {
		body["name"] = createLoadBalancerListenerPolicyOptions.Name
	}
	if createLoadBalancerListenerPolicyOptions.Rules != nil {
		body["rules"] = createLoadBalancerListenerPolicyOptions.Rules
	}
	if createLoadBalancerListenerPolicyOptions.Target != nil {
		body["target"] = createLoadBalancerListenerPolicyOptions.Target
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicy)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteLoadBalancerListenerPolicy : Delete a load balancer listener policy
// Deletes a policy of the load balancer listener. This operation cannot be reversed.
func (vpc *VpcV1) DeleteLoadBalancerListenerPolicy(deleteLoadBalancerListenerPolicyOptions *DeleteLoadBalancerListenerPolicyOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteLoadBalancerListenerPolicyWithContext(context.Background(), deleteLoadBalancerListenerPolicyOptions)
}

// DeleteLoadBalancerListenerPolicyWithContext is an alternate form of the DeleteLoadBalancerListenerPolicy method which supports a Context parameter
func (vpc *VpcV1) DeleteLoadBalancerListenerPolicyWithContext(ctx context.Context, deleteLoadBalancerListenerPolicyOptions *DeleteLoadBalancerListenerPolicyOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteLoadBalancerListenerPolicyOptions, "deleteLoadBalancerListenerPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteLoadBalancerListenerPolicyOptions, "deleteLoadBalancerListenerPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *deleteLoadBalancerListenerPolicyOptions.LoadBalancerID,
		"listener_id":      *deleteLoadBalancerListenerPolicyOptions.ListenerID,
		"id":               *deleteLoadBalancerListenerPolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteLoadBalancerListenerPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteLoadBalancerListenerPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetLoadBalancerListenerPolicy : Retrieve a load balancer listener policy
// Retrieve a single policy specified by the identifier in the URL path.
func (vpc *VpcV1) GetLoadBalancerListenerPolicy(getLoadBalancerListenerPolicyOptions *GetLoadBalancerListenerPolicyOptions) (result *LoadBalancerListenerPolicy, response *core.DetailedResponse, err error) {
	return vpc.GetLoadBalancerListenerPolicyWithContext(context.Background(), getLoadBalancerListenerPolicyOptions)
}

// GetLoadBalancerListenerPolicyWithContext is an alternate form of the GetLoadBalancerListenerPolicy method which supports a Context parameter
func (vpc *VpcV1) GetLoadBalancerListenerPolicyWithContext(ctx context.Context, getLoadBalancerListenerPolicyOptions *GetLoadBalancerListenerPolicyOptions) (result *LoadBalancerListenerPolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getLoadBalancerListenerPolicyOptions, "getLoadBalancerListenerPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getLoadBalancerListenerPolicyOptions, "getLoadBalancerListenerPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *getLoadBalancerListenerPolicyOptions.LoadBalancerID,
		"listener_id":      *getLoadBalancerListenerPolicyOptions.ListenerID,
		"id":               *getLoadBalancerListenerPolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getLoadBalancerListenerPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetLoadBalancerListenerPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicy)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateLoadBalancerListenerPolicy : Update a load balancer listener policy
// Updates a policy from a policy patch.
func (vpc *VpcV1) UpdateLoadBalancerListenerPolicy(updateLoadBalancerListenerPolicyOptions *UpdateLoadBalancerListenerPolicyOptions) (result *LoadBalancerListenerPolicy, response *core.DetailedResponse, err error) {
	return vpc.UpdateLoadBalancerListenerPolicyWithContext(context.Background(), updateLoadBalancerListenerPolicyOptions)
}

// UpdateLoadBalancerListenerPolicyWithContext is an alternate form of the UpdateLoadBalancerListenerPolicy method which supports a Context parameter
func (vpc *VpcV1) UpdateLoadBalancerListenerPolicyWithContext(ctx context.Context, updateLoadBalancerListenerPolicyOptions *UpdateLoadBalancerListenerPolicyOptions) (result *LoadBalancerListenerPolicy, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateLoadBalancerListenerPolicyOptions, "updateLoadBalancerListenerPolicyOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateLoadBalancerListenerPolicyOptions, "updateLoadBalancerListenerPolicyOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *updateLoadBalancerListenerPolicyOptions.LoadBalancerID,
		"listener_id":      *updateLoadBalancerListenerPolicyOptions.ListenerID,
		"id":               *updateLoadBalancerListenerPolicyOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateLoadBalancerListenerPolicyOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateLoadBalancerListenerPolicy")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateLoadBalancerListenerPolicyOptions.LoadBalancerListenerPolicyPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicy)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListLoadBalancerListenerPolicyRules : List all rules of a load balancer listener policy
// This request lists all rules of a load balancer listener policy.
func (vpc *VpcV1) ListLoadBalancerListenerPolicyRules(listLoadBalancerListenerPolicyRulesOptions *ListLoadBalancerListenerPolicyRulesOptions) (result *LoadBalancerListenerPolicyRuleCollection, response *core.DetailedResponse, err error) {
	return vpc.ListLoadBalancerListenerPolicyRulesWithContext(context.Background(), listLoadBalancerListenerPolicyRulesOptions)
}

// ListLoadBalancerListenerPolicyRulesWithContext is an alternate form of the ListLoadBalancerListenerPolicyRules method which supports a Context parameter
func (vpc *VpcV1) ListLoadBalancerListenerPolicyRulesWithContext(ctx context.Context, listLoadBalancerListenerPolicyRulesOptions *ListLoadBalancerListenerPolicyRulesOptions) (result *LoadBalancerListenerPolicyRuleCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listLoadBalancerListenerPolicyRulesOptions, "listLoadBalancerListenerPolicyRulesOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listLoadBalancerListenerPolicyRulesOptions, "listLoadBalancerListenerPolicyRulesOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *listLoadBalancerListenerPolicyRulesOptions.LoadBalancerID,
		"listener_id":      *listLoadBalancerListenerPolicyRulesOptions.ListenerID,
		"policy_id":        *listLoadBalancerListenerPolicyRulesOptions.PolicyID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{policy_id}/rules`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listLoadBalancerListenerPolicyRulesOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListLoadBalancerListenerPolicyRules")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicyRuleCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateLoadBalancerListenerPolicyRule : Create a rule for a load balancer listener policy
// Creates a new rule for the load balancer listener policy.
func (vpc *VpcV1) CreateLoadBalancerListenerPolicyRule(createLoadBalancerListenerPolicyRuleOptions *CreateLoadBalancerListenerPolicyRuleOptions) (result *LoadBalancerListenerPolicyRule, response *core.DetailedResponse, err error) {
	return vpc.CreateLoadBalancerListenerPolicyRuleWithContext(context.Background(), createLoadBalancerListenerPolicyRuleOptions)
}

// CreateLoadBalancerListenerPolicyRuleWithContext is an alternate form of the CreateLoadBalancerListenerPolicyRule method which supports a Context parameter
func (vpc *VpcV1) CreateLoadBalancerListenerPolicyRuleWithContext(ctx context.Context, createLoadBalancerListenerPolicyRuleOptions *CreateLoadBalancerListenerPolicyRuleOptions) (result *LoadBalancerListenerPolicyRule, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createLoadBalancerListenerPolicyRuleOptions, "createLoadBalancerListenerPolicyRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createLoadBalancerListenerPolicyRuleOptions, "createLoadBalancerListenerPolicyRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *createLoadBalancerListenerPolicyRuleOptions.LoadBalancerID,
		"listener_id":      *createLoadBalancerListenerPolicyRuleOptions.ListenerID,
		"policy_id":        *createLoadBalancerListenerPolicyRuleOptions.PolicyID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{policy_id}/rules`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createLoadBalancerListenerPolicyRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateLoadBalancerListenerPolicyRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createLoadBalancerListenerPolicyRuleOptions.Condition != nil {
		body["condition"] = createLoadBalancerListenerPolicyRuleOptions.Condition
	}
	if createLoadBalancerListenerPolicyRuleOptions.Type != nil {
		body["type"] = createLoadBalancerListenerPolicyRuleOptions.Type
	}
	if createLoadBalancerListenerPolicyRuleOptions.Value != nil {
		body["value"] = createLoadBalancerListenerPolicyRuleOptions.Value
	}
	if createLoadBalancerListenerPolicyRuleOptions.Field != nil {
		body["field"] = createLoadBalancerListenerPolicyRuleOptions.Field
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicyRule)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteLoadBalancerListenerPolicyRule : Delete a load balancer listener policy rule
// Deletes a rule from the load balancer listener policy. This operation cannot be reversed.
func (vpc *VpcV1) DeleteLoadBalancerListenerPolicyRule(deleteLoadBalancerListenerPolicyRuleOptions *DeleteLoadBalancerListenerPolicyRuleOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteLoadBalancerListenerPolicyRuleWithContext(context.Background(), deleteLoadBalancerListenerPolicyRuleOptions)
}

// DeleteLoadBalancerListenerPolicyRuleWithContext is an alternate form of the DeleteLoadBalancerListenerPolicyRule method which supports a Context parameter
func (vpc *VpcV1) DeleteLoadBalancerListenerPolicyRuleWithContext(ctx context.Context, deleteLoadBalancerListenerPolicyRuleOptions *DeleteLoadBalancerListenerPolicyRuleOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteLoadBalancerListenerPolicyRuleOptions, "deleteLoadBalancerListenerPolicyRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteLoadBalancerListenerPolicyRuleOptions, "deleteLoadBalancerListenerPolicyRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *deleteLoadBalancerListenerPolicyRuleOptions.LoadBalancerID,
		"listener_id":      *deleteLoadBalancerListenerPolicyRuleOptions.ListenerID,
		"policy_id":        *deleteLoadBalancerListenerPolicyRuleOptions.PolicyID,
		"id":               *deleteLoadBalancerListenerPolicyRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{policy_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteLoadBalancerListenerPolicyRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteLoadBalancerListenerPolicyRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetLoadBalancerListenerPolicyRule : Retrieve a load balancer listener policy rule
// Retrieves a single rule specified by the identifier in the URL path.
func (vpc *VpcV1) GetLoadBalancerListenerPolicyRule(getLoadBalancerListenerPolicyRuleOptions *GetLoadBalancerListenerPolicyRuleOptions) (result *LoadBalancerListenerPolicyRule, response *core.DetailedResponse, err error) {
	return vpc.GetLoadBalancerListenerPolicyRuleWithContext(context.Background(), getLoadBalancerListenerPolicyRuleOptions)
}

// GetLoadBalancerListenerPolicyRuleWithContext is an alternate form of the GetLoadBalancerListenerPolicyRule method which supports a Context parameter
func (vpc *VpcV1) GetLoadBalancerListenerPolicyRuleWithContext(ctx context.Context, getLoadBalancerListenerPolicyRuleOptions *GetLoadBalancerListenerPolicyRuleOptions) (result *LoadBalancerListenerPolicyRule, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getLoadBalancerListenerPolicyRuleOptions, "getLoadBalancerListenerPolicyRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getLoadBalancerListenerPolicyRuleOptions, "getLoadBalancerListenerPolicyRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *getLoadBalancerListenerPolicyRuleOptions.LoadBalancerID,
		"listener_id":      *getLoadBalancerListenerPolicyRuleOptions.ListenerID,
		"policy_id":        *getLoadBalancerListenerPolicyRuleOptions.PolicyID,
		"id":               *getLoadBalancerListenerPolicyRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{policy_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getLoadBalancerListenerPolicyRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetLoadBalancerListenerPolicyRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicyRule)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateLoadBalancerListenerPolicyRule : Update a load balancer listener policy rule
// Updates a rule of the load balancer listener policy.
func (vpc *VpcV1) UpdateLoadBalancerListenerPolicyRule(updateLoadBalancerListenerPolicyRuleOptions *UpdateLoadBalancerListenerPolicyRuleOptions) (result *LoadBalancerListenerPolicyRule, response *core.DetailedResponse, err error) {
	return vpc.UpdateLoadBalancerListenerPolicyRuleWithContext(context.Background(), updateLoadBalancerListenerPolicyRuleOptions)
}

// UpdateLoadBalancerListenerPolicyRuleWithContext is an alternate form of the UpdateLoadBalancerListenerPolicyRule method which supports a Context parameter
func (vpc *VpcV1) UpdateLoadBalancerListenerPolicyRuleWithContext(ctx context.Context, updateLoadBalancerListenerPolicyRuleOptions *UpdateLoadBalancerListenerPolicyRuleOptions) (result *LoadBalancerListenerPolicyRule, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateLoadBalancerListenerPolicyRuleOptions, "updateLoadBalancerListenerPolicyRuleOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateLoadBalancerListenerPolicyRuleOptions, "updateLoadBalancerListenerPolicyRuleOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *updateLoadBalancerListenerPolicyRuleOptions.LoadBalancerID,
		"listener_id":      *updateLoadBalancerListenerPolicyRuleOptions.ListenerID,
		"policy_id":        *updateLoadBalancerListenerPolicyRuleOptions.PolicyID,
		"id":               *updateLoadBalancerListenerPolicyRuleOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{policy_id}/rules/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateLoadBalancerListenerPolicyRuleOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateLoadBalancerListenerPolicyRule")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateLoadBalancerListenerPolicyRuleOptions.LoadBalancerListenerPolicyRulePatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerListenerPolicyRule)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListLoadBalancerPools : List all pools of a load balancer
// This request lists all pools of a load balancer.
func (vpc *VpcV1) ListLoadBalancerPools(listLoadBalancerPoolsOptions *ListLoadBalancerPoolsOptions) (result *LoadBalancerPoolCollection, response *core.DetailedResponse, err error) {
	return vpc.ListLoadBalancerPoolsWithContext(context.Background(), listLoadBalancerPoolsOptions)
}

// ListLoadBalancerPoolsWithContext is an alternate form of the ListLoadBalancerPools method which supports a Context parameter
func (vpc *VpcV1) ListLoadBalancerPoolsWithContext(ctx context.Context, listLoadBalancerPoolsOptions *ListLoadBalancerPoolsOptions) (result *LoadBalancerPoolCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listLoadBalancerPoolsOptions, "listLoadBalancerPoolsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listLoadBalancerPoolsOptions, "listLoadBalancerPoolsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *listLoadBalancerPoolsOptions.LoadBalancerID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listLoadBalancerPoolsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListLoadBalancerPools")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPoolCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateLoadBalancerPool : Create a load balancer pool
// This request creates a new pool from a pool prototype object.
func (vpc *VpcV1) CreateLoadBalancerPool(createLoadBalancerPoolOptions *CreateLoadBalancerPoolOptions) (result *LoadBalancerPool, response *core.DetailedResponse, err error) {
	return vpc.CreateLoadBalancerPoolWithContext(context.Background(), createLoadBalancerPoolOptions)
}

// CreateLoadBalancerPoolWithContext is an alternate form of the CreateLoadBalancerPool method which supports a Context parameter
func (vpc *VpcV1) CreateLoadBalancerPoolWithContext(ctx context.Context, createLoadBalancerPoolOptions *CreateLoadBalancerPoolOptions) (result *LoadBalancerPool, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createLoadBalancerPoolOptions, "createLoadBalancerPoolOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createLoadBalancerPoolOptions, "createLoadBalancerPoolOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *createLoadBalancerPoolOptions.LoadBalancerID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createLoadBalancerPoolOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateLoadBalancerPool")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createLoadBalancerPoolOptions.Algorithm != nil {
		body["algorithm"] = createLoadBalancerPoolOptions.Algorithm
	}
	if createLoadBalancerPoolOptions.HealthMonitor != nil {
		body["health_monitor"] = createLoadBalancerPoolOptions.HealthMonitor
	}
	if createLoadBalancerPoolOptions.Protocol != nil {
		body["protocol"] = createLoadBalancerPoolOptions.Protocol
	}
	if createLoadBalancerPoolOptions.Members != nil {
		body["members"] = createLoadBalancerPoolOptions.Members
	}
	if createLoadBalancerPoolOptions.Name != nil {
		body["name"] = createLoadBalancerPoolOptions.Name
	}
	if createLoadBalancerPoolOptions.ProxyProtocol != nil {
		body["proxy_protocol"] = createLoadBalancerPoolOptions.ProxyProtocol
	}
	if createLoadBalancerPoolOptions.SessionPersistence != nil {
		body["session_persistence"] = createLoadBalancerPoolOptions.SessionPersistence
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPool)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteLoadBalancerPool : Delete a load balancer pool
// This request deletes a load balancer pool. This operation cannot be reversed. The pool must not currently be the
// default pool for any listener in the load balancer.
func (vpc *VpcV1) DeleteLoadBalancerPool(deleteLoadBalancerPoolOptions *DeleteLoadBalancerPoolOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteLoadBalancerPoolWithContext(context.Background(), deleteLoadBalancerPoolOptions)
}

// DeleteLoadBalancerPoolWithContext is an alternate form of the DeleteLoadBalancerPool method which supports a Context parameter
func (vpc *VpcV1) DeleteLoadBalancerPoolWithContext(ctx context.Context, deleteLoadBalancerPoolOptions *DeleteLoadBalancerPoolOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteLoadBalancerPoolOptions, "deleteLoadBalancerPoolOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteLoadBalancerPoolOptions, "deleteLoadBalancerPoolOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *deleteLoadBalancerPoolOptions.LoadBalancerID,
		"id":               *deleteLoadBalancerPoolOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteLoadBalancerPoolOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteLoadBalancerPool")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetLoadBalancerPool : Retrieve a load balancer pool
// This request retrieves a single pool specified by the identifier in the URL path.
func (vpc *VpcV1) GetLoadBalancerPool(getLoadBalancerPoolOptions *GetLoadBalancerPoolOptions) (result *LoadBalancerPool, response *core.DetailedResponse, err error) {
	return vpc.GetLoadBalancerPoolWithContext(context.Background(), getLoadBalancerPoolOptions)
}

// GetLoadBalancerPoolWithContext is an alternate form of the GetLoadBalancerPool method which supports a Context parameter
func (vpc *VpcV1) GetLoadBalancerPoolWithContext(ctx context.Context, getLoadBalancerPoolOptions *GetLoadBalancerPoolOptions) (result *LoadBalancerPool, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getLoadBalancerPoolOptions, "getLoadBalancerPoolOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getLoadBalancerPoolOptions, "getLoadBalancerPoolOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *getLoadBalancerPoolOptions.LoadBalancerID,
		"id":               *getLoadBalancerPoolOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getLoadBalancerPoolOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetLoadBalancerPool")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPool)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateLoadBalancerPool : Update a load balancer pool
// This request updates a load balancer pool from a pool patch.
func (vpc *VpcV1) UpdateLoadBalancerPool(updateLoadBalancerPoolOptions *UpdateLoadBalancerPoolOptions) (result *LoadBalancerPool, response *core.DetailedResponse, err error) {
	return vpc.UpdateLoadBalancerPoolWithContext(context.Background(), updateLoadBalancerPoolOptions)
}

// UpdateLoadBalancerPoolWithContext is an alternate form of the UpdateLoadBalancerPool method which supports a Context parameter
func (vpc *VpcV1) UpdateLoadBalancerPoolWithContext(ctx context.Context, updateLoadBalancerPoolOptions *UpdateLoadBalancerPoolOptions) (result *LoadBalancerPool, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateLoadBalancerPoolOptions, "updateLoadBalancerPoolOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateLoadBalancerPoolOptions, "updateLoadBalancerPoolOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *updateLoadBalancerPoolOptions.LoadBalancerID,
		"id":               *updateLoadBalancerPoolOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateLoadBalancerPoolOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateLoadBalancerPool")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateLoadBalancerPoolOptions.LoadBalancerPoolPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPool)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListLoadBalancerPoolMembers : List all members of a load balancer pool
// This request lists all members of a load balancer pool.
func (vpc *VpcV1) ListLoadBalancerPoolMembers(listLoadBalancerPoolMembersOptions *ListLoadBalancerPoolMembersOptions) (result *LoadBalancerPoolMemberCollection, response *core.DetailedResponse, err error) {
	return vpc.ListLoadBalancerPoolMembersWithContext(context.Background(), listLoadBalancerPoolMembersOptions)
}

// ListLoadBalancerPoolMembersWithContext is an alternate form of the ListLoadBalancerPoolMembers method which supports a Context parameter
func (vpc *VpcV1) ListLoadBalancerPoolMembersWithContext(ctx context.Context, listLoadBalancerPoolMembersOptions *ListLoadBalancerPoolMembersOptions) (result *LoadBalancerPoolMemberCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listLoadBalancerPoolMembersOptions, "listLoadBalancerPoolMembersOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listLoadBalancerPoolMembersOptions, "listLoadBalancerPoolMembersOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *listLoadBalancerPoolMembersOptions.LoadBalancerID,
		"pool_id":          *listLoadBalancerPoolMembersOptions.PoolID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{pool_id}/members`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listLoadBalancerPoolMembersOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListLoadBalancerPoolMembers")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPoolMemberCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateLoadBalancerPoolMember : Create a member in a load balancer pool
// This request creates a new member and adds the member to the pool.
func (vpc *VpcV1) CreateLoadBalancerPoolMember(createLoadBalancerPoolMemberOptions *CreateLoadBalancerPoolMemberOptions) (result *LoadBalancerPoolMember, response *core.DetailedResponse, err error) {
	return vpc.CreateLoadBalancerPoolMemberWithContext(context.Background(), createLoadBalancerPoolMemberOptions)
}

// CreateLoadBalancerPoolMemberWithContext is an alternate form of the CreateLoadBalancerPoolMember method which supports a Context parameter
func (vpc *VpcV1) CreateLoadBalancerPoolMemberWithContext(ctx context.Context, createLoadBalancerPoolMemberOptions *CreateLoadBalancerPoolMemberOptions) (result *LoadBalancerPoolMember, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createLoadBalancerPoolMemberOptions, "createLoadBalancerPoolMemberOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createLoadBalancerPoolMemberOptions, "createLoadBalancerPoolMemberOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *createLoadBalancerPoolMemberOptions.LoadBalancerID,
		"pool_id":          *createLoadBalancerPoolMemberOptions.PoolID,
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{pool_id}/members`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range createLoadBalancerPoolMemberOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateLoadBalancerPoolMember")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createLoadBalancerPoolMemberOptions.Port != nil {
		body["port"] = createLoadBalancerPoolMemberOptions.Port
	}
	if createLoadBalancerPoolMemberOptions.Target != nil {
		body["target"] = createLoadBalancerPoolMemberOptions.Target
	}
	if createLoadBalancerPoolMemberOptions.Weight != nil {
		body["weight"] = createLoadBalancerPoolMemberOptions.Weight
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPoolMember)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ReplaceLoadBalancerPoolMembers : Replace load balancer pool members
// This request replaces the existing members of the load balancer pool with new members created from the collection of
// member prototype objects.
func (vpc *VpcV1) ReplaceLoadBalancerPoolMembers(replaceLoadBalancerPoolMembersOptions *ReplaceLoadBalancerPoolMembersOptions) (result *LoadBalancerPoolMemberCollection, response *core.DetailedResponse, err error) {
	return vpc.ReplaceLoadBalancerPoolMembersWithContext(context.Background(), replaceLoadBalancerPoolMembersOptions)
}

// ReplaceLoadBalancerPoolMembersWithContext is an alternate form of the ReplaceLoadBalancerPoolMembers method which supports a Context parameter
func (vpc *VpcV1) ReplaceLoadBalancerPoolMembersWithContext(ctx context.Context, replaceLoadBalancerPoolMembersOptions *ReplaceLoadBalancerPoolMembersOptions) (result *LoadBalancerPoolMemberCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(replaceLoadBalancerPoolMembersOptions, "replaceLoadBalancerPoolMembersOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(replaceLoadBalancerPoolMembersOptions, "replaceLoadBalancerPoolMembersOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *replaceLoadBalancerPoolMembersOptions.LoadBalancerID,
		"pool_id":          *replaceLoadBalancerPoolMembersOptions.PoolID,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{pool_id}/members`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range replaceLoadBalancerPoolMembersOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ReplaceLoadBalancerPoolMembers")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if replaceLoadBalancerPoolMembersOptions.Members != nil {
		body["members"] = replaceLoadBalancerPoolMembersOptions.Members
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPoolMemberCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteLoadBalancerPoolMember : Delete a load balancer pool member
// This request deletes a member from the pool. This operation cannot be reversed.
func (vpc *VpcV1) DeleteLoadBalancerPoolMember(deleteLoadBalancerPoolMemberOptions *DeleteLoadBalancerPoolMemberOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteLoadBalancerPoolMemberWithContext(context.Background(), deleteLoadBalancerPoolMemberOptions)
}

// DeleteLoadBalancerPoolMemberWithContext is an alternate form of the DeleteLoadBalancerPoolMember method which supports a Context parameter
func (vpc *VpcV1) DeleteLoadBalancerPoolMemberWithContext(ctx context.Context, deleteLoadBalancerPoolMemberOptions *DeleteLoadBalancerPoolMemberOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteLoadBalancerPoolMemberOptions, "deleteLoadBalancerPoolMemberOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteLoadBalancerPoolMemberOptions, "deleteLoadBalancerPoolMemberOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *deleteLoadBalancerPoolMemberOptions.LoadBalancerID,
		"pool_id":          *deleteLoadBalancerPoolMemberOptions.PoolID,
		"id":               *deleteLoadBalancerPoolMemberOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{pool_id}/members/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteLoadBalancerPoolMemberOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteLoadBalancerPoolMember")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetLoadBalancerPoolMember : Retrieve a load balancer pool member
// This request retrieves a single member specified by the identifier in the URL path.
func (vpc *VpcV1) GetLoadBalancerPoolMember(getLoadBalancerPoolMemberOptions *GetLoadBalancerPoolMemberOptions) (result *LoadBalancerPoolMember, response *core.DetailedResponse, err error) {
	return vpc.GetLoadBalancerPoolMemberWithContext(context.Background(), getLoadBalancerPoolMemberOptions)
}

// GetLoadBalancerPoolMemberWithContext is an alternate form of the GetLoadBalancerPoolMember method which supports a Context parameter
func (vpc *VpcV1) GetLoadBalancerPoolMemberWithContext(ctx context.Context, getLoadBalancerPoolMemberOptions *GetLoadBalancerPoolMemberOptions) (result *LoadBalancerPoolMember, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getLoadBalancerPoolMemberOptions, "getLoadBalancerPoolMemberOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getLoadBalancerPoolMemberOptions, "getLoadBalancerPoolMemberOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *getLoadBalancerPoolMemberOptions.LoadBalancerID,
		"pool_id":          *getLoadBalancerPoolMemberOptions.PoolID,
		"id":               *getLoadBalancerPoolMemberOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{pool_id}/members/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getLoadBalancerPoolMemberOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetLoadBalancerPoolMember")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPoolMember)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateLoadBalancerPoolMember : Update a load balancer pool member
// This request updates an existing member from a member patch.
func (vpc *VpcV1) UpdateLoadBalancerPoolMember(updateLoadBalancerPoolMemberOptions *UpdateLoadBalancerPoolMemberOptions) (result *LoadBalancerPoolMember, response *core.DetailedResponse, err error) {
	return vpc.UpdateLoadBalancerPoolMemberWithContext(context.Background(), updateLoadBalancerPoolMemberOptions)
}

// UpdateLoadBalancerPoolMemberWithContext is an alternate form of the UpdateLoadBalancerPoolMember method which supports a Context parameter
func (vpc *VpcV1) UpdateLoadBalancerPoolMemberWithContext(ctx context.Context, updateLoadBalancerPoolMemberOptions *UpdateLoadBalancerPoolMemberOptions) (result *LoadBalancerPoolMember, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateLoadBalancerPoolMemberOptions, "updateLoadBalancerPoolMemberOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateLoadBalancerPoolMemberOptions, "updateLoadBalancerPoolMemberOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"load_balancer_id": *updateLoadBalancerPoolMemberOptions.LoadBalancerID,
		"pool_id":          *updateLoadBalancerPoolMemberOptions.PoolID,
		"id":               *updateLoadBalancerPoolMemberOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/load_balancers/{load_balancer_id}/pools/{pool_id}/members/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateLoadBalancerPoolMemberOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateLoadBalancerPoolMember")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateLoadBalancerPoolMemberOptions.LoadBalancerPoolMemberPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLoadBalancerPoolMember)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListEndpointGateways : List all endpoint gateways
// This request lists all endpoint gateways in the region. An endpoint gateway maps one or more reserved IPs in a VPC to
// a target outside the VPC.
func (vpc *VpcV1) ListEndpointGateways(listEndpointGatewaysOptions *ListEndpointGatewaysOptions) (result *EndpointGatewayCollection, response *core.DetailedResponse, err error) {
	return vpc.ListEndpointGatewaysWithContext(context.Background(), listEndpointGatewaysOptions)
}

// ListEndpointGatewaysWithContext is an alternate form of the ListEndpointGateways method which supports a Context parameter
func (vpc *VpcV1) ListEndpointGatewaysWithContext(ctx context.Context, listEndpointGatewaysOptions *ListEndpointGatewaysOptions) (result *EndpointGatewayCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listEndpointGatewaysOptions, "listEndpointGatewaysOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/endpoint_gateways`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listEndpointGatewaysOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListEndpointGateways")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listEndpointGatewaysOptions.Name != nil {
		builder.AddQuery("name", fmt.Sprint(*listEndpointGatewaysOptions.Name))
	}
	if listEndpointGatewaysOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listEndpointGatewaysOptions.Start))
	}
	if listEndpointGatewaysOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listEndpointGatewaysOptions.Limit))
	}
	if listEndpointGatewaysOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listEndpointGatewaysOptions.ResourceGroupID))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalEndpointGatewayCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateEndpointGateway : Create an endpoint gateway
// This request creates a new endpoint gateway. An endpoint gateway maps one or more reserved IPs in a VPC to a target
// outside the VPC.
func (vpc *VpcV1) CreateEndpointGateway(createEndpointGatewayOptions *CreateEndpointGatewayOptions) (result *EndpointGateway, response *core.DetailedResponse, err error) {
	return vpc.CreateEndpointGatewayWithContext(context.Background(), createEndpointGatewayOptions)
}

// CreateEndpointGatewayWithContext is an alternate form of the CreateEndpointGateway method which supports a Context parameter
func (vpc *VpcV1) CreateEndpointGatewayWithContext(ctx context.Context, createEndpointGatewayOptions *CreateEndpointGatewayOptions) (result *EndpointGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createEndpointGatewayOptions, "createEndpointGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createEndpointGatewayOptions, "createEndpointGatewayOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/endpoint_gateways`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createEndpointGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateEndpointGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createEndpointGatewayOptions.Target != nil {
		body["target"] = createEndpointGatewayOptions.Target
	}
	if createEndpointGatewayOptions.VPC != nil {
		body["vpc"] = createEndpointGatewayOptions.VPC
	}
	if createEndpointGatewayOptions.Ips != nil {
		body["ips"] = createEndpointGatewayOptions.Ips
	}
	if createEndpointGatewayOptions.Name != nil {
		body["name"] = createEndpointGatewayOptions.Name
	}
	if createEndpointGatewayOptions.ResourceGroup != nil {
		body["resource_group"] = createEndpointGatewayOptions.ResourceGroup
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalEndpointGateway)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListEndpointGatewayIps : List all reserved IPs bound to an endpoint gateway
// This request lists all reserved IPs bound to an endpoint gateway.
func (vpc *VpcV1) ListEndpointGatewayIps(listEndpointGatewayIpsOptions *ListEndpointGatewayIpsOptions) (result *ReservedIPCollectionEndpointGatewayContext, response *core.DetailedResponse, err error) {
	return vpc.ListEndpointGatewayIpsWithContext(context.Background(), listEndpointGatewayIpsOptions)
}

// ListEndpointGatewayIpsWithContext is an alternate form of the ListEndpointGatewayIps method which supports a Context parameter
func (vpc *VpcV1) ListEndpointGatewayIpsWithContext(ctx context.Context, listEndpointGatewayIpsOptions *ListEndpointGatewayIpsOptions) (result *ReservedIPCollectionEndpointGatewayContext, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listEndpointGatewayIpsOptions, "listEndpointGatewayIpsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listEndpointGatewayIpsOptions, "listEndpointGatewayIpsOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"endpoint_gateway_id": *listEndpointGatewayIpsOptions.EndpointGatewayID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/endpoint_gateways/{endpoint_gateway_id}/ips`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range listEndpointGatewayIpsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListEndpointGatewayIps")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listEndpointGatewayIpsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listEndpointGatewayIpsOptions.Start))
	}
	if listEndpointGatewayIpsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listEndpointGatewayIpsOptions.Limit))
	}
	if listEndpointGatewayIpsOptions.Sort != nil {
		builder.AddQuery("sort", fmt.Sprint(*listEndpointGatewayIpsOptions.Sort))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalReservedIPCollectionEndpointGatewayContext)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// RemoveEndpointGatewayIP : Unbind a reserved IP from an endpoint gateway
// This request unbinds the specified reserved IP from the specified endpoint gateway. If the reserved IP has
// `auto_delete` set to `true`, the reserved IP will be deleted.
func (vpc *VpcV1) RemoveEndpointGatewayIP(removeEndpointGatewayIPOptions *RemoveEndpointGatewayIPOptions) (response *core.DetailedResponse, err error) {
	return vpc.RemoveEndpointGatewayIPWithContext(context.Background(), removeEndpointGatewayIPOptions)
}

// RemoveEndpointGatewayIPWithContext is an alternate form of the RemoveEndpointGatewayIP method which supports a Context parameter
func (vpc *VpcV1) RemoveEndpointGatewayIPWithContext(ctx context.Context, removeEndpointGatewayIPOptions *RemoveEndpointGatewayIPOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(removeEndpointGatewayIPOptions, "removeEndpointGatewayIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(removeEndpointGatewayIPOptions, "removeEndpointGatewayIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"endpoint_gateway_id": *removeEndpointGatewayIPOptions.EndpointGatewayID,
		"id":                  *removeEndpointGatewayIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/endpoint_gateways/{endpoint_gateway_id}/ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range removeEndpointGatewayIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "RemoveEndpointGatewayIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetEndpointGatewayIP : Retrieve a reserved IP bound to an endpoint gateway
// This request a retrieves the specified reserved IP address if it is bound to the endpoint gateway specified in the
// URL.
func (vpc *VpcV1) GetEndpointGatewayIP(getEndpointGatewayIPOptions *GetEndpointGatewayIPOptions) (result *ReservedIP, response *core.DetailedResponse, err error) {
	return vpc.GetEndpointGatewayIPWithContext(context.Background(), getEndpointGatewayIPOptions)
}

// GetEndpointGatewayIPWithContext is an alternate form of the GetEndpointGatewayIP method which supports a Context parameter
func (vpc *VpcV1) GetEndpointGatewayIPWithContext(ctx context.Context, getEndpointGatewayIPOptions *GetEndpointGatewayIPOptions) (result *ReservedIP, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getEndpointGatewayIPOptions, "getEndpointGatewayIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getEndpointGatewayIPOptions, "getEndpointGatewayIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"endpoint_gateway_id": *getEndpointGatewayIPOptions.EndpointGatewayID,
		"id":                  *getEndpointGatewayIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/endpoint_gateways/{endpoint_gateway_id}/ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getEndpointGatewayIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetEndpointGatewayIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalReservedIP)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// AddEndpointGatewayIP : Bind a reserved IP to an endpoint gateway
// This request binds the specified reserved IP to the specified endpoint gateway. The reserved IP:
//
// - must currently be unbound
// - must not be in the same zone as any other reserved IP bound to the endpoint gateway.
func (vpc *VpcV1) AddEndpointGatewayIP(addEndpointGatewayIPOptions *AddEndpointGatewayIPOptions) (result *ReservedIP, response *core.DetailedResponse, err error) {
	return vpc.AddEndpointGatewayIPWithContext(context.Background(), addEndpointGatewayIPOptions)
}

// AddEndpointGatewayIPWithContext is an alternate form of the AddEndpointGatewayIP method which supports a Context parameter
func (vpc *VpcV1) AddEndpointGatewayIPWithContext(ctx context.Context, addEndpointGatewayIPOptions *AddEndpointGatewayIPOptions) (result *ReservedIP, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(addEndpointGatewayIPOptions, "addEndpointGatewayIPOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(addEndpointGatewayIPOptions, "addEndpointGatewayIPOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"endpoint_gateway_id": *addEndpointGatewayIPOptions.EndpointGatewayID,
		"id":                  *addEndpointGatewayIPOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PUT)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/endpoint_gateways/{endpoint_gateway_id}/ips/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range addEndpointGatewayIPOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "AddEndpointGatewayIP")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalReservedIP)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteEndpointGateway : Delete an endpoint gateway
// This request deletes an endpoint gateway. This operation cannot be reversed.
//
// Reserved IPs that were bound to the endpoint gateway will be released if their
// `auto_delete` property is set to true.
func (vpc *VpcV1) DeleteEndpointGateway(deleteEndpointGatewayOptions *DeleteEndpointGatewayOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteEndpointGatewayWithContext(context.Background(), deleteEndpointGatewayOptions)
}

// DeleteEndpointGatewayWithContext is an alternate form of the DeleteEndpointGateway method which supports a Context parameter
func (vpc *VpcV1) DeleteEndpointGatewayWithContext(ctx context.Context, deleteEndpointGatewayOptions *DeleteEndpointGatewayOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteEndpointGatewayOptions, "deleteEndpointGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteEndpointGatewayOptions, "deleteEndpointGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteEndpointGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/endpoint_gateways/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteEndpointGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteEndpointGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetEndpointGateway : Retrieve an endpoint gateway
// This request retrieves a single endpoint gateway specified by the identifier in the URL.
func (vpc *VpcV1) GetEndpointGateway(getEndpointGatewayOptions *GetEndpointGatewayOptions) (result *EndpointGateway, response *core.DetailedResponse, err error) {
	return vpc.GetEndpointGatewayWithContext(context.Background(), getEndpointGatewayOptions)
}

// GetEndpointGatewayWithContext is an alternate form of the GetEndpointGateway method which supports a Context parameter
func (vpc *VpcV1) GetEndpointGatewayWithContext(ctx context.Context, getEndpointGatewayOptions *GetEndpointGatewayOptions) (result *EndpointGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getEndpointGatewayOptions, "getEndpointGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getEndpointGatewayOptions, "getEndpointGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getEndpointGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/endpoint_gateways/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getEndpointGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetEndpointGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalEndpointGateway)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateEndpointGateway : Update an endpoint gateway
// This request updates an endpoint gateway's name.
func (vpc *VpcV1) UpdateEndpointGateway(updateEndpointGatewayOptions *UpdateEndpointGatewayOptions) (result *EndpointGateway, response *core.DetailedResponse, err error) {
	return vpc.UpdateEndpointGatewayWithContext(context.Background(), updateEndpointGatewayOptions)
}

// UpdateEndpointGatewayWithContext is an alternate form of the UpdateEndpointGateway method which supports a Context parameter
func (vpc *VpcV1) UpdateEndpointGatewayWithContext(ctx context.Context, updateEndpointGatewayOptions *UpdateEndpointGatewayOptions) (result *EndpointGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateEndpointGatewayOptions, "updateEndpointGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateEndpointGatewayOptions, "updateEndpointGatewayOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateEndpointGatewayOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/endpoint_gateways/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateEndpointGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateEndpointGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateEndpointGatewayOptions.EndpointGatewayPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalEndpointGateway)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// ListFlowLogCollectors : List all flow log collectors
// This request lists all flow log collectors in the region. A flow log collector summarizes data sent over one or more
// network interfaces within a VPC, depending on the chosen target.
func (vpc *VpcV1) ListFlowLogCollectors(listFlowLogCollectorsOptions *ListFlowLogCollectorsOptions) (result *FlowLogCollectorCollection, response *core.DetailedResponse, err error) {
	return vpc.ListFlowLogCollectorsWithContext(context.Background(), listFlowLogCollectorsOptions)
}

// ListFlowLogCollectorsWithContext is an alternate form of the ListFlowLogCollectors method which supports a Context parameter
func (vpc *VpcV1) ListFlowLogCollectorsWithContext(ctx context.Context, listFlowLogCollectorsOptions *ListFlowLogCollectorsOptions) (result *FlowLogCollectorCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listFlowLogCollectorsOptions, "listFlowLogCollectorsOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/flow_log_collectors`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range listFlowLogCollectorsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "ListFlowLogCollectors")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))
	if listFlowLogCollectorsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listFlowLogCollectorsOptions.Start))
	}
	if listFlowLogCollectorsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listFlowLogCollectorsOptions.Limit))
	}
	if listFlowLogCollectorsOptions.ResourceGroupID != nil {
		builder.AddQuery("resource_group.id", fmt.Sprint(*listFlowLogCollectorsOptions.ResourceGroupID))
	}
	if listFlowLogCollectorsOptions.Name != nil {
		builder.AddQuery("name", fmt.Sprint(*listFlowLogCollectorsOptions.Name))
	}
	if listFlowLogCollectorsOptions.VPCID != nil {
		builder.AddQuery("vpc.id", fmt.Sprint(*listFlowLogCollectorsOptions.VPCID))
	}
	if listFlowLogCollectorsOptions.VPCCRN != nil {
		builder.AddQuery("vpc.crn", fmt.Sprint(*listFlowLogCollectorsOptions.VPCCRN))
	}
	if listFlowLogCollectorsOptions.VPCName != nil {
		builder.AddQuery("vpc.name", fmt.Sprint(*listFlowLogCollectorsOptions.VPCName))
	}
	if listFlowLogCollectorsOptions.TargetID != nil {
		builder.AddQuery("target.id", fmt.Sprint(*listFlowLogCollectorsOptions.TargetID))
	}
	if listFlowLogCollectorsOptions.TargetResourceType != nil {
		builder.AddQuery("target.resource_type", fmt.Sprint(*listFlowLogCollectorsOptions.TargetResourceType))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFlowLogCollectorCollection)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// CreateFlowLogCollector : Create a flow log collector
// This request creates and starts a new flow log collector from a flow log collector prototype object. The prototype
// object is structured in the same way as a retrieved flow log collector, and contains the information necessary to
// create and start the new flow log collector.
func (vpc *VpcV1) CreateFlowLogCollector(createFlowLogCollectorOptions *CreateFlowLogCollectorOptions) (result *FlowLogCollector, response *core.DetailedResponse, err error) {
	return vpc.CreateFlowLogCollectorWithContext(context.Background(), createFlowLogCollectorOptions)
}

// CreateFlowLogCollectorWithContext is an alternate form of the CreateFlowLogCollector method which supports a Context parameter
func (vpc *VpcV1) CreateFlowLogCollectorWithContext(ctx context.Context, createFlowLogCollectorOptions *CreateFlowLogCollectorOptions) (result *FlowLogCollector, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createFlowLogCollectorOptions, "createFlowLogCollectorOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createFlowLogCollectorOptions, "createFlowLogCollectorOptions")
	if err != nil {
		return
	}

	builder := core.NewRequestBuilder(core.POST)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/flow_log_collectors`, nil)
	if err != nil {
		return
	}

	for headerName, headerValue := range createFlowLogCollectorOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "CreateFlowLogCollector")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	body := make(map[string]interface{})
	if createFlowLogCollectorOptions.StorageBucket != nil {
		body["storage_bucket"] = createFlowLogCollectorOptions.StorageBucket
	}
	if createFlowLogCollectorOptions.Target != nil {
		body["target"] = createFlowLogCollectorOptions.Target
	}
	if createFlowLogCollectorOptions.Active != nil {
		body["active"] = createFlowLogCollectorOptions.Active
	}
	if createFlowLogCollectorOptions.Name != nil {
		body["name"] = createFlowLogCollectorOptions.Name
	}
	if createFlowLogCollectorOptions.ResourceGroup != nil {
		body["resource_group"] = createFlowLogCollectorOptions.ResourceGroup
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFlowLogCollector)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// DeleteFlowLogCollector : Delete a flow log collector
// This request stops and deletes a flow log collector. Collected flow logs remain available within the flow log
// collector's bucket.
func (vpc *VpcV1) DeleteFlowLogCollector(deleteFlowLogCollectorOptions *DeleteFlowLogCollectorOptions) (response *core.DetailedResponse, err error) {
	return vpc.DeleteFlowLogCollectorWithContext(context.Background(), deleteFlowLogCollectorOptions)
}

// DeleteFlowLogCollectorWithContext is an alternate form of the DeleteFlowLogCollector method which supports a Context parameter
func (vpc *VpcV1) DeleteFlowLogCollectorWithContext(ctx context.Context, deleteFlowLogCollectorOptions *DeleteFlowLogCollectorOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteFlowLogCollectorOptions, "deleteFlowLogCollectorOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteFlowLogCollectorOptions, "deleteFlowLogCollectorOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *deleteFlowLogCollectorOptions.ID,
	}

	builder := core.NewRequestBuilder(core.DELETE)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/flow_log_collectors/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteFlowLogCollectorOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "DeleteFlowLogCollector")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = vpc.Service.Request(request, nil)

	return
}

// GetFlowLogCollector : Retrieve a flow log collector
// This request retrieves a single flow log collector specified by the identifier in the URL.
func (vpc *VpcV1) GetFlowLogCollector(getFlowLogCollectorOptions *GetFlowLogCollectorOptions) (result *FlowLogCollector, response *core.DetailedResponse, err error) {
	return vpc.GetFlowLogCollectorWithContext(context.Background(), getFlowLogCollectorOptions)
}

// GetFlowLogCollectorWithContext is an alternate form of the GetFlowLogCollector method which supports a Context parameter
func (vpc *VpcV1) GetFlowLogCollectorWithContext(ctx context.Context, getFlowLogCollectorOptions *GetFlowLogCollectorOptions) (result *FlowLogCollector, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getFlowLogCollectorOptions, "getFlowLogCollectorOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getFlowLogCollectorOptions, "getFlowLogCollectorOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *getFlowLogCollectorOptions.ID,
	}

	builder := core.NewRequestBuilder(core.GET)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/flow_log_collectors/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range getFlowLogCollectorOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "GetFlowLogCollector")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFlowLogCollector)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// UpdateFlowLogCollector : Update a flow log collector
// This request updates a flow log collector with the information in a provided flow log collector patch. The flow log
// collector patch object is structured in the same way as a retrieved flow log collector and contains only the
// information to be updated.
func (vpc *VpcV1) UpdateFlowLogCollector(updateFlowLogCollectorOptions *UpdateFlowLogCollectorOptions) (result *FlowLogCollector, response *core.DetailedResponse, err error) {
	return vpc.UpdateFlowLogCollectorWithContext(context.Background(), updateFlowLogCollectorOptions)
}

// UpdateFlowLogCollectorWithContext is an alternate form of the UpdateFlowLogCollector method which supports a Context parameter
func (vpc *VpcV1) UpdateFlowLogCollectorWithContext(ctx context.Context, updateFlowLogCollectorOptions *UpdateFlowLogCollectorOptions) (result *FlowLogCollector, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateFlowLogCollectorOptions, "updateFlowLogCollectorOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateFlowLogCollectorOptions, "updateFlowLogCollectorOptions")
	if err != nil {
		return
	}

	pathParamsMap := map[string]string{
		"id": *updateFlowLogCollectorOptions.ID,
	}

	builder := core.NewRequestBuilder(core.PATCH)
	builder = builder.WithContext(ctx)
	builder.EnableGzipCompression = vpc.GetEnableGzipCompression()
	_, err = builder.ResolveRequestURL(vpc.Service.Options.URL, `/flow_log_collectors/{id}`, pathParamsMap)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateFlowLogCollectorOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("vpc", "V1", "UpdateFlowLogCollector")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/merge-patch+json")

	builder.AddQuery("version", fmt.Sprint(*vpc.Version))
	builder.AddQuery("generation", fmt.Sprint(*vpc.generation))

	_, err = builder.SetBodyContentJSON(updateFlowLogCollectorOptions.FlowLogCollectorPatch)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = vpc.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	if rawResponse != nil {
		err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalFlowLogCollector)
		if err != nil {
			return
		}
		response.Result = result
	}

	return
}

// AddEndpointGatewayIPOptions : The AddEndpointGatewayIP options.
type AddEndpointGatewayIPOptions struct {
	// The endpoint gateway identifier.
	EndpointGatewayID *string `validate:"required,ne="`

	// The reserved IP identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewAddEndpointGatewayIPOptions : Instantiate AddEndpointGatewayIPOptions
func (*VpcV1) NewAddEndpointGatewayIPOptions(endpointGatewayID string, id string) *AddEndpointGatewayIPOptions {
	return &AddEndpointGatewayIPOptions{
		EndpointGatewayID: core.StringPtr(endpointGatewayID),
		ID:                core.StringPtr(id),
	}
}

// SetEndpointGatewayID : Allow user to set EndpointGatewayID
func (options *AddEndpointGatewayIPOptions) SetEndpointGatewayID(endpointGatewayID string) *AddEndpointGatewayIPOptions {
	options.EndpointGatewayID = core.StringPtr(endpointGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *AddEndpointGatewayIPOptions) SetID(id string) *AddEndpointGatewayIPOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *AddEndpointGatewayIPOptions) SetHeaders(param map[string]string) *AddEndpointGatewayIPOptions {
	options.Headers = param
	return options
}

// AddInstanceNetworkInterfaceFloatingIPOptions : The AddInstanceNetworkInterfaceFloatingIP options.
type AddInstanceNetworkInterfaceFloatingIPOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The network interface identifier.
	NetworkInterfaceID *string `validate:"required,ne="`

	// The floating IP identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewAddInstanceNetworkInterfaceFloatingIPOptions : Instantiate AddInstanceNetworkInterfaceFloatingIPOptions
func (*VpcV1) NewAddInstanceNetworkInterfaceFloatingIPOptions(instanceID string, networkInterfaceID string, id string) *AddInstanceNetworkInterfaceFloatingIPOptions {
	return &AddInstanceNetworkInterfaceFloatingIPOptions{
		InstanceID:         core.StringPtr(instanceID),
		NetworkInterfaceID: core.StringPtr(networkInterfaceID),
		ID:                 core.StringPtr(id),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *AddInstanceNetworkInterfaceFloatingIPOptions) SetInstanceID(instanceID string) *AddInstanceNetworkInterfaceFloatingIPOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetNetworkInterfaceID : Allow user to set NetworkInterfaceID
func (options *AddInstanceNetworkInterfaceFloatingIPOptions) SetNetworkInterfaceID(networkInterfaceID string) *AddInstanceNetworkInterfaceFloatingIPOptions {
	options.NetworkInterfaceID = core.StringPtr(networkInterfaceID)
	return options
}

// SetID : Allow user to set ID
func (options *AddInstanceNetworkInterfaceFloatingIPOptions) SetID(id string) *AddInstanceNetworkInterfaceFloatingIPOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *AddInstanceNetworkInterfaceFloatingIPOptions) SetHeaders(param map[string]string) *AddInstanceNetworkInterfaceFloatingIPOptions {
	options.Headers = param
	return options
}

// AddSecurityGroupNetworkInterfaceOptions : The AddSecurityGroupNetworkInterface options.
type AddSecurityGroupNetworkInterfaceOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// The network interface identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewAddSecurityGroupNetworkInterfaceOptions : Instantiate AddSecurityGroupNetworkInterfaceOptions
func (*VpcV1) NewAddSecurityGroupNetworkInterfaceOptions(securityGroupID string, id string) *AddSecurityGroupNetworkInterfaceOptions {
	return &AddSecurityGroupNetworkInterfaceOptions{
		SecurityGroupID: core.StringPtr(securityGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *AddSecurityGroupNetworkInterfaceOptions) SetSecurityGroupID(securityGroupID string) *AddSecurityGroupNetworkInterfaceOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *AddSecurityGroupNetworkInterfaceOptions) SetID(id string) *AddSecurityGroupNetworkInterfaceOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *AddSecurityGroupNetworkInterfaceOptions) SetHeaders(param map[string]string) *AddSecurityGroupNetworkInterfaceOptions {
	options.Headers = param
	return options
}

// AddVPNGatewayConnectionLocalCIDROptions : The AddVPNGatewayConnectionLocalCIDR options.
type AddVPNGatewayConnectionLocalCIDROptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// The address prefix part of the CIDR.
	CIDRPrefix *string `validate:"required,ne="`

	// The prefix length part of the CIDR.
	PrefixLength *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewAddVPNGatewayConnectionLocalCIDROptions : Instantiate AddVPNGatewayConnectionLocalCIDROptions
func (*VpcV1) NewAddVPNGatewayConnectionLocalCIDROptions(vpnGatewayID string, id string, cidrPrefix string, prefixLength string) *AddVPNGatewayConnectionLocalCIDROptions {
	return &AddVPNGatewayConnectionLocalCIDROptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
		CIDRPrefix:   core.StringPtr(cidrPrefix),
		PrefixLength: core.StringPtr(prefixLength),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *AddVPNGatewayConnectionLocalCIDROptions) SetVPNGatewayID(vpnGatewayID string) *AddVPNGatewayConnectionLocalCIDROptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *AddVPNGatewayConnectionLocalCIDROptions) SetID(id string) *AddVPNGatewayConnectionLocalCIDROptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetCIDRPrefix : Allow user to set CIDRPrefix
func (options *AddVPNGatewayConnectionLocalCIDROptions) SetCIDRPrefix(cidrPrefix string) *AddVPNGatewayConnectionLocalCIDROptions {
	options.CIDRPrefix = core.StringPtr(cidrPrefix)
	return options
}

// SetPrefixLength : Allow user to set PrefixLength
func (options *AddVPNGatewayConnectionLocalCIDROptions) SetPrefixLength(prefixLength string) *AddVPNGatewayConnectionLocalCIDROptions {
	options.PrefixLength = core.StringPtr(prefixLength)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *AddVPNGatewayConnectionLocalCIDROptions) SetHeaders(param map[string]string) *AddVPNGatewayConnectionLocalCIDROptions {
	options.Headers = param
	return options
}

// AddVPNGatewayConnectionPeerCIDROptions : The AddVPNGatewayConnectionPeerCIDR options.
type AddVPNGatewayConnectionPeerCIDROptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// The address prefix part of the CIDR.
	CIDRPrefix *string `validate:"required,ne="`

	// The prefix length part of the CIDR.
	PrefixLength *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewAddVPNGatewayConnectionPeerCIDROptions : Instantiate AddVPNGatewayConnectionPeerCIDROptions
func (*VpcV1) NewAddVPNGatewayConnectionPeerCIDROptions(vpnGatewayID string, id string, cidrPrefix string, prefixLength string) *AddVPNGatewayConnectionPeerCIDROptions {
	return &AddVPNGatewayConnectionPeerCIDROptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
		CIDRPrefix:   core.StringPtr(cidrPrefix),
		PrefixLength: core.StringPtr(prefixLength),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *AddVPNGatewayConnectionPeerCIDROptions) SetVPNGatewayID(vpnGatewayID string) *AddVPNGatewayConnectionPeerCIDROptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *AddVPNGatewayConnectionPeerCIDROptions) SetID(id string) *AddVPNGatewayConnectionPeerCIDROptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetCIDRPrefix : Allow user to set CIDRPrefix
func (options *AddVPNGatewayConnectionPeerCIDROptions) SetCIDRPrefix(cidrPrefix string) *AddVPNGatewayConnectionPeerCIDROptions {
	options.CIDRPrefix = core.StringPtr(cidrPrefix)
	return options
}

// SetPrefixLength : Allow user to set PrefixLength
func (options *AddVPNGatewayConnectionPeerCIDROptions) SetPrefixLength(prefixLength string) *AddVPNGatewayConnectionPeerCIDROptions {
	options.PrefixLength = core.StringPtr(prefixLength)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *AddVPNGatewayConnectionPeerCIDROptions) SetHeaders(param map[string]string) *AddVPNGatewayConnectionPeerCIDROptions {
	options.Headers = param
	return options
}

// AddressPrefix : AddressPrefix struct
type AddressPrefix struct {
	// The CIDR block for this prefix.
	CIDR *string `json:"cidr" validate:"required"`

	// The date and time that the prefix was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// Indicates whether subnets exist with addresses from this prefix.
	HasSubnets *bool `json:"has_subnets" validate:"required"`

	// The URL for this address prefix.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this address prefix.
	ID *string `json:"id" validate:"required"`

	// Indicates whether this is the default prefix for this zone in this VPC. If a default prefix was automatically
	// created when the VPC was created, the prefix is automatically named using a hyphenated list of randomly-selected
	// words, but may be updated with a user-specified name.
	IsDefault *bool `json:"is_default" validate:"required"`

	// The user-defined name for this address prefix. Names must be unique within the VPC the address prefix resides in.
	Name *string `json:"name" validate:"required"`

	// The zone this address prefix resides in.
	Zone *ZoneReference `json:"zone" validate:"required"`
}

// UnmarshalAddressPrefix unmarshals an instance of AddressPrefix from the specified map of raw messages.
func UnmarshalAddressPrefix(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(AddressPrefix)
	err = core.UnmarshalPrimitive(m, "cidr", &obj.CIDR)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "has_subnets", &obj.HasSubnets)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "is_default", &obj.IsDefault)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AddressPrefixCollection : AddressPrefixCollection struct
type AddressPrefixCollection struct {
	// Collection of address prefixes.
	AddressPrefixes []AddressPrefix `json:"address_prefixes" validate:"required"`

	// A link to the first page of resources.
	First *AddressPrefixCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *AddressPrefixCollectionNext `json:"next,omitempty"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`
}

// UnmarshalAddressPrefixCollection unmarshals an instance of AddressPrefixCollection from the specified map of raw messages.
func UnmarshalAddressPrefixCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(AddressPrefixCollection)
	err = core.UnmarshalModel(m, "address_prefixes", &obj.AddressPrefixes, UnmarshalAddressPrefix)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalAddressPrefixCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalAddressPrefixCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AddressPrefixCollectionFirst : A link to the first page of resources.
type AddressPrefixCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalAddressPrefixCollectionFirst unmarshals an instance of AddressPrefixCollectionFirst from the specified map of raw messages.
func UnmarshalAddressPrefixCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(AddressPrefixCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AddressPrefixCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type AddressPrefixCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalAddressPrefixCollectionNext unmarshals an instance of AddressPrefixCollectionNext from the specified map of raw messages.
func UnmarshalAddressPrefixCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(AddressPrefixCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AddressPrefixPatch : AddressPrefixPatch struct
type AddressPrefixPatch struct {
	// Indicates whether this is the default prefix for this zone in this VPC. Updating to true makes this prefix the
	// default prefix for this zone in this VPC, provided the VPC currently has no default address prefix for this zone.
	// Updating to false removes the default prefix for this zone in this VPC.
	IsDefault *bool `json:"is_default,omitempty"`

	// The user-defined name for this address prefix. Names must be unique within the VPC the address prefix resides in.
	Name *string `json:"name,omitempty"`
}

// UnmarshalAddressPrefixPatch unmarshals an instance of AddressPrefixPatch from the specified map of raw messages.
func UnmarshalAddressPrefixPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(AddressPrefixPatch)
	err = core.UnmarshalPrimitive(m, "is_default", &obj.IsDefault)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the AddressPrefixPatch
func (addressPrefixPatch *AddressPrefixPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(addressPrefixPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// CertificateInstanceIdentity : Identifies a certificate instance by a unique property.
// Models which "extend" this model:
// - CertificateInstanceIdentityByCRN
type CertificateInstanceIdentity struct {
	// The CRN for this certificate instance.
	CRN *string `json:"crn,omitempty"`
}

func (*CertificateInstanceIdentity) isaCertificateInstanceIdentity() bool {
	return true
}

type CertificateInstanceIdentityIntf interface {
	isaCertificateInstanceIdentity() bool
}

// UnmarshalCertificateInstanceIdentity unmarshals an instance of CertificateInstanceIdentity from the specified map of raw messages.
func UnmarshalCertificateInstanceIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(CertificateInstanceIdentity)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// CertificateInstanceReference : CertificateInstanceReference struct
type CertificateInstanceReference struct {
	// The CRN for this certificate instance.
	CRN *string `json:"crn" validate:"required"`
}

// UnmarshalCertificateInstanceReference unmarshals an instance of CertificateInstanceReference from the specified map of raw messages.
func UnmarshalCertificateInstanceReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(CertificateInstanceReference)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// CheckVPNGatewayConnectionLocalCIDROptions : The CheckVPNGatewayConnectionLocalCIDR options.
type CheckVPNGatewayConnectionLocalCIDROptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// The address prefix part of the CIDR.
	CIDRPrefix *string `validate:"required,ne="`

	// The prefix length part of the CIDR.
	PrefixLength *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCheckVPNGatewayConnectionLocalCIDROptions : Instantiate CheckVPNGatewayConnectionLocalCIDROptions
func (*VpcV1) NewCheckVPNGatewayConnectionLocalCIDROptions(vpnGatewayID string, id string, cidrPrefix string, prefixLength string) *CheckVPNGatewayConnectionLocalCIDROptions {
	return &CheckVPNGatewayConnectionLocalCIDROptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
		CIDRPrefix:   core.StringPtr(cidrPrefix),
		PrefixLength: core.StringPtr(prefixLength),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *CheckVPNGatewayConnectionLocalCIDROptions) SetVPNGatewayID(vpnGatewayID string) *CheckVPNGatewayConnectionLocalCIDROptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *CheckVPNGatewayConnectionLocalCIDROptions) SetID(id string) *CheckVPNGatewayConnectionLocalCIDROptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetCIDRPrefix : Allow user to set CIDRPrefix
func (options *CheckVPNGatewayConnectionLocalCIDROptions) SetCIDRPrefix(cidrPrefix string) *CheckVPNGatewayConnectionLocalCIDROptions {
	options.CIDRPrefix = core.StringPtr(cidrPrefix)
	return options
}

// SetPrefixLength : Allow user to set PrefixLength
func (options *CheckVPNGatewayConnectionLocalCIDROptions) SetPrefixLength(prefixLength string) *CheckVPNGatewayConnectionLocalCIDROptions {
	options.PrefixLength = core.StringPtr(prefixLength)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CheckVPNGatewayConnectionLocalCIDROptions) SetHeaders(param map[string]string) *CheckVPNGatewayConnectionLocalCIDROptions {
	options.Headers = param
	return options
}

// CheckVPNGatewayConnectionPeerCIDROptions : The CheckVPNGatewayConnectionPeerCIDR options.
type CheckVPNGatewayConnectionPeerCIDROptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// The address prefix part of the CIDR.
	CIDRPrefix *string `validate:"required,ne="`

	// The prefix length part of the CIDR.
	PrefixLength *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCheckVPNGatewayConnectionPeerCIDROptions : Instantiate CheckVPNGatewayConnectionPeerCIDROptions
func (*VpcV1) NewCheckVPNGatewayConnectionPeerCIDROptions(vpnGatewayID string, id string, cidrPrefix string, prefixLength string) *CheckVPNGatewayConnectionPeerCIDROptions {
	return &CheckVPNGatewayConnectionPeerCIDROptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
		CIDRPrefix:   core.StringPtr(cidrPrefix),
		PrefixLength: core.StringPtr(prefixLength),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *CheckVPNGatewayConnectionPeerCIDROptions) SetVPNGatewayID(vpnGatewayID string) *CheckVPNGatewayConnectionPeerCIDROptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *CheckVPNGatewayConnectionPeerCIDROptions) SetID(id string) *CheckVPNGatewayConnectionPeerCIDROptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetCIDRPrefix : Allow user to set CIDRPrefix
func (options *CheckVPNGatewayConnectionPeerCIDROptions) SetCIDRPrefix(cidrPrefix string) *CheckVPNGatewayConnectionPeerCIDROptions {
	options.CIDRPrefix = core.StringPtr(cidrPrefix)
	return options
}

// SetPrefixLength : Allow user to set PrefixLength
func (options *CheckVPNGatewayConnectionPeerCIDROptions) SetPrefixLength(prefixLength string) *CheckVPNGatewayConnectionPeerCIDROptions {
	options.PrefixLength = core.StringPtr(prefixLength)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CheckVPNGatewayConnectionPeerCIDROptions) SetHeaders(param map[string]string) *CheckVPNGatewayConnectionPeerCIDROptions {
	options.Headers = param
	return options
}

// CloudObjectStorageBucketIdentity : Identifies a Cloud Object Storage bucket by a unique property.
// Models which "extend" this model:
// - CloudObjectStorageBucketIdentityByName
type CloudObjectStorageBucketIdentity struct {
	// The globally unique name of this COS bucket.
	Name *string `json:"name,omitempty"`
}

func (*CloudObjectStorageBucketIdentity) isaCloudObjectStorageBucketIdentity() bool {
	return true
}

type CloudObjectStorageBucketIdentityIntf interface {
	isaCloudObjectStorageBucketIdentity() bool
}

// UnmarshalCloudObjectStorageBucketIdentity unmarshals an instance of CloudObjectStorageBucketIdentity from the specified map of raw messages.
func UnmarshalCloudObjectStorageBucketIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(CloudObjectStorageBucketIdentity)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// CloudObjectStorageBucketReference : CloudObjectStorageBucketReference struct
type CloudObjectStorageBucketReference struct {
	// The globally unique name of this COS bucket.
	Name *string `json:"name" validate:"required"`
}

// UnmarshalCloudObjectStorageBucketReference unmarshals an instance of CloudObjectStorageBucketReference from the specified map of raw messages.
func UnmarshalCloudObjectStorageBucketReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(CloudObjectStorageBucketReference)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// CreateDedicatedHostGroupOptions : The CreateDedicatedHostGroup options.
type CreateDedicatedHostGroupOptions struct {
	// The dedicated host profile class for hosts in this group.
	Class *string

	// The dedicated host profile family for hosts in this group.
	Family *string

	// The unique user-defined name for this dedicated host group. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// The zone this dedicated host group will reside in.
	Zone ZoneIdentityIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateDedicatedHostGroupOptions.Family property.
// The dedicated host profile family for hosts in this group.
const (
	CreateDedicatedHostGroupOptionsFamilyBalancedConst = "balanced"
	CreateDedicatedHostGroupOptionsFamilyComputeConst  = "compute"
	CreateDedicatedHostGroupOptionsFamilyMemoryConst   = "memory"
)

// NewCreateDedicatedHostGroupOptions : Instantiate CreateDedicatedHostGroupOptions
func (*VpcV1) NewCreateDedicatedHostGroupOptions() *CreateDedicatedHostGroupOptions {
	return &CreateDedicatedHostGroupOptions{}
}

// SetClass : Allow user to set Class
func (options *CreateDedicatedHostGroupOptions) SetClass(class string) *CreateDedicatedHostGroupOptions {
	options.Class = core.StringPtr(class)
	return options
}

// SetFamily : Allow user to set Family
func (options *CreateDedicatedHostGroupOptions) SetFamily(family string) *CreateDedicatedHostGroupOptions {
	options.Family = core.StringPtr(family)
	return options
}

// SetName : Allow user to set Name
func (options *CreateDedicatedHostGroupOptions) SetName(name string) *CreateDedicatedHostGroupOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateDedicatedHostGroupOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateDedicatedHostGroupOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetZone : Allow user to set Zone
func (options *CreateDedicatedHostGroupOptions) SetZone(zone ZoneIdentityIntf) *CreateDedicatedHostGroupOptions {
	options.Zone = zone
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateDedicatedHostGroupOptions) SetHeaders(param map[string]string) *CreateDedicatedHostGroupOptions {
	options.Headers = param
	return options
}

// CreateDedicatedHostOptions : The CreateDedicatedHost options.
type CreateDedicatedHostOptions struct {
	// The dedicated host prototype object.
	DedicatedHostPrototype DedicatedHostPrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateDedicatedHostOptions : Instantiate CreateDedicatedHostOptions
func (*VpcV1) NewCreateDedicatedHostOptions(dedicatedHostPrototype DedicatedHostPrototypeIntf) *CreateDedicatedHostOptions {
	return &CreateDedicatedHostOptions{
		DedicatedHostPrototype: dedicatedHostPrototype,
	}
}

// SetDedicatedHostPrototype : Allow user to set DedicatedHostPrototype
func (options *CreateDedicatedHostOptions) SetDedicatedHostPrototype(dedicatedHostPrototype DedicatedHostPrototypeIntf) *CreateDedicatedHostOptions {
	options.DedicatedHostPrototype = dedicatedHostPrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateDedicatedHostOptions) SetHeaders(param map[string]string) *CreateDedicatedHostOptions {
	options.Headers = param
	return options
}

// CreateEndpointGatewayOptions : The CreateEndpointGateway options.
type CreateEndpointGatewayOptions struct {
	// The target for this endpoint gateway.
	Target EndpointGatewayTargetPrototypeIntf `validate:"required"`

	// The VPC this endpoint gateway will serve.
	VPC VPCIdentityIntf `validate:"required"`

	// The reserved IPs to bind to this endpoint gateway. At most one reserved IP per zone is allowed.
	Ips []EndpointGatewayReservedIPIntf

	// The user-defined name for this endpoint gateway. If unspecified, the name will be a hyphenated list of
	// randomly-selected words. Names must be unique within the VPC this endpoint gateway is serving.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateEndpointGatewayOptions : Instantiate CreateEndpointGatewayOptions
func (*VpcV1) NewCreateEndpointGatewayOptions(target EndpointGatewayTargetPrototypeIntf, vpc VPCIdentityIntf) *CreateEndpointGatewayOptions {
	return &CreateEndpointGatewayOptions{
		Target: target,
		VPC:    vpc,
	}
}

// SetTarget : Allow user to set Target
func (options *CreateEndpointGatewayOptions) SetTarget(target EndpointGatewayTargetPrototypeIntf) *CreateEndpointGatewayOptions {
	options.Target = target
	return options
}

// SetVPC : Allow user to set VPC
func (options *CreateEndpointGatewayOptions) SetVPC(vpc VPCIdentityIntf) *CreateEndpointGatewayOptions {
	options.VPC = vpc
	return options
}

// SetIps : Allow user to set Ips
func (options *CreateEndpointGatewayOptions) SetIps(ips []EndpointGatewayReservedIPIntf) *CreateEndpointGatewayOptions {
	options.Ips = ips
	return options
}

// SetName : Allow user to set Name
func (options *CreateEndpointGatewayOptions) SetName(name string) *CreateEndpointGatewayOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateEndpointGatewayOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateEndpointGatewayOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateEndpointGatewayOptions) SetHeaders(param map[string]string) *CreateEndpointGatewayOptions {
	options.Headers = param
	return options
}

// CreateFloatingIPOptions : The CreateFloatingIP options.
type CreateFloatingIPOptions struct {
	// The floating IP prototype object.
	FloatingIPPrototype FloatingIPPrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateFloatingIPOptions : Instantiate CreateFloatingIPOptions
func (*VpcV1) NewCreateFloatingIPOptions(floatingIPPrototype FloatingIPPrototypeIntf) *CreateFloatingIPOptions {
	return &CreateFloatingIPOptions{
		FloatingIPPrototype: floatingIPPrototype,
	}
}

// SetFloatingIPPrototype : Allow user to set FloatingIPPrototype
func (options *CreateFloatingIPOptions) SetFloatingIPPrototype(floatingIPPrototype FloatingIPPrototypeIntf) *CreateFloatingIPOptions {
	options.FloatingIPPrototype = floatingIPPrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateFloatingIPOptions) SetHeaders(param map[string]string) *CreateFloatingIPOptions {
	options.Headers = param
	return options
}

// CreateFlowLogCollectorOptions : The CreateFlowLogCollector options.
type CreateFlowLogCollectorOptions struct {
	// The Cloud Object Storage bucket where the collected flows will be logged.
	// The bucket must exist and an IAM service authorization must grant
	// `IBM Cloud Flow Logs` resources of `VPC Infrastructure Services` writer
	// access to the bucket.
	StorageBucket CloudObjectStorageBucketIdentityIntf `validate:"required"`

	// The target this collector will collect flow logs for. If the target is an instance,
	// subnet, or VPC, flow logs will not be collected for any network interfaces within the
	// target that are themselves the target of a more specific flow log collector.
	Target FlowLogCollectorTargetPrototypeIntf `validate:"required"`

	// Indicates whether this collector will be active upon creation.
	Active *bool

	// The unique user-defined name for this flow log collector. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateFlowLogCollectorOptions : Instantiate CreateFlowLogCollectorOptions
func (*VpcV1) NewCreateFlowLogCollectorOptions(storageBucket CloudObjectStorageBucketIdentityIntf, target FlowLogCollectorTargetPrototypeIntf) *CreateFlowLogCollectorOptions {
	return &CreateFlowLogCollectorOptions{
		StorageBucket: storageBucket,
		Target:        target,
	}
}

// SetStorageBucket : Allow user to set StorageBucket
func (options *CreateFlowLogCollectorOptions) SetStorageBucket(storageBucket CloudObjectStorageBucketIdentityIntf) *CreateFlowLogCollectorOptions {
	options.StorageBucket = storageBucket
	return options
}

// SetTarget : Allow user to set Target
func (options *CreateFlowLogCollectorOptions) SetTarget(target FlowLogCollectorTargetPrototypeIntf) *CreateFlowLogCollectorOptions {
	options.Target = target
	return options
}

// SetActive : Allow user to set Active
func (options *CreateFlowLogCollectorOptions) SetActive(active bool) *CreateFlowLogCollectorOptions {
	options.Active = core.BoolPtr(active)
	return options
}

// SetName : Allow user to set Name
func (options *CreateFlowLogCollectorOptions) SetName(name string) *CreateFlowLogCollectorOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateFlowLogCollectorOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateFlowLogCollectorOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateFlowLogCollectorOptions) SetHeaders(param map[string]string) *CreateFlowLogCollectorOptions {
	options.Headers = param
	return options
}

// CreateIkePolicyOptions : The CreateIkePolicy options.
type CreateIkePolicyOptions struct {
	// The authentication algorithm.
	AuthenticationAlgorithm *string `validate:"required"`

	// The Diffie-Hellman group.
	DhGroup *int64 `validate:"required"`

	// The encryption algorithm.
	EncryptionAlgorithm *string `validate:"required"`

	// The IKE protocol version.
	IkeVersion *int64 `validate:"required"`

	// The key lifetime in seconds.
	KeyLifetime *int64

	// The user-defined name for this IKE policy.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateIkePolicyOptions.AuthenticationAlgorithm property.
// The authentication algorithm.
const (
	CreateIkePolicyOptionsAuthenticationAlgorithmMd5Const    = "md5"
	CreateIkePolicyOptionsAuthenticationAlgorithmSha1Const   = "sha1"
	CreateIkePolicyOptionsAuthenticationAlgorithmSha256Const = "sha256"
	CreateIkePolicyOptionsAuthenticationAlgorithmSha512Const = "sha512"
)

// Constants associated with the CreateIkePolicyOptions.EncryptionAlgorithm property.
// The encryption algorithm.
const (
	CreateIkePolicyOptionsEncryptionAlgorithmAes128Const    = "aes128"
	CreateIkePolicyOptionsEncryptionAlgorithmAes256Const    = "aes256"
	CreateIkePolicyOptionsEncryptionAlgorithmTripleDesConst = "triple_des"
)

// NewCreateIkePolicyOptions : Instantiate CreateIkePolicyOptions
func (*VpcV1) NewCreateIkePolicyOptions(authenticationAlgorithm string, dhGroup int64, encryptionAlgorithm string, ikeVersion int64) *CreateIkePolicyOptions {
	return &CreateIkePolicyOptions{
		AuthenticationAlgorithm: core.StringPtr(authenticationAlgorithm),
		DhGroup:                 core.Int64Ptr(dhGroup),
		EncryptionAlgorithm:     core.StringPtr(encryptionAlgorithm),
		IkeVersion:              core.Int64Ptr(ikeVersion),
	}
}

// SetAuthenticationAlgorithm : Allow user to set AuthenticationAlgorithm
func (options *CreateIkePolicyOptions) SetAuthenticationAlgorithm(authenticationAlgorithm string) *CreateIkePolicyOptions {
	options.AuthenticationAlgorithm = core.StringPtr(authenticationAlgorithm)
	return options
}

// SetDhGroup : Allow user to set DhGroup
func (options *CreateIkePolicyOptions) SetDhGroup(dhGroup int64) *CreateIkePolicyOptions {
	options.DhGroup = core.Int64Ptr(dhGroup)
	return options
}

// SetEncryptionAlgorithm : Allow user to set EncryptionAlgorithm
func (options *CreateIkePolicyOptions) SetEncryptionAlgorithm(encryptionAlgorithm string) *CreateIkePolicyOptions {
	options.EncryptionAlgorithm = core.StringPtr(encryptionAlgorithm)
	return options
}

// SetIkeVersion : Allow user to set IkeVersion
func (options *CreateIkePolicyOptions) SetIkeVersion(ikeVersion int64) *CreateIkePolicyOptions {
	options.IkeVersion = core.Int64Ptr(ikeVersion)
	return options
}

// SetKeyLifetime : Allow user to set KeyLifetime
func (options *CreateIkePolicyOptions) SetKeyLifetime(keyLifetime int64) *CreateIkePolicyOptions {
	options.KeyLifetime = core.Int64Ptr(keyLifetime)
	return options
}

// SetName : Allow user to set Name
func (options *CreateIkePolicyOptions) SetName(name string) *CreateIkePolicyOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateIkePolicyOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateIkePolicyOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateIkePolicyOptions) SetHeaders(param map[string]string) *CreateIkePolicyOptions {
	options.Headers = param
	return options
}

// CreateImageOptions : The CreateImage options.
type CreateImageOptions struct {
	// The image prototype object.
	ImagePrototype ImagePrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateImageOptions : Instantiate CreateImageOptions
func (*VpcV1) NewCreateImageOptions(imagePrototype ImagePrototypeIntf) *CreateImageOptions {
	return &CreateImageOptions{
		ImagePrototype: imagePrototype,
	}
}

// SetImagePrototype : Allow user to set ImagePrototype
func (options *CreateImageOptions) SetImagePrototype(imagePrototype ImagePrototypeIntf) *CreateImageOptions {
	options.ImagePrototype = imagePrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateImageOptions) SetHeaders(param map[string]string) *CreateImageOptions {
	options.Headers = param
	return options
}

// CreateInstanceActionOptions : The CreateInstanceAction options.
type CreateInstanceActionOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The type of action.
	Type *string `validate:"required"`

	// If set to true, the action will be forced immediately, and all queued actions deleted. Ignored for the start action.
	Force *bool

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateInstanceActionOptions.Type property.
// The type of action.
const (
	CreateInstanceActionOptionsTypeRebootConst = "reboot"
	CreateInstanceActionOptionsTypeStartConst  = "start"
	CreateInstanceActionOptionsTypeStopConst   = "stop"
)

// NewCreateInstanceActionOptions : Instantiate CreateInstanceActionOptions
func (*VpcV1) NewCreateInstanceActionOptions(instanceID string, typeVar string) *CreateInstanceActionOptions {
	return &CreateInstanceActionOptions{
		InstanceID: core.StringPtr(instanceID),
		Type:       core.StringPtr(typeVar),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *CreateInstanceActionOptions) SetInstanceID(instanceID string) *CreateInstanceActionOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetType : Allow user to set Type
func (options *CreateInstanceActionOptions) SetType(typeVar string) *CreateInstanceActionOptions {
	options.Type = core.StringPtr(typeVar)
	return options
}

// SetForce : Allow user to set Force
func (options *CreateInstanceActionOptions) SetForce(force bool) *CreateInstanceActionOptions {
	options.Force = core.BoolPtr(force)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateInstanceActionOptions) SetHeaders(param map[string]string) *CreateInstanceActionOptions {
	options.Headers = param
	return options
}

// CreateInstanceConsoleAccessTokenOptions : The CreateInstanceConsoleAccessToken options.
type CreateInstanceConsoleAccessTokenOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The instance console type for which this token may be used.
	ConsoleType *string `validate:"required"`

	// Indicates whether to disconnect an existing serial console session as the serial console cannot be shared.  This has
	// no effect on VNC consoles.
	Force *bool

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateInstanceConsoleAccessTokenOptions.ConsoleType property.
// The instance console type for which this token may be used.
const (
	CreateInstanceConsoleAccessTokenOptionsConsoleTypeSerialConst = "serial"
	CreateInstanceConsoleAccessTokenOptionsConsoleTypeVncConst    = "vnc"
)

// NewCreateInstanceConsoleAccessTokenOptions : Instantiate CreateInstanceConsoleAccessTokenOptions
func (*VpcV1) NewCreateInstanceConsoleAccessTokenOptions(instanceID string, consoleType string) *CreateInstanceConsoleAccessTokenOptions {
	return &CreateInstanceConsoleAccessTokenOptions{
		InstanceID:  core.StringPtr(instanceID),
		ConsoleType: core.StringPtr(consoleType),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *CreateInstanceConsoleAccessTokenOptions) SetInstanceID(instanceID string) *CreateInstanceConsoleAccessTokenOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetConsoleType : Allow user to set ConsoleType
func (options *CreateInstanceConsoleAccessTokenOptions) SetConsoleType(consoleType string) *CreateInstanceConsoleAccessTokenOptions {
	options.ConsoleType = core.StringPtr(consoleType)
	return options
}

// SetForce : Allow user to set Force
func (options *CreateInstanceConsoleAccessTokenOptions) SetForce(force bool) *CreateInstanceConsoleAccessTokenOptions {
	options.Force = core.BoolPtr(force)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateInstanceConsoleAccessTokenOptions) SetHeaders(param map[string]string) *CreateInstanceConsoleAccessTokenOptions {
	options.Headers = param
	return options
}

// CreateInstanceGroupManagerActionOptions : The CreateInstanceGroupManagerAction options.
type CreateInstanceGroupManagerActionOptions struct {
	// The instance group identifier.
	InstanceGroupID *string `validate:"required,ne="`

	// The instance group manager identifier.
	InstanceGroupManagerID *string `validate:"required,ne="`

	// The instance group manager action prototype object.
	InstanceGroupManagerActionPrototype InstanceGroupManagerActionPrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateInstanceGroupManagerActionOptions : Instantiate CreateInstanceGroupManagerActionOptions
func (*VpcV1) NewCreateInstanceGroupManagerActionOptions(instanceGroupID string, instanceGroupManagerID string, instanceGroupManagerActionPrototype InstanceGroupManagerActionPrototypeIntf) *CreateInstanceGroupManagerActionOptions {
	return &CreateInstanceGroupManagerActionOptions{
		InstanceGroupID:                     core.StringPtr(instanceGroupID),
		InstanceGroupManagerID:              core.StringPtr(instanceGroupManagerID),
		InstanceGroupManagerActionPrototype: instanceGroupManagerActionPrototype,
	}
}

// SetInstanceGroupID : Allow user to set InstanceGroupID
func (options *CreateInstanceGroupManagerActionOptions) SetInstanceGroupID(instanceGroupID string) *CreateInstanceGroupManagerActionOptions {
	options.InstanceGroupID = core.StringPtr(instanceGroupID)
	return options
}

// SetInstanceGroupManagerID : Allow user to set InstanceGroupManagerID
func (options *CreateInstanceGroupManagerActionOptions) SetInstanceGroupManagerID(instanceGroupManagerID string) *CreateInstanceGroupManagerActionOptions {
	options.InstanceGroupManagerID = core.StringPtr(instanceGroupManagerID)
	return options
}

// SetInstanceGroupManagerActionPrototype : Allow user to set InstanceGroupManagerActionPrototype
func (options *CreateInstanceGroupManagerActionOptions) SetInstanceGroupManagerActionPrototype(instanceGroupManagerActionPrototype InstanceGroupManagerActionPrototypeIntf) *CreateInstanceGroupManagerActionOptions {
	options.InstanceGroupManagerActionPrototype = instanceGroupManagerActionPrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateInstanceGroupManagerActionOptions) SetHeaders(param map[string]string) *CreateInstanceGroupManagerActionOptions {
	options.Headers = param
	return options
}

// CreateInstanceGroupManagerOptions : The CreateInstanceGroupManager options.
type CreateInstanceGroupManagerOptions struct {
	// The instance group identifier.
	InstanceGroupID *string `validate:"required,ne="`

	// The instance group manager prototype object.
	InstanceGroupManagerPrototype InstanceGroupManagerPrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateInstanceGroupManagerOptions : Instantiate CreateInstanceGroupManagerOptions
func (*VpcV1) NewCreateInstanceGroupManagerOptions(instanceGroupID string, instanceGroupManagerPrototype InstanceGroupManagerPrototypeIntf) *CreateInstanceGroupManagerOptions {
	return &CreateInstanceGroupManagerOptions{
		InstanceGroupID:               core.StringPtr(instanceGroupID),
		InstanceGroupManagerPrototype: instanceGroupManagerPrototype,
	}
}

// SetInstanceGroupID : Allow user to set InstanceGroupID
func (options *CreateInstanceGroupManagerOptions) SetInstanceGroupID(instanceGroupID string) *CreateInstanceGroupManagerOptions {
	options.InstanceGroupID = core.StringPtr(instanceGroupID)
	return options
}

// SetInstanceGroupManagerPrototype : Allow user to set InstanceGroupManagerPrototype
func (options *CreateInstanceGroupManagerOptions) SetInstanceGroupManagerPrototype(instanceGroupManagerPrototype InstanceGroupManagerPrototypeIntf) *CreateInstanceGroupManagerOptions {
	options.InstanceGroupManagerPrototype = instanceGroupManagerPrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateInstanceGroupManagerOptions) SetHeaders(param map[string]string) *CreateInstanceGroupManagerOptions {
	options.Headers = param
	return options
}

// CreateInstanceGroupManagerPolicyOptions : The CreateInstanceGroupManagerPolicy options.
type CreateInstanceGroupManagerPolicyOptions struct {
	// The instance group identifier.
	InstanceGroupID *string `validate:"required,ne="`

	// The instance group manager identifier.
	InstanceGroupManagerID *string `validate:"required,ne="`

	// The instance group manager policy prototype object.
	InstanceGroupManagerPolicyPrototype InstanceGroupManagerPolicyPrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateInstanceGroupManagerPolicyOptions : Instantiate CreateInstanceGroupManagerPolicyOptions
func (*VpcV1) NewCreateInstanceGroupManagerPolicyOptions(instanceGroupID string, instanceGroupManagerID string, instanceGroupManagerPolicyPrototype InstanceGroupManagerPolicyPrototypeIntf) *CreateInstanceGroupManagerPolicyOptions {
	return &CreateInstanceGroupManagerPolicyOptions{
		InstanceGroupID:                     core.StringPtr(instanceGroupID),
		InstanceGroupManagerID:              core.StringPtr(instanceGroupManagerID),
		InstanceGroupManagerPolicyPrototype: instanceGroupManagerPolicyPrototype,
	}
}

// SetInstanceGroupID : Allow user to set InstanceGroupID
func (options *CreateInstanceGroupManagerPolicyOptions) SetInstanceGroupID(instanceGroupID string) *CreateInstanceGroupManagerPolicyOptions {
	options.InstanceGroupID = core.StringPtr(instanceGroupID)
	return options
}

// SetInstanceGroupManagerID : Allow user to set InstanceGroupManagerID
func (options *CreateInstanceGroupManagerPolicyOptions) SetInstanceGroupManagerID(instanceGroupManagerID string) *CreateInstanceGroupManagerPolicyOptions {
	options.InstanceGroupManagerID = core.StringPtr(instanceGroupManagerID)
	return options
}

// SetInstanceGroupManagerPolicyPrototype : Allow user to set InstanceGroupManagerPolicyPrototype
func (options *CreateInstanceGroupManagerPolicyOptions) SetInstanceGroupManagerPolicyPrototype(instanceGroupManagerPolicyPrototype InstanceGroupManagerPolicyPrototypeIntf) *CreateInstanceGroupManagerPolicyOptions {
	options.InstanceGroupManagerPolicyPrototype = instanceGroupManagerPolicyPrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateInstanceGroupManagerPolicyOptions) SetHeaders(param map[string]string) *CreateInstanceGroupManagerPolicyOptions {
	options.Headers = param
	return options
}

// CreateInstanceGroupOptions : The CreateInstanceGroup options.
type CreateInstanceGroupOptions struct {
	// Instance template to use when creating new instances.
	InstanceTemplate InstanceTemplateIdentityIntf `validate:"required"`

	// The subnets to use when creating new instances.
	Subnets []SubnetIdentityIntf `validate:"required"`

	// Required if specifying a load balancer pool only. Used by the instance group when scaling up instances to supply the
	// port for the load balancer pool member.
	ApplicationPort *int64

	// The load balancer that the load balancer pool used by this group
	// is in. Must be supplied when using a load balancer pool.
	LoadBalancer LoadBalancerIdentityIntf

	// When specified, the load balancer pool will be managed by this
	// group. Instances created by this group will have a new load
	// balancer pool member in that pool created. Must be used with
	// `application_port`.
	LoadBalancerPool LoadBalancerPoolIdentityIntf

	// The number of instances in the instance group.
	MembershipCount *int64

	// The user-defined name for this instance group.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateInstanceGroupOptions : Instantiate CreateInstanceGroupOptions
func (*VpcV1) NewCreateInstanceGroupOptions(instanceTemplate InstanceTemplateIdentityIntf, subnets []SubnetIdentityIntf) *CreateInstanceGroupOptions {
	return &CreateInstanceGroupOptions{
		InstanceTemplate: instanceTemplate,
		Subnets:          subnets,
	}
}

// SetInstanceTemplate : Allow user to set InstanceTemplate
func (options *CreateInstanceGroupOptions) SetInstanceTemplate(instanceTemplate InstanceTemplateIdentityIntf) *CreateInstanceGroupOptions {
	options.InstanceTemplate = instanceTemplate
	return options
}

// SetSubnets : Allow user to set Subnets
func (options *CreateInstanceGroupOptions) SetSubnets(subnets []SubnetIdentityIntf) *CreateInstanceGroupOptions {
	options.Subnets = subnets
	return options
}

// SetApplicationPort : Allow user to set ApplicationPort
func (options *CreateInstanceGroupOptions) SetApplicationPort(applicationPort int64) *CreateInstanceGroupOptions {
	options.ApplicationPort = core.Int64Ptr(applicationPort)
	return options
}

// SetLoadBalancer : Allow user to set LoadBalancer
func (options *CreateInstanceGroupOptions) SetLoadBalancer(loadBalancer LoadBalancerIdentityIntf) *CreateInstanceGroupOptions {
	options.LoadBalancer = loadBalancer
	return options
}

// SetLoadBalancerPool : Allow user to set LoadBalancerPool
func (options *CreateInstanceGroupOptions) SetLoadBalancerPool(loadBalancerPool LoadBalancerPoolIdentityIntf) *CreateInstanceGroupOptions {
	options.LoadBalancerPool = loadBalancerPool
	return options
}

// SetMembershipCount : Allow user to set MembershipCount
func (options *CreateInstanceGroupOptions) SetMembershipCount(membershipCount int64) *CreateInstanceGroupOptions {
	options.MembershipCount = core.Int64Ptr(membershipCount)
	return options
}

// SetName : Allow user to set Name
func (options *CreateInstanceGroupOptions) SetName(name string) *CreateInstanceGroupOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateInstanceGroupOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateInstanceGroupOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateInstanceGroupOptions) SetHeaders(param map[string]string) *CreateInstanceGroupOptions {
	options.Headers = param
	return options
}

// CreateInstanceNetworkInterfaceOptions : The CreateInstanceNetworkInterface options.
type CreateInstanceNetworkInterfaceOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The associated subnet.
	Subnet SubnetIdentityIntf `validate:"required"`

	// Indicates whether source IP spoofing is allowed on this interface. If false, source IP spoofing is prevented on this
	// interface. If true, source IP spoofing is allowed on this interface.
	AllowIPSpoofing *bool

	// The user-defined name for this network interface. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string

	// The primary IPv4 address. If specified, it must be an available address on the network interface's subnet. If
	// unspecified, an available address on the subnet will be automatically selected.
	PrimaryIpv4Address *string

	// Collection of security groups.
	SecurityGroups []SecurityGroupIdentityIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateInstanceNetworkInterfaceOptions : Instantiate CreateInstanceNetworkInterfaceOptions
func (*VpcV1) NewCreateInstanceNetworkInterfaceOptions(instanceID string, subnet SubnetIdentityIntf) *CreateInstanceNetworkInterfaceOptions {
	return &CreateInstanceNetworkInterfaceOptions{
		InstanceID: core.StringPtr(instanceID),
		Subnet:     subnet,
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *CreateInstanceNetworkInterfaceOptions) SetInstanceID(instanceID string) *CreateInstanceNetworkInterfaceOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetSubnet : Allow user to set Subnet
func (options *CreateInstanceNetworkInterfaceOptions) SetSubnet(subnet SubnetIdentityIntf) *CreateInstanceNetworkInterfaceOptions {
	options.Subnet = subnet
	return options
}

// SetAllowIPSpoofing : Allow user to set AllowIPSpoofing
func (options *CreateInstanceNetworkInterfaceOptions) SetAllowIPSpoofing(allowIPSpoofing bool) *CreateInstanceNetworkInterfaceOptions {
	options.AllowIPSpoofing = core.BoolPtr(allowIPSpoofing)
	return options
}

// SetName : Allow user to set Name
func (options *CreateInstanceNetworkInterfaceOptions) SetName(name string) *CreateInstanceNetworkInterfaceOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetPrimaryIpv4Address : Allow user to set PrimaryIpv4Address
func (options *CreateInstanceNetworkInterfaceOptions) SetPrimaryIpv4Address(primaryIpv4Address string) *CreateInstanceNetworkInterfaceOptions {
	options.PrimaryIpv4Address = core.StringPtr(primaryIpv4Address)
	return options
}

// SetSecurityGroups : Allow user to set SecurityGroups
func (options *CreateInstanceNetworkInterfaceOptions) SetSecurityGroups(securityGroups []SecurityGroupIdentityIntf) *CreateInstanceNetworkInterfaceOptions {
	options.SecurityGroups = securityGroups
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateInstanceNetworkInterfaceOptions) SetHeaders(param map[string]string) *CreateInstanceNetworkInterfaceOptions {
	options.Headers = param
	return options
}

// CreateInstanceOptions : The CreateInstance options.
type CreateInstanceOptions struct {
	// The instance prototype object.
	InstancePrototype InstancePrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateInstanceOptions : Instantiate CreateInstanceOptions
func (*VpcV1) NewCreateInstanceOptions(instancePrototype InstancePrototypeIntf) *CreateInstanceOptions {
	return &CreateInstanceOptions{
		InstancePrototype: instancePrototype,
	}
}

// SetInstancePrototype : Allow user to set InstancePrototype
func (options *CreateInstanceOptions) SetInstancePrototype(instancePrototype InstancePrototypeIntf) *CreateInstanceOptions {
	options.InstancePrototype = instancePrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateInstanceOptions) SetHeaders(param map[string]string) *CreateInstanceOptions {
	options.Headers = param
	return options
}

// CreateInstanceTemplateOptions : The CreateInstanceTemplate options.
type CreateInstanceTemplateOptions struct {
	// The instance template prototype object.
	InstanceTemplatePrototype InstanceTemplatePrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateInstanceTemplateOptions : Instantiate CreateInstanceTemplateOptions
func (*VpcV1) NewCreateInstanceTemplateOptions(instanceTemplatePrototype InstanceTemplatePrototypeIntf) *CreateInstanceTemplateOptions {
	return &CreateInstanceTemplateOptions{
		InstanceTemplatePrototype: instanceTemplatePrototype,
	}
}

// SetInstanceTemplatePrototype : Allow user to set InstanceTemplatePrototype
func (options *CreateInstanceTemplateOptions) SetInstanceTemplatePrototype(instanceTemplatePrototype InstanceTemplatePrototypeIntf) *CreateInstanceTemplateOptions {
	options.InstanceTemplatePrototype = instanceTemplatePrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateInstanceTemplateOptions) SetHeaders(param map[string]string) *CreateInstanceTemplateOptions {
	options.Headers = param
	return options
}

// CreateInstanceVolumeAttachmentOptions : The CreateInstanceVolumeAttachment options.
type CreateInstanceVolumeAttachmentOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// An existing volume to attach to the instance, or a prototype object for a new volume.
	Volume VolumeAttachmentPrototypeVolumeIntf `validate:"required"`

	// If set to true, when deleting the instance the volume will also be deleted.
	DeleteVolumeOnInstanceDelete *bool

	// The user-defined name for this volume attachment. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateInstanceVolumeAttachmentOptions : Instantiate CreateInstanceVolumeAttachmentOptions
func (*VpcV1) NewCreateInstanceVolumeAttachmentOptions(instanceID string, volume VolumeAttachmentPrototypeVolumeIntf) *CreateInstanceVolumeAttachmentOptions {
	return &CreateInstanceVolumeAttachmentOptions{
		InstanceID: core.StringPtr(instanceID),
		Volume:     volume,
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *CreateInstanceVolumeAttachmentOptions) SetInstanceID(instanceID string) *CreateInstanceVolumeAttachmentOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetVolume : Allow user to set Volume
func (options *CreateInstanceVolumeAttachmentOptions) SetVolume(volume VolumeAttachmentPrototypeVolumeIntf) *CreateInstanceVolumeAttachmentOptions {
	options.Volume = volume
	return options
}

// SetDeleteVolumeOnInstanceDelete : Allow user to set DeleteVolumeOnInstanceDelete
func (options *CreateInstanceVolumeAttachmentOptions) SetDeleteVolumeOnInstanceDelete(deleteVolumeOnInstanceDelete bool) *CreateInstanceVolumeAttachmentOptions {
	options.DeleteVolumeOnInstanceDelete = core.BoolPtr(deleteVolumeOnInstanceDelete)
	return options
}

// SetName : Allow user to set Name
func (options *CreateInstanceVolumeAttachmentOptions) SetName(name string) *CreateInstanceVolumeAttachmentOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateInstanceVolumeAttachmentOptions) SetHeaders(param map[string]string) *CreateInstanceVolumeAttachmentOptions {
	options.Headers = param
	return options
}

// CreateIpsecPolicyOptions : The CreateIpsecPolicy options.
type CreateIpsecPolicyOptions struct {
	// The authentication algorithm.
	AuthenticationAlgorithm *string `validate:"required"`

	// The encryption algorithm.
	EncryptionAlgorithm *string `validate:"required"`

	// Perfect Forward Secrecy.
	Pfs *string `validate:"required"`

	// The key lifetime in seconds.
	KeyLifetime *int64

	// The user-defined name for this IPsec policy.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateIpsecPolicyOptions.AuthenticationAlgorithm property.
// The authentication algorithm.
const (
	CreateIpsecPolicyOptionsAuthenticationAlgorithmMd5Const    = "md5"
	CreateIpsecPolicyOptionsAuthenticationAlgorithmSha1Const   = "sha1"
	CreateIpsecPolicyOptionsAuthenticationAlgorithmSha256Const = "sha256"
	CreateIpsecPolicyOptionsAuthenticationAlgorithmSha512Const = "sha512"
)

// Constants associated with the CreateIpsecPolicyOptions.EncryptionAlgorithm property.
// The encryption algorithm.
const (
	CreateIpsecPolicyOptionsEncryptionAlgorithmAes128Const    = "aes128"
	CreateIpsecPolicyOptionsEncryptionAlgorithmAes256Const    = "aes256"
	CreateIpsecPolicyOptionsEncryptionAlgorithmTripleDesConst = "triple_des"
)

// Constants associated with the CreateIpsecPolicyOptions.Pfs property.
// Perfect Forward Secrecy.
const (
	CreateIpsecPolicyOptionsPfsDisabledConst = "disabled"
	CreateIpsecPolicyOptionsPfsGroup14Const  = "group_14"
	CreateIpsecPolicyOptionsPfsGroup19Const  = "group_19"
	CreateIpsecPolicyOptionsPfsGroup2Const   = "group_2"
	CreateIpsecPolicyOptionsPfsGroup5Const   = "group_5"
)

// NewCreateIpsecPolicyOptions : Instantiate CreateIpsecPolicyOptions
func (*VpcV1) NewCreateIpsecPolicyOptions(authenticationAlgorithm string, encryptionAlgorithm string, pfs string) *CreateIpsecPolicyOptions {
	return &CreateIpsecPolicyOptions{
		AuthenticationAlgorithm: core.StringPtr(authenticationAlgorithm),
		EncryptionAlgorithm:     core.StringPtr(encryptionAlgorithm),
		Pfs:                     core.StringPtr(pfs),
	}
}

// SetAuthenticationAlgorithm : Allow user to set AuthenticationAlgorithm
func (options *CreateIpsecPolicyOptions) SetAuthenticationAlgorithm(authenticationAlgorithm string) *CreateIpsecPolicyOptions {
	options.AuthenticationAlgorithm = core.StringPtr(authenticationAlgorithm)
	return options
}

// SetEncryptionAlgorithm : Allow user to set EncryptionAlgorithm
func (options *CreateIpsecPolicyOptions) SetEncryptionAlgorithm(encryptionAlgorithm string) *CreateIpsecPolicyOptions {
	options.EncryptionAlgorithm = core.StringPtr(encryptionAlgorithm)
	return options
}

// SetPfs : Allow user to set Pfs
func (options *CreateIpsecPolicyOptions) SetPfs(pfs string) *CreateIpsecPolicyOptions {
	options.Pfs = core.StringPtr(pfs)
	return options
}

// SetKeyLifetime : Allow user to set KeyLifetime
func (options *CreateIpsecPolicyOptions) SetKeyLifetime(keyLifetime int64) *CreateIpsecPolicyOptions {
	options.KeyLifetime = core.Int64Ptr(keyLifetime)
	return options
}

// SetName : Allow user to set Name
func (options *CreateIpsecPolicyOptions) SetName(name string) *CreateIpsecPolicyOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateIpsecPolicyOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateIpsecPolicyOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateIpsecPolicyOptions) SetHeaders(param map[string]string) *CreateIpsecPolicyOptions {
	options.Headers = param
	return options
}

// CreateKeyOptions : The CreateKey options.
type CreateKeyOptions struct {
	// A unique public SSH key to import, encoded in PEM format. The key (prior to encoding) must be either 2048 or 4096
	// bits long.
	PublicKey *string `validate:"required"`

	// The unique user-defined name for this key. If unspecified, the name will be a hyphenated list of randomly-selected
	// words.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// The crypto-system used by this key.
	Type *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateKeyOptions.Type property.
// The crypto-system used by this key.
const (
	CreateKeyOptionsTypeRsaConst = "rsa"
)

// NewCreateKeyOptions : Instantiate CreateKeyOptions
func (*VpcV1) NewCreateKeyOptions(publicKey string) *CreateKeyOptions {
	return &CreateKeyOptions{
		PublicKey: core.StringPtr(publicKey),
	}
}

// SetPublicKey : Allow user to set PublicKey
func (options *CreateKeyOptions) SetPublicKey(publicKey string) *CreateKeyOptions {
	options.PublicKey = core.StringPtr(publicKey)
	return options
}

// SetName : Allow user to set Name
func (options *CreateKeyOptions) SetName(name string) *CreateKeyOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateKeyOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateKeyOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetType : Allow user to set Type
func (options *CreateKeyOptions) SetType(typeVar string) *CreateKeyOptions {
	options.Type = core.StringPtr(typeVar)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateKeyOptions) SetHeaders(param map[string]string) *CreateKeyOptions {
	options.Headers = param
	return options
}

// CreateLoadBalancerListenerOptions : The CreateLoadBalancerListener options.
type CreateLoadBalancerListenerOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener port number. Each listener in the load balancer must have a unique
	// `port` and `protocol` combination.
	Port *int64 `validate:"required"`

	// The listener protocol. Load balancers in the `network` family support `tcp`. Load balancers in the `application`
	// family support `tcp`, `http`, and `https`. Each listener in the load balancer must have a unique `port` and
	// `protocol` combination.
	Protocol *string `validate:"required"`

	// If set to `true`, this listener will accept and forward PROXY protocol information. Supported by load balancers in
	// the `application` family (otherwise always `false`).
	AcceptProxyProtocol *bool

	// The certificate instance used for SSL termination. It is applicable only to `https`
	// protocol.
	CertificateInstance CertificateInstanceIdentityIntf

	// The connection limit of the listener.
	ConnectionLimit *int64

	// The default pool associated with the listener. The specified pool must:
	//
	// - Belong to this load balancer
	// - Have the same `protocol` as this listener
	// - Not already be the default pool for another listener.
	DefaultPool LoadBalancerPoolIdentityIntf

	// The policy prototype objects for this listener.
	Policies []LoadBalancerListenerPolicyPrototype

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateLoadBalancerListenerOptions.Protocol property.
// The listener protocol. Load balancers in the `network` family support `tcp`. Load balancers in the `application`
// family support `tcp`, `http`, and `https`. Each listener in the load balancer must have a unique `port` and
// `protocol` combination.
const (
	CreateLoadBalancerListenerOptionsProtocolHTTPConst  = "http"
	CreateLoadBalancerListenerOptionsProtocolHTTPSConst = "https"
	CreateLoadBalancerListenerOptionsProtocolTCPConst   = "tcp"
)

// NewCreateLoadBalancerListenerOptions : Instantiate CreateLoadBalancerListenerOptions
func (*VpcV1) NewCreateLoadBalancerListenerOptions(loadBalancerID string, port int64, protocol string) *CreateLoadBalancerListenerOptions {
	return &CreateLoadBalancerListenerOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		Port:           core.Int64Ptr(port),
		Protocol:       core.StringPtr(protocol),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *CreateLoadBalancerListenerOptions) SetLoadBalancerID(loadBalancerID string) *CreateLoadBalancerListenerOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetPort : Allow user to set Port
func (options *CreateLoadBalancerListenerOptions) SetPort(port int64) *CreateLoadBalancerListenerOptions {
	options.Port = core.Int64Ptr(port)
	return options
}

// SetProtocol : Allow user to set Protocol
func (options *CreateLoadBalancerListenerOptions) SetProtocol(protocol string) *CreateLoadBalancerListenerOptions {
	options.Protocol = core.StringPtr(protocol)
	return options
}

// SetAcceptProxyProtocol : Allow user to set AcceptProxyProtocol
func (options *CreateLoadBalancerListenerOptions) SetAcceptProxyProtocol(acceptProxyProtocol bool) *CreateLoadBalancerListenerOptions {
	options.AcceptProxyProtocol = core.BoolPtr(acceptProxyProtocol)
	return options
}

// SetCertificateInstance : Allow user to set CertificateInstance
func (options *CreateLoadBalancerListenerOptions) SetCertificateInstance(certificateInstance CertificateInstanceIdentityIntf) *CreateLoadBalancerListenerOptions {
	options.CertificateInstance = certificateInstance
	return options
}

// SetConnectionLimit : Allow user to set ConnectionLimit
func (options *CreateLoadBalancerListenerOptions) SetConnectionLimit(connectionLimit int64) *CreateLoadBalancerListenerOptions {
	options.ConnectionLimit = core.Int64Ptr(connectionLimit)
	return options
}

// SetDefaultPool : Allow user to set DefaultPool
func (options *CreateLoadBalancerListenerOptions) SetDefaultPool(defaultPool LoadBalancerPoolIdentityIntf) *CreateLoadBalancerListenerOptions {
	options.DefaultPool = defaultPool
	return options
}

// SetPolicies : Allow user to set Policies
func (options *CreateLoadBalancerListenerOptions) SetPolicies(policies []LoadBalancerListenerPolicyPrototype) *CreateLoadBalancerListenerOptions {
	options.Policies = policies
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateLoadBalancerListenerOptions) SetHeaders(param map[string]string) *CreateLoadBalancerListenerOptions {
	options.Headers = param
	return options
}

// CreateLoadBalancerListenerPolicyOptions : The CreateLoadBalancerListenerPolicy options.
type CreateLoadBalancerListenerPolicyOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// The policy action.
	Action *string `validate:"required"`

	// Priority of the policy. Lower value indicates higher priority.
	Priority *int64 `validate:"required"`

	// The user-defined name for this policy. Names must be unique within the load balancer listener the policy resides in.
	Name *string

	// The rule prototype objects for this policy.
	Rules []LoadBalancerListenerPolicyRulePrototype

	// When `action` is `forward`, `LoadBalancerPoolIdentity` is required to specify which
	// pool the load balancer forwards the traffic to. When `action` is `redirect`,
	// `LoadBalancerListenerPolicyRedirectURLPrototype` is required to specify the url and
	// http status code used in the redirect response.
	Target LoadBalancerListenerPolicyTargetPrototypeIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateLoadBalancerListenerPolicyOptions.Action property.
// The policy action.
const (
	CreateLoadBalancerListenerPolicyOptionsActionForwardConst  = "forward"
	CreateLoadBalancerListenerPolicyOptionsActionRedirectConst = "redirect"
	CreateLoadBalancerListenerPolicyOptionsActionRejectConst   = "reject"
)

// NewCreateLoadBalancerListenerPolicyOptions : Instantiate CreateLoadBalancerListenerPolicyOptions
func (*VpcV1) NewCreateLoadBalancerListenerPolicyOptions(loadBalancerID string, listenerID string, action string, priority int64) *CreateLoadBalancerListenerPolicyOptions {
	return &CreateLoadBalancerListenerPolicyOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ListenerID:     core.StringPtr(listenerID),
		Action:         core.StringPtr(action),
		Priority:       core.Int64Ptr(priority),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *CreateLoadBalancerListenerPolicyOptions) SetLoadBalancerID(loadBalancerID string) *CreateLoadBalancerListenerPolicyOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *CreateLoadBalancerListenerPolicyOptions) SetListenerID(listenerID string) *CreateLoadBalancerListenerPolicyOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetAction : Allow user to set Action
func (options *CreateLoadBalancerListenerPolicyOptions) SetAction(action string) *CreateLoadBalancerListenerPolicyOptions {
	options.Action = core.StringPtr(action)
	return options
}

// SetPriority : Allow user to set Priority
func (options *CreateLoadBalancerListenerPolicyOptions) SetPriority(priority int64) *CreateLoadBalancerListenerPolicyOptions {
	options.Priority = core.Int64Ptr(priority)
	return options
}

// SetName : Allow user to set Name
func (options *CreateLoadBalancerListenerPolicyOptions) SetName(name string) *CreateLoadBalancerListenerPolicyOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetRules : Allow user to set Rules
func (options *CreateLoadBalancerListenerPolicyOptions) SetRules(rules []LoadBalancerListenerPolicyRulePrototype) *CreateLoadBalancerListenerPolicyOptions {
	options.Rules = rules
	return options
}

// SetTarget : Allow user to set Target
func (options *CreateLoadBalancerListenerPolicyOptions) SetTarget(target LoadBalancerListenerPolicyTargetPrototypeIntf) *CreateLoadBalancerListenerPolicyOptions {
	options.Target = target
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateLoadBalancerListenerPolicyOptions) SetHeaders(param map[string]string) *CreateLoadBalancerListenerPolicyOptions {
	options.Headers = param
	return options
}

// CreateLoadBalancerListenerPolicyRuleOptions : The CreateLoadBalancerListenerPolicyRule options.
type CreateLoadBalancerListenerPolicyRuleOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// The policy identifier.
	PolicyID *string `validate:"required,ne="`

	// The condition of the rule.
	Condition *string `validate:"required"`

	// The type of the rule.
	//
	// Body rules are applied to form-encoded request bodies using the `UTF-8` character set.
	Type *string `validate:"required"`

	// Value to be matched for rule condition.
	//
	// If the rule type is `query` and the rule condition is not `matches_regex`, the value must be percent-encoded.
	Value *string `validate:"required"`

	// The field. This is applicable to `header`, `query`, and `body` rule types.
	//
	// If the rule type is `header`, this field is required.
	//
	// If the rule type is `query`, this is optional. If specified and the rule condition is not
	// `matches_regex`, the value must be percent-encoded.
	//
	// If the rule type is `body`, this is optional.
	Field *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateLoadBalancerListenerPolicyRuleOptions.Condition property.
// The condition of the rule.
const (
	CreateLoadBalancerListenerPolicyRuleOptionsConditionContainsConst     = "contains"
	CreateLoadBalancerListenerPolicyRuleOptionsConditionEqualsConst       = "equals"
	CreateLoadBalancerListenerPolicyRuleOptionsConditionMatchesRegexConst = "matches_regex"
)

// Constants associated with the CreateLoadBalancerListenerPolicyRuleOptions.Type property.
// The type of the rule.
//
// Body rules are applied to form-encoded request bodies using the `UTF-8` character set.
const (
	CreateLoadBalancerListenerPolicyRuleOptionsTypeBodyConst     = "body"
	CreateLoadBalancerListenerPolicyRuleOptionsTypeHeaderConst   = "header"
	CreateLoadBalancerListenerPolicyRuleOptionsTypeHostnameConst = "hostname"
	CreateLoadBalancerListenerPolicyRuleOptionsTypePathConst     = "path"
	CreateLoadBalancerListenerPolicyRuleOptionsTypeQueryConst    = "query"
)

// NewCreateLoadBalancerListenerPolicyRuleOptions : Instantiate CreateLoadBalancerListenerPolicyRuleOptions
func (*VpcV1) NewCreateLoadBalancerListenerPolicyRuleOptions(loadBalancerID string, listenerID string, policyID string, condition string, typeVar string, value string) *CreateLoadBalancerListenerPolicyRuleOptions {
	return &CreateLoadBalancerListenerPolicyRuleOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ListenerID:     core.StringPtr(listenerID),
		PolicyID:       core.StringPtr(policyID),
		Condition:      core.StringPtr(condition),
		Type:           core.StringPtr(typeVar),
		Value:          core.StringPtr(value),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetLoadBalancerID(loadBalancerID string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetListenerID(listenerID string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetPolicyID : Allow user to set PolicyID
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetPolicyID(policyID string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.PolicyID = core.StringPtr(policyID)
	return options
}

// SetCondition : Allow user to set Condition
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetCondition(condition string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.Condition = core.StringPtr(condition)
	return options
}

// SetType : Allow user to set Type
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetType(typeVar string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.Type = core.StringPtr(typeVar)
	return options
}

// SetValue : Allow user to set Value
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetValue(value string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.Value = core.StringPtr(value)
	return options
}

// SetField : Allow user to set Field
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetField(field string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.Field = core.StringPtr(field)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateLoadBalancerListenerPolicyRuleOptions) SetHeaders(param map[string]string) *CreateLoadBalancerListenerPolicyRuleOptions {
	options.Headers = param
	return options
}

// CreateLoadBalancerOptions : The CreateLoadBalancer options.
type CreateLoadBalancerOptions struct {
	// Indicates whether this load balancer is public or private.
	IsPublic *bool `validate:"required"`

	// The subnets to provision this load balancer.
	Subnets []SubnetIdentityIntf `validate:"required"`

	// The listeners of this load balancer.
	Listeners []LoadBalancerListenerPrototypeLoadBalancerContext

	// The logging configuration to use for this load balancer. See [VPC Datapath
	// Logging](https://cloud.ibm.com/docs/vpc?topic=vpc-datapath-logging)
	// on the logging format, fields and permitted values.
	//
	// To activate logging, the load balancer profile must support the specified logging
	// type.
	Logging *LoadBalancerLogging

	// The user-defined name for this load balancer. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string

	// The pools of this load balancer.
	Pools []LoadBalancerPoolPrototype

	// The profile to use for this load balancer.
	Profile LoadBalancerProfileIdentityIntf

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// The security groups to use for this load balancer.
	//
	// The load balancer profile must support security groups.
	SecurityGroups []SecurityGroupIdentityIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateLoadBalancerOptions : Instantiate CreateLoadBalancerOptions
func (*VpcV1) NewCreateLoadBalancerOptions(isPublic bool, subnets []SubnetIdentityIntf) *CreateLoadBalancerOptions {
	return &CreateLoadBalancerOptions{
		IsPublic: core.BoolPtr(isPublic),
		Subnets:  subnets,
	}
}

// SetIsPublic : Allow user to set IsPublic
func (options *CreateLoadBalancerOptions) SetIsPublic(isPublic bool) *CreateLoadBalancerOptions {
	options.IsPublic = core.BoolPtr(isPublic)
	return options
}

// SetSubnets : Allow user to set Subnets
func (options *CreateLoadBalancerOptions) SetSubnets(subnets []SubnetIdentityIntf) *CreateLoadBalancerOptions {
	options.Subnets = subnets
	return options
}

// SetListeners : Allow user to set Listeners
func (options *CreateLoadBalancerOptions) SetListeners(listeners []LoadBalancerListenerPrototypeLoadBalancerContext) *CreateLoadBalancerOptions {
	options.Listeners = listeners
	return options
}

// SetLogging : Allow user to set Logging
func (options *CreateLoadBalancerOptions) SetLogging(logging *LoadBalancerLogging) *CreateLoadBalancerOptions {
	options.Logging = logging
	return options
}

// SetName : Allow user to set Name
func (options *CreateLoadBalancerOptions) SetName(name string) *CreateLoadBalancerOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetPools : Allow user to set Pools
func (options *CreateLoadBalancerOptions) SetPools(pools []LoadBalancerPoolPrototype) *CreateLoadBalancerOptions {
	options.Pools = pools
	return options
}

// SetProfile : Allow user to set Profile
func (options *CreateLoadBalancerOptions) SetProfile(profile LoadBalancerProfileIdentityIntf) *CreateLoadBalancerOptions {
	options.Profile = profile
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateLoadBalancerOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateLoadBalancerOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetSecurityGroups : Allow user to set SecurityGroups
func (options *CreateLoadBalancerOptions) SetSecurityGroups(securityGroups []SecurityGroupIdentityIntf) *CreateLoadBalancerOptions {
	options.SecurityGroups = securityGroups
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateLoadBalancerOptions) SetHeaders(param map[string]string) *CreateLoadBalancerOptions {
	options.Headers = param
	return options
}

// CreateLoadBalancerPoolMemberOptions : The CreateLoadBalancerPoolMember options.
type CreateLoadBalancerPoolMemberOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The pool identifier.
	PoolID *string `validate:"required,ne="`

	// The port number of the application running in the server member.
	Port *int64 `validate:"required"`

	// The pool member target. Load balancers in the `network` family support virtual server
	// instances. Load balancers in the `application` family support IP addresses.
	Target LoadBalancerPoolMemberTargetPrototypeIntf `validate:"required"`

	// Weight of the server member. Applicable only if the pool algorithm is
	// `weighted_round_robin`.
	Weight *int64

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateLoadBalancerPoolMemberOptions : Instantiate CreateLoadBalancerPoolMemberOptions
func (*VpcV1) NewCreateLoadBalancerPoolMemberOptions(loadBalancerID string, poolID string, port int64, target LoadBalancerPoolMemberTargetPrototypeIntf) *CreateLoadBalancerPoolMemberOptions {
	return &CreateLoadBalancerPoolMemberOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		PoolID:         core.StringPtr(poolID),
		Port:           core.Int64Ptr(port),
		Target:         target,
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *CreateLoadBalancerPoolMemberOptions) SetLoadBalancerID(loadBalancerID string) *CreateLoadBalancerPoolMemberOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetPoolID : Allow user to set PoolID
func (options *CreateLoadBalancerPoolMemberOptions) SetPoolID(poolID string) *CreateLoadBalancerPoolMemberOptions {
	options.PoolID = core.StringPtr(poolID)
	return options
}

// SetPort : Allow user to set Port
func (options *CreateLoadBalancerPoolMemberOptions) SetPort(port int64) *CreateLoadBalancerPoolMemberOptions {
	options.Port = core.Int64Ptr(port)
	return options
}

// SetTarget : Allow user to set Target
func (options *CreateLoadBalancerPoolMemberOptions) SetTarget(target LoadBalancerPoolMemberTargetPrototypeIntf) *CreateLoadBalancerPoolMemberOptions {
	options.Target = target
	return options
}

// SetWeight : Allow user to set Weight
func (options *CreateLoadBalancerPoolMemberOptions) SetWeight(weight int64) *CreateLoadBalancerPoolMemberOptions {
	options.Weight = core.Int64Ptr(weight)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateLoadBalancerPoolMemberOptions) SetHeaders(param map[string]string) *CreateLoadBalancerPoolMemberOptions {
	options.Headers = param
	return options
}

// CreateLoadBalancerPoolOptions : The CreateLoadBalancerPool options.
type CreateLoadBalancerPoolOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The load balancing algorithm.
	Algorithm *string `validate:"required"`

	// The health monitor of this pool.
	HealthMonitor *LoadBalancerPoolHealthMonitorPrototype `validate:"required"`

	// The protocol used for this load balancer pool. Load balancers in the `network` family support `tcp`. Load balancers
	// in the `application` family support `tcp`, `http`, and
	// `https`.
	Protocol *string `validate:"required"`

	// The members for this load balancer pool. For load balancers in the `network` family, the same `port` and `target`
	// tuple cannot be shared by a pool member of any other load balancer in the same VPC.
	Members []LoadBalancerPoolMemberPrototype

	// The user-defined name for this load balancer pool. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string

	// The PROXY protocol setting for this pool:
	// - `v1`: Enabled with version 1 (human-readable header format)
	// - `v2`: Enabled with version 2 (binary header format)
	// - `disabled`: Disabled
	//
	// Supported by load balancers in the `application` family (otherwise always `disabled`).
	ProxyProtocol *string

	// The session persistence of this pool.
	SessionPersistence *LoadBalancerPoolSessionPersistencePrototype

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateLoadBalancerPoolOptions.Algorithm property.
// The load balancing algorithm.
const (
	CreateLoadBalancerPoolOptionsAlgorithmLeastConnectionsConst   = "least_connections"
	CreateLoadBalancerPoolOptionsAlgorithmRoundRobinConst         = "round_robin"
	CreateLoadBalancerPoolOptionsAlgorithmWeightedRoundRobinConst = "weighted_round_robin"
)

// Constants associated with the CreateLoadBalancerPoolOptions.Protocol property.
// The protocol used for this load balancer pool. Load balancers in the `network` family support `tcp`. Load balancers
// in the `application` family support `tcp`, `http`, and
// `https`.
const (
	CreateLoadBalancerPoolOptionsProtocolHTTPConst  = "http"
	CreateLoadBalancerPoolOptionsProtocolHTTPSConst = "https"
	CreateLoadBalancerPoolOptionsProtocolTCPConst   = "tcp"
)

// Constants associated with the CreateLoadBalancerPoolOptions.ProxyProtocol property.
// The PROXY protocol setting for this pool:
// - `v1`: Enabled with version 1 (human-readable header format)
// - `v2`: Enabled with version 2 (binary header format)
// - `disabled`: Disabled
//
// Supported by load balancers in the `application` family (otherwise always `disabled`).
const (
	CreateLoadBalancerPoolOptionsProxyProtocolDisabledConst = "disabled"
	CreateLoadBalancerPoolOptionsProxyProtocolV1Const       = "v1"
	CreateLoadBalancerPoolOptionsProxyProtocolV2Const       = "v2"
)

// NewCreateLoadBalancerPoolOptions : Instantiate CreateLoadBalancerPoolOptions
func (*VpcV1) NewCreateLoadBalancerPoolOptions(loadBalancerID string, algorithm string, healthMonitor *LoadBalancerPoolHealthMonitorPrototype, protocol string) *CreateLoadBalancerPoolOptions {
	return &CreateLoadBalancerPoolOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		Algorithm:      core.StringPtr(algorithm),
		HealthMonitor:  healthMonitor,
		Protocol:       core.StringPtr(protocol),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *CreateLoadBalancerPoolOptions) SetLoadBalancerID(loadBalancerID string) *CreateLoadBalancerPoolOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetAlgorithm : Allow user to set Algorithm
func (options *CreateLoadBalancerPoolOptions) SetAlgorithm(algorithm string) *CreateLoadBalancerPoolOptions {
	options.Algorithm = core.StringPtr(algorithm)
	return options
}

// SetHealthMonitor : Allow user to set HealthMonitor
func (options *CreateLoadBalancerPoolOptions) SetHealthMonitor(healthMonitor *LoadBalancerPoolHealthMonitorPrototype) *CreateLoadBalancerPoolOptions {
	options.HealthMonitor = healthMonitor
	return options
}

// SetProtocol : Allow user to set Protocol
func (options *CreateLoadBalancerPoolOptions) SetProtocol(protocol string) *CreateLoadBalancerPoolOptions {
	options.Protocol = core.StringPtr(protocol)
	return options
}

// SetMembers : Allow user to set Members
func (options *CreateLoadBalancerPoolOptions) SetMembers(members []LoadBalancerPoolMemberPrototype) *CreateLoadBalancerPoolOptions {
	options.Members = members
	return options
}

// SetName : Allow user to set Name
func (options *CreateLoadBalancerPoolOptions) SetName(name string) *CreateLoadBalancerPoolOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetProxyProtocol : Allow user to set ProxyProtocol
func (options *CreateLoadBalancerPoolOptions) SetProxyProtocol(proxyProtocol string) *CreateLoadBalancerPoolOptions {
	options.ProxyProtocol = core.StringPtr(proxyProtocol)
	return options
}

// SetSessionPersistence : Allow user to set SessionPersistence
func (options *CreateLoadBalancerPoolOptions) SetSessionPersistence(sessionPersistence *LoadBalancerPoolSessionPersistencePrototype) *CreateLoadBalancerPoolOptions {
	options.SessionPersistence = sessionPersistence
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateLoadBalancerPoolOptions) SetHeaders(param map[string]string) *CreateLoadBalancerPoolOptions {
	options.Headers = param
	return options
}

// CreateNetworkACLOptions : The CreateNetworkACL options.
type CreateNetworkACLOptions struct {
	// The network ACL prototype object.
	NetworkACLPrototype NetworkACLPrototypeIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateNetworkACLOptions : Instantiate CreateNetworkACLOptions
func (*VpcV1) NewCreateNetworkACLOptions() *CreateNetworkACLOptions {
	return &CreateNetworkACLOptions{}
}

// SetNetworkACLPrototype : Allow user to set NetworkACLPrototype
func (options *CreateNetworkACLOptions) SetNetworkACLPrototype(networkACLPrototype NetworkACLPrototypeIntf) *CreateNetworkACLOptions {
	options.NetworkACLPrototype = networkACLPrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateNetworkACLOptions) SetHeaders(param map[string]string) *CreateNetworkACLOptions {
	options.Headers = param
	return options
}

// CreateNetworkACLRuleOptions : The CreateNetworkACLRule options.
type CreateNetworkACLRuleOptions struct {
	// The network ACL identifier.
	NetworkACLID *string `validate:"required,ne="`

	// The network ACL rule prototype object.
	NetworkACLRulePrototype NetworkACLRulePrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateNetworkACLRuleOptions : Instantiate CreateNetworkACLRuleOptions
func (*VpcV1) NewCreateNetworkACLRuleOptions(networkACLID string, networkACLRulePrototype NetworkACLRulePrototypeIntf) *CreateNetworkACLRuleOptions {
	return &CreateNetworkACLRuleOptions{
		NetworkACLID:            core.StringPtr(networkACLID),
		NetworkACLRulePrototype: networkACLRulePrototype,
	}
}

// SetNetworkACLID : Allow user to set NetworkACLID
func (options *CreateNetworkACLRuleOptions) SetNetworkACLID(networkACLID string) *CreateNetworkACLRuleOptions {
	options.NetworkACLID = core.StringPtr(networkACLID)
	return options
}

// SetNetworkACLRulePrototype : Allow user to set NetworkACLRulePrototype
func (options *CreateNetworkACLRuleOptions) SetNetworkACLRulePrototype(networkACLRulePrototype NetworkACLRulePrototypeIntf) *CreateNetworkACLRuleOptions {
	options.NetworkACLRulePrototype = networkACLRulePrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateNetworkACLRuleOptions) SetHeaders(param map[string]string) *CreateNetworkACLRuleOptions {
	options.Headers = param
	return options
}

// CreatePublicGatewayOptions : The CreatePublicGateway options.
type CreatePublicGatewayOptions struct {
	// The VPC this public gateway will serve.
	VPC VPCIdentityIntf `validate:"required"`

	// The zone this public gateway will reside in.
	Zone ZoneIdentityIntf `validate:"required"`

	FloatingIP PublicGatewayFloatingIPPrototypeIntf

	// The user-defined name for this public gateway. Names must be unique within the VPC the public gateway resides in. If
	// unspecified, the name will be a hyphenated list of randomly-selected words.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreatePublicGatewayOptions : Instantiate CreatePublicGatewayOptions
func (*VpcV1) NewCreatePublicGatewayOptions(vpc VPCIdentityIntf, zone ZoneIdentityIntf) *CreatePublicGatewayOptions {
	return &CreatePublicGatewayOptions{
		VPC:  vpc,
		Zone: zone,
	}
}

// SetVPC : Allow user to set VPC
func (options *CreatePublicGatewayOptions) SetVPC(vpc VPCIdentityIntf) *CreatePublicGatewayOptions {
	options.VPC = vpc
	return options
}

// SetZone : Allow user to set Zone
func (options *CreatePublicGatewayOptions) SetZone(zone ZoneIdentityIntf) *CreatePublicGatewayOptions {
	options.Zone = zone
	return options
}

// SetFloatingIP : Allow user to set FloatingIP
func (options *CreatePublicGatewayOptions) SetFloatingIP(floatingIP PublicGatewayFloatingIPPrototypeIntf) *CreatePublicGatewayOptions {
	options.FloatingIP = floatingIP
	return options
}

// SetName : Allow user to set Name
func (options *CreatePublicGatewayOptions) SetName(name string) *CreatePublicGatewayOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreatePublicGatewayOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreatePublicGatewayOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreatePublicGatewayOptions) SetHeaders(param map[string]string) *CreatePublicGatewayOptions {
	options.Headers = param
	return options
}

// CreateSecurityGroupOptions : The CreateSecurityGroup options.
type CreateSecurityGroupOptions struct {
	// The VPC this security group is to be a part of.
	VPC VPCIdentityIntf `validate:"required"`

	// The user-defined name for this security group. If unspecified, the name will be a hyphenated list of
	// randomly-selected words. Names must be unique within the VPC the security group resides in.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// The prototype objects for rules to be created for this security group. If unspecified, no rules will be created,
	// resulting in all traffic being denied.
	Rules []SecurityGroupRulePrototypeIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateSecurityGroupOptions : Instantiate CreateSecurityGroupOptions
func (*VpcV1) NewCreateSecurityGroupOptions(vpc VPCIdentityIntf) *CreateSecurityGroupOptions {
	return &CreateSecurityGroupOptions{
		VPC: vpc,
	}
}

// SetVPC : Allow user to set VPC
func (options *CreateSecurityGroupOptions) SetVPC(vpc VPCIdentityIntf) *CreateSecurityGroupOptions {
	options.VPC = vpc
	return options
}

// SetName : Allow user to set Name
func (options *CreateSecurityGroupOptions) SetName(name string) *CreateSecurityGroupOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateSecurityGroupOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateSecurityGroupOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetRules : Allow user to set Rules
func (options *CreateSecurityGroupOptions) SetRules(rules []SecurityGroupRulePrototypeIntf) *CreateSecurityGroupOptions {
	options.Rules = rules
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateSecurityGroupOptions) SetHeaders(param map[string]string) *CreateSecurityGroupOptions {
	options.Headers = param
	return options
}

// CreateSecurityGroupRuleOptions : The CreateSecurityGroupRule options.
type CreateSecurityGroupRuleOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// The properties of the security group rule to be created.
	SecurityGroupRulePrototype SecurityGroupRulePrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateSecurityGroupRuleOptions : Instantiate CreateSecurityGroupRuleOptions
func (*VpcV1) NewCreateSecurityGroupRuleOptions(securityGroupID string, securityGroupRulePrototype SecurityGroupRulePrototypeIntf) *CreateSecurityGroupRuleOptions {
	return &CreateSecurityGroupRuleOptions{
		SecurityGroupID:            core.StringPtr(securityGroupID),
		SecurityGroupRulePrototype: securityGroupRulePrototype,
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *CreateSecurityGroupRuleOptions) SetSecurityGroupID(securityGroupID string) *CreateSecurityGroupRuleOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetSecurityGroupRulePrototype : Allow user to set SecurityGroupRulePrototype
func (options *CreateSecurityGroupRuleOptions) SetSecurityGroupRulePrototype(securityGroupRulePrototype SecurityGroupRulePrototypeIntf) *CreateSecurityGroupRuleOptions {
	options.SecurityGroupRulePrototype = securityGroupRulePrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateSecurityGroupRuleOptions) SetHeaders(param map[string]string) *CreateSecurityGroupRuleOptions {
	options.Headers = param
	return options
}

// CreateSecurityGroupTargetBindingOptions : The CreateSecurityGroupTargetBinding options.
type CreateSecurityGroupTargetBindingOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// The security group target identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateSecurityGroupTargetBindingOptions : Instantiate CreateSecurityGroupTargetBindingOptions
func (*VpcV1) NewCreateSecurityGroupTargetBindingOptions(securityGroupID string, id string) *CreateSecurityGroupTargetBindingOptions {
	return &CreateSecurityGroupTargetBindingOptions{
		SecurityGroupID: core.StringPtr(securityGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *CreateSecurityGroupTargetBindingOptions) SetSecurityGroupID(securityGroupID string) *CreateSecurityGroupTargetBindingOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *CreateSecurityGroupTargetBindingOptions) SetID(id string) *CreateSecurityGroupTargetBindingOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateSecurityGroupTargetBindingOptions) SetHeaders(param map[string]string) *CreateSecurityGroupTargetBindingOptions {
	options.Headers = param
	return options
}

// CreateSnapshotOptions : The CreateSnapshot options.
type CreateSnapshotOptions struct {
	// The volume to snapshot.
	SourceVolume VolumeIdentityIntf `validate:"required"`

	// The unique user-defined name for this snapshot. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateSnapshotOptions : Instantiate CreateSnapshotOptions
func (*VpcV1) NewCreateSnapshotOptions(sourceVolume VolumeIdentityIntf) *CreateSnapshotOptions {
	return &CreateSnapshotOptions{
		SourceVolume: sourceVolume,
	}
}

// SetSourceVolume : Allow user to set SourceVolume
func (options *CreateSnapshotOptions) SetSourceVolume(sourceVolume VolumeIdentityIntf) *CreateSnapshotOptions {
	options.SourceVolume = sourceVolume
	return options
}

// SetName : Allow user to set Name
func (options *CreateSnapshotOptions) SetName(name string) *CreateSnapshotOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateSnapshotOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateSnapshotOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateSnapshotOptions) SetHeaders(param map[string]string) *CreateSnapshotOptions {
	options.Headers = param
	return options
}

// CreateSubnetOptions : The CreateSubnet options.
type CreateSubnetOptions struct {
	// The subnet prototype object.
	SubnetPrototype SubnetPrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateSubnetOptions : Instantiate CreateSubnetOptions
func (*VpcV1) NewCreateSubnetOptions(subnetPrototype SubnetPrototypeIntf) *CreateSubnetOptions {
	return &CreateSubnetOptions{
		SubnetPrototype: subnetPrototype,
	}
}

// SetSubnetPrototype : Allow user to set SubnetPrototype
func (options *CreateSubnetOptions) SetSubnetPrototype(subnetPrototype SubnetPrototypeIntf) *CreateSubnetOptions {
	options.SubnetPrototype = subnetPrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateSubnetOptions) SetHeaders(param map[string]string) *CreateSubnetOptions {
	options.Headers = param
	return options
}

// CreateSubnetReservedIPOptions : The CreateSubnetReservedIP options.
type CreateSubnetReservedIPOptions struct {
	// The subnet identifier.
	SubnetID *string `validate:"required,ne="`

	// If set to `true`, this reserved IP will be automatically deleted when the target is deleted or when the reserved IP
	// is unbound. The value cannot be set to `true` if the reserved IP is unbound.
	AutoDelete *bool

	// The user-defined name for this reserved IP. If not specified, the name will be a hyphenated list of
	// randomly-selected words. Names must be unique within the subnet the reserved IP resides in. Names beginning with
	// `ibm-` are reserved for provider-owned resources.
	Name *string

	// The target this reserved IP is to be bound to.
	Target ReservedIPTargetPrototypeIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateSubnetReservedIPOptions : Instantiate CreateSubnetReservedIPOptions
func (*VpcV1) NewCreateSubnetReservedIPOptions(subnetID string) *CreateSubnetReservedIPOptions {
	return &CreateSubnetReservedIPOptions{
		SubnetID: core.StringPtr(subnetID),
	}
}

// SetSubnetID : Allow user to set SubnetID
func (options *CreateSubnetReservedIPOptions) SetSubnetID(subnetID string) *CreateSubnetReservedIPOptions {
	options.SubnetID = core.StringPtr(subnetID)
	return options
}

// SetAutoDelete : Allow user to set AutoDelete
func (options *CreateSubnetReservedIPOptions) SetAutoDelete(autoDelete bool) *CreateSubnetReservedIPOptions {
	options.AutoDelete = core.BoolPtr(autoDelete)
	return options
}

// SetName : Allow user to set Name
func (options *CreateSubnetReservedIPOptions) SetName(name string) *CreateSubnetReservedIPOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetTarget : Allow user to set Target
func (options *CreateSubnetReservedIPOptions) SetTarget(target ReservedIPTargetPrototypeIntf) *CreateSubnetReservedIPOptions {
	options.Target = target
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateSubnetReservedIPOptions) SetHeaders(param map[string]string) *CreateSubnetReservedIPOptions {
	options.Headers = param
	return options
}

// CreateVolumeOptions : The CreateVolume options.
type CreateVolumeOptions struct {
	// The volume prototype object.
	VolumePrototype VolumePrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateVolumeOptions : Instantiate CreateVolumeOptions
func (*VpcV1) NewCreateVolumeOptions(volumePrototype VolumePrototypeIntf) *CreateVolumeOptions {
	return &CreateVolumeOptions{
		VolumePrototype: volumePrototype,
	}
}

// SetVolumePrototype : Allow user to set VolumePrototype
func (options *CreateVolumeOptions) SetVolumePrototype(volumePrototype VolumePrototypeIntf) *CreateVolumeOptions {
	options.VolumePrototype = volumePrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateVolumeOptions) SetHeaders(param map[string]string) *CreateVolumeOptions {
	options.Headers = param
	return options
}

// CreateVPCAddressPrefixOptions : The CreateVPCAddressPrefix options.
type CreateVPCAddressPrefixOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The IPv4 range of the address prefix, expressed in CIDR format. The request must not overlap with any existing
	// address prefixes in the VPC or any of the following reserved address ranges:
	//   - `127.0.0.0/8` (IPv4 loopback addresses)
	//   - `161.26.0.0/16` (IBM services)
	//   - `166.8.0.0/14` (Cloud Service Endpoints)
	//   - `169.254.0.0/16` (IPv4 link-local addresses)
	//   - `224.0.0.0/4` (IPv4 multicast addresses)
	//
	// The prefix length of the address prefix's CIDR must be between `/9` (8,388,608 addresses) and `/29` (8 addresses).
	CIDR *string `validate:"required"`

	// The zone this address prefix will reside in.
	Zone ZoneIdentityIntf `validate:"required"`

	// Indicates whether this is the default prefix for this zone in this VPC. If true, this prefix will become the default
	// prefix for this zone in this VPC. This fails if the VPC currently has a default address prefix for this zone.
	IsDefault *bool

	// The user-defined name for this address prefix. Names must be unique within the VPC the address prefix resides in. If
	// unspecified, the name will be a hyphenated list of randomly-selected words.
	Name *string

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateVPCAddressPrefixOptions : Instantiate CreateVPCAddressPrefixOptions
func (*VpcV1) NewCreateVPCAddressPrefixOptions(vpcID string, cidr string, zone ZoneIdentityIntf) *CreateVPCAddressPrefixOptions {
	return &CreateVPCAddressPrefixOptions{
		VPCID: core.StringPtr(vpcID),
		CIDR:  core.StringPtr(cidr),
		Zone:  zone,
	}
}

// SetVPCID : Allow user to set VPCID
func (options *CreateVPCAddressPrefixOptions) SetVPCID(vpcID string) *CreateVPCAddressPrefixOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetCIDR : Allow user to set CIDR
func (options *CreateVPCAddressPrefixOptions) SetCIDR(cidr string) *CreateVPCAddressPrefixOptions {
	options.CIDR = core.StringPtr(cidr)
	return options
}

// SetZone : Allow user to set Zone
func (options *CreateVPCAddressPrefixOptions) SetZone(zone ZoneIdentityIntf) *CreateVPCAddressPrefixOptions {
	options.Zone = zone
	return options
}

// SetIsDefault : Allow user to set IsDefault
func (options *CreateVPCAddressPrefixOptions) SetIsDefault(isDefault bool) *CreateVPCAddressPrefixOptions {
	options.IsDefault = core.BoolPtr(isDefault)
	return options
}

// SetName : Allow user to set Name
func (options *CreateVPCAddressPrefixOptions) SetName(name string) *CreateVPCAddressPrefixOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateVPCAddressPrefixOptions) SetHeaders(param map[string]string) *CreateVPCAddressPrefixOptions {
	options.Headers = param
	return options
}

// CreateVPCOptions : The CreateVPC options.
type CreateVPCOptions struct {
	// Indicates whether a default address prefix should be automatically created for each zone in this VPC. If `manual`,
	// this VPC will be created with no default address prefixes.
	AddressPrefixManagement *string

	// Indicates whether this VPC should be connected to Classic Infrastructure. If true, this VPC's resources will have
	// private network connectivity to the account's Classic Infrastructure resources. Only one VPC, per region, may be
	// connected in this way. This value is set at creation and subsequently immutable.
	ClassicAccess *bool

	// The unique user-defined name for this VPC. If unspecified, the name will be a hyphenated list of randomly-selected
	// words.
	Name *string

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateVPCOptions.AddressPrefixManagement property.
// Indicates whether a default address prefix should be automatically created for each zone in this VPC. If `manual`,
// this VPC will be created with no default address prefixes.
const (
	CreateVPCOptionsAddressPrefixManagementAutoConst   = "auto"
	CreateVPCOptionsAddressPrefixManagementManualConst = "manual"
)

// NewCreateVPCOptions : Instantiate CreateVPCOptions
func (*VpcV1) NewCreateVPCOptions() *CreateVPCOptions {
	return &CreateVPCOptions{}
}

// SetAddressPrefixManagement : Allow user to set AddressPrefixManagement
func (options *CreateVPCOptions) SetAddressPrefixManagement(addressPrefixManagement string) *CreateVPCOptions {
	options.AddressPrefixManagement = core.StringPtr(addressPrefixManagement)
	return options
}

// SetClassicAccess : Allow user to set ClassicAccess
func (options *CreateVPCOptions) SetClassicAccess(classicAccess bool) *CreateVPCOptions {
	options.ClassicAccess = core.BoolPtr(classicAccess)
	return options
}

// SetName : Allow user to set Name
func (options *CreateVPCOptions) SetName(name string) *CreateVPCOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateVPCOptions) SetResourceGroup(resourceGroup ResourceGroupIdentityIntf) *CreateVPCOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateVPCOptions) SetHeaders(param map[string]string) *CreateVPCOptions {
	options.Headers = param
	return options
}

// CreateVPCRouteOptions : The CreateVPCRoute options.
type CreateVPCRouteOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The destination of the route. At most two routes per `zone` in a table can have the same destination, and only if
	// both routes have an `action` of `deliver` and the
	// `next_hop` is an IP address.
	Destination *string `validate:"required"`

	// The zone to apply the route to. (Traffic from subnets in this zone will be
	// subject to this route.).
	Zone ZoneIdentityIntf `validate:"required"`

	// The action to perform with a packet matching the route:
	// - `delegate`: delegate to the system's built-in routes
	// - `delegate_vpc`: delegate to the system's built-in routes, ignoring Internet-bound
	//   routes
	// - `deliver`: deliver the packet to the specified `next_hop`
	// - `drop`: drop the packet.
	Action *string

	// The user-defined name for this route. If unspecified, the name will be a hyphenated list of randomly-selected words.
	// Names must be unique within the VPC routing table the route resides in.
	Name *string

	// If `action` is `deliver`, the next hop that packets will be delivered to.  For
	// other `action` values, it must be omitted or specified as `0.0.0.0`.
	NextHop RouteNextHopPrototypeIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateVPCRouteOptions.Action property.
// The action to perform with a packet matching the route:
// - `delegate`: delegate to the system's built-in routes
// - `delegate_vpc`: delegate to the system's built-in routes, ignoring Internet-bound
//   routes
// - `deliver`: deliver the packet to the specified `next_hop`
// - `drop`: drop the packet.
const (
	CreateVPCRouteOptionsActionDelegateConst    = "delegate"
	CreateVPCRouteOptionsActionDelegateVPCConst = "delegate_vpc"
	CreateVPCRouteOptionsActionDeliverConst     = "deliver"
	CreateVPCRouteOptionsActionDropConst        = "drop"
)

// NewCreateVPCRouteOptions : Instantiate CreateVPCRouteOptions
func (*VpcV1) NewCreateVPCRouteOptions(vpcID string, destination string, zone ZoneIdentityIntf) *CreateVPCRouteOptions {
	return &CreateVPCRouteOptions{
		VPCID:       core.StringPtr(vpcID),
		Destination: core.StringPtr(destination),
		Zone:        zone,
	}
}

// SetVPCID : Allow user to set VPCID
func (options *CreateVPCRouteOptions) SetVPCID(vpcID string) *CreateVPCRouteOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetDestination : Allow user to set Destination
func (options *CreateVPCRouteOptions) SetDestination(destination string) *CreateVPCRouteOptions {
	options.Destination = core.StringPtr(destination)
	return options
}

// SetZone : Allow user to set Zone
func (options *CreateVPCRouteOptions) SetZone(zone ZoneIdentityIntf) *CreateVPCRouteOptions {
	options.Zone = zone
	return options
}

// SetAction : Allow user to set Action
func (options *CreateVPCRouteOptions) SetAction(action string) *CreateVPCRouteOptions {
	options.Action = core.StringPtr(action)
	return options
}

// SetName : Allow user to set Name
func (options *CreateVPCRouteOptions) SetName(name string) *CreateVPCRouteOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetNextHop : Allow user to set NextHop
func (options *CreateVPCRouteOptions) SetNextHop(nextHop RouteNextHopPrototypeIntf) *CreateVPCRouteOptions {
	options.NextHop = nextHop
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateVPCRouteOptions) SetHeaders(param map[string]string) *CreateVPCRouteOptions {
	options.Headers = param
	return options
}

// CreateVPCRoutingTableOptions : The CreateVPCRoutingTable options.
type CreateVPCRoutingTableOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The user-defined name for this routing table. Names must be unique within the VPC the routing table resides in. If
	// unspecified, the name will be a hyphenated list of randomly-selected words.
	Name *string

	// If set to `true`, this routing table will be used to route traffic that originates from [Direct
	// Link](https://cloud.ibm.com/docs/dl/) to this VPC. For this to succeed, the VPC must not already have a routing
	// table with this property set to `true`.
	//
	// Incoming traffic will be routed according to the routing table with one exception: routes with an `action` of
	// `deliver` are treated as `drop` unless the `next_hop` is an IP address within the VPC's address prefix ranges.
	// Therefore, if an incoming packet matches a route with a `next_hop` of an internet-bound IP address or a VPN gateway
	// connection, the packet will be dropped.
	RouteDirectLinkIngress *bool

	// If set to `true`, this routing table will be used to route traffic that originates from [Transit
	// Gateway](https://cloud.ibm.com/cloud/transit-gateway/) to this VPC. For this to succeed, the VPC must not already
	// have a routing table with this property set to `true`.
	//
	// Incoming traffic will be routed according to the routing table with one exception: routes with an `action` of
	// `deliver` are treated as `drop` unless the `next_hop` is an IP address within the VPC's address prefix ranges.
	// Therefore, if an incoming packet matches a route with a `next_hop` of an internet-bound IP address or a VPN gateway
	// connection, the packet will be dropped.
	//
	// If [Classic Access](https://cloud.ibm.com/docs/vpc?topic=vpc-setting-up-access-to-classic-infrastructure) is enabled
	// for this VPC, and this property is set to `true`, its incoming traffic will also be routed according to this routing
	// table.
	RouteTransitGatewayIngress *bool

	// If set to `true`, this routing table will be used to route traffic that originates from subnets in other zones in
	// this VPC. For this to succeed, the VPC must not already have a routing table with this property set to `true`.
	//
	// Incoming traffic will be routed according to the routing table with one exception: routes with an `action` of
	// `deliver` are treated as `drop` unless the `next_hop` is an IP address within the VPC's address prefix ranges.
	// Therefore, if an incoming packet matches a route with a `next_hop` of an internet-bound IP address or a VPN gateway
	// connection, the packet will be dropped.
	RouteVPCZoneIngress *bool

	// The prototype objects for routes to create for this routing table. If unspecified, the routing table will be created
	// with no routes.
	Routes []RoutePrototype

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateVPCRoutingTableOptions : Instantiate CreateVPCRoutingTableOptions
func (*VpcV1) NewCreateVPCRoutingTableOptions(vpcID string) *CreateVPCRoutingTableOptions {
	return &CreateVPCRoutingTableOptions{
		VPCID: core.StringPtr(vpcID),
	}
}

// SetVPCID : Allow user to set VPCID
func (options *CreateVPCRoutingTableOptions) SetVPCID(vpcID string) *CreateVPCRoutingTableOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetName : Allow user to set Name
func (options *CreateVPCRoutingTableOptions) SetName(name string) *CreateVPCRoutingTableOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetRouteDirectLinkIngress : Allow user to set RouteDirectLinkIngress
func (options *CreateVPCRoutingTableOptions) SetRouteDirectLinkIngress(routeDirectLinkIngress bool) *CreateVPCRoutingTableOptions {
	options.RouteDirectLinkIngress = core.BoolPtr(routeDirectLinkIngress)
	return options
}

// SetRouteTransitGatewayIngress : Allow user to set RouteTransitGatewayIngress
func (options *CreateVPCRoutingTableOptions) SetRouteTransitGatewayIngress(routeTransitGatewayIngress bool) *CreateVPCRoutingTableOptions {
	options.RouteTransitGatewayIngress = core.BoolPtr(routeTransitGatewayIngress)
	return options
}

// SetRouteVPCZoneIngress : Allow user to set RouteVPCZoneIngress
func (options *CreateVPCRoutingTableOptions) SetRouteVPCZoneIngress(routeVPCZoneIngress bool) *CreateVPCRoutingTableOptions {
	options.RouteVPCZoneIngress = core.BoolPtr(routeVPCZoneIngress)
	return options
}

// SetRoutes : Allow user to set Routes
func (options *CreateVPCRoutingTableOptions) SetRoutes(routes []RoutePrototype) *CreateVPCRoutingTableOptions {
	options.Routes = routes
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateVPCRoutingTableOptions) SetHeaders(param map[string]string) *CreateVPCRoutingTableOptions {
	options.Headers = param
	return options
}

// CreateVPCRoutingTableRouteOptions : The CreateVPCRoutingTableRoute options.
type CreateVPCRoutingTableRouteOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The routing table identifier.
	RoutingTableID *string `validate:"required,ne="`

	// The destination of the route. At most two routes per `zone` in a table can have the same destination, and only if
	// both routes have an `action` of `deliver` and the
	// `next_hop` is an IP address.
	Destination *string `validate:"required"`

	// The zone to apply the route to. (Traffic from subnets in this zone will be
	// subject to this route.).
	Zone ZoneIdentityIntf `validate:"required"`

	// The action to perform with a packet matching the route:
	// - `delegate`: delegate to the system's built-in routes
	// - `delegate_vpc`: delegate to the system's built-in routes, ignoring Internet-bound
	//   routes
	// - `deliver`: deliver the packet to the specified `next_hop`
	// - `drop`: drop the packet.
	Action *string

	// The user-defined name for this route. If unspecified, the name will be a hyphenated list of randomly-selected words.
	// Names must be unique within the VPC routing table the route resides in.
	Name *string

	// If `action` is `deliver`, the next hop that packets will be delivered to.  For
	// other `action` values, it must be omitted or specified as `0.0.0.0`.
	NextHop RouteNextHopPrototypeIntf

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateVPCRoutingTableRouteOptions.Action property.
// The action to perform with a packet matching the route:
// - `delegate`: delegate to the system's built-in routes
// - `delegate_vpc`: delegate to the system's built-in routes, ignoring Internet-bound
//   routes
// - `deliver`: deliver the packet to the specified `next_hop`
// - `drop`: drop the packet.
const (
	CreateVPCRoutingTableRouteOptionsActionDelegateConst    = "delegate"
	CreateVPCRoutingTableRouteOptionsActionDelegateVPCConst = "delegate_vpc"
	CreateVPCRoutingTableRouteOptionsActionDeliverConst     = "deliver"
	CreateVPCRoutingTableRouteOptionsActionDropConst        = "drop"
)

// NewCreateVPCRoutingTableRouteOptions : Instantiate CreateVPCRoutingTableRouteOptions
func (*VpcV1) NewCreateVPCRoutingTableRouteOptions(vpcID string, routingTableID string, destination string, zone ZoneIdentityIntf) *CreateVPCRoutingTableRouteOptions {
	return &CreateVPCRoutingTableRouteOptions{
		VPCID:          core.StringPtr(vpcID),
		RoutingTableID: core.StringPtr(routingTableID),
		Destination:    core.StringPtr(destination),
		Zone:           zone,
	}
}

// SetVPCID : Allow user to set VPCID
func (options *CreateVPCRoutingTableRouteOptions) SetVPCID(vpcID string) *CreateVPCRoutingTableRouteOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetRoutingTableID : Allow user to set RoutingTableID
func (options *CreateVPCRoutingTableRouteOptions) SetRoutingTableID(routingTableID string) *CreateVPCRoutingTableRouteOptions {
	options.RoutingTableID = core.StringPtr(routingTableID)
	return options
}

// SetDestination : Allow user to set Destination
func (options *CreateVPCRoutingTableRouteOptions) SetDestination(destination string) *CreateVPCRoutingTableRouteOptions {
	options.Destination = core.StringPtr(destination)
	return options
}

// SetZone : Allow user to set Zone
func (options *CreateVPCRoutingTableRouteOptions) SetZone(zone ZoneIdentityIntf) *CreateVPCRoutingTableRouteOptions {
	options.Zone = zone
	return options
}

// SetAction : Allow user to set Action
func (options *CreateVPCRoutingTableRouteOptions) SetAction(action string) *CreateVPCRoutingTableRouteOptions {
	options.Action = core.StringPtr(action)
	return options
}

// SetName : Allow user to set Name
func (options *CreateVPCRoutingTableRouteOptions) SetName(name string) *CreateVPCRoutingTableRouteOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetNextHop : Allow user to set NextHop
func (options *CreateVPCRoutingTableRouteOptions) SetNextHop(nextHop RouteNextHopPrototypeIntf) *CreateVPCRoutingTableRouteOptions {
	options.NextHop = nextHop
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateVPCRoutingTableRouteOptions) SetHeaders(param map[string]string) *CreateVPCRoutingTableRouteOptions {
	options.Headers = param
	return options
}

// CreateVPNGatewayConnectionOptions : The CreateVPNGatewayConnection options.
type CreateVPNGatewayConnectionOptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection prototype object.
	VPNGatewayConnectionPrototype VPNGatewayConnectionPrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateVPNGatewayConnectionOptions : Instantiate CreateVPNGatewayConnectionOptions
func (*VpcV1) NewCreateVPNGatewayConnectionOptions(vpnGatewayID string, vpnGatewayConnectionPrototype VPNGatewayConnectionPrototypeIntf) *CreateVPNGatewayConnectionOptions {
	return &CreateVPNGatewayConnectionOptions{
		VPNGatewayID:                  core.StringPtr(vpnGatewayID),
		VPNGatewayConnectionPrototype: vpnGatewayConnectionPrototype,
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *CreateVPNGatewayConnectionOptions) SetVPNGatewayID(vpnGatewayID string) *CreateVPNGatewayConnectionOptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetVPNGatewayConnectionPrototype : Allow user to set VPNGatewayConnectionPrototype
func (options *CreateVPNGatewayConnectionOptions) SetVPNGatewayConnectionPrototype(vpnGatewayConnectionPrototype VPNGatewayConnectionPrototypeIntf) *CreateVPNGatewayConnectionOptions {
	options.VPNGatewayConnectionPrototype = vpnGatewayConnectionPrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateVPNGatewayConnectionOptions) SetHeaders(param map[string]string) *CreateVPNGatewayConnectionOptions {
	options.Headers = param
	return options
}

// CreateVPNGatewayOptions : The CreateVPNGateway options.
type CreateVPNGatewayOptions struct {
	// The VPN gateway prototype object.
	VPNGatewayPrototype VPNGatewayPrototypeIntf `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateVPNGatewayOptions : Instantiate CreateVPNGatewayOptions
func (*VpcV1) NewCreateVPNGatewayOptions(vpnGatewayPrototype VPNGatewayPrototypeIntf) *CreateVPNGatewayOptions {
	return &CreateVPNGatewayOptions{
		VPNGatewayPrototype: vpnGatewayPrototype,
	}
}

// SetVPNGatewayPrototype : Allow user to set VPNGatewayPrototype
func (options *CreateVPNGatewayOptions) SetVPNGatewayPrototype(vpnGatewayPrototype VPNGatewayPrototypeIntf) *CreateVPNGatewayOptions {
	options.VPNGatewayPrototype = vpnGatewayPrototype
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateVPNGatewayOptions) SetHeaders(param map[string]string) *CreateVPNGatewayOptions {
	options.Headers = param
	return options
}

// DedicatedHost : DedicatedHost struct
type DedicatedHost struct {
	// The amount of memory in gibibytes that is currently available for instances.
	AvailableMemory *int64 `json:"available_memory" validate:"required"`

	// The available VCPU for the dedicated host.
	AvailableVcpu *Vcpu `json:"available_vcpu" validate:"required"`

	// The date and time that the dedicated host was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN for this dedicated host.
	CRN *string `json:"crn" validate:"required"`

	// Collection of the dedicated host's disks.
	Disks []DedicatedHostDisk `json:"disks" validate:"required"`

	// The dedicated host group this dedicated host is in.
	Group *DedicatedHostGroupReference `json:"group" validate:"required"`

	// The URL for this dedicated host.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this dedicated host.
	ID *string `json:"id" validate:"required"`

	// If set to true, instances can be placed on this dedicated host.
	InstancePlacementEnabled *bool `json:"instance_placement_enabled" validate:"required"`

	// The instances that are allocated to this dedicated host.
	Instances []InstanceReference `json:"instances" validate:"required"`

	// The lifecycle state of the dedicated host resource.
	LifecycleState *string `json:"lifecycle_state" validate:"required"`

	// The total amount of memory in gibibytes for this host.
	Memory *int64 `json:"memory" validate:"required"`

	// The unique user-defined name for this dedicated host. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string `json:"name" validate:"required"`

	// The profile this dedicated host uses.
	Profile *DedicatedHostProfileReference `json:"profile" validate:"required"`

	// Indicates whether this dedicated host is available for instance creation.
	Provisionable *bool `json:"provisionable" validate:"required"`

	// The resource group for this dedicated host.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The type of resource referenced.
	ResourceType *string `json:"resource_type" validate:"required"`

	// The total number of sockets for this host.
	SocketCount *int64 `json:"socket_count" validate:"required"`

	// The administrative state of the dedicated host.
	//
	// The enumerated values for this property are expected to expand in the future. When processing this property, check
	// for and log unknown values. Optionally halt processing and surface the error, or bypass the dedicated host on which
	// the unexpected property value was encountered.
	State *string `json:"state" validate:"required"`

	// The instance profiles usable by instances placed on this dedicated host.
	SupportedInstanceProfiles []InstanceProfileReference `json:"supported_instance_profiles" validate:"required"`

	// The total VCPU of the dedicated host.
	Vcpu *Vcpu `json:"vcpu" validate:"required"`

	// The zone this dedicated host resides in.
	Zone *ZoneReference `json:"zone" validate:"required"`
}

// Constants associated with the DedicatedHost.LifecycleState property.
// The lifecycle state of the dedicated host resource.
const (
	DedicatedHostLifecycleStateDeletingConst  = "deleting"
	DedicatedHostLifecycleStateFailedConst    = "failed"
	DedicatedHostLifecycleStatePendingConst   = "pending"
	DedicatedHostLifecycleStateStableConst    = "stable"
	DedicatedHostLifecycleStateSuspendedConst = "suspended"
	DedicatedHostLifecycleStateUpdatingConst  = "updating"
	DedicatedHostLifecycleStateWaitingConst   = "waiting"
)

// Constants associated with the DedicatedHost.ResourceType property.
// The type of resource referenced.
const (
	DedicatedHostResourceTypeDedicatedHostConst = "dedicated_host"
)

// Constants associated with the DedicatedHost.State property.
// The administrative state of the dedicated host.
//
// The enumerated values for this property are expected to expand in the future. When processing this property, check
// for and log unknown values. Optionally halt processing and surface the error, or bypass the dedicated host on which
// the unexpected property value was encountered.
const (
	DedicatedHostStateAvailableConst   = "available"
	DedicatedHostStateDegradedConst    = "degraded"
	DedicatedHostStateMigratingConst   = "migrating"
	DedicatedHostStateUnavailableConst = "unavailable"
)

// UnmarshalDedicatedHost unmarshals an instance of DedicatedHost from the specified map of raw messages.
func UnmarshalDedicatedHost(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHost)
	err = core.UnmarshalPrimitive(m, "available_memory", &obj.AvailableMemory)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "available_vcpu", &obj.AvailableVcpu, UnmarshalVcpu)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "disks", &obj.Disks, UnmarshalDedicatedHostDisk)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "group", &obj.Group, UnmarshalDedicatedHostGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "instance_placement_enabled", &obj.InstancePlacementEnabled)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "instances", &obj.Instances, UnmarshalInstanceReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "lifecycle_state", &obj.LifecycleState)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "memory", &obj.Memory)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "profile", &obj.Profile, UnmarshalDedicatedHostProfileReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "provisionable", &obj.Provisionable)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "socket_count", &obj.SocketCount)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "state", &obj.State)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "supported_instance_profiles", &obj.SupportedInstanceProfiles, UnmarshalInstanceProfileReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vcpu", &obj.Vcpu, UnmarshalVcpu)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostCollection : DedicatedHostCollection struct
type DedicatedHostCollection struct {
	// Collection of dedicated hosts.
	DedicatedHosts []DedicatedHost `json:"dedicated_hosts" validate:"required"`

	// A link to the first page of resources.
	First *DedicatedHostCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *DedicatedHostCollectionNext `json:"next,omitempty"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`
}

// UnmarshalDedicatedHostCollection unmarshals an instance of DedicatedHostCollection from the specified map of raw messages.
func UnmarshalDedicatedHostCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostCollection)
	err = core.UnmarshalModel(m, "dedicated_hosts", &obj.DedicatedHosts, UnmarshalDedicatedHost)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalDedicatedHostCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalDedicatedHostCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostCollectionFirst : A link to the first page of resources.
type DedicatedHostCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalDedicatedHostCollectionFirst unmarshals an instance of DedicatedHostCollectionFirst from the specified map of raw messages.
func UnmarshalDedicatedHostCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type DedicatedHostCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalDedicatedHostCollectionNext unmarshals an instance of DedicatedHostCollectionNext from the specified map of raw messages.
func UnmarshalDedicatedHostCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostDisk : DedicatedHostDisk struct
type DedicatedHostDisk struct {
	// The remaining space left for instance placement in GB (gigabytes).
	Available *int64 `json:"available" validate:"required"`

	// The date and time that the disk was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The URL for this disk.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this disk.
	ID *string `json:"id" validate:"required"`

	// Instance disks that are on this dedicated host disk.
	InstanceDisks []InstanceDiskReference `json:"instance_disks" validate:"required"`

	// The disk interface used for attaching the disk
	//
	// The enumerated values for this property are expected to expand in the future. When processing this property, check
	// for and log unknown values. Optionally halt processing and surface the error, or bypass the resource on which the
	// unexpected property value was encountered.
	InterfaceType *string `json:"interface_type" validate:"required"`

	// The lifecycle state of this dedicated host disk.
	LifecycleState *string `json:"lifecycle_state,omitempty"`

	// The user-defined or system-provided name for this disk.
	Name *string `json:"name" validate:"required"`

	// Indicates whether this dedicated host disk is available for instance disk creation.
	Provisionable *bool `json:"provisionable" validate:"required"`

	// The type of resource referenced.
	ResourceType *string `json:"resource_type" validate:"required"`

	// The size of the disk in GB (gigabytes).
	Size *int64 `json:"size" validate:"required"`

	// The instance disk interfaces supported for this dedicated host disk.
	SupportedInstanceInterfaceTypes []string `json:"supported_instance_interface_types" validate:"required"`
}

// Constants associated with the DedicatedHostDisk.InterfaceType property.
// The disk interface used for attaching the disk
//
// The enumerated values for this property are expected to expand in the future. When processing this property, check
// for and log unknown values. Optionally halt processing and surface the error, or bypass the resource on which the
// unexpected property value was encountered.
const (
	DedicatedHostDiskInterfaceTypeNvmeConst = "nvme"
)

// Constants associated with the DedicatedHostDisk.LifecycleState property.
// The lifecycle state of this dedicated host disk.
const (
	DedicatedHostDiskLifecycleStateDeletingConst  = "deleting"
	DedicatedHostDiskLifecycleStateFailedConst    = "failed"
	DedicatedHostDiskLifecycleStatePendingConst   = "pending"
	DedicatedHostDiskLifecycleStateStableConst    = "stable"
	DedicatedHostDiskLifecycleStateSuspendedConst = "suspended"
	DedicatedHostDiskLifecycleStateUpdatingConst  = "updating"
	DedicatedHostDiskLifecycleStateWaitingConst   = "waiting"
)

// Constants associated with the DedicatedHostDisk.ResourceType property.
// The type of resource referenced.
const (
	DedicatedHostDiskResourceTypeDedicatedHostDiskConst = "dedicated_host_disk"
)

// Constants associated with the DedicatedHostDisk.SupportedInstanceInterfaceTypes property.
// The disk interface used for attaching the disk.
//
// The enumerated values for this property are expected to expand in the future. When processing this property, check
// for and log unknown values. Optionally halt processing and surface the error, or bypass the resource on which the
// unexpected property value was encountered.
const (
	DedicatedHostDiskSupportedInstanceInterfaceTypesNvmeConst      = "nvme"
	DedicatedHostDiskSupportedInstanceInterfaceTypesVirtioBlkConst = "virtio_blk"
)

// UnmarshalDedicatedHostDisk unmarshals an instance of DedicatedHostDisk from the specified map of raw messages.
func UnmarshalDedicatedHostDisk(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostDisk)
	err = core.UnmarshalPrimitive(m, "available", &obj.Available)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "instance_disks", &obj.InstanceDisks, UnmarshalInstanceDiskReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "interface_type", &obj.InterfaceType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "lifecycle_state", &obj.LifecycleState)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "provisionable", &obj.Provisionable)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "size", &obj.Size)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "supported_instance_interface_types", &obj.SupportedInstanceInterfaceTypes)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostDiskCollection : DedicatedHostDiskCollection struct
type DedicatedHostDiskCollection struct {
	// Collection of the dedicated host's disks.
	Disks []DedicatedHostDisk `json:"disks" validate:"required"`
}

// UnmarshalDedicatedHostDiskCollection unmarshals an instance of DedicatedHostDiskCollection from the specified map of raw messages.
func UnmarshalDedicatedHostDiskCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostDiskCollection)
	err = core.UnmarshalModel(m, "disks", &obj.Disks, UnmarshalDedicatedHostDisk)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostDiskPatch : DedicatedHostDiskPatch struct
type DedicatedHostDiskPatch struct {
	// The user-defined name for this disk.
	Name *string `json:"name,omitempty"`
}

// UnmarshalDedicatedHostDiskPatch unmarshals an instance of DedicatedHostDiskPatch from the specified map of raw messages.
func UnmarshalDedicatedHostDiskPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostDiskPatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the DedicatedHostDiskPatch
func (dedicatedHostDiskPatch *DedicatedHostDiskPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(dedicatedHostDiskPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// DedicatedHostGroup : DedicatedHostGroup struct
type DedicatedHostGroup struct {
	// The dedicated host profile class for hosts in this group.
	Class *string `json:"class" validate:"required"`

	// The date and time that the dedicated host group was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN for this dedicated host group.
	CRN *string `json:"crn" validate:"required"`

	// The dedicated hosts that are in this dedicated host group.
	DedicatedHosts []DedicatedHostReference `json:"dedicated_hosts" validate:"required"`

	// The dedicated host profile family for hosts in this group.
	Family *string `json:"family" validate:"required"`

	// The URL for this dedicated host group.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this dedicated host group.
	ID *string `json:"id" validate:"required"`

	// The unique user-defined name for this dedicated host group. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string `json:"name" validate:"required"`

	// The resource group for this dedicated host group.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The type of resource referenced.
	ResourceType *string `json:"resource_type" validate:"required"`

	// The instance profiles usable by instances placed on this dedicated host group.
	SupportedInstanceProfiles []InstanceProfileReference `json:"supported_instance_profiles" validate:"required"`

	// The zone this dedicated host group resides in.
	Zone *ZoneReference `json:"zone" validate:"required"`
}

// Constants associated with the DedicatedHostGroup.Family property.
// The dedicated host profile family for hosts in this group.
const (
	DedicatedHostGroupFamilyBalancedConst = "balanced"
	DedicatedHostGroupFamilyComputeConst  = "compute"
	DedicatedHostGroupFamilyMemoryConst   = "memory"
)

// Constants associated with the DedicatedHostGroup.ResourceType property.
// The type of resource referenced.
const (
	DedicatedHostGroupResourceTypeDedicatedHostGroupConst = "dedicated_host_group"
)

// UnmarshalDedicatedHostGroup unmarshals an instance of DedicatedHostGroup from the specified map of raw messages.
func UnmarshalDedicatedHostGroup(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostGroup)
	err = core.UnmarshalPrimitive(m, "class", &obj.Class)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "dedicated_hosts", &obj.DedicatedHosts, UnmarshalDedicatedHostReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "family", &obj.Family)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "supported_instance_profiles", &obj.SupportedInstanceProfiles, UnmarshalInstanceProfileReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostGroupCollection : DedicatedHostGroupCollection struct
type DedicatedHostGroupCollection struct {
	// A link to the first page of resources.
	First *DedicatedHostGroupCollectionFirst `json:"first" validate:"required"`

	// Collection of dedicated host groups.
	Groups []DedicatedHostGroup `json:"groups" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *DedicatedHostGroupCollectionNext `json:"next,omitempty"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`
}

// UnmarshalDedicatedHostGroupCollection unmarshals an instance of DedicatedHostGroupCollection from the specified map of raw messages.
func UnmarshalDedicatedHostGroupCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostGroupCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalDedicatedHostGroupCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "groups", &obj.Groups, UnmarshalDedicatedHostGroup)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalDedicatedHostGroupCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostGroupCollectionFirst : A link to the first page of resources.
type DedicatedHostGroupCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalDedicatedHostGroupCollectionFirst unmarshals an instance of DedicatedHostGroupCollectionFirst from the specified map of raw messages.
func UnmarshalDedicatedHostGroupCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostGroupCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostGroupCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type DedicatedHostGroupCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalDedicatedHostGroupCollectionNext unmarshals an instance of DedicatedHostGroupCollectionNext from the specified map of raw messages.
func UnmarshalDedicatedHostGroupCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostGroupCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostGroupIdentity : Identifies a dedicated host group by a unique property.
// Models which "extend" this model:
// - DedicatedHostGroupIdentityByID
// - DedicatedHostGroupIdentityByCRN
// - DedicatedHostGroupIdentityByHref
type DedicatedHostGroupIdentity struct {
	// The unique identifier for this dedicated host group.
	ID *string `json:"id,omitempty"`

	// The CRN for this dedicated host group.
	CRN *string `json:"crn,omitempty"`

	// The URL for this dedicated host group.
	Href *string `json:"href,omitempty"`
}

func (*DedicatedHostGroupIdentity) isaDedicatedHostGroupIdentity() bool {
	return true
}

type DedicatedHostGroupIdentityIntf interface {
	isaDedicatedHostGroupIdentity() bool
}

// UnmarshalDedicatedHostGroupIdentity unmarshals an instance of DedicatedHostGroupIdentity from the specified map of raw messages.
func UnmarshalDedicatedHostGroupIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostGroupIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostGroupPatch : DedicatedHostGroupPatch struct
type DedicatedHostGroupPatch struct {
	// The unique user-defined name for this dedicated host group. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string `json:"name,omitempty"`
}

// UnmarshalDedicatedHostGroupPatch unmarshals an instance of DedicatedHostGroupPatch from the specified map of raw messages.
func UnmarshalDedicatedHostGroupPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostGroupPatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the DedicatedHostGroupPatch
func (dedicatedHostGroupPatch *DedicatedHostGroupPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(dedicatedHostGroupPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// DedicatedHostGroupPrototypeDedicatedHostByZoneContext : DedicatedHostGroupPrototypeDedicatedHostByZoneContext struct
type DedicatedHostGroupPrototypeDedicatedHostByZoneContext struct {
	// The unique user-defined name for this dedicated host group. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string `json:"name,omitempty"`

	// The resource group to use. If unspecified, the host's resource group is used.
	ResourceGroup ResourceGroupIdentityIntf `json:"resource_group,omitempty"`
}

// UnmarshalDedicatedHostGroupPrototypeDedicatedHostByZoneContext unmarshals an instance of DedicatedHostGroupPrototypeDedicatedHostByZoneContext from the specified map of raw messages.
func UnmarshalDedicatedHostGroupPrototypeDedicatedHostByZoneContext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostGroupPrototypeDedicatedHostByZoneContext)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupIdentity)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostGroupReference : DedicatedHostGroupReference struct
type DedicatedHostGroupReference struct {
	// The CRN for this dedicated host group.
	CRN *string `json:"crn" validate:"required"`

	// If present, this property indicates the referenced resource has been deleted and provides
	// some supplementary information.
	Deleted *DedicatedHostGroupReferenceDeleted `json:"deleted,omitempty"`

	// The URL for this dedicated host group.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this dedicated host group.
	ID *string `json:"id" validate:"required"`

	// The unique user-defined name for this dedicated host group. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string `json:"name" validate:"required"`

	// The type of resource referenced.
	ResourceType *string `json:"resource_type" validate:"required"`
}

// Constants associated with the DedicatedHostGroupReference.ResourceType property.
// The type of resource referenced.
const (
	DedicatedHostGroupReferenceResourceTypeDedicatedHostGroupConst = "dedicated_host_group"
)

// UnmarshalDedicatedHostGroupReference unmarshals an instance of DedicatedHostGroupReference from the specified map of raw messages.
func UnmarshalDedicatedHostGroupReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostGroupReference)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "deleted", &obj.Deleted, UnmarshalDedicatedHostGroupReferenceDeleted)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostGroupReferenceDeleted : If present, this property indicates the referenced resource has been deleted and provides some supplementary
// information.
type DedicatedHostGroupReferenceDeleted struct {
	// Link to documentation about deleted resources.
	MoreInfo *string `json:"more_info" validate:"required"`
}

// UnmarshalDedicatedHostGroupReferenceDeleted unmarshals an instance of DedicatedHostGroupReferenceDeleted from the specified map of raw messages.
func UnmarshalDedicatedHostGroupReferenceDeleted(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostGroupReferenceDeleted)
	err = core.UnmarshalPrimitive(m, "more_info", &obj.MoreInfo)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostPatch : DedicatedHostPatch struct
type DedicatedHostPatch struct {
	// If set to true, instances can be placed on this dedicated host.
	InstancePlacementEnabled *bool `json:"instance_placement_enabled,omitempty"`

	// The unique user-defined name for this dedicated host. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string `json:"name,omitempty"`
}

// UnmarshalDedicatedHostPatch unmarshals an instance of DedicatedHostPatch from the specified map of raw messages.
func UnmarshalDedicatedHostPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostPatch)
	err = core.UnmarshalPrimitive(m, "instance_placement_enabled", &obj.InstancePlacementEnabled)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the DedicatedHostPatch
func (dedicatedHostPatch *DedicatedHostPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(dedicatedHostPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// DedicatedHostProfile : DedicatedHostProfile struct
type DedicatedHostProfile struct {
	// The product class this dedicated host profile belongs to.
	Class *string `json:"class" validate:"required"`

	// Collection of the dedicated host profile's disks.
	Disks []DedicatedHostProfileDisk `json:"disks" validate:"required"`

	// The product family this dedicated host profile belongs to
	//
	// The enumerated values for this property are expected to expand in the future. When processing this property, check
	// for and log unknown values. Optionally halt processing and surface the error, or bypass the resource on which the
	// unexpected property value was encountered.
	Family *string `json:"family" validate:"required"`

	// The URL for this dedicated host.
	Href *string `json:"href" validate:"required"`

	Memory DedicatedHostProfileMemoryIntf `json:"memory" validate:"required"`

	// The globally unique name for this dedicated host profile.
	Name *string `json:"name" validate:"required"`

	SocketCount DedicatedHostProfileSocketIntf `json:"socket_count" validate:"required"`

	// The instance profiles usable by instances placed on dedicated hosts with this profile.
	SupportedInstanceProfiles []InstanceProfileReference `json:"supported_instance_profiles" validate:"required"`

	VcpuArchitecture *DedicatedHostProfileVcpuArchitecture `json:"vcpu_architecture" validate:"required"`

	VcpuCount DedicatedHostProfileVcpuIntf `json:"vcpu_count" validate:"required"`
}

// Constants associated with the DedicatedHostProfile.Family property.
// The product family this dedicated host profile belongs to
//
// The enumerated values for this property are expected to expand in the future. When processing this property, check
// for and log unknown values. Optionally halt processing and surface the error, or bypass the resource on which the
// unexpected property value was encountered.
const (
	DedicatedHostProfileFamilyBalancedConst = "balanced"
	DedicatedHostProfileFamilyComputeConst  = "compute"
	DedicatedHostProfileFamilyMemoryConst   = "memory"
)

// UnmarshalDedicatedHostProfile unmarshals an instance of DedicatedHostProfile from the specified map of raw messages.
func UnmarshalDedicatedHostProfile(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostProfile)
	err = core.UnmarshalPrimitive(m, "class", &obj.Class)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "disks", &obj.Disks, UnmarshalDedicatedHostProfileDisk)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "family", &obj.Family)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "memory", &obj.Memory, UnmarshalDedicatedHostProfileMemory)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "socket_count", &obj.SocketCount, UnmarshalDedicatedHostProfileSocket)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "supported_instance_profiles", &obj.SupportedInstanceProfiles, UnmarshalInstanceProfileReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vcpu_architecture", &obj.VcpuArchitecture, UnmarshalDedicatedHostProfileVcpuArchitecture)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vcpu_count", &obj.VcpuCount, UnmarshalDedicatedHostProfileVcpu)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostProfileCollection : DedicatedHostProfileCollection struct
type DedicatedHostProfileCollection struct {
	// A link to the first page of resources.
	First *DedicatedHostProfileCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *DedicatedHostProfileCollectionNext `json:"next,omitempty"`

	// Collection of dedicated host profiles.
	Profiles []DedicatedHostProfile `json:"profiles" validate:"required"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`
}

// UnmarshalDedicatedHostProfileCollection unmarshals an instance of DedicatedHostProfileCollection from the specified map of raw messages.
func UnmarshalDedicatedHostProfileCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostProfileCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalDedicatedHostProfileCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalDedicatedHostProfileCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "profiles", &obj.Profiles, UnmarshalDedicatedHostProfile)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostProfileCollectionFirst : A link to the first page of resources.
type DedicatedHostProfileCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalDedicatedHostProfileCollectionFirst unmarshals an instance of DedicatedHostProfileCollectionFirst from the specified map of raw messages.
func UnmarshalDedicatedHostProfileCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostProfileCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostProfileCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type DedicatedHostProfileCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalDedicatedHostProfileCollectionNext unmarshals an instance of DedicatedHostProfileCollectionNext from the specified map of raw messages.
func UnmarshalDedicatedHostProfileCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostProfileCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostProfileDisk : Disks provided by this profile.
type DedicatedHostProfileDisk struct {
	InterfaceType *DedicatedHostProfileDiskInterface `json:"interface_type" validate:"required"`

	// The number of disks of this type for a dedicated host with this profile.
	Quantity *DedicatedHostProfileDiskQuantity `json:"quantity" validate:"required"`

	// The size of the disk in GB (gigabytes).
	Size *DedicatedHostProfileDiskSize `json:"size" validate:"required"`

	SupportedInstanceInterfaceTypes *DedicatedHostProfileDiskSupportedInterfaces `json:"supported_instance_interface_types" validate:"required"`
}

// UnmarshalDedicatedHostProfileDisk unmarshals an instance of DedicatedHostProfileDisk from the specified map of raw messages.
func UnmarshalDedicatedHostProfileDisk(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostProfileDisk)
	err = core.UnmarshalModel(m, "interface_type", &obj.InterfaceType, UnmarshalDedicatedHostProfileDiskInterface)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "quantity", &obj.Quantity, UnmarshalDedicatedHostProfileDiskQuantity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "size", &obj.Size, UnmarshalDedicatedHostProfileDiskSize)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "supported_instance_interface_types", &obj.SupportedInstanceInterfaceTypes, UnmarshalDedicatedHostProfileDiskSupportedInterfaces)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostProfileDiskInterface : DedicatedHostProfileDiskInterface struct
type DedicatedHostProfileDiskInterface struct {
	// The type for this profile field.
	Type *string `json:"type" validate:"required"`

	// The interface of the disk for a dedicated host with this profile
	//
	// The enumerated values for this property are expected to expand in the future. When processing this property, check
	// for and log unknown values. Optionally halt processing and surface the error, or bypass the resource on which the
	// unexpected property value was encountered.
	Value *string `json:"value" validate:"required"`
}

// Constants associated with the DedicatedHostProfileDiskInterface.Type property.
// The type for this profile field.
const (
	DedicatedHostProfileDiskInterfaceTypeFixedConst = "fixed"
)

// Constants associated with the DedicatedHostProfileDiskInterface.Value property.
// The interface of the disk for a dedicated host with this profile
//
// The enumerated values for this property are expected to expand in the future. When processing this property, check
// for and log unknown values. Optionally halt processing and surface the error, or bypass the resource on which the
// unexpected property value was encountered.
const (
	DedicatedHostProfileDiskInterfaceValueNvmeConst = "nvme"
)

// UnmarshalDedicatedHostProfileDiskInterface unmarshals an instance of DedicatedHostProfileDiskInterface from the specified map of raw messages.
func UnmarshalDedicatedHostProfileDiskInterface(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostProfileDiskInterface)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "value", &obj.Value)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostProfileDiskQuantity : The number of disks of this type for a dedicated host with this profile.
type DedicatedHostProfileDiskQuantity struct {
	// The type for this profile field.
	Type *string `json:"type" validate:"required"`

	// The value for this profile field.
	Value *int64 `json:"value" validate:"required"`
}

// Constants associated with the DedicatedHostProfileDiskQuantity.Type property.
// The type for this profile field.
const (
	DedicatedHostProfileDiskQuantityTypeFixedConst = "fixed"
)

// UnmarshalDedicatedHostProfileDiskQuantity unmarshals an instance of DedicatedHostProfileDiskQuantity from the specified map of raw messages.
func UnmarshalDedicatedHostProfileDiskQuantity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostProfileDiskQuantity)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "value", &obj.Value)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostProfileDiskSize : The size of the disk in GB (gigabytes).
type DedicatedHostProfileDiskSize struct {
	// The type for this profile field.
	Type *string `json:"type" validate:"required"`

	// The size of the disk in GB (gigabytes).
	Value *int64 `json:"value" validate:"required"`
}

// Constants associated with the DedicatedHostProfileDiskSize.Type property.
// The type for this profile field.
const (
	DedicatedHostProfileDiskSizeTypeFixedConst = "fixed"
)

// UnmarshalDedicatedHostProfileDiskSize unmarshals an instance of DedicatedHostProfileDiskSize from the specified map of raw messages.
func UnmarshalDedicatedHostProfileDiskSize(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostProfileDiskSize)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "value", &obj.Value)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostProfileDiskSupportedInterfaces : DedicatedHostProfileDiskSupportedInterfaces struct
type DedicatedHostProfileDiskSupportedInterfaces struct {
	// The type for this profile field.
	Type *string `json:"type" validate:"required"`

	// The instance disk interfaces supported for a dedicated host with this profile.
	Value []string `json:"value" validate:"required"`
}

// Constants associated with the DedicatedHostProfileDiskSupportedInterfaces.Type property.
// The type for this profile field.
const (
	DedicatedHostProfileDiskSupportedInterfacesTypeFixedConst = "fixed"
)

// Constants associated with the DedicatedHostProfileDiskSupportedInterfaces.Value property.
// The disk interface used for attaching the disk.
//
// The enumerated values for this property are expected to expand in the future. When processing this property, check
// for and log unknown values. Optionally halt processing and surface the error, or bypass the resource on which the
// unexpected property value was encountered.
const (
	DedicatedHostProfileDiskSupportedInterfacesValueNvmeConst      = "nvme"
	DedicatedHostProfileDiskSupportedInterfacesValueVirtioBlkConst = "virtio_blk"
)

// UnmarshalDedicatedHostProfileDiskSupportedInterfaces unmarshals an instance of DedicatedHostProfileDiskSupportedInterfaces from the specified map of raw messages.
func UnmarshalDedicatedHostProfileDiskSupportedInterfaces(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostProfileDiskSupportedInterfaces)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "value", &obj.Value)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostProfileIdentity : Identifies a dedicated host profile by a unique property.
// Models which "extend" this model:
// - DedicatedHostProfileIdentityByName
// - DedicatedHostProfileIdentityByHref
type DedicatedHostProfileIdentity struct {
	// The globally unique name for this dedicated host profile.
	Name *string `json:"name,omitempty"`

	// The URL for this dedicated host profile.
	Href *string `json:"href,omitempty"`
}

func (*DedicatedHostProfileIdentity) isaDedicatedHostProfileIdentity() bool {
	return true
}

type DedicatedHostProfileIdentityIntf interface {
	isaDedicatedHostProfileIdentity() bool
}

// UnmarshalDedicatedHostProfileIdentity unmarshals an instance of DedicatedHostProfileIdentity from the specified map of raw messages.
func UnmarshalDedicatedHostProfileIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostProfileIdentity)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostProfileMemory : DedicatedHostProfileMemory struct
// Models which "extend" this model:
// - DedicatedHostProfileMemoryFixed
// - DedicatedHostProfileMemoryRange
// - DedicatedHostProfileMemoryEnum
// - DedicatedHostProfileMemoryDependent
type DedicatedHostProfileMemory struct {
	// The type for this profile field.
	Type *string `json:"type,omitempty"`

	// The value for this profile field.
	Value *int64 `json:"value,omitempty"`

	// The default value for this profile field.
	Default *int64 `json:"default,omitempty"`

	// The maximum value for this profile field.
	Max *int64 `json:"max,omitempty"`

	// The minimum value for this profile field.
	Min *int64 `json:"min,omitempty"`

	// The increment step value for this profile field.
	Step *int64 `json:"step,omitempty"`

	// The permitted values for this profile field.
	Values []int64 `json:"values,omitempty"`
}

// Constants associated with the DedicatedHostProfileMemory.Type property.
// The type for this profile field.
const (
	DedicatedHostProfileMemoryTypeFixedConst = "fixed"
)

func (*DedicatedHostProfileMemory) isaDedicatedHostProfileMemory() bool {
	return true
}

type DedicatedHostProfileMemoryIntf interface {
	isaDedicatedHostProfileMemory() bool
}

// UnmarshalDedicatedHostProfileMemory unmarshals an instance of DedicatedHostProfileMemory from the specified map of raw messages.
func UnmarshalDedicatedHostProfileMemory(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostProfileMemory)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "value", &obj.Value)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "default", &obj.Default)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "max", &obj.Max)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "min", &obj.Min)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "step", &obj.Step)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "values", &obj.Values)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostProfileReference : DedicatedHostProfileReference struct
type DedicatedHostProfileReference struct {
	// The URL for this dedicated host.
	Href *string `json:"href" validate:"required"`

	// The globally unique name for this dedicated host profile.
	Name *string `json:"name" validate:"required"`
}

// UnmarshalDedicatedHostProfileReference unmarshals an instance of DedicatedHostProfileReference from the specified map of raw messages.
func UnmarshalDedicatedHostProfileReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostProfileReference)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostProfileSocket : DedicatedHostProfileSocket struct
// Models which "extend" this model:
// - DedicatedHostProfileSocketFixed
// - DedicatedHostProfileSocketRange
// - DedicatedHostProfileSocketEnum
// - DedicatedHostProfileSocketDependent
type DedicatedHostProfileSocket struct {
	// The type for this profile field.
	Type *string `json:"type,omitempty"`

	// The value for this profile field.
	Value *int64 `json:"value,omitempty"`

	// The default value for this profile field.
	Default *int64 `json:"default,omitempty"`

	// The maximum value for this profile field.
	Max *int64 `json:"max,omitempty"`

	// The minimum value for this profile field.
	Min *int64 `json:"min,omitempty"`

	// The increment step value for this profile field.
	Step *int64 `json:"step,omitempty"`

	// The permitted values for this profile field.
	Values []int64 `json:"values,omitempty"`
}

// Constants associated with the DedicatedHostProfileSocket.Type property.
// The type for this profile field.
const (
	DedicatedHostProfileSocketTypeFixedConst = "fixed"
)

func (*DedicatedHostProfileSocket) isaDedicatedHostProfileSocket() bool {
	return true
}

type DedicatedHostProfileSocketIntf interface {
	isaDedicatedHostProfileSocket() bool
}

// UnmarshalDedicatedHostProfileSocket unmarshals an instance of DedicatedHostProfileSocket from the specified map of raw messages.
func UnmarshalDedicatedHostProfileSocket(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostProfileSocket)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "value", &obj.Value)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "default", &obj.Default)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "max", &obj.Max)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "min", &obj.Min)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "step", &obj.Step)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "values", &obj.Values)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostProfileVcpu : DedicatedHostProfileVcpu struct
// Models which "extend" this model:
// - DedicatedHostProfileVcpuFixed
// - DedicatedHostProfileVcpuRange
// - DedicatedHostProfileVcpuEnum
// - DedicatedHostProfileVcpuDependent
type DedicatedHostProfileVcpu struct {
	// The type for this profile field.
	Type *string `json:"type,omitempty"`

	// The value for this profile field.
	Value *int64 `json:"value,omitempty"`

	// The default value for this profile field.
	Default *int64 `json:"default,omitempty"`

	// The maximum value for this profile field.
	Max *int64 `json:"max,omitempty"`

	// The minimum value for this profile field.
	Min *int64 `json:"min,omitempty"`

	// The increment step value for this profile field.
	Step *int64 `json:"step,omitempty"`

	// The permitted values for this profile field.
	Values []int64 `json:"values,omitempty"`
}

// Constants associated with the DedicatedHostProfileVcpu.Type property.
// The type for this profile field.
const (
	DedicatedHostProfileVcpuTypeFixedConst = "fixed"
)

func (*DedicatedHostProfileVcpu) isaDedicatedHostProfileVcpu() bool {
	return true
}

type DedicatedHostProfileVcpuIntf interface {
	isaDedicatedHostProfileVcpu() bool
}

// UnmarshalDedicatedHostProfileVcpu unmarshals an instance of DedicatedHostProfileVcpu from the specified map of raw messages.
func UnmarshalDedicatedHostProfileVcpu(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostProfileVcpu)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "value", &obj.Value)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "default", &obj.Default)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "max", &obj.Max)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "min", &obj.Min)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "step", &obj.Step)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "values", &obj.Values)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostProfileVcpuArchitecture : DedicatedHostProfileVcpuArchitecture struct
type DedicatedHostProfileVcpuArchitecture struct {
	// The type for this profile field.
	Type *string `json:"type" validate:"required"`

	// The VCPU architecture for a dedicated host with this profile.
	Value *string `json:"value" validate:"required"`
}

// Constants associated with the DedicatedHostProfileVcpuArchitecture.Type property.
// The type for this profile field.
const (
	DedicatedHostProfileVcpuArchitectureTypeFixedConst = "fixed"
)

// UnmarshalDedicatedHostProfileVcpuArchitecture unmarshals an instance of DedicatedHostProfileVcpuArchitecture from the specified map of raw messages.
func UnmarshalDedicatedHostProfileVcpuArchitecture(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostProfileVcpuArchitecture)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "value", &obj.Value)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostPrototype : DedicatedHostPrototype struct
// Models which "extend" this model:
// - DedicatedHostPrototypeDedicatedHostByGroup
// - DedicatedHostPrototypeDedicatedHostByZone
type DedicatedHostPrototype struct {
	// If set to true, instances can be placed on this dedicated host.
	InstancePlacementEnabled *bool `json:"instance_placement_enabled,omitempty"`

	// The unique user-defined name for this dedicated host. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string `json:"name,omitempty"`

	// The profile to use for this dedicated host.
	Profile DedicatedHostProfileIdentityIntf `json:"profile" validate:"required"`

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf `json:"resource_group,omitempty"`

	// The dedicated host group for this dedicated host.
	Group DedicatedHostGroupIdentityIntf `json:"group,omitempty"`

	// The zone this dedicated host will reside in.
	Zone ZoneIdentityIntf `json:"zone,omitempty"`
}

func (*DedicatedHostPrototype) isaDedicatedHostPrototype() bool {
	return true
}

type DedicatedHostPrototypeIntf interface {
	isaDedicatedHostPrototype() bool
}

// UnmarshalDedicatedHostPrototype unmarshals an instance of DedicatedHostPrototype from the specified map of raw messages.
func UnmarshalDedicatedHostPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostPrototype)
	err = core.UnmarshalPrimitive(m, "instance_placement_enabled", &obj.InstancePlacementEnabled)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "profile", &obj.Profile, UnmarshalDedicatedHostProfileIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "group", &obj.Group, UnmarshalDedicatedHostGroupIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneIdentity)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostReference : DedicatedHostReference struct
type DedicatedHostReference struct {
	// The CRN for this dedicated host.
	CRN *string `json:"crn" validate:"required"`

	// If present, this property indicates the referenced resource has been deleted and provides
	// some supplementary information.
	Deleted *DedicatedHostReferenceDeleted `json:"deleted,omitempty"`

	// The URL for this dedicated host.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this dedicated host.
	ID *string `json:"id" validate:"required"`

	// The unique user-defined name for this dedicated host. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string `json:"name" validate:"required"`

	// The type of resource referenced.
	ResourceType *string `json:"resource_type" validate:"required"`
}

// Constants associated with the DedicatedHostReference.ResourceType property.
// The type of resource referenced.
const (
	DedicatedHostReferenceResourceTypeDedicatedHostConst = "dedicated_host"
)

// UnmarshalDedicatedHostReference unmarshals an instance of DedicatedHostReference from the specified map of raw messages.
func UnmarshalDedicatedHostReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostReference)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "deleted", &obj.Deleted, UnmarshalDedicatedHostReferenceDeleted)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DedicatedHostReferenceDeleted : If present, this property indicates the referenced resource has been deleted and provides some supplementary
// information.
type DedicatedHostReferenceDeleted struct {
	// Link to documentation about deleted resources.
	MoreInfo *string `json:"more_info" validate:"required"`
}

// UnmarshalDedicatedHostReferenceDeleted unmarshals an instance of DedicatedHostReferenceDeleted from the specified map of raw messages.
func UnmarshalDedicatedHostReferenceDeleted(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DedicatedHostReferenceDeleted)
	err = core.UnmarshalPrimitive(m, "more_info", &obj.MoreInfo)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DefaultNetworkACL : DefaultNetworkACL struct
type DefaultNetworkACL struct {
	// The date and time that the network ACL was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN for this network ACL.
	CRN *string `json:"crn" validate:"required"`

	// The URL for this network ACL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this network ACL.
	ID *string `json:"id" validate:"required"`

	// The name of the default network ACL created for a VPC. The name will be a hyphenated list of randomly-selected words
	// at creation, but may be user-specified with a subsequent request.
	Name *string `json:"name" validate:"required"`

	// The resource group for the default network ACL for a VPC. Set to the VPC's
	// resource group at creation.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The ordered rules for the default network ACL for a VPC.  Defaults to two rules which allow all inbound and outbound
	// traffic, respectively.  Rules for the default network ACL may be changed, added, or removed.
	Rules []NetworkACLRuleItemIntf `json:"rules" validate:"required"`

	// The subnets to which this network ACL is attached.
	Subnets []SubnetReference `json:"subnets" validate:"required"`

	// The VPC this network ACL is a part of.
	VPC *VPCReference `json:"vpc" validate:"required"`
}

// UnmarshalDefaultNetworkACL unmarshals an instance of DefaultNetworkACL from the specified map of raw messages.
func UnmarshalDefaultNetworkACL(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DefaultNetworkACL)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "rules", &obj.Rules, UnmarshalNetworkACLRuleItem)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "subnets", &obj.Subnets, UnmarshalSubnetReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vpc", &obj.VPC, UnmarshalVPCReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DefaultRoutingTable : DefaultRoutingTable struct
type DefaultRoutingTable struct {
	// The date and time that this routing table was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The URL for this routing table.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this routing table.
	ID *string `json:"id" validate:"required"`

	// Indicates whether this is the default routing table for this VPC.
	IsDefault *bool `json:"is_default" validate:"required"`

	// The lifecycle state of the routing table.
	LifecycleState *string `json:"lifecycle_state" validate:"required"`

	// The name of the default routing table created for this VPC. The name will be a hyphenated list of randomly-selected
	// words at creation, but may be user-specified with a subsequent request.
	Name *string `json:"name" validate:"required"`

	// The resource type.
	ResourceType *string `json:"resource_type" validate:"required"`

	// Indicates whether this routing table is used to route traffic that originates from
	// [Direct Link](https://cloud.ibm.com/docs/dl/) to this VPC.
	//
	// Incoming traffic will be routed according to the routing table with one exception: routes with an `action` of
	// `deliver` are treated as `drop` unless the `next_hop` is an IP address within the VPC's address prefix ranges.
	// Therefore, if an incoming packet matches a route with a `next_hop` of an internet-bound IP address or a VPN gateway
	// connection, the packet will be dropped.
	RouteDirectLinkIngress *bool `json:"route_direct_link_ingress" validate:"required"`

	// Indicates whether this routing table is used to route traffic that originates from from [Transit
	// Gateway](https://cloud.ibm.com/cloud/transit-gateway/) to this VPC.
	//
	// Incoming traffic will be routed according to the routing table with one exception: routes with an `action` of
	// `deliver` are treated as `drop` unless the `next_hop` is an IP address within the VPC's address prefix ranges.
	// Therefore, if an incoming packet matches a route with a `next_hop` of an internet-bound IP address or a VPN gateway
	// connection, the packet will be dropped.
	RouteTransitGatewayIngress *bool `json:"route_transit_gateway_ingress" validate:"required"`

	// Indicates whether this routing table is used to route traffic that originates from subnets in other zones in this
	// VPC.
	//
	// Incoming traffic will be routed according to the routing table with one exception: routes with an `action` of
	// `deliver` are treated as `drop` unless the `next_hop` is an IP address within the VPC's address prefix ranges.
	// Therefore, if an incoming packet matches a route with a `next_hop` of an internet-bound IP address or a VPN gateway
	// connection, the packet will be dropped.
	RouteVPCZoneIngress *bool `json:"route_vpc_zone_ingress" validate:"required"`

	// The routes for the default routing table for this VPC. The table is created with no routes, but routes may be added,
	// changed, or removed with a subsequent request.
	Routes []RouteReference `json:"routes" validate:"required"`

	// The subnets to which this routing table is attached.
	Subnets []SubnetReference `json:"subnets" validate:"required"`
}

// Constants associated with the DefaultRoutingTable.LifecycleState property.
// The lifecycle state of the routing table.
const (
	DefaultRoutingTableLifecycleStateDeletingConst  = "deleting"
	DefaultRoutingTableLifecycleStateFailedConst    = "failed"
	DefaultRoutingTableLifecycleStatePendingConst   = "pending"
	DefaultRoutingTableLifecycleStateStableConst    = "stable"
	DefaultRoutingTableLifecycleStateSuspendedConst = "suspended"
	DefaultRoutingTableLifecycleStateUpdatingConst  = "updating"
	DefaultRoutingTableLifecycleStateWaitingConst   = "waiting"
)

// Constants associated with the DefaultRoutingTable.ResourceType property.
// The resource type.
const (
	DefaultRoutingTableResourceTypeRoutingTableConst = "routing_table"
)

// UnmarshalDefaultRoutingTable unmarshals an instance of DefaultRoutingTable from the specified map of raw messages.
func UnmarshalDefaultRoutingTable(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DefaultRoutingTable)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "is_default", &obj.IsDefault)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "lifecycle_state", &obj.LifecycleState)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "route_direct_link_ingress", &obj.RouteDirectLinkIngress)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "route_transit_gateway_ingress", &obj.RouteTransitGatewayIngress)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "route_vpc_zone_ingress", &obj.RouteVPCZoneIngress)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "routes", &obj.Routes, UnmarshalRouteReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "subnets", &obj.Subnets, UnmarshalSubnetReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DefaultSecurityGroup : DefaultSecurityGroup struct
type DefaultSecurityGroup struct {
	// The date and time that this security group was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The security group's CRN.
	CRN *string `json:"crn" validate:"required"`

	// The security group's canonical URL.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this security group.
	ID *string `json:"id" validate:"required"`

	// The name of the default security group created for a VPC. The name will be a hyphenated list of randomly-selected
	// words at creation, but may be user-specified with a subsequent request.
	Name *string `json:"name" validate:"required"`

	// The resource group for this security group.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The rules for the default security group for a VPC. Defaults to allowing all outbound traffic, and allowing all
	// inbound traffic from other interfaces in the VPC's default security group. Rules in the default security group may
	// be changed, added or removed.
	Rules []SecurityGroupRuleIntf `json:"rules" validate:"required"`

	// The VPC this security group is a part of.
	VPC *VPCReference `json:"vpc" validate:"required"`
}

// UnmarshalDefaultSecurityGroup unmarshals an instance of DefaultSecurityGroup from the specified map of raw messages.
func UnmarshalDefaultSecurityGroup(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(DefaultSecurityGroup)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "rules", &obj.Rules, UnmarshalSecurityGroupRule)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vpc", &obj.VPC, UnmarshalVPCReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DeleteDedicatedHostGroupOptions : The DeleteDedicatedHostGroup options.
type DeleteDedicatedHostGroupOptions struct {
	// The dedicated host group identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteDedicatedHostGroupOptions : Instantiate DeleteDedicatedHostGroupOptions
func (*VpcV1) NewDeleteDedicatedHostGroupOptions(id string) *DeleteDedicatedHostGroupOptions {
	return &DeleteDedicatedHostGroupOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteDedicatedHostGroupOptions) SetID(id string) *DeleteDedicatedHostGroupOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteDedicatedHostGroupOptions) SetHeaders(param map[string]string) *DeleteDedicatedHostGroupOptions {
	options.Headers = param
	return options
}

// DeleteDedicatedHostOptions : The DeleteDedicatedHost options.
type DeleteDedicatedHostOptions struct {
	// The dedicated host identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteDedicatedHostOptions : Instantiate DeleteDedicatedHostOptions
func (*VpcV1) NewDeleteDedicatedHostOptions(id string) *DeleteDedicatedHostOptions {
	return &DeleteDedicatedHostOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteDedicatedHostOptions) SetID(id string) *DeleteDedicatedHostOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteDedicatedHostOptions) SetHeaders(param map[string]string) *DeleteDedicatedHostOptions {
	options.Headers = param
	return options
}

// DeleteEndpointGatewayOptions : The DeleteEndpointGateway options.
type DeleteEndpointGatewayOptions struct {
	// The endpoint gateway identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteEndpointGatewayOptions : Instantiate DeleteEndpointGatewayOptions
func (*VpcV1) NewDeleteEndpointGatewayOptions(id string) *DeleteEndpointGatewayOptions {
	return &DeleteEndpointGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteEndpointGatewayOptions) SetID(id string) *DeleteEndpointGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteEndpointGatewayOptions) SetHeaders(param map[string]string) *DeleteEndpointGatewayOptions {
	options.Headers = param
	return options
}

// DeleteFloatingIPOptions : The DeleteFloatingIP options.
type DeleteFloatingIPOptions struct {
	// The floating IP identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteFloatingIPOptions : Instantiate DeleteFloatingIPOptions
func (*VpcV1) NewDeleteFloatingIPOptions(id string) *DeleteFloatingIPOptions {
	return &DeleteFloatingIPOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteFloatingIPOptions) SetID(id string) *DeleteFloatingIPOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteFloatingIPOptions) SetHeaders(param map[string]string) *DeleteFloatingIPOptions {
	options.Headers = param
	return options
}

// DeleteFlowLogCollectorOptions : The DeleteFlowLogCollector options.
type DeleteFlowLogCollectorOptions struct {
	// The flow log collector identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteFlowLogCollectorOptions : Instantiate DeleteFlowLogCollectorOptions
func (*VpcV1) NewDeleteFlowLogCollectorOptions(id string) *DeleteFlowLogCollectorOptions {
	return &DeleteFlowLogCollectorOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteFlowLogCollectorOptions) SetID(id string) *DeleteFlowLogCollectorOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteFlowLogCollectorOptions) SetHeaders(param map[string]string) *DeleteFlowLogCollectorOptions {
	options.Headers = param
	return options
}

// DeleteIkePolicyOptions : The DeleteIkePolicy options.
type DeleteIkePolicyOptions struct {
	// The IKE policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteIkePolicyOptions : Instantiate DeleteIkePolicyOptions
func (*VpcV1) NewDeleteIkePolicyOptions(id string) *DeleteIkePolicyOptions {
	return &DeleteIkePolicyOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteIkePolicyOptions) SetID(id string) *DeleteIkePolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteIkePolicyOptions) SetHeaders(param map[string]string) *DeleteIkePolicyOptions {
	options.Headers = param
	return options
}

// DeleteImageOptions : The DeleteImage options.
type DeleteImageOptions struct {
	// The image identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteImageOptions : Instantiate DeleteImageOptions
func (*VpcV1) NewDeleteImageOptions(id string) *DeleteImageOptions {
	return &DeleteImageOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteImageOptions) SetID(id string) *DeleteImageOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteImageOptions) SetHeaders(param map[string]string) *DeleteImageOptions {
	options.Headers = param
	return options
}

// DeleteInstanceGroupLoadBalancerOptions : The DeleteInstanceGroupLoadBalancer options.
type DeleteInstanceGroupLoadBalancerOptions struct {
	// The instance group identifier.
	InstanceGroupID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteInstanceGroupLoadBalancerOptions : Instantiate DeleteInstanceGroupLoadBalancerOptions
func (*VpcV1) NewDeleteInstanceGroupLoadBalancerOptions(instanceGroupID string) *DeleteInstanceGroupLoadBalancerOptions {
	return &DeleteInstanceGroupLoadBalancerOptions{
		InstanceGroupID: core.StringPtr(instanceGroupID),
	}
}

// SetInstanceGroupID : Allow user to set InstanceGroupID
func (options *DeleteInstanceGroupLoadBalancerOptions) SetInstanceGroupID(instanceGroupID string) *DeleteInstanceGroupLoadBalancerOptions {
	options.InstanceGroupID = core.StringPtr(instanceGroupID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteInstanceGroupLoadBalancerOptions) SetHeaders(param map[string]string) *DeleteInstanceGroupLoadBalancerOptions {
	options.Headers = param
	return options
}

// DeleteInstanceGroupManagerActionOptions : The DeleteInstanceGroupManagerAction options.
type DeleteInstanceGroupManagerActionOptions struct {
	// The instance group identifier.
	InstanceGroupID *string `validate:"required,ne="`

	// The instance group manager identifier.
	InstanceGroupManagerID *string `validate:"required,ne="`

	// The instance group manager action identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteInstanceGroupManagerActionOptions : Instantiate DeleteInstanceGroupManagerActionOptions
func (*VpcV1) NewDeleteInstanceGroupManagerActionOptions(instanceGroupID string, instanceGroupManagerID string, id string) *DeleteInstanceGroupManagerActionOptions {
	return &DeleteInstanceGroupManagerActionOptions{
		InstanceGroupID:        core.StringPtr(instanceGroupID),
		InstanceGroupManagerID: core.StringPtr(instanceGroupManagerID),
		ID:                     core.StringPtr(id),
	}
}

// SetInstanceGroupID : Allow user to set InstanceGroupID
func (options *DeleteInstanceGroupManagerActionOptions) SetInstanceGroupID(instanceGroupID string) *DeleteInstanceGroupManagerActionOptions {
	options.InstanceGroupID = core.StringPtr(instanceGroupID)
	return options
}

// SetInstanceGroupManagerID : Allow user to set InstanceGroupManagerID
func (options *DeleteInstanceGroupManagerActionOptions) SetInstanceGroupManagerID(instanceGroupManagerID string) *DeleteInstanceGroupManagerActionOptions {
	options.InstanceGroupManagerID = core.StringPtr(instanceGroupManagerID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteInstanceGroupManagerActionOptions) SetID(id string) *DeleteInstanceGroupManagerActionOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteInstanceGroupManagerActionOptions) SetHeaders(param map[string]string) *DeleteInstanceGroupManagerActionOptions {
	options.Headers = param
	return options
}

// DeleteInstanceGroupManagerOptions : The DeleteInstanceGroupManager options.
type DeleteInstanceGroupManagerOptions struct {
	// The instance group identifier.
	InstanceGroupID *string `validate:"required,ne="`

	// The instance group manager identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteInstanceGroupManagerOptions : Instantiate DeleteInstanceGroupManagerOptions
func (*VpcV1) NewDeleteInstanceGroupManagerOptions(instanceGroupID string, id string) *DeleteInstanceGroupManagerOptions {
	return &DeleteInstanceGroupManagerOptions{
		InstanceGroupID: core.StringPtr(instanceGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetInstanceGroupID : Allow user to set InstanceGroupID
func (options *DeleteInstanceGroupManagerOptions) SetInstanceGroupID(instanceGroupID string) *DeleteInstanceGroupManagerOptions {
	options.InstanceGroupID = core.StringPtr(instanceGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteInstanceGroupManagerOptions) SetID(id string) *DeleteInstanceGroupManagerOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteInstanceGroupManagerOptions) SetHeaders(param map[string]string) *DeleteInstanceGroupManagerOptions {
	options.Headers = param
	return options
}

// DeleteInstanceGroupManagerPolicyOptions : The DeleteInstanceGroupManagerPolicy options.
type DeleteInstanceGroupManagerPolicyOptions struct {
	// The instance group identifier.
	InstanceGroupID *string `validate:"required,ne="`

	// The instance group manager identifier.
	InstanceGroupManagerID *string `validate:"required,ne="`

	// The instance group manager policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteInstanceGroupManagerPolicyOptions : Instantiate DeleteInstanceGroupManagerPolicyOptions
func (*VpcV1) NewDeleteInstanceGroupManagerPolicyOptions(instanceGroupID string, instanceGroupManagerID string, id string) *DeleteInstanceGroupManagerPolicyOptions {
	return &DeleteInstanceGroupManagerPolicyOptions{
		InstanceGroupID:        core.StringPtr(instanceGroupID),
		InstanceGroupManagerID: core.StringPtr(instanceGroupManagerID),
		ID:                     core.StringPtr(id),
	}
}

// SetInstanceGroupID : Allow user to set InstanceGroupID
func (options *DeleteInstanceGroupManagerPolicyOptions) SetInstanceGroupID(instanceGroupID string) *DeleteInstanceGroupManagerPolicyOptions {
	options.InstanceGroupID = core.StringPtr(instanceGroupID)
	return options
}

// SetInstanceGroupManagerID : Allow user to set InstanceGroupManagerID
func (options *DeleteInstanceGroupManagerPolicyOptions) SetInstanceGroupManagerID(instanceGroupManagerID string) *DeleteInstanceGroupManagerPolicyOptions {
	options.InstanceGroupManagerID = core.StringPtr(instanceGroupManagerID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteInstanceGroupManagerPolicyOptions) SetID(id string) *DeleteInstanceGroupManagerPolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteInstanceGroupManagerPolicyOptions) SetHeaders(param map[string]string) *DeleteInstanceGroupManagerPolicyOptions {
	options.Headers = param
	return options
}

// DeleteInstanceGroupMembershipOptions : The DeleteInstanceGroupMembership options.
type DeleteInstanceGroupMembershipOptions struct {
	// The instance group identifier.
	InstanceGroupID *string `validate:"required,ne="`

	// The instance group membership identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteInstanceGroupMembershipOptions : Instantiate DeleteInstanceGroupMembershipOptions
func (*VpcV1) NewDeleteInstanceGroupMembershipOptions(instanceGroupID string, id string) *DeleteInstanceGroupMembershipOptions {
	return &DeleteInstanceGroupMembershipOptions{
		InstanceGroupID: core.StringPtr(instanceGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetInstanceGroupID : Allow user to set InstanceGroupID
func (options *DeleteInstanceGroupMembershipOptions) SetInstanceGroupID(instanceGroupID string) *DeleteInstanceGroupMembershipOptions {
	options.InstanceGroupID = core.StringPtr(instanceGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteInstanceGroupMembershipOptions) SetID(id string) *DeleteInstanceGroupMembershipOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteInstanceGroupMembershipOptions) SetHeaders(param map[string]string) *DeleteInstanceGroupMembershipOptions {
	options.Headers = param
	return options
}

// DeleteInstanceGroupMembershipsOptions : The DeleteInstanceGroupMemberships options.
type DeleteInstanceGroupMembershipsOptions struct {
	// The instance group identifier.
	InstanceGroupID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteInstanceGroupMembershipsOptions : Instantiate DeleteInstanceGroupMembershipsOptions
func (*VpcV1) NewDeleteInstanceGroupMembershipsOptions(instanceGroupID string) *DeleteInstanceGroupMembershipsOptions {
	return &DeleteInstanceGroupMembershipsOptions{
		InstanceGroupID: core.StringPtr(instanceGroupID),
	}
}

// SetInstanceGroupID : Allow user to set InstanceGroupID
func (options *DeleteInstanceGroupMembershipsOptions) SetInstanceGroupID(instanceGroupID string) *DeleteInstanceGroupMembershipsOptions {
	options.InstanceGroupID = core.StringPtr(instanceGroupID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteInstanceGroupMembershipsOptions) SetHeaders(param map[string]string) *DeleteInstanceGroupMembershipsOptions {
	options.Headers = param
	return options
}

// DeleteInstanceGroupOptions : The DeleteInstanceGroup options.
type DeleteInstanceGroupOptions struct {
	// The instance group identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteInstanceGroupOptions : Instantiate DeleteInstanceGroupOptions
func (*VpcV1) NewDeleteInstanceGroupOptions(id string) *DeleteInstanceGroupOptions {
	return &DeleteInstanceGroupOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteInstanceGroupOptions) SetID(id string) *DeleteInstanceGroupOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteInstanceGroupOptions) SetHeaders(param map[string]string) *DeleteInstanceGroupOptions {
	options.Headers = param
	return options
}

// DeleteInstanceNetworkInterfaceOptions : The DeleteInstanceNetworkInterface options.
type DeleteInstanceNetworkInterfaceOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The network interface identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteInstanceNetworkInterfaceOptions : Instantiate DeleteInstanceNetworkInterfaceOptions
func (*VpcV1) NewDeleteInstanceNetworkInterfaceOptions(instanceID string, id string) *DeleteInstanceNetworkInterfaceOptions {
	return &DeleteInstanceNetworkInterfaceOptions{
		InstanceID: core.StringPtr(instanceID),
		ID:         core.StringPtr(id),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *DeleteInstanceNetworkInterfaceOptions) SetInstanceID(instanceID string) *DeleteInstanceNetworkInterfaceOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteInstanceNetworkInterfaceOptions) SetID(id string) *DeleteInstanceNetworkInterfaceOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteInstanceNetworkInterfaceOptions) SetHeaders(param map[string]string) *DeleteInstanceNetworkInterfaceOptions {
	options.Headers = param
	return options
}

// DeleteInstanceOptions : The DeleteInstance options.
type DeleteInstanceOptions struct {
	// The instance identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteInstanceOptions : Instantiate DeleteInstanceOptions
func (*VpcV1) NewDeleteInstanceOptions(id string) *DeleteInstanceOptions {
	return &DeleteInstanceOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteInstanceOptions) SetID(id string) *DeleteInstanceOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteInstanceOptions) SetHeaders(param map[string]string) *DeleteInstanceOptions {
	options.Headers = param
	return options
}

// DeleteInstanceTemplateOptions : The DeleteInstanceTemplate options.
type DeleteInstanceTemplateOptions struct {
	// The instance template identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteInstanceTemplateOptions : Instantiate DeleteInstanceTemplateOptions
func (*VpcV1) NewDeleteInstanceTemplateOptions(id string) *DeleteInstanceTemplateOptions {
	return &DeleteInstanceTemplateOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteInstanceTemplateOptions) SetID(id string) *DeleteInstanceTemplateOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteInstanceTemplateOptions) SetHeaders(param map[string]string) *DeleteInstanceTemplateOptions {
	options.Headers = param
	return options
}

// DeleteInstanceVolumeAttachmentOptions : The DeleteInstanceVolumeAttachment options.
type DeleteInstanceVolumeAttachmentOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The volume attachment identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteInstanceVolumeAttachmentOptions : Instantiate DeleteInstanceVolumeAttachmentOptions
func (*VpcV1) NewDeleteInstanceVolumeAttachmentOptions(instanceID string, id string) *DeleteInstanceVolumeAttachmentOptions {
	return &DeleteInstanceVolumeAttachmentOptions{
		InstanceID: core.StringPtr(instanceID),
		ID:         core.StringPtr(id),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *DeleteInstanceVolumeAttachmentOptions) SetInstanceID(instanceID string) *DeleteInstanceVolumeAttachmentOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteInstanceVolumeAttachmentOptions) SetID(id string) *DeleteInstanceVolumeAttachmentOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteInstanceVolumeAttachmentOptions) SetHeaders(param map[string]string) *DeleteInstanceVolumeAttachmentOptions {
	options.Headers = param
	return options
}

// DeleteIpsecPolicyOptions : The DeleteIpsecPolicy options.
type DeleteIpsecPolicyOptions struct {
	// The IPsec policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteIpsecPolicyOptions : Instantiate DeleteIpsecPolicyOptions
func (*VpcV1) NewDeleteIpsecPolicyOptions(id string) *DeleteIpsecPolicyOptions {
	return &DeleteIpsecPolicyOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteIpsecPolicyOptions) SetID(id string) *DeleteIpsecPolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteIpsecPolicyOptions) SetHeaders(param map[string]string) *DeleteIpsecPolicyOptions {
	options.Headers = param
	return options
}

// DeleteKeyOptions : The DeleteKey options.
type DeleteKeyOptions struct {
	// The key identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteKeyOptions : Instantiate DeleteKeyOptions
func (*VpcV1) NewDeleteKeyOptions(id string) *DeleteKeyOptions {
	return &DeleteKeyOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteKeyOptions) SetID(id string) *DeleteKeyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteKeyOptions) SetHeaders(param map[string]string) *DeleteKeyOptions {
	options.Headers = param
	return options
}

// DeleteLoadBalancerListenerOptions : The DeleteLoadBalancerListener options.
type DeleteLoadBalancerListenerOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteLoadBalancerListenerOptions : Instantiate DeleteLoadBalancerListenerOptions
func (*VpcV1) NewDeleteLoadBalancerListenerOptions(loadBalancerID string, id string) *DeleteLoadBalancerListenerOptions {
	return &DeleteLoadBalancerListenerOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *DeleteLoadBalancerListenerOptions) SetLoadBalancerID(loadBalancerID string) *DeleteLoadBalancerListenerOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteLoadBalancerListenerOptions) SetID(id string) *DeleteLoadBalancerListenerOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteLoadBalancerListenerOptions) SetHeaders(param map[string]string) *DeleteLoadBalancerListenerOptions {
	options.Headers = param
	return options
}

// DeleteLoadBalancerListenerPolicyOptions : The DeleteLoadBalancerListenerPolicy options.
type DeleteLoadBalancerListenerPolicyOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// The policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteLoadBalancerListenerPolicyOptions : Instantiate DeleteLoadBalancerListenerPolicyOptions
func (*VpcV1) NewDeleteLoadBalancerListenerPolicyOptions(loadBalancerID string, listenerID string, id string) *DeleteLoadBalancerListenerPolicyOptions {
	return &DeleteLoadBalancerListenerPolicyOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ListenerID:     core.StringPtr(listenerID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *DeleteLoadBalancerListenerPolicyOptions) SetLoadBalancerID(loadBalancerID string) *DeleteLoadBalancerListenerPolicyOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *DeleteLoadBalancerListenerPolicyOptions) SetListenerID(listenerID string) *DeleteLoadBalancerListenerPolicyOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteLoadBalancerListenerPolicyOptions) SetID(id string) *DeleteLoadBalancerListenerPolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteLoadBalancerListenerPolicyOptions) SetHeaders(param map[string]string) *DeleteLoadBalancerListenerPolicyOptions {
	options.Headers = param
	return options
}

// DeleteLoadBalancerListenerPolicyRuleOptions : The DeleteLoadBalancerListenerPolicyRule options.
type DeleteLoadBalancerListenerPolicyRuleOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// The policy identifier.
	PolicyID *string `validate:"required,ne="`

	// The rule identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteLoadBalancerListenerPolicyRuleOptions : Instantiate DeleteLoadBalancerListenerPolicyRuleOptions
func (*VpcV1) NewDeleteLoadBalancerListenerPolicyRuleOptions(loadBalancerID string, listenerID string, policyID string, id string) *DeleteLoadBalancerListenerPolicyRuleOptions {
	return &DeleteLoadBalancerListenerPolicyRuleOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ListenerID:     core.StringPtr(listenerID),
		PolicyID:       core.StringPtr(policyID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *DeleteLoadBalancerListenerPolicyRuleOptions) SetLoadBalancerID(loadBalancerID string) *DeleteLoadBalancerListenerPolicyRuleOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *DeleteLoadBalancerListenerPolicyRuleOptions) SetListenerID(listenerID string) *DeleteLoadBalancerListenerPolicyRuleOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetPolicyID : Allow user to set PolicyID
func (options *DeleteLoadBalancerListenerPolicyRuleOptions) SetPolicyID(policyID string) *DeleteLoadBalancerListenerPolicyRuleOptions {
	options.PolicyID = core.StringPtr(policyID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteLoadBalancerListenerPolicyRuleOptions) SetID(id string) *DeleteLoadBalancerListenerPolicyRuleOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteLoadBalancerListenerPolicyRuleOptions) SetHeaders(param map[string]string) *DeleteLoadBalancerListenerPolicyRuleOptions {
	options.Headers = param
	return options
}

// DeleteLoadBalancerOptions : The DeleteLoadBalancer options.
type DeleteLoadBalancerOptions struct {
	// The load balancer identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteLoadBalancerOptions : Instantiate DeleteLoadBalancerOptions
func (*VpcV1) NewDeleteLoadBalancerOptions(id string) *DeleteLoadBalancerOptions {
	return &DeleteLoadBalancerOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteLoadBalancerOptions) SetID(id string) *DeleteLoadBalancerOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteLoadBalancerOptions) SetHeaders(param map[string]string) *DeleteLoadBalancerOptions {
	options.Headers = param
	return options
}

// DeleteLoadBalancerPoolMemberOptions : The DeleteLoadBalancerPoolMember options.
type DeleteLoadBalancerPoolMemberOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The pool identifier.
	PoolID *string `validate:"required,ne="`

	// The member identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteLoadBalancerPoolMemberOptions : Instantiate DeleteLoadBalancerPoolMemberOptions
func (*VpcV1) NewDeleteLoadBalancerPoolMemberOptions(loadBalancerID string, poolID string, id string) *DeleteLoadBalancerPoolMemberOptions {
	return &DeleteLoadBalancerPoolMemberOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		PoolID:         core.StringPtr(poolID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *DeleteLoadBalancerPoolMemberOptions) SetLoadBalancerID(loadBalancerID string) *DeleteLoadBalancerPoolMemberOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetPoolID : Allow user to set PoolID
func (options *DeleteLoadBalancerPoolMemberOptions) SetPoolID(poolID string) *DeleteLoadBalancerPoolMemberOptions {
	options.PoolID = core.StringPtr(poolID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteLoadBalancerPoolMemberOptions) SetID(id string) *DeleteLoadBalancerPoolMemberOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteLoadBalancerPoolMemberOptions) SetHeaders(param map[string]string) *DeleteLoadBalancerPoolMemberOptions {
	options.Headers = param
	return options
}

// DeleteLoadBalancerPoolOptions : The DeleteLoadBalancerPool options.
type DeleteLoadBalancerPoolOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The pool identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteLoadBalancerPoolOptions : Instantiate DeleteLoadBalancerPoolOptions
func (*VpcV1) NewDeleteLoadBalancerPoolOptions(loadBalancerID string, id string) *DeleteLoadBalancerPoolOptions {
	return &DeleteLoadBalancerPoolOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *DeleteLoadBalancerPoolOptions) SetLoadBalancerID(loadBalancerID string) *DeleteLoadBalancerPoolOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteLoadBalancerPoolOptions) SetID(id string) *DeleteLoadBalancerPoolOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteLoadBalancerPoolOptions) SetHeaders(param map[string]string) *DeleteLoadBalancerPoolOptions {
	options.Headers = param
	return options
}

// DeleteNetworkACLOptions : The DeleteNetworkACL options.
type DeleteNetworkACLOptions struct {
	// The network ACL identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteNetworkACLOptions : Instantiate DeleteNetworkACLOptions
func (*VpcV1) NewDeleteNetworkACLOptions(id string) *DeleteNetworkACLOptions {
	return &DeleteNetworkACLOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteNetworkACLOptions) SetID(id string) *DeleteNetworkACLOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteNetworkACLOptions) SetHeaders(param map[string]string) *DeleteNetworkACLOptions {
	options.Headers = param
	return options
}

// DeleteNetworkACLRuleOptions : The DeleteNetworkACLRule options.
type DeleteNetworkACLRuleOptions struct {
	// The network ACL identifier.
	NetworkACLID *string `validate:"required,ne="`

	// The rule identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteNetworkACLRuleOptions : Instantiate DeleteNetworkACLRuleOptions
func (*VpcV1) NewDeleteNetworkACLRuleOptions(networkACLID string, id string) *DeleteNetworkACLRuleOptions {
	return &DeleteNetworkACLRuleOptions{
		NetworkACLID: core.StringPtr(networkACLID),
		ID:           core.StringPtr(id),
	}
}

// SetNetworkACLID : Allow user to set NetworkACLID
func (options *DeleteNetworkACLRuleOptions) SetNetworkACLID(networkACLID string) *DeleteNetworkACLRuleOptions {
	options.NetworkACLID = core.StringPtr(networkACLID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteNetworkACLRuleOptions) SetID(id string) *DeleteNetworkACLRuleOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteNetworkACLRuleOptions) SetHeaders(param map[string]string) *DeleteNetworkACLRuleOptions {
	options.Headers = param
	return options
}

// DeletePublicGatewayOptions : The DeletePublicGateway options.
type DeletePublicGatewayOptions struct {
	// The public gateway identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeletePublicGatewayOptions : Instantiate DeletePublicGatewayOptions
func (*VpcV1) NewDeletePublicGatewayOptions(id string) *DeletePublicGatewayOptions {
	return &DeletePublicGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeletePublicGatewayOptions) SetID(id string) *DeletePublicGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeletePublicGatewayOptions) SetHeaders(param map[string]string) *DeletePublicGatewayOptions {
	options.Headers = param
	return options
}

// DeleteSecurityGroupOptions : The DeleteSecurityGroup options.
type DeleteSecurityGroupOptions struct {
	// The security group identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteSecurityGroupOptions : Instantiate DeleteSecurityGroupOptions
func (*VpcV1) NewDeleteSecurityGroupOptions(id string) *DeleteSecurityGroupOptions {
	return &DeleteSecurityGroupOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteSecurityGroupOptions) SetID(id string) *DeleteSecurityGroupOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteSecurityGroupOptions) SetHeaders(param map[string]string) *DeleteSecurityGroupOptions {
	options.Headers = param
	return options
}

// DeleteSecurityGroupRuleOptions : The DeleteSecurityGroupRule options.
type DeleteSecurityGroupRuleOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// The rule identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteSecurityGroupRuleOptions : Instantiate DeleteSecurityGroupRuleOptions
func (*VpcV1) NewDeleteSecurityGroupRuleOptions(securityGroupID string, id string) *DeleteSecurityGroupRuleOptions {
	return &DeleteSecurityGroupRuleOptions{
		SecurityGroupID: core.StringPtr(securityGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *DeleteSecurityGroupRuleOptions) SetSecurityGroupID(securityGroupID string) *DeleteSecurityGroupRuleOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteSecurityGroupRuleOptions) SetID(id string) *DeleteSecurityGroupRuleOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteSecurityGroupRuleOptions) SetHeaders(param map[string]string) *DeleteSecurityGroupRuleOptions {
	options.Headers = param
	return options
}

// DeleteSecurityGroupTargetBindingOptions : The DeleteSecurityGroupTargetBinding options.
type DeleteSecurityGroupTargetBindingOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// The security group target identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteSecurityGroupTargetBindingOptions : Instantiate DeleteSecurityGroupTargetBindingOptions
func (*VpcV1) NewDeleteSecurityGroupTargetBindingOptions(securityGroupID string, id string) *DeleteSecurityGroupTargetBindingOptions {
	return &DeleteSecurityGroupTargetBindingOptions{
		SecurityGroupID: core.StringPtr(securityGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *DeleteSecurityGroupTargetBindingOptions) SetSecurityGroupID(securityGroupID string) *DeleteSecurityGroupTargetBindingOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteSecurityGroupTargetBindingOptions) SetID(id string) *DeleteSecurityGroupTargetBindingOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteSecurityGroupTargetBindingOptions) SetHeaders(param map[string]string) *DeleteSecurityGroupTargetBindingOptions {
	options.Headers = param
	return options
}

// DeleteSnapshotOptions : The DeleteSnapshot options.
type DeleteSnapshotOptions struct {
	// The snapshot identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteSnapshotOptions : Instantiate DeleteSnapshotOptions
func (*VpcV1) NewDeleteSnapshotOptions(id string) *DeleteSnapshotOptions {
	return &DeleteSnapshotOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteSnapshotOptions) SetID(id string) *DeleteSnapshotOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteSnapshotOptions) SetHeaders(param map[string]string) *DeleteSnapshotOptions {
	options.Headers = param
	return options
}

// DeleteSnapshotsOptions : The DeleteSnapshots options.
type DeleteSnapshotsOptions struct {
	// Filters the collection to resources with the source volume with the specified identifier.
	SourceVolumeID *string `validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteSnapshotsOptions : Instantiate DeleteSnapshotsOptions
func (*VpcV1) NewDeleteSnapshotsOptions(sourceVolumeID string) *DeleteSnapshotsOptions {
	return &DeleteSnapshotsOptions{
		SourceVolumeID: core.StringPtr(sourceVolumeID),
	}
}

// SetSourceVolumeID : Allow user to set SourceVolumeID
func (options *DeleteSnapshotsOptions) SetSourceVolumeID(sourceVolumeID string) *DeleteSnapshotsOptions {
	options.SourceVolumeID = core.StringPtr(sourceVolumeID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteSnapshotsOptions) SetHeaders(param map[string]string) *DeleteSnapshotsOptions {
	options.Headers = param
	return options
}

// DeleteSubnetOptions : The DeleteSubnet options.
type DeleteSubnetOptions struct {
	// The subnet identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteSubnetOptions : Instantiate DeleteSubnetOptions
func (*VpcV1) NewDeleteSubnetOptions(id string) *DeleteSubnetOptions {
	return &DeleteSubnetOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteSubnetOptions) SetID(id string) *DeleteSubnetOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteSubnetOptions) SetHeaders(param map[string]string) *DeleteSubnetOptions {
	options.Headers = param
	return options
}

// DeleteSubnetReservedIPOptions : The DeleteSubnetReservedIP options.
type DeleteSubnetReservedIPOptions struct {
	// The subnet identifier.
	SubnetID *string `validate:"required,ne="`

	// The reserved IP identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteSubnetReservedIPOptions : Instantiate DeleteSubnetReservedIPOptions
func (*VpcV1) NewDeleteSubnetReservedIPOptions(subnetID string, id string) *DeleteSubnetReservedIPOptions {
	return &DeleteSubnetReservedIPOptions{
		SubnetID: core.StringPtr(subnetID),
		ID:       core.StringPtr(id),
	}
}

// SetSubnetID : Allow user to set SubnetID
func (options *DeleteSubnetReservedIPOptions) SetSubnetID(subnetID string) *DeleteSubnetReservedIPOptions {
	options.SubnetID = core.StringPtr(subnetID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteSubnetReservedIPOptions) SetID(id string) *DeleteSubnetReservedIPOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteSubnetReservedIPOptions) SetHeaders(param map[string]string) *DeleteSubnetReservedIPOptions {
	options.Headers = param
	return options
}

// DeleteVolumeOptions : The DeleteVolume options.
type DeleteVolumeOptions struct {
	// The volume identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteVolumeOptions : Instantiate DeleteVolumeOptions
func (*VpcV1) NewDeleteVolumeOptions(id string) *DeleteVolumeOptions {
	return &DeleteVolumeOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteVolumeOptions) SetID(id string) *DeleteVolumeOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteVolumeOptions) SetHeaders(param map[string]string) *DeleteVolumeOptions {
	options.Headers = param
	return options
}

// DeleteVPCAddressPrefixOptions : The DeleteVPCAddressPrefix options.
type DeleteVPCAddressPrefixOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The prefix identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteVPCAddressPrefixOptions : Instantiate DeleteVPCAddressPrefixOptions
func (*VpcV1) NewDeleteVPCAddressPrefixOptions(vpcID string, id string) *DeleteVPCAddressPrefixOptions {
	return &DeleteVPCAddressPrefixOptions{
		VPCID: core.StringPtr(vpcID),
		ID:    core.StringPtr(id),
	}
}

// SetVPCID : Allow user to set VPCID
func (options *DeleteVPCAddressPrefixOptions) SetVPCID(vpcID string) *DeleteVPCAddressPrefixOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteVPCAddressPrefixOptions) SetID(id string) *DeleteVPCAddressPrefixOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteVPCAddressPrefixOptions) SetHeaders(param map[string]string) *DeleteVPCAddressPrefixOptions {
	options.Headers = param
	return options
}

// DeleteVPCOptions : The DeleteVPC options.
type DeleteVPCOptions struct {
	// The VPC identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteVPCOptions : Instantiate DeleteVPCOptions
func (*VpcV1) NewDeleteVPCOptions(id string) *DeleteVPCOptions {
	return &DeleteVPCOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteVPCOptions) SetID(id string) *DeleteVPCOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteVPCOptions) SetHeaders(param map[string]string) *DeleteVPCOptions {
	options.Headers = param
	return options
}

// DeleteVPCRouteOptions : The DeleteVPCRoute options.
type DeleteVPCRouteOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The route identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteVPCRouteOptions : Instantiate DeleteVPCRouteOptions
func (*VpcV1) NewDeleteVPCRouteOptions(vpcID string, id string) *DeleteVPCRouteOptions {
	return &DeleteVPCRouteOptions{
		VPCID: core.StringPtr(vpcID),
		ID:    core.StringPtr(id),
	}
}

// SetVPCID : Allow user to set VPCID
func (options *DeleteVPCRouteOptions) SetVPCID(vpcID string) *DeleteVPCRouteOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteVPCRouteOptions) SetID(id string) *DeleteVPCRouteOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteVPCRouteOptions) SetHeaders(param map[string]string) *DeleteVPCRouteOptions {
	options.Headers = param
	return options
}

// DeleteVPCRoutingTableOptions : The DeleteVPCRoutingTable options.
type DeleteVPCRoutingTableOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The routing table identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteVPCRoutingTableOptions : Instantiate DeleteVPCRoutingTableOptions
func (*VpcV1) NewDeleteVPCRoutingTableOptions(vpcID string, id string) *DeleteVPCRoutingTableOptions {
	return &DeleteVPCRoutingTableOptions{
		VPCID: core.StringPtr(vpcID),
		ID:    core.StringPtr(id),
	}
}

// SetVPCID : Allow user to set VPCID
func (options *DeleteVPCRoutingTableOptions) SetVPCID(vpcID string) *DeleteVPCRoutingTableOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteVPCRoutingTableOptions) SetID(id string) *DeleteVPCRoutingTableOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteVPCRoutingTableOptions) SetHeaders(param map[string]string) *DeleteVPCRoutingTableOptions {
	options.Headers = param
	return options
}

// DeleteVPCRoutingTableRouteOptions : The DeleteVPCRoutingTableRoute options.
type DeleteVPCRoutingTableRouteOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The routing table identifier.
	RoutingTableID *string `validate:"required,ne="`

	// The VPC routing table route identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteVPCRoutingTableRouteOptions : Instantiate DeleteVPCRoutingTableRouteOptions
func (*VpcV1) NewDeleteVPCRoutingTableRouteOptions(vpcID string, routingTableID string, id string) *DeleteVPCRoutingTableRouteOptions {
	return &DeleteVPCRoutingTableRouteOptions{
		VPCID:          core.StringPtr(vpcID),
		RoutingTableID: core.StringPtr(routingTableID),
		ID:             core.StringPtr(id),
	}
}

// SetVPCID : Allow user to set VPCID
func (options *DeleteVPCRoutingTableRouteOptions) SetVPCID(vpcID string) *DeleteVPCRoutingTableRouteOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetRoutingTableID : Allow user to set RoutingTableID
func (options *DeleteVPCRoutingTableRouteOptions) SetRoutingTableID(routingTableID string) *DeleteVPCRoutingTableRouteOptions {
	options.RoutingTableID = core.StringPtr(routingTableID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteVPCRoutingTableRouteOptions) SetID(id string) *DeleteVPCRoutingTableRouteOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteVPCRoutingTableRouteOptions) SetHeaders(param map[string]string) *DeleteVPCRoutingTableRouteOptions {
	options.Headers = param
	return options
}

// DeleteVPNGatewayConnectionOptions : The DeleteVPNGatewayConnection options.
type DeleteVPNGatewayConnectionOptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteVPNGatewayConnectionOptions : Instantiate DeleteVPNGatewayConnectionOptions
func (*VpcV1) NewDeleteVPNGatewayConnectionOptions(vpnGatewayID string, id string) *DeleteVPNGatewayConnectionOptions {
	return &DeleteVPNGatewayConnectionOptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *DeleteVPNGatewayConnectionOptions) SetVPNGatewayID(vpnGatewayID string) *DeleteVPNGatewayConnectionOptions {
	options.VPNGatewayID = core.StringPtr(vpnGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteVPNGatewayConnectionOptions) SetID(id string) *DeleteVPNGatewayConnectionOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteVPNGatewayConnectionOptions) SetHeaders(param map[string]string) *DeleteVPNGatewayConnectionOptions {
	options.Headers = param
	return options
}

// DeleteVPNGatewayOptions : The DeleteVPNGateway options.
type DeleteVPNGatewayOptions struct {
	// The VPN gateway identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteVPNGatewayOptions : Instantiate DeleteVPNGatewayOptions
func (*VpcV1) NewDeleteVPNGatewayOptions(id string) *DeleteVPNGatewayOptions {
	return &DeleteVPNGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteVPNGatewayOptions) SetID(id string) *DeleteVPNGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteVPNGatewayOptions) SetHeaders(param map[string]string) *DeleteVPNGatewayOptions {
	options.Headers = param
	return options
}

// EncryptionKeyIdentity : Identifies an encryption key by a unique property.
// Models which "extend" this model:
// - EncryptionKeyIdentityByCRN
type EncryptionKeyIdentity struct {
	// The CRN of the [Key Protect Root
	// Key](https://cloud.ibm.com/docs/key-protect?topic=key-protect-getting-started-tutorial) or [Hyper Protect Crypto
	// Service Root Key](https://cloud.ibm.com/docs/hs-crypto?topic=hs-crypto-get-started) for this resource.
	CRN *string `json:"crn,omitempty"`
}

func (*EncryptionKeyIdentity) isaEncryptionKeyIdentity() bool {
	return true
}

type EncryptionKeyIdentityIntf interface {
	isaEncryptionKeyIdentity() bool
}

// UnmarshalEncryptionKeyIdentity unmarshals an instance of EncryptionKeyIdentity from the specified map of raw messages.
func UnmarshalEncryptionKeyIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(EncryptionKeyIdentity)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// EncryptionKeyReference : EncryptionKeyReference struct
type EncryptionKeyReference struct {
	// The CRN of the [Key Protect Root
	// Key](https://cloud.ibm.com/docs/key-protect?topic=key-protect-getting-started-tutorial) or [Hyper Protect Crypto
	// Service Root Key](https://cloud.ibm.com/docs/hs-crypto?topic=hs-crypto-get-started) for this resource.
	CRN *string `json:"crn" validate:"required"`
}

// UnmarshalEncryptionKeyReference unmarshals an instance of EncryptionKeyReference from the specified map of raw messages.
func UnmarshalEncryptionKeyReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(EncryptionKeyReference)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// EndpointGateway : EndpointGateway struct
type EndpointGateway struct {
	// The date and time that the endpoint gateway was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN for this endpoint gateway.
	CRN *string `json:"crn" validate:"required"`

	// The health of this resource.
	// - `ok`: Healthy
	// - `degraded`: Suffering from compromised performance, capacity, or connectivity
	// - `faulted`: Completely unreachable, inoperative, or otherwise entirely incapacitated
	// - `inapplicable`: The health state does not apply because of the current lifecycle state. A resource with a
	// lifecycle state of `failed` or `deleting` will have a health state of `inapplicable`. A `pending` resource may also
	// have this state.
	HealthState *string `json:"health_state" validate:"required"`

	// The URL for this endpoint gateway.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this endpoint gateway.
	ID *string `json:"id" validate:"required"`

	// The reserved IPs bound to this endpoint gateway.
	Ips []ReservedIPReference `json:"ips" validate:"required"`

	// The lifecycle state of the endpoint gateway.
	LifecycleState *string `json:"lifecycle_state" validate:"required"`

	// The unique user-defined name for this endpoint gateway.
	Name *string `json:"name" validate:"required"`

	// The resource group for this endpoint gateway.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The type of resource referenced.
	ResourceType *string `json:"resource_type" validate:"required"`

	// The fully qualified domain name for the target service.
	ServiceEndpoint *string `json:"service_endpoint,omitempty"`

	// The fully qualified domain names for the target service.
	ServiceEndpoints []string `json:"service_endpoints" validate:"required"`

	// The target for this endpoint gateway.
	Target EndpointGatewayTargetIntf `json:"target" validate:"required"`

	// The VPC this endpoint gateway is serving.
	VPC *VPCReference `json:"vpc" validate:"required"`
}

// Constants associated with the EndpointGateway.HealthState property.
// The health of this resource.
// - `ok`: Healthy
// - `degraded`: Suffering from compromised performance, capacity, or connectivity
// - `faulted`: Completely unreachable, inoperative, or otherwise entirely incapacitated
// - `inapplicable`: The health state does not apply because of the current lifecycle state. A resource with a lifecycle
// state of `failed` or `deleting` will have a health state of `inapplicable`. A `pending` resource may also have this
// state.
const (
	EndpointGatewayHealthStateDegradedConst     = "degraded"
	EndpointGatewayHealthStateFaultedConst      = "faulted"
	EndpointGatewayHealthStateInapplicableConst = "inapplicable"
	EndpointGatewayHealthStateOkConst           = "ok"
)

// Constants associated with the EndpointGateway.LifecycleState property.
// The lifecycle state of the endpoint gateway.
const (
	EndpointGatewayLifecycleStateDeletingConst  = "deleting"
	EndpointGatewayLifecycleStateFailedConst    = "failed"
	EndpointGatewayLifecycleStatePendingConst   = "pending"
	EndpointGatewayLifecycleStateStableConst    = "stable"
	EndpointGatewayLifecycleStateSuspendedConst = "suspended"
	EndpointGatewayLifecycleStateUpdatingConst  = "updating"
	EndpointGatewayLifecycleStateWaitingConst   = "waiting"
)

// Constants associated with the EndpointGateway.ResourceType property.
// The type of resource referenced.
const (
	EndpointGatewayResourceTypeEndpointGatewayConst = "endpoint_gateway"
)

// UnmarshalEndpointGateway unmarshals an instance of EndpointGateway from the specified map of raw messages.
func UnmarshalEndpointGateway(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(EndpointGateway)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "health_state", &obj.HealthState)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "ips", &obj.Ips, UnmarshalReservedIPReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "lifecycle_state", &obj.LifecycleState)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "service_endpoint", &obj.ServiceEndpoint)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "service_endpoints", &obj.ServiceEndpoints)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "target", &obj.Target, UnmarshalEndpointGatewayTarget)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vpc", &obj.VPC, UnmarshalVPCReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// EndpointGatewayCollection : EndpointGatewayCollection struct
type EndpointGatewayCollection struct {
	// Collection of endpoint gateways.
	EndpointGateways []EndpointGateway `json:"endpoint_gateways" validate:"required"`

	// A link to the first page of resources.
	First *EndpointGatewayCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *EndpointGatewayCollectionNext `json:"next,omitempty"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`
}

// UnmarshalEndpointGatewayCollection unmarshals an instance of EndpointGatewayCollection from the specified map of raw messages.
func UnmarshalEndpointGatewayCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(EndpointGatewayCollection)
	err = core.UnmarshalModel(m, "endpoint_gateways", &obj.EndpointGateways, UnmarshalEndpointGateway)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalEndpointGatewayCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalEndpointGatewayCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// EndpointGatewayCollectionFirst : A link to the first page of resources.
type EndpointGatewayCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalEndpointGatewayCollectionFirst unmarshals an instance of EndpointGatewayCollectionFirst from the specified map of raw messages.
func UnmarshalEndpointGatewayCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(EndpointGatewayCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// EndpointGatewayCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type EndpointGatewayCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalEndpointGatewayCollectionNext unmarshals an instance of EndpointGatewayCollectionNext from the specified map of raw messages.
func UnmarshalEndpointGatewayCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(EndpointGatewayCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// EndpointGatewayPatch : EndpointGatewayPatch struct
type EndpointGatewayPatch struct {
	// The user-defined name for this endpoint gateway. Names must be unique within the VPC this endpoint gateway is
	// serving.
	Name *string `json:"name,omitempty"`
}

// UnmarshalEndpointGatewayPatch unmarshals an instance of EndpointGatewayPatch from the specified map of raw messages.
func UnmarshalEndpointGatewayPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(EndpointGatewayPatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the EndpointGatewayPatch
func (endpointGatewayPatch *EndpointGatewayPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(endpointGatewayPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// EndpointGatewayReferenceDeleted : If present, this property indicates the referenced resource has been deleted and provides some supplementary
// information.
type EndpointGatewayReferenceDeleted struct {
	// Link to documentation about deleted resources.
	MoreInfo *string `json:"more_info" validate:"required"`
}

// UnmarshalEndpointGatewayReferenceDeleted unmarshals an instance of EndpointGatewayReferenceDeleted from the specified map of raw messages.
func UnmarshalEndpointGatewayReferenceDeleted(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(EndpointGatewayReferenceDeleted)
	err = core.UnmarshalPrimitive(m, "more_info", &obj.MoreInfo)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// EndpointGatewayReservedIP : A reserved IP to bind to the endpoint gateway. This can be specified using an existing reserved IP, or a prototype
// object for a new reserved IP. The reserved IP will be bound to the endpoint gateway to function as a virtual private
// endpoint for the service.
// Models which "extend" this model:
// - EndpointGatewayReservedIPReservedIPIdentity
// - EndpointGatewayReservedIPReservedIPPrototypeTargetContext
type EndpointGatewayReservedIP struct {
	// The unique identifier for this reserved IP.
	ID *string `json:"id,omitempty"`

	// The URL for this reserved IP.
	Href *string `json:"href,omitempty"`

	// If set to `true`, this reserved IP will be automatically deleted when the target is deleted or when the reserved IP
	// is unbound.
	AutoDelete *bool `json:"auto_delete,omitempty"`

	// The user-defined name for this reserved IP. If not specified, the name will be a hyphenated list of
	// randomly-selected words. Names must be unique within the subnet the reserved IP resides in. Names beginning with
	// `ibm-` are reserved for provider-owned resources.
	Name *string `json:"name,omitempty"`

	// The subnet in which to create this reserved IP.
	Subnet SubnetIdentityIntf `json:"subnet,omitempty"`
}

func (*EndpointGatewayReservedIP) isaEndpointGatewayReservedIP() bool {
	return true
}

type EndpointGatewayReservedIPIntf interface {
	isaEndpointGatewayReservedIP() bool
}

// UnmarshalEndpointGatewayReservedIP unmarshals an instance of EndpointGatewayReservedIP from the specified map of raw messages.
func UnmarshalEndpointGatewayReservedIP(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(EndpointGatewayReservedIP)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "auto_delete", &obj.AutoDelete)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "subnet", &obj.Subnet, UnmarshalSubnetIdentity)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// EndpointGatewayTarget : The target for this endpoint gateway.
// Models which "extend" this model:
// - EndpointGatewayTargetProviderCloudServiceReference
// - EndpointGatewayTargetProviderInfrastructureServiceReference
type EndpointGatewayTarget struct {
	// The CRN for this provider cloud service, or the CRN for the user's instance of a provider cloud service.
	CRN *string `json:"crn,omitempty"`

	// The type of target.
	ResourceType *string `json:"resource_type,omitempty"`

	// The name of a provider infrastructure service. Must be:
	// - `ibm-ntp-server`: An NTP (Network Time Protocol) server provided by IBM.
	Name *string `json:"name,omitempty"`
}

// Constants associated with the EndpointGatewayTarget.ResourceType property.
// The type of target.
const (
	EndpointGatewayTargetResourceTypeProviderCloudServiceConst = "provider_cloud_service"
)

func (*EndpointGatewayTarget) isaEndpointGatewayTarget() bool {
	return true
}

type EndpointGatewayTargetIntf interface {
	isaEndpointGatewayTarget() bool
}

// UnmarshalEndpointGatewayTarget unmarshals an instance of EndpointGatewayTarget from the specified map of raw messages.
func UnmarshalEndpointGatewayTarget(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(EndpointGatewayTarget)
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// EndpointGatewayTargetPrototype : The target for this endpoint gateway.
// Models which "extend" this model:
// - EndpointGatewayTargetPrototypeProviderCloudServiceIdentity
// - EndpointGatewayTargetPrototypeProviderInfrastructureServiceIdentity
type EndpointGatewayTargetPrototype struct {
	// The type of target for this endpoint gateway.
	ResourceType *string `json:"resource_type" validate:"required"`

	// The CRN for this provider cloud service, or the CRN for the user's instance of a provider cloud service.
	CRN *string `json:"crn,omitempty"`

	// The name of a provider infrastructure service. Must be:
	// - `ibm-ntp-server`: An NTP (Network Time Protocol) server provided by IBM.
	Name *string `json:"name,omitempty"`
}

// Constants associated with the EndpointGatewayTargetPrototype.ResourceType property.
// The type of target for this endpoint gateway.
const (
	EndpointGatewayTargetPrototypeResourceTypeProviderCloudServiceConst          = "provider_cloud_service"
	EndpointGatewayTargetPrototypeResourceTypeProviderInfrastructureServiceConst = "provider_infrastructure_service"
)

func (*EndpointGatewayTargetPrototype) isaEndpointGatewayTargetPrototype() bool {
	return true
}

type EndpointGatewayTargetPrototypeIntf interface {
	isaEndpointGatewayTargetPrototype() bool
}

// UnmarshalEndpointGatewayTargetPrototype unmarshals an instance of EndpointGatewayTargetPrototype from the specified map of raw messages.
func UnmarshalEndpointGatewayTargetPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	// Retrieve discriminator value to determine correct "subclass".
	var discValue string
	err = core.UnmarshalPrimitive(m, "resource_type", &discValue)
	if err != nil {
		err = fmt.Errorf("error unmarshalling discriminator property 'resource_type': %s", err.Error())
		return
	}
	if discValue == "" {
		err = fmt.Errorf("required discriminator property 'resource_type' not found in JSON object")
		return
	}
	if discValue == "provider_cloud_service" {
		err = core.UnmarshalModel(m, "", result, UnmarshalEndpointGatewayTargetPrototypeProviderCloudServiceIdentity)
	} else if discValue == "provider_infrastructure_service" {
		err = core.UnmarshalModel(m, "", result, UnmarshalEndpointGatewayTargetPrototypeProviderInfrastructureServiceIdentity)
	} else {
		err = fmt.Errorf("unrecognized value for discriminator property 'resource_type': %s", discValue)
	}
	return
}

// FloatingIP : FloatingIP struct
type FloatingIP struct {
	// The globally unique IP address.
	Address *string `json:"address" validate:"required"`

	// The date and time that the floating IP was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN for this floating IP.
	CRN *string `json:"crn" validate:"required"`

	// The URL for this floating IP.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this floating IP.
	ID *string `json:"id" validate:"required"`

	// The unique user-defined name for this floating IP.
	Name *string `json:"name" validate:"required"`

	// The resource group for this floating IP.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The status of the floating IP.
	Status *string `json:"status" validate:"required"`

	// The target of this floating IP.
	Target FloatingIPTargetIntf `json:"target,omitempty"`

	// The zone this floating IP resides in.
	Zone *ZoneReference `json:"zone" validate:"required"`
}

// Constants associated with the FloatingIP.Status property.
// The status of the floating IP.
const (
	FloatingIPStatusAvailableConst = "available"
	FloatingIPStatusDeletingConst  = "deleting"
	FloatingIPStatusFailedConst    = "failed"
	FloatingIPStatusPendingConst   = "pending"
)

// UnmarshalFloatingIP unmarshals an instance of FloatingIP from the specified map of raw messages.
func UnmarshalFloatingIP(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIP)
	err = core.UnmarshalPrimitive(m, "address", &obj.Address)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "target", &obj.Target, UnmarshalFloatingIPTarget)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPByTargetNetworkInterfaceIdentity : The network interface this floating IP is to be bound to.
// Models which "extend" this model:
// - FloatingIPByTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByID
// - FloatingIPByTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByHref
type FloatingIPByTargetNetworkInterfaceIdentity struct {
	// The unique identifier for this network interface.
	ID *string `json:"id,omitempty"`

	// The URL for this network interface.
	Href *string `json:"href,omitempty"`
}

func (*FloatingIPByTargetNetworkInterfaceIdentity) isaFloatingIPByTargetNetworkInterfaceIdentity() bool {
	return true
}

type FloatingIPByTargetNetworkInterfaceIdentityIntf interface {
	isaFloatingIPByTargetNetworkInterfaceIdentity() bool
}

// UnmarshalFloatingIPByTargetNetworkInterfaceIdentity unmarshals an instance of FloatingIPByTargetNetworkInterfaceIdentity from the specified map of raw messages.
func UnmarshalFloatingIPByTargetNetworkInterfaceIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPByTargetNetworkInterfaceIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPCollection : FloatingIPCollection struct
type FloatingIPCollection struct {
	// A link to the first page of resources.
	First *FloatingIPCollectionFirst `json:"first" validate:"required"`

	// Collection of floating IPs.
	FloatingIps []FloatingIP `json:"floating_ips" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *FloatingIPCollectionNext `json:"next,omitempty"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`
}

// UnmarshalFloatingIPCollection unmarshals an instance of FloatingIPCollection from the specified map of raw messages.
func UnmarshalFloatingIPCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalFloatingIPCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "floating_ips", &obj.FloatingIps, UnmarshalFloatingIP)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalFloatingIPCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPCollectionFirst : A link to the first page of resources.
type FloatingIPCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalFloatingIPCollectionFirst unmarshals an instance of FloatingIPCollectionFirst from the specified map of raw messages.
func UnmarshalFloatingIPCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type FloatingIPCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalFloatingIPCollectionNext unmarshals an instance of FloatingIPCollectionNext from the specified map of raw messages.
func UnmarshalFloatingIPCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPPatch : FloatingIPPatch struct
type FloatingIPPatch struct {
	// The unique user-defined name for this floating IP.
	Name *string `json:"name,omitempty"`

	// A new network interface to bind this floating IP to, replacing any existing binding.
	// For this request to succeed, the existing floating IP must not be required by another
	// resource, such as a public gateway.
	Target FloatingIPPatchTargetNetworkInterfaceIdentityIntf `json:"target,omitempty"`
}

// UnmarshalFloatingIPPatch unmarshals an instance of FloatingIPPatch from the specified map of raw messages.
func UnmarshalFloatingIPPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPPatch)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "target", &obj.Target, UnmarshalFloatingIPPatchTargetNetworkInterfaceIdentity)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the FloatingIPPatch
func (floatingIPPatch *FloatingIPPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(floatingIPPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// FloatingIPPatchTargetNetworkInterfaceIdentity : A new network interface to bind this floating IP to, replacing any existing binding. For this request to succeed, the
// existing floating IP must not be required by another resource, such as a public gateway.
// Models which "extend" this model:
// - FloatingIPPatchTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByID
// - FloatingIPPatchTargetNetworkInterfaceIdentityNetworkInterfaceIdentityByHref
type FloatingIPPatchTargetNetworkInterfaceIdentity struct {
	// The unique identifier for this network interface.
	ID *string `json:"id,omitempty"`

	// The URL for this network interface.
	Href *string `json:"href,omitempty"`
}

func (*FloatingIPPatchTargetNetworkInterfaceIdentity) isaFloatingIPPatchTargetNetworkInterfaceIdentity() bool {
	return true
}

type FloatingIPPatchTargetNetworkInterfaceIdentityIntf interface {
	isaFloatingIPPatchTargetNetworkInterfaceIdentity() bool
}

// UnmarshalFloatingIPPatchTargetNetworkInterfaceIdentity unmarshals an instance of FloatingIPPatchTargetNetworkInterfaceIdentity from the specified map of raw messages.
func UnmarshalFloatingIPPatchTargetNetworkInterfaceIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPPatchTargetNetworkInterfaceIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPPrototype : FloatingIPPrototype struct
// Models which "extend" this model:
// - FloatingIPPrototypeFloatingIPByZone
// - FloatingIPPrototypeFloatingIPByTarget
type FloatingIPPrototype struct {
	// The unique user-defined name for this floating IP. If unspecified, the name will be a hyphenated list of
	// randomly-selected words.
	Name *string `json:"name,omitempty"`

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup ResourceGroupIdentityIntf `json:"resource_group,omitempty"`

	// The zone this floating IP will reside in.
	Zone ZoneIdentityIntf `json:"zone,omitempty"`

	// The network interface this floating IP is to be bound to.
	Target FloatingIPByTargetNetworkInterfaceIdentityIntf `json:"target,omitempty"`
}

func (*FloatingIPPrototype) isaFloatingIPPrototype() bool {
	return true
}

type FloatingIPPrototypeIntf interface {
	isaFloatingIPPrototype() bool
}

// UnmarshalFloatingIPPrototype unmarshals an instance of FloatingIPPrototype from the specified map of raw messages.
func UnmarshalFloatingIPPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPPrototype)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "zone", &obj.Zone, UnmarshalZoneIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "target", &obj.Target, UnmarshalFloatingIPByTargetNetworkInterfaceIdentity)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPReference : FloatingIPReference struct
type FloatingIPReference struct {
	// The globally unique IP address.
	Address *string `json:"address" validate:"required"`

	// The CRN for this floating IP.
	CRN *string `json:"crn" validate:"required"`

	// If present, this property indicates the referenced resource has been deleted and provides
	// some supplementary information.
	Deleted *FloatingIPReferenceDeleted `json:"deleted,omitempty"`

	// The URL for this floating IP.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this floating IP.
	ID *string `json:"id" validate:"required"`

	// The unique user-defined name for this floating IP.
	Name *string `json:"name" validate:"required"`
}

// UnmarshalFloatingIPReference unmarshals an instance of FloatingIPReference from the specified map of raw messages.
func UnmarshalFloatingIPReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPReference)
	err = core.UnmarshalPrimitive(m, "address", &obj.Address)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "deleted", &obj.Deleted, UnmarshalFloatingIPReferenceDeleted)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPReferenceDeleted : If present, this property indicates the referenced resource has been deleted and provides some supplementary
// information.
type FloatingIPReferenceDeleted struct {
	// Link to documentation about deleted resources.
	MoreInfo *string `json:"more_info" validate:"required"`
}

// UnmarshalFloatingIPReferenceDeleted unmarshals an instance of FloatingIPReferenceDeleted from the specified map of raw messages.
func UnmarshalFloatingIPReferenceDeleted(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPReferenceDeleted)
	err = core.UnmarshalPrimitive(m, "more_info", &obj.MoreInfo)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPTarget : The target of this floating IP.
// Models which "extend" this model:
// - FloatingIPTargetNetworkInterfaceReference
// - FloatingIPTargetPublicGatewayReference
type FloatingIPTarget struct {
	// If present, this property indicates the referenced resource has been deleted and provides
	// some supplementary information.
	Deleted *NetworkInterfaceReferenceDeleted `json:"deleted,omitempty"`

	// The URL for this network interface.
	Href *string `json:"href,omitempty"`

	// The unique identifier for this network interface.
	ID *string `json:"id,omitempty"`

	// The user-defined name for this network interface.
	Name *string `json:"name,omitempty"`

	// The primary IPv4 address.
	PrimaryIpv4Address *string `json:"primary_ipv4_address,omitempty"`

	// The resource type.
	ResourceType *string `json:"resource_type,omitempty"`

	// The CRN for this public gateway.
	CRN *string `json:"crn,omitempty"`
}

// Constants associated with the FloatingIPTarget.ResourceType property.
// The resource type.
const (
	FloatingIPTargetResourceTypeNetworkInterfaceConst = "network_interface"
)

func (*FloatingIPTarget) isaFloatingIPTarget() bool {
	return true
}

type FloatingIPTargetIntf interface {
	isaFloatingIPTarget() bool
}

// UnmarshalFloatingIPTarget unmarshals an instance of FloatingIPTarget from the specified map of raw messages.
func UnmarshalFloatingIPTarget(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPTarget)
	err = core.UnmarshalModel(m, "deleted", &obj.Deleted, UnmarshalNetworkInterfaceReferenceDeleted)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "primary_ipv4_address", &obj.PrimaryIpv4Address)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FloatingIPUnpaginatedCollection : FloatingIPUnpaginatedCollection struct
type FloatingIPUnpaginatedCollection struct {
	// Collection of floating IPs.
	FloatingIps []FloatingIP `json:"floating_ips" validate:"required"`
}

// UnmarshalFloatingIPUnpaginatedCollection unmarshals an instance of FloatingIPUnpaginatedCollection from the specified map of raw messages.
func UnmarshalFloatingIPUnpaginatedCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FloatingIPUnpaginatedCollection)
	err = core.UnmarshalModel(m, "floating_ips", &obj.FloatingIps, UnmarshalFloatingIP)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FlowLogCollector : FlowLogCollector struct
type FlowLogCollector struct {
	// Indicates whether this collector is active.
	Active *bool `json:"active" validate:"required"`

	// If set to `true`, this flow log collector will be automatically deleted when the target is deleted.
	AutoDelete *bool `json:"auto_delete" validate:"required"`

	// The date and time that the flow log collector was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN for this flow log collector.
	CRN *string `json:"crn" validate:"required"`

	// The URL for this flow log collector.
	Href *string `json:"href" validate:"required"`

	// The unique identifier for this flow log collector.
	ID *string `json:"id" validate:"required"`

	// The lifecycle state of the flow log collector.
	LifecycleState *string `json:"lifecycle_state" validate:"required"`

	// The unique user-defined name for this flow log collector.
	Name *string `json:"name" validate:"required"`

	// The resource group for this flow log collector.
	ResourceGroup *ResourceGroupReference `json:"resource_group" validate:"required"`

	// The Cloud Object Storage bucket where the collected flows are logged.
	StorageBucket *CloudObjectStorageBucketReference `json:"storage_bucket" validate:"required"`

	// The target this collector is collecting flow logs for. If the target is an instance,
	// subnet, or VPC, flow logs will not be collected for any network interfaces within the
	// target that are themselves the target of a more specific flow log collector.
	Target FlowLogCollectorTargetIntf `json:"target" validate:"required"`

	// The VPC this flow log collector is associated with.
	VPC *VPCReference `json:"vpc" validate:"required"`
}

// Constants associated with the FlowLogCollector.LifecycleState property.
// The lifecycle state of the flow log collector.
const (
	FlowLogCollectorLifecycleStateDeletingConst  = "deleting"
	FlowLogCollectorLifecycleStateFailedConst    = "failed"
	FlowLogCollectorLifecycleStatePendingConst   = "pending"
	FlowLogCollectorLifecycleStateStableConst    = "stable"
	FlowLogCollectorLifecycleStateSuspendedConst = "suspended"
	FlowLogCollectorLifecycleStateUpdatingConst  = "updating"
	FlowLogCollectorLifecycleStateWaitingConst   = "waiting"
)

// UnmarshalFlowLogCollector unmarshals an instance of FlowLogCollector from the specified map of raw messages.
func UnmarshalFlowLogCollector(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FlowLogCollector)
	err = core.UnmarshalPrimitive(m, "active", &obj.Active)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "auto_delete", &obj.AutoDelete)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "lifecycle_state", &obj.LifecycleState)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "storage_bucket", &obj.StorageBucket, UnmarshalCloudObjectStorageBucketReference)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "target", &obj.Target, UnmarshalFlowLogCollectorTarget)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "vpc", &obj.VPC, UnmarshalVPCReference)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FlowLogCollectorCollection : FlowLogCollectorCollection struct
type FlowLogCollectorCollection struct {
	// A link to the first page of resources.
	First *FlowLogCollectorCollectionFirst `json:"first" validate:"required"`

	// Collection of flow log collectors.
	FlowLogCollectors []FlowLogCollector `json:"flow_log_collectors" validate:"required"`

	// The maximum number of resources that can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A link to the next page of resources. This property is present for all pages
	// except the last page.
	Next *FlowLogCollectorCollectionNext `json:"next,omitempty"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`
}

// UnmarshalFlowLogCollectorCollection unmarshals an instance of FlowLogCollectorCollection from the specified map of raw messages.
func UnmarshalFlowLogCollectorCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FlowLogCollectorCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalFlowLogCollectorCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "flow_log_collectors", &obj.FlowLogCollectors, UnmarshalFlowLogCollector)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalFlowLogCollectorCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FlowLogCollectorCollectionFirst : A link to the first page of resources.
type FlowLogCollectorCollectionFirst struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalFlowLogCollectorCollectionFirst unmarshals an instance of FlowLogCollectorCollectionFirst from the specified map of raw messages.
func UnmarshalFlowLogCollectorCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FlowLogCollectorCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FlowLogCollectorCollectionNext : A link to the next page of resources. This property is present for all pages except the last page.
type FlowLogCollectorCollectionNext struct {
	// The URL for a page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalFlowLogCollectorCollectionNext unmarshals an instance of FlowLogCollectorCollectionNext from the specified map of raw messages.
func UnmarshalFlowLogCollectorCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FlowLogCollectorCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FlowLogCollectorPatch : FlowLogCollectorPatch struct
type FlowLogCollectorPatch struct {
	// Indicates whether this collector is active. Updating to false deactivates the collector and updating to true
	// activates the collector.
	Active *bool `json:"active,omitempty"`

	// The unique user-defined name for this flow log collector.
	Name *string `json:"name,omitempty"`
}

// UnmarshalFlowLogCollectorPatch unmarshals an instance of FlowLogCollectorPatch from the specified map of raw messages.
func UnmarshalFlowLogCollectorPatch(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FlowLogCollectorPatch)
	err = core.UnmarshalPrimitive(m, "active", &obj.Active)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// AsPatch returns a generic map representation of the FlowLogCollectorPatch
func (flowLogCollectorPatch *FlowLogCollectorPatch) AsPatch() (patch map[string]interface{}, err error) {
	var jsonData []byte
	jsonData, err = json.Marshal(flowLogCollectorPatch)
	if err == nil {
		err = json.Unmarshal(jsonData, &patch)
	}
	return
}

// FlowLogCollectorTarget : The target this collector is collecting flow logs for. If the target is an instance, subnet, or VPC, flow logs will
// not be collected for any network interfaces within the target that are themselves the target of a more specific flow
// log collector.
// Models which "extend" this model:
// - FlowLogCollectorTargetNetworkInterfaceReferenceTargetContext
// - FlowLogCollectorTargetInstanceReference
// - FlowLogCollectorTargetSubnetReference
// - FlowLogCollectorTargetVPCReference
type FlowLogCollectorTarget struct {
	// If present, this property indicates the referenced resource has been deleted and provides
	// some supplementary information.
	Deleted *NetworkInterfaceReferenceTargetContextDeleted `json:"deleted,omitempty"`

	// The URL for this network interface.
	Href *string `json:"href,omitempty"`

	// The unique identifier for this network interface.
	ID *string `json:"id,omitempty"`

	// The user-defined name for this network interface.
	Name *string `json:"name,omitempty"`

	// The resource type.
	ResourceType *string `json:"resource_type,omitempty"`

	// The CRN for this virtual server instance.
	CRN *string `json:"crn,omitempty"`
}

// Constants associated with the FlowLogCollectorTarget.ResourceType property.
// The resource type.
const (
	FlowLogCollectorTargetResourceTypeNetworkInterfaceConst = "network_interface"
)

func (*FlowLogCollectorTarget) isaFlowLogCollectorTarget() bool {
	return true
}

type FlowLogCollectorTargetIntf interface {
	isaFlowLogCollectorTarget() bool
}

// UnmarshalFlowLogCollectorTarget unmarshals an instance of FlowLogCollectorTarget from the specified map of raw messages.
func UnmarshalFlowLogCollectorTarget(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FlowLogCollectorTarget)
	err = core.UnmarshalModel(m, "deleted", &obj.Deleted, UnmarshalNetworkInterfaceReferenceTargetContextDeleted)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "resource_type", &obj.ResourceType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// FlowLogCollectorTargetPrototype : The target this collector will collect flow logs for. If the target is an instance, subnet, or VPC, flow logs will
// not be collected for any network interfaces within the target that are themselves the target of a more specific flow
// log collector.
// Models which "extend" this model:
// - FlowLogCollectorTargetPrototypeNetworkInterfaceIdentity
// - FlowLogCollectorTargetPrototypeInstanceIdentity
// - FlowLogCollectorTargetPrototypeSubnetIdentity
// - FlowLogCollectorTargetPrototypeVPCIdentity
type FlowLogCollectorTargetPrototype struct {
	// The unique identifier for this network interface.
	ID *string `json:"id,omitempty"`

	// The URL for this network interface.
	Href *string `json:"href,omitempty"`

	// The CRN for this virtual server instance.
	CRN *string `json:"crn,omitempty"`
}

func (*FlowLogCollectorTargetPrototype) isaFlowLogCollectorTargetPrototype() bool {
	return true
}

type FlowLogCollectorTargetPrototypeIntf interface {
	isaFlowLogCollectorTargetPrototype() bool
}

// UnmarshalFlowLogCollectorTargetPrototype unmarshals an instance of FlowLogCollectorTargetPrototype from the specified map of raw messages.
func UnmarshalFlowLogCollectorTargetPrototype(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(FlowLogCollectorTargetPrototype)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.CRN)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GetDedicatedHostDiskOptions : The GetDedicatedHostDisk options.
type GetDedicatedHostDiskOptions struct {
	// The dedicated host identifier.
	DedicatedHostID *string `validate:"required,ne="`

	// The dedicated host disk identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetDedicatedHostDiskOptions : Instantiate GetDedicatedHostDiskOptions
func (*VpcV1) NewGetDedicatedHostDiskOptions(dedicatedHostID string, id string) *GetDedicatedHostDiskOptions {
	return &GetDedicatedHostDiskOptions{
		DedicatedHostID: core.StringPtr(dedicatedHostID),
		ID:              core.StringPtr(id),
	}
}

// SetDedicatedHostID : Allow user to set DedicatedHostID
func (options *GetDedicatedHostDiskOptions) SetDedicatedHostID(dedicatedHostID string) *GetDedicatedHostDiskOptions {
	options.DedicatedHostID = core.StringPtr(dedicatedHostID)
	return options
}

// SetID : Allow user to set ID
func (options *GetDedicatedHostDiskOptions) SetID(id string) *GetDedicatedHostDiskOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetDedicatedHostDiskOptions) SetHeaders(param map[string]string) *GetDedicatedHostDiskOptions {
	options.Headers = param
	return options
}

// GetDedicatedHostGroupOptions : The GetDedicatedHostGroup options.
type GetDedicatedHostGroupOptions struct {
	// The dedicated host group identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetDedicatedHostGroupOptions : Instantiate GetDedicatedHostGroupOptions
func (*VpcV1) NewGetDedicatedHostGroupOptions(id string) *GetDedicatedHostGroupOptions {
	return &GetDedicatedHostGroupOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetDedicatedHostGroupOptions) SetID(id string) *GetDedicatedHostGroupOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetDedicatedHostGroupOptions) SetHeaders(param map[string]string) *GetDedicatedHostGroupOptions {
	options.Headers = param
	return options
}

// GetDedicatedHostOptions : The GetDedicatedHost options.
type GetDedicatedHostOptions struct {
	// The dedicated host identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetDedicatedHostOptions : Instantiate GetDedicatedHostOptions
func (*VpcV1) NewGetDedicatedHostOptions(id string) *GetDedicatedHostOptions {
	return &GetDedicatedHostOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetDedicatedHostOptions) SetID(id string) *GetDedicatedHostOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetDedicatedHostOptions) SetHeaders(param map[string]string) *GetDedicatedHostOptions {
	options.Headers = param
	return options
}

// GetDedicatedHostProfileOptions : The GetDedicatedHostProfile options.
type GetDedicatedHostProfileOptions struct {
	// The dedicated host profile name.
	Name *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetDedicatedHostProfileOptions : Instantiate GetDedicatedHostProfileOptions
func (*VpcV1) NewGetDedicatedHostProfileOptions(name string) *GetDedicatedHostProfileOptions {
	return &GetDedicatedHostProfileOptions{
		Name: core.StringPtr(name),
	}
}

// SetName : Allow user to set Name
func (options *GetDedicatedHostProfileOptions) SetName(name string) *GetDedicatedHostProfileOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetDedicatedHostProfileOptions) SetHeaders(param map[string]string) *GetDedicatedHostProfileOptions {
	options.Headers = param
	return options
}

// GetEndpointGatewayIPOptions : The GetEndpointGatewayIP options.
type GetEndpointGatewayIPOptions struct {
	// The endpoint gateway identifier.
	EndpointGatewayID *string `validate:"required,ne="`

	// The reserved IP identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetEndpointGatewayIPOptions : Instantiate GetEndpointGatewayIPOptions
func (*VpcV1) NewGetEndpointGatewayIPOptions(endpointGatewayID string, id string) *GetEndpointGatewayIPOptions {
	return &GetEndpointGatewayIPOptions{
		EndpointGatewayID: core.StringPtr(endpointGatewayID),
		ID:                core.StringPtr(id),
	}
}

// SetEndpointGatewayID : Allow user to set EndpointGatewayID
func (options *GetEndpointGatewayIPOptions) SetEndpointGatewayID(endpointGatewayID string) *GetEndpointGatewayIPOptions {
	options.EndpointGatewayID = core.StringPtr(endpointGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *GetEndpointGatewayIPOptions) SetID(id string) *GetEndpointGatewayIPOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetEndpointGatewayIPOptions) SetHeaders(param map[string]string) *GetEndpointGatewayIPOptions {
	options.Headers = param
	return options
}

// GetEndpointGatewayOptions : The GetEndpointGateway options.
type GetEndpointGatewayOptions struct {
	// The endpoint gateway identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetEndpointGatewayOptions : Instantiate GetEndpointGatewayOptions
func (*VpcV1) NewGetEndpointGatewayOptions(id string) *GetEndpointGatewayOptions {
	return &GetEndpointGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetEndpointGatewayOptions) SetID(id string) *GetEndpointGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetEndpointGatewayOptions) SetHeaders(param map[string]string) *GetEndpointGatewayOptions {
	options.Headers = param
	return options
}

// GetFloatingIPOptions : The GetFloatingIP options.
type GetFloatingIPOptions struct {
	// The floating IP identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetFloatingIPOptions : Instantiate GetFloatingIPOptions
func (*VpcV1) NewGetFloatingIPOptions(id string) *GetFloatingIPOptions {
	return &GetFloatingIPOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetFloatingIPOptions) SetID(id string) *GetFloatingIPOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetFloatingIPOptions) SetHeaders(param map[string]string) *GetFloatingIPOptions {
	options.Headers = param
	return options
}

// GetFlowLogCollectorOptions : The GetFlowLogCollector options.
type GetFlowLogCollectorOptions struct {
	// The flow log collector identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetFlowLogCollectorOptions : Instantiate GetFlowLogCollectorOptions
func (*VpcV1) NewGetFlowLogCollectorOptions(id string) *GetFlowLogCollectorOptions {
	return &GetFlowLogCollectorOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetFlowLogCollectorOptions) SetID(id string) *GetFlowLogCollectorOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetFlowLogCollectorOptions) SetHeaders(param map[string]string) *GetFlowLogCollectorOptions {
	options.Headers = param
	return options
}

// GetIkePolicyOptions : The GetIkePolicy options.
type GetIkePolicyOptions struct {
	// The IKE policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetIkePolicyOptions : Instantiate GetIkePolicyOptions
func (*VpcV1) NewGetIkePolicyOptions(id string) *GetIkePolicyOptions {
	return &GetIkePolicyOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetIkePolicyOptions) SetID(id string) *GetIkePolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetIkePolicyOptions) SetHeaders(param map[string]string) *GetIkePolicyOptions {
	options.Headers = param
	return options
}

// GetImageOptions : The GetImage options.
type GetImageOptions struct {
	// The image identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetImageOptions : Instantiate GetImageOptions
func (*VpcV1) NewGetImageOptions(id string) *GetImageOptions {
	return &GetImageOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetImageOptions) SetID(id string) *GetImageOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetImageOptions) SetHeaders(param map[string]string) *GetImageOptions {
	options.Headers = param
	return options
}

// GetInstanceDiskOptions : The GetInstanceDisk options.
type GetInstanceDiskOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The instance disk identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceDiskOptions : Instantiate GetInstanceDiskOptions
func (*VpcV1) NewGetInstanceDiskOptions(instanceID string, id string) *GetInstanceDiskOptions {
	return &GetInstanceDiskOptions{
		InstanceID: core.StringPtr(instanceID),
		ID:         core.StringPtr(id),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *GetInstanceDiskOptions) SetInstanceID(instanceID string) *GetInstanceDiskOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetID : Allow user to set ID
func (options *GetInstanceDiskOptions) SetID(id string) *GetInstanceDiskOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceDiskOptions) SetHeaders(param map[string]string) *GetInstanceDiskOptions {
	options.Headers = param
	return options
}

// GetInstanceGroupManagerActionOptions : The GetInstanceGroupManagerAction options.
type GetInstanceGroupManagerActionOptions struct {
	// The instance group identifier.
	InstanceGroupID *string `validate:"required,ne="`

	// The instance group manager identifier.
	InstanceGroupManagerID *string `validate:"required,ne="`

	// The instance group manager action identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceGroupManagerActionOptions : Instantiate GetInstanceGroupManagerActionOptions
func (*VpcV1) NewGetInstanceGroupManagerActionOptions(instanceGroupID string, instanceGroupManagerID string, id string) *GetInstanceGroupManagerActionOptions {
	return &GetInstanceGroupManagerActionOptions{
		InstanceGroupID:        core.StringPtr(instanceGroupID),
		InstanceGroupManagerID: core.StringPtr(instanceGroupManagerID),
		ID:                     core.StringPtr(id),
	}
}

// SetInstanceGroupID : Allow user to set InstanceGroupID
func (options *GetInstanceGroupManagerActionOptions) SetInstanceGroupID(instanceGroupID string) *GetInstanceGroupManagerActionOptions {
	options.InstanceGroupID = core.StringPtr(instanceGroupID)
	return options
}

// SetInstanceGroupManagerID : Allow user to set InstanceGroupManagerID
func (options *GetInstanceGroupManagerActionOptions) SetInstanceGroupManagerID(instanceGroupManagerID string) *GetInstanceGroupManagerActionOptions {
	options.InstanceGroupManagerID = core.StringPtr(instanceGroupManagerID)
	return options
}

// SetID : Allow user to set ID
func (options *GetInstanceGroupManagerActionOptions) SetID(id string) *GetInstanceGroupManagerActionOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceGroupManagerActionOptions) SetHeaders(param map[string]string) *GetInstanceGroupManagerActionOptions {
	options.Headers = param
	return options
}

// GetInstanceGroupManagerOptions : The GetInstanceGroupManager options.
type GetInstanceGroupManagerOptions struct {
	// The instance group identifier.
	InstanceGroupID *string `validate:"required,ne="`

	// The instance group manager identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceGroupManagerOptions : Instantiate GetInstanceGroupManagerOptions
func (*VpcV1) NewGetInstanceGroupManagerOptions(instanceGroupID string, id string) *GetInstanceGroupManagerOptions {
	return &GetInstanceGroupManagerOptions{
		InstanceGroupID: core.StringPtr(instanceGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetInstanceGroupID : Allow user to set InstanceGroupID
func (options *GetInstanceGroupManagerOptions) SetInstanceGroupID(instanceGroupID string) *GetInstanceGroupManagerOptions {
	options.InstanceGroupID = core.StringPtr(instanceGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *GetInstanceGroupManagerOptions) SetID(id string) *GetInstanceGroupManagerOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceGroupManagerOptions) SetHeaders(param map[string]string) *GetInstanceGroupManagerOptions {
	options.Headers = param
	return options
}

// GetInstanceGroupManagerPolicyOptions : The GetInstanceGroupManagerPolicy options.
type GetInstanceGroupManagerPolicyOptions struct {
	// The instance group identifier.
	InstanceGroupID *string `validate:"required,ne="`

	// The instance group manager identifier.
	InstanceGroupManagerID *string `validate:"required,ne="`

	// The instance group manager policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceGroupManagerPolicyOptions : Instantiate GetInstanceGroupManagerPolicyOptions
func (*VpcV1) NewGetInstanceGroupManagerPolicyOptions(instanceGroupID string, instanceGroupManagerID string, id string) *GetInstanceGroupManagerPolicyOptions {
	return &GetInstanceGroupManagerPolicyOptions{
		InstanceGroupID:        core.StringPtr(instanceGroupID),
		InstanceGroupManagerID: core.StringPtr(instanceGroupManagerID),
		ID:                     core.StringPtr(id),
	}
}

// SetInstanceGroupID : Allow user to set InstanceGroupID
func (options *GetInstanceGroupManagerPolicyOptions) SetInstanceGroupID(instanceGroupID string) *GetInstanceGroupManagerPolicyOptions {
	options.InstanceGroupID = core.StringPtr(instanceGroupID)
	return options
}

// SetInstanceGroupManagerID : Allow user to set InstanceGroupManagerID
func (options *GetInstanceGroupManagerPolicyOptions) SetInstanceGroupManagerID(instanceGroupManagerID string) *GetInstanceGroupManagerPolicyOptions {
	options.InstanceGroupManagerID = core.StringPtr(instanceGroupManagerID)
	return options
}

// SetID : Allow user to set ID
func (options *GetInstanceGroupManagerPolicyOptions) SetID(id string) *GetInstanceGroupManagerPolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceGroupManagerPolicyOptions) SetHeaders(param map[string]string) *GetInstanceGroupManagerPolicyOptions {
	options.Headers = param
	return options
}

// GetInstanceGroupMembershipOptions : The GetInstanceGroupMembership options.
type GetInstanceGroupMembershipOptions struct {
	// The instance group identifier.
	InstanceGroupID *string `validate:"required,ne="`

	// The instance group membership identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceGroupMembershipOptions : Instantiate GetInstanceGroupMembershipOptions
func (*VpcV1) NewGetInstanceGroupMembershipOptions(instanceGroupID string, id string) *GetInstanceGroupMembershipOptions {
	return &GetInstanceGroupMembershipOptions{
		InstanceGroupID: core.StringPtr(instanceGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetInstanceGroupID : Allow user to set InstanceGroupID
func (options *GetInstanceGroupMembershipOptions) SetInstanceGroupID(instanceGroupID string) *GetInstanceGroupMembershipOptions {
	options.InstanceGroupID = core.StringPtr(instanceGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *GetInstanceGroupMembershipOptions) SetID(id string) *GetInstanceGroupMembershipOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceGroupMembershipOptions) SetHeaders(param map[string]string) *GetInstanceGroupMembershipOptions {
	options.Headers = param
	return options
}

// GetInstanceGroupOptions : The GetInstanceGroup options.
type GetInstanceGroupOptions struct {
	// The instance group identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceGroupOptions : Instantiate GetInstanceGroupOptions
func (*VpcV1) NewGetInstanceGroupOptions(id string) *GetInstanceGroupOptions {
	return &GetInstanceGroupOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetInstanceGroupOptions) SetID(id string) *GetInstanceGroupOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceGroupOptions) SetHeaders(param map[string]string) *GetInstanceGroupOptions {
	options.Headers = param
	return options
}

// GetInstanceInitializationOptions : The GetInstanceInitialization options.
type GetInstanceInitializationOptions struct {
	// The instance identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceInitializationOptions : Instantiate GetInstanceInitializationOptions
func (*VpcV1) NewGetInstanceInitializationOptions(id string) *GetInstanceInitializationOptions {
	return &GetInstanceInitializationOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetInstanceInitializationOptions) SetID(id string) *GetInstanceInitializationOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceInitializationOptions) SetHeaders(param map[string]string) *GetInstanceInitializationOptions {
	options.Headers = param
	return options
}

// GetInstanceNetworkInterfaceFloatingIPOptions : The GetInstanceNetworkInterfaceFloatingIP options.
type GetInstanceNetworkInterfaceFloatingIPOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The network interface identifier.
	NetworkInterfaceID *string `validate:"required,ne="`

	// The floating IP identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceNetworkInterfaceFloatingIPOptions : Instantiate GetInstanceNetworkInterfaceFloatingIPOptions
func (*VpcV1) NewGetInstanceNetworkInterfaceFloatingIPOptions(instanceID string, networkInterfaceID string, id string) *GetInstanceNetworkInterfaceFloatingIPOptions {
	return &GetInstanceNetworkInterfaceFloatingIPOptions{
		InstanceID:         core.StringPtr(instanceID),
		NetworkInterfaceID: core.StringPtr(networkInterfaceID),
		ID:                 core.StringPtr(id),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *GetInstanceNetworkInterfaceFloatingIPOptions) SetInstanceID(instanceID string) *GetInstanceNetworkInterfaceFloatingIPOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetNetworkInterfaceID : Allow user to set NetworkInterfaceID
func (options *GetInstanceNetworkInterfaceFloatingIPOptions) SetNetworkInterfaceID(networkInterfaceID string) *GetInstanceNetworkInterfaceFloatingIPOptions {
	options.NetworkInterfaceID = core.StringPtr(networkInterfaceID)
	return options
}

// SetID : Allow user to set ID
func (options *GetInstanceNetworkInterfaceFloatingIPOptions) SetID(id string) *GetInstanceNetworkInterfaceFloatingIPOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceNetworkInterfaceFloatingIPOptions) SetHeaders(param map[string]string) *GetInstanceNetworkInterfaceFloatingIPOptions {
	options.Headers = param
	return options
}

// GetInstanceNetworkInterfaceOptions : The GetInstanceNetworkInterface options.
type GetInstanceNetworkInterfaceOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The network interface identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceNetworkInterfaceOptions : Instantiate GetInstanceNetworkInterfaceOptions
func (*VpcV1) NewGetInstanceNetworkInterfaceOptions(instanceID string, id string) *GetInstanceNetworkInterfaceOptions {
	return &GetInstanceNetworkInterfaceOptions{
		InstanceID: core.StringPtr(instanceID),
		ID:         core.StringPtr(id),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *GetInstanceNetworkInterfaceOptions) SetInstanceID(instanceID string) *GetInstanceNetworkInterfaceOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetID : Allow user to set ID
func (options *GetInstanceNetworkInterfaceOptions) SetID(id string) *GetInstanceNetworkInterfaceOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceNetworkInterfaceOptions) SetHeaders(param map[string]string) *GetInstanceNetworkInterfaceOptions {
	options.Headers = param
	return options
}

// GetInstanceOptions : The GetInstance options.
type GetInstanceOptions struct {
	// The instance identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceOptions : Instantiate GetInstanceOptions
func (*VpcV1) NewGetInstanceOptions(id string) *GetInstanceOptions {
	return &GetInstanceOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetInstanceOptions) SetID(id string) *GetInstanceOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceOptions) SetHeaders(param map[string]string) *GetInstanceOptions {
	options.Headers = param
	return options
}

// GetInstanceProfileOptions : The GetInstanceProfile options.
type GetInstanceProfileOptions struct {
	// The instance profile name.
	Name *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceProfileOptions : Instantiate GetInstanceProfileOptions
func (*VpcV1) NewGetInstanceProfileOptions(name string) *GetInstanceProfileOptions {
	return &GetInstanceProfileOptions{
		Name: core.StringPtr(name),
	}
}

// SetName : Allow user to set Name
func (options *GetInstanceProfileOptions) SetName(name string) *GetInstanceProfileOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceProfileOptions) SetHeaders(param map[string]string) *GetInstanceProfileOptions {
	options.Headers = param
	return options
}

// GetInstanceTemplateOptions : The GetInstanceTemplate options.
type GetInstanceTemplateOptions struct {
	// The instance template identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceTemplateOptions : Instantiate GetInstanceTemplateOptions
func (*VpcV1) NewGetInstanceTemplateOptions(id string) *GetInstanceTemplateOptions {
	return &GetInstanceTemplateOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetInstanceTemplateOptions) SetID(id string) *GetInstanceTemplateOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceTemplateOptions) SetHeaders(param map[string]string) *GetInstanceTemplateOptions {
	options.Headers = param
	return options
}

// GetInstanceVolumeAttachmentOptions : The GetInstanceVolumeAttachment options.
type GetInstanceVolumeAttachmentOptions struct {
	// The instance identifier.
	InstanceID *string `validate:"required,ne="`

	// The volume attachment identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetInstanceVolumeAttachmentOptions : Instantiate GetInstanceVolumeAttachmentOptions
func (*VpcV1) NewGetInstanceVolumeAttachmentOptions(instanceID string, id string) *GetInstanceVolumeAttachmentOptions {
	return &GetInstanceVolumeAttachmentOptions{
		InstanceID: core.StringPtr(instanceID),
		ID:         core.StringPtr(id),
	}
}

// SetInstanceID : Allow user to set InstanceID
func (options *GetInstanceVolumeAttachmentOptions) SetInstanceID(instanceID string) *GetInstanceVolumeAttachmentOptions {
	options.InstanceID = core.StringPtr(instanceID)
	return options
}

// SetID : Allow user to set ID
func (options *GetInstanceVolumeAttachmentOptions) SetID(id string) *GetInstanceVolumeAttachmentOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetInstanceVolumeAttachmentOptions) SetHeaders(param map[string]string) *GetInstanceVolumeAttachmentOptions {
	options.Headers = param
	return options
}

// GetIpsecPolicyOptions : The GetIpsecPolicy options.
type GetIpsecPolicyOptions struct {
	// The IPsec policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetIpsecPolicyOptions : Instantiate GetIpsecPolicyOptions
func (*VpcV1) NewGetIpsecPolicyOptions(id string) *GetIpsecPolicyOptions {
	return &GetIpsecPolicyOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetIpsecPolicyOptions) SetID(id string) *GetIpsecPolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetIpsecPolicyOptions) SetHeaders(param map[string]string) *GetIpsecPolicyOptions {
	options.Headers = param
	return options
}

// GetKeyOptions : The GetKey options.
type GetKeyOptions struct {
	// The key identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetKeyOptions : Instantiate GetKeyOptions
func (*VpcV1) NewGetKeyOptions(id string) *GetKeyOptions {
	return &GetKeyOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetKeyOptions) SetID(id string) *GetKeyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetKeyOptions) SetHeaders(param map[string]string) *GetKeyOptions {
	options.Headers = param
	return options
}

// GetLoadBalancerListenerOptions : The GetLoadBalancerListener options.
type GetLoadBalancerListenerOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetLoadBalancerListenerOptions : Instantiate GetLoadBalancerListenerOptions
func (*VpcV1) NewGetLoadBalancerListenerOptions(loadBalancerID string, id string) *GetLoadBalancerListenerOptions {
	return &GetLoadBalancerListenerOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *GetLoadBalancerListenerOptions) SetLoadBalancerID(loadBalancerID string) *GetLoadBalancerListenerOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetID : Allow user to set ID
func (options *GetLoadBalancerListenerOptions) SetID(id string) *GetLoadBalancerListenerOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetLoadBalancerListenerOptions) SetHeaders(param map[string]string) *GetLoadBalancerListenerOptions {
	options.Headers = param
	return options
}

// GetLoadBalancerListenerPolicyOptions : The GetLoadBalancerListenerPolicy options.
type GetLoadBalancerListenerPolicyOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// The policy identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetLoadBalancerListenerPolicyOptions : Instantiate GetLoadBalancerListenerPolicyOptions
func (*VpcV1) NewGetLoadBalancerListenerPolicyOptions(loadBalancerID string, listenerID string, id string) *GetLoadBalancerListenerPolicyOptions {
	return &GetLoadBalancerListenerPolicyOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ListenerID:     core.StringPtr(listenerID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *GetLoadBalancerListenerPolicyOptions) SetLoadBalancerID(loadBalancerID string) *GetLoadBalancerListenerPolicyOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *GetLoadBalancerListenerPolicyOptions) SetListenerID(listenerID string) *GetLoadBalancerListenerPolicyOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetID : Allow user to set ID
func (options *GetLoadBalancerListenerPolicyOptions) SetID(id string) *GetLoadBalancerListenerPolicyOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetLoadBalancerListenerPolicyOptions) SetHeaders(param map[string]string) *GetLoadBalancerListenerPolicyOptions {
	options.Headers = param
	return options
}

// GetLoadBalancerListenerPolicyRuleOptions : The GetLoadBalancerListenerPolicyRule options.
type GetLoadBalancerListenerPolicyRuleOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The listener identifier.
	ListenerID *string `validate:"required,ne="`

	// The policy identifier.
	PolicyID *string `validate:"required,ne="`

	// The rule identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetLoadBalancerListenerPolicyRuleOptions : Instantiate GetLoadBalancerListenerPolicyRuleOptions
func (*VpcV1) NewGetLoadBalancerListenerPolicyRuleOptions(loadBalancerID string, listenerID string, policyID string, id string) *GetLoadBalancerListenerPolicyRuleOptions {
	return &GetLoadBalancerListenerPolicyRuleOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ListenerID:     core.StringPtr(listenerID),
		PolicyID:       core.StringPtr(policyID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *GetLoadBalancerListenerPolicyRuleOptions) SetLoadBalancerID(loadBalancerID string) *GetLoadBalancerListenerPolicyRuleOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetListenerID : Allow user to set ListenerID
func (options *GetLoadBalancerListenerPolicyRuleOptions) SetListenerID(listenerID string) *GetLoadBalancerListenerPolicyRuleOptions {
	options.ListenerID = core.StringPtr(listenerID)
	return options
}

// SetPolicyID : Allow user to set PolicyID
func (options *GetLoadBalancerListenerPolicyRuleOptions) SetPolicyID(policyID string) *GetLoadBalancerListenerPolicyRuleOptions {
	options.PolicyID = core.StringPtr(policyID)
	return options
}

// SetID : Allow user to set ID
func (options *GetLoadBalancerListenerPolicyRuleOptions) SetID(id string) *GetLoadBalancerListenerPolicyRuleOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetLoadBalancerListenerPolicyRuleOptions) SetHeaders(param map[string]string) *GetLoadBalancerListenerPolicyRuleOptions {
	options.Headers = param
	return options
}

// GetLoadBalancerOptions : The GetLoadBalancer options.
type GetLoadBalancerOptions struct {
	// The load balancer identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetLoadBalancerOptions : Instantiate GetLoadBalancerOptions
func (*VpcV1) NewGetLoadBalancerOptions(id string) *GetLoadBalancerOptions {
	return &GetLoadBalancerOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetLoadBalancerOptions) SetID(id string) *GetLoadBalancerOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetLoadBalancerOptions) SetHeaders(param map[string]string) *GetLoadBalancerOptions {
	options.Headers = param
	return options
}

// GetLoadBalancerPoolMemberOptions : The GetLoadBalancerPoolMember options.
type GetLoadBalancerPoolMemberOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The pool identifier.
	PoolID *string `validate:"required,ne="`

	// The member identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetLoadBalancerPoolMemberOptions : Instantiate GetLoadBalancerPoolMemberOptions
func (*VpcV1) NewGetLoadBalancerPoolMemberOptions(loadBalancerID string, poolID string, id string) *GetLoadBalancerPoolMemberOptions {
	return &GetLoadBalancerPoolMemberOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		PoolID:         core.StringPtr(poolID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *GetLoadBalancerPoolMemberOptions) SetLoadBalancerID(loadBalancerID string) *GetLoadBalancerPoolMemberOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetPoolID : Allow user to set PoolID
func (options *GetLoadBalancerPoolMemberOptions) SetPoolID(poolID string) *GetLoadBalancerPoolMemberOptions {
	options.PoolID = core.StringPtr(poolID)
	return options
}

// SetID : Allow user to set ID
func (options *GetLoadBalancerPoolMemberOptions) SetID(id string) *GetLoadBalancerPoolMemberOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetLoadBalancerPoolMemberOptions) SetHeaders(param map[string]string) *GetLoadBalancerPoolMemberOptions {
	options.Headers = param
	return options
}

// GetLoadBalancerPoolOptions : The GetLoadBalancerPool options.
type GetLoadBalancerPoolOptions struct {
	// The load balancer identifier.
	LoadBalancerID *string `validate:"required,ne="`

	// The pool identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetLoadBalancerPoolOptions : Instantiate GetLoadBalancerPoolOptions
func (*VpcV1) NewGetLoadBalancerPoolOptions(loadBalancerID string, id string) *GetLoadBalancerPoolOptions {
	return &GetLoadBalancerPoolOptions{
		LoadBalancerID: core.StringPtr(loadBalancerID),
		ID:             core.StringPtr(id),
	}
}

// SetLoadBalancerID : Allow user to set LoadBalancerID
func (options *GetLoadBalancerPoolOptions) SetLoadBalancerID(loadBalancerID string) *GetLoadBalancerPoolOptions {
	options.LoadBalancerID = core.StringPtr(loadBalancerID)
	return options
}

// SetID : Allow user to set ID
func (options *GetLoadBalancerPoolOptions) SetID(id string) *GetLoadBalancerPoolOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetLoadBalancerPoolOptions) SetHeaders(param map[string]string) *GetLoadBalancerPoolOptions {
	options.Headers = param
	return options
}

// GetLoadBalancerProfileOptions : The GetLoadBalancerProfile options.
type GetLoadBalancerProfileOptions struct {
	// The load balancer profile name.
	Name *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetLoadBalancerProfileOptions : Instantiate GetLoadBalancerProfileOptions
func (*VpcV1) NewGetLoadBalancerProfileOptions(name string) *GetLoadBalancerProfileOptions {
	return &GetLoadBalancerProfileOptions{
		Name: core.StringPtr(name),
	}
}

// SetName : Allow user to set Name
func (options *GetLoadBalancerProfileOptions) SetName(name string) *GetLoadBalancerProfileOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetLoadBalancerProfileOptions) SetHeaders(param map[string]string) *GetLoadBalancerProfileOptions {
	options.Headers = param
	return options
}

// GetLoadBalancerStatisticsOptions : The GetLoadBalancerStatistics options.
type GetLoadBalancerStatisticsOptions struct {
	// The load balancer identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetLoadBalancerStatisticsOptions : Instantiate GetLoadBalancerStatisticsOptions
func (*VpcV1) NewGetLoadBalancerStatisticsOptions(id string) *GetLoadBalancerStatisticsOptions {
	return &GetLoadBalancerStatisticsOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetLoadBalancerStatisticsOptions) SetID(id string) *GetLoadBalancerStatisticsOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetLoadBalancerStatisticsOptions) SetHeaders(param map[string]string) *GetLoadBalancerStatisticsOptions {
	options.Headers = param
	return options
}

// GetNetworkACLOptions : The GetNetworkACL options.
type GetNetworkACLOptions struct {
	// The network ACL identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetNetworkACLOptions : Instantiate GetNetworkACLOptions
func (*VpcV1) NewGetNetworkACLOptions(id string) *GetNetworkACLOptions {
	return &GetNetworkACLOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetNetworkACLOptions) SetID(id string) *GetNetworkACLOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetNetworkACLOptions) SetHeaders(param map[string]string) *GetNetworkACLOptions {
	options.Headers = param
	return options
}

// GetNetworkACLRuleOptions : The GetNetworkACLRule options.
type GetNetworkACLRuleOptions struct {
	// The network ACL identifier.
	NetworkACLID *string `validate:"required,ne="`

	// The rule identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetNetworkACLRuleOptions : Instantiate GetNetworkACLRuleOptions
func (*VpcV1) NewGetNetworkACLRuleOptions(networkACLID string, id string) *GetNetworkACLRuleOptions {
	return &GetNetworkACLRuleOptions{
		NetworkACLID: core.StringPtr(networkACLID),
		ID:           core.StringPtr(id),
	}
}

// SetNetworkACLID : Allow user to set NetworkACLID
func (options *GetNetworkACLRuleOptions) SetNetworkACLID(networkACLID string) *GetNetworkACLRuleOptions {
	options.NetworkACLID = core.StringPtr(networkACLID)
	return options
}

// SetID : Allow user to set ID
func (options *GetNetworkACLRuleOptions) SetID(id string) *GetNetworkACLRuleOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetNetworkACLRuleOptions) SetHeaders(param map[string]string) *GetNetworkACLRuleOptions {
	options.Headers = param
	return options
}

// GetOperatingSystemOptions : The GetOperatingSystem options.
type GetOperatingSystemOptions struct {
	// The operating system name.
	Name *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetOperatingSystemOptions : Instantiate GetOperatingSystemOptions
func (*VpcV1) NewGetOperatingSystemOptions(name string) *GetOperatingSystemOptions {
	return &GetOperatingSystemOptions{
		Name: core.StringPtr(name),
	}
}

// SetName : Allow user to set Name
func (options *GetOperatingSystemOptions) SetName(name string) *GetOperatingSystemOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetOperatingSystemOptions) SetHeaders(param map[string]string) *GetOperatingSystemOptions {
	options.Headers = param
	return options
}

// GetPublicGatewayOptions : The GetPublicGateway options.
type GetPublicGatewayOptions struct {
	// The public gateway identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetPublicGatewayOptions : Instantiate GetPublicGatewayOptions
func (*VpcV1) NewGetPublicGatewayOptions(id string) *GetPublicGatewayOptions {
	return &GetPublicGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetPublicGatewayOptions) SetID(id string) *GetPublicGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetPublicGatewayOptions) SetHeaders(param map[string]string) *GetPublicGatewayOptions {
	options.Headers = param
	return options
}

// GetRegionOptions : The GetRegion options.
type GetRegionOptions struct {
	// The region name.
	Name *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetRegionOptions : Instantiate GetRegionOptions
func (*VpcV1) NewGetRegionOptions(name string) *GetRegionOptions {
	return &GetRegionOptions{
		Name: core.StringPtr(name),
	}
}

// SetName : Allow user to set Name
func (options *GetRegionOptions) SetName(name string) *GetRegionOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetRegionOptions) SetHeaders(param map[string]string) *GetRegionOptions {
	options.Headers = param
	return options
}

// GetRegionZoneOptions : The GetRegionZone options.
type GetRegionZoneOptions struct {
	// The region name.
	RegionName *string `validate:"required,ne="`

	// The zone name.
	Name *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetRegionZoneOptions : Instantiate GetRegionZoneOptions
func (*VpcV1) NewGetRegionZoneOptions(regionName string, name string) *GetRegionZoneOptions {
	return &GetRegionZoneOptions{
		RegionName: core.StringPtr(regionName),
		Name:       core.StringPtr(name),
	}
}

// SetRegionName : Allow user to set RegionName
func (options *GetRegionZoneOptions) SetRegionName(regionName string) *GetRegionZoneOptions {
	options.RegionName = core.StringPtr(regionName)
	return options
}

// SetName : Allow user to set Name
func (options *GetRegionZoneOptions) SetName(name string) *GetRegionZoneOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetRegionZoneOptions) SetHeaders(param map[string]string) *GetRegionZoneOptions {
	options.Headers = param
	return options
}

// GetSecurityGroupNetworkInterfaceOptions : The GetSecurityGroupNetworkInterface options.
type GetSecurityGroupNetworkInterfaceOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// The network interface identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSecurityGroupNetworkInterfaceOptions : Instantiate GetSecurityGroupNetworkInterfaceOptions
func (*VpcV1) NewGetSecurityGroupNetworkInterfaceOptions(securityGroupID string, id string) *GetSecurityGroupNetworkInterfaceOptions {
	return &GetSecurityGroupNetworkInterfaceOptions{
		SecurityGroupID: core.StringPtr(securityGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *GetSecurityGroupNetworkInterfaceOptions) SetSecurityGroupID(securityGroupID string) *GetSecurityGroupNetworkInterfaceOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *GetSecurityGroupNetworkInterfaceOptions) SetID(id string) *GetSecurityGroupNetworkInterfaceOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSecurityGroupNetworkInterfaceOptions) SetHeaders(param map[string]string) *GetSecurityGroupNetworkInterfaceOptions {
	options.Headers = param
	return options
}

// GetSecurityGroupOptions : The GetSecurityGroup options.
type GetSecurityGroupOptions struct {
	// The security group identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSecurityGroupOptions : Instantiate GetSecurityGroupOptions
func (*VpcV1) NewGetSecurityGroupOptions(id string) *GetSecurityGroupOptions {
	return &GetSecurityGroupOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetSecurityGroupOptions) SetID(id string) *GetSecurityGroupOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSecurityGroupOptions) SetHeaders(param map[string]string) *GetSecurityGroupOptions {
	options.Headers = param
	return options
}

// GetSecurityGroupRuleOptions : The GetSecurityGroupRule options.
type GetSecurityGroupRuleOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// The rule identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSecurityGroupRuleOptions : Instantiate GetSecurityGroupRuleOptions
func (*VpcV1) NewGetSecurityGroupRuleOptions(securityGroupID string, id string) *GetSecurityGroupRuleOptions {
	return &GetSecurityGroupRuleOptions{
		SecurityGroupID: core.StringPtr(securityGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *GetSecurityGroupRuleOptions) SetSecurityGroupID(securityGroupID string) *GetSecurityGroupRuleOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *GetSecurityGroupRuleOptions) SetID(id string) *GetSecurityGroupRuleOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSecurityGroupRuleOptions) SetHeaders(param map[string]string) *GetSecurityGroupRuleOptions {
	options.Headers = param
	return options
}

// GetSecurityGroupTargetOptions : The GetSecurityGroupTarget options.
type GetSecurityGroupTargetOptions struct {
	// The security group identifier.
	SecurityGroupID *string `validate:"required,ne="`

	// The security group target identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSecurityGroupTargetOptions : Instantiate GetSecurityGroupTargetOptions
func (*VpcV1) NewGetSecurityGroupTargetOptions(securityGroupID string, id string) *GetSecurityGroupTargetOptions {
	return &GetSecurityGroupTargetOptions{
		SecurityGroupID: core.StringPtr(securityGroupID),
		ID:              core.StringPtr(id),
	}
}

// SetSecurityGroupID : Allow user to set SecurityGroupID
func (options *GetSecurityGroupTargetOptions) SetSecurityGroupID(securityGroupID string) *GetSecurityGroupTargetOptions {
	options.SecurityGroupID = core.StringPtr(securityGroupID)
	return options
}

// SetID : Allow user to set ID
func (options *GetSecurityGroupTargetOptions) SetID(id string) *GetSecurityGroupTargetOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSecurityGroupTargetOptions) SetHeaders(param map[string]string) *GetSecurityGroupTargetOptions {
	options.Headers = param
	return options
}

// GetSnapshotOptions : The GetSnapshot options.
type GetSnapshotOptions struct {
	// The snapshot identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSnapshotOptions : Instantiate GetSnapshotOptions
func (*VpcV1) NewGetSnapshotOptions(id string) *GetSnapshotOptions {
	return &GetSnapshotOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetSnapshotOptions) SetID(id string) *GetSnapshotOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSnapshotOptions) SetHeaders(param map[string]string) *GetSnapshotOptions {
	options.Headers = param
	return options
}

// GetSubnetNetworkACLOptions : The GetSubnetNetworkACL options.
type GetSubnetNetworkACLOptions struct {
	// The subnet identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSubnetNetworkACLOptions : Instantiate GetSubnetNetworkACLOptions
func (*VpcV1) NewGetSubnetNetworkACLOptions(id string) *GetSubnetNetworkACLOptions {
	return &GetSubnetNetworkACLOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetSubnetNetworkACLOptions) SetID(id string) *GetSubnetNetworkACLOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSubnetNetworkACLOptions) SetHeaders(param map[string]string) *GetSubnetNetworkACLOptions {
	options.Headers = param
	return options
}

// GetSubnetOptions : The GetSubnet options.
type GetSubnetOptions struct {
	// The subnet identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSubnetOptions : Instantiate GetSubnetOptions
func (*VpcV1) NewGetSubnetOptions(id string) *GetSubnetOptions {
	return &GetSubnetOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetSubnetOptions) SetID(id string) *GetSubnetOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSubnetOptions) SetHeaders(param map[string]string) *GetSubnetOptions {
	options.Headers = param
	return options
}

// GetSubnetPublicGatewayOptions : The GetSubnetPublicGateway options.
type GetSubnetPublicGatewayOptions struct {
	// The subnet identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSubnetPublicGatewayOptions : Instantiate GetSubnetPublicGatewayOptions
func (*VpcV1) NewGetSubnetPublicGatewayOptions(id string) *GetSubnetPublicGatewayOptions {
	return &GetSubnetPublicGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetSubnetPublicGatewayOptions) SetID(id string) *GetSubnetPublicGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSubnetPublicGatewayOptions) SetHeaders(param map[string]string) *GetSubnetPublicGatewayOptions {
	options.Headers = param
	return options
}

// GetSubnetReservedIPOptions : The GetSubnetReservedIP options.
type GetSubnetReservedIPOptions struct {
	// The subnet identifier.
	SubnetID *string `validate:"required,ne="`

	// The reserved IP identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSubnetReservedIPOptions : Instantiate GetSubnetReservedIPOptions
func (*VpcV1) NewGetSubnetReservedIPOptions(subnetID string, id string) *GetSubnetReservedIPOptions {
	return &GetSubnetReservedIPOptions{
		SubnetID: core.StringPtr(subnetID),
		ID:       core.StringPtr(id),
	}
}

// SetSubnetID : Allow user to set SubnetID
func (options *GetSubnetReservedIPOptions) SetSubnetID(subnetID string) *GetSubnetReservedIPOptions {
	options.SubnetID = core.StringPtr(subnetID)
	return options
}

// SetID : Allow user to set ID
func (options *GetSubnetReservedIPOptions) SetID(id string) *GetSubnetReservedIPOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSubnetReservedIPOptions) SetHeaders(param map[string]string) *GetSubnetReservedIPOptions {
	options.Headers = param
	return options
}

// GetSubnetRoutingTableOptions : The GetSubnetRoutingTable options.
type GetSubnetRoutingTableOptions struct {
	// The subnet identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetSubnetRoutingTableOptions : Instantiate GetSubnetRoutingTableOptions
func (*VpcV1) NewGetSubnetRoutingTableOptions(id string) *GetSubnetRoutingTableOptions {
	return &GetSubnetRoutingTableOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetSubnetRoutingTableOptions) SetID(id string) *GetSubnetRoutingTableOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetSubnetRoutingTableOptions) SetHeaders(param map[string]string) *GetSubnetRoutingTableOptions {
	options.Headers = param
	return options
}

// GetVolumeOptions : The GetVolume options.
type GetVolumeOptions struct {
	// The volume identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVolumeOptions : Instantiate GetVolumeOptions
func (*VpcV1) NewGetVolumeOptions(id string) *GetVolumeOptions {
	return &GetVolumeOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetVolumeOptions) SetID(id string) *GetVolumeOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVolumeOptions) SetHeaders(param map[string]string) *GetVolumeOptions {
	options.Headers = param
	return options
}

// GetVolumeProfileOptions : The GetVolumeProfile options.
type GetVolumeProfileOptions struct {
	// The volume profile name.
	Name *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVolumeProfileOptions : Instantiate GetVolumeProfileOptions
func (*VpcV1) NewGetVolumeProfileOptions(name string) *GetVolumeProfileOptions {
	return &GetVolumeProfileOptions{
		Name: core.StringPtr(name),
	}
}

// SetName : Allow user to set Name
func (options *GetVolumeProfileOptions) SetName(name string) *GetVolumeProfileOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVolumeProfileOptions) SetHeaders(param map[string]string) *GetVolumeProfileOptions {
	options.Headers = param
	return options
}

// GetVPCAddressPrefixOptions : The GetVPCAddressPrefix options.
type GetVPCAddressPrefixOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The prefix identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVPCAddressPrefixOptions : Instantiate GetVPCAddressPrefixOptions
func (*VpcV1) NewGetVPCAddressPrefixOptions(vpcID string, id string) *GetVPCAddressPrefixOptions {
	return &GetVPCAddressPrefixOptions{
		VPCID: core.StringPtr(vpcID),
		ID:    core.StringPtr(id),
	}
}

// SetVPCID : Allow user to set VPCID
func (options *GetVPCAddressPrefixOptions) SetVPCID(vpcID string) *GetVPCAddressPrefixOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetID : Allow user to set ID
func (options *GetVPCAddressPrefixOptions) SetID(id string) *GetVPCAddressPrefixOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVPCAddressPrefixOptions) SetHeaders(param map[string]string) *GetVPCAddressPrefixOptions {
	options.Headers = param
	return options
}

// GetVPCDefaultNetworkACLOptions : The GetVPCDefaultNetworkACL options.
type GetVPCDefaultNetworkACLOptions struct {
	// The VPC identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVPCDefaultNetworkACLOptions : Instantiate GetVPCDefaultNetworkACLOptions
func (*VpcV1) NewGetVPCDefaultNetworkACLOptions(id string) *GetVPCDefaultNetworkACLOptions {
	return &GetVPCDefaultNetworkACLOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetVPCDefaultNetworkACLOptions) SetID(id string) *GetVPCDefaultNetworkACLOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVPCDefaultNetworkACLOptions) SetHeaders(param map[string]string) *GetVPCDefaultNetworkACLOptions {
	options.Headers = param
	return options
}

// GetVPCDefaultRoutingTableOptions : The GetVPCDefaultRoutingTable options.
type GetVPCDefaultRoutingTableOptions struct {
	// The VPC identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVPCDefaultRoutingTableOptions : Instantiate GetVPCDefaultRoutingTableOptions
func (*VpcV1) NewGetVPCDefaultRoutingTableOptions(id string) *GetVPCDefaultRoutingTableOptions {
	return &GetVPCDefaultRoutingTableOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetVPCDefaultRoutingTableOptions) SetID(id string) *GetVPCDefaultRoutingTableOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVPCDefaultRoutingTableOptions) SetHeaders(param map[string]string) *GetVPCDefaultRoutingTableOptions {
	options.Headers = param
	return options
}

// GetVPCDefaultSecurityGroupOptions : The GetVPCDefaultSecurityGroup options.
type GetVPCDefaultSecurityGroupOptions struct {
	// The VPC identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVPCDefaultSecurityGroupOptions : Instantiate GetVPCDefaultSecurityGroupOptions
func (*VpcV1) NewGetVPCDefaultSecurityGroupOptions(id string) *GetVPCDefaultSecurityGroupOptions {
	return &GetVPCDefaultSecurityGroupOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetVPCDefaultSecurityGroupOptions) SetID(id string) *GetVPCDefaultSecurityGroupOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVPCDefaultSecurityGroupOptions) SetHeaders(param map[string]string) *GetVPCDefaultSecurityGroupOptions {
	options.Headers = param
	return options
}

// GetVPCOptions : The GetVPC options.
type GetVPCOptions struct {
	// The VPC identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVPCOptions : Instantiate GetVPCOptions
func (*VpcV1) NewGetVPCOptions(id string) *GetVPCOptions {
	return &GetVPCOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetVPCOptions) SetID(id string) *GetVPCOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVPCOptions) SetHeaders(param map[string]string) *GetVPCOptions {
	options.Headers = param
	return options
}

// GetVPCRouteOptions : The GetVPCRoute options.
type GetVPCRouteOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The route identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVPCRouteOptions : Instantiate GetVPCRouteOptions
func (*VpcV1) NewGetVPCRouteOptions(vpcID string, id string) *GetVPCRouteOptions {
	return &GetVPCRouteOptions{
		VPCID: core.StringPtr(vpcID),
		ID:    core.StringPtr(id),
	}
}

// SetVPCID : Allow user to set VPCID
func (options *GetVPCRouteOptions) SetVPCID(vpcID string) *GetVPCRouteOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetID : Allow user to set ID
func (options *GetVPCRouteOptions) SetID(id string) *GetVPCRouteOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVPCRouteOptions) SetHeaders(param map[string]string) *GetVPCRouteOptions {
	options.Headers = param
	return options
}

// GetVPCRoutingTableOptions : The GetVPCRoutingTable options.
type GetVPCRoutingTableOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The routing table identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVPCRoutingTableOptions : Instantiate GetVPCRoutingTableOptions
func (*VpcV1) NewGetVPCRoutingTableOptions(vpcID string, id string) *GetVPCRoutingTableOptions {
	return &GetVPCRoutingTableOptions{
		VPCID: core.StringPtr(vpcID),
		ID:    core.StringPtr(id),
	}
}

// SetVPCID : Allow user to set VPCID
func (options *GetVPCRoutingTableOptions) SetVPCID(vpcID string) *GetVPCRoutingTableOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetID : Allow user to set ID
func (options *GetVPCRoutingTableOptions) SetID(id string) *GetVPCRoutingTableOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVPCRoutingTableOptions) SetHeaders(param map[string]string) *GetVPCRoutingTableOptions {
	options.Headers = param
	return options
}

// GetVPCRoutingTableRouteOptions : The GetVPCRoutingTableRoute options.
type GetVPCRoutingTableRouteOptions struct {
	// The VPC identifier.
	VPCID *string `validate:"required,ne="`

	// The routing table identifier.
	RoutingTableID *string `validate:"required,ne="`

	// The VPC routing table route identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVPCRoutingTableRouteOptions : Instantiate GetVPCRoutingTableRouteOptions
func (*VpcV1) NewGetVPCRoutingTableRouteOptions(vpcID string, routingTableID string, id string) *GetVPCRoutingTableRouteOptions {
	return &GetVPCRoutingTableRouteOptions{
		VPCID:          core.StringPtr(vpcID),
		RoutingTableID: core.StringPtr(routingTableID),
		ID:             core.StringPtr(id),
	}
}

// SetVPCID : Allow user to set VPCID
func (options *GetVPCRoutingTableRouteOptions) SetVPCID(vpcID string) *GetVPCRoutingTableRouteOptions {
	options.VPCID = core.StringPtr(vpcID)
	return options
}

// SetRoutingTableID : Allow user to set RoutingTableID
func (options *GetVPCRoutingTableRouteOptions) SetRoutingTableID(routingTableID string) *GetVPCRoutingTableRouteOptions {
	options.RoutingTableID = core.StringPtr(routingTableID)
	return options
}

// SetID : Allow user to set ID
func (options *GetVPCRoutingTableRouteOptions) SetID(id string) *GetVPCRoutingTableRouteOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetVPCRoutingTableRouteOptions) SetHeaders(param map[string]string) *GetVPCRoutingTableRouteOptions {
	options.Headers = param
	return options
}

// GetVPNGatewayConnectionOptions : The GetVPNGatewayConnection options.
type GetVPNGatewayConnectionOptions struct {
	// The VPN gateway identifier.
	VPNGatewayID *string `validate:"required,ne="`

	// The VPN gateway connection identifier.
	ID *string `validate:"required,ne="`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetVPNGatewayConnectionOptions : Instantiate GetVPNGatewayConnectionOptions
func (*VpcV1) NewGetVPNGatewayConnectionOptions(vpnGatewayID string, id string) *GetVPNGatewayConnectionOptions {
	return &GetVPNGatewayConnectionOptions{
		VPNGatewayID: core.StringPtr(vpnGatewayID),
		ID:           core.StringPtr(id),
	}
}

// SetVPNGatewayID : Allow user to set VPNGatewayID
func (options *GetVPNGatewayConnectionOptions) SetVPNGatewayID(vpnGatewayID strin