/**
 * (C) Copyright IBM Corp. 2021.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// Package directlinkv1 : Operations and models for the DirectLinkV1 service
package directlinkv1

import (
	"encoding/json"
	"fmt"
	"io"
	"reflect"

	"github.com/IBM/go-sdk-core/v4/core"
	common "github.com/IBM/networking-go-sdk/common"
	"github.com/go-openapi/strfmt"
)

// DirectLinkV1 : No description provided (generated by Openapi Generator
// https://github.com/openapitools/openapi-generator)
//
// Version: __VERSION__
type DirectLinkV1 struct {
	Service *core.BaseService

	// Requests the version of the API as a date in the format `YYYY-MM-DD`. Any date from 2019-12-13 up to the current
	// date may be provided. Specify the current date to request the latest version.
	Version *string
}

// DefaultServiceURL is the default URL to make service requests to.
const DefaultServiceURL = "https://directlink.cloud.ibm.com/v1"

// DefaultServiceName is the default key used to find external configuration information.
const DefaultServiceName = "direct_link"

// DirectLinkV1Options : Service options
type DirectLinkV1Options struct {
	ServiceName   string
	URL           string
	Authenticator core.Authenticator

	// Requests the version of the API as a date in the format `YYYY-MM-DD`. Any date from 2019-12-13 up to the current
	// date may be provided. Specify the current date to request the latest version.
	Version *string `validate:"required"`
}

// NewDirectLinkV1UsingExternalConfig : constructs an instance of DirectLinkV1 with passed in options and external configuration.
func NewDirectLinkV1UsingExternalConfig(options *DirectLinkV1Options) (directLink *DirectLinkV1, err error) {
	if options.ServiceName == "" {
		options.ServiceName = DefaultServiceName
	}

	if options.Authenticator == nil {
		options.Authenticator, err = core.GetAuthenticatorFromEnvironment(options.ServiceName)
		if err != nil {
			return
		}
	}

	directLink, err = NewDirectLinkV1(options)
	if err != nil {
		return
	}

	err = directLink.Service.ConfigureService(options.ServiceName)
	if err != nil {
		return
	}

	if options.URL != "" {
		err = directLink.Service.SetServiceURL(options.URL)
	}
	return
}

// NewDirectLinkV1 : constructs an instance of DirectLinkV1 with passed in options.
func NewDirectLinkV1(options *DirectLinkV1Options) (service *DirectLinkV1, err error) {
	serviceOptions := &core.ServiceOptions{
		URL:           DefaultServiceURL,
		Authenticator: options.Authenticator,
	}

	err = core.ValidateStruct(options, "options")
	if err != nil {
		return
	}

	baseService, err := core.NewBaseService(serviceOptions)
	if err != nil {
		return
	}

	if options.URL != "" {
		err = baseService.SetServiceURL(options.URL)
		if err != nil {
			return
		}
	}

	service = &DirectLinkV1{
		Service: baseService,
		Version: options.Version,
	}

	return
}

// SetServiceURL sets the service URL
func (directLink *DirectLinkV1) SetServiceURL(url string) error {
	return directLink.Service.SetServiceURL(url)
}

// ListGateways : List gateways
// List all Direct Link gateways in this account.  Gateways in other accounts with connections to networks in this
// account are also returned.
func (directLink *DirectLinkV1) ListGateways(listGatewaysOptions *ListGatewaysOptions) (result *GatewayCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listGatewaysOptions, "listGatewaysOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways"}
	pathParameters := []string{}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range listGatewaysOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "ListGateways")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLink.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalGatewayCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateGateway : Create gateway
// Creates a Direct Link gateway based on the supplied template.
func (directLink *DirectLinkV1) CreateGateway(createGatewayOptions *CreateGatewayOptions) (result *Gateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createGatewayOptions, "createGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createGatewayOptions, "createGatewayOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways"}
	pathParameters := []string{}

	builder := core.NewRequestBuilder(core.POST)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range createGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "CreateGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	_, err = builder.SetBodyContentJSON(createGatewayOptions.GatewayTemplate)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLink.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteGateway : Delete gateway
// Delete a Direct Link gateway.
func (directLink *DirectLinkV1) DeleteGateway(deleteGatewayOptions *DeleteGatewayOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteGatewayOptions, "deleteGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteGatewayOptions, "deleteGatewayOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways"}
	pathParameters := []string{*deleteGatewayOptions.ID}

	builder := core.NewRequestBuilder(core.DELETE)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "DeleteGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = directLink.Service.Request(request, nil)

	return
}

// GetGateway : Get gateway
// Retrieve a Direct Link gateway.
func (directLink *DirectLinkV1) GetGateway(getGatewayOptions *GetGatewayOptions) (result *Gateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getGatewayOptions, "getGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getGatewayOptions, "getGatewayOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways"}
	pathParameters := []string{*getGatewayOptions.ID}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range getGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "GetGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLink.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateGateway : Update gateway
// Update a Direct Link gateway.
func (directLink *DirectLinkV1) UpdateGateway(updateGatewayOptions *UpdateGatewayOptions) (result *Gateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateGatewayOptions, "updateGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateGatewayOptions, "updateGatewayOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways"}
	pathParameters := []string{*updateGatewayOptions.ID}

	builder := core.NewRequestBuilder(core.PATCH)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "UpdateGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	body := make(map[string]interface{})
	if updateGatewayOptions.Global != nil {
		body["global"] = updateGatewayOptions.Global
	}
	if updateGatewayOptions.LoaRejectReason != nil {
		body["loa_reject_reason"] = updateGatewayOptions.LoaRejectReason
	}
	if updateGatewayOptions.MacsecConfig != nil {
		body["macsec_config"] = updateGatewayOptions.MacsecConfig
	}
	if updateGatewayOptions.Metered != nil {
		body["metered"] = updateGatewayOptions.Metered
	}
	if updateGatewayOptions.Name != nil {
		body["name"] = updateGatewayOptions.Name
	}
	if updateGatewayOptions.OperationalStatus != nil {
		body["operational_status"] = updateGatewayOptions.OperationalStatus
	}
	if updateGatewayOptions.SpeedMbps != nil {
		body["speed_mbps"] = updateGatewayOptions.SpeedMbps
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLink.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateGatewayAction : Approve or reject change requests
// Approve or reject a gateway's current oustanding change request.
//
// This API is only used for provider created Direct Link Connect gateways to approve or reject specific changes
// initiated from a provider portal.
func (directLink *DirectLinkV1) CreateGatewayAction(createGatewayActionOptions *CreateGatewayActionOptions) (result *Gateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createGatewayActionOptions, "createGatewayActionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createGatewayActionOptions, "createGatewayActionOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways", "actions"}
	pathParameters := []string{*createGatewayActionOptions.ID}

	builder := core.NewRequestBuilder(core.POST)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range createGatewayActionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "CreateGatewayAction")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	body := make(map[string]interface{})
	if createGatewayActionOptions.Action != nil {
		body["action"] = createGatewayActionOptions.Action
	}
	if createGatewayActionOptions.Global != nil {
		body["global"] = createGatewayActionOptions.Global
	}
	if createGatewayActionOptions.Metered != nil {
		body["metered"] = createGatewayActionOptions.Metered
	}
	if createGatewayActionOptions.ResourceGroup != nil {
		body["resource_group"] = createGatewayActionOptions.ResourceGroup
	}
	if createGatewayActionOptions.Updates != nil {
		body["updates"] = createGatewayActionOptions.Updates
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLink.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListGatewayCompletionNotice : Get completion notice
// Retrieve a Direct Link Dedicated gateway's completion notice.
func (directLink *DirectLinkV1) ListGatewayCompletionNotice(listGatewayCompletionNoticeOptions *ListGatewayCompletionNoticeOptions) (result io.ReadCloser, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listGatewayCompletionNoticeOptions, "listGatewayCompletionNoticeOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listGatewayCompletionNoticeOptions, "listGatewayCompletionNoticeOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways", "completion_notice"}
	pathParameters := []string{*listGatewayCompletionNoticeOptions.ID}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range listGatewayCompletionNoticeOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "ListGatewayCompletionNotice")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/pdf")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = directLink.Service.Request(request, &result)

	return
}

// CreateGatewayCompletionNotice : Create completion notice
// Upload a Direct Link Dedicated gateway completion notice.
func (directLink *DirectLinkV1) CreateGatewayCompletionNotice(createGatewayCompletionNoticeOptions *CreateGatewayCompletionNoticeOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createGatewayCompletionNoticeOptions, "createGatewayCompletionNoticeOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createGatewayCompletionNoticeOptions, "createGatewayCompletionNoticeOptions")
	if err != nil {
		return
	}
	if createGatewayCompletionNoticeOptions.Upload == nil {
		err = fmt.Errorf("at least one of  or upload must be supplied")
		return
	}

	pathSegments := []string{"gateways", "completion_notice"}
	pathParameters := []string{*createGatewayCompletionNoticeOptions.ID}

	builder := core.NewRequestBuilder(core.PUT)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range createGatewayCompletionNoticeOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "CreateGatewayCompletionNotice")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	if createGatewayCompletionNoticeOptions.Upload != nil {
		builder.AddFormData("upload", "filename",
			core.StringNilMapper(createGatewayCompletionNoticeOptions.UploadContentType), createGatewayCompletionNoticeOptions.Upload)
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = directLink.Service.Request(request, nil)

	return
}

// ListGatewayLetterOfAuthorization : Get letter of authorization
// Retrieve a Direct Link Dedicated gateway's Letter of Authorization.
func (directLink *DirectLinkV1) ListGatewayLetterOfAuthorization(listGatewayLetterOfAuthorizationOptions *ListGatewayLetterOfAuthorizationOptions) (result io.ReadCloser, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listGatewayLetterOfAuthorizationOptions, "listGatewayLetterOfAuthorizationOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listGatewayLetterOfAuthorizationOptions, "listGatewayLetterOfAuthorizationOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways", "letter_of_authorization"}
	pathParameters := []string{*listGatewayLetterOfAuthorizationOptions.ID}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range listGatewayLetterOfAuthorizationOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "ListGatewayLetterOfAuthorization")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/pdf")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = directLink.Service.Request(request, &result)

	return
}

// GetGatewayStatistics : Gateway statistics
// Retrieve gateway statistics.  Specify statistic to retrieve using required `type` query parameter.  Currently data
// retrieval is only supported for MACsec configurations.
func (directLink *DirectLinkV1) GetGatewayStatistics(getGatewayStatisticsOptions *GetGatewayStatisticsOptions) (result *GatewayStatisticCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getGatewayStatisticsOptions, "getGatewayStatisticsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getGatewayStatisticsOptions, "getGatewayStatisticsOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways", "statistics"}
	pathParameters := []string{*getGatewayStatisticsOptions.ID}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range getGatewayStatisticsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "GetGatewayStatistics")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("type", fmt.Sprint(*getGatewayStatisticsOptions.Type))
	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLink.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalGatewayStatisticCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListOfferingTypeLocations : List available locations
// Retrieve the list of valid locations for the specified Direct Link offering.
func (directLink *DirectLinkV1) ListOfferingTypeLocations(listOfferingTypeLocationsOptions *ListOfferingTypeLocationsOptions) (result *LocationCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listOfferingTypeLocationsOptions, "listOfferingTypeLocationsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listOfferingTypeLocationsOptions, "listOfferingTypeLocationsOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"offering_types", "locations"}
	pathParameters := []string{*listOfferingTypeLocationsOptions.OfferingType}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range listOfferingTypeLocationsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "ListOfferingTypeLocations")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLink.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLocationCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListOfferingTypeLocationCrossConnectRouters : List routers
// Retrieve location specific cross connect router information.  Only valid for offering_type=dedicated locations.
func (directLink *DirectLinkV1) ListOfferingTypeLocationCrossConnectRouters(listOfferingTypeLocationCrossConnectRoutersOptions *ListOfferingTypeLocationCrossConnectRoutersOptions) (result *LocationCrossConnectRouterCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listOfferingTypeLocationCrossConnectRoutersOptions, "listOfferingTypeLocationCrossConnectRoutersOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listOfferingTypeLocationCrossConnectRoutersOptions, "listOfferingTypeLocationCrossConnectRoutersOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"offering_types", "locations", "cross_connect_routers"}
	pathParameters := []string{*listOfferingTypeLocationCrossConnectRoutersOptions.OfferingType, *listOfferingTypeLocationCrossConnectRoutersOptions.LocationName}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range listOfferingTypeLocationCrossConnectRoutersOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "ListOfferingTypeLocationCrossConnectRouters")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLink.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalLocationCrossConnectRouterCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListOfferingTypeSpeeds : List speed options
// List the available Direct Link speeds.
func (directLink *DirectLinkV1) ListOfferingTypeSpeeds(listOfferingTypeSpeedsOptions *ListOfferingTypeSpeedsOptions) (result *OfferingSpeedCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listOfferingTypeSpeedsOptions, "listOfferingTypeSpeedsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listOfferingTypeSpeedsOptions, "listOfferingTypeSpeedsOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"offering_types", "speeds"}
	pathParameters := []string{*listOfferingTypeSpeedsOptions.OfferingType}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range listOfferingTypeSpeedsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "ListOfferingTypeSpeeds")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLink.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalOfferingSpeedCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListPorts : List ports
// Retrieve list of available Direct Link connect ports.  These ports can be used to create Direct Link connect
// gateways.
func (directLink *DirectLinkV1) ListPorts(listPortsOptions *ListPortsOptions) (result *PortCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listPortsOptions, "listPortsOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"ports"}
	pathParameters := []string{}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range listPortsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "ListPorts")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))
	if listPortsOptions.Start != nil {
		builder.AddQuery("start", fmt.Sprint(*listPortsOptions.Start))
	}
	if listPortsOptions.Limit != nil {
		builder.AddQuery("limit", fmt.Sprint(*listPortsOptions.Limit))
	}
	if listPortsOptions.LocationName != nil {
		builder.AddQuery("location_name", fmt.Sprint(*listPortsOptions.LocationName))
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLink.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalPortCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetPort : Get port
// Retrieve Direct Link Connect provider port.
func (directLink *DirectLinkV1) GetPort(getPortOptions *GetPortOptions) (result *Port, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getPortOptions, "getPortOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getPortOptions, "getPortOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"ports"}
	pathParameters := []string{*getPortOptions.ID}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range getPortOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "GetPort")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLink.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalPort)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListGatewayVirtualConnections : List virtual connections
// List a gateway's virtual connections.   For gateway in other account with virtual connections that connect to network
// in this account.  Only virtual connections that connect to this account are returned.
func (directLink *DirectLinkV1) ListGatewayVirtualConnections(listGatewayVirtualConnectionsOptions *ListGatewayVirtualConnectionsOptions) (result *GatewayVirtualConnectionCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listGatewayVirtualConnectionsOptions, "listGatewayVirtualConnectionsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listGatewayVirtualConnectionsOptions, "listGatewayVirtualConnectionsOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways", "virtual_connections"}
	pathParameters := []string{*listGatewayVirtualConnectionsOptions.GatewayID}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range listGatewayVirtualConnectionsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "ListGatewayVirtualConnections")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLink.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalGatewayVirtualConnectionCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateGatewayVirtualConnection : Create virtual connection
// Create a virtual connection to the specified network.
func (directLink *DirectLinkV1) CreateGatewayVirtualConnection(createGatewayVirtualConnectionOptions *CreateGatewayVirtualConnectionOptions) (result *GatewayVirtualConnection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createGatewayVirtualConnectionOptions, "createGatewayVirtualConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createGatewayVirtualConnectionOptions, "createGatewayVirtualConnectionOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways", "virtual_connections"}
	pathParameters := []string{*createGatewayVirtualConnectionOptions.GatewayID}

	builder := core.NewRequestBuilder(core.POST)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range createGatewayVirtualConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "CreateGatewayVirtualConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	body := make(map[string]interface{})
	if createGatewayVirtualConnectionOptions.Name != nil {
		body["name"] = createGatewayVirtualConnectionOptions.Name
	}
	if createGatewayVirtualConnectionOptions.Type != nil {
		body["type"] = createGatewayVirtualConnectionOptions.Type
	}
	if createGatewayVirtualConnectionOptions.NetworkID != nil {
		body["network_id"] = createGatewayVirtualConnectionOptions.NetworkID
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLink.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalGatewayVirtualConnection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteGatewayVirtualConnection : Delete virtual connection
// Delete the virtual connection.
func (directLink *DirectLinkV1) DeleteGatewayVirtualConnection(deleteGatewayVirtualConnectionOptions *DeleteGatewayVirtualConnectionOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteGatewayVirtualConnectionOptions, "deleteGatewayVirtualConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteGatewayVirtualConnectionOptions, "deleteGatewayVirtualConnectionOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways", "virtual_connections"}
	pathParameters := []string{*deleteGatewayVirtualConnectionOptions.GatewayID, *deleteGatewayVirtualConnectionOptions.ID}

	builder := core.NewRequestBuilder(core.DELETE)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteGatewayVirtualConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "DeleteGatewayVirtualConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = directLink.Service.Request(request, nil)

	return
}

// GetGatewayVirtualConnection : Get virtual connection
// Retrieve a virtual connection.
func (directLink *DirectLinkV1) GetGatewayVirtualConnection(getGatewayVirtualConnectionOptions *GetGatewayVirtualConnectionOptions) (result *GatewayVirtualConnection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getGatewayVirtualConnectionOptions, "getGatewayVirtualConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getGatewayVirtualConnectionOptions, "getGatewayVirtualConnectionOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways", "virtual_connections"}
	pathParameters := []string{*getGatewayVirtualConnectionOptions.GatewayID, *getGatewayVirtualConnectionOptions.ID}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range getGatewayVirtualConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "GetGatewayVirtualConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLink.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalGatewayVirtualConnection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateGatewayVirtualConnection : Update virtual connection
// Update a virtual connection.
func (directLink *DirectLinkV1) UpdateGatewayVirtualConnection(updateGatewayVirtualConnectionOptions *UpdateGatewayVirtualConnectionOptions) (result *GatewayVirtualConnection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateGatewayVirtualConnectionOptions, "updateGatewayVirtualConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateGatewayVirtualConnectionOptions, "updateGatewayVirtualConnectionOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"gateways", "virtual_connections"}
	pathParameters := []string{*updateGatewayVirtualConnectionOptions.GatewayID, *updateGatewayVirtualConnectionOptions.ID}

	builder := core.NewRequestBuilder(core.PATCH)
	_, err = builder.ConstructHTTPURL(directLink.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateGatewayVirtualConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("direct_link", "V1", "UpdateGatewayVirtualConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*directLink.Version))

	body := make(map[string]interface{})
	if updateGatewayVirtualConnectionOptions.Name != nil {
		body["name"] = updateGatewayVirtualConnectionOptions.Name
	}
	if updateGatewayVirtualConnectionOptions.Status != nil {
		body["status"] = updateGatewayVirtualConnectionOptions.Status
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = directLink.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalGatewayVirtualConnection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateGatewayActionOptions : The CreateGatewayAction options.
type CreateGatewayActionOptions struct {
	// Direct Link Connect gateway identifier.
	ID *string `json:"id" validate:"required"`

	// Action request.
	Action *string `json:"action" validate:"required"`

	// Required for create_gateway_approve requests to select the gateway's routing option.  Gateways with global routing
	// (`true`) can connect to networks outside of their associated region.
	Global *bool `json:"global,omitempty"`

	// Required for create_gateway_approve requests to select the gateway's metered billing option.  When `true` gateway
	// usage is billed per gigabyte.  When `false` there is no per gigabyte usage charge, instead a flat rate is charged
	// for the gateway.
	Metered *bool `json:"metered,omitempty"`

	// Set for create_gateway_approve requests to select the gateway's resource group.  If unspecified on
	// create_gateway_approve, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup *ResourceGroupIdentity `json:"resource_group,omitempty"`

	// Specify attribute updates being approved or rejected, update_attributes_approve and update_attributes_reject actions
	// must provide an updates field that matches the gateway's current pending changes.
	Updates []GatewayActionTemplateUpdatesItemIntf `json:"updates,omitempty"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateGatewayActionOptions.Action property.
// Action request.
const (
	CreateGatewayActionOptions_Action_CreateGatewayApprove    = "create_gateway_approve"
	CreateGatewayActionOptions_Action_CreateGatewayReject     = "create_gateway_reject"
	CreateGatewayActionOptions_Action_DeleteGatewayApprove    = "delete_gateway_approve"
	CreateGatewayActionOptions_Action_DeleteGatewayReject     = "delete_gateway_reject"
	CreateGatewayActionOptions_Action_UpdateAttributesApprove = "update_attributes_approve"
	CreateGatewayActionOptions_Action_UpdateAttributesReject  = "update_attributes_reject"
)

// NewCreateGatewayActionOptions : Instantiate CreateGatewayActionOptions
func (*DirectLinkV1) NewCreateGatewayActionOptions(id string, action string) *CreateGatewayActionOptions {
	return &CreateGatewayActionOptions{
		ID:     core.StringPtr(id),
		Action: core.StringPtr(action),
	}
}

// SetID : Allow user to set ID
func (options *CreateGatewayActionOptions) SetID(id string) *CreateGatewayActionOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetAction : Allow user to set Action
func (options *CreateGatewayActionOptions) SetAction(action string) *CreateGatewayActionOptions {
	options.Action = core.StringPtr(action)
	return options
}

// SetGlobal : Allow user to set Global
func (options *CreateGatewayActionOptions) SetGlobal(global bool) *CreateGatewayActionOptions {
	options.Global = core.BoolPtr(global)
	return options
}

// SetMetered : Allow user to set Metered
func (options *CreateGatewayActionOptions) SetMetered(metered bool) *CreateGatewayActionOptions {
	options.Metered = core.BoolPtr(metered)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateGatewayActionOptions) SetResourceGroup(resourceGroup *ResourceGroupIdentity) *CreateGatewayActionOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetUpdates : Allow user to set Updates
func (options *CreateGatewayActionOptions) SetUpdates(updates []GatewayActionTemplateUpdatesItemIntf) *CreateGatewayActionOptions {
	options.Updates = updates
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateGatewayActionOptions) SetHeaders(param map[string]string) *CreateGatewayActionOptions {
	options.Headers = param
	return options
}

// CreateGatewayCompletionNoticeOptions : The CreateGatewayCompletionNotice options.
type CreateGatewayCompletionNoticeOptions struct {
	// Direct Link Dedicated gateway identifier.
	ID *string `json:"id" validate:"required"`

	// Completion notice PDF file.
	Upload io.ReadCloser `json:"upload,omitempty"`

	// The content type of upload.
	UploadContentType *string `json:"upload_content_type,omitempty"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateGatewayCompletionNoticeOptions : Instantiate CreateGatewayCompletionNoticeOptions
func (*DirectLinkV1) NewCreateGatewayCompletionNoticeOptions(id string) *CreateGatewayCompletionNoticeOptions {
	return &CreateGatewayCompletionNoticeOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *CreateGatewayCompletionNoticeOptions) SetID(id string) *CreateGatewayCompletionNoticeOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetUpload : Allow user to set Upload
func (options *CreateGatewayCompletionNoticeOptions) SetUpload(upload io.ReadCloser) *CreateGatewayCompletionNoticeOptions {
	options.Upload = upload
	return options
}

// SetUploadContentType : Allow user to set UploadContentType
func (options *CreateGatewayCompletionNoticeOptions) SetUploadContentType(uploadContentType string) *CreateGatewayCompletionNoticeOptions {
	options.UploadContentType = core.StringPtr(uploadContentType)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateGatewayCompletionNoticeOptions) SetHeaders(param map[string]string) *CreateGatewayCompletionNoticeOptions {
	options.Headers = param
	return options
}

// CreateGatewayOptions : The CreateGateway options.
type CreateGatewayOptions struct {
	// The Direct Link Gateway template.
	GatewayTemplate GatewayTemplateIntf `json:"GatewayTemplate" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateGatewayOptions : Instantiate CreateGatewayOptions
func (*DirectLinkV1) NewCreateGatewayOptions(gatewayTemplate GatewayTemplateIntf) *CreateGatewayOptions {
	return &CreateGatewayOptions{
		GatewayTemplate: gatewayTemplate,
	}
}

// SetGatewayTemplate : Allow user to set GatewayTemplate
func (options *CreateGatewayOptions) SetGatewayTemplate(gatewayTemplate GatewayTemplateIntf) *CreateGatewayOptions {
	options.GatewayTemplate = gatewayTemplate
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateGatewayOptions) SetHeaders(param map[string]string) *CreateGatewayOptions {
	options.Headers = param
	return options
}

// CreateGatewayVirtualConnectionOptions : The CreateGatewayVirtualConnection options.
type CreateGatewayVirtualConnectionOptions struct {
	// Direct Link gateway identifier.
	GatewayID *string `json:"gateway_id" validate:"required"`

	// The user-defined name for this virtual connection.  Virtual connection names are unique within a gateway.  This is
	// the name of the virtual connection itself, the network being connected may have its own name attribute.
	Name *string `json:"name" validate:"required"`

	// The type of virtual connection.
	Type *string `json:"type" validate:"required"`

	// Unique identifier of the target network.  For type=vpc virtual connections this is the CRN of the target VPC.  This
	// field does not apply to type=classic connections.
	NetworkID *string `json:"network_id,omitempty"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateGatewayVirtualConnectionOptions.Type property.
// The type of virtual connection.
const (
	CreateGatewayVirtualConnectionOptions_Type_Classic = "classic"
	CreateGatewayVirtualConnectionOptions_Type_Vpc     = "vpc"
)

// NewCreateGatewayVirtualConnectionOptions : Instantiate CreateGatewayVirtualConnectionOptions
func (*DirectLinkV1) NewCreateGatewayVirtualConnectionOptions(gatewayID string, name string, typeVar string) *CreateGatewayVirtualConnectionOptions {
	return &CreateGatewayVirtualConnectionOptions{
		GatewayID: core.StringPtr(gatewayID),
		Name:      core.StringPtr(name),
		Type:      core.StringPtr(typeVar),
	}
}

// SetGatewayID : Allow user to set GatewayID
func (options *CreateGatewayVirtualConnectionOptions) SetGatewayID(gatewayID string) *CreateGatewayVirtualConnectionOptions {
	options.GatewayID = core.StringPtr(gatewayID)
	return options
}

// SetName : Allow user to set Name
func (options *CreateGatewayVirtualConnectionOptions) SetName(name string) *CreateGatewayVirtualConnectionOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetType : Allow user to set Type
func (options *CreateGatewayVirtualConnectionOptions) SetType(typeVar string) *CreateGatewayVirtualConnectionOptions {
	options.Type = core.StringPtr(typeVar)
	return options
}

// SetNetworkID : Allow user to set NetworkID
func (options *CreateGatewayVirtualConnectionOptions) SetNetworkID(networkID string) *CreateGatewayVirtualConnectionOptions {
	options.NetworkID = core.StringPtr(networkID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateGatewayVirtualConnectionOptions) SetHeaders(param map[string]string) *CreateGatewayVirtualConnectionOptions {
	options.Headers = param
	return options
}

// CrossConnectRouter : Cross Connect Router details.
type CrossConnectRouter struct {
	// Array of capabilities for this router.
	Capabilities []string `json:"capabilities,omitempty"`

	// The name of the Router.
	RouterName *string `json:"router_name,omitempty"`

	// Count of existing Direct Link Dedicated gateways on this router for this account.
	TotalConnections *int64 `json:"total_connections,omitempty"`
}

// UnmarshalCrossConnectRouter unmarshals an instance of CrossConnectRouter from the specified map of raw messages.
func UnmarshalCrossConnectRouter(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(CrossConnectRouter)
	err = core.UnmarshalPrimitive(m, "capabilities", &obj.Capabilities)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "router_name", &obj.RouterName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_connections", &obj.TotalConnections)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// DeleteGatewayOptions : The DeleteGateway options.
type DeleteGatewayOptions struct {
	// Direct Link gateway identifier.
	ID *string `json:"id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteGatewayOptions : Instantiate DeleteGatewayOptions
func (*DirectLinkV1) NewDeleteGatewayOptions(id string) *DeleteGatewayOptions {
	return &DeleteGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteGatewayOptions) SetID(id string) *DeleteGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteGatewayOptions) SetHeaders(param map[string]string) *DeleteGatewayOptions {
	options.Headers = param
	return options
}

// DeleteGatewayVirtualConnectionOptions : The DeleteGatewayVirtualConnection options.
type DeleteGatewayVirtualConnectionOptions struct {
	// Direct Link gateway identifier.
	GatewayID *string `json:"gateway_id" validate:"required"`

	// The virtual connection identifier.
	ID *string `json:"id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteGatewayVirtualConnectionOptions : Instantiate DeleteGatewayVirtualConnectionOptions
func (*DirectLinkV1) NewDeleteGatewayVirtualConnectionOptions(gatewayID string, id string) *DeleteGatewayVirtualConnectionOptions {
	return &DeleteGatewayVirtualConnectionOptions{
		GatewayID: core.StringPtr(gatewayID),
		ID:        core.StringPtr(id),
	}
}

// SetGatewayID : Allow user to set GatewayID
func (options *DeleteGatewayVirtualConnectionOptions) SetGatewayID(gatewayID string) *DeleteGatewayVirtualConnectionOptions {
	options.GatewayID = core.StringPtr(gatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteGatewayVirtualConnectionOptions) SetID(id string) *DeleteGatewayVirtualConnectionOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteGatewayVirtualConnectionOptions) SetHeaders(param map[string]string) *DeleteGatewayVirtualConnectionOptions {
	options.Headers = param
	return options
}

// Gateway : gateway.
type Gateway struct {
	// Customer BGP ASN.
	BgpAsn *int64 `json:"bgp_asn" validate:"required"`

	// (DEPRECATED) BGP base CIDR is deprecated and no longer recognized by the Direct Link APIs.
	//
	// See bgp_cer_cidr and bgp_ibm_cidr fields instead for IP related information.
	//
	// Deprecated field bgp_base_cidr will be removed from the API specificiation after 15-MAR-2021.
	BgpBaseCidr *string `json:"bgp_base_cidr,omitempty"`

	// BGP customer edge router CIDR.
	BgpCerCidr *string `json:"bgp_cer_cidr,omitempty"`

	// IBM BGP ASN.
	BgpIbmAsn *int64 `json:"bgp_ibm_asn,omitempty"`

	// BGP IBM CIDR.
	BgpIbmCidr *string `json:"bgp_ibm_cidr,omitempty"`

	// Gateway BGP status. The list of enumerated values for this property may expand in the future. Code and processes
	// using this field  must tolerate unexpected values.
	BgpStatus *string `json:"bgp_status,omitempty"`

	// Carrier name.  Only set for type=dedicated gateways.
	CarrierName *string `json:"carrier_name,omitempty"`

	// Changes pending approval for provider managed Direct Link Connect gateways.
	ChangeRequest GatewayChangeRequestIntf `json:"change_request,omitempty"`

	// Reason for completion notice rejection.  Only included on type=dedicated gateways with a rejected completion notice.
	CompletionNoticeRejectReason *string `json:"completion_notice_reject_reason,omitempty"`

	// The date and time resource was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The CRN (Cloud Resource Name) of this gateway.
	Crn *string `json:"crn" validate:"required"`

	// Cross connect router.  Only included on type=dedicated gateways.
	CrossConnectRouter *string `json:"cross_connect_router,omitempty"`

	// Customer name.  Only set for type=dedicated gateways.
	CustomerName *string `json:"customer_name,omitempty"`

	// Gateways with global routing (`true`) can connect to networks outside their associated region.
	Global *bool `json:"global" validate:"required"`

	// The unique identifier of this gateway.
	ID *string `json:"id" validate:"required"`

	// Gateway link status.  Only included on type=dedicated gateways. The list of enumerated values for this property may
	// expand in the future. Code and processes using this field  must tolerate unexpected values.
	LinkStatus *string `json:"link_status,omitempty"`

	// Gateway location long name.
	LocationDisplayName *string `json:"location_display_name" validate:"required"`

	// Gateway location.
	LocationName *string `json:"location_name" validate:"required"`

	// MACsec configuration information.  For Dedicated Gateways with MACsec configured, return configuration information.
	// Contact IBM support for access to MACsec.
	MacsecConfig *GatewayMacsecConfig `json:"macsec_config,omitempty"`

	// Metered billing option.  When `true` gateway usage is billed per gigabyte.  When `false` there is no per gigabyte
	// usage charge, instead a flat rate is charged for the gateway.
	Metered *bool `json:"metered" validate:"required"`

	// The unique user-defined name for this gateway.
	Name *string `json:"name" validate:"required"`

	// Gateway operational status. The list of enumerated values for this property may expand in the future. Code and
	// processes using this field  must tolerate unexpected values.
	OperationalStatus *string `json:"operational_status" validate:"required"`

	// gateway port for type=connect gateways.
	Port *GatewayPort `json:"port,omitempty"`

	// Indicates whether gateway changes must be made via a provider portal.
	ProviderApiManaged *bool `json:"provider_api_managed,omitempty"`

	// Resource group reference.
	ResourceGroup *ResourceGroupReference `json:"resource_group,omitempty"`

	// Gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps" validate:"required"`

	// Gateway type. The list of enumerated values for this property may expand in the future. Code and processes using
	// this field  must tolerate unexpected values.
	Type *string `json:"type" validate:"required"`

	// VLAN allocated for this gateway.  Only set for type=connect gateways.
	Vlan *int64 `json:"vlan,omitempty"`
}

// Constants associated with the Gateway.BgpStatus property.
// Gateway BGP status. The list of enumerated values for this property may expand in the future. Code and processes
// using this field  must tolerate unexpected values.
const (
	Gateway_BgpStatus_Active      = "active"
	Gateway_BgpStatus_Connect     = "connect"
	Gateway_BgpStatus_Established = "established"
	Gateway_BgpStatus_Idle        = "idle"
)

// Constants associated with the Gateway.LinkStatus property.
// Gateway link status.  Only included on type=dedicated gateways. The list of enumerated values for this property may
// expand in the future. Code and processes using this field  must tolerate unexpected values.
const (
	Gateway_LinkStatus_Down = "down"
	Gateway_LinkStatus_Up   = "up"
)

// Constants associated with the Gateway.OperationalStatus property.
// Gateway operational status. The list of enumerated values for this property may expand in the future. Code and
// processes using this field  must tolerate unexpected values.
const (
	Gateway_OperationalStatus_AwaitingCompletionNotice = "awaiting_completion_notice"
	Gateway_OperationalStatus_AwaitingLoa              = "awaiting_loa"
	Gateway_OperationalStatus_CompletionNoticeApproved = "completion_notice_approved"
	Gateway_OperationalStatus_CompletionNoticeReceived = "completion_notice_received"
	Gateway_OperationalStatus_CompletionNoticeRejected = "completion_notice_rejected"
	Gateway_OperationalStatus_Configuring              = "configuring"
	Gateway_OperationalStatus_CreatePending            = "create_pending"
	Gateway_OperationalStatus_CreateRejected           = "create_rejected"
	Gateway_OperationalStatus_DeletePending            = "delete_pending"
	Gateway_OperationalStatus_LoaAccepted              = "loa_accepted"
	Gateway_OperationalStatus_LoaCreated               = "loa_created"
	Gateway_OperationalStatus_LoaRejected              = "loa_rejected"
	Gateway_OperationalStatus_Provisioned              = "provisioned"
)

// Constants associated with the Gateway.Type property.
// Gateway type. The list of enumerated values for this property may expand in the future. Code and processes using this
// field  must tolerate unexpected values.
const (
	Gateway_Type_Connect   = "connect"
	Gateway_Type_Dedicated = "dedicated"
)

// UnmarshalGateway unmarshals an instance of Gateway from the specified map of raw messages.
func UnmarshalGateway(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(Gateway)
	err = core.UnmarshalPrimitive(m, "bgp_asn", &obj.BgpAsn)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_base_cidr", &obj.BgpBaseCidr)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_cer_cidr", &obj.BgpCerCidr)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_ibm_asn", &obj.BgpIbmAsn)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_ibm_cidr", &obj.BgpIbmCidr)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_status", &obj.BgpStatus)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "carrier_name", &obj.CarrierName)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "change_request", &obj.ChangeRequest, UnmarshalGatewayChangeRequest)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "completion_notice_reject_reason", &obj.CompletionNoticeRejectReason)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.Crn)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "cross_connect_router", &obj.CrossConnectRouter)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "customer_name", &obj.CustomerName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "global", &obj.Global)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "link_status", &obj.LinkStatus)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "location_display_name", &obj.LocationDisplayName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "location_name", &obj.LocationName)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "macsec_config", &obj.MacsecConfig, UnmarshalGatewayMacsecConfig)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "metered", &obj.Metered)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "operational_status", &obj.OperationalStatus)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "port", &obj.Port, UnmarshalGatewayPort)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "provider_api_managed", &obj.ProviderApiManaged)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "speed_mbps", &obj.SpeedMbps)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "vlan", &obj.Vlan)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayActionTemplateUpdatesItem : GatewayActionTemplateUpdatesItem struct
// Models which "extend" this model:
// - GatewayActionTemplateUpdatesItemGatewayClientSpeedUpdate
type GatewayActionTemplateUpdatesItem struct {
	// New gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps,omitempty"`
}

func (*GatewayActionTemplateUpdatesItem) isaGatewayActionTemplateUpdatesItem() bool {
	return true
}

type GatewayActionTemplateUpdatesItemIntf interface {
	isaGatewayActionTemplateUpdatesItem() bool
}

// UnmarshalGatewayActionTemplateUpdatesItem unmarshals an instance of GatewayActionTemplateUpdatesItem from the specified map of raw messages.
func UnmarshalGatewayActionTemplateUpdatesItem(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayActionTemplateUpdatesItem)
	err = core.UnmarshalPrimitive(m, "speed_mbps", &obj.SpeedMbps)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayChangeRequest : GatewayChangeRequest struct
// Models which "extend" this model:
// - GatewayChangeRequestGatewayClientGatewayCreate
// - GatewayChangeRequestGatewayClientGatewayDelete
// - GatewayChangeRequestGatewayClientGatewayUpdateAttributes
type GatewayChangeRequest struct {
	// type of gateway change request.
	Type *string `json:"type,omitempty"`

	// array of pending updates.
	Updates []GatewayChangeRequestUpdatesItemIntf `json:"updates,omitempty"`
}

// Constants associated with the GatewayChangeRequest.Type property.
// type of gateway change request.
const (
	GatewayChangeRequest_Type_CreateGateway = "create_gateway"
)

func (*GatewayChangeRequest) isaGatewayChangeRequest() bool {
	return true
}

type GatewayChangeRequestIntf interface {
	isaGatewayChangeRequest() bool
}

// UnmarshalGatewayChangeRequest unmarshals an instance of GatewayChangeRequest from the specified map of raw messages.
func UnmarshalGatewayChangeRequest(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayChangeRequest)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "updates", &obj.Updates, UnmarshalGatewayChangeRequestUpdatesItem)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItem : GatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItem struct
// Models which "extend" this model:
// - GatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItemGatewayClientSpeedUpdate
type GatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItem struct {
	// New gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps,omitempty"`
}

func (*GatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItem) isaGatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItem() bool {
	return true
}

type GatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItemIntf interface {
	isaGatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItem() bool
}

// UnmarshalGatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItem unmarshals an instance of GatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItem from the specified map of raw messages.
func UnmarshalGatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItem(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItem)
	err = core.UnmarshalPrimitive(m, "speed_mbps", &obj.SpeedMbps)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayChangeRequestUpdatesItem : GatewayChangeRequestUpdatesItem struct
// Models which "extend" this model:
// - GatewayChangeRequestUpdatesItemGatewayClientSpeedUpdate
type GatewayChangeRequestUpdatesItem struct {
	// New gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps,omitempty"`
}

func (*GatewayChangeRequestUpdatesItem) isaGatewayChangeRequestUpdatesItem() bool {
	return true
}

type GatewayChangeRequestUpdatesItemIntf interface {
	isaGatewayChangeRequestUpdatesItem() bool
}

// UnmarshalGatewayChangeRequestUpdatesItem unmarshals an instance of GatewayChangeRequestUpdatesItem from the specified map of raw messages.
func UnmarshalGatewayChangeRequestUpdatesItem(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayChangeRequestUpdatesItem)
	err = core.UnmarshalPrimitive(m, "speed_mbps", &obj.SpeedMbps)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayCollection : GatewayCollection struct
type GatewayCollection struct {
	// Collection of Direct Link gateways.
	Gateways []Gateway `json:"gateways" validate:"required"`
}

// UnmarshalGatewayCollection unmarshals an instance of GatewayCollection from the specified map of raw messages.
func UnmarshalGatewayCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayCollection)
	err = core.UnmarshalModel(m, "gateways", &obj.Gateways, UnmarshalGateway)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayMacsecConfig : MACsec configuration information.  For Dedicated Gateways with MACsec configured, return configuration information.
// Contact IBM support for access to MACsec.
type GatewayMacsecConfig struct {
	// Indicate whether MACsec should currently be active (true) or inactive (false) for a MACsec enabled gateway.   To be
	// MACsec enabled a `macsec_config` must be specified at gateway create time.
	Active *bool `json:"active" validate:"required"`

	// Active connectivity association key.
	//
	// During normal operation `active_cak` will match the desired `primary_cak`.  During CAK changes this field can be
	// used to indicate which key is currently active on the gateway.
	ActiveCak *GatewayMacsecConfigActiveCak `json:"active_cak,omitempty"`

	// SAK cipher suite.
	CipherSuite *string `json:"cipher_suite,omitempty"`

	// confidentiality offset.
	ConfidentialityOffset *int64 `json:"confidentiality_offset,omitempty"`

	// cryptographic algorithm.
	CryptographicAlgorithm *string `json:"cryptographic_algorithm,omitempty"`

	// fallback connectivity association key.
	FallbackCak *GatewayMacsecConfigFallbackCak `json:"fallback_cak,omitempty"`

	// key server priority.
	KeyServerPriority *int64 `json:"key_server_priority,omitempty"`

	// desired primary connectivity association key.
	PrimaryCak *GatewayMacsecConfigPrimaryCak `json:"primary_cak" validate:"required"`

	// Secure Association Key (SAK) expiry time in seconds.
	SakExpiryTime *int64 `json:"sak_expiry_time,omitempty"`

	// Packets without MACsec headers are dropped when security_policy is `must_secure`.
	SecurityPolicy *string `json:"security_policy,omitempty"`

	// Current status of MACsec on this gateway.
	//
	// Status 'offline' is returned during gateway creation and deletion.
	Status *string `json:"status" validate:"required"`

	// replay protection window size.
	WindowSize *int64 `json:"window_size,omitempty"`
}

// Constants associated with the GatewayMacsecConfig.CipherSuite property.
// SAK cipher suite.
const (
	GatewayMacsecConfig_CipherSuite_GcmAesXpn256 = "gcm_aes_xpn_256"
)

// Constants associated with the GatewayMacsecConfig.CryptographicAlgorithm property.
// cryptographic algorithm.
const (
	GatewayMacsecConfig_CryptographicAlgorithm_Aes256Cmac = "aes_256_cmac"
)

// Constants associated with the GatewayMacsecConfig.SecurityPolicy property.
// Packets without MACsec headers are dropped when security_policy is `must_secure`.
const (
	GatewayMacsecConfig_SecurityPolicy_MustSecure = "must_secure"
)

// Constants associated with the GatewayMacsecConfig.Status property.
// Current status of MACsec on this gateway.
//
// Status 'offline' is returned during gateway creation and deletion.
const (
	GatewayMacsecConfig_Status_Init    = "init"
	GatewayMacsecConfig_Status_Offline = "offline"
	GatewayMacsecConfig_Status_Pending = "pending"
	GatewayMacsecConfig_Status_Secured = "secured"
)

// UnmarshalGatewayMacsecConfig unmarshals an instance of GatewayMacsecConfig from the specified map of raw messages.
func UnmarshalGatewayMacsecConfig(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayMacsecConfig)
	err = core.UnmarshalPrimitive(m, "active", &obj.Active)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "active_cak", &obj.ActiveCak, UnmarshalGatewayMacsecConfigActiveCak)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "cipher_suite", &obj.CipherSuite)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "confidentiality_offset", &obj.ConfidentialityOffset)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "cryptographic_algorithm", &obj.CryptographicAlgorithm)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "fallback_cak", &obj.FallbackCak, UnmarshalGatewayMacsecConfigFallbackCak)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "key_server_priority", &obj.KeyServerPriority)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "primary_cak", &obj.PrimaryCak, UnmarshalGatewayMacsecConfigPrimaryCak)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "sak_expiry_time", &obj.SakExpiryTime)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "security_policy", &obj.SecurityPolicy)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "window_size", &obj.WindowSize)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayMacsecConfigActiveCak : Active connectivity association key.
//
// During normal operation `active_cak` will match the desired `primary_cak`.  During CAK changes this field can be used
// to indicate which key is currently active on the gateway.
type GatewayMacsecConfigActiveCak struct {
	// connectivity association key crn.
	Crn *string `json:"crn" validate:"required"`

	// connectivity association key status.
	Status *string `json:"status" validate:"required"`
}

// UnmarshalGatewayMacsecConfigActiveCak unmarshals an instance of GatewayMacsecConfigActiveCak from the specified map of raw messages.
func UnmarshalGatewayMacsecConfigActiveCak(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayMacsecConfigActiveCak)
	err = core.UnmarshalPrimitive(m, "crn", &obj.Crn)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayMacsecConfigFallbackCak : fallback connectivity association key.
type GatewayMacsecConfigFallbackCak struct {
	// connectivity association key crn.
	Crn *string `json:"crn" validate:"required"`

	// connectivity association key status.
	Status *string `json:"status" validate:"required"`
}

// UnmarshalGatewayMacsecConfigFallbackCak unmarshals an instance of GatewayMacsecConfigFallbackCak from the specified map of raw messages.
func UnmarshalGatewayMacsecConfigFallbackCak(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayMacsecConfigFallbackCak)
	err = core.UnmarshalPrimitive(m, "crn", &obj.Crn)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayMacsecConfigPatchTemplate : MACsec configuration information.  When patching any macsec_config fields, no other fields may be specified in the
// patch request.  Contact IBM support for access to MACsec.
//
// A MACsec config cannot be added to a gateway created without MACsec.
type GatewayMacsecConfigPatchTemplate struct {
	// Indicate whether MACsec protection should be active (true) or inactive (false) for this MACsec enabled gateway.
	Active *bool `json:"active,omitempty"`

	// Fallback connectivity association key.
	//
	// MACsec keys must be type=standard with key name lengths between 2 to 64 inclusive and contain only characters
	// [a-fA-F0-9].
	// The key material must be exactly 64 characters in length and contain only [a-fA-F0-9].
	//
	// To clear the optional `fallback_cak` field patch its crn to `""`.
	//
	// A gateway's `fallback_cak` crn cannot match its `primary_cak` crn.
	FallbackCak *GatewayMacsecConfigPatchTemplateFallbackCak `json:"fallback_cak,omitempty"`

	// Desired primary connectivity association key.
	//
	// MACsec keys must be type=standard with key name lengths between 2 to 64 inclusive and contain only characters
	// [a-fA-F0-9].
	// The key material must be exactly 64 characters in length and contain only [a-fA-F0-9].
	//
	// A gateway's `primary_cak` crn cannot match its `fallback_cak` crn.
	PrimaryCak *GatewayMacsecConfigPatchTemplatePrimaryCak `json:"primary_cak,omitempty"`

	// replay protection window size.
	WindowSize *int64 `json:"window_size,omitempty"`
}

// UnmarshalGatewayMacsecConfigPatchTemplate unmarshals an instance of GatewayMacsecConfigPatchTemplate from the specified map of raw messages.
func UnmarshalGatewayMacsecConfigPatchTemplate(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayMacsecConfigPatchTemplate)
	err = core.UnmarshalPrimitive(m, "active", &obj.Active)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "fallback_cak", &obj.FallbackCak, UnmarshalGatewayMacsecConfigPatchTemplateFallbackCak)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "primary_cak", &obj.PrimaryCak, UnmarshalGatewayMacsecConfigPatchTemplatePrimaryCak)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "window_size", &obj.WindowSize)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayMacsecConfigPatchTemplateFallbackCak : Fallback connectivity association key.
//
// MACsec keys must be type=standard with key name lengths between 2 to 64 inclusive and contain only characters
// [a-fA-F0-9]. The key material must be exactly 64 characters in length and contain only [a-fA-F0-9].
//
// To clear the optional `fallback_cak` field patch its crn to `""`.
//
// A gateway's `fallback_cak` crn cannot match its `primary_cak` crn.
type GatewayMacsecConfigPatchTemplateFallbackCak struct {
	// connectivity association key crn.
	Crn *string `json:"crn" validate:"required"`
}

// NewGatewayMacsecConfigPatchTemplateFallbackCak : Instantiate GatewayMacsecConfigPatchTemplateFallbackCak (Generic Model Constructor)
func (*DirectLinkV1) NewGatewayMacsecConfigPatchTemplateFallbackCak(crn string) (model *GatewayMacsecConfigPatchTemplateFallbackCak, err error) {
	model = &GatewayMacsecConfigPatchTemplateFallbackCak{
		Crn: core.StringPtr(crn),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalGatewayMacsecConfigPatchTemplateFallbackCak unmarshals an instance of GatewayMacsecConfigPatchTemplateFallbackCak from the specified map of raw messages.
func UnmarshalGatewayMacsecConfigPatchTemplateFallbackCak(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayMacsecConfigPatchTemplateFallbackCak)
	err = core.UnmarshalPrimitive(m, "crn", &obj.Crn)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayMacsecConfigPatchTemplatePrimaryCak : Desired primary connectivity association key.
//
// MACsec keys must be type=standard with key name lengths between 2 to 64 inclusive and contain only characters
// [a-fA-F0-9]. The key material must be exactly 64 characters in length and contain only [a-fA-F0-9].
//
// A gateway's `primary_cak` crn cannot match its `fallback_cak` crn.
type GatewayMacsecConfigPatchTemplatePrimaryCak struct {
	// connectivity association key crn.
	Crn *string `json:"crn" validate:"required"`
}

// NewGatewayMacsecConfigPatchTemplatePrimaryCak : Instantiate GatewayMacsecConfigPatchTemplatePrimaryCak (Generic Model Constructor)
func (*DirectLinkV1) NewGatewayMacsecConfigPatchTemplatePrimaryCak(crn string) (model *GatewayMacsecConfigPatchTemplatePrimaryCak, err error) {
	model = &GatewayMacsecConfigPatchTemplatePrimaryCak{
		Crn: core.StringPtr(crn),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalGatewayMacsecConfigPatchTemplatePrimaryCak unmarshals an instance of GatewayMacsecConfigPatchTemplatePrimaryCak from the specified map of raw messages.
func UnmarshalGatewayMacsecConfigPatchTemplatePrimaryCak(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayMacsecConfigPatchTemplatePrimaryCak)
	err = core.UnmarshalPrimitive(m, "crn", &obj.Crn)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayMacsecConfigPrimaryCak : desired primary connectivity association key.
type GatewayMacsecConfigPrimaryCak struct {
	// connectivity association key crn.
	Crn *string `json:"crn" validate:"required"`

	// connectivity association key status.
	Status *string `json:"status" validate:"required"`
}

// UnmarshalGatewayMacsecConfigPrimaryCak unmarshals an instance of GatewayMacsecConfigPrimaryCak from the specified map of raw messages.
func UnmarshalGatewayMacsecConfigPrimaryCak(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayMacsecConfigPrimaryCak)
	err = core.UnmarshalPrimitive(m, "crn", &obj.Crn)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayMacsecConfigTemplate : MACsec configuration information.  Contact IBM support for access to MACsec.
type GatewayMacsecConfigTemplate struct {
	// Indicate whether MACsec protection should be active (true) or inactive (false) for this MACsec enabled gateway.
	Active *bool `json:"active" validate:"required"`

	// Fallback connectivity association key.
	//
	// The `fallback_cak` crn cannot match the `primary_cak` crn.
	// MACsec keys must be type=standard with key name lengths between 2 to 64 inclusive and contain only characters
	// [a-fA-F0-9].
	// The key material must be exactly 64 characters in length and contain only [a-fA-F0-9].
	FallbackCak *GatewayMacsecConfigTemplateFallbackCak `json:"fallback_cak,omitempty"`

	// Desired primary connectivity association key.
	//
	// MACsec keys must be type=standard with key name lengths between 2 to 64 inclusive and contain only characters
	// [a-fA-F0-9].
	// The key material must be exactly 64 characters in length and contain only [a-fA-F0-9].
	PrimaryCak *GatewayMacsecConfigTemplatePrimaryCak `json:"primary_cak" validate:"required"`

	// replay protection window size.
	WindowSize *int64 `json:"window_size,omitempty"`
}

// NewGatewayMacsecConfigTemplate : Instantiate GatewayMacsecConfigTemplate (Generic Model Constructor)
func (*DirectLinkV1) NewGatewayMacsecConfigTemplate(active bool, primaryCak *GatewayMacsecConfigTemplatePrimaryCak) (model *GatewayMacsecConfigTemplate, err error) {
	model = &GatewayMacsecConfigTemplate{
		Active:     core.BoolPtr(active),
		PrimaryCak: primaryCak,
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalGatewayMacsecConfigTemplate unmarshals an instance of GatewayMacsecConfigTemplate from the specified map of raw messages.
func UnmarshalGatewayMacsecConfigTemplate(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayMacsecConfigTemplate)
	err = core.UnmarshalPrimitive(m, "active", &obj.Active)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "fallback_cak", &obj.FallbackCak, UnmarshalGatewayMacsecConfigTemplateFallbackCak)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "primary_cak", &obj.PrimaryCak, UnmarshalGatewayMacsecConfigTemplatePrimaryCak)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "window_size", &obj.WindowSize)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayMacsecConfigTemplateFallbackCak : Fallback connectivity association key.
//
// The `fallback_cak` crn cannot match the `primary_cak` crn. MACsec keys must be type=standard with key name lengths
// between 2 to 64 inclusive and contain only characters [a-fA-F0-9]. The key material must be exactly 64 characters in
// length and contain only [a-fA-F0-9].
type GatewayMacsecConfigTemplateFallbackCak struct {
	// connectivity association key crn.
	Crn *string `json:"crn" validate:"required"`
}

// NewGatewayMacsecConfigTemplateFallbackCak : Instantiate GatewayMacsecConfigTemplateFallbackCak (Generic Model Constructor)
func (*DirectLinkV1) NewGatewayMacsecConfigTemplateFallbackCak(crn string) (model *GatewayMacsecConfigTemplateFallbackCak, err error) {
	model = &GatewayMacsecConfigTemplateFallbackCak{
		Crn: core.StringPtr(crn),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalGatewayMacsecConfigTemplateFallbackCak unmarshals an instance of GatewayMacsecConfigTemplateFallbackCak from the specified map of raw messages.
func UnmarshalGatewayMacsecConfigTemplateFallbackCak(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayMacsecConfigTemplateFallbackCak)
	err = core.UnmarshalPrimitive(m, "crn", &obj.Crn)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayMacsecConfigTemplatePrimaryCak : Desired primary connectivity association key.
//
// MACsec keys must be type=standard with key name lengths between 2 to 64 inclusive and contain only characters
// [a-fA-F0-9]. The key material must be exactly 64 characters in length and contain only [a-fA-F0-9].
type GatewayMacsecConfigTemplatePrimaryCak struct {
	// connectivity association key crn.
	Crn *string `json:"crn" validate:"required"`
}

// NewGatewayMacsecConfigTemplatePrimaryCak : Instantiate GatewayMacsecConfigTemplatePrimaryCak (Generic Model Constructor)
func (*DirectLinkV1) NewGatewayMacsecConfigTemplatePrimaryCak(crn string) (model *GatewayMacsecConfigTemplatePrimaryCak, err error) {
	model = &GatewayMacsecConfigTemplatePrimaryCak{
		Crn: core.StringPtr(crn),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalGatewayMacsecConfigTemplatePrimaryCak unmarshals an instance of GatewayMacsecConfigTemplatePrimaryCak from the specified map of raw messages.
func UnmarshalGatewayMacsecConfigTemplatePrimaryCak(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayMacsecConfigTemplatePrimaryCak)
	err = core.UnmarshalPrimitive(m, "crn", &obj.Crn)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayPort : gateway port for type=connect gateways.
type GatewayPort struct {
	// Port Identifier.
	ID *string `json:"id" validate:"required"`
}

// UnmarshalGatewayPort unmarshals an instance of GatewayPort from the specified map of raw messages.
func UnmarshalGatewayPort(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayPort)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayPortIdentity : Select Port Label for new type=connect gateway.
type GatewayPortIdentity struct {
	// port id.
	ID *string `json:"id" validate:"required"`
}

// NewGatewayPortIdentity : Instantiate GatewayPortIdentity (Generic Model Constructor)
func (*DirectLinkV1) NewGatewayPortIdentity(id string) (model *GatewayPortIdentity, err error) {
	model = &GatewayPortIdentity{
		ID: core.StringPtr(id),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalGatewayPortIdentity unmarshals an instance of GatewayPortIdentity from the specified map of raw messages.
func UnmarshalGatewayPortIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayPortIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayStatistic : Gateway statistics.  Currently data retrieval is only supported for MACsec configurations.
type GatewayStatistic struct {
	// Date and time data was collected.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// statistics output.
	Data *string `json:"data" validate:"required"`

	// statistic type.
	Type *string `json:"type" validate:"required"`
}

// Constants associated with the GatewayStatistic.Type property.
// statistic type.
const (
	GatewayStatistic_Type_MacsecMkaSession    = "macsec_mka_session"
	GatewayStatistic_Type_MacsecMkaStatistics = "macsec_mka_statistics"
	GatewayStatistic_Type_MacsecPolicy        = "macsec_policy"
)

// UnmarshalGatewayStatistic unmarshals an instance of GatewayStatistic from the specified map of raw messages.
func UnmarshalGatewayStatistic(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayStatistic)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "data", &obj.Data)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayStatisticCollection : gateway statistics.
type GatewayStatisticCollection struct {
	// Collection of gateway statistics.
	Statistics []GatewayStatistic `json:"statistics" validate:"required"`
}

// UnmarshalGatewayStatisticCollection unmarshals an instance of GatewayStatisticCollection from the specified map of raw messages.
func UnmarshalGatewayStatisticCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayStatisticCollection)
	err = core.UnmarshalModel(m, "statistics", &obj.Statistics, UnmarshalGatewayStatistic)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayTemplate : Create gateway template.
// Models which "extend" this model:
// - GatewayTemplateGatewayTypeDedicatedTemplate
// - GatewayTemplateGatewayTypeConnectTemplate
type GatewayTemplate struct {
	// BGP ASN.
	BgpAsn *int64 `json:"bgp_asn" validate:"required"`

	// (DEPRECATED) BGP base CIDR.
	//
	// Field is deprecated.  See bgp_ibm_cidr and bgp_cer_cidr for details on how to create a gateway using either
	// automatic or explicit IP assignment.  Any bgp_base_cidr value set will be ignored.
	//
	// Deprecated field bgp_base_cidr will be removed from the API specificiation after 15-MAR-2021.
	BgpBaseCidr *string `json:"bgp_base_cidr,omitempty"`

	// BGP customer edge router CIDR.
	//
	// For auto IP assignment, omit bgp_cer_cidr and bgp_ibm_cidr.  IBM will automatically select values for bgp_cer_cidr
	// and bgp_ibm_cidr.
	//
	// For explicit IP assignment set a valid bgp_cer_cidr and bgp_ibm_cidr CIDR, the value must reside in one of
	// "10.254.0.0/16", "172.16.0.0/12", "192.168.0.0/16", "169.254.0.0/16" or an owned public CIDR.  bgp_cer_cidr and
	// bgp_ibm_cidr must have matching network and subnet mask values.
	BgpCerCidr *string `json:"bgp_cer_cidr,omitempty"`

	// BGP IBM CIDR.
	//
	// For auto IP assignment, omit bgp_cer_cidr and bgp_ibm_cidr.  IBM will automatically select values for bgp_cer_cidr
	// and bgp_ibm_cidr.
	//
	// For explicit IP assignment set a valid bgp_cer_cidr and bgp_ibm_cidr CIDR, the value must reside in one of
	// "10.254.0.0/16", "172.16.0.0/12", "192.168.0.0/16", "169.254.0.0/16" or an owned public CIDR.  bgp_cer_cidr and
	// bgp_ibm_cidr must have matching network and subnet mask values.
	BgpIbmCidr *string `json:"bgp_ibm_cidr,omitempty"`

	// Gateways with global routing (`true`) can connect to networks outside their associated region.
	Global *bool `json:"global" validate:"required"`

	// Metered billing option.  When `true` gateway usage is billed per gigabyte.  When `false` there is no per gigabyte
	// usage charge, instead a flat rate is charged for the gateway.
	Metered *bool `json:"metered" validate:"required"`

	// The unique user-defined name for this gateway.
	Name *string `json:"name" validate:"required"`

	// Resource group for this resource. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup *ResourceGroupIdentity `json:"resource_group,omitempty"`

	// Gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps" validate:"required"`

	// Gateway type.
	Type *string `json:"type" validate:"required"`

	// Carrier name.
	CarrierName *string `json:"carrier_name,omitempty"`

	// Cross connect router.
	CrossConnectRouter *string `json:"cross_connect_router,omitempty"`

	// Customer name.
	CustomerName *string `json:"customer_name,omitempty"`

	// Gateway location.
	LocationName *string `json:"location_name,omitempty"`

	// MACsec configuration information.  Contact IBM support for access to MACsec.
	MacsecConfig *GatewayMacsecConfigTemplate `json:"macsec_config,omitempty"`

	// Select Port Label for new type=connect gateway.
	Port *GatewayPortIdentity `json:"port,omitempty"`
}

// Constants associated with the GatewayTemplate.Type property.
// Gateway type.
const (
	GatewayTemplate_Type_Connect   = "connect"
	GatewayTemplate_Type_Dedicated = "dedicated"
)

func (*GatewayTemplate) isaGatewayTemplate() bool {
	return true
}

type GatewayTemplateIntf interface {
	isaGatewayTemplate() bool
}

// UnmarshalGatewayTemplate unmarshals an instance of GatewayTemplate from the specified map of raw messages.
func UnmarshalGatewayTemplate(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayTemplate)
	err = core.UnmarshalPrimitive(m, "bgp_asn", &obj.BgpAsn)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_base_cidr", &obj.BgpBaseCidr)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_cer_cidr", &obj.BgpCerCidr)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_ibm_cidr", &obj.BgpIbmCidr)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "global", &obj.Global)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "metered", &obj.Metered)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "speed_mbps", &obj.SpeedMbps)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "carrier_name", &obj.CarrierName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "cross_connect_router", &obj.CrossConnectRouter)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "customer_name", &obj.CustomerName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "location_name", &obj.LocationName)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "macsec_config", &obj.MacsecConfig, UnmarshalGatewayMacsecConfigTemplate)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "port", &obj.Port, UnmarshalGatewayPortIdentity)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayVirtualConnection : Virtual connection.
type GatewayVirtualConnection struct {
	// The date and time resource was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// The unique identifier for this virtual connection.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this virtual connection.  Virtual connection names are unique within a gateway.  This is
	// the name of the virtual connection itself, the network being connected may have its own name attribute.
	Name *string `json:"name" validate:"required"`

	// For virtual connections across two different IBM Cloud Accounts network_account indicates the account that owns the
	// target network.
	NetworkAccount *string `json:"network_account,omitempty"`

	// Unique identifier of the target network.  For type=vpc virtual connections this is the CRN of the target VPC.  This
	// field does not apply to type=classic connections.
	NetworkID *string `json:"network_id,omitempty"`

	// Status of the virtual connection.
	//
	// The list of enumerated values for this property may expand in the future. Code and processes using this field  must
	// tolerate unexpected values.
	Status *string `json:"status" validate:"required"`

	// Virtual connection type.
	//
	// The list of enumerated values for this property may expand in the future. Code and processes using this field  must
	// tolerate unexpected values.
	Type *string `json:"type" validate:"required"`
}

// Constants associated with the GatewayVirtualConnection.Status property.
// Status of the virtual connection.
//
// The list of enumerated values for this property may expand in the future. Code and processes using this field  must
// tolerate unexpected values.
const (
	GatewayVirtualConnection_Status_ApprovalPending          = "approval_pending"
	GatewayVirtualConnection_Status_Attached                 = "attached"
	GatewayVirtualConnection_Status_Deleting                 = "deleting"
	GatewayVirtualConnection_Status_DetachedByNetwork        = "detached_by_network"
	GatewayVirtualConnection_Status_DetachedByNetworkPending = "detached_by_network_pending"
	GatewayVirtualConnection_Status_Expired                  = "expired"
	GatewayVirtualConnection_Status_Pending                  = "pending"
	GatewayVirtualConnection_Status_Rejected                 = "rejected"
)

// Constants associated with the GatewayVirtualConnection.Type property.
// Virtual connection type.
//
// The list of enumerated values for this property may expand in the future. Code and processes using this field  must
// tolerate unexpected values.
const (
	GatewayVirtualConnection_Type_Classic = "classic"
	GatewayVirtualConnection_Type_Vpc     = "vpc"
)

// UnmarshalGatewayVirtualConnection unmarshals an instance of GatewayVirtualConnection from the specified map of raw messages.
func UnmarshalGatewayVirtualConnection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayVirtualConnection)
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "network_account", &obj.NetworkAccount)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "network_id", &obj.NetworkID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayVirtualConnectionCollection : Virtual connection collection.
type GatewayVirtualConnectionCollection struct {
	// virtual connection array.
	VirtualConnections []GatewayVirtualConnection `json:"virtual_connections" validate:"required"`
}

// UnmarshalGatewayVirtualConnectionCollection unmarshals an instance of GatewayVirtualConnectionCollection from the specified map of raw messages.
func UnmarshalGatewayVirtualConnectionCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayVirtualConnectionCollection)
	err = core.UnmarshalModel(m, "virtual_connections", &obj.VirtualConnections, UnmarshalGatewayVirtualConnection)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GetGatewayOptions : The GetGateway options.
type GetGatewayOptions struct {
	// Direct Link gateway identifier.
	ID *string `json:"id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetGatewayOptions : Instantiate GetGatewayOptions
func (*DirectLinkV1) NewGetGatewayOptions(id string) *GetGatewayOptions {
	return &GetGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetGatewayOptions) SetID(id string) *GetGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetGatewayOptions) SetHeaders(param map[string]string) *GetGatewayOptions {
	options.Headers = param
	return options
}

// GetGatewayStatisticsOptions : The GetGatewayStatistics options.
type GetGatewayStatisticsOptions struct {
	// Direct Link Dedicated gateway identifier.
	ID *string `json:"id" validate:"required"`

	// specify statistic to retrieve.
	Type *string `json:"type" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the GetGatewayStatisticsOptions.Type property.
// specify statistic to retrieve.
const (
	GetGatewayStatisticsOptions_Type_MacsecMka      = "macsec_mka"
	GetGatewayStatisticsOptions_Type_MacsecSecurity = "macsec_security"
)

// NewGetGatewayStatisticsOptions : Instantiate GetGatewayStatisticsOptions
func (*DirectLinkV1) NewGetGatewayStatisticsOptions(id string, typeVar string) *GetGatewayStatisticsOptions {
	return &GetGatewayStatisticsOptions{
		ID:   core.StringPtr(id),
		Type: core.StringPtr(typeVar),
	}
}

// SetID : Allow user to set ID
func (options *GetGatewayStatisticsOptions) SetID(id string) *GetGatewayStatisticsOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetType : Allow user to set Type
func (options *GetGatewayStatisticsOptions) SetType(typeVar string) *GetGatewayStatisticsOptions {
	options.Type = core.StringPtr(typeVar)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetGatewayStatisticsOptions) SetHeaders(param map[string]string) *GetGatewayStatisticsOptions {
	options.Headers = param
	return options
}

// GetGatewayVirtualConnectionOptions : The GetGatewayVirtualConnection options.
type GetGatewayVirtualConnectionOptions struct {
	// Direct Link gateway identifier.
	GatewayID *string `json:"gateway_id" validate:"required"`

	// The virtual connection identifier.
	ID *string `json:"id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetGatewayVirtualConnectionOptions : Instantiate GetGatewayVirtualConnectionOptions
func (*DirectLinkV1) NewGetGatewayVirtualConnectionOptions(gatewayID string, id string) *GetGatewayVirtualConnectionOptions {
	return &GetGatewayVirtualConnectionOptions{
		GatewayID: core.StringPtr(gatewayID),
		ID:        core.StringPtr(id),
	}
}

// SetGatewayID : Allow user to set GatewayID
func (options *GetGatewayVirtualConnectionOptions) SetGatewayID(gatewayID string) *GetGatewayVirtualConnectionOptions {
	options.GatewayID = core.StringPtr(gatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *GetGatewayVirtualConnectionOptions) SetID(id string) *GetGatewayVirtualConnectionOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetGatewayVirtualConnectionOptions) SetHeaders(param map[string]string) *GetGatewayVirtualConnectionOptions {
	options.Headers = param
	return options
}

// GetPortOptions : The GetPort options.
type GetPortOptions struct {
	// The port identifier.
	ID *string `json:"id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetPortOptions : Instantiate GetPortOptions
func (*DirectLinkV1) NewGetPortOptions(id string) *GetPortOptions {
	return &GetPortOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetPortOptions) SetID(id string) *GetPortOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetPortOptions) SetHeaders(param map[string]string) *GetPortOptions {
	options.Headers = param
	return options
}

// ListGatewayCompletionNoticeOptions : The ListGatewayCompletionNotice options.
type ListGatewayCompletionNoticeOptions struct {
	// Direct Link Dedicated gateway identifier.
	ID *string `json:"id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListGatewayCompletionNoticeOptions : Instantiate ListGatewayCompletionNoticeOptions
func (*DirectLinkV1) NewListGatewayCompletionNoticeOptions(id string) *ListGatewayCompletionNoticeOptions {
	return &ListGatewayCompletionNoticeOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *ListGatewayCompletionNoticeOptions) SetID(id string) *ListGatewayCompletionNoticeOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListGatewayCompletionNoticeOptions) SetHeaders(param map[string]string) *ListGatewayCompletionNoticeOptions {
	options.Headers = param
	return options
}

// ListGatewayLetterOfAuthorizationOptions : The ListGatewayLetterOfAuthorization options.
type ListGatewayLetterOfAuthorizationOptions struct {
	// Direct Link Dedicated gateway identifier.
	ID *string `json:"id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListGatewayLetterOfAuthorizationOptions : Instantiate ListGatewayLetterOfAuthorizationOptions
func (*DirectLinkV1) NewListGatewayLetterOfAuthorizationOptions(id string) *ListGatewayLetterOfAuthorizationOptions {
	return &ListGatewayLetterOfAuthorizationOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *ListGatewayLetterOfAuthorizationOptions) SetID(id string) *ListGatewayLetterOfAuthorizationOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListGatewayLetterOfAuthorizationOptions) SetHeaders(param map[string]string) *ListGatewayLetterOfAuthorizationOptions {
	options.Headers = param
	return options
}

// ListGatewayVirtualConnectionsOptions : The ListGatewayVirtualConnections options.
type ListGatewayVirtualConnectionsOptions struct {
	// Direct Link gateway identifier.
	GatewayID *string `json:"gateway_id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListGatewayVirtualConnectionsOptions : Instantiate ListGatewayVirtualConnectionsOptions
func (*DirectLinkV1) NewListGatewayVirtualConnectionsOptions(gatewayID string) *ListGatewayVirtualConnectionsOptions {
	return &ListGatewayVirtualConnectionsOptions{
		GatewayID: core.StringPtr(gatewayID),
	}
}

// SetGatewayID : Allow user to set GatewayID
func (options *ListGatewayVirtualConnectionsOptions) SetGatewayID(gatewayID string) *ListGatewayVirtualConnectionsOptions {
	options.GatewayID = core.StringPtr(gatewayID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListGatewayVirtualConnectionsOptions) SetHeaders(param map[string]string) *ListGatewayVirtualConnectionsOptions {
	options.Headers = param
	return options
}

// ListGatewaysOptions : The ListGateways options.
type ListGatewaysOptions struct {

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListGatewaysOptions : Instantiate ListGatewaysOptions
func (*DirectLinkV1) NewListGatewaysOptions() *ListGatewaysOptions {
	return &ListGatewaysOptions{}
}

// SetHeaders : Allow user to set Headers
func (options *ListGatewaysOptions) SetHeaders(param map[string]string) *ListGatewaysOptions {
	options.Headers = param
	return options
}

// ListOfferingTypeLocationCrossConnectRoutersOptions : The ListOfferingTypeLocationCrossConnectRouters options.
type ListOfferingTypeLocationCrossConnectRoutersOptions struct {
	// The Direct Link offering type.  Only value `"dedicated"` is supported for this API.
	OfferingType *string `json:"offering_type" validate:"required"`

	// The name of the Direct Link location.
	LocationName *string `json:"location_name" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the ListOfferingTypeLocationCrossConnectRoutersOptions.OfferingType property.
// The Direct Link offering type.  Only value `"dedicated"` is supported for this API.
const (
	ListOfferingTypeLocationCrossConnectRoutersOptions_OfferingType_Dedicated = "dedicated"
)

// NewListOfferingTypeLocationCrossConnectRoutersOptions : Instantiate ListOfferingTypeLocationCrossConnectRoutersOptions
func (*DirectLinkV1) NewListOfferingTypeLocationCrossConnectRoutersOptions(offeringType string, locationName string) *ListOfferingTypeLocationCrossConnectRoutersOptions {
	return &ListOfferingTypeLocationCrossConnectRoutersOptions{
		OfferingType: core.StringPtr(offeringType),
		LocationName: core.StringPtr(locationName),
	}
}

// SetOfferingType : Allow user to set OfferingType
func (options *ListOfferingTypeLocationCrossConnectRoutersOptions) SetOfferingType(offeringType string) *ListOfferingTypeLocationCrossConnectRoutersOptions {
	options.OfferingType = core.StringPtr(offeringType)
	return options
}

// SetLocationName : Allow user to set LocationName
func (options *ListOfferingTypeLocationCrossConnectRoutersOptions) SetLocationName(locationName string) *ListOfferingTypeLocationCrossConnectRoutersOptions {
	options.LocationName = core.StringPtr(locationName)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListOfferingTypeLocationCrossConnectRoutersOptions) SetHeaders(param map[string]string) *ListOfferingTypeLocationCrossConnectRoutersOptions {
	options.Headers = param
	return options
}

// ListOfferingTypeLocationsOptions : The ListOfferingTypeLocations options.
type ListOfferingTypeLocationsOptions struct {
	// The Direct Link offering type.  Current supported values are `"dedicated"` and `"connect"`.
	OfferingType *string `json:"offering_type" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the ListOfferingTypeLocationsOptions.OfferingType property.
// The Direct Link offering type.  Current supported values are `"dedicated"` and `"connect"`.
const (
	ListOfferingTypeLocationsOptions_OfferingType_Connect   = "connect"
	ListOfferingTypeLocationsOptions_OfferingType_Dedicated = "dedicated"
)

// NewListOfferingTypeLocationsOptions : Instantiate ListOfferingTypeLocationsOptions
func (*DirectLinkV1) NewListOfferingTypeLocationsOptions(offeringType string) *ListOfferingTypeLocationsOptions {
	return &ListOfferingTypeLocationsOptions{
		OfferingType: core.StringPtr(offeringType),
	}
}

// SetOfferingType : Allow user to set OfferingType
func (options *ListOfferingTypeLocationsOptions) SetOfferingType(offeringType string) *ListOfferingTypeLocationsOptions {
	options.OfferingType = core.StringPtr(offeringType)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListOfferingTypeLocationsOptions) SetHeaders(param map[string]string) *ListOfferingTypeLocationsOptions {
	options.Headers = param
	return options
}

// ListOfferingTypeSpeedsOptions : The ListOfferingTypeSpeeds options.
type ListOfferingTypeSpeedsOptions struct {
	// The Direct Link offering type.  Current supported values are `"dedicated"` and `"connect"`.
	OfferingType *string `json:"offering_type" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the ListOfferingTypeSpeedsOptions.OfferingType property.
// The Direct Link offering type.  Current supported values are `"dedicated"` and `"connect"`.
const (
	ListOfferingTypeSpeedsOptions_OfferingType_Connect   = "connect"
	ListOfferingTypeSpeedsOptions_OfferingType_Dedicated = "dedicated"
)

// NewListOfferingTypeSpeedsOptions : Instantiate ListOfferingTypeSpeedsOptions
func (*DirectLinkV1) NewListOfferingTypeSpeedsOptions(offeringType string) *ListOfferingTypeSpeedsOptions {
	return &ListOfferingTypeSpeedsOptions{
		OfferingType: core.StringPtr(offeringType),
	}
}

// SetOfferingType : Allow user to set OfferingType
func (options *ListOfferingTypeSpeedsOptions) SetOfferingType(offeringType string) *ListOfferingTypeSpeedsOptions {
	options.OfferingType = core.StringPtr(offeringType)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListOfferingTypeSpeedsOptions) SetHeaders(param map[string]string) *ListOfferingTypeSpeedsOptions {
	options.Headers = param
	return options
}

// ListPortsOptions : The ListPorts options.
type ListPortsOptions struct {
	// A server-supplied token determining which resource to start the page on.
	Start *string `json:"start,omitempty"`

	// The number of resources to return on a page.
	Limit *int64 `json:"limit,omitempty"`

	// Direct Link location short name.
	LocationName *string `json:"location_name,omitempty"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListPortsOptions : Instantiate ListPortsOptions
func (*DirectLinkV1) NewListPortsOptions() *ListPortsOptions {
	return &ListPortsOptions{}
}

// SetStart : Allow user to set Start
func (options *ListPortsOptions) SetStart(start string) *ListPortsOptions {
	options.Start = core.StringPtr(start)
	return options
}

// SetLimit : Allow user to set Limit
func (options *ListPortsOptions) SetLimit(limit int64) *ListPortsOptions {
	options.Limit = core.Int64Ptr(limit)
	return options
}

// SetLocationName : Allow user to set LocationName
func (options *ListPortsOptions) SetLocationName(locationName string) *ListPortsOptions {
	options.LocationName = core.StringPtr(locationName)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListPortsOptions) SetHeaders(param map[string]string) *ListPortsOptions {
	options.Headers = param
	return options
}

// LocationCollection : location collection.
type LocationCollection struct {
	// Collection of Direct Link locations.
	Locations []LocationOutput `json:"locations" validate:"required"`
}

// UnmarshalLocationCollection unmarshals an instance of LocationCollection from the specified map of raw messages.
func UnmarshalLocationCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LocationCollection)
	err = core.UnmarshalModel(m, "locations", &obj.Locations, UnmarshalLocationOutput)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LocationCrossConnectRouterCollection : List of cross connect router details.
type LocationCrossConnectRouterCollection struct {
	// Array of Routers for this location.
	CrossConnectRouters []CrossConnectRouter `json:"cross_connect_routers,omitempty"`
}

// UnmarshalLocationCrossConnectRouterCollection unmarshals an instance of LocationCrossConnectRouterCollection from the specified map of raw messages.
func UnmarshalLocationCrossConnectRouterCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LocationCrossConnectRouterCollection)
	err = core.UnmarshalModel(m, "cross_connect_routers", &obj.CrossConnectRouters, UnmarshalCrossConnectRouter)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// LocationOutput : location.
type LocationOutput struct {
	// Billing location.  Only present for locations where provisioning is enabled.
	BillingLocation *string `json:"billing_location,omitempty"`

	// Building colocation owner.  Only present for offering_type=dedicated locations where provisioning is enabled.
	BuildingColocationOwner *string `json:"building_colocation_owner,omitempty"`

	// Location long name.
	DisplayName *string `json:"display_name" validate:"required"`

	// Location type.
	LocationType *string `json:"location_type" validate:"required"`

	// Indicate whether location supports MACsec.  Only returned for gateway type=dedicated locations.  Contact IBM support
	// for access to MACsec.
	MacsecEnabled *bool `json:"macsec_enabled,omitempty"`

	// Location market.
	Market *string `json:"market" validate:"required"`

	// Location geography.  Only present for locations where provisioning is enabled.
	MarketGeography *string `json:"market_geography,omitempty"`

	// Is location a multi-zone region (MZR).  Only present for locations where provisioning is enabled.
	Mzr *bool `json:"mzr,omitempty"`

	// Location short name.
	Name *string `json:"name" validate:"required"`

	// Location offering type.
	OfferingType *string `json:"offering_type" validate:"required"`

	// Indicates for the specific offering_type whether this location supports gateway provisioning.
	ProvisionEnabled *bool `json:"provision_enabled" validate:"required"`

	// Location's VPC region.  Only present for locations where provisioning is enabled.
	VpcRegion *string `json:"vpc_region,omitempty"`
}

// UnmarshalLocationOutput unmarshals an instance of LocationOutput from the specified map of raw messages.
func UnmarshalLocationOutput(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(LocationOutput)
	err = core.UnmarshalPrimitive(m, "billing_location", &obj.BillingLocation)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "building_colocation_owner", &obj.BuildingColocationOwner)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "display_name", &obj.DisplayName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "location_type", &obj.LocationType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "macsec_enabled", &obj.MacsecEnabled)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "market", &obj.Market)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "market_geography", &obj.MarketGeography)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "mzr", &obj.Mzr)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "offering_type", &obj.OfferingType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "provision_enabled", &obj.ProvisionEnabled)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "vpc_region", &obj.VpcRegion)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// OfferingSpeed : Speed.
type OfferingSpeed struct {
	// Array of capabilities for billing option.
	Capabilities []string `json:"capabilities" validate:"required"`

	// Link speed in megabits per second.
	LinkSpeed *int64 `json:"link_speed" validate:"required"`

	// Indicate whether speed supports MACsec.  Only returned for gateway type=dedicated speeds.  Contact IBM support for
	// access to MACsec.
	MacsecEnabled *bool `json:"macsec_enabled,omitempty"`
}

// UnmarshalOfferingSpeed unmarshals an instance of OfferingSpeed from the specified map of raw messages.
func UnmarshalOfferingSpeed(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(OfferingSpeed)
	err = core.UnmarshalPrimitive(m, "capabilities", &obj.Capabilities)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "link_speed", &obj.LinkSpeed)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "macsec_enabled", &obj.MacsecEnabled)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// OfferingSpeedCollection : OfferingSpeedCollection struct
type OfferingSpeedCollection struct {
	// speed list.
	Speeds []OfferingSpeed `json:"speeds" validate:"required"`
}

// UnmarshalOfferingSpeedCollection unmarshals an instance of OfferingSpeedCollection from the specified map of raw messages.
func UnmarshalOfferingSpeedCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(OfferingSpeedCollection)
	err = core.UnmarshalModel(m, "speeds", &obj.Speeds, UnmarshalOfferingSpeed)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// Port : Provider port details.
type Port struct {
	// Count of existing Direct Link gateways in this account on this port.
	DirectLinkCount *int64 `json:"direct_link_count" validate:"required"`

	// Port identifier.
	ID *string `json:"id" validate:"required"`

	// Port Label.
	Label *string `json:"label" validate:"required"`

	// Port location long name.
	LocationDisplayName *string `json:"location_display_name" validate:"required"`

	// Port location name identifier.
	LocationName *string `json:"location_name" validate:"required"`

	// Port's provider name.
	ProviderName *string `json:"provider_name" validate:"required"`

	// Port's supported speeds in megabits per second.
	SupportedLinkSpeeds []int64 `json:"supported_link_speeds" validate:"required"`
}

// UnmarshalPort unmarshals an instance of Port from the specified map of raw messages.
func UnmarshalPort(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(Port)
	err = core.UnmarshalPrimitive(m, "direct_link_count", &obj.DirectLinkCount)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "label", &obj.Label)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "location_display_name", &obj.LocationDisplayName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "location_name", &obj.LocationName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "provider_name", &obj.ProviderName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "supported_link_speeds", &obj.SupportedLinkSpeeds)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// PortCollection : List of port label details.
type PortCollection struct {
	// A reference to the first page of resources.
	First *PortsPaginatedCollectionFirst `json:"first" validate:"required"`

	// The maximum number of resources can be returned by the request.
	Limit *int64 `json:"limit" validate:"required"`

	// A reference to the next page of resources; this reference is included for all pages except the last page.
	Next *PortsPaginatedCollectionNext `json:"next,omitempty"`

	// The total number of resources across all pages.
	TotalCount *int64 `json:"total_count" validate:"required"`

	// Array of ports.
	Ports []Port `json:"ports,omitempty"`
}

// UnmarshalPortCollection unmarshals an instance of PortCollection from the specified map of raw messages.
func UnmarshalPortCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(PortCollection)
	err = core.UnmarshalModel(m, "first", &obj.First, UnmarshalPortsPaginatedCollectionFirst)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "limit", &obj.Limit)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "next", &obj.Next, UnmarshalPortsPaginatedCollectionNext)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "total_count", &obj.TotalCount)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "ports", &obj.Ports, UnmarshalPort)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// PortsPaginatedCollectionFirst : A reference to the first page of resources.
type PortsPaginatedCollectionFirst struct {
	// The URL for the first page of resources.
	Href *string `json:"href" validate:"required"`
}

// UnmarshalPortsPaginatedCollectionFirst unmarshals an instance of PortsPaginatedCollectionFirst from the specified map of raw messages.
func UnmarshalPortsPaginatedCollectionFirst(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(PortsPaginatedCollectionFirst)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// PortsPaginatedCollectionNext : A reference to the next page of resources; this reference is included for all pages except the last page.
type PortsPaginatedCollectionNext struct {
	// URL for the next page of resources.
	Href *string `json:"href" validate:"required"`

	// start token for the next page of resources.
	Start *string `json:"start,omitempty"`
}

// UnmarshalPortsPaginatedCollectionNext unmarshals an instance of PortsPaginatedCollectionNext from the specified map of raw messages.
func UnmarshalPortsPaginatedCollectionNext(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(PortsPaginatedCollectionNext)
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "start", &obj.Start)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ResourceGroupIdentity : Resource group for this resource. If unspecified, the account's [default resource
// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
type ResourceGroupIdentity struct {
	// Resource group identifier.
	ID *string `json:"id" validate:"required"`
}

// NewResourceGroupIdentity : Instantiate ResourceGroupIdentity (Generic Model Constructor)
func (*DirectLinkV1) NewResourceGroupIdentity(id string) (model *ResourceGroupIdentity, err error) {
	model = &ResourceGroupIdentity{
		ID: core.StringPtr(id),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalResourceGroupIdentity unmarshals an instance of ResourceGroupIdentity from the specified map of raw messages.
func UnmarshalResourceGroupIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ResourceGroupIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ResourceGroupReference : Resource group reference.
type ResourceGroupReference struct {
	// Resource group identifier.
	ID *string `json:"id" validate:"required"`
}

// UnmarshalResourceGroupReference unmarshals an instance of ResourceGroupReference from the specified map of raw messages.
func UnmarshalResourceGroupReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ResourceGroupReference)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// UpdateGatewayOptions : The UpdateGateway options.
type UpdateGatewayOptions struct {
	// Direct Link gateway identifier.
	ID *string `json:"id" validate:"required"`

	// Gateways with global routing (`true`) can connect to networks outside of their associated region.
	Global *bool `json:"global,omitempty"`

	// Use this field during LOA rejection to provide the reason for the rejection.
	//
	// Only allowed for type=dedicated gateways.
	LoaRejectReason *string `json:"loa_reject_reason,omitempty"`

	// MACsec configuration information.  When patching any macsec_config fields, no other fields may be specified in the
	// patch request.  Contact IBM support for access to MACsec.
	//
	// A MACsec config cannot be added to a gateway created without MACsec.
	MacsecConfig *GatewayMacsecConfigPatchTemplate `json:"macsec_config,omitempty"`

	// Metered billing option.  When `true` gateway usage is billed per gigabyte.  When `false` there is no per gigabyte
	// usage charge, instead a flat rate is charged for the gateway.
	Metered *bool `json:"metered,omitempty"`

	// The unique user-defined name for this gateway.
	Name *string `json:"name,omitempty"`

	// Gateway operational status.
	//
	// For gateways pending LOA approval, patch operational_status to the appropriate value to approve or reject its LOA.
	// When rejecting an LOA, provide reject reasoning in `loa_reject_reason`.
	//
	// Only allowed for type=dedicated gateways.
	OperationalStatus *string `json:"operational_status,omitempty"`

	// Gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps,omitempty"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the UpdateGatewayOptions.OperationalStatus property.
// Gateway operational status.
//
// For gateways pending LOA approval, patch operational_status to the appropriate value to approve or reject its LOA.
// When rejecting an LOA, provide reject reasoning in `loa_reject_reason`.
//
// Only allowed for type=dedicated gateways.
const (
	UpdateGatewayOptions_OperationalStatus_LoaAccepted = "loa_accepted"
	UpdateGatewayOptions_OperationalStatus_LoaRejected = "loa_rejected"
)

// NewUpdateGatewayOptions : Instantiate UpdateGatewayOptions
func (*DirectLinkV1) NewUpdateGatewayOptions(id string) *UpdateGatewayOptions {
	return &UpdateGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *UpdateGatewayOptions) SetID(id string) *UpdateGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetGlobal : Allow user to set Global
func (options *UpdateGatewayOptions) SetGlobal(global bool) *UpdateGatewayOptions {
	options.Global = core.BoolPtr(global)
	return options
}

// SetLoaRejectReason : Allow user to set LoaRejectReason
func (options *UpdateGatewayOptions) SetLoaRejectReason(loaRejectReason string) *UpdateGatewayOptions {
	options.LoaRejectReason = core.StringPtr(loaRejectReason)
	return options
}

// SetMacsecConfig : Allow user to set MacsecConfig
func (options *UpdateGatewayOptions) SetMacsecConfig(macsecConfig *GatewayMacsecConfigPatchTemplate) *UpdateGatewayOptions {
	options.MacsecConfig = macsecConfig
	return options
}

// SetMetered : Allow user to set Metered
func (options *UpdateGatewayOptions) SetMetered(metered bool) *UpdateGatewayOptions {
	options.Metered = core.BoolPtr(metered)
	return options
}

// SetName : Allow user to set Name
func (options *UpdateGatewayOptions) SetName(name string) *UpdateGatewayOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetOperationalStatus : Allow user to set OperationalStatus
func (options *UpdateGatewayOptions) SetOperationalStatus(operationalStatus string) *UpdateGatewayOptions {
	options.OperationalStatus = core.StringPtr(operationalStatus)
	return options
}

// SetSpeedMbps : Allow user to set SpeedMbps
func (options *UpdateGatewayOptions) SetSpeedMbps(speedMbps int64) *UpdateGatewayOptions {
	options.SpeedMbps = core.Int64Ptr(speedMbps)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateGatewayOptions) SetHeaders(param map[string]string) *UpdateGatewayOptions {
	options.Headers = param
	return options
}

// UpdateGatewayVirtualConnectionOptions : The UpdateGatewayVirtualConnection options.
type UpdateGatewayVirtualConnectionOptions struct {
	// Direct Link gateway identifier.
	GatewayID *string `json:"gateway_id" validate:"required"`

	// The virtual connection identifier.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this virtual connection.  Virtual connection names are unique within a gateway.  This is
	// the name of the virtual connection itself, the network being connected may have its own name attribute.
	Name *string `json:"name,omitempty"`

	// Status of the virtual connection.  Virtual connections that span IBM Cloud Accounts are created in approval_pending
	// status.  The owner of the target network can accept or reject connection requests by patching status to attached or
	// rejected respectively.
	Status *string `json:"status,omitempty"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the UpdateGatewayVirtualConnectionOptions.Status property.
// Status of the virtual connection.  Virtual connections that span IBM Cloud Accounts are created in approval_pending
// status.  The owner of the target network can accept or reject connection requests by patching status to attached or
// rejected respectively.
const (
	UpdateGatewayVirtualConnectionOptions_Status_Attached = "attached"
	UpdateGatewayVirtualConnectionOptions_Status_Rejected = "rejected"
)

// NewUpdateGatewayVirtualConnectionOptions : Instantiate UpdateGatewayVirtualConnectionOptions
func (*DirectLinkV1) NewUpdateGatewayVirtualConnectionOptions(gatewayID string, id string) *UpdateGatewayVirtualConnectionOptions {
	return &UpdateGatewayVirtualConnectionOptions{
		GatewayID: core.StringPtr(gatewayID),
		ID:        core.StringPtr(id),
	}
}

// SetGatewayID : Allow user to set GatewayID
func (options *UpdateGatewayVirtualConnectionOptions) SetGatewayID(gatewayID string) *UpdateGatewayVirtualConnectionOptions {
	options.GatewayID = core.StringPtr(gatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *UpdateGatewayVirtualConnectionOptions) SetID(id string) *UpdateGatewayVirtualConnectionOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetName : Allow user to set Name
func (options *UpdateGatewayVirtualConnectionOptions) SetName(name string) *UpdateGatewayVirtualConnectionOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetStatus : Allow user to set Status
func (options *UpdateGatewayVirtualConnectionOptions) SetStatus(status string) *UpdateGatewayVirtualConnectionOptions {
	options.Status = core.StringPtr(status)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateGatewayVirtualConnectionOptions) SetHeaders(param map[string]string) *UpdateGatewayVirtualConnectionOptions {
	options.Headers = param
	return options
}

// GatewayActionTemplateUpdatesItemGatewayClientSpeedUpdate : gateway speed change.
// This model "extends" GatewayActionTemplateUpdatesItem
type GatewayActionTemplateUpdatesItemGatewayClientSpeedUpdate struct {
	// New gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps,omitempty"`
}

func (*GatewayActionTemplateUpdatesItemGatewayClientSpeedUpdate) isaGatewayActionTemplateUpdatesItem() bool {
	return true
}

// UnmarshalGatewayActionTemplateUpdatesItemGatewayClientSpeedUpdate unmarshals an instance of GatewayActionTemplateUpdatesItemGatewayClientSpeedUpdate from the specified map of raw messages.
func UnmarshalGatewayActionTemplateUpdatesItemGatewayClientSpeedUpdate(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayActionTemplateUpdatesItemGatewayClientSpeedUpdate)
	err = core.UnmarshalPrimitive(m, "speed_mbps", &obj.SpeedMbps)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItemGatewayClientSpeedUpdate : gateway speed change.
// This model "extends" GatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItem
type GatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItemGatewayClientSpeedUpdate struct {
	// New gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps,omitempty"`
}

func (*GatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItemGatewayClientSpeedUpdate) isaGatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItem() bool {
	return true
}

// UnmarshalGatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItemGatewayClientSpeedUpdate unmarshals an instance of GatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItemGatewayClientSpeedUpdate from the specified map of raw messages.
func UnmarshalGatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItemGatewayClientSpeedUpdate(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItemGatewayClientSpeedUpdate)
	err = core.UnmarshalPrimitive(m, "speed_mbps", &obj.SpeedMbps)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayChangeRequestUpdatesItemGatewayClientSpeedUpdate : gateway speed change.
// This model "extends" GatewayChangeRequestUpdatesItem
type GatewayChangeRequestUpdatesItemGatewayClientSpeedUpdate struct {
	// New gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps,omitempty"`
}

func (*GatewayChangeRequestUpdatesItemGatewayClientSpeedUpdate) isaGatewayChangeRequestUpdatesItem() bool {
	return true
}

// UnmarshalGatewayChangeRequestUpdatesItemGatewayClientSpeedUpdate unmarshals an instance of GatewayChangeRequestUpdatesItemGatewayClientSpeedUpdate from the specified map of raw messages.
func UnmarshalGatewayChangeRequestUpdatesItemGatewayClientSpeedUpdate(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayChangeRequestUpdatesItemGatewayClientSpeedUpdate)
	err = core.UnmarshalPrimitive(m, "speed_mbps", &obj.SpeedMbps)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayChangeRequestGatewayClientGatewayCreate : gateway create.
// This model "extends" GatewayChangeRequest
type GatewayChangeRequestGatewayClientGatewayCreate struct {
	// type of gateway change request.
	Type *string `json:"type" validate:"required"`
}

// Constants associated with the GatewayChangeRequestGatewayClientGatewayCreate.Type property.
// type of gateway change request.
const (
	GatewayChangeRequestGatewayClientGatewayCreate_Type_CreateGateway = "create_gateway"
)

func (*GatewayChangeRequestGatewayClientGatewayCreate) isaGatewayChangeRequest() bool {
	return true
}

// UnmarshalGatewayChangeRequestGatewayClientGatewayCreate unmarshals an instance of GatewayChangeRequestGatewayClientGatewayCreate from the specified map of raw messages.
func UnmarshalGatewayChangeRequestGatewayClientGatewayCreate(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayChangeRequestGatewayClientGatewayCreate)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayChangeRequestGatewayClientGatewayDelete : gateway delete.
// This model "extends" GatewayChangeRequest
type GatewayChangeRequestGatewayClientGatewayDelete struct {
	// type of gateway change request.
	Type *string `json:"type" validate:"required"`
}

// Constants associated with the GatewayChangeRequestGatewayClientGatewayDelete.Type property.
// type of gateway change request.
const (
	GatewayChangeRequestGatewayClientGatewayDelete_Type_DeleteGateway = "delete_gateway"
)

func (*GatewayChangeRequestGatewayClientGatewayDelete) isaGatewayChangeRequest() bool {
	return true
}

// UnmarshalGatewayChangeRequestGatewayClientGatewayDelete unmarshals an instance of GatewayChangeRequestGatewayClientGatewayDelete from the specified map of raw messages.
func UnmarshalGatewayChangeRequestGatewayClientGatewayDelete(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayChangeRequestGatewayClientGatewayDelete)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayChangeRequestGatewayClientGatewayUpdateAttributes : gateway attributes update.
// This model "extends" GatewayChangeRequest
type GatewayChangeRequestGatewayClientGatewayUpdateAttributes struct {
	// type of gateway change request.
	Type *string `json:"type" validate:"required"`

	// array of pending updates.
	Updates []GatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItemIntf `json:"updates" validate:"required"`
}

// Constants associated with the GatewayChangeRequestGatewayClientGatewayUpdateAttributes.Type property.
// type of gateway change request.
const (
	GatewayChangeRequestGatewayClientGatewayUpdateAttributes_Type_UpdateAttributes = "update_attributes"
)

func (*GatewayChangeRequestGatewayClientGatewayUpdateAttributes) isaGatewayChangeRequest() bool {
	return true
}

// UnmarshalGatewayChangeRequestGatewayClientGatewayUpdateAttributes unmarshals an instance of GatewayChangeRequestGatewayClientGatewayUpdateAttributes from the specified map of raw messages.
func UnmarshalGatewayChangeRequestGatewayClientGatewayUpdateAttributes(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayChangeRequestGatewayClientGatewayUpdateAttributes)
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "updates", &obj.Updates, UnmarshalGatewayChangeRequestGatewayClientGatewayUpdateAttributesUpdatesItem)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayTemplateGatewayTypeConnectTemplate : Gateway fields specific to type=connect gateway create.
// This model "extends" GatewayTemplate
type GatewayTemplateGatewayTypeConnectTemplate struct {
	// BGP ASN.
	BgpAsn *int64 `json:"bgp_asn" validate:"required"`

	// (DEPRECATED) BGP base CIDR.
	//
	// Field is deprecated.  See bgp_ibm_cidr and bgp_cer_cidr for details on how to create a gateway using either
	// automatic or explicit IP assignment.  Any bgp_base_cidr value set will be ignored.
	//
	// Deprecated field bgp_base_cidr will be removed from the API specificiation after 15-MAR-2021.
	BgpBaseCidr *string `json:"bgp_base_cidr,omitempty"`

	// BGP customer edge router CIDR.
	//
	// For auto IP assignment, omit bgp_cer_cidr and bgp_ibm_cidr.  IBM will automatically select values for bgp_cer_cidr
	// and bgp_ibm_cidr.
	//
	// For explicit IP assignment set a valid bgp_cer_cidr and bgp_ibm_cidr CIDR, the value must reside in one of
	// "10.254.0.0/16", "172.16.0.0/12", "192.168.0.0/16", "169.254.0.0/16" or an owned public CIDR.  bgp_cer_cidr and
	// bgp_ibm_cidr must have matching network and subnet mask values.
	BgpCerCidr *string `json:"bgp_cer_cidr,omitempty"`

	// BGP IBM CIDR.
	//
	// For auto IP assignment, omit bgp_cer_cidr and bgp_ibm_cidr.  IBM will automatically select values for bgp_cer_cidr
	// and bgp_ibm_cidr.
	//
	// For explicit IP assignment set a valid bgp_cer_cidr and bgp_ibm_cidr CIDR, the value must reside in one of
	// "10.254.0.0/16", "172.16.0.0/12", "192.168.0.0/16", "169.254.0.0/16" or an owned public CIDR.  bgp_cer_cidr and
	// bgp_ibm_cidr must have matching network and subnet mask values.
	BgpIbmCidr *string `json:"bgp_ibm_cidr,omitempty"`

	// Gateways with global routing (`true`) can connect to networks outside their associated region.
	Global *bool `json:"global" validate:"required"`

	// Metered billing option.  When `true` gateway usage is billed per gigabyte.  When `false` there is no per gigabyte
	// usage charge, instead a flat rate is charged for the gateway.
	Metered *bool `json:"metered" validate:"required"`

	// The unique user-defined name for this gateway.
	Name *string `json:"name" validate:"required"`

	// Resource group for this resource. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup *ResourceGroupIdentity `json:"resource_group,omitempty"`

	// Gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps" validate:"required"`

	// Gateway type.
	Type *string `json:"type" validate:"required"`

	// Select Port Label for new type=connect gateway.
	Port *GatewayPortIdentity `json:"port" validate:"required"`
}

// Constants associated with the GatewayTemplateGatewayTypeConnectTemplate.Type property.
// Gateway type.
const (
	GatewayTemplateGatewayTypeConnectTemplate_Type_Connect   = "connect"
	GatewayTemplateGatewayTypeConnectTemplate_Type_Dedicated = "dedicated"
)

// NewGatewayTemplateGatewayTypeConnectTemplate : Instantiate GatewayTemplateGatewayTypeConnectTemplate (Generic Model Constructor)
func (*DirectLinkV1) NewGatewayTemplateGatewayTypeConnectTemplate(bgpAsn int64, global bool, metered bool, name string, speedMbps int64, typeVar string, port *GatewayPortIdentity) (model *GatewayTemplateGatewayTypeConnectTemplate, err error) {
	model = &GatewayTemplateGatewayTypeConnectTemplate{
		BgpAsn:    core.Int64Ptr(bgpAsn),
		Global:    core.BoolPtr(global),
		Metered:   core.BoolPtr(metered),
		Name:      core.StringPtr(name),
		SpeedMbps: core.Int64Ptr(speedMbps),
		Type:      core.StringPtr(typeVar),
		Port:      port,
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

func (*GatewayTemplateGatewayTypeConnectTemplate) isaGatewayTemplate() bool {
	return true
}

// UnmarshalGatewayTemplateGatewayTypeConnectTemplate unmarshals an instance of GatewayTemplateGatewayTypeConnectTemplate from the specified map of raw messages.
func UnmarshalGatewayTemplateGatewayTypeConnectTemplate(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayTemplateGatewayTypeConnectTemplate)
	err = core.UnmarshalPrimitive(m, "bgp_asn", &obj.BgpAsn)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_base_cidr", &obj.BgpBaseCidr)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_cer_cidr", &obj.BgpCerCidr)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_ibm_cidr", &obj.BgpIbmCidr)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "global", &obj.Global)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "metered", &obj.Metered)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "speed_mbps", &obj.SpeedMbps)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "port", &obj.Port, UnmarshalGatewayPortIdentity)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// GatewayTemplateGatewayTypeDedicatedTemplate : Gateway fields specific to type=dedicated gateway create.
// This model "extends" GatewayTemplate
type GatewayTemplateGatewayTypeDedicatedTemplate struct {
	// BGP ASN.
	BgpAsn *int64 `json:"bgp_asn" validate:"required"`

	// (DEPRECATED) BGP base CIDR.
	//
	// Field is deprecated.  See bgp_ibm_cidr and bgp_cer_cidr for details on how to create a gateway using either
	// automatic or explicit IP assignment.  Any bgp_base_cidr value set will be ignored.
	//
	// Deprecated field bgp_base_cidr will be removed from the API specificiation after 15-MAR-2021.
	BgpBaseCidr *string `json:"bgp_base_cidr,omitempty"`

	// BGP customer edge router CIDR.
	//
	// For auto IP assignment, omit bgp_cer_cidr and bgp_ibm_cidr.  IBM will automatically select values for bgp_cer_cidr
	// and bgp_ibm_cidr.
	//
	// For explicit IP assignment set a valid bgp_cer_cidr and bgp_ibm_cidr CIDR, the value must reside in one of
	// "10.254.0.0/16", "172.16.0.0/12", "192.168.0.0/16", "169.254.0.0/16" or an owned public CIDR.  bgp_cer_cidr and
	// bgp_ibm_cidr must have matching network and subnet mask values.
	BgpCerCidr *string `json:"bgp_cer_cidr,omitempty"`

	// BGP IBM CIDR.
	//
	// For auto IP assignment, omit bgp_cer_cidr and bgp_ibm_cidr.  IBM will automatically select values for bgp_cer_cidr
	// and bgp_ibm_cidr.
	//
	// For explicit IP assignment set a valid bgp_cer_cidr and bgp_ibm_cidr CIDR, the value must reside in one of
	// "10.254.0.0/16", "172.16.0.0/12", "192.168.0.0/16", "169.254.0.0/16" or an owned public CIDR.  bgp_cer_cidr and
	// bgp_ibm_cidr must have matching network and subnet mask values.
	BgpIbmCidr *string `json:"bgp_ibm_cidr,omitempty"`

	// Gateways with global routing (`true`) can connect to networks outside their associated region.
	Global *bool `json:"global" validate:"required"`

	// Metered billing option.  When `true` gateway usage is billed per gigabyte.  When `false` there is no per gigabyte
	// usage charge, instead a flat rate is charged for the gateway.
	Metered *bool `json:"metered" validate:"required"`

	// The unique user-defined name for this gateway.
	Name *string `json:"name" validate:"required"`

	// Resource group for this resource. If unspecified, the account's [default resource
	// group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
	ResourceGroup *ResourceGroupIdentity `json:"resource_group,omitempty"`

	// Gateway speed in megabits per second.
	SpeedMbps *int64 `json:"speed_mbps" validate:"required"`

	// Gateway type.
	Type *string `json:"type" validate:"required"`

	// Carrier name.
	CarrierName *string `json:"carrier_name" validate:"required"`

	// Cross connect router.
	CrossConnectRouter *string `json:"cross_connect_router" validate:"required"`

	// Customer name.
	CustomerName *string `json:"customer_name" validate:"required"`

	// Gateway location.
	LocationName *string `json:"location_name" validate:"required"`

	// MACsec configuration information.  Contact IBM support for access to MACsec.
	MacsecConfig *GatewayMacsecConfigTemplate `json:"macsec_config,omitempty"`
}

// Constants associated with the GatewayTemplateGatewayTypeDedicatedTemplate.Type property.
// Gateway type.
const (
	GatewayTemplateGatewayTypeDedicatedTemplate_Type_Connect   = "connect"
	GatewayTemplateGatewayTypeDedicatedTemplate_Type_Dedicated = "dedicated"
)

// NewGatewayTemplateGatewayTypeDedicatedTemplate : Instantiate GatewayTemplateGatewayTypeDedicatedTemplate (Generic Model Constructor)
func (*DirectLinkV1) NewGatewayTemplateGatewayTypeDedicatedTemplate(bgpAsn int64, global bool, metered bool, name string, speedMbps int64, typeVar string, carrierName string, crossConnectRouter string, customerName string, locationName string) (model *GatewayTemplateGatewayTypeDedicatedTemplate, err error) {
	model = &GatewayTemplateGatewayTypeDedicatedTemplate{
		BgpAsn:             core.Int64Ptr(bgpAsn),
		Global:             core.BoolPtr(global),
		Metered:            core.BoolPtr(metered),
		Name:               core.StringPtr(name),
		SpeedMbps:          core.Int64Ptr(speedMbps),
		Type:               core.StringPtr(typeVar),
		CarrierName:        core.StringPtr(carrierName),
		CrossConnectRouter: core.StringPtr(crossConnectRouter),
		CustomerName:       core.StringPtr(customerName),
		LocationName:       core.StringPtr(locationName),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

func (*GatewayTemplateGatewayTypeDedicatedTemplate) isaGatewayTemplate() bool {
	return true
}

// UnmarshalGatewayTemplateGatewayTypeDedicatedTemplate unmarshals an instance of GatewayTemplateGatewayTypeDedicatedTemplate from the specified map of raw messages.
func UnmarshalGatewayTemplateGatewayTypeDedicatedTemplate(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(GatewayTemplateGatewayTypeDedicatedTemplate)
	err = core.UnmarshalPrimitive(m, "bgp_asn", &obj.BgpAsn)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_base_cidr", &obj.BgpBaseCidr)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_cer_cidr", &obj.BgpCerCidr)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "bgp_ibm_cidr", &obj.BgpIbmCidr)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "global", &obj.Global)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "metered", &obj.Metered)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupIdentity)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "speed_mbps", &obj.SpeedMbps)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "carrier_name", &obj.CarrierName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "cross_connect_router", &obj.CrossConnectRouter)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "customer_name", &obj.CustomerName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "location_name", &obj.LocationName)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "macsec_config", &obj.MacsecConfig, UnmarshalGatewayMacsecConfigTemplate)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}
