package mssql_test

import (
	"context"
	"fmt"
	"testing"

	"github.com/Azure/azure-sdk-for-go/services/preview/sql/mgmt/v3.0/sql"
	"github.com/hashicorp/terraform-plugin-sdk/helper/resource"
	"github.com/hashicorp/terraform-plugin-sdk/terraform"

	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/acceptance"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/acceptance/check"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/clients"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/services/mssql/parse"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/utils"
)

type MsSqlDatabaseVulnerabilityAssessmentRuleBaselineResource struct{}

func TestAccMsSqlDatabaseVulnerabilityAssessmentRuleBaseline_basic(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_mssql_database_vulnerability_assessment_rule_baseline", "test")
	r := MsSqlDatabaseVulnerabilityAssessmentRuleBaselineResource{}

	data.ResourceTest(t, r, []resource.TestStep{
		{
			Config: r.basic(data),
			Check: resource.ComposeTestCheckFunc(
				check.That(data.ResourceName).ExistsInAzure(r),
			),
		},
		data.ImportStep(),
	})
}

func TestAccMsSqlDatabaseVulnerabilityAssessmentRuleBaseline_primary(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_mssql_database_vulnerability_assessment_rule_baseline", "test")
	r := MsSqlDatabaseVulnerabilityAssessmentRuleBaselineResource{}

	data.ResourceTest(t, r, []resource.TestStep{
		{
			Config: r.primary(data),
			Check: resource.ComposeTestCheckFunc(
				check.That(data.ResourceName).ExistsInAzure(r),
			),
		},
		data.ImportStep(),
		{
			Config: r.update(data),
			Check: resource.ComposeTestCheckFunc(
				check.That(data.ResourceName).ExistsInAzure(r),
			),
		},
		data.ImportStep(),
	})
}

func (MsSqlDatabaseVulnerabilityAssessmentRuleBaselineResource) Exists(ctx context.Context, client *clients.Client, state *terraform.InstanceState) (*bool, error) {
	id, err := parse.DatabaseVulnerabilityAssessmentRuleBaselineID(state.ID)
	if err != nil {
		return nil, err
	}

	baselineName := sql.VulnerabilityAssessmentPolicyBaselineName(id.BaselineName)
	resp, err := client.MSSQL.DatabaseVulnerabilityAssessmentRuleBaselinesClient.Get(ctx, id.ResourceGroup, id.ServerName, id.DatabaseName, id.RuleName, baselineName)
	if err != nil {
		if utils.ResponseWasNotFound(resp.Response) {
			return nil, fmt.Errorf("Baseline %q not found (Resource Group %q, Server %q, Database %q, Rule %q)",
				id.BaselineName, id.ResourceGroup, id.ServerName, id.DatabaseName, id.RuleName)
		}
		return nil, fmt.Errorf("reading Baseline %q (Resource Group %q, Server %q, Database %q, Rule %q): %v",
			id.BaselineName, id.ResourceGroup, id.ServerName, id.DatabaseName, id.RuleName, err)
	}

	return utils.Bool(resp.ID != nil), nil
}

func (r MsSqlDatabaseVulnerabilityAssessmentRuleBaselineResource) basic(data acceptance.TestData) string {
	return fmt.Sprintf(`
%[1]s

resource "azurerm_mssql_database_vulnerability_assessment_rule_baseline" "test" {
  server_vulnerability_assessment_id = azurerm_mssql_server_vulnerability_assessment.test.id

  database_name = azurerm_sql_database.test.name
  rule_id       = "VA2111"
  baseline_name = "default"

  baseline_result {
    result = [
      "SCHEMA",
      "dbo",
      "CONTROL",
      "SQL_USER",
      "adminuser1"
    ]
  }

  baseline_result {
    result = [
      "SCHEMA",
      "dbo",
      "CONTROL",
      "SQL_USER",
      "adminuser2"
    ]
  }
}
`, r.database(data))
}

func (r MsSqlDatabaseVulnerabilityAssessmentRuleBaselineResource) primary(data acceptance.TestData) string {
	return fmt.Sprintf(`
%[1]s

resource "azurerm_mssql_database_vulnerability_assessment_rule_baseline" "test" {
  server_vulnerability_assessment_id = azurerm_mssql_server_vulnerability_assessment.test.id

  database_name = azurerm_sql_database.test.name
  rule_id       = "VA2065"
  baseline_name = "master"

  baseline_result {
    result = [
      "clientip",
      "255.255.255.255",
      "255.255.255.255"
    ]
  }
}
`, r.database(data))
}

func (r MsSqlDatabaseVulnerabilityAssessmentRuleBaselineResource) update(data acceptance.TestData) string {
	return fmt.Sprintf(`
%[1]s

resource "azurerm_mssql_database_vulnerability_assessment_rule_baseline" "test" {
  server_vulnerability_assessment_id = azurerm_mssql_server_vulnerability_assessment.test.id

  database_name = azurerm_sql_database.test.name
  rule_id       = "VA2065"
  baseline_name = "master"

  baseline_result {
    result = [
      "clientips",
      "255.255.255.255",
      "255.255.255.255"
    ]
  }

  baseline_result {
    result = [
      "myip",
      "255.255.255.0",
      "255.255.255.0"
    ]
  }
}
`, r.database(data))
}

func (MsSqlDatabaseVulnerabilityAssessmentRuleBaselineResource) database(data acceptance.TestData) string {
	return fmt.Sprintf(`
provider "azurerm" {
  features {}
}

resource "azurerm_resource_group" "test" {
  name     = "acctestRG-mssql-%[1]d"
  location = "%[2]s"
}

resource "azurerm_sql_server" "test" {
  name                         = "acctestsqlserver%[1]d"
  resource_group_name          = azurerm_resource_group.test.name
  location                     = azurerm_resource_group.test.location
  version                      = "12.0"
  administrator_login          = "mradministrator"
  administrator_login_password = "thisIsDog11"
}

resource "azurerm_storage_account" "test" {
  name                     = "accsa%[1]d"
  resource_group_name      = azurerm_resource_group.test.name
  location                 = "%[2]s"
  account_tier             = "Standard"
  account_replication_type = "GRS"
}

resource "azurerm_storage_container" "test" {
  name                  = "acctestsc%[1]d"
  storage_account_name  = azurerm_storage_account.test.name
  container_access_type = "private"
}

resource "azurerm_mssql_server_security_alert_policy" "test" {
  resource_group_name = azurerm_resource_group.test.name
  server_name         = azurerm_sql_server.test.name
  state               = "Enabled"
}

resource "azurerm_mssql_server_vulnerability_assessment" "test" {
  server_security_alert_policy_id = azurerm_mssql_server_security_alert_policy.test.id
  storage_container_path          = "${azurerm_storage_account.test.primary_blob_endpoint}${azurerm_storage_container.test.name}/"
  storage_account_access_key      = azurerm_storage_account.test.primary_access_key
}

resource "azurerm_sql_database" "test" {
  name                = "acctestdb%[1]d"
  resource_group_name = azurerm_resource_group.test.name
  server_name         = azurerm_sql_server.test.name
  location            = azurerm_resource_group.test.location
  edition             = "Standard"
}
`, data.RandomInteger, data.Locations.Primary)
}
