/**
 * (C) Copyright IBM Corp. 2020, 2021.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package vulnerabilityadvisorv3_test

import (
	"bytes"
	"context"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"os"
	"time"

	"github.com/IBM/container-registry-go-sdk/vulnerabilityadvisorv3"
	"github.com/IBM/go-sdk-core/v5/core"
	"github.com/go-openapi/strfmt"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
)

var _ = Describe(`VulnerabilityAdvisorV3`, func() {
	var testServer *httptest.Server
	Describe(`Service constructor tests`, func() {
		account := "testString"
		It(`Instantiate service client`, func() {
			vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
				Authenticator: &core.NoAuthAuthenticator{},
				Account:       core.StringPtr(account),
			})
			Expect(vulnerabilityAdvisorService).ToNot(BeNil())
			Expect(serviceErr).To(BeNil())
		})
		It(`Instantiate service client with error: Invalid URL`, func() {
			vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
				URL:     "{BAD_URL_STRING",
				Account: core.StringPtr(account),
			})
			Expect(vulnerabilityAdvisorService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Invalid Auth`, func() {
			vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
				URL:     "https://vulnerabilityadvisorv3/api",
				Account: core.StringPtr(account),
				Authenticator: &core.BasicAuthenticator{
					Username: "",
					Password: "",
				},
			})
			Expect(vulnerabilityAdvisorService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Validation Error`, func() {
			vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{})
			Expect(vulnerabilityAdvisorService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
	})
	Describe(`Service constructor tests using external config`, func() {
		account := "testString"
		Context(`Using external config, construct service client instances`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VULNERABILITY_ADVISOR_URL":       "https://vulnerabilityadvisorv3/api",
				"VULNERABILITY_ADVISOR_AUTH_TYPE": "noauth",
			}

			It(`Create service client using external config successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3UsingExternalConfig(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					Account: core.StringPtr(account),
				})
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				ClearTestEnvironment(testEnvironment)

				clone := vulnerabilityAdvisorService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vulnerabilityAdvisorService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vulnerabilityAdvisorService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vulnerabilityAdvisorService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url from constructor successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3UsingExternalConfig(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:     "https://testService/api",
					Account: core.StringPtr(account),
				})
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vulnerabilityAdvisorService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vulnerabilityAdvisorService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vulnerabilityAdvisorService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vulnerabilityAdvisorService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url programatically successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3UsingExternalConfig(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					Account: core.StringPtr(account),
				})
				err := vulnerabilityAdvisorService.SetServiceURL("https://testService/api")
				Expect(err).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vulnerabilityAdvisorService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vulnerabilityAdvisorService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vulnerabilityAdvisorService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vulnerabilityAdvisorService.Service.Options.Authenticator))
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid Auth`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VULNERABILITY_ADVISOR_URL":       "https://vulnerabilityadvisorv3/api",
				"VULNERABILITY_ADVISOR_AUTH_TYPE": "someOtherAuth",
			}

			SetTestEnvironment(testEnvironment)
			vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3UsingExternalConfig(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
				Account: core.StringPtr(account),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vulnerabilityAdvisorService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid URL`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VULNERABILITY_ADVISOR_AUTH_TYPE": "NOAuth",
			}

			SetTestEnvironment(testEnvironment)
			vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3UsingExternalConfig(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
				URL:     "{BAD_URL_STRING",
				Account: core.StringPtr(account),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vulnerabilityAdvisorService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
	})
	Describe(`Regional endpoint tests`, func() {
		It(`GetServiceURLForRegion(region string)`, func() {
			var url string
			var err error
			url, err = vulnerabilityadvisorv3.GetServiceURLForRegion("INVALID_REGION")
			Expect(url).To(BeEmpty())
			Expect(err).ToNot(BeNil())
			fmt.Fprintf(GinkgoWriter, "Expected error: %s\n", err.Error())
		})
	})
	Describe(`AccountReportQueryPath(accountReportQueryPathOptions *AccountReportQueryPathOptions) - Operation response error`, func() {
		account := "testString"
		acceptLanguage := "testString"
		accountReportQueryPathPath := "/va/api/v3/report/account"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(accountReportQueryPathPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["repository"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["includeIBM"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["includePrivate"]).To(Equal([]string{"testString"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke AccountReportQueryPath with error: Operation response processing error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the AccountReportQueryPathOptions model
				accountReportQueryPathOptionsModel := new(vulnerabilityadvisorv3.AccountReportQueryPathOptions)
				accountReportQueryPathOptionsModel.Repository = core.StringPtr("testString")
				accountReportQueryPathOptionsModel.IncludeIBM = core.StringPtr("testString")
				accountReportQueryPathOptionsModel.IncludePrivate = core.StringPtr("testString")
				accountReportQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vulnerabilityAdvisorService.AccountReportQueryPath(accountReportQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vulnerabilityAdvisorService.EnableRetries(0, 0)
				result, response, operationErr = vulnerabilityAdvisorService.AccountReportQueryPath(accountReportQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`AccountReportQueryPath(accountReportQueryPathOptions *AccountReportQueryPathOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		accountReportQueryPathPath := "/va/api/v3/report/account"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(accountReportQueryPathPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["repository"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["includeIBM"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["includePrivate"]).To(Equal([]string{"testString"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"assessments": {"mapKey": {"configuration_issues": [{"corrective_action": "CorrectiveAction", "description": "Description", "exempt": true, "meta": {"mapKey": "Inner"}, "type": "Type"}], "id": "ID", "scan_time": 8, "status": "Status", "vulnerabilities": [{"cve_exempt": false, "cve_id": "CveID", "exempt_security_notice_count": 25, "exempt_status": "ExemptStatus", "security_notice_count": 19, "security_notices": [{"notice": "Notice", "notice_exempt": true, "notice_id": "NoticeID", "summary": "Summary", "vulnerable_packages": [{"corrective_action": "CorrectiveAction", "description": "Description", "fix_version": "FixVersion", "installed_version": "InstalledVersion", "package_name": "PackageName"}]}], "summary": "Summary", "total_security_notice_count": 24}]}}}`)
				}))
			})
			It(`Invoke AccountReportQueryPath successfully with retries`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				vulnerabilityAdvisorService.EnableRetries(0, 0)

				// Construct an instance of the AccountReportQueryPathOptions model
				accountReportQueryPathOptionsModel := new(vulnerabilityadvisorv3.AccountReportQueryPathOptions)
				accountReportQueryPathOptionsModel.Repository = core.StringPtr("testString")
				accountReportQueryPathOptionsModel.IncludeIBM = core.StringPtr("testString")
				accountReportQueryPathOptionsModel.IncludePrivate = core.StringPtr("testString")
				accountReportQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vulnerabilityAdvisorService.AccountReportQueryPathWithContext(ctx, accountReportQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vulnerabilityAdvisorService.DisableRetries()
				result, response, operationErr := vulnerabilityAdvisorService.AccountReportQueryPath(accountReportQueryPathOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vulnerabilityAdvisorService.AccountReportQueryPathWithContext(ctx, accountReportQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(accountReportQueryPathPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["repository"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["includeIBM"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["includePrivate"]).To(Equal([]string{"testString"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"assessments": {"mapKey": {"configuration_issues": [{"corrective_action": "CorrectiveAction", "description": "Description", "exempt": true, "meta": {"mapKey": "Inner"}, "type": "Type"}], "id": "ID", "scan_time": 8, "status": "Status", "vulnerabilities": [{"cve_exempt": false, "cve_id": "CveID", "exempt_security_notice_count": 25, "exempt_status": "ExemptStatus", "security_notice_count": 19, "security_notices": [{"notice": "Notice", "notice_exempt": true, "notice_id": "NoticeID", "summary": "Summary", "vulnerable_packages": [{"corrective_action": "CorrectiveAction", "description": "Description", "fix_version": "FixVersion", "installed_version": "InstalledVersion", "package_name": "PackageName"}]}], "summary": "Summary", "total_security_notice_count": 24}]}}}`)
				}))
			})
			It(`Invoke AccountReportQueryPath successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vulnerabilityAdvisorService.AccountReportQueryPath(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the AccountReportQueryPathOptions model
				accountReportQueryPathOptionsModel := new(vulnerabilityadvisorv3.AccountReportQueryPathOptions)
				accountReportQueryPathOptionsModel.Repository = core.StringPtr("testString")
				accountReportQueryPathOptionsModel.IncludeIBM = core.StringPtr("testString")
				accountReportQueryPathOptionsModel.IncludePrivate = core.StringPtr("testString")
				accountReportQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vulnerabilityAdvisorService.AccountReportQueryPath(accountReportQueryPathOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke AccountReportQueryPath with error: Operation request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the AccountReportQueryPathOptions model
				accountReportQueryPathOptionsModel := new(vulnerabilityadvisorv3.AccountReportQueryPathOptions)
				accountReportQueryPathOptionsModel.Repository = core.StringPtr("testString")
				accountReportQueryPathOptionsModel.IncludeIBM = core.StringPtr("testString")
				accountReportQueryPathOptionsModel.IncludePrivate = core.StringPtr("testString")
				accountReportQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vulnerabilityAdvisorService.AccountReportQueryPath(accountReportQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`AccountStatusQueryPath(accountStatusQueryPathOptions *AccountStatusQueryPathOptions) - Operation response error`, func() {
		account := "testString"
		acceptLanguage := "testString"
		accountStatusQueryPathPath := "/va/api/v3/report/account/status"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(accountStatusQueryPathPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["repository"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["includeIBM"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["includePrivate"]).To(Equal([]string{"testString"}))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke AccountStatusQueryPath with error: Operation response processing error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the AccountStatusQueryPathOptions model
				accountStatusQueryPathOptionsModel := new(vulnerabilityadvisorv3.AccountStatusQueryPathOptions)
				accountStatusQueryPathOptionsModel.Repository = core.StringPtr("testString")
				accountStatusQueryPathOptionsModel.IncludeIBM = core.StringPtr("testString")
				accountStatusQueryPathOptionsModel.IncludePrivate = core.StringPtr("testString")
				accountStatusQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vulnerabilityAdvisorService.AccountStatusQueryPath(accountStatusQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vulnerabilityAdvisorService.EnableRetries(0, 0)
				result, response, operationErr = vulnerabilityAdvisorService.AccountStatusQueryPath(accountStatusQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`AccountStatusQueryPath(accountStatusQueryPathOptions *AccountStatusQueryPathOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		accountStatusQueryPathPath := "/va/api/v3/report/account/status"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(accountStatusQueryPathPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["repository"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["includeIBM"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["includePrivate"]).To(Equal([]string{"testString"}))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"images": [{"configuration_issue_count": 23, "created_time": 11, "exempt_configuration_issue_count": 29, "exempt_issue_count": 16, "exempt_vulnerability_count": 24, "issue_count": 10, "name": "Name", "scan_time": 8, "status": "Status", "vulnerability_count": 18}]}`)
				}))
			})
			It(`Invoke AccountStatusQueryPath successfully with retries`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				vulnerabilityAdvisorService.EnableRetries(0, 0)

				// Construct an instance of the AccountStatusQueryPathOptions model
				accountStatusQueryPathOptionsModel := new(vulnerabilityadvisorv3.AccountStatusQueryPathOptions)
				accountStatusQueryPathOptionsModel.Repository = core.StringPtr("testString")
				accountStatusQueryPathOptionsModel.IncludeIBM = core.StringPtr("testString")
				accountStatusQueryPathOptionsModel.IncludePrivate = core.StringPtr("testString")
				accountStatusQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vulnerabilityAdvisorService.AccountStatusQueryPathWithContext(ctx, accountStatusQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vulnerabilityAdvisorService.DisableRetries()
				result, response, operationErr := vulnerabilityAdvisorService.AccountStatusQueryPath(accountStatusQueryPathOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vulnerabilityAdvisorService.AccountStatusQueryPathWithContext(ctx, accountStatusQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(accountStatusQueryPathPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.URL.Query()["repository"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["includeIBM"]).To(Equal([]string{"testString"}))
					Expect(req.URL.Query()["includePrivate"]).To(Equal([]string{"testString"}))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"images": [{"configuration_issue_count": 23, "created_time": 11, "exempt_configuration_issue_count": 29, "exempt_issue_count": 16, "exempt_vulnerability_count": 24, "issue_count": 10, "name": "Name", "scan_time": 8, "status": "Status", "vulnerability_count": 18}]}`)
				}))
			})
			It(`Invoke AccountStatusQueryPath successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vulnerabilityAdvisorService.AccountStatusQueryPath(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the AccountStatusQueryPathOptions model
				accountStatusQueryPathOptionsModel := new(vulnerabilityadvisorv3.AccountStatusQueryPathOptions)
				accountStatusQueryPathOptionsModel.Repository = core.StringPtr("testString")
				accountStatusQueryPathOptionsModel.IncludeIBM = core.StringPtr("testString")
				accountStatusQueryPathOptionsModel.IncludePrivate = core.StringPtr("testString")
				accountStatusQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vulnerabilityAdvisorService.AccountStatusQueryPath(accountStatusQueryPathOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke AccountStatusQueryPath with error: Operation request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the AccountStatusQueryPathOptions model
				accountStatusQueryPathOptionsModel := new(vulnerabilityadvisorv3.AccountStatusQueryPathOptions)
				accountStatusQueryPathOptionsModel.Repository = core.StringPtr("testString")
				accountStatusQueryPathOptionsModel.IncludeIBM = core.StringPtr("testString")
				accountStatusQueryPathOptionsModel.IncludePrivate = core.StringPtr("testString")
				accountStatusQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vulnerabilityAdvisorService.AccountStatusQueryPath(accountStatusQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ImageReportQueryPath(imageReportQueryPathOptions *ImageReportQueryPathOptions) - Operation response error`, func() {
		account := "testString"
		acceptLanguage := "testString"
		imageReportQueryPathPath := "/va/api/v3/report/image/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(imageReportQueryPathPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ImageReportQueryPath with error: Operation response processing error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ImageReportQueryPathOptions model
				imageReportQueryPathOptionsModel := new(vulnerabilityadvisorv3.ImageReportQueryPathOptions)
				imageReportQueryPathOptionsModel.Name = core.StringPtr("testString")
				imageReportQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vulnerabilityAdvisorService.ImageReportQueryPath(imageReportQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vulnerabilityAdvisorService.EnableRetries(0, 0)
				result, response, operationErr = vulnerabilityAdvisorService.ImageReportQueryPath(imageReportQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ImageReportQueryPath(imageReportQueryPathOptions *ImageReportQueryPathOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		imageReportQueryPathPath := "/va/api/v3/report/image/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(imageReportQueryPathPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"configuration_issues": [{"corrective_action": "CorrectiveAction", "description": "Description", "exempt": true, "meta": {"mapKey": "Inner"}, "type": "Type"}], "id": "ID", "scan_time": 8, "status": "Status", "vulnerabilities": [{"cve_exempt": false, "cve_id": "CveID", "exempt_security_notice_count": 25, "exempt_status": "ExemptStatus", "security_notice_count": 19, "security_notices": [{"notice": "Notice", "notice_exempt": true, "notice_id": "NoticeID", "summary": "Summary", "vulnerable_packages": [{"corrective_action": "CorrectiveAction", "description": "Description", "fix_version": "FixVersion", "installed_version": "InstalledVersion", "package_name": "PackageName"}]}], "summary": "Summary", "total_security_notice_count": 24}]}`)
				}))
			})
			It(`Invoke ImageReportQueryPath successfully with retries`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				vulnerabilityAdvisorService.EnableRetries(0, 0)

				// Construct an instance of the ImageReportQueryPathOptions model
				imageReportQueryPathOptionsModel := new(vulnerabilityadvisorv3.ImageReportQueryPathOptions)
				imageReportQueryPathOptionsModel.Name = core.StringPtr("testString")
				imageReportQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vulnerabilityAdvisorService.ImageReportQueryPathWithContext(ctx, imageReportQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vulnerabilityAdvisorService.DisableRetries()
				result, response, operationErr := vulnerabilityAdvisorService.ImageReportQueryPath(imageReportQueryPathOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vulnerabilityAdvisorService.ImageReportQueryPathWithContext(ctx, imageReportQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(imageReportQueryPathPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"configuration_issues": [{"corrective_action": "CorrectiveAction", "description": "Description", "exempt": true, "meta": {"mapKey": "Inner"}, "type": "Type"}], "id": "ID", "scan_time": 8, "status": "Status", "vulnerabilities": [{"cve_exempt": false, "cve_id": "CveID", "exempt_security_notice_count": 25, "exempt_status": "ExemptStatus", "security_notice_count": 19, "security_notices": [{"notice": "Notice", "notice_exempt": true, "notice_id": "NoticeID", "summary": "Summary", "vulnerable_packages": [{"corrective_action": "CorrectiveAction", "description": "Description", "fix_version": "FixVersion", "installed_version": "InstalledVersion", "package_name": "PackageName"}]}], "summary": "Summary", "total_security_notice_count": 24}]}`)
				}))
			})
			It(`Invoke ImageReportQueryPath successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vulnerabilityAdvisorService.ImageReportQueryPath(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ImageReportQueryPathOptions model
				imageReportQueryPathOptionsModel := new(vulnerabilityadvisorv3.ImageReportQueryPathOptions)
				imageReportQueryPathOptionsModel.Name = core.StringPtr("testString")
				imageReportQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vulnerabilityAdvisorService.ImageReportQueryPath(imageReportQueryPathOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ImageReportQueryPath with error: Operation validation and request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ImageReportQueryPathOptions model
				imageReportQueryPathOptionsModel := new(vulnerabilityadvisorv3.ImageReportQueryPathOptions)
				imageReportQueryPathOptionsModel.Name = core.StringPtr("testString")
				imageReportQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vulnerabilityAdvisorService.ImageReportQueryPath(imageReportQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ImageReportQueryPathOptions model with no property values
				imageReportQueryPathOptionsModelNew := new(vulnerabilityadvisorv3.ImageReportQueryPathOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vulnerabilityAdvisorService.ImageReportQueryPath(imageReportQueryPathOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ImageStatusQueryPath(imageStatusQueryPathOptions *ImageStatusQueryPathOptions) - Operation response error`, func() {
		account := "testString"
		acceptLanguage := "testString"
		imageStatusQueryPathPath := "/va/api/v3/report/image/status/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(imageStatusQueryPathPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ImageStatusQueryPath with error: Operation response processing error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ImageStatusQueryPathOptions model
				imageStatusQueryPathOptionsModel := new(vulnerabilityadvisorv3.ImageStatusQueryPathOptions)
				imageStatusQueryPathOptionsModel.Name = core.StringPtr("testString")
				imageStatusQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vulnerabilityAdvisorService.ImageStatusQueryPath(imageStatusQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vulnerabilityAdvisorService.EnableRetries(0, 0)
				result, response, operationErr = vulnerabilityAdvisorService.ImageStatusQueryPath(imageStatusQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ImageStatusQueryPath(imageStatusQueryPathOptions *ImageStatusQueryPathOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		imageStatusQueryPathPath := "/va/api/v3/report/image/status/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(imageStatusQueryPathPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"configuration_issue_count": 23, "exempt_configuration_issue_count": 29, "exempt_issue_count": 16, "exempt_vulnerability_count": 24, "issue_count": 10, "scan_time": 8, "status": "Status", "vulnerability_count": 18}`)
				}))
			})
			It(`Invoke ImageStatusQueryPath successfully with retries`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				vulnerabilityAdvisorService.EnableRetries(0, 0)

				// Construct an instance of the ImageStatusQueryPathOptions model
				imageStatusQueryPathOptionsModel := new(vulnerabilityadvisorv3.ImageStatusQueryPathOptions)
				imageStatusQueryPathOptionsModel.Name = core.StringPtr("testString")
				imageStatusQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vulnerabilityAdvisorService.ImageStatusQueryPathWithContext(ctx, imageStatusQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vulnerabilityAdvisorService.DisableRetries()
				result, response, operationErr := vulnerabilityAdvisorService.ImageStatusQueryPath(imageStatusQueryPathOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vulnerabilityAdvisorService.ImageStatusQueryPathWithContext(ctx, imageStatusQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(imageStatusQueryPathPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"configuration_issue_count": 23, "exempt_configuration_issue_count": 29, "exempt_issue_count": 16, "exempt_vulnerability_count": 24, "issue_count": 10, "scan_time": 8, "status": "Status", "vulnerability_count": 18}`)
				}))
			})
			It(`Invoke ImageStatusQueryPath successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vulnerabilityAdvisorService.ImageStatusQueryPath(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ImageStatusQueryPathOptions model
				imageStatusQueryPathOptionsModel := new(vulnerabilityadvisorv3.ImageStatusQueryPathOptions)
				imageStatusQueryPathOptionsModel.Name = core.StringPtr("testString")
				imageStatusQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vulnerabilityAdvisorService.ImageStatusQueryPath(imageStatusQueryPathOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ImageStatusQueryPath with error: Operation validation and request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ImageStatusQueryPathOptions model
				imageStatusQueryPathOptionsModel := new(vulnerabilityadvisorv3.ImageStatusQueryPathOptions)
				imageStatusQueryPathOptionsModel.Name = core.StringPtr("testString")
				imageStatusQueryPathOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vulnerabilityAdvisorService.ImageStatusQueryPath(imageStatusQueryPathOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ImageStatusQueryPathOptions model with no property values
				imageStatusQueryPathOptionsModelNew := new(vulnerabilityadvisorv3.ImageStatusQueryPathOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vulnerabilityAdvisorService.ImageStatusQueryPath(imageStatusQueryPathOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`Service constructor tests`, func() {
		account := "testString"
		It(`Instantiate service client`, func() {
			vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
				Authenticator: &core.NoAuthAuthenticator{},
				Account:       core.StringPtr(account),
			})
			Expect(vulnerabilityAdvisorService).ToNot(BeNil())
			Expect(serviceErr).To(BeNil())
		})
		It(`Instantiate service client with error: Invalid URL`, func() {
			vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
				URL:     "{BAD_URL_STRING",
				Account: core.StringPtr(account),
			})
			Expect(vulnerabilityAdvisorService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Invalid Auth`, func() {
			vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
				URL:     "https://vulnerabilityadvisorv3/api",
				Account: core.StringPtr(account),
				Authenticator: &core.BasicAuthenticator{
					Username: "",
					Password: "",
				},
			})
			Expect(vulnerabilityAdvisorService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
		It(`Instantiate service client with error: Validation Error`, func() {
			vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{})
			Expect(vulnerabilityAdvisorService).To(BeNil())
			Expect(serviceErr).ToNot(BeNil())
		})
	})
	Describe(`Service constructor tests using external config`, func() {
		account := "testString"
		Context(`Using external config, construct service client instances`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VULNERABILITY_ADVISOR_URL":       "https://vulnerabilityadvisorv3/api",
				"VULNERABILITY_ADVISOR_AUTH_TYPE": "noauth",
			}

			It(`Create service client using external config successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3UsingExternalConfig(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					Account: core.StringPtr(account),
				})
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				ClearTestEnvironment(testEnvironment)

				clone := vulnerabilityAdvisorService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vulnerabilityAdvisorService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vulnerabilityAdvisorService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vulnerabilityAdvisorService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url from constructor successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3UsingExternalConfig(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:     "https://testService/api",
					Account: core.StringPtr(account),
				})
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vulnerabilityAdvisorService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vulnerabilityAdvisorService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vulnerabilityAdvisorService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vulnerabilityAdvisorService.Service.Options.Authenticator))
			})
			It(`Create service client using external config and set url programatically successfully`, func() {
				SetTestEnvironment(testEnvironment)
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3UsingExternalConfig(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					Account: core.StringPtr(account),
				})
				err := vulnerabilityAdvisorService.SetServiceURL("https://testService/api")
				Expect(err).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService.Service.GetServiceURL()).To(Equal("https://testService/api"))
				ClearTestEnvironment(testEnvironment)

				clone := vulnerabilityAdvisorService.Clone()
				Expect(clone).ToNot(BeNil())
				Expect(clone.Service != vulnerabilityAdvisorService.Service).To(BeTrue())
				Expect(clone.GetServiceURL()).To(Equal(vulnerabilityAdvisorService.GetServiceURL()))
				Expect(clone.Service.Options.Authenticator).To(Equal(vulnerabilityAdvisorService.Service.Options.Authenticator))
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid Auth`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VULNERABILITY_ADVISOR_URL":       "https://vulnerabilityadvisorv3/api",
				"VULNERABILITY_ADVISOR_AUTH_TYPE": "someOtherAuth",
			}

			SetTestEnvironment(testEnvironment)
			vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3UsingExternalConfig(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
				Account: core.StringPtr(account),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vulnerabilityAdvisorService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
		Context(`Using external config, construct service client instances with error: Invalid URL`, func() {
			// Map containing environment variables used in testing.
			var testEnvironment = map[string]string{
				"VULNERABILITY_ADVISOR_AUTH_TYPE": "NOAuth",
			}

			SetTestEnvironment(testEnvironment)
			vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3UsingExternalConfig(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
				URL:     "{BAD_URL_STRING",
				Account: core.StringPtr(account),
			})

			It(`Instantiate service client with error`, func() {
				Expect(vulnerabilityAdvisorService).To(BeNil())
				Expect(serviceErr).ToNot(BeNil())
				ClearTestEnvironment(testEnvironment)
			})
		})
	})
	Describe(`Regional endpoint tests`, func() {
		It(`GetServiceURLForRegion(region string)`, func() {
			var url string
			var err error
			url, err = vulnerabilityadvisorv3.GetServiceURLForRegion("INVALID_REGION")
			Expect(url).To(BeEmpty())
			Expect(err).ToNot(BeNil())
			fmt.Fprintf(GinkgoWriter, "Expected error: %s\n", err.Error())
		})
	})
	Describe(`ListExemptionAccount(listExemptionAccountOptions *ListExemptionAccountOptions) - Operation response error`, func() {
		account := "testString"
		acceptLanguage := "testString"
		listExemptionAccountPath := "/va/api/v3/exempt/image"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listExemptionAccountPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListExemptionAccount with error: Operation response processing error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ListExemptionAccountOptions model
				listExemptionAccountOptionsModel := new(vulnerabilityadvisorv3.ListExemptionAccountOptions)
				listExemptionAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vulnerabilityAdvisorService.ListExemptionAccount(listExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vulnerabilityAdvisorService.EnableRetries(0, 0)
				result, response, operationErr = vulnerabilityAdvisorService.ListExemptionAccount(listExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListExemptionAccount(listExemptionAccountOptions *ListExemptionAccountOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		listExemptionAccountPath := "/va/api/v3/exempt/image"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listExemptionAccountPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}]`)
				}))
			})
			It(`Invoke ListExemptionAccount successfully with retries`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				vulnerabilityAdvisorService.EnableRetries(0, 0)

				// Construct an instance of the ListExemptionAccountOptions model
				listExemptionAccountOptionsModel := new(vulnerabilityadvisorv3.ListExemptionAccountOptions)
				listExemptionAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vulnerabilityAdvisorService.ListExemptionAccountWithContext(ctx, listExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vulnerabilityAdvisorService.DisableRetries()
				result, response, operationErr := vulnerabilityAdvisorService.ListExemptionAccount(listExemptionAccountOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vulnerabilityAdvisorService.ListExemptionAccountWithContext(ctx, listExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listExemptionAccountPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}]`)
				}))
			})
			It(`Invoke ListExemptionAccount successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vulnerabilityAdvisorService.ListExemptionAccount(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListExemptionAccountOptions model
				listExemptionAccountOptionsModel := new(vulnerabilityadvisorv3.ListExemptionAccountOptions)
				listExemptionAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vulnerabilityAdvisorService.ListExemptionAccount(listExemptionAccountOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListExemptionAccount with error: Operation request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ListExemptionAccountOptions model
				listExemptionAccountOptionsModel := new(vulnerabilityadvisorv3.ListExemptionAccountOptions)
				listExemptionAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vulnerabilityAdvisorService.ListExemptionAccount(listExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetExemptionAccount(getExemptionAccountOptions *GetExemptionAccountOptions) - Operation response error`, func() {
		account := "testString"
		acceptLanguage := "testString"
		getExemptionAccountPath := "/va/api/v3/exempt/image/issue/testString/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getExemptionAccountPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetExemptionAccount with error: Operation response processing error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the GetExemptionAccountOptions model
				getExemptionAccountOptionsModel := new(vulnerabilityadvisorv3.GetExemptionAccountOptions)
				getExemptionAccountOptionsModel.IssueType = core.StringPtr("testString")
				getExemptionAccountOptionsModel.IssueID = core.StringPtr("testString")
				getExemptionAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vulnerabilityAdvisorService.GetExemptionAccount(getExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vulnerabilityAdvisorService.EnableRetries(0, 0)
				result, response, operationErr = vulnerabilityAdvisorService.GetExemptionAccount(getExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetExemptionAccount(getExemptionAccountOptions *GetExemptionAccountOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		getExemptionAccountPath := "/va/api/v3/exempt/image/issue/testString/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getExemptionAccountPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}`)
				}))
			})
			It(`Invoke GetExemptionAccount successfully with retries`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				vulnerabilityAdvisorService.EnableRetries(0, 0)

				// Construct an instance of the GetExemptionAccountOptions model
				getExemptionAccountOptionsModel := new(vulnerabilityadvisorv3.GetExemptionAccountOptions)
				getExemptionAccountOptionsModel.IssueType = core.StringPtr("testString")
				getExemptionAccountOptionsModel.IssueID = core.StringPtr("testString")
				getExemptionAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vulnerabilityAdvisorService.GetExemptionAccountWithContext(ctx, getExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vulnerabilityAdvisorService.DisableRetries()
				result, response, operationErr := vulnerabilityAdvisorService.GetExemptionAccount(getExemptionAccountOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vulnerabilityAdvisorService.GetExemptionAccountWithContext(ctx, getExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getExemptionAccountPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}`)
				}))
			})
			It(`Invoke GetExemptionAccount successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vulnerabilityAdvisorService.GetExemptionAccount(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetExemptionAccountOptions model
				getExemptionAccountOptionsModel := new(vulnerabilityadvisorv3.GetExemptionAccountOptions)
				getExemptionAccountOptionsModel.IssueType = core.StringPtr("testString")
				getExemptionAccountOptionsModel.IssueID = core.StringPtr("testString")
				getExemptionAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vulnerabilityAdvisorService.GetExemptionAccount(getExemptionAccountOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetExemptionAccount with error: Operation validation and request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the GetExemptionAccountOptions model
				getExemptionAccountOptionsModel := new(vulnerabilityadvisorv3.GetExemptionAccountOptions)
				getExemptionAccountOptionsModel.IssueType = core.StringPtr("testString")
				getExemptionAccountOptionsModel.IssueID = core.StringPtr("testString")
				getExemptionAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vulnerabilityAdvisorService.GetExemptionAccount(getExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetExemptionAccountOptions model with no property values
				getExemptionAccountOptionsModelNew := new(vulnerabilityadvisorv3.GetExemptionAccountOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vulnerabilityAdvisorService.GetExemptionAccount(getExemptionAccountOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateExemptionAccount(createExemptionAccountOptions *CreateExemptionAccountOptions) - Operation response error`, func() {
		account := "testString"
		acceptLanguage := "testString"
		createExemptionAccountPath := "/va/api/v3/exempt/image/issue/testString/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createExemptionAccountPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateExemptionAccount with error: Operation response processing error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the CreateExemptionAccountOptions model
				createExemptionAccountOptionsModel := new(vulnerabilityadvisorv3.CreateExemptionAccountOptions)
				createExemptionAccountOptionsModel.IssueType = core.StringPtr("testString")
				createExemptionAccountOptionsModel.IssueID = core.StringPtr("testString")
				createExemptionAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vulnerabilityAdvisorService.CreateExemptionAccount(createExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vulnerabilityAdvisorService.EnableRetries(0, 0)
				result, response, operationErr = vulnerabilityAdvisorService.CreateExemptionAccount(createExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`CreateExemptionAccount(createExemptionAccountOptions *CreateExemptionAccountOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		createExemptionAccountPath := "/va/api/v3/exempt/image/issue/testString/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createExemptionAccountPath))
					Expect(req.Method).To(Equal("POST"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}`)
				}))
			})
			It(`Invoke CreateExemptionAccount successfully with retries`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				vulnerabilityAdvisorService.EnableRetries(0, 0)

				// Construct an instance of the CreateExemptionAccountOptions model
				createExemptionAccountOptionsModel := new(vulnerabilityadvisorv3.CreateExemptionAccountOptions)
				createExemptionAccountOptionsModel.IssueType = core.StringPtr("testString")
				createExemptionAccountOptionsModel.IssueID = core.StringPtr("testString")
				createExemptionAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vulnerabilityAdvisorService.CreateExemptionAccountWithContext(ctx, createExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vulnerabilityAdvisorService.DisableRetries()
				result, response, operationErr := vulnerabilityAdvisorService.CreateExemptionAccount(createExemptionAccountOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vulnerabilityAdvisorService.CreateExemptionAccountWithContext(ctx, createExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createExemptionAccountPath))
					Expect(req.Method).To(Equal("POST"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}`)
				}))
			})
			It(`Invoke CreateExemptionAccount successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vulnerabilityAdvisorService.CreateExemptionAccount(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the CreateExemptionAccountOptions model
				createExemptionAccountOptionsModel := new(vulnerabilityadvisorv3.CreateExemptionAccountOptions)
				createExemptionAccountOptionsModel.IssueType = core.StringPtr("testString")
				createExemptionAccountOptionsModel.IssueID = core.StringPtr("testString")
				createExemptionAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vulnerabilityAdvisorService.CreateExemptionAccount(createExemptionAccountOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateExemptionAccount with error: Operation validation and request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the CreateExemptionAccountOptions model
				createExemptionAccountOptionsModel := new(vulnerabilityadvisorv3.CreateExemptionAccountOptions)
				createExemptionAccountOptionsModel.IssueType = core.StringPtr("testString")
				createExemptionAccountOptionsModel.IssueID = core.StringPtr("testString")
				createExemptionAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vulnerabilityAdvisorService.CreateExemptionAccount(createExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateExemptionAccountOptions model with no property values
				createExemptionAccountOptionsModelNew := new(vulnerabilityadvisorv3.CreateExemptionAccountOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vulnerabilityAdvisorService.CreateExemptionAccount(createExemptionAccountOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`DeleteExemptionAccount(deleteExemptionAccountOptions *DeleteExemptionAccountOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		deleteExemptionAccountPath := "/va/api/v3/exempt/image/issue/testString/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteExemptionAccountPath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.WriteHeader(200)
				}))
			})
			It(`Invoke DeleteExemptionAccount successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vulnerabilityAdvisorService.DeleteExemptionAccount(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteExemptionAccountOptions model
				deleteExemptionAccountOptionsModel := new(vulnerabilityadvisorv3.DeleteExemptionAccountOptions)
				deleteExemptionAccountOptionsModel.IssueType = core.StringPtr("testString")
				deleteExemptionAccountOptionsModel.IssueID = core.StringPtr("testString")
				deleteExemptionAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vulnerabilityAdvisorService.DeleteExemptionAccount(deleteExemptionAccountOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteExemptionAccount with error: Operation validation and request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the DeleteExemptionAccountOptions model
				deleteExemptionAccountOptionsModel := new(vulnerabilityadvisorv3.DeleteExemptionAccountOptions)
				deleteExemptionAccountOptionsModel.IssueType = core.StringPtr("testString")
				deleteExemptionAccountOptionsModel.IssueID = core.StringPtr("testString")
				deleteExemptionAccountOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vulnerabilityAdvisorService.DeleteExemptionAccount(deleteExemptionAccountOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteExemptionAccountOptions model with no property values
				deleteExemptionAccountOptionsModelNew := new(vulnerabilityadvisorv3.DeleteExemptionAccountOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vulnerabilityAdvisorService.DeleteExemptionAccount(deleteExemptionAccountOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListExemptionResource(listExemptionResourceOptions *ListExemptionResourceOptions) - Operation response error`, func() {
		account := "testString"
		acceptLanguage := "testString"
		listExemptionResourcePath := "/va/api/v3/exempt/image/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listExemptionResourcePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListExemptionResource with error: Operation response processing error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ListExemptionResourceOptions model
				listExemptionResourceOptionsModel := new(vulnerabilityadvisorv3.ListExemptionResourceOptions)
				listExemptionResourceOptionsModel.Resource = core.StringPtr("testString")
				listExemptionResourceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vulnerabilityAdvisorService.ListExemptionResource(listExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vulnerabilityAdvisorService.EnableRetries(0, 0)
				result, response, operationErr = vulnerabilityAdvisorService.ListExemptionResource(listExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListExemptionResource(listExemptionResourceOptions *ListExemptionResourceOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		listExemptionResourcePath := "/va/api/v3/exempt/image/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listExemptionResourcePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}]`)
				}))
			})
			It(`Invoke ListExemptionResource successfully with retries`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				vulnerabilityAdvisorService.EnableRetries(0, 0)

				// Construct an instance of the ListExemptionResourceOptions model
				listExemptionResourceOptionsModel := new(vulnerabilityadvisorv3.ListExemptionResourceOptions)
				listExemptionResourceOptionsModel.Resource = core.StringPtr("testString")
				listExemptionResourceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vulnerabilityAdvisorService.ListExemptionResourceWithContext(ctx, listExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vulnerabilityAdvisorService.DisableRetries()
				result, response, operationErr := vulnerabilityAdvisorService.ListExemptionResource(listExemptionResourceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vulnerabilityAdvisorService.ListExemptionResourceWithContext(ctx, listExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listExemptionResourcePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}]`)
				}))
			})
			It(`Invoke ListExemptionResource successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vulnerabilityAdvisorService.ListExemptionResource(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListExemptionResourceOptions model
				listExemptionResourceOptionsModel := new(vulnerabilityadvisorv3.ListExemptionResourceOptions)
				listExemptionResourceOptionsModel.Resource = core.StringPtr("testString")
				listExemptionResourceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vulnerabilityAdvisorService.ListExemptionResource(listExemptionResourceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListExemptionResource with error: Operation validation and request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ListExemptionResourceOptions model
				listExemptionResourceOptionsModel := new(vulnerabilityadvisorv3.ListExemptionResourceOptions)
				listExemptionResourceOptionsModel.Resource = core.StringPtr("testString")
				listExemptionResourceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vulnerabilityAdvisorService.ListExemptionResource(listExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListExemptionResourceOptions model with no property values
				listExemptionResourceOptionsModelNew := new(vulnerabilityadvisorv3.ListExemptionResourceOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vulnerabilityAdvisorService.ListExemptionResource(listExemptionResourceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`GetExemptionResource(getExemptionResourceOptions *GetExemptionResourceOptions) - Operation response error`, func() {
		account := "testString"
		acceptLanguage := "testString"
		getExemptionResourcePath := "/va/api/v3/exempt/image/testString/issue/testString/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getExemptionResourcePath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke GetExemptionResource with error: Operation response processing error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the GetExemptionResourceOptions model
				getExemptionResourceOptionsModel := new(vulnerabilityadvisorv3.GetExemptionResourceOptions)
				getExemptionResourceOptionsModel.Resource = core.StringPtr("testString")
				getExemptionResourceOptionsModel.IssueType = core.StringPtr("testString")
				getExemptionResourceOptionsModel.IssueID = core.StringPtr("testString")
				getExemptionResourceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vulnerabilityAdvisorService.GetExemptionResource(getExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vulnerabilityAdvisorService.EnableRetries(0, 0)
				result, response, operationErr = vulnerabilityAdvisorService.GetExemptionResource(getExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`GetExemptionResource(getExemptionResourceOptions *GetExemptionResourceOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		getExemptionResourcePath := "/va/api/v3/exempt/image/testString/issue/testString/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getExemptionResourcePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}`)
				}))
			})
			It(`Invoke GetExemptionResource successfully with retries`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				vulnerabilityAdvisorService.EnableRetries(0, 0)

				// Construct an instance of the GetExemptionResourceOptions model
				getExemptionResourceOptionsModel := new(vulnerabilityadvisorv3.GetExemptionResourceOptions)
				getExemptionResourceOptionsModel.Resource = core.StringPtr("testString")
				getExemptionResourceOptionsModel.IssueType = core.StringPtr("testString")
				getExemptionResourceOptionsModel.IssueID = core.StringPtr("testString")
				getExemptionResourceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vulnerabilityAdvisorService.GetExemptionResourceWithContext(ctx, getExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vulnerabilityAdvisorService.DisableRetries()
				result, response, operationErr := vulnerabilityAdvisorService.GetExemptionResource(getExemptionResourceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vulnerabilityAdvisorService.GetExemptionResourceWithContext(ctx, getExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(getExemptionResourcePath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}`)
				}))
			})
			It(`Invoke GetExemptionResource successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vulnerabilityAdvisorService.GetExemptionResource(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the GetExemptionResourceOptions model
				getExemptionResourceOptionsModel := new(vulnerabilityadvisorv3.GetExemptionResourceOptions)
				getExemptionResourceOptionsModel.Resource = core.StringPtr("testString")
				getExemptionResourceOptionsModel.IssueType = core.StringPtr("testString")
				getExemptionResourceOptionsModel.IssueID = core.StringPtr("testString")
				getExemptionResourceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vulnerabilityAdvisorService.GetExemptionResource(getExemptionResourceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke GetExemptionResource with error: Operation validation and request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the GetExemptionResourceOptions model
				getExemptionResourceOptionsModel := new(vulnerabilityadvisorv3.GetExemptionResourceOptions)
				getExemptionResourceOptionsModel.Resource = core.StringPtr("testString")
				getExemptionResourceOptionsModel.IssueType = core.StringPtr("testString")
				getExemptionResourceOptionsModel.IssueID = core.StringPtr("testString")
				getExemptionResourceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vulnerabilityAdvisorService.GetExemptionResource(getExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the GetExemptionResourceOptions model with no property values
				getExemptionResourceOptionsModelNew := new(vulnerabilityadvisorv3.GetExemptionResourceOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vulnerabilityAdvisorService.GetExemptionResource(getExemptionResourceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`CreateExemptionResource(createExemptionResourceOptions *CreateExemptionResourceOptions) - Operation response error`, func() {
		account := "testString"
		acceptLanguage := "testString"
		createExemptionResourcePath := "/va/api/v3/exempt/image/testString/issue/testString/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createExemptionResourcePath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke CreateExemptionResource with error: Operation response processing error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the CreateExemptionResourceOptions model
				createExemptionResourceOptionsModel := new(vulnerabilityadvisorv3.CreateExemptionResourceOptions)
				createExemptionResourceOptionsModel.Resource = core.StringPtr("testString")
				createExemptionResourceOptionsModel.IssueType = core.StringPtr("testString")
				createExemptionResourceOptionsModel.IssueID = core.StringPtr("testString")
				createExemptionResourceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vulnerabilityAdvisorService.CreateExemptionResource(createExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vulnerabilityAdvisorService.EnableRetries(0, 0)
				result, response, operationErr = vulnerabilityAdvisorService.CreateExemptionResource(createExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`CreateExemptionResource(createExemptionResourceOptions *CreateExemptionResourceOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		createExemptionResourcePath := "/va/api/v3/exempt/image/testString/issue/testString/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createExemptionResourcePath))
					Expect(req.Method).To(Equal("POST"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}`)
				}))
			})
			It(`Invoke CreateExemptionResource successfully with retries`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				vulnerabilityAdvisorService.EnableRetries(0, 0)

				// Construct an instance of the CreateExemptionResourceOptions model
				createExemptionResourceOptionsModel := new(vulnerabilityadvisorv3.CreateExemptionResourceOptions)
				createExemptionResourceOptionsModel.Resource = core.StringPtr("testString")
				createExemptionResourceOptionsModel.IssueType = core.StringPtr("testString")
				createExemptionResourceOptionsModel.IssueID = core.StringPtr("testString")
				createExemptionResourceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vulnerabilityAdvisorService.CreateExemptionResourceWithContext(ctx, createExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vulnerabilityAdvisorService.DisableRetries()
				result, response, operationErr := vulnerabilityAdvisorService.CreateExemptionResource(createExemptionResourceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vulnerabilityAdvisorService.CreateExemptionResourceWithContext(ctx, createExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(createExemptionResourcePath))
					Expect(req.Method).To(Equal("POST"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(201)
					fmt.Fprintf(res, "%s", `{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}`)
				}))
			})
			It(`Invoke CreateExemptionResource successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vulnerabilityAdvisorService.CreateExemptionResource(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the CreateExemptionResourceOptions model
				createExemptionResourceOptionsModel := new(vulnerabilityadvisorv3.CreateExemptionResourceOptions)
				createExemptionResourceOptionsModel.Resource = core.StringPtr("testString")
				createExemptionResourceOptionsModel.IssueType = core.StringPtr("testString")
				createExemptionResourceOptionsModel.IssueID = core.StringPtr("testString")
				createExemptionResourceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vulnerabilityAdvisorService.CreateExemptionResource(createExemptionResourceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke CreateExemptionResource with error: Operation validation and request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the CreateExemptionResourceOptions model
				createExemptionResourceOptionsModel := new(vulnerabilityadvisorv3.CreateExemptionResourceOptions)
				createExemptionResourceOptionsModel.Resource = core.StringPtr("testString")
				createExemptionResourceOptionsModel.IssueType = core.StringPtr("testString")
				createExemptionResourceOptionsModel.IssueID = core.StringPtr("testString")
				createExemptionResourceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vulnerabilityAdvisorService.CreateExemptionResource(createExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the CreateExemptionResourceOptions model with no property values
				createExemptionResourceOptionsModelNew := new(vulnerabilityadvisorv3.CreateExemptionResourceOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vulnerabilityAdvisorService.CreateExemptionResource(createExemptionResourceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`DeleteExemptionResource(deleteExemptionResourceOptions *DeleteExemptionResourceOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		deleteExemptionResourcePath := "/va/api/v3/exempt/image/testString/issue/testString/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(deleteExemptionResourcePath))
					Expect(req.Method).To(Equal("DELETE"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.WriteHeader(200)
				}))
			})
			It(`Invoke DeleteExemptionResource successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				response, operationErr := vulnerabilityAdvisorService.DeleteExemptionResource(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())

				// Construct an instance of the DeleteExemptionResourceOptions model
				deleteExemptionResourceOptionsModel := new(vulnerabilityadvisorv3.DeleteExemptionResourceOptions)
				deleteExemptionResourceOptionsModel.Resource = core.StringPtr("testString")
				deleteExemptionResourceOptionsModel.IssueType = core.StringPtr("testString")
				deleteExemptionResourceOptionsModel.IssueID = core.StringPtr("testString")
				deleteExemptionResourceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				response, operationErr = vulnerabilityAdvisorService.DeleteExemptionResource(deleteExemptionResourceOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
			})
			It(`Invoke DeleteExemptionResource with error: Operation validation and request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the DeleteExemptionResourceOptions model
				deleteExemptionResourceOptionsModel := new(vulnerabilityadvisorv3.DeleteExemptionResourceOptions)
				deleteExemptionResourceOptionsModel.Resource = core.StringPtr("testString")
				deleteExemptionResourceOptionsModel.IssueType = core.StringPtr("testString")
				deleteExemptionResourceOptionsModel.IssueID = core.StringPtr("testString")
				deleteExemptionResourceOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				response, operationErr := vulnerabilityAdvisorService.DeleteExemptionResource(deleteExemptionResourceOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				// Construct a second instance of the DeleteExemptionResourceOptions model with no property values
				deleteExemptionResourceOptionsModelNew := new(vulnerabilityadvisorv3.DeleteExemptionResourceOptions)
				// Invoke operation with invalid model (negative test)
				response, operationErr = vulnerabilityAdvisorService.DeleteExemptionResource(deleteExemptionResourceOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ExemptHandler(exemptHandlerOptions *ExemptHandlerOptions) - Operation response error`, func() {
		account := "testString"
		acceptLanguage := "testString"
		exemptHandlerPath := "/va/api/v3/exempt/types"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(exemptHandlerPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ExemptHandler with error: Operation response processing error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ExemptHandlerOptions model
				exemptHandlerOptionsModel := new(vulnerabilityadvisorv3.ExemptHandlerOptions)
				exemptHandlerOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vulnerabilityAdvisorService.ExemptHandler(exemptHandlerOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vulnerabilityAdvisorService.EnableRetries(0, 0)
				result, response, operationErr = vulnerabilityAdvisorService.ExemptHandler(exemptHandlerOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ExemptHandler(exemptHandlerOptions *ExemptHandlerOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		exemptHandlerPath := "/va/api/v3/exempt/types"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(exemptHandlerPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"display_name": "DisplayName", "identifier": "Identifier", "translationKey": "TranslationKey", "validation_regex": "ValidationRegex"}]`)
				}))
			})
			It(`Invoke ExemptHandler successfully with retries`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				vulnerabilityAdvisorService.EnableRetries(0, 0)

				// Construct an instance of the ExemptHandlerOptions model
				exemptHandlerOptionsModel := new(vulnerabilityadvisorv3.ExemptHandlerOptions)
				exemptHandlerOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vulnerabilityAdvisorService.ExemptHandlerWithContext(ctx, exemptHandlerOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vulnerabilityAdvisorService.DisableRetries()
				result, response, operationErr := vulnerabilityAdvisorService.ExemptHandler(exemptHandlerOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vulnerabilityAdvisorService.ExemptHandlerWithContext(ctx, exemptHandlerOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(exemptHandlerPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"display_name": "DisplayName", "identifier": "Identifier", "translationKey": "TranslationKey", "validation_regex": "ValidationRegex"}]`)
				}))
			})
			It(`Invoke ExemptHandler successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vulnerabilityAdvisorService.ExemptHandler(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ExemptHandlerOptions model
				exemptHandlerOptionsModel := new(vulnerabilityadvisorv3.ExemptHandlerOptions)
				exemptHandlerOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vulnerabilityAdvisorService.ExemptHandler(exemptHandlerOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ExemptHandler with error: Operation request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ExemptHandlerOptions model
				exemptHandlerOptionsModel := new(vulnerabilityadvisorv3.ExemptHandlerOptions)
				exemptHandlerOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vulnerabilityAdvisorService.ExemptHandler(exemptHandlerOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListAccountExemptions(listAccountExemptionsOptions *ListAccountExemptionsOptions) - Operation response error`, func() {
		account := "testString"
		acceptLanguage := "testString"
		listAccountExemptionsPath := "/va/api/v3/exemptions/account"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listAccountExemptionsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListAccountExemptions with error: Operation response processing error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ListAccountExemptionsOptions model
				listAccountExemptionsOptionsModel := new(vulnerabilityadvisorv3.ListAccountExemptionsOptions)
				listAccountExemptionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vulnerabilityAdvisorService.ListAccountExemptions(listAccountExemptionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vulnerabilityAdvisorService.EnableRetries(0, 0)
				result, response, operationErr = vulnerabilityAdvisorService.ListAccountExemptions(listAccountExemptionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListAccountExemptions(listAccountExemptionsOptions *ListAccountExemptionsOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		listAccountExemptionsPath := "/va/api/v3/exemptions/account"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listAccountExemptionsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}]`)
				}))
			})
			It(`Invoke ListAccountExemptions successfully with retries`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				vulnerabilityAdvisorService.EnableRetries(0, 0)

				// Construct an instance of the ListAccountExemptionsOptions model
				listAccountExemptionsOptionsModel := new(vulnerabilityadvisorv3.ListAccountExemptionsOptions)
				listAccountExemptionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vulnerabilityAdvisorService.ListAccountExemptionsWithContext(ctx, listAccountExemptionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vulnerabilityAdvisorService.DisableRetries()
				result, response, operationErr := vulnerabilityAdvisorService.ListAccountExemptions(listAccountExemptionsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vulnerabilityAdvisorService.ListAccountExemptionsWithContext(ctx, listAccountExemptionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listAccountExemptionsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}]`)
				}))
			})
			It(`Invoke ListAccountExemptions successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vulnerabilityAdvisorService.ListAccountExemptions(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListAccountExemptionsOptions model
				listAccountExemptionsOptionsModel := new(vulnerabilityadvisorv3.ListAccountExemptionsOptions)
				listAccountExemptionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vulnerabilityAdvisorService.ListAccountExemptions(listAccountExemptionsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListAccountExemptions with error: Operation request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ListAccountExemptionsOptions model
				listAccountExemptionsOptionsModel := new(vulnerabilityadvisorv3.ListAccountExemptionsOptions)
				listAccountExemptionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vulnerabilityAdvisorService.ListAccountExemptions(listAccountExemptionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ExemptionsAccountDeleteHandler(exemptionsAccountDeleteHandlerOptions *ExemptionsAccountDeleteHandlerOptions) - Operation response error`, func() {
		account := "testString"
		acceptLanguage := "testString"
		exemptionsAccountDeleteHandlerPath := "/va/api/v3/exemptions/deleteAll"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(exemptionsAccountDeleteHandlerPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ExemptionsAccountDeleteHandler with error: Operation response processing error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ExemptionsAccountDeleteHandlerOptions model
				exemptionsAccountDeleteHandlerOptionsModel := new(vulnerabilityadvisorv3.ExemptionsAccountDeleteHandlerOptions)
				exemptionsAccountDeleteHandlerOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vulnerabilityAdvisorService.ExemptionsAccountDeleteHandler(exemptionsAccountDeleteHandlerOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vulnerabilityAdvisorService.EnableRetries(0, 0)
				result, response, operationErr = vulnerabilityAdvisorService.ExemptionsAccountDeleteHandler(exemptionsAccountDeleteHandlerOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ExemptionsAccountDeleteHandler(exemptionsAccountDeleteHandlerOptions *ExemptionsAccountDeleteHandlerOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		exemptionsAccountDeleteHandlerPath := "/va/api/v3/exemptions/deleteAll"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(exemptionsAccountDeleteHandlerPath))
					Expect(req.Method).To(Equal("POST"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"number_of_exemptions_deleted": 25}`)
				}))
			})
			It(`Invoke ExemptionsAccountDeleteHandler successfully with retries`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				vulnerabilityAdvisorService.EnableRetries(0, 0)

				// Construct an instance of the ExemptionsAccountDeleteHandlerOptions model
				exemptionsAccountDeleteHandlerOptionsModel := new(vulnerabilityadvisorv3.ExemptionsAccountDeleteHandlerOptions)
				exemptionsAccountDeleteHandlerOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vulnerabilityAdvisorService.ExemptionsAccountDeleteHandlerWithContext(ctx, exemptionsAccountDeleteHandlerOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vulnerabilityAdvisorService.DisableRetries()
				result, response, operationErr := vulnerabilityAdvisorService.ExemptionsAccountDeleteHandler(exemptionsAccountDeleteHandlerOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vulnerabilityAdvisorService.ExemptionsAccountDeleteHandlerWithContext(ctx, exemptionsAccountDeleteHandlerOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(exemptionsAccountDeleteHandlerPath))
					Expect(req.Method).To(Equal("POST"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"number_of_exemptions_deleted": 25}`)
				}))
			})
			It(`Invoke ExemptionsAccountDeleteHandler successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vulnerabilityAdvisorService.ExemptionsAccountDeleteHandler(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ExemptionsAccountDeleteHandlerOptions model
				exemptionsAccountDeleteHandlerOptionsModel := new(vulnerabilityadvisorv3.ExemptionsAccountDeleteHandlerOptions)
				exemptionsAccountDeleteHandlerOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vulnerabilityAdvisorService.ExemptionsAccountDeleteHandler(exemptionsAccountDeleteHandlerOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ExemptionsAccountDeleteHandler with error: Operation request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ExemptionsAccountDeleteHandlerOptions model
				exemptionsAccountDeleteHandlerOptionsModel := new(vulnerabilityadvisorv3.ExemptionsAccountDeleteHandlerOptions)
				exemptionsAccountDeleteHandlerOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vulnerabilityAdvisorService.ExemptionsAccountDeleteHandler(exemptionsAccountDeleteHandlerOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListImageExemptions(listImageExemptionsOptions *ListImageExemptionsOptions) - Operation response error`, func() {
		account := "testString"
		acceptLanguage := "testString"
		listImageExemptionsPath := "/va/api/v3/exemptions/image/testString"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listImageExemptionsPath))
					Expect(req.Method).To(Equal("GET"))
					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// TODO: Add check for includeScope query parameter
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListImageExemptions with error: Operation response processing error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ListImageExemptionsOptions model
				listImageExemptionsOptionsModel := new(vulnerabilityadvisorv3.ListImageExemptionsOptions)
				listImageExemptionsOptionsModel.Resource = core.StringPtr("testString")
				listImageExemptionsOptionsModel.IncludeScope = core.BoolPtr(true)
				listImageExemptionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vulnerabilityAdvisorService.ListImageExemptions(listImageExemptionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vulnerabilityAdvisorService.EnableRetries(0, 0)
				result, response, operationErr = vulnerabilityAdvisorService.ListImageExemptions(listImageExemptionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListImageExemptions(listImageExemptionsOptions *ListImageExemptionsOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		listImageExemptionsPath := "/va/api/v3/exemptions/image/testString"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listImageExemptionsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// TODO: Add check for includeScope query parameter
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}]`)
				}))
			})
			It(`Invoke ListImageExemptions successfully with retries`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				vulnerabilityAdvisorService.EnableRetries(0, 0)

				// Construct an instance of the ListImageExemptionsOptions model
				listImageExemptionsOptionsModel := new(vulnerabilityadvisorv3.ListImageExemptionsOptions)
				listImageExemptionsOptionsModel.Resource = core.StringPtr("testString")
				listImageExemptionsOptionsModel.IncludeScope = core.BoolPtr(true)
				listImageExemptionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vulnerabilityAdvisorService.ListImageExemptionsWithContext(ctx, listImageExemptionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vulnerabilityAdvisorService.DisableRetries()
				result, response, operationErr := vulnerabilityAdvisorService.ListImageExemptions(listImageExemptionsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vulnerabilityAdvisorService.ListImageExemptionsWithContext(ctx, listImageExemptionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listImageExemptionsPath))
					Expect(req.Method).To(Equal("GET"))

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// TODO: Add check for includeScope query parameter
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `[{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}]`)
				}))
			})
			It(`Invoke ListImageExemptions successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vulnerabilityAdvisorService.ListImageExemptions(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListImageExemptionsOptions model
				listImageExemptionsOptionsModel := new(vulnerabilityadvisorv3.ListImageExemptionsOptions)
				listImageExemptionsOptionsModel.Resource = core.StringPtr("testString")
				listImageExemptionsOptionsModel.IncludeScope = core.BoolPtr(true)
				listImageExemptionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vulnerabilityAdvisorService.ListImageExemptions(listImageExemptionsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListImageExemptions with error: Operation validation and request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ListImageExemptionsOptions model
				listImageExemptionsOptionsModel := new(vulnerabilityadvisorv3.ListImageExemptionsOptions)
				listImageExemptionsOptionsModel.Resource = core.StringPtr("testString")
				listImageExemptionsOptionsModel.IncludeScope = core.BoolPtr(true)
				listImageExemptionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vulnerabilityAdvisorService.ListImageExemptions(listImageExemptionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListImageExemptionsOptions model with no property values
				listImageExemptionsOptionsModelNew := new(vulnerabilityadvisorv3.ListImageExemptionsOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vulnerabilityAdvisorService.ListImageExemptions(listImageExemptionsOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`ListBulkImageExemptions(listBulkImageExemptionsOptions *ListBulkImageExemptionsOptions) - Operation response error`, func() {
		account := "testString"
		acceptLanguage := "testString"
		listBulkImageExemptionsPath := "/va/api/v3/exemptions/images"
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listBulkImageExemptionsPath))
					Expect(req.Method).To(Equal("POST"))
					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, `} this is not valid json {`)
				}))
			})
			It(`Invoke ListBulkImageExemptions with error: Operation response processing error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ListBulkImageExemptionsOptions model
				listBulkImageExemptionsOptionsModel := new(vulnerabilityadvisorv3.ListBulkImageExemptionsOptions)
				listBulkImageExemptionsOptionsModel.Body = []string{"us.icr.io/birds/woodpecker:green", "us.icr.io/birds/grebe:crested"}
				listBulkImageExemptionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Expect response parsing to fail since we are receiving a text/plain response
				result, response, operationErr := vulnerabilityAdvisorService.ListBulkImageExemptions(listBulkImageExemptionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())

				// Enable retries and test again
				vulnerabilityAdvisorService.EnableRetries(0, 0)
				result, response, operationErr = vulnerabilityAdvisorService.ListBulkImageExemptions(listBulkImageExemptionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})

	Describe(`ListBulkImageExemptions(listBulkImageExemptionsOptions *ListBulkImageExemptionsOptions)`, func() {
		account := "testString"
		acceptLanguage := "testString"
		listBulkImageExemptionsPath := "/va/api/v3/exemptions/images"
		Context(`Using mock server endpoint with timeout`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listBulkImageExemptionsPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Sleep a short time to support a timeout test
					time.Sleep(100 * time.Millisecond)

					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"mapKey": [{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}]}`)
				}))
			})
			It(`Invoke ListBulkImageExemptions successfully with retries`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())
				vulnerabilityAdvisorService.EnableRetries(0, 0)

				// Construct an instance of the ListBulkImageExemptionsOptions model
				listBulkImageExemptionsOptionsModel := new(vulnerabilityadvisorv3.ListBulkImageExemptionsOptions)
				listBulkImageExemptionsOptionsModel.Body = []string{"us.icr.io/birds/woodpecker:green", "us.icr.io/birds/grebe:crested"}
				listBulkImageExemptionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with a Context to test a timeout error
				ctx, cancelFunc := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc()
				_, _, operationErr := vulnerabilityAdvisorService.ListBulkImageExemptionsWithContext(ctx, listBulkImageExemptionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))

				// Disable retries and test again
				vulnerabilityAdvisorService.DisableRetries()
				result, response, operationErr := vulnerabilityAdvisorService.ListBulkImageExemptions(listBulkImageExemptionsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

				// Re-test the timeout error with retries disabled
				ctx, cancelFunc2 := context.WithTimeout(context.Background(), 80*time.Millisecond)
				defer cancelFunc2()
				_, _, operationErr = vulnerabilityAdvisorService.ListBulkImageExemptionsWithContext(ctx, listBulkImageExemptionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring("deadline exceeded"))
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
		Context(`Using mock server endpoint`, func() {
			BeforeEach(func() {
				testServer = httptest.NewServer(http.HandlerFunc(func(res http.ResponseWriter, req *http.Request) {
					defer GinkgoRecover()

					// Verify the contents of the request
					Expect(req.URL.EscapedPath()).To(Equal(listBulkImageExemptionsPath))
					Expect(req.Method).To(Equal("POST"))

					// For gzip-disabled operation, verify Content-Encoding is not set.
					Expect(req.Header.Get("Content-Encoding")).To(BeEmpty())

					// If there is a body, then make sure we can read it
					bodyBuf := new(bytes.Buffer)
					if req.Header.Get("Content-Encoding") == "gzip" {
						body, err := core.NewGzipDecompressionReader(req.Body)
						Expect(err).To(BeNil())
						_, err = bodyBuf.ReadFrom(body)
						Expect(err).To(BeNil())
					} else {
						_, err := bodyBuf.ReadFrom(req.Body)
						Expect(err).To(BeNil())
					}
					fmt.Fprintf(GinkgoWriter, "  Request body: %s", bodyBuf.String())

					Expect(req.Header["Account"]).ToNot(BeNil())
					Expect(req.Header["Account"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					Expect(req.Header["Accept-Language"]).ToNot(BeNil())
					Expect(req.Header["Accept-Language"][0]).To(Equal(fmt.Sprintf("%v", "testString")))
					// Set mock response
					res.Header().Set("Content-type", "application/json")
					res.WriteHeader(200)
					fmt.Fprintf(res, "%s", `{"mapKey": [{"account_id": "AccountID", "issue_id": "IssueID", "issue_type": "IssueType", "scope": {"scope_type": "ScopeType", "namespace": "Namespace", "repository": "Repository", "tag": "Tag"}}]}`)
				}))
			})
			It(`Invoke ListBulkImageExemptions successfully`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Invoke operation with nil options model (negative test)
				result, response, operationErr := vulnerabilityAdvisorService.ListBulkImageExemptions(nil)
				Expect(operationErr).NotTo(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())

				// Construct an instance of the ListBulkImageExemptionsOptions model
				listBulkImageExemptionsOptionsModel := new(vulnerabilityadvisorv3.ListBulkImageExemptionsOptions)
				listBulkImageExemptionsOptionsModel.Body = []string{"us.icr.io/birds/woodpecker:green", "us.icr.io/birds/grebe:crested"}
				listBulkImageExemptionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}

				// Invoke operation with valid options model (positive test)
				result, response, operationErr = vulnerabilityAdvisorService.ListBulkImageExemptions(listBulkImageExemptionsOptionsModel)
				Expect(operationErr).To(BeNil())
				Expect(response).ToNot(BeNil())
				Expect(result).ToNot(BeNil())

			})
			It(`Invoke ListBulkImageExemptions with error: Operation validation and request error`, func() {
				vulnerabilityAdvisorService, serviceErr := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
					URL:            testServer.URL,
					Authenticator:  &core.NoAuthAuthenticator{},
					Account:        core.StringPtr(account),
					AcceptLanguage: core.StringPtr(acceptLanguage),
				})
				Expect(serviceErr).To(BeNil())
				Expect(vulnerabilityAdvisorService).ToNot(BeNil())

				// Construct an instance of the ListBulkImageExemptionsOptions model
				listBulkImageExemptionsOptionsModel := new(vulnerabilityadvisorv3.ListBulkImageExemptionsOptions)
				listBulkImageExemptionsOptionsModel.Body = []string{"us.icr.io/birds/woodpecker:green", "us.icr.io/birds/grebe:crested"}
				listBulkImageExemptionsOptionsModel.Headers = map[string]string{"x-custom-header": "x-custom-value"}
				// Invoke operation with empty URL (negative test)
				err := vulnerabilityAdvisorService.SetServiceURL("")
				Expect(err).To(BeNil())
				result, response, operationErr := vulnerabilityAdvisorService.ListBulkImageExemptions(listBulkImageExemptionsOptionsModel)
				Expect(operationErr).ToNot(BeNil())
				Expect(operationErr.Error()).To(ContainSubstring(core.ERRORMSG_SERVICE_URL_MISSING))
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
				// Construct a second instance of the ListBulkImageExemptionsOptions model with no property values
				listBulkImageExemptionsOptionsModelNew := new(vulnerabilityadvisorv3.ListBulkImageExemptionsOptions)
				// Invoke operation with invalid model (negative test)
				result, response, operationErr = vulnerabilityAdvisorService.ListBulkImageExemptions(listBulkImageExemptionsOptionsModelNew)
				Expect(operationErr).ToNot(BeNil())
				Expect(response).To(BeNil())
				Expect(result).To(BeNil())
			})
			AfterEach(func() {
				testServer.Close()
			})
		})
	})
	Describe(`Model constructor tests`, func() {
		Context(`Using a service client instance`, func() {
			account := "testString"
			acceptLanguage := "testString"
			vulnerabilityAdvisorService, _ := vulnerabilityadvisorv3.NewVulnerabilityAdvisorV3(&vulnerabilityadvisorv3.VulnerabilityAdvisorV3Options{
				URL:            "http://vulnerabilityadvisorv3modelgenerator.com",
				Authenticator:  &core.NoAuthAuthenticator{},
				Account:        core.StringPtr(account),
				AcceptLanguage: core.StringPtr(acceptLanguage),
			})
			It(`Invoke NewAccountReportQueryPathOptions successfully`, func() {
				// Construct an instance of the AccountReportQueryPathOptions model
				accountReportQueryPathOptionsModel := vulnerabilityAdvisorService.NewAccountReportQueryPathOptions()
				accountReportQueryPathOptionsModel.SetRepository("testString")
				accountReportQueryPathOptionsModel.SetIncludeIBM("testString")
				accountReportQueryPathOptionsModel.SetIncludePrivate("testString")
				accountReportQueryPathOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(accountReportQueryPathOptionsModel).ToNot(BeNil())
				Expect(accountReportQueryPathOptionsModel.Repository).To(Equal(core.StringPtr("testString")))
				Expect(accountReportQueryPathOptionsModel.IncludeIBM).To(Equal(core.StringPtr("testString")))
				Expect(accountReportQueryPathOptionsModel.IncludePrivate).To(Equal(core.StringPtr("testString")))
				Expect(accountReportQueryPathOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewAccountStatusQueryPathOptions successfully`, func() {
				// Construct an instance of the AccountStatusQueryPathOptions model
				accountStatusQueryPathOptionsModel := vulnerabilityAdvisorService.NewAccountStatusQueryPathOptions()
				accountStatusQueryPathOptionsModel.SetRepository("testString")
				accountStatusQueryPathOptionsModel.SetIncludeIBM("testString")
				accountStatusQueryPathOptionsModel.SetIncludePrivate("testString")
				accountStatusQueryPathOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(accountStatusQueryPathOptionsModel).ToNot(BeNil())
				Expect(accountStatusQueryPathOptionsModel.Repository).To(Equal(core.StringPtr("testString")))
				Expect(accountStatusQueryPathOptionsModel.IncludeIBM).To(Equal(core.StringPtr("testString")))
				Expect(accountStatusQueryPathOptionsModel.IncludePrivate).To(Equal(core.StringPtr("testString")))
				Expect(accountStatusQueryPathOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewCreateExemptionAccountOptions successfully`, func() {
				// Construct an instance of the CreateExemptionAccountOptions model
				issueType := "testString"
				issueID := "testString"
				createExemptionAccountOptionsModel := vulnerabilityAdvisorService.NewCreateExemptionAccountOptions(issueType, issueID)
				createExemptionAccountOptionsModel.SetIssueType("testString")
				createExemptionAccountOptionsModel.SetIssueID("testString")
				createExemptionAccountOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(createExemptionAccountOptionsModel).ToNot(BeNil())
				Expect(createExemptionAccountOptionsModel.IssueType).To(Equal(core.StringPtr("testString")))
				Expect(createExemptionAccountOptionsModel.IssueID).To(Equal(core.StringPtr("testString")))
				Expect(createExemptionAccountOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewCreateExemptionResourceOptions successfully`, func() {
				// Construct an instance of the CreateExemptionResourceOptions model
				resource := "testString"
				issueType := "testString"
				issueID := "testString"
				createExemptionResourceOptionsModel := vulnerabilityAdvisorService.NewCreateExemptionResourceOptions(resource, issueType, issueID)
				createExemptionResourceOptionsModel.SetResource("testString")
				createExemptionResourceOptionsModel.SetIssueType("testString")
				createExemptionResourceOptionsModel.SetIssueID("testString")
				createExemptionResourceOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(createExemptionResourceOptionsModel).ToNot(BeNil())
				Expect(createExemptionResourceOptionsModel.Resource).To(Equal(core.StringPtr("testString")))
				Expect(createExemptionResourceOptionsModel.IssueType).To(Equal(core.StringPtr("testString")))
				Expect(createExemptionResourceOptionsModel.IssueID).To(Equal(core.StringPtr("testString")))
				Expect(createExemptionResourceOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewDeleteExemptionAccountOptions successfully`, func() {
				// Construct an instance of the DeleteExemptionAccountOptions model
				issueType := "testString"
				issueID := "testString"
				deleteExemptionAccountOptionsModel := vulnerabilityAdvisorService.NewDeleteExemptionAccountOptions(issueType, issueID)
				deleteExemptionAccountOptionsModel.SetIssueType("testString")
				deleteExemptionAccountOptionsModel.SetIssueID("testString")
				deleteExemptionAccountOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(deleteExemptionAccountOptionsModel).ToNot(BeNil())
				Expect(deleteExemptionAccountOptionsModel.IssueType).To(Equal(core.StringPtr("testString")))
				Expect(deleteExemptionAccountOptionsModel.IssueID).To(Equal(core.StringPtr("testString")))
				Expect(deleteExemptionAccountOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewDeleteExemptionResourceOptions successfully`, func() {
				// Construct an instance of the DeleteExemptionResourceOptions model
				resource := "testString"
				issueType := "testString"
				issueID := "testString"
				deleteExemptionResourceOptionsModel := vulnerabilityAdvisorService.NewDeleteExemptionResourceOptions(resource, issueType, issueID)
				deleteExemptionResourceOptionsModel.SetResource("testString")
				deleteExemptionResourceOptionsModel.SetIssueType("testString")
				deleteExemptionResourceOptionsModel.SetIssueID("testString")
				deleteExemptionResourceOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(deleteExemptionResourceOptionsModel).ToNot(BeNil())
				Expect(deleteExemptionResourceOptionsModel.Resource).To(Equal(core.StringPtr("testString")))
				Expect(deleteExemptionResourceOptionsModel.IssueType).To(Equal(core.StringPtr("testString")))
				Expect(deleteExemptionResourceOptionsModel.IssueID).To(Equal(core.StringPtr("testString")))
				Expect(deleteExemptionResourceOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewExemptHandlerOptions successfully`, func() {
				// Construct an instance of the ExemptHandlerOptions model
				exemptHandlerOptionsModel := vulnerabilityAdvisorService.NewExemptHandlerOptions()
				exemptHandlerOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(exemptHandlerOptionsModel).ToNot(BeNil())
				Expect(exemptHandlerOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewExemptionsAccountDeleteHandlerOptions successfully`, func() {
				// Construct an instance of the ExemptionsAccountDeleteHandlerOptions model
				exemptionsAccountDeleteHandlerOptionsModel := vulnerabilityAdvisorService.NewExemptionsAccountDeleteHandlerOptions()
				exemptionsAccountDeleteHandlerOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(exemptionsAccountDeleteHandlerOptionsModel).ToNot(BeNil())
				Expect(exemptionsAccountDeleteHandlerOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetExemptionAccountOptions successfully`, func() {
				// Construct an instance of the GetExemptionAccountOptions model
				issueType := "testString"
				issueID := "testString"
				getExemptionAccountOptionsModel := vulnerabilityAdvisorService.NewGetExemptionAccountOptions(issueType, issueID)
				getExemptionAccountOptionsModel.SetIssueType("testString")
				getExemptionAccountOptionsModel.SetIssueID("testString")
				getExemptionAccountOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getExemptionAccountOptionsModel).ToNot(BeNil())
				Expect(getExemptionAccountOptionsModel.IssueType).To(Equal(core.StringPtr("testString")))
				Expect(getExemptionAccountOptionsModel.IssueID).To(Equal(core.StringPtr("testString")))
				Expect(getExemptionAccountOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewGetExemptionResourceOptions successfully`, func() {
				// Construct an instance of the GetExemptionResourceOptions model
				resource := "testString"
				issueType := "testString"
				issueID := "testString"
				getExemptionResourceOptionsModel := vulnerabilityAdvisorService.NewGetExemptionResourceOptions(resource, issueType, issueID)
				getExemptionResourceOptionsModel.SetResource("testString")
				getExemptionResourceOptionsModel.SetIssueType("testString")
				getExemptionResourceOptionsModel.SetIssueID("testString")
				getExemptionResourceOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(getExemptionResourceOptionsModel).ToNot(BeNil())
				Expect(getExemptionResourceOptionsModel.Resource).To(Equal(core.StringPtr("testString")))
				Expect(getExemptionResourceOptionsModel.IssueType).To(Equal(core.StringPtr("testString")))
				Expect(getExemptionResourceOptionsModel.IssueID).To(Equal(core.StringPtr("testString")))
				Expect(getExemptionResourceOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewImageReportQueryPathOptions successfully`, func() {
				// Construct an instance of the ImageReportQueryPathOptions model
				name := "testString"
				imageReportQueryPathOptionsModel := vulnerabilityAdvisorService.NewImageReportQueryPathOptions(name)
				imageReportQueryPathOptionsModel.SetName("testString")
				imageReportQueryPathOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(imageReportQueryPathOptionsModel).ToNot(BeNil())
				Expect(imageReportQueryPathOptionsModel.Name).To(Equal(core.StringPtr("testString")))
				Expect(imageReportQueryPathOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewImageStatusQueryPathOptions successfully`, func() {
				// Construct an instance of the ImageStatusQueryPathOptions model
				name := "testString"
				imageStatusQueryPathOptionsModel := vulnerabilityAdvisorService.NewImageStatusQueryPathOptions(name)
				imageStatusQueryPathOptionsModel.SetName("testString")
				imageStatusQueryPathOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(imageStatusQueryPathOptionsModel).ToNot(BeNil())
				Expect(imageStatusQueryPathOptionsModel.Name).To(Equal(core.StringPtr("testString")))
				Expect(imageStatusQueryPathOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewListAccountExemptionsOptions successfully`, func() {
				// Construct an instance of the ListAccountExemptionsOptions model
				listAccountExemptionsOptionsModel := vulnerabilityAdvisorService.NewListAccountExemptionsOptions()
				listAccountExemptionsOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(listAccountExemptionsOptionsModel).ToNot(BeNil())
				Expect(listAccountExemptionsOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewListBulkImageExemptionsOptions successfully`, func() {
				// Construct an instance of the ListBulkImageExemptionsOptions model
				body := []string{"us.icr.io/birds/woodpecker:green", "us.icr.io/birds/grebe:crested"}
				listBulkImageExemptionsOptionsModel := vulnerabilityAdvisorService.NewListBulkImageExemptionsOptions(body)
				listBulkImageExemptionsOptionsModel.SetBody([]string{"us.icr.io/birds/woodpecker:green", "us.icr.io/birds/grebe:crested"})
				listBulkImageExemptionsOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(listBulkImageExemptionsOptionsModel).ToNot(BeNil())
				Expect(listBulkImageExemptionsOptionsModel.Body).To(Equal([]string{"us.icr.io/birds/woodpecker:green", "us.icr.io/birds/grebe:crested"}))
				Expect(listBulkImageExemptionsOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewListExemptionAccountOptions successfully`, func() {
				// Construct an instance of the ListExemptionAccountOptions model
				listExemptionAccountOptionsModel := vulnerabilityAdvisorService.NewListExemptionAccountOptions()
				listExemptionAccountOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(listExemptionAccountOptionsModel).ToNot(BeNil())
				Expect(listExemptionAccountOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewListExemptionResourceOptions successfully`, func() {
				// Construct an instance of the ListExemptionResourceOptions model
				resource := "testString"
				listExemptionResourceOptionsModel := vulnerabilityAdvisorService.NewListExemptionResourceOptions(resource)
				listExemptionResourceOptionsModel.SetResource("testString")
				listExemptionResourceOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(listExemptionResourceOptionsModel).ToNot(BeNil())
				Expect(listExemptionResourceOptionsModel.Resource).To(Equal(core.StringPtr("testString")))
				Expect(listExemptionResourceOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
			It(`Invoke NewListImageExemptionsOptions successfully`, func() {
				// Construct an instance of the ListImageExemptionsOptions model
				resource := "testString"
				listImageExemptionsOptionsModel := vulnerabilityAdvisorService.NewListImageExemptionsOptions(resource)
				listImageExemptionsOptionsModel.SetResource("testString")
				listImageExemptionsOptionsModel.SetIncludeScope(true)
				listImageExemptionsOptionsModel.SetHeaders(map[string]string{"foo": "bar"})
				Expect(listImageExemptionsOptionsModel).ToNot(BeNil())
				Expect(listImageExemptionsOptionsModel.Resource).To(Equal(core.StringPtr("testString")))
				Expect(listImageExemptionsOptionsModel.IncludeScope).To(Equal(core.BoolPtr(true)))
				Expect(listImageExemptionsOptionsModel.Headers).To(Equal(map[string]string{"foo": "bar"}))
			})
		})
	})
	Describe(`Utility function tests`, func() {
		It(`Invoke CreateMockByteArray() successfully`, func() {
			mockByteArray := CreateMockByteArray("This is a test")
			Expect(mockByteArray).ToNot(BeNil())
		})
		It(`Invoke CreateMockUUID() successfully`, func() {
			mockUUID := CreateMockUUID("9fab83da-98cb-4f18-a7ba-b6f0435c9673")
			Expect(mockUUID).ToNot(BeNil())
		})
		It(`Invoke CreateMockReader() successfully`, func() {
			mockReader := CreateMockReader("This is a test.")
			Expect(mockReader).ToNot(BeNil())
		})
		It(`Invoke CreateMockDate() successfully`, func() {
			mockDate := CreateMockDate()
			Expect(mockDate).ToNot(BeNil())
		})
		It(`Invoke CreateMockDateTime() successfully`, func() {
			mockDateTime := CreateMockDateTime()
			Expect(mockDateTime).ToNot(BeNil())
		})
	})
})

//
// Utility functions used by the generated test code
//

func CreateMockByteArray(mockData string) *[]byte {
	ba := make([]byte, 0)
	ba = append(ba, mockData...)
	return &ba
}

func CreateMockUUID(mockData string) *strfmt.UUID {
	uuid := strfmt.UUID(mockData)
	return &uuid
}

func CreateMockReader(mockData string) io.ReadCloser {
	return ioutil.NopCloser(bytes.NewReader([]byte(mockData)))
}

func CreateMockDate() *strfmt.Date {
	d := strfmt.Date(time.Date(2009, time.November, 10, 23, 0, 0, 0, time.UTC))
	return &d
}

func CreateMockDateTime() *strfmt.DateTime {
	d := strfmt.DateTime(time.Date(2009, time.November, 10, 23, 0, 0, 0, time.UTC))
	return &d
}

func SetTestEnvironment(testEnvironment map[string]string) {
	for key, value := range testEnvironment {
		os.Setenv(key, value)
	}
}

func ClearTestEnvironment(testEnvironment map[string]string) {
	for key := range testEnvironment {
		os.Unsetenv(key)
	}
}
