/*
Copyright 2019 The Knative Authors

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package http

import (
	"bytes"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"

	"knative.dev/serving/pkg/network"
)

var defaultRevInfo = &RequestLogRevision{
	Name:          "rev",
	Namespace:     "ns",
	Service:       "svc",
	Configuration: "cfg",
	PodName:       "pn",
	PodIP:         "ip",
}

func TestRequestLogHandler(t *testing.T) {
	baseHandler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
	})

	tests := []struct {
		name     string
		url      string
		body     string
		template string
		want     string
		wantErr  bool
		isProbe  bool
	}{{
		name:     "empty template",
		url:      "http://example.com/testpage",
		body:     "test",
		template: "",
		want:     "",
	}, {
		name:     "template with new line",
		url:      "http://example.com/testpage",
		body:     "test",
		template: "{{.Request.URL}}\n",
		want:     "http://example.com/testpage\n",
	}, {
		name:     "template without new line",
		url:      "http://example.com",
		body:     "test",
		template: "{{.Request.ContentLength}}",
		want:     "4\n",
	}, {
		name:     "invalid template",
		url:      "http://example.com",
		body:     "test",
		template: "{{}}",
		want:     "",
		wantErr:  true,
	}, {
		name:     "revision info",
		url:      "http://example.com",
		body:     "test",
		template: "{{.Revision.Name}}, {{.Revision.Namespace}}, {{.Revision.Service}}, {{.Revision.Configuration}}, {{.Revision.PodName}}, {{.Revision.PodIP}}",
		want:     "rev, ns, svc, cfg, pn, ip\n",
	}, {
		name:     "probe request",
		url:      "http://example.com",
		body:     "test",
		template: "{{.Request.ContentLength}}",
		want:     "",
		isProbe:  true,
	}}

	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			buf := bytes.NewBufferString("")
			handler, err := NewRequestLogHandler(baseHandler, buf, test.template,
				RequestLogTemplateInputGetterFromRevision(defaultRevInfo))
			if test.wantErr != (err != nil) {
				t.Errorf("got %v, want error %v", err, test.wantErr)
			}

			if !test.wantErr {
				resp := httptest.NewRecorder()
				req := httptest.NewRequest(http.MethodPost, test.url, bytes.NewBufferString(test.body))
				if test.isProbe {
					req.Header.Set(network.ProbeHeaderName, "activator")
				}
				handler.ServeHTTP(resp, req)

				got := buf.String()
				if got != test.want {
					t.Errorf("got '%v', want '%v'", got, test.want)
				}
			}
		})
	}
}

func TestPanickingHandler(t *testing.T) {
	baseHandler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		panic("no!")
	})
	buf := bytes.NewBufferString("")
	handler, err := NewRequestLogHandler(baseHandler, buf, "{{.Request.URL}}",
		RequestLogTemplateInputGetterFromRevision(defaultRevInfo))
	if err != nil {
		t.Errorf("got %v, want error: %v", err, false)
	}

	resp := httptest.NewRecorder()
	req := httptest.NewRequest(http.MethodPost, "http://example.com", bytes.NewBufferString("test"))
	defer func() {
		err := recover()
		if err == nil {
			t.Error("want ServeHTTP to panic, got nothing.")
		}

		got := buf.String()
		if want := "http://example.com\n"; got != want {
			t.Errorf("got '%v', want '%v'", got, want)
		}
	}()
	handler.ServeHTTP(resp, req)
}

func TestFailedTemplateExecution(t *testing.T) {
	baseHandler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
	})
	buf := bytes.NewBufferString("")
	handler, err := NewRequestLogHandler(baseHandler, buf, "{{.Request.Something}}",
		RequestLogTemplateInputGetterFromRevision(defaultRevInfo))
	if err != nil {
		t.Errorf("got %v, wantErr %v, ", err, false)
	}

	resp := httptest.NewRecorder()
	req := httptest.NewRequest(http.MethodGet, "http://example.com", bytes.NewBufferString("test"))
	handler.ServeHTTP(resp, req)

	got := buf.String()
	if want := "Invalid request log template: "; !strings.HasPrefix(got, want) {
		t.Errorf("got: '%v', want: '%v'", got, want)
	}
}

func TestSetTemplate(t *testing.T) {
	baseHandler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
	})

	url, body := "http://example.com/testpage", "test"
	tests := []struct {
		name     string
		template string
		want     string
		wantErr  bool
	}{{
		name:     "empty template 1",
		template: "",
		want:     "",
		wantErr:  false,
	}, {
		name:     "template with new line",
		template: "{{.Request.URL}}\n",
		want:     "http://example.com/testpage\n",
		wantErr:  false,
	}, {
		name:     "empty template 2",
		template: "",
		want:     "",
		wantErr:  false,
	}, {
		name:     "template without new line",
		template: "{{.Request.ContentLength}}",
		want:     "4\n",
		wantErr:  false,
	}, {
		name:     "empty template 3",
		template: "",
		want:     "",
		wantErr:  false,
	}, {
		name:     "invalid template",
		template: "{{}}",
		want:     "",
		wantErr:  true,
	}}

	buf := bytes.NewBufferString("")
	handler, err := NewRequestLogHandler(baseHandler, buf, "",
		RequestLogTemplateInputGetterFromRevision(defaultRevInfo))
	if err != nil {
		t.Fatalf("want: no error, got: %v", err)
	}

	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			err := handler.SetTemplate(test.template)
			if test.wantErr != (err != nil) {
				t.Errorf("got %v, want error %v", err, test.wantErr)
			}

			if !test.wantErr {
				buf.Reset()
				resp := httptest.NewRecorder()
				req := httptest.NewRequest(http.MethodPost, url, bytes.NewBufferString(body))
				handler.ServeHTTP(resp, req)
				got := buf.String()
				if got != test.want {
					t.Errorf("got '%v', want '%v'", got, test.want)
				}
			}
		})
	}
}
