#!/usr/bin/env python
import argparse
import copy
import datetime
import json
import pycurl
import re
import sys
import yaml

from io import BytesIO
from dictdiffer import diff


csv_f = 'manifests/' \
        'kubevirt-hyperconverged-operator.v2.0.0.clusterserviceversion.yaml'

upstream_csv_f = 'upstream_csv.yaml'

us_ds_ignore = set([
    'metadata.annotations.alm-examples',
])


ENVV_TO_BE_UPDATED = [
    'OPERATOR_IMAGE',
    'DOCKER_TAG',
    'OPERATOR_TAG',
    'WEBUI_TAG',
    'VIRT_LAUNCHER_TAG',
    'LINUX_BRIDGE_IMAGE',
    'LINUX_BRIDGE_MARKER_IMAGE',
    'SRIOV_DP_IMAGE',
    'SRIOV_CNI_IMAGE',
    'KUBEMACPOOL_IMAGE',
]
ENVV_TO_BE_UPDATED_TAG_ONLY = [
    'DOCKER_TAG',
    'OPERATOR_TAG',
    'WEBUI_TAG',
    'VIRT_LAUNCHER_TAG',
]
DEFAULT_ERRATA = '39812'
DEFAULT_CHANNEL = "RHEL-8-CNV-2.0"
LOCKED_IMAGES = [
]
VIRT_TAG = ''
UI_TAG = ""

def fetch(url, secure=True):
    buffer = BytesIO()
    c = pycurl.Curl()
    c.setopt(c.URL, url)
    c.setopt(c.WRITEFUNCTION, buffer.write)
    c.setopt(c.HTTPAUTH, c.HTTPAUTH_GSSNEGOTIATE)
    c.setopt(c.USERPWD, ':')
    c.setopt(c.USERAGENT, 'Python Client')
    if not secure:
        c.setopt(c.SSL_VERIFYPEER, False)
        c.setopt(c.SSL_VERIFYHOST, False)
    c.perform()
    response_code = c.getinfo(c.RESPONSE_CODE)
    c.close()

    if response_code == 401:
        raise RuntimeError(
            'auth error: please run kinit to get a valid kerberos ticket'
        )
    elif response_code != 200:
        raise RuntimeError(
            'pycurl error {rc}'.format(rc=response_code)
        )

    data = buffer.getvalue()
    buffer.close()
    jdata = json.loads(data)
    return jdata


def get_errata_images(errata, verbose, secure, channel=DEFAULT_CHANNEL):
    url = 'https://errata.devel.redhat.com/api/v1/erratum/{e}/builds'.format(
        e=errata,
    )
    print("Fetching errata info from '{u}'".format(u=url))
    images = {}

    errata_out = fetch(url, secure=secure)
    builds = [i.keys()[0] for i in errata_out[channel]["builds"]]
    for i in builds:
        s = i.rsplit('-', 3)
        images[s[-4]] = '-'.join(s[-2:])
    print(images)
    return images


def update_image(image, errata_images, verbose=False, tag_only=False):
    i = re.split(':|/', image)
    cname = i[-2]
    ov = i[-1]
    if cname in LOCKED_IMAGES:
        if tag_only:
            return '{v}'.format(v=ov)
        return image
    if verbose:
        print("Processing {cname} currently at {ov}".format(
            cname=cname,
            ov=ov,
        ))
    if cname in errata_images:
        nv = errata_images[cname]
        if ov != nv:
            if verbose:
                print(
                    "Updating '{cname}' from {ov} to {nv}".format(
                        cname=cname,
                        ov=ov,
                        nv=nv,
                    )
                )
            if tag_only:
                return '{v}'.format(v=nv)
            updated_image = "{b}:{v}".format(
                b=image.rsplit(':', 1)[-2],
                v=nv,
            )
            return updated_image
    if tag_only:
        return '{v}'.format(v=ov)
    return image


def update(
    dry_run=False,
    errata=DEFAULT_ERRATA,
    verbose=False,
    secure=True,
    channel=DEFAULT_CHANNEL,
):
    errata_images = get_errata_images(errata, verbose, secure, channel)
    if verbose:
        print("Builds in the errata:")
        for b in errata_images:
            print("\t{b}:{v}".format(b=b, v=errata_images[b]))

            if "kubevirt-web-ui" == b:
                print errata_images[b]
                UI_TAG = "brew-pulp-docker01.web.prod.ext.phx2.redhat.com:8888/container-native-virtualization/kubevirt-web-ui:" + errata_images[b]

        print("")
    with open(csv_f, 'r') as stream:
        csv = yaml.safe_load(stream)
        original_csv = copy.deepcopy(csv)
        containerImage = csv['metadata']['annotations']['containerImage']
        csv['metadata']['annotations']['containerImage'] = update_image(
            containerImage,
            errata_images,
            verbose=verbose
        )

        deployments = csv['spec']['install']['spec']['deployments']
        for d in deployments:
            containers = d['spec']['template']['spec']['containers']
            for c in containers:
                image = c['image']
                c['image'] = update_image(
                    image,
                    errata_images,
                    verbose=verbose
                )
                if "virt-operator" in c['image']:
                    VIRT_TAG = c['image']

                if 'env' in c:
                    env = c['env']
                    for e in env:
                        if (
                            'name' in e and
                            e['name'] in ENVV_TO_BE_UPDATED
                        ):
                            if 'value' in e:
                                tag_only = False
                                value = e['value']
                                if e['name'] in ENVV_TO_BE_UPDATED_TAG_ONLY:
                                    tag_only = True
                                    value = image
                                    if "VIRT_LAUNCHER_TAG" in e['name']:
                                        value = VIRT_TAG
                                    if "WEBUI_TAG" in e['name']:
                                        value = UI_TAG

                                e['value'] = update_image(
                                    value,
                                    errata_images,
                                    verbose=verbose,
                                    tag_only=tag_only
                                )

        csv['metadata']['annotations']['createdAt'] = datetime.datetime.now()
        ddiff = diff(original_csv, csv)
        print("diff:")
        for d in ddiff:
            print("\t{d}".format(d=d))

        print("")
        try:
            with open(upstream_csv_f, 'r') as ustream:
                ucsv = yaml.safe_load(ustream)
                udiff = diff(ucsv, csv, ignore=us_ds_ignore)
                print("diff with upstream CSV:")
                for d in udiff:
                    print("\t{d}".format(d=d))
        except IOError:
            print(
                "Please clone upstream CSV to '{ucsv}' "
                "if you want to compute the diff "
                "with that".format(ucsv=upstream_csv_f)
            )

        if not dry_run:
            with open(csv_f, 'w') as outfile:
                yaml.dump(
                    csv,
                    outfile,
                    default_flow_style=False,
                    sort_keys=True
                )


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description='helper to align versions in the CSV file at build time'
    )
    parser.add_argument(
        '--verbose',
        help='increase output verbosity',
        action='store_true'
    )
    parser.add_argument(
        '--dry-run',
        help='skips writing to the CSV file',
        action='store_true'
    )
    parser.add_argument(
        '--insecure',
        help='Allow insecure server connections when using SSL',
        action='store_false',
        dest='secure'
    )
    parser.add_argument(
        '--errata',
        help='fetch version info from a specific errata'
             ' [Default: {d}]'.format(d=DEFAULT_ERRATA),
        default=DEFAULT_ERRATA
    )
    parser.add_argument(
        '--channel',
        help='errata channel'
             ' [Default: {d}]'.format(d=DEFAULT_CHANNEL),
        default=DEFAULT_CHANNEL,
    )

    args = parser.parse_args()

    try:
        update(
            dry_run=args.dry_run,
            errata=args.errata,
            verbose=args.verbose,
            secure=args.secure,
            channel=args.channel,
        )
        sys.exit(0)
    except yaml.YAMLError as exc:
        print(exc)
        sys.exit(-1)
