#!/usr/bin/python3

import datetime
import json
import os
import subprocess
import logging

NAMESPACE_PLACEHOLDER = "openshift-cnv"
new_csv_version = "2.1.0"
old_csv_version = "2.0.0"

OC_CMD = ['oc', 'version']
DOCKER_CMD = ['docker', '--version']

YAML_DELIMITER = '---'

# Keep the CSVs and CRDs content in a dict to be written only after
# all the CSVs and CRDs were generated.
csvs = {}
crds = {}

logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.INFO)


def writeTofile(file_path, data):
    with open(file_path, "w") as f:
        f.write(data.replace('\r\n', '\n'))


def remove_pod_oc(podName):
    subprocess.run(
        ['oc', 'delete', 'pod', '--ignore-not-found', podName],
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE
    )


def is_cmd_available(cmd):
    try:
        subprocess.run(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    except BaseException:
        return False

    return True


def save_csv_crd(operatorName, output):
    odocs = output.split(YAML_DELIMITER)
    if '' in odocs:
        odocs.remove('')
    if len(odocs) < 2:
        raise Exception('Failed splitting %s CSV and CRDs' % operatorName)

    csvs[operatorName] = YAML_DELIMITER + '\n' + odocs[0]
    crds[operatorName] = YAML_DELIMITER + YAML_DELIMITER.join(odocs[1:])


def split_crds(operatorName, crd_data):
    crds = crd_data.split('---')
    idx = 1
    for crd in crds[1:]:
        CRD_save_path = "%s/CRDs/%s.%d.crd.yml" % (
            os.getcwd(), operatorName, idx
        )

        crd_to_write = "---%s" % crd

        logger.info(
            "Saving {operator_name} CRD {num} in {crd_path}".format(
                operator_name=operatorName,
                num=idx,
                crd_path=CRD_save_path
            )
        )
        writeTofile(CRD_save_path, crd_to_write)
        idx += 1


def write_csvs_crds():
    for operatorName, csv_data in csvs.items():
        CSV_save_path = "%s/CSVs/%s.csv" % (os.getcwd(), operatorName)
        logger.info(
            "Saving {operator_name} CSV in {csv_path}".format(
                operator_name=operatorName,
                csv_path=CSV_save_path
            )
        )
        writeTofile(CSV_save_path, csv_data)

    for operatorName, crd_data in crds.items():
        if crd_data.count('---') > 1:
            split_crds(operatorName, crd_data)
        else:
            CRD_save_path = "%s/CRDs/%s.crd.yml" % (os.getcwd(), operatorName)
            logger.info(
                "Saving {operator_name} CRD in {crd_path}".format(
                    operator_name=operatorName,
                    crd_path=CRD_save_path
                )
            )
            writeTofile(CRD_save_path, crd_data)


def run_pod(operatorArgs, image_pull_url, operatorName):
    ocArgs = [
        "oc",
        "run",
        operatorName,
        "-i",
        "-t",
        "--quiet=true",
        "--image=%s" % image_pull_url,
        "--restart=Never",
        "--command",
        "--",
        "/usr/bin/csv-generator",
    ]

    dockerArgs = [
        "docker",
        "run",
        "--rm",
        "--entrypoint=/usr/bin/csv-generator",
        image_pull_url,
    ]

    manual_build = False
    if "MANUAL_BUILD" in os.environ:
        manual_build = True

    tool = None

    if is_cmd_available(OC_CMD) and not manual_build:
        runArgs = ocArgs.extend(operatorArgs)
        runArgs = ocArgs
        tool = "oc"
    elif is_cmd_available(DOCKER_CMD):
        dockerArgs.extend(operatorArgs)
        runArgs = dockerArgs
        tool = "docker"
    else:
        raise Exception(
            "Nor oc binary or docker are installed in the system. "
            "Cannot proceed"
        )

    try:
        cmd_result = subprocess.run(
            runArgs,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        )
    finally:
        if tool == "oc":
            remove_pod_oc(operatorName)

    if cmd_result.returncode != 0:
        print(runArgs)
        raise Exception(
            'Failed to execute %s, due to: %s' % (
                operatorName,
                cmd_result.stderr.decode("utf-8")
            )
        )

    logger.info("Generated CSV and CRD from %s" % image_pull_url)
    logger.info(
        "%s CSV+CRD %s=" % (operatorName, cmd_result.stdout.decode("utf-8"))
    )
    save_csv_crd(operatorName, cmd_result.stdout.decode("utf-8"))


def get_env_var(envVar):
    try:
        env_value = os.environ[envVar]
    except KeyError:
        raise Exception("Environment variable %s not found" % envVar)

    logger.info("Received EnvVar: %s=%s " % (envVar, env_value))
    return env_value


def getBuildInfo(envVar):
    return json.loads(get_env_var(envVar))


# temporary workaround for
# https://mojo.redhat.com/groups/container-build-system/
# blog/2019/08/12/brew-container-images-migrating-to-quay
# all the images are going to move to quay.io
# but brew is still returnign the old pull url
# for images built in the past
def fix_brew_pu(pullURL):
    return pullURL.replace(
        'brew-pulp-docker01.web.prod.ext.phx2.redhat.com:8888/'
        'container-native-virtualization/',
        'registry-proxy.engineering.redhat.com/rh-osbs/'
        'container-native-virtualization-',
    )


# PLEASE NOTE
# podman has a bug and fail to pull images using
# sha256 tags instead of semantic version tags
# see: https://github.com/cri-o/cri-o/issues/2157
def getSha(envVar):
    buildInfo = getBuildInfo(envVar)
    pullURL = fix_brew_pu(buildInfo["extra"]["image"]["index"]["pull"][0])
    return "@sha256:%s" % pullURL.split("@sha256:")[1]


def getImageVersion(envVar):
    buildInfo = getBuildInfo(envVar)
    return buildInfo["extra"]["image"]["index"]["tags"][0]


def getImageName(pullURL):
    cname_with_version = pullURL.split('/')[-1]
    return cname_with_version.split(':')[0]


def getPullURL(envVar):
    buildInfo = getBuildInfo(envVar)
    return fix_brew_pu(buildInfo["extra"]["image"]["index"]["pull"][1])


def create_cna_csv():
    logger.info("Creating CNA csv.....")
    buildInfo = getBuildInfo(
        "CLUSTER_NETWORK_ADDONS_OPERATOR_BUILD_INFO_JSON"
    )
    pullURL = getPullURL("CLUSTER_NETWORK_ADDONS_OPERATOR_BUILD_INFO_JSON")

    # multus is used only on upstream
    multus_image = "registry-proxy.engineering.redhat.com/rh-osbs/" \
                   "openshift-ose-multus-cni:v4.2"

    bases = pullURL.split('/')
    cprefix = '/'.join(bases[:-1])
    iname = bases[-1].split(':')[0]
    operator_version = buildInfo["extra"]["image"]["index"]["tags"][0]

    args = [
        "--version=%s" % new_csv_version,
        "--version-replaces=%s" % old_csv_version,
        "--operator-version=%s" % operator_version,
        "--namespace=%s" % NAMESPACE_PLACEHOLDER,
        "--container-prefix=%s" % cprefix,
        "--image-name=%s" % iname,
        "--container-tag=%s" % operator_version,
        "--multus-image=%s" % multus_image,
        "--linux-bridge-cni-image=%s" % getPullURL(
            "CNV_CONTAINERNETWORKING_PLUGINS_BUILD_INFO_JSON"
        ),
        "--linux-bridge-marker-image=%s" % getPullURL(
            "BRIDGE_MARKER_BUILD_INFO_JSON"
        ),
        "--kubemacpool-image=%s" % getPullURL(
            "KUBEMACPOOL_BUILD_INFO_JSON"
        ),
        "--nm-state-handler-image=%s" % getPullURL(
            "KUBERNETES_NMSTATE_HANDLER_BUILD_INFO_JSON"
        ),
        "--ovs-cni-image=%s" % getPullURL(
            "OVS_CNI_PLUGIN_BUILD_INFO_JSON"
        ),
        "--ovs-marker-image=%s" % getPullURL(
            "OVS_CNI_MARKER_BUILD_INFO_JSON"
        ),
        "--image-pull-policy=IfNotPresent",
        "--dump-crds",
    ]

    run_pod(
        args,
        get_env_var("CLUSTER_NETWORK_ADDONS_OPERATOR_IMAGE_PULL_URL"),
        buildInfo["name"]
    )


def create_virt_csv():
    logger.info("Creating VIRT csv....")
    buildInfo = getBuildInfo("VIRT_OPERATOR_BUILD_INFO_JSON")
    pullURL = getPullURL("VIRT_OPERATOR_BUILD_INFO_JSON")

    bases = pullURL.split('/')
    cprefix = '/'.join(bases[:-1])
    # temporary workaround until omps-playbooks support a regex
    # virt-oeprator will only work with stage
    cprefix = "registry.stage.redhat.io/container-native-virtualization"

    args = [
        "--namespace=%s" % NAMESPACE_PLACEHOLDER,
        "--operatorImageVersion=%s" % buildInfo[
            "extra"
        ]["image"]["index"]["tags"][0],
        "--dockerPrefix=%s" % cprefix,
        "--kubeVirtVersion=%s" % buildInfo[
            "extra"
        ]["image"]["index"]["tags"][0],
        # skipping image SHA values due to
        # https://github.com/cri-o/cri-o/issues/2157
        # "--apiSha=%s" % getSha("VIRT_API_BUILD_INFO_JSON"),
        # "--controllerSha=%s" % getSha(
        #     "VIRT_CONTROLLER_BUILD_INFO_JSON"
        # ),
        # "--handlerSha=%s" % getSha("VIRT_HANDLER_BUILD_INFO_JSON"),
        # "--launcherSha=%s" % getSha("VIRT_LAUNCHER_BUILD_INFO_JSON"),
        # "--kubevirtLogo=kubevirtLogo",
        "--csvVersion=%s" % new_csv_version,
        "--replacesCsvVersion=%s" % old_csv_version,
        "--csvCreatedAtTimestamp=%s" % str(datetime.datetime.now()),
        "--dumpCRDs",
    ]

    run_pod(
        args,
        get_env_var("VIRT_OPERATOR_IMAGE_PULL_URL"),
        buildInfo["name"]
    )


def create_ssp_csv():
    logger.info("Creating SSP csv.....")
    buildInfo = getBuildInfo("KUBEVIRT_SSP_OPERATOR_BUILD_INFO_JSON")
    pullURL = getPullURL("KUBEVIRT_SSP_OPERATOR_BUILD_INFO_JSON")
    args = [
        "--csv-version=%s" % new_csv_version,
        "--namespace=%s" % NAMESPACE_PLACEHOLDER,
        "--operator-image=%s" % pullURL,
        "--kvm-info-tag=%s" % getImageVersion(
            "KUBEVIRT_KVM_INFO_NFD_PLUGIN_BUILD_INFO_JSON"
        ),
        "--validator-tag=%s" % getImageVersion(
            "KUBEVIRT_TEMPLATE_VALIDATOR_BUILD_INFO_JSON"
        ),
        "--virt-launcher-tag=%s" % getImageVersion(
            "VIRT_LAUNCHER_BUILD_INFO_JSON"
        ),
        "--node-labeller-tag=%s" % getImageVersion(
            "KUBEVIRT_CPU_NODE_LABELLER_BUILD_INFO_JSON"
        ),
        "--cpu-plugin-tag=%s" % getImageVersion(
            "KUBEVIRT_CPU_MODEL_NFD_PLUGIN_BUILD_INFO_JSON"
        ),
        "--dump-crds",
    ]

    run_pod(
        args,
        get_env_var("KUBEVIRT_SSP_OPERATOR_IMAGE_PULL_URL"),
        buildInfo["name"]
    )


def create_cdi_csv():
    logger.info("Creating CDI csv.....")
    buildInfo = getBuildInfo("VIRT_CDI_OPERATOR_BUILD_INFO_JSON")
    pullURL = getPullURL("VIRT_CDI_OPERATOR_BUILD_INFO_JSON")

    bases = pullURL.split('/')
    cprefix = '/'.join(bases[:-1])

    # temporary workaround until omps-playbooks support a regex
    # CDI will only work with stage
    cprefix = "registry.stage.redhat.io/container-native-virtualization"
    operator_version = buildInfo["extra"]["image"]["index"]["tags"][0]

    args = [
        "--csv-version=%s" % new_csv_version,
        "--replaces-csv-version=%s" % old_csv_version,
        "--docker-repo=%s" % cprefix,
        "--namespace=" + NAMESPACE_PLACEHOLDER,
        "--cdi-logo-base64=logo",
        "--docker-tag=%s" % operator_version,
        "--operator-image-name=%s" % getImageName(
            pullURL
        ).replace('container-native-virtualization-',''),
        "--controller-image-name=%s" % getImageName(
            getPullURL("VIRT_CDI_CONTROLLER_BUILD_INFO_JSON")
        ).replace('container-native-virtualization-',''),
        "--importer-image-name=%s" % getImageName(
            getPullURL("VIRT_CDI_IMPORTER_BUILD_INFO_JSON")
        ).replace('container-native-virtualization-',''),
        "--cloner-image-name=%s" % getImageName(
            getPullURL("VIRT_CDI_CLONER_BUILD_INFO_JSON")
        ).replace('container-native-virtualization-',''),
        "--apiserver-image-name=%s" % getImageName(
            getPullURL("VIRT_CDI_APISERVER_BUILD_INFO_JSON")
        ).replace('container-native-virtualization-',''),
        "--uploadproxy-image-name=%s" % getImageName(
            getPullURL("VIRT_CDI_UPLOADPROXY_BUILD_INFO_JSON")
        ).replace('container-native-virtualization-',''),
        "--uploadserver-image-name=%s" % getImageName(
            getPullURL("VIRT_CDI_UPLOADSERVER_BUILD_INFO_JSON")
        ).replace('container-native-virtualization-',''),
        "--pull-policy=IfNotPresent",
        "--dump-crds",
    ]

    run_pod(
        args,
        get_env_var("VIRT_CDI_OPERATOR_IMAGE_PULL_URL"),
        buildInfo["name"]
    )


def create_nmo_csv():
    logger.info("Creating NMO csv.....")
    buildInfo = getBuildInfo("NODE_MAINTENANCE_OPERATOR_BUILD_INFO_JSON")
    pullURL = getPullURL("NODE_MAINTENANCE_OPERATOR_BUILD_INFO_JSON")
    args = [
        "--csv-version=%s" % new_csv_version,
        "--namespace=%s" % NAMESPACE_PLACEHOLDER,
        "--operator-image=%s" % pullURL,
        "--watch-namespace=%s" % NAMESPACE_PLACEHOLDER,
        "--dump-crds",
    ]

    run_pod(
        args,
        get_env_var("NODE_MAINTENANCE_OPERATOR_IMAGE_PULL_URL"),
        buildInfo["name"]
    )


def create_all_csvs():
    create_cna_csv()
    create_virt_csv()
    create_ssp_csv()
    create_nmo_csv()
    create_cdi_csv()


create_all_csvs()
write_csvs_crds()
