#!/bin/bash -ex

# begin of hack - remove when we are sure that CPaaS always pass CPAAS_PRODUCT_VERSION
# due to some bug CPaaS sometimes forgets to properly pass CPAAS_PRODUCT_VERSION
# the right approach is to fail the build but, in order to avoid getting stuck now,
# we can try guessing its value from HYPERCONVERGED_CLUSTER_OPERATOR_IMAGE_PULL_URL
# which is basically an hack due to the facts that attaching an HCO build from the wrong branch
# now can potentially shift the whole compose to a different version.
if [ -z "$CPAAS_PRODUCT_VERSION" ]; then
    echo "CPAAS_PRODUCT_VERSION is unset, trying to guess its value from HYPERCONVERGED_CLUSTER_OPERATOR_IMAGE_PULL_URL"
    case "$HYPERCONVERGED_CLUSTER_OPERATOR_IMAGE_PULL_URL" in
        *2\.2\.0*)
                  CPAAS_PRODUCT_VERSION="2.2.0"
        ;;
        *2\.3\.0*)
                  CPAAS_PRODUCT_VERSION="2.3.0"
        ;;
        *)
                  echo "FATAL: CPAAS_PRODUCT_VERSION is unset and I also failed guessing its value from HYPERCONVERGED_CLUSTER_OPERATOR_IMAGE_PULL_URL"
                  exit 2
        ;;
    esac
fi
# end of hack

readonly CSV_MERGER_POD_NAME=csv-merger
readonly BUNDLE_DIR="olm-catalog/kubevirt-hyperconverged/${CPAAS_PRODUCT_VERSION}"
readonly CSV_FILE_PATH="${BUNDLE_DIR}/kubevirt-hyperconverged-operator.v${CPAAS_PRODUCT_VERSION}.clusterserviceversion.yaml"
readonly TEMP_CSV_FILE_PATH="CSVs/kubevirt-hyperconverged-operator.v${CPAAS_PRODUCT_VERSION}.clusterserviceversion.yaml"

case ${CPAAS_PRODUCT_VERSION} in
  2.2.0)
         REPLACES_PRODUCT_VERSION="2.1.0"
         IMAGE_SHA_DIGEST=false
  ;;
  2.3.0)
         REPLACES_PRODUCT_VERSION="2.2.0"
         IMAGE_SHA_DIGEST=true
  ;;
  *)
         echo "Unsupported release: ${CPAAS_PRODUCT_VERSION}"
         exit 2
  ;;
esac

if [ -z "$MANUAL_BUILD" ]
then
  hco_image_pipeline="${HYPERCONVERGED_CLUSTER_OPERATOR_IMAGE_PULL_URL:?}"
else
  echo "!!! Manual execution !!!"
  python update_env_vars.py
  source downstream_vars
  export $(cut -d= -f1 downstream_vars)
  hco_image_pipeline=$(echo ${HYPERCONVERGED_CLUSTER_OPERATOR_BUILD_INFO_JSON} | python3 -c 'import json,sys;obj=json.load(sys.stdin);print(obj["extra"]["image"]["index"]["pull"][1])')
fi


case $IMAGE_SHA_DIGEST in
  (true)
        hco_image_csv=$(echo ${HYPERCONVERGED_CLUSTER_OPERATOR_BUILD_INFO_JSON} | python3 -c 'import json,sys;obj=json.load(sys.stdin);print(obj["extra"]["image"]["index"]["pull"][0])')
        ims_v2v_image=$(echo ${KUBEVIRT_V2V_CONVERSION_BUILD_INFO_JSON} | python3 -c 'import json,sys;obj=json.load(sys.stdin);print(obj["extra"]["image"]["index"]["pull"][0])')
        ims_vmware_image=$(echo ${KUBEVIRT_VMWARE_BUILD_INFO_JSON} | python3 -c 'import json,sys;obj=json.load(sys.stdin);print(obj["extra"]["image"]["index"]["pull"][0])')
  ;;
  (false)
        hco_image_csv=$(echo ${HYPERCONVERGED_CLUSTER_OPERATOR_BUILD_INFO_JSON} | python3 -c 'import json,sys;obj=json.load(sys.stdin);print(obj["extra"]["image"]["index"]["pull"][1])')
        ims_v2v_image=$(echo ${KUBEVIRT_V2V_CONVERSION_BUILD_INFO_JSON} | python3 -c 'import json,sys;obj=json.load(sys.stdin);print(obj["extra"]["image"]["index"]["pull"][1])')
        ims_vmware_image=$(echo ${KUBEVIRT_VMWARE_BUILD_INFO_JSON} | python3 -c 'import json,sys;obj=json.load(sys.stdin);print(obj["extra"]["image"]["index"]["pull"][1])')
  ;;
esac

get_fcontent(){
    csv_path=$1
    if test -f "${csv_path}"; then
        cat ${csv_path}
        return
    fi
    echo "ERROR: ${csv_path} is missing."
    exit 1
}

clean_env(){
    rm -rf CSVs
    rm -rf CRDs
}

init_env(){
    clean_env
    mkdir CSVs
    mkdir CRDs
}



# Create 'CSVs' and 'CRDs' directories
init_env


# Create all CSV and CRDs files
IMAGE_SHA_DIGEST=${IMAGE_SHA_DIGEST} python3 cnv-operators-csv-generator.py
if [[ $? != 0 ]]; then
    echo "ERROR: CSVs generation failed"
    exit 1
fi

cnaCsv=$(get_fcontent "CSVs/cluster-network-addons-operator-container.csv")
virtCsv=$(get_fcontent "CSVs/virt-operator-container.csv")
sspCsv=$(get_fcontent "CSVs/kubevirt-ssp-operator-container.csv")
cdiCsv=$(get_fcontent "CSVs/virt-cdi-operator-container.csv")
nmoCsv=$(get_fcontent "CSVs/node-maintenance-operator-container.csv")
hppCsv=$(get_fcontent "CSVs/hostpath-provisioner-operator-container.csv")
csvOverrides=$(get_fcontent "csvOverrides.yaml")

imageList=$(get_fcontent "CSVs/imagelist.txt")
specDescription=$(get_fcontent "description.txt")
specDisplayName="Container-native virtualization"
metadataDescription="Creates and maintains a CNV Deployment"
crdDisplay="CNV Operator"

case ${CPAAS_PRODUCT_VERSION} in
  2.2.0)
      CSV_OVERRIDES=""
      CRD_DISPLAY=""
  ;;
  *)
      CSV_OVERRIDES="--csv-overrides=${csvOverrides}"
      CRD_DISPLAY="--crd-display=${crdDisplay}"
  ;;
esac



# OC - HCO - Using all the other CSVs
run_with_oc(){
    oc run \
        "${CSV_MERGER_POD_NAME}" \
        -i -t \
        --serviceaccount='jenkins' \
        --quiet=true \
        --image="${hco_image_pipeline}" \
        --restart="Never" \
        --command -- /usr/local/bin/csv-merger \
        --cna-csv="${cnaCsv}" \
        --virt-csv="${virtCsv}" \
        --ssp-csv="${sspCsv}" \
        --cdi-csv="${cdiCsv}" \
        --nmo-csv="${nmoCsv}" \
        --hpp-csv="${hppCsv}" \
        --ims-conversion-image-name="${ims_v2v_image}" \
        --ims-vmware-image-name="${ims_vmware_image}" \
        --operator-image-name="${hco_image_csv}" \
        --csv-version=${CPAAS_PRODUCT_VERSION} \
        --replaces-csv-version=${REPLACES_PRODUCT_VERSION} \
        --spec-description="${specDescription}" \
        --spec-displayname="${specDisplayName}" \
        --metadata-description="${metadataDescription}" \
        --related-images-list="${imageList}" \
        "${CSV_OVERRIDES}" \
        "${CRD_DISPLAY}" \
    | sed -e "s/\r//" > "${TEMP_CSV_FILE_PATH}"

    cp CRDs/* "${BUNDLE_DIR}"
}

clean_pod() {
    oc delete pod --ignore-not-found "${CSV_MERGER_POD_NAME}" 2>/dev/null
}

# DOCKER - HCO - Using all the other CSVs
run_with_docker(){
    docker run \
        --rm --entrypoint=/usr/local/bin/csv-merger \
        "${hco_image_pipeline}" \
        --cna-csv="${cnaCsv}" \
        --virt-csv="${virtCsv}" \
        --ssp-csv="${sspCsv}" \
        --cdi-csv="${cdiCsv}" \
        --nmo-csv="${nmoCsv}" \
        --hpp-csv="${hppCsv}" \
        --ims-conversion-image-name="${ims_v2v_image}" \
        --ims-vmware-image-name="${ims_vmware_image}" \
        --operator-image-name="${hco_image_csv}" \
        --csv-version=${CPAAS_PRODUCT_VERSION} \
        --replaces-csv-version=${REPLACES_PRODUCT_VERSION} \
        --spec-description="${specDescription}" \
        --spec-displayname="${specDisplayName}" \
        --metadata-description="${metadataDescription}" \
        --related-images-list="${imageList}" \
        "${CSV_OVERRIDES}" \
        "${CRD_DISPLAY}" \
    | sed -e "s/\r//" > "${TEMP_CSV_FILE_PATH}"

    cp CRDs/* "${BUNDLE_DIR}"
}

if [[ -z "${MANUAL_BUILD}" ]]; then
  # TODO: Add kubectl
  if [ -x /usr/bin/oc ]; then
    clean_pod
    trap 'clean_pod' EXIT
    run_with_oc
  elif [ -x /usr/bin/docker ]; then
    run_with_docker
  else
    echo "ERROR: Nor oc or docker are installed."
    exit 1
  fi
else
  run_with_docker
fi

echo "Checking CSV diff"
diff ${TEMP_CSV_FILE_PATH} ${CSV_FILE_PATH} || :

if  [ $(diff ${TEMP_CSV_FILE_PATH} ${CSV_FILE_PATH} | grep '^[<>]' | wc -l) == 2 ] && \
    [ "$(diff ${TEMP_CSV_FILE_PATH} ${CSV_FILE_PATH} | grep -e '<*createdAt:')" != "" ]
then
    echo "The CSV has not changed"
else
    cp ${TEMP_CSV_FILE_PATH} ${CSV_FILE_PATH}
fi

clean_env

