#!/usr/bin/python3

import datetime
import json
import os
import subprocess
import logging

NAMESPACE_PLACEHOLDER = "openshift-cnv"
new_csv_version = "2.2.0"
old_csv_version = "2.1.0"

OC_CMD = ['oc', 'version']
DOCKER_CMD = ['docker', '--version']

YAML_DELIMITER = '---'

# Keep the CSVs and CRDs content in a dict to be written only after
# all the CSVs and CRDs were generated.
csvs = {}
crds = {}

logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.INFO)


def writeTofile(file_path, data):
    with open(file_path, "w") as f:
        f.write(data.replace('\r\n', '\n'))


def remove_pod_oc(podName):
    subprocess.run(
        [
            'oc', 'delete', 'pod', '--ignore-not-found',
            podName
        ],
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE
    )


def is_cmd_available(cmd):
    try:
        subprocess.run(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    except BaseException:
        return False

    return True


def save_csv_crd(operatorName, output):
    odocs = output.split(YAML_DELIMITER)
    if '' in odocs:
        odocs.remove('')
    if len(odocs) < 2:
        raise Exception('Failed splitting %s CSV and CRDs' % operatorName)

    csvs[operatorName] = YAML_DELIMITER + '\n' + odocs[0]
    crds[operatorName] = YAML_DELIMITER + YAML_DELIMITER.join(odocs[1:])


def save_crd(operatorName, output):
    crds[operatorName] = output


def split_crds(operatorName, crd_data):
    crds = crd_data.split('---')
    idx = 1
    for crd in crds[1:]:
        CRD_save_path = "%s/CRDs/%s.%d.crd.yml" % (
            os.getcwd(), operatorName, idx
        )

        crd_to_write = "---%s" % crd

        logger.info(
            "Saving {operator_name} CRD {num} in {crd_path}".format(
                operator_name=operatorName,
                num=idx,
                crd_path=CRD_save_path
            )
        )
        writeTofile(CRD_save_path, crd_to_write)
        idx += 1


def write_csvs_crds():
    for operatorName, csv_data in csvs.items():
        CSV_save_path = "%s/CSVs/%s.csv" % (os.getcwd(), operatorName)
        logger.info(
            "Saving {operator_name} CSV in {csv_path}".format(
                operator_name=operatorName,
                csv_path=CSV_save_path
            )
        )
        writeTofile(CSV_save_path, csv_data)

    for operatorName, crd_data in crds.items():
        if crd_data.count('---') > 1:
            split_crds(operatorName, crd_data)
        else:
            CRD_save_path = "%s/CRDs/%s.crd.yml" % (os.getcwd(), operatorName)
            logger.info(
                "Saving {operator_name} CRD in {crd_path}".format(
                    operator_name=operatorName,
                    crd_path=CRD_save_path
                )
            )
            writeTofile(CRD_save_path, crd_data)
    imagelistfile = "%s/CSVs/imagelist.txt" % os.getcwd()
    writeTofile(imagelistfile, getAllImages())


def run_pod(
        operatorArgs,
        image_pull_url,
        operatorName, nvr,
        entrypoint="/usr/bin/csv-generator",
        crds_only=False):
    pod_name = nvr_to_pod_name(nvr)
    ocArgs = [
        "oc",
        "run",
        pod_name,
        "-i",
        "-t",
        "--quiet=true",
        "--image=%s" % image_pull_url,
        "--restart=Never",
        "--command",
        "--",
        entrypoint,
    ]

    dockerArgs = [
        "docker",
        "run",
        "--rm",
        "--entrypoint=%s" % entrypoint,
        image_pull_url,
    ]

    manual_build = False
    if "MANUAL_BUILD" in os.environ:
        manual_build = True

    tool = None

    if is_cmd_available(OC_CMD) and not manual_build:
        ocArgs.extend(operatorArgs)
        runArgs = ocArgs
        tool = "oc"
    elif is_cmd_available(DOCKER_CMD):
        dockerArgs.extend(operatorArgs)
        runArgs = dockerArgs
        tool = "docker"
    else:
        raise Exception(
            "Nor oc binary or docker are installed in the system. "
            "Cannot proceed"
        )

    try:
        cmd_result = subprocess.run(
            runArgs,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        )
    finally:
        if tool == "oc":
            remove_pod_oc(pod_name)

    if cmd_result.returncode != 0:
        print(runArgs)
        raise Exception(
            'Failed to execute %s, due to: %s' % (
                pod_name,
                cmd_result.stderr.decode("utf-8")
            )
        )
    if not crds_only:
        logger.info("Generated CSV and CRDs from %s" % image_pull_url)
        logger.info(
            "%s CSV+CRD %s=" % (
                operatorName,
                cmd_result.stdout.decode("utf-8")
            )
        )
        save_csv_crd(operatorName, cmd_result.stdout.decode("utf-8"))
    else:
        logger.info("Generated CRDs from %s" % image_pull_url)
        logger.info(
            "%s CRDs %s=" % (operatorName, cmd_result.stdout.decode("utf-8"))
        )
        save_crd(operatorName, cmd_result.stdout.decode("utf-8"))


def get_env_var(envVar):
    try:
        env_value = os.environ[envVar]
    except KeyError:
        raise Exception("Environment variable %s not found" % envVar)

    logger.info("Received EnvVar: %s=%s " % (envVar, env_value))
    return env_value


def getBuildInfo(envVar):
    return json.loads(get_env_var(envVar))


def getAllImages():
    if use_sha_digest():
        return ','.join([
            getShaPullURL(envk)+'|'+getPullURL(envk)
            for envk in os.environ
            if envk.endswith('_BUILD_INFO_JSON')
        ])
    return ','.join([
        getPullURL(envk)
        for envk in os.environ
        if envk.endswith('_BUILD_INFO_JSON')
    ])


def getSha(envVar):
    buildInfo = getBuildInfo(envVar)
    pullURL = buildInfo["extra"]["image"]["index"]["pull"][0]
    return "sha256:%s" % pullURL.split("@sha256:")[1]


def use_sha_digest():
    if 'IMAGE_SHA_DIGEST' in os.environ and os.environ[
        'IMAGE_SHA_DIGEST'
    ].lower() == 'true':
        return True
    return False


def get_product_version():
    return os.getenv('CPAAS_PRODUCT_VERSION', '0.0.0')


def getImageVersion(envVar):
    buildInfo = getBuildInfo(envVar)
    if use_sha_digest():
        key = list(buildInfo["extra"]["image"]["index"]["digests"].keys())[0]
        return buildInfo["extra"]["image"]["index"]["digests"][key]
    return buildInfo["extra"]["image"]["index"]["tags"][0]


def getImageName(pullURL):
    cname_with_version = pullURL.split('/')[-1]
    return cname_with_version.split(':')[0]


def getPullURL(envVar):
    buildInfo = getBuildInfo(envVar)
    return buildInfo["extra"]["image"]["index"]["pull"][1]


def getShaPullURL(envVar):
    buildInfo = getBuildInfo(envVar)
    return buildInfo["extra"]["image"]["index"]["pull"][0]


def getImageURL(envVar):
    if use_sha_digest():
        return getShaPullURL(envVar)
    return getPullURL(envVar)


def create_cna_csv():
    logger.info("Creating CNA csv.....")
    variableName = "CLUSTER_NETWORK_ADDONS_OPERATOR_BUILD_INFO_JSON"
    buildInfo = getBuildInfo(variableName)
    pullURL = getImageURL(variableName)

    # multus is used only on upstream
    multus_image = "brew-pulp-docker01.web.prod.ext.phx2.redhat.com:8888/" \
                   "openshift/ose-multus-cni:v4.1.0-7.1556622314"

    bases = pullURL.split('/')
    cprefix = '/'.join(bases[:-1])
    iname = bases[-1].split(':')[0]
    operator_version = getImageVersion(variableName)
    container_tag = operator_version
    if container_tag.startswith("sha256:"):
        container_tag = operator_version[len("sha256:"):]

    args = [
        "--version=%s" % new_csv_version,
        "--version-replaces=%s" % old_csv_version,
        "--operator-version=%s" % operator_version,
        "--namespace=%s" % NAMESPACE_PLACEHOLDER,
        "--container-prefix=%s" % cprefix,
        "--image-name=%s" % iname,
        "--container-tag=%s" % container_tag,
        "--multus-image=%s" % multus_image,
        "--linux-bridge-cni-image=%s" % getImageURL(
            "CNV_CONTAINERNETWORKING_PLUGINS_BUILD_INFO_JSON"
        ),
        "--linux-bridge-marker-image=%s" % getImageURL(
            "BRIDGE_MARKER_BUILD_INFO_JSON"
        ),
        "--kubemacpool-image=%s" % getImageURL(
            "KUBEMACPOOL_BUILD_INFO_JSON"
        ),
        "--nm-state-handler-image=%s" % getImageURL(
            "KUBERNETES_NMSTATE_HANDLER_BUILD_INFO_JSON"
        ),
        "--ovs-cni-image=%s" % getImageURL(
            "OVS_CNI_PLUGIN_BUILD_INFO_JSON"
        ),
        "--ovs-marker-image=%s" % getImageURL(
            "OVS_CNI_MARKER_BUILD_INFO_JSON"
        ),
        "--image-pull-policy=IfNotPresent",
        "--dump-crds",
    ]

    run_pod(
        args,
        get_env_var("CLUSTER_NETWORK_ADDONS_OPERATOR_IMAGE_PULL_URL"),
        buildInfo["name"],
        buildInfo["nvr"]
    )


def create_virt_csv():
    logger.info("Creating VIRT csv....")
    variableName = "VIRT_OPERATOR_BUILD_INFO_JSON"
    buildInfo = getBuildInfo(variableName)
    pullURL = getImageURL(variableName)
    imagePrefix = "container-native-virtualization-"

    bases = pullURL.split('/')
    cprefix = '/'.join(bases[:-1])

    args = [
        "--namespace=%s" % NAMESPACE_PLACEHOLDER,
        "--operatorImageVersion=%s" % getImageVersion(variableName),
        "--dockerPrefix=%s" % cprefix,
        "--kubeVirtVersion=%s" % getImageVersion(variableName),
        "--imagePrefix=%s" % imagePrefix,
        "--csvVersion=%s" % new_csv_version,
        "--replacesCsvVersion=%s" % old_csv_version,
        "--csvCreatedAtTimestamp=%s" % str(datetime.datetime.now()),
        "--dumpCRDs",
    ]
    if use_sha_digest():
        sha_arg = [
            "--apiSha=%s" % getSha("VIRT_API_BUILD_INFO_JSON"),
            "--controllerSha=%s" % getSha(
                "VIRT_CONTROLLER_BUILD_INFO_JSON"
            ),
            "--handlerSha=%s" % getSha("VIRT_HANDLER_BUILD_INFO_JSON"),
            "--launcherSha=%s" % getSha("VIRT_LAUNCHER_BUILD_INFO_JSON"),
            "--kubevirtLogo=kubevirtLogo",
        ]
        args = args + sha_arg

    run_pod(
        args,
        get_env_var("VIRT_OPERATOR_IMAGE_PULL_URL"),
        buildInfo["name"],
        buildInfo["nvr"]
    )


def create_ssp_csv():
    logger.info("Creating SSP csv.....")
    buildInfo = getBuildInfo("KUBEVIRT_SSP_OPERATOR_BUILD_INFO_JSON")
    pullURL = getImageURL("KUBEVIRT_SSP_OPERATOR_BUILD_INFO_JSON")
    imagePrefix = "container-native-virtualization-"
    args = [
        "--csv-version=%s" % new_csv_version,
        "--namespace=%s" % NAMESPACE_PLACEHOLDER,
        "--operator-image=%s" % pullURL,
        "--image-name-prefix=%s" % imagePrefix,
        "--kvm-info-tag=%s" % getImageVersion(
            "KUBEVIRT_KVM_INFO_NFD_PLUGIN_BUILD_INFO_JSON"
        ),
        "--validator-tag=%s" % getImageVersion(
            "KUBEVIRT_TEMPLATE_VALIDATOR_BUILD_INFO_JSON"
        ),
        "--virt-launcher-tag=%s" % getImageVersion(
            "VIRT_LAUNCHER_BUILD_INFO_JSON"
        ),
        "--node-labeller-tag=%s" % getImageVersion(
            "KUBEVIRT_CPU_NODE_LABELLER_BUILD_INFO_JSON"
        ),
        "--cpu-plugin-tag=%s" % getImageVersion(
            "KUBEVIRT_CPU_MODEL_NFD_PLUGIN_BUILD_INFO_JSON"
        ),
        "--dump-crds",
    ]

    run_pod(
        args,
        get_env_var("KUBEVIRT_SSP_OPERATOR_IMAGE_PULL_URL"),
        buildInfo["name"],
        buildInfo["nvr"]
    )


def create_cdi_csv():
    logger.info("Creating CDI csv.....")
    variableName = "VIRT_CDI_OPERATOR_BUILD_INFO_JSON"
    buildInfo = getBuildInfo(variableName)
    pullURL = getImageURL(variableName)

    args = [
        "--csv-version=%s" % new_csv_version,
        "--replaces-csv-version=%s" % old_csv_version,
        "--namespace=" + NAMESPACE_PLACEHOLDER,
        "--cdi-logo-base64=logo",
        "--operator-version=%s" % 'v' + get_product_version(),
        "--operator-image=%s" % pullURL,
        "--controller-image=%s" % getImageURL(
            "VIRT_CDI_CONTROLLER_BUILD_INFO_JSON"
        ),
        "--importer-image=%s" % getImageURL(
            "VIRT_CDI_IMPORTER_BUILD_INFO_JSON"
        ),
        "--cloner-image=%s" % getImageURL(
            "VIRT_CDI_CLONER_BUILD_INFO_JSON"
        ),
        "--apiserver-image=%s" % getImageURL(
            "VIRT_CDI_APISERVER_BUILD_INFO_JSON"
        ),
        "--uploadproxy-image=%s" % getImageURL(
            "VIRT_CDI_UPLOADPROXY_BUILD_INFO_JSON"
        ),
        "--uploadserver-image=%s" % getImageURL(
            "VIRT_CDI_UPLOADSERVER_BUILD_INFO_JSON"
        ),
        "--pull-policy=IfNotPresent",
        "--dump-crds",
    ]

    run_pod(
        args,
        get_env_var("VIRT_CDI_OPERATOR_IMAGE_PULL_URL"),
        buildInfo["name"],
        buildInfo["nvr"]
    )


def create_nmo_csv():
    logger.info("Creating NMO csv.....")
    buildInfo = getBuildInfo("NODE_MAINTENANCE_OPERATOR_BUILD_INFO_JSON")
    pullURL = getImageURL("NODE_MAINTENANCE_OPERATOR_BUILD_INFO_JSON")
    args = [
        "--csv-version=%s" % new_csv_version,
        "--namespace=%s" % NAMESPACE_PLACEHOLDER,
        "--operator-image=%s" % pullURL,
        "--watch-namespace=%s" % NAMESPACE_PLACEHOLDER,
        "--dump-crds",
    ]

    run_pod(
        args,
        get_env_var("NODE_MAINTENANCE_OPERATOR_IMAGE_PULL_URL"),
        buildInfo["name"],
        buildInfo["nvr"]
    )


def create_hpp_csv():
    logger.info("Creating HPP csv.....")
    buildInfo = getBuildInfo("HOSTPATH_PROVISIONER_OPERATOR_BUILD_INFO_JSON")
    op_pullURL = getImageURL("HOSTPATH_PROVISIONER_OPERATOR_BUILD_INFO_JSON")
    pullURL = getImageURL("HOSTPATH_PROVISIONER_BUILD_INFO_JSON")
    args = [
        "--csv-version=%s" % new_csv_version,
        "--operator-image-name=%s" % op_pullURL,
        "--namespace=%s" % NAMESPACE_PLACEHOLDER,
        "--pull-policy=IfNotPresent",
        "--provisioner-image-name=%s" % pullURL,
        "--dump-crds",
    ]

    run_pod(
        args,
        get_env_var("HOSTPATH_PROVISIONER_OPERATOR_IMAGE_PULL_URL"),
        buildInfo["name"],
        buildInfo["nvr"]
    )


def create_all_csvs():
    create_cna_csv()
    create_virt_csv()
    create_ssp_csv()
    create_nmo_csv()
    create_cdi_csv()
    create_hpp_csv()


def create_hco_crds():
    if get_product_version()[:3] != "2.2":
        logger.info("Creating HCO CRDs.....")
        buildInfo = getBuildInfo(
            "HYPERCONVERGED_CLUSTER_OPERATOR_BUILD_INFO_JSON"
        )
        args = [
            "--output-mode=CRDs",
            "--namespace=%s" % NAMESPACE_PLACEHOLDER,
        ]
        run_pod(
            args,
            get_env_var("HYPERCONVERGED_CLUSTER_OPERATOR_IMAGE_PULL_URL"),
            buildInfo["name"],
            buildInfo["nvr"],
            entrypoint="/usr/local/bin/csv-merger",
            crds_only=True,
        )


def nvr_to_pod_name(nvr):
    return nvr.lower().replace('.', '-') + '-pod'


create_all_csvs()
create_hco_crds()
write_csvs_crds()
