#!/usr/bin/env python
import koji as brew
import json
import os

import pycurl
from io import BytesIO

CPAAS_PRODUCT_VERSION = os.environ['CPAAS_PRODUCT_VERSION']

ADDITIONAL_CHANNEL = None
if CPAAS_PRODUCT_VERSION == '2.2.0':
    DEFAULT_ERRATA = '45581'
    DEFAULT_CHANNEL = "RHEL-8-CNV-2.2"
    ADDITIONAL_CHANNEL = "RHEL-7-CNV-2.2"
elif CPAAS_PRODUCT_VERSION == '2.2.1':
    DEFAULT_ERRATA = '51253'
    DEFAULT_CHANNEL = "RHEL-8-CNV-2.2"
    ADDITIONAL_CHANNEL = "RHEL-7-CNV-2.2"
elif CPAAS_PRODUCT_VERSION == '2.3.0':
    DEFAULT_ERRATA = '48747'
    DEFAULT_CHANNEL = "RHEL-8-CNV-2.3"
    ADDITIONAL_CHANNEL = "RHEL-7-CNV-2.3"
elif CPAAS_PRODUCT_VERSION == '2.3.1':
    DEFAULT_ERRATA = '53127'
    DEFAULT_CHANNEL = "RHEL-8-CNV-2.3"
    ADDITIONAL_CHANNEL = "RHEL-7-CNV-2.3"
elif CPAAS_PRODUCT_VERSION == '2.4.0':
    DEFAULT_ERRATA = '51552'
    DEFAULT_CHANNEL = "RHEL-8-CNV-2.4"
    ADDITIONAL_CHANNEL = "RHEL-7-CNV-2.4"
elif CPAAS_PRODUCT_VERSION == '2.5.0':
    DEFAULT_ERRATA = '51552'
    DEFAULT_CHANNEL = "RHEL-8-CNV-2.4"
    ADDITIONAL_CHANNEL = "RHEL-7-CNV-2.4"

else:
    raise RuntimeError(
        "Unsupported release: {r}".format(r=CPAAS_PRODUCT_VERSION)
    )


builds = {}


replacements = {
    'VM_IMPORT_CONTROLLER_OPERATOR_BUILD_INFO_JSON':
    'VM_IMPORT_CONTROLLER_BUILD_INFO_JSON',
}


def fetch(url, secure=True):
    buffer = BytesIO()
    c = pycurl.Curl()
    c.setopt(c.URL, url)
    c.setopt(c.WRITEFUNCTION, buffer.write)
    c.setopt(c.HTTPAUTH, c.HTTPAUTH_GSSNEGOTIATE)
    c.setopt(c.USERPWD, ':')
    c.setopt(c.USERAGENT, 'Python Client')
    if not secure:
        c.setopt(c.SSL_VERIFYPEER, False)
        c.setopt(c.SSL_VERIFYHOST, False)
    c.perform()
    response_code = c.getinfo(c.RESPONSE_CODE)
    c.close()

    if response_code == 401:
        raise RuntimeError(
            'auth error: please run kinit to get a valid kerberos ticket'
        )
    elif response_code != 200:
        raise RuntimeError(
            'pycurl error {rc}'.format(rc=response_code)
        )

    data = buffer.getvalue()
    buffer.close()
    jdata = json.loads(data)
    return jdata


def get_errata_images(errata, secure, channel=DEFAULT_CHANNEL):
    url = 'https://errata.devel.redhat.com/api/v1/erratum/{e}/builds'.format(
        e=errata,
    )

    errata_out = fetch(url, secure=secure)
    # import pdb; pdb.set_trace()
    for i in errata_out[channel]["builds"]:
        build = list(i.keys())[0]
        build_id = i[build]["id"]

        env_var = build.split(
            '-container-'
        )[0].replace('-', '_').upper() + "_BUILD_INFO_JSON"
        if env_var in replacements:
            env_var = replacements[env_var]
        builds[env_var] = build_id


def main():
    hub = brew.ClientSession('http://brewhub.devel.redhat.com/brewhub')
    get_errata_images(DEFAULT_ERRATA, True)
    if ADDITIONAL_CHANNEL:
        get_errata_images(DEFAULT_ERRATA, True, ADDITIONAL_CHANNEL)

    with open("downstream_vars", "w") as f:
        for env_var in builds.keys():
            print(
                "Pulling buildinfo for build: %s for env_var: %s" % (
                    builds[env_var],
                    env_var
                )
            )
            build_info_dict = hub.getBuild(int(builds[env_var]))
            f.write("%s=\'%s\'\n" % (env_var, json.dumps(build_info_dict)))
            f.write("%s=\'%s\'\n" % (
                env_var.replace("_BUILD_INFO_JSON", "_IMAGE_PULL_URL"),
                build_info_dict["extra"]["image"]["index"]["pull"][1]
            ))


if __name__ == "__main__":
    main()
