package main

import (
	"fmt"
	"kubevirt.io/hostpath-provisioner-operator/pkg/controller/hostpathprovisioner"
	"sort"
	"strings"
)

// constant parts of the file
const (
	title      = "# hostpath-provisioner-operator metrics\n"
	background = "This document aims to help users that are not familiar with metrics exposed by the hostpath-provisioner-operator.\n" +
		"All metrics documented here are auto-generated by the utility tool `tools/metricsdocs` and reflects exactly what is being exposed.\n\n"

	KVSpecificMetrics = "## hostpath-provisioner-operator Metrics List\n"

	opening = title +
		background +
		KVSpecificMetrics

	// footer
	footerHeading = "## Developing new metrics\n"
	footerContent = "After developing new metrics or changing old ones, please run `make generate-doc` to regenerate this document.\n\n" +
		"If you feel that the new metric doesn't follow these rules, please change `tools/metricsdocs` with your needs.\n"

	footer = footerHeading + footerContent
)

func main() {
	metricsList := recordRulesDescToMetricList(hostpathprovisioner.GetRecordRulesDesc(""))
	sort.Sort(metricsList)
	writeToFile(metricsList)
}

func writeToFile(metricsList metricList) {
	fmt.Print(opening)
	metricsList.writeOut()
	fmt.Print(footer)
}

type metric struct {
	name        string
	description string
	mtype       string
}

func recordRulesDescToMetricList(mdl []hostpathprovisioner.RecordRulesDesc) metricList {
	res := make([]metric, len(mdl))
	for i, md := range mdl {
		res[i] = metricDescriptionToMetric(md)
	}

	return res
}

func metricDescriptionToMetric(rrd hostpathprovisioner.RecordRulesDesc) metric {
	return metric{
		name:        rrd.Name,
		description: rrd.Description,
		mtype:       rrd.Type,
	}
}

func (m metric) writeOut() {
	fmt.Println("###", m.name)
	fmt.Println(m.description + ". Type: " + m.mtype + ".")
}

type metricList []metric

// Len implements sort.Interface.Len
func (m metricList) Len() int {
	return len(m)
}

// Less implements sort.Interface.Less
func (m metricList) Less(i, j int) bool {
	return m[i].name < m[j].name
}

// Swap implements sort.Interface.Swap
func (m metricList) Swap(i, j int) {
	m[i], m[j] = m[j], m[i]
}

func (m *metricList) add(line string) {
	split := strings.Split(line, " ")
	name := split[2]
	split[3] = strings.Title(split[3])
	description := strings.Join(split[3:], " ")
	*m = append(*m, metric{name: name, description: description})
}

func (m metricList) writeOut() {
	for _, met := range m {
		met.writeOut()
	}
}
