package components

import (
	"fmt"
	"strings"

	appsv1 "k8s.io/api/apps/v1"
	corev1 "k8s.io/api/core/v1"
	rbacv1 "k8s.io/api/rbac/v1"
	extv1beta1 "k8s.io/apiextensions-apiserver/pkg/apis/apiextensions/v1beta1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

const Name = "kubevirt-web-ui-operator"

func getBaseTag(tag string) string {
	last := strings.LastIndex(tag, "-")
	if last < 0 {
		last = len(tag)
	}
	return tag[0:last]
}

func GetDeployment(namespace string, repository string, tag string, imagePullPolicy string) *appsv1.Deployment {
	image := fmt.Sprintf("%s/%s:%s", repository, Name, tag)
	deployment := &appsv1.Deployment{
		TypeMeta: metav1.TypeMeta{
			APIVersion: "apps/v1",
			Kind:       "Deployment",
		},
		ObjectMeta: metav1.ObjectMeta{
			Name:      Name,
			Namespace: namespace,
		},
		Spec: appsv1.DeploymentSpec{
			Replicas: int32Ptr(1),
			Selector: &metav1.LabelSelector{
				MatchLabels: map[string]string{
					"name": Name,
				},
			},
			Template: corev1.PodTemplateSpec{
				ObjectMeta: metav1.ObjectMeta{
					Labels: map[string]string{
						"name": Name,
					},
				},
				Spec: corev1.PodSpec{
					ServiceAccountName: Name,
					Containers: []corev1.Container{
						{
							Name:            Name,
							Image:           image,
							ImagePullPolicy: corev1.PullPolicy(imagePullPolicy),
							Ports: []corev1.ContainerPort{
								{
									Name:          "metrics",
									Protocol:      corev1.ProtocolTCP,
									ContainerPort: 60000,
								},
							},
							ReadinessProbe: &corev1.Probe{
								Handler: corev1.Handler{
									Exec: &corev1.ExecAction{
										Command: []string{
											"stat",
											"/tmp/operator-sdk-ready",
										},
									},
								},
								InitialDelaySeconds: 4,
								PeriodSeconds:       10,
								FailureThreshold:    1,
							},
							Env: []corev1.EnvVar{
								{
									Name:  "OPERATOR_NAME",
									Value: Name,
								},
								{
									Name:  "OPERATOR_REGISTRY",
									Value: repository,
								},
								{
									Name:  "OPERATOR_TAG",
									Value: tag,
								},
								{
									Name:  "WEBUI_TAG",
									Value: getBaseTag(tag),
								},
								{
									Name:  "BRANDING",
									Value: "okdvirt", // or openshiftvirt
								},
								{
									Name:  "IMAGE_PULL_POLICY",
									Value: imagePullPolicy,
								},
								{
									Name: "OPERATOR_NAMESPACE",
									ValueFrom: &corev1.EnvVarSource{
										FieldRef: &corev1.ObjectFieldSelector{
											FieldPath: "metadata.namespace",
										},
									},
								},
								{
									Name: "POD_NAME",
									ValueFrom: &corev1.EnvVarSource{
										FieldRef: &corev1.ObjectFieldSelector{
											FieldPath: "metadata.name",
										},
									},
								},
								{
									Name: "WATCH_NAMESPACE",
									ValueFrom: &corev1.EnvVarSource{
										FieldRef: &corev1.ObjectFieldSelector{
											FieldPath: "metadata.namespace",
										},
									},
								},
							},
						},
					},
				},
			},
		},
	}
	return deployment
}


// The GetRole() is probably not needed since the use of `components.go` implicates operator's deployment for cluster-scope (within HCO).
// To avoid confusion: If deployed independently using the deploy/*.yaml files, the operator is namespace-scoped.
// TODO: validate the statement above and optionally remove this GetRole() function.
func GetRole(namespace string) *rbacv1.Role {
	role := &rbacv1.Role{
		TypeMeta: metav1.TypeMeta{
			APIVersion: "rbac.authorization.k8s.io/v1",
			Kind:       "Role",
		},
		ObjectMeta: metav1.ObjectMeta{
			Name:      Name,
			Namespace: namespace,
			Labels: map[string]string{
				"name": Name,
			},
		},
		Rules: []rbacv1.PolicyRule{
			{
				APIGroups: []string{
					"",
				},
				Resources: []string{
					"pods",
					"configmaps",
					"services",
					"endpoints",
					"persistentvolumeclaims",
					"events",
					"secrets",
					"replicationcontrollers",
					"serviceaccounts",
				},
				Verbs: []string{
					"get",
					"list",
					"watch",
					"create",
					"patch",
					"update",
					"delete",
				},
			},
			{
				APIGroups: []string{
					"extensions",
					"apps",
				},
				Resources: []string{
					"deployments",
					"replicasets",
				},
				Verbs: []string{
					"get",
					"list",
					"watch",
					"create",
					"patch",
					"update",
					"delete",
				},
			},
			{
				APIGroups: []string{
					"",
				},
				Resources: []string{
					"namespaces",
				},
				Verbs: []string{
					"get",
					"list",
				},
			},
			{
				APIGroups: []string{
					"apps",
				},
				Resources: []string{
					"namespaces",
					"deployments",
					"daemonsets",
					"replicasets",
					"statefulsets",
				},
				Verbs: []string{
					"get",
					"list",
					"watch",
					"create",
					"patch",
					"update",
					"delete",
				},
			},
			{
				APIGroups: []string{
					"monitoring.coreos.com",
				},
				Resources: []string{
					"servicemonitors",
				},
				Verbs: []string{
					"get",
					"create",
				},
			},
			{
				APIGroups: []string{
					"kubevirt.io",
					"template.openshift.io",
					"route.openshift.io",
				},
				Resources: []string{
					"*",
				},
				Verbs: []string{
					"*",
				},
			},
		},
	}
	return role
}

func GetClusterRole() *rbacv1.ClusterRole {
	role := &rbacv1.ClusterRole{
		TypeMeta: metav1.TypeMeta{
			APIVersion: "rbac.authorization.k8s.io/v1",
			Kind:       "ClusterRole",
		},
		ObjectMeta: metav1.ObjectMeta{
			Name: Name,
			Labels: map[string]string{
				"name": Name,
			},
		},
		Rules: []rbacv1.PolicyRule{
			{
				APIGroups: []string{
					"oauth.openshift.io",
                                        "project.openshift.io",
                                        "template.openshift.io",
                                        "route.openshift.io",
					"apiextensions.k8s.io",
					"kubevirt.io",
					"extensions",
					"apps",
                                        "monitoring.coreos.com",
				},
				Resources: []string{
					"*",
				},
				Verbs: []string{
					"*",
				},
			},
			{
				APIGroups: []string{
					"",
				},
				Resources: []string{
					"configmaps",
					"pods",
					"namespaces",
					"services",
					"endpoints",
					"persistentvolumeclaims",
					"events",
					"secrets",
					"replicationcontrollers",
					"serviceaccounts",
					"statefulsets",

				},
				Verbs: []string{
					"*",
				},
			},
		},
	}
	return role
}

func GetCrd() *extv1beta1.CustomResourceDefinition {
	crd := &extv1beta1.CustomResourceDefinition{
		TypeMeta: metav1.TypeMeta{
			APIVersion: "apiextensions.k8s.io/v1beta1",
			Kind:       "CustomResourceDefinition",
		},
		ObjectMeta: metav1.ObjectMeta{
			Name: "kwebuis.kubevirt.io",
		},
		Spec: extv1beta1.CustomResourceDefinitionSpec{
			Group:   "kubevirt.io",
			Version: "v1alpha1",
			Scope:   "Cluster", // TODO: verify, yaml states "Namespaced"

			Names: extv1beta1.CustomResourceDefinitionNames{
				Plural:   "kwebuis",
				Singular: "kwebui",
				Kind:     "KWebUI",
				ListKind: "KWebUIList",
			},
		},
	}
	return crd
}

func int32Ptr(i int32) *int32 {
	return &i
}
