package main

import (
	"fmt"
	"sort"

	"github.com/kubevirt/hyperconverged-cluster-operator/pkg/metrics"
)

// constant parts of the file
const (
	title      = "# Hyperconverged Cluster Operator metrics\n"
	background = "This document aims to help users that are not familiar with metrics exposed by the Hyperconverged Cluster Operator.\n" +
		"All metrics documented here are auto-generated by the utility tool `tools/metricsdocs` and reflects exactly what is being exposed.\n\n"

	KVSpecificMetrics = "## Hyperconverged Cluster Operator Metrics List\n"

	opening = title +
		background +
		KVSpecificMetrics

	// footer
	footerHeading = "## Developing new metrics\n"
	footerContent = "After developing new metrics or changing old ones, please run `make generate-doc` to regenerate this document.\n\n" +
		"If you feel that the new metric doesn't follow these rules, please change `tools/metricsdocs` with your needs.\n"

	footer = footerHeading + footerContent
)

func main() {
	hcoMetricList := metricDescriptionListToMetricList(metrics.HcoMetrics.GetMetricDesc())
	sort.Sort(hcoMetricList)
	writeToFile(hcoMetricList)
}

func writeToFile(metricsList metricList) {
	fmt.Print(opening)
	metricsList.writeOut()
	fmt.Print(footer)
}

type metric struct {
	name        string
	description string
}

func metricDescriptionToMetric(md metrics.MetricDescription) metric {
	return metric{
		name:        md.FqName,
		description: md.Help,
	}
}

func (m metric) writeOut() {
	fmt.Println("###", m.name)
	fmt.Println(m.description)
}

type metricList []metric

func metricDescriptionListToMetricList(mdl []metrics.MetricDescription) metricList {
	res := make([]metric, len(mdl))
	for i, md := range mdl {
		res[i] = metricDescriptionToMetric(md)
	}

	return res
}

// Len implements sort.Interface.Len
func (m metricList) Len() int {
	return len(m)
}

// Less implements sort.Interface.Less
func (m metricList) Less(i, j int) bool {
	return m[i].name < m[j].name
}

// Swap implements sort.Interface.Swap
func (m metricList) Swap(i, j int) {
	m[i], m[j] = m[j], m[i]
}

func (m metricList) writeOut() {
	for _, met := range m {
		met.writeOut()
	}
}
