'use strict'

const { cwd } = process
const { homedir } = require('os')
const { isAbsolute, join, normalize, resolve, sep } = require('path')

const SEPARATOR_CC = sep === '\\' ? '[/\\\\]' : '/'
const DOT_RELATIVE_RX = new RegExp(`^\\.${SEPARATOR_CC}`)
const HOME_RELATIVE_RX = new RegExp(`^~${SEPARATOR_CC}`)
const PWD_RELATIVE_RX = new RegExp(`^~\\+${SEPARATOR_CC}`)

/**
 * Converts path to an absolute path.
 *
 * If path is absolute, its value is normalized. If the first segment of path
 * is a tilde (~), that segment is replaced with the home directory of the
 * current user. If the first segment of path is a tilde followed by a plus
 * (~+), that segment is replaced with the current working directory of the
 * process. If path is relative, path is resolved starting from start. If path
 * has a leading dot (.) segment, that segment is replaced with dot after that
 * value is expanded.
 *
 * @param {String} path - The path to expand.
 * @param {String} [start='~+'] - The path to use as a starting point for a
 *   relative path. Unless absolute, the value will first be expanded starting
 *   from the current working directory of the process.
 * @param {String} [dot='.'] - The value used to replace a leading dot segment.
 *   If the value is '.', then the start value is used.
 *
 * @returns {String} An absolute path.
 */
function expandPath (path, start = '~+', dot = '.') {
  if (path === '.') {
    if (dot === '.') dot = start
    return dot === '~' ? homedir() : (dot === '~+' ? cwd() : resolve(dot))
  } else if (path === '~') {
    return homedir()
  } else if (path === '~+') {
    return cwd()
  } else if (!path) {
    throw new TypeError('path must be a string. Received ' + path)
  }
  const ch0 = path.charAt()
  if (ch0 === '.') {
    if (DOT_RELATIVE_RX.test(path) && dot !== '.') start = dot
  } else if (ch0 === '~') {
    if (HOME_RELATIVE_RX.test(path)) {
      return join(homedir(), path.substr(2))
    } else if (PWD_RELATIVE_RX.test(path)) {
      return join(cwd(), path.substr(3))
    }
  } else if (isAbsolute(path)) {
    return normalize(path)
  }
  if (start === '.') start = dot
  return start === '~+' ? join(cwd(), path) : (start === '~' ? join(homedir(), path) : resolve(start, path))
}

module.exports = expandPath
