package rocksdb

import (
	"context"
	"testing"

	"github.com/stackrox/rox/generated/storage"
	"github.com/stackrox/rox/pkg/alert/convert"
	"github.com/stackrox/rox/pkg/fixtures"
	"github.com/stackrox/rox/pkg/rocksdb"
	"github.com/stackrox/rox/pkg/testutils/rocksdbtest"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestFullStoreImpl(t *testing.T) {
	ctx := context.Background()
	db, err := rocksdb.NewTemp(t.Name())
	require.NoError(t, err)
	defer rocksdbtest.TearDownRocksDB(db)

	store := NewFullStore(db)
	alert := fixtures.GetAlert()
	listAlert := convert.AlertToListAlert(alert)

	// Doesn't exist
	_, exists, err := store.ListAlert(ctx, alert.GetId())
	assert.NoError(t, err)
	assert.False(t, exists)

	// Explicitly test flows that are no encapsulated by autogenerated code
	assert.NoError(t, store.Upsert(ctx, alert))

	retrievedListAlert, exists, err := store.ListAlert(ctx, alert.GetId())
	assert.NoError(t, err)
	assert.True(t, exists)
	assert.Equal(t, listAlert, retrievedListAlert)

	// Add alert 2
	alert2 := fixtures.GetAlert()
	alert2.Id = "alert2"
	listAlert2 := convert.AlertToListAlert(alert2)
	assert.NoError(t, store.Upsert(ctx, alert2))

	listAlerts, missingIndices, err := store.GetListAlerts(ctx, []string{alert.GetId(), "alert3", alert2.GetId()})
	assert.NoError(t, err)
	assert.Equal(t, []int{1}, missingIndices)
	assert.Equal(t, []*storage.ListAlert{listAlert, listAlert2}, listAlerts)

	var retrievedListAlertsViaWalk []*storage.ListAlert
	err = store.Walk(ctx, func(la *storage.ListAlert) error {
		retrievedListAlertsViaWalk = append(retrievedListAlertsViaWalk, la)
		return nil
	})
	assert.NoError(t, err)
	assert.Equal(t, []*storage.ListAlert{listAlert, listAlert2}, retrievedListAlertsViaWalk)
}
