package resolvers

import (
	"context"
	"encoding/json"
	"fmt"
	"reflect"
	"testing"
	"time"

	"github.com/blevesearch/bleve"
	"github.com/gogo/protobuf/types"
	"github.com/golang/mock/gomock"
	"github.com/graph-gophers/graphql-go"
	"github.com/stackrox/rox/central/audit"
	clusterIndex "github.com/stackrox/rox/central/cluster/index"
	componentCVEEdgeDackBox "github.com/stackrox/rox/central/componentcveedge/dackbox"
	componentCVEEdgeDS "github.com/stackrox/rox/central/componentcveedge/datastore"
	componentCVEEdgeIndex "github.com/stackrox/rox/central/componentcveedge/index"
	componentCVEEdgeSearcher "github.com/stackrox/rox/central/componentcveedge/search"
	componentCVEEdgeStore "github.com/stackrox/rox/central/componentcveedge/store/dackbox"
	cveDackBox "github.com/stackrox/rox/central/cve/dackbox"
	cveDS "github.com/stackrox/rox/central/cve/datastore"
	cveIndex "github.com/stackrox/rox/central/cve/index"
	cveSearcher "github.com/stackrox/rox/central/cve/search"
	cveStore "github.com/stackrox/rox/central/cve/store/dackbox"
	deploymentMockDS "github.com/stackrox/rox/central/deployment/datastore/mocks"
	deploymentIndex "github.com/stackrox/rox/central/deployment/index"
	"github.com/stackrox/rox/central/globalindex"
	imageDackBox "github.com/stackrox/rox/central/image/dackbox"
	imageDS "github.com/stackrox/rox/central/image/datastore"
	imageIndex "github.com/stackrox/rox/central/image/index"
	componentDackBox "github.com/stackrox/rox/central/imagecomponent/dackbox"
	componentIndex "github.com/stackrox/rox/central/imagecomponent/index"
	imageComponentEdgeDackBox "github.com/stackrox/rox/central/imagecomponentedge/dackbox"
	imageComponentEdgeIndex "github.com/stackrox/rox/central/imagecomponentedge/index"
	imageCVEEdgeDackBox "github.com/stackrox/rox/central/imagecveedge/dackbox"
	imageCVEEdgeDS "github.com/stackrox/rox/central/imagecveedge/datastore"
	imageCVEEdgeIndex "github.com/stackrox/rox/central/imagecveedge/index"
	imageCVEEdgeSearcher "github.com/stackrox/rox/central/imagecveedge/search"
	imageCVEEdgeStore "github.com/stackrox/rox/central/imagecveedge/store/dackbox"
	nodeIndex "github.com/stackrox/rox/central/node/index"
	nodeComponentEdgeIndex "github.com/stackrox/rox/central/nodecomponentedge/index"
	notifierMocks "github.com/stackrox/rox/central/notifier/processor/mocks"
	"github.com/stackrox/rox/central/ranking"
	reprocessorMocks "github.com/stackrox/rox/central/reprocessor/mocks"
	riskManager "github.com/stackrox/rox/central/risk/manager/mocks"
	sensorConnMgrMocks "github.com/stackrox/rox/central/sensor/service/connection/mocks"
	vulnReqCache "github.com/stackrox/rox/central/vulnerabilityrequest/cache"
	vulnReqDS "github.com/stackrox/rox/central/vulnerabilityrequest/datastore"
	vulnReqManager "github.com/stackrox/rox/central/vulnerabilityrequest/manager/requestmgr"
	v1 "github.com/stackrox/rox/generated/api/v1"
	"github.com/stackrox/rox/generated/storage"
	"github.com/stackrox/rox/pkg/concurrency"
	"github.com/stackrox/rox/pkg/dackbox"
	"github.com/stackrox/rox/pkg/dackbox/indexer"
	"github.com/stackrox/rox/pkg/dackbox/utils/queue"
	"github.com/stackrox/rox/pkg/features"
	"github.com/stackrox/rox/pkg/fixtures"
	"github.com/stackrox/rox/pkg/grpc/authn"
	mockIdentity "github.com/stackrox/rox/pkg/grpc/authn/mocks"
	"github.com/stackrox/rox/pkg/rocksdb"
	"github.com/stackrox/rox/pkg/sac"
	"github.com/stackrox/rox/pkg/testutils/envisolator"
	"github.com/stackrox/rox/pkg/testutils/rocksdbtest"
	"github.com/stretchr/testify/suite"
)

var (
	allAllowedCtx = sac.WithGlobalAccessScopeChecker(context.Background(), sac.AllowAllAccessScopeChecker())
)

const (
	fakeUserID = "user-id-a"

	vulnerabilityRequestSelector = `
		id
		status
		targetState
		expired
		comments {
			message
			user {
				name
			}
		}
		requestor {
			name
		}
		approvers {
			name
		}
		scope {
			imageScope {
				registry
				remote
				tag
			}
		}
		deferralReq {
			expiresOn
			expiresWhenFixed
		}
		updatedDeferralReq {
			expiresOn
			expiresWhenFixed
		}
		cves {
			ids
		}`
)

// While it would've been nice to use the proto object, because of the oneofs and enums json unmarshalling into that object is a struggle
type vulnResponse struct {
	ID          string `json:"id"`
	Status      string `json:"status"`
	TargetState string `json:"targetState"`
	Expired     bool   `json:"expired"`
	Comments    []struct {
		Message string `json:"message"`
		User    struct {
			Name string `json:"name"`
		} `json:"user"`
	} `json:"comments"`
	Requestor struct {
		Name string `json:"name"`
	} `json:"requestor"`
	Approvers []struct {
		Name string `json:"name"`
	} `json:"approvers"`
	Scope struct {
		ImageScope struct {
			Registry string `json:"registry"`
			Remote   string `json:"remote"`
			Tag      string `json:"tag"`
		} `json:"imageScope"`
	} `json:"scope"`
	DeferralReq        *expiryResponse `json:"deferralReq"`
	UpdatedDeferralReq *expiryResponse `json:"updatedDeferralReq"`
	Cves               struct {
		Ids []string `json:"ids"`
	} `json:"cves"`
}

type expiryResponse struct {
	ExpiresOn        *time.Time `json:"expiresOn"`
	ExpiresWhenFixed bool       `json:"expiresWhenFixed"`
}

func TestVulnRequestsResolver(t *testing.T) {
	suite.Run(t, new(VulnRequestResolverTestSuite))
}

type VulnRequestResolverTestSuite struct {
	mockCtrl *gomock.Controller
	suite.Suite
	envIsolator *envisolator.EnvIsolator

	db                    *rocksdb.RocksDB
	indexQ                queue.WaitableQueue
	bleveIndex            bleve.Index
	vulnReqDataStore      vulnReqDS.DataStore
	deployments           *deploymentMockDS.MockDataStore
	cveDataStore          cveDS.DataStore
	imageDataStore        imageDS.DataStore
	componentCVEDataStore componentCVEEdgeDS.DataStore
	imageCVEDataStore     imageCVEEdgeDS.DataStore
	sensorConnMgrMocks    *sensorConnMgrMocks.MockManager
	riskManager           *riskManager.MockManager
	reprocessor           *reprocessorMocks.MockLoop
	vulnReqManager        vulnReqManager.Manager

	resolver *Resolver
	schema   *graphql.Schema

	mockContext context.Context
}

func (s *VulnRequestResolverTestSuite) TearDownTest() {
	rocksdbtest.TearDownRocksDB(s.db)
	s.envIsolator.RestoreAll()
	s.mockCtrl.Finish()
}

func (s *VulnRequestResolverTestSuite) SetupTest() {
	s.envIsolator = envisolator.NewEnvIsolator(s.T())
	s.envIsolator.Setenv(features.VulnRiskManagement.EnvVar(), "true")

	s.mockCtrl = gomock.NewController(s.T())
	s.db = rocksdbtest.RocksDBForT(s.T())

	bleveIndex, err := globalindex.MemOnlyIndex()
	s.Require().NoError(err)
	s.bleveIndex = bleveIndex

	s.indexQ = queue.NewWaitableQueue()
	dacky, err := dackbox.NewRocksDBDackBox(s.db, s.indexQ, []byte("graph"), []byte("dirty"), []byte("valid"))
	s.Require().NoError(err, "failed to create dackbox")

	reg := indexer.NewWrapperRegistry()
	indexer.NewLazy(s.indexQ, reg, bleveIndex, dacky.AckIndexed).Start()
	reg.RegisterWrapper(cveDackBox.Bucket, cveIndex.Wrapper{})
	reg.RegisterWrapper(componentDackBox.Bucket, componentIndex.Wrapper{})
	reg.RegisterWrapper(componentCVEEdgeDackBox.Bucket, componentCVEEdgeIndex.Wrapper{})
	reg.RegisterWrapper(imageDackBox.Bucket, imageIndex.Wrapper{})
	reg.RegisterWrapper(imageComponentEdgeDackBox.Bucket, imageComponentEdgeIndex.Wrapper{})
	reg.RegisterWrapper(imageCVEEdgeDackBox.Bucket, imageCVEEdgeIndex.Wrapper{})

	pendingReqCache, activeReqCache := vulnReqCache.New(), vulnReqCache.New()
	s.createImageDataStore(dacky)
	s.createCVEDataStore(dacky)
	s.createImageCVEDataStore(dacky)
	s.createComponentCVEDataStore(dacky)
	s.createVulnRequestDataStore(pendingReqCache, activeReqCache)
	s.deployments = deploymentMockDS.NewMockDataStore(s.mockCtrl)
	s.sensorConnMgrMocks = sensorConnMgrMocks.NewMockManager(s.mockCtrl)
	s.riskManager = riskManager.NewMockManager(s.mockCtrl)
	s.reprocessor = reprocessorMocks.NewMockLoop(s.mockCtrl)
	notifierMock := notifierMocks.NewMockProcessor(s.mockCtrl)

	id := mockIdentity.NewMockIdentity(s.mockCtrl)
	id.EXPECT().UID().Return(fakeUserID).AnyTimes()
	id.EXPECT().FullName().Return("First Last").AnyTimes()
	id.EXPECT().FriendlyName().Return("DefinitelyNotBob").AnyTimes()
	id.EXPECT().Permissions().Return(map[string]storage.Access{
		"VulnerabilityManagementApprovals": storage.Access_READ_WRITE_ACCESS,
		"VulnerabilityManagementRequests":  storage.Access_READ_WRITE_ACCESS,
	}).AnyTimes()
	s.mockContext = authn.ContextWithIdentity(sac.WithAllAccess(context.Background()), id, s.T())

	s.riskManager.EXPECT().CalculateRiskAndUpsertImage(gomock.Any()).Return(nil).AnyTimes()
	s.sensorConnMgrMocks.EXPECT().BroadcastMessage(gomock.Any()).AnyTimes()
	s.deployments.EXPECT().SearchDeployments(gomock.Any(), gomock.Any()).Return([]*v1.SearchResult{}, nil).AnyTimes()
	s.reprocessor.EXPECT().ReprocessRiskForDeployments(gomock.Any()).AnyTimes()

	s.vulnReqManager = vulnReqManager.New(
		s.deployments,
		s.vulnReqDataStore,
		pendingReqCache,
		vulnReqCache.New(),
		s.imageDataStore,
		s.imageCVEDataStore,
		s.cveDataStore,
		s.componentCVEDataStore,
		s.sensorConnMgrMocks,
		s.reprocessor,
	)

	notifierMock.EXPECT().HasEnabledAuditNotifiers().Return(false).AnyTimes()
	s.resolver = &Resolver{
		ImageDataStore: s.imageDataStore,
		CVEDataStore:   s.cveDataStore,
		vulnReqMgr:     s.vulnReqManager,
		vulnReqStore:   s.vulnReqDataStore,
		AuditLogger:    audit.New(notifierMock),
	}

	s.schema, err = graphql.ParseSchema(Schema(), s.resolver)
	s.NoError(err)

	if !features.VulnRiskManagement.Enabled() {
		s.T().Skip("Skip vuln requests resolver tests")
		s.T().SkipNow()
	}
}

func (s *VulnRequestResolverTestSuite) createImageDataStore(dacky *dackbox.DackBox) {
	imageDataStore := imageDS.New(dacky, concurrency.NewKeyFence(), s.bleveIndex, s.bleveIndex, true, nil, ranking.ImageRanker(), ranking.ComponentRanker())
	s.imageDataStore = imageDataStore
}

func (s *VulnRequestResolverTestSuite) createCVEDataStore(dacky *dackbox.DackBox) {
	cveStore := cveStore.New(dacky, concurrency.NewKeyFence())
	cveIndexer := cveIndex.New(s.bleveIndex)
	cveDataStore, err := cveDS.New(dacky,
		cveStore,
		cveIndexer,
		cveSearcher.New(cveStore, dacky, cveIndexer, nil, nil, nil, nil, imageCVEEdgeIndex.New(s.bleveIndex), nil, nil, nil, nil, nil),
	)
	s.Require().NoError(err)
	s.cveDataStore = cveDataStore
}

func (s *VulnRequestResolverTestSuite) createImageCVEDataStore(dacky *dackbox.DackBox) {
	imageCVEEdgeStore := imageCVEEdgeStore.New(dacky, concurrency.NewKeyFence())
	s.imageCVEDataStore = imageCVEEdgeDS.New(dacky,
		imageCVEEdgeStore,
		imageCVEEdgeSearcher.New(imageCVEEdgeStore,
			cveIndex.New(s.bleveIndex),
			imageCVEEdgeIndex.New(s.bleveIndex),
			componentCVEEdgeIndex.New(s.bleveIndex),
			componentIndex.New(s.bleveIndex),
			imageComponentEdgeIndex.New(s.bleveIndex),
			imageIndex.New(s.bleveIndex),
			deploymentIndex.New(s.bleveIndex, nil),
			clusterIndex.New(s.bleveIndex),
		),
	)
}

func (s *VulnRequestResolverTestSuite) createComponentCVEDataStore(dacky *dackbox.DackBox) {
	componentCVEEdgeStore, err := componentCVEEdgeStore.New(dacky)
	s.NoError(err)
	s.componentCVEDataStore, err = componentCVEEdgeDS.New(dacky,
		componentCVEEdgeStore,
		componentCVEEdgeIndex.New(s.bleveIndex),
		componentCVEEdgeSearcher.New(componentCVEEdgeStore, dacky,
			componentCVEEdgeIndex.New(s.bleveIndex),
			cveIndex.New(s.bleveIndex),
			componentIndex.New(s.bleveIndex),
			imageComponentEdgeIndex.New(s.bleveIndex),
			imageCVEEdgeIndex.New(s.bleveIndex),
			imageIndex.New(s.bleveIndex),
			nodeComponentEdgeIndex.New(s.bleveIndex),
			nodeIndex.New(s.bleveIndex),
			deploymentIndex.New(s.bleveIndex, nil),
			clusterIndex.New(s.bleveIndex),
		),
	)
	s.NoError(err)
}

func (s *VulnRequestResolverTestSuite) createVulnRequestDataStore(pendingReqCache, activeReqCache vulnReqCache.VulnReqCache) {
	ds, err := vulnReqDS.NewForTestOnly(s.T(), s.db, s.bleveIndex, pendingReqCache, activeReqCache)
	s.NoError(err)
	s.vulnReqDataStore = ds
}

func (s *VulnRequestResolverTestSuite) TestDeferVulnerability() {
	img := fixtures.GetImage()
	s.addImageToStore(img)

	cve := img.GetScan().GetComponents()[0].GetVulns()[0].GetCve()
	query := getQueryWithVulnReqSelector(
		"mutation createDeferral($request: DeferVulnRequest!)",
		"deferVulnerability(request: $request)",
	)

	cases := []struct {
		name             string
		expiresWhenFixed *bool
		expiresOn        *graphql.Time
	}{
		{
			name:      "request that expires on time",
			expiresOn: &graphql.Time{Time: time.Now().Add(1 * time.Hour)},
		},
		{
			name:             "request that expires when fixed",
			expiresWhenFixed: boolPtr(true),
			expiresOn:        nil,
		},
		{
			name:             "missing both expiry and expires on is treated as default (indefinite expiry)",
			expiresWhenFixed: nil,
			expiresOn:        nil,
		},
		{
			name:             "zero value for expiry is treat as default (indefinite expiry)",
			expiresWhenFixed: nil,
			expiresOn:        &graphql.Time{Time: time.Time{}},
		},
	}
	for _, c := range cases {
		s.T().Run(c.name, func(t *testing.T) {

			requestVars := map[string]interface{}{
				"cve":     cve,
				"comment": "test defer",
				"scope":   getImageScopeVar(img.GetName()),
			}

			if c.expiresOn != nil {
				requestVars["expiresOn"] = c.expiresOn.Time
			}
			if c.expiresWhenFixed != nil {
				requestVars["expiresWhenFixed"] = *c.expiresWhenFixed
			}

			response := s.schema.Exec(s.mockContext,
				query, "createDeferral",
				map[string]interface{}{"request": requestVars})

			s.Len(response.Errors, 0)
			var resp struct {
				DeferVulnerability vulnResponse `json:"deferVulnerability"`
			}
			s.NoError(json.Unmarshal(response.Data, &resp))

			s.validateReturnedRequest(resp.DeferVulnerability, cve, storage.VulnerabilityState_DEFERRED, storage.RequestStatus_PENDING, false, img)

			shouldExpireWhenFixed := c.expiresWhenFixed != nil && *c.expiresWhenFixed
			var expiresOn *types.Timestamp
			if c.expiresOn != nil {
				expiresOn, _ = types.TimestampProto(c.expiresOn.Time)
			}
			s.validateDeferralReq(resp.DeferVulnerability, shouldExpireWhenFixed, expiresOn)
			s.validateUpdatedDeferralReq(resp.DeferVulnerability, false, false, nil)
		})
	}
}

func (s *VulnRequestResolverTestSuite) TestMarkVulnerabilityFP() {
	img := fixtures.GetImage()
	s.addImageToStore(img)

	cve := img.GetScan().GetComponents()[0].GetVulns()[0].GetCve()

	query := getQueryWithVulnReqSelector(
		"mutation createFP($request: FalsePositiveVulnRequest!)",
		"markVulnerabilityFalsePositive(request: $request)",
	)

	response := s.schema.Exec(s.mockContext,
		query, "createFP", map[string]interface{}{
			"request": map[string]interface{}{
				"cve":     cve,
				"comment": "test fp",
				"scope":   getImageScopeVar(img.GetName()),
			},
		})

	s.Len(response.Errors, 0)
	var resp struct {
		MarkVulnerabilityFalsePositive vulnResponse `json:"markVulnerabilityFalsePositive"`
	}
	s.NoError(json.Unmarshal(response.Data, &resp))

	s.validateReturnedRequest(resp.MarkVulnerabilityFalsePositive, cve, storage.VulnerabilityState_FALSE_POSITIVE, storage.RequestStatus_PENDING, false, img)
}

func (s *VulnRequestResolverTestSuite) TestApproveRequest() {
	img := fixtures.GetImage()
	s.addImageToStore(img)

	cve := img.GetScan().GetComponents()[0].GetVulns()[0].GetCve()

	// Add a pending request first
	req := fixtures.GetImageScopeDeferralRequest(img.GetName().GetRegistry(), img.GetName().GetRemote(), img.GetName().GetTag(), cve)
	s.NoError(s.vulnReqDataStore.AddRequest(allAllowedCtx, req))

	query := getQueryWithVulnReqSelector(
		"mutation approveReq($requestID: ID!, $comment: String!)",
		"approveVulnerabilityRequest(requestID: $requestID, comment: $comment)",
	)

	response := s.schema.Exec(s.mockContext,
		query, "approveReq", map[string]interface{}{
			"requestID": req.GetId(),
			"comment":   "approve me",
		})

	s.Len(response.Errors, 0)
	var resp struct {
		ApproveVulnerabilityRequest vulnResponse `json:"approveVulnerabilityRequest"`
	}
	s.NoError(json.Unmarshal(response.Data, &resp))

	s.validateReturnedRequest(resp.ApproveVulnerabilityRequest, cve, storage.VulnerabilityState_DEFERRED, storage.RequestStatus_APPROVED, false, img)
	s.validateDeferralReq(resp.ApproveVulnerabilityRequest, false, req.GetDeferralReq().GetExpiry().GetExpiresOn())
	s.validateUpdatedDeferralReq(resp.ApproveVulnerabilityRequest, false, false, nil)

	// Validate the req in the data store just to be sure
	s.verifyRequestInStore(req.GetId(), storage.VulnerabilityState_DEFERRED, storage.RequestStatus_APPROVED, false)
}

func (s *VulnRequestResolverTestSuite) TestDenyRequest() {
	img := fixtures.GetImage()
	s.addImageToStore(img)

	cve := img.GetScan().GetComponents()[0].GetVulns()[0].GetCve()

	// Add a pending request first
	req := fixtures.GetImageScopeDeferralRequest(img.GetName().GetRegistry(), img.GetName().GetRemote(), img.GetName().GetTag(), cve)
	s.NoError(s.vulnReqDataStore.AddRequest(allAllowedCtx, req))

	query := getQueryWithVulnReqSelector(
		"mutation denyReq($requestID: ID!, $comment: String!)",
		"denyVulnerabilityRequest(requestID: $requestID, comment: $comment)",
	)

	response := s.schema.Exec(s.mockContext,
		query, "denyReq", map[string]interface{}{
			"requestID": req.GetId(),
			"comment":   "NO.",
		})

	s.Len(response.Errors, 0)
	var resp struct {
		DenyVulnerabilityRequest vulnResponse `json:"denyVulnerabilityRequest"`
	}
	s.NoError(json.Unmarshal(response.Data, &resp))

	s.validateReturnedRequest(resp.DenyVulnerabilityRequest, cve, storage.VulnerabilityState_DEFERRED, storage.RequestStatus_DENIED, true, img)
	s.validateDeferralReq(resp.DenyVulnerabilityRequest, false, req.GetDeferralReq().GetExpiry().GetExpiresOn())
	s.validateUpdatedDeferralReq(resp.DenyVulnerabilityRequest, false, false, nil)

	// Validate the req in the data store just to be sure
	s.verifyRequestInStore(req.GetId(), storage.VulnerabilityState_DEFERRED, storage.RequestStatus_DENIED, true)
}

func (s *VulnRequestResolverTestSuite) TestUndoRequest() {
	img := fixtures.GetImage()
	s.addImageToStore(img)

	cve := img.GetScan().GetComponents()[0].GetVulns()[0].GetCve()

	// Add an approved request first
	req := fixtures.GetApprovedDeferralRequest(img.GetName().GetRegistry(), img.GetName().GetRemote(), img.GetName().GetTag(), cve)
	s.NoError(s.vulnReqDataStore.AddRequest(allAllowedCtx, req))

	query := getQueryWithVulnReqSelector(
		"mutation reObserveReq($requestID: ID!)",
		"undoVulnerabilityRequest(requestID: $requestID)",
	)

	response := s.schema.Exec(s.mockContext,
		query, "reObserveReq", map[string]interface{}{
			"requestID": req.GetId(),
		})

	s.Len(response.Errors, 0)
	var resp struct {
		UndoVulnerabilityRequest vulnResponse `json:"undoVulnerabilityRequest"`
	}
	s.NoError(json.Unmarshal(response.Data, &resp))

	// It should have the same state but be marked as expired
	s.validateReturnedRequest(resp.UndoVulnerabilityRequest, cve, storage.VulnerabilityState_DEFERRED, storage.RequestStatus_APPROVED, true, img)
	s.validateDeferralReq(resp.UndoVulnerabilityRequest, false, req.GetDeferralReq().GetExpiry().GetExpiresOn())
	s.validateUpdatedDeferralReq(resp.UndoVulnerabilityRequest, false, false, nil)

	// Validate the req in the data store just to be sure
	s.verifyRequestInStore(req.GetId(), storage.VulnerabilityState_DEFERRED, storage.RequestStatus_APPROVED, true)
}

func (s *VulnRequestResolverTestSuite) TestRemoveRequest() {
	img := fixtures.GetImage()
	s.addImageToStore(img)

	cve := img.GetScan().GetComponents()[0].GetVulns()[0].GetCve()

	// Add a pending request first and set the request to current user
	req := fixtures.GetImageScopeDeferralRequest(img.GetName().GetRegistry(), img.GetName().GetRemote(), img.GetName().GetTag(), cve)
	req.Requestor.Id = fakeUserID
	s.NoError(s.vulnReqDataStore.AddRequest(allAllowedCtx, req))

	query := `mutation cancelReq($requestID: ID!) {
		success: deleteVulnerabilityRequest(requestID: $requestID)
	}`

	response := s.schema.Exec(s.mockContext,
		query, "cancelReq", map[string]interface{}{
			"requestID": req.GetId(),
		})

	s.Len(response.Errors, 0)
	var resp map[string]interface{}
	s.NoError(json.Unmarshal(response.Data, &resp))

	s.True(resp["success"].(bool))

	// Verify that the request is not in the store anymore
	_, ok, err := s.vulnReqDataStore.Get(allAllowedCtx, req.GetId())
	s.NoError(err)
	s.False(ok)
}

func (s *VulnRequestResolverTestSuite) TestUpdateRequest() {
	img := fixtures.GetImage()
	s.addImageToStore(img)

	cve := img.GetScan().GetComponents()[0].GetVulns()[0].GetCve()

	// Add an approved request with a timed expiry first
	req := fixtures.GetApprovedDeferralRequest(img.GetName().GetRegistry(), img.GetName().GetRemote(), img.GetName().GetTag(), cve)
	s.NoError(s.vulnReqDataStore.AddRequest(allAllowedCtx, req))

	query := getQueryWithVulnReqSelector(
		"mutation updateReq($requestID: ID!, $comment: String!, $expiry: VulnReqExpiry!)",
		"updateVulnerabilityRequest(requestID: $requestID, comment: $comment, expiry: $expiry)",
	)

	response := s.schema.Exec(s.mockContext,
		query, "updateReq", map[string]interface{}{
			"requestID": req.GetId(),
			"comment":   "update me",
			"expiry": map[string]interface{}{
				"expiresWhenFixed": true,
			},
		})

	s.Len(response.Errors, 0)
	var resp struct {
		UpdateVulnerabilityRequest vulnResponse `json:"updateVulnerabilityRequest"`
	}
	s.NoError(json.Unmarshal(response.Data, &resp))

	s.validateReturnedRequest(resp.UpdateVulnerabilityRequest, cve, storage.VulnerabilityState_DEFERRED, storage.RequestStatus_APPROVED_PENDING_UPDATE, false, img)
	// Validate original deferral req remains
	s.validateDeferralReq(resp.UpdateVulnerabilityRequest, false, req.GetDeferralReq().GetExpiry().GetExpiresOn())
	// validate there's an updated obj with expires when fixed
	s.validateUpdatedDeferralReq(resp.UpdateVulnerabilityRequest, true, true, nil)

	// Validate the req in the data store just to be sure
	reqInStore := s.verifyRequestInStore(req.GetId(), storage.VulnerabilityState_DEFERRED, storage.RequestStatus_APPROVED_PENDING_UPDATE, false)
	s.NotNil(reqInStore.GetUpdatedDeferralReq())
	s.True(reqInStore.GetUpdatedDeferralReq().GetExpiry().GetExpiresWhenFixed())
}

func (s *VulnRequestResolverTestSuite) TestApproveUpdatedRequest() {
	img := fixtures.GetImage()
	s.addImageToStore(img)

	cve := img.GetScan().GetComponents()[0].GetVulns()[0].GetCve()

	// Add a request pending update first
	req := fixtures.GetApprovedDeferralRequest(img.GetName().GetRegistry(), img.GetName().GetRemote(), img.GetName().GetTag(), cve)
	req.Status = storage.RequestStatus_APPROVED_PENDING_UPDATE
	req.UpdatedReq = &storage.VulnerabilityRequest_UpdatedDeferralReq{UpdatedDeferralReq: &storage.DeferralRequest{
		Expiry: &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresWhenFixed{ExpiresWhenFixed: true}},
	}}
	s.NoError(s.vulnReqDataStore.AddRequest(allAllowedCtx, req))

	query := getQueryWithVulnReqSelector(
		"mutation approveReq($requestID: ID!, $comment: String!)",
		"approveVulnerabilityRequest(requestID: $requestID, comment: $comment)",
	)

	response := s.schema.Exec(s.mockContext,
		query, "approveReq", map[string]interface{}{
			"requestID": req.GetId(),
			"comment":   "approve me",
		})

	s.Len(response.Errors, 0)
	var resp struct {
		ApproveVulnerabilityRequest vulnResponse `json:"approveVulnerabilityRequest"`
	}
	s.NoError(json.Unmarshal(response.Data, &resp))

	s.validateReturnedRequest(resp.ApproveVulnerabilityRequest, cve, storage.VulnerabilityState_DEFERRED, storage.RequestStatus_APPROVED, false, img)
	// Validate that it was updated to the newest expiry
	s.validateDeferralReq(resp.ApproveVulnerabilityRequest, true, nil)
	// Validate that the deferral obj was cleared out after update
	s.validateUpdatedDeferralReq(resp.ApproveVulnerabilityRequest, false, false, nil)

	// Validate the req in the data store just to be sure
	reqInStore := s.verifyRequestInStore(req.GetId(), storage.VulnerabilityState_DEFERRED, storage.RequestStatus_APPROVED, false)
	s.Nil(reqInStore.GetUpdatedDeferralReq())
}

func (s *VulnRequestResolverTestSuite) TestDenyUpdatedRequest() {
	img := fixtures.GetImage()
	s.addImageToStore(img)

	cve := img.GetScan().GetComponents()[0].GetVulns()[0].GetCve()

	// Add a request pending update first
	req := fixtures.GetApprovedDeferralRequest(img.GetName().GetRegistry(), img.GetName().GetRemote(), img.GetName().GetTag(), cve)
	req.Status = storage.RequestStatus_APPROVED_PENDING_UPDATE
	req.UpdatedReq = &storage.VulnerabilityRequest_UpdatedDeferralReq{UpdatedDeferralReq: &storage.DeferralRequest{
		Expiry: &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresWhenFixed{ExpiresWhenFixed: true}},
	}}
	s.NoError(s.vulnReqDataStore.AddRequest(allAllowedCtx, req))

	query := getQueryWithVulnReqSelector(
		"mutation denyReq($requestID: ID!, $comment: String!)",
		"denyVulnerabilityRequest(requestID: $requestID, comment: $comment)",
	)

	response := s.schema.Exec(s.mockContext,
		query, "denyReq", map[string]interface{}{
			"requestID": req.GetId(),
			"comment":   "NO.",
		})

	s.Len(response.Errors, 0)

	var resp struct {
		DenyVulnerabilityRequest vulnResponse `json:"denyVulnerabilityRequest"`
	}
	s.NoError(json.Unmarshal(response.Data, &resp))

	// It should be in the approved state even after denial because it went back to the previous state
	s.validateReturnedRequest(resp.DenyVulnerabilityRequest, cve, storage.VulnerabilityState_DEFERRED, storage.RequestStatus_APPROVED, false, img)
	// Validate that it was NOT updated to the newest expiry
	s.validateDeferralReq(resp.DenyVulnerabilityRequest, false, req.GetDeferralReq().GetExpiry().GetExpiresOn())
	// Validate that the deferral obj was cleared out after update
	s.validateUpdatedDeferralReq(resp.DenyVulnerabilityRequest, false, false, nil)

	// Validate the req in the data store just to be sure
	reqInStore := s.verifyRequestInStore(req.GetId(), storage.VulnerabilityState_DEFERRED, storage.RequestStatus_APPROVED, false)
	s.Nil(reqInStore.GetUpdatedDeferralReq())
}

func (s *VulnRequestResolverTestSuite) TestGetVulnRequestsCount() {
	img := fixtures.GetImage()
	s.addImageToStore(img)

	pendingDeferrals, pendingFPs, approvedDeferrals, approvedFPs := s.getCategorizedRequests(img)

	query := `query vulnCount($query: String!) {
		count: vulnerabilityRequestsCount(query: $query)
	}`

	cases := []struct {
		countQuery    string
		expectedCount int
	}{
		{
			countQuery:    "Requested Vulnerability State:DEFERRED",
			expectedCount: len(pendingDeferrals) + len(approvedDeferrals),
		},
		{
			countQuery:    "Requested Vulnerability State:FALSE_POSITIVE",
			expectedCount: len(pendingFPs) + len(approvedFPs),
		},
		{
			countQuery:    "Request Status:PENDING",
			expectedCount: len(pendingDeferrals) + len(pendingFPs),
		},
		{
			countQuery:    "Request Status:APPROVED",
			expectedCount: len(approvedDeferrals) + len(approvedFPs),
		},
		{
			countQuery:    "Request Status:PENDING+Requested Vulnerability State:DEFERRED",
			expectedCount: len(pendingDeferrals),
		},
		{
			countQuery:    "Request Status:APPROVED+Requested Vulnerability State:FALSE_POSITIVE",
			expectedCount: len(approvedFPs),
		},
		{
			countQuery:    "Request Status:APPROVED_PENDING_UPDATE",
			expectedCount: 0,
		},
		{
			countQuery:    "Request Status:*",
			expectedCount: len(approvedDeferrals) + len(approvedFPs) + len(pendingFPs) + len(pendingDeferrals),
		},
	}
	for _, c := range cases {
		s.T().Run(c.countQuery, func(t *testing.T) {
			response := s.schema.Exec(s.mockContext,
				query, "vulnCount", map[string]interface{}{
					"query": c.countQuery,
				})

			s.Len(response.Errors, 0)
			var resp map[string]interface{}
			s.NoError(json.Unmarshal(response.Data, &resp))
			s.Equal(float64(c.expectedCount), resp["count"]) // for some reason the response deserializes as a float even though it's an int
		})
	}
}

func (s *VulnRequestResolverTestSuite) TestGetVulnerabilityRequest() {
	img := fixtures.GetImage()
	s.addImageToStore(img)

	cve := img.GetScan().GetComponents()[0].GetVulns()[0].GetCve()

	// Add an approved request with all the details to the store
	req := fixtures.GetApprovedDeferralRequest(img.GetName().GetRegistry(), img.GetName().GetRemote(), img.GetName().GetTag(), cve)
	s.NoError(s.vulnReqDataStore.AddRequest(allAllowedCtx, req))

	query := getQueryWithVulnReqSelector(
		"query getReq($id: ID!)",
		"vulnerabilityRequest(id: $id)",
	)

	response := s.schema.Exec(s.mockContext,
		query, "getReq", map[string]interface{}{
			"id": req.GetId(),
		})

	s.Len(response.Errors, 0)
	var resp struct {
		VulnerabilityRequest vulnResponse `json:"vulnerabilityRequest"`
	}
	s.NoError(json.Unmarshal(response.Data, &resp))

	s.validateReturnedRequest(resp.VulnerabilityRequest, cve, storage.VulnerabilityState_DEFERRED, storage.RequestStatus_APPROVED, false, img)
	s.validateDeferralReq(resp.VulnerabilityRequest, false, req.GetDeferralReq().GetExpiry().GetExpiresOn())
	s.validateUpdatedDeferralReq(resp.VulnerabilityRequest, false, false, nil)
	s.validateComments(resp.VulnerabilityRequest, req.GetComments())

	s.Equal(resp.VulnerabilityRequest.Requestor.Name, req.GetRequestor().GetName())
	for i, a := range resp.VulnerabilityRequest.Approvers {
		s.Equal(a.Name, req.GetApprovers()[i].GetName())
	}
}

func (s *VulnRequestResolverTestSuite) TestGetVulnerabilityRequests() {
	img := fixtures.GetImage()
	s.addImageToStore(img)

	pendingDeferrals, _, _, _ := s.getCategorizedRequests(img)
	deferredReqIDs := reflect.ValueOf(pendingDeferrals).MapKeys()

	query := getQueryWithVulnReqSelector(
		"query getReqs($query: String, $requestIDSelector: String)",
		"vulnerabilityRequests(query: $query, requestIDSelector: $requestIDSelector)",
	)

	cases := []struct {
		name                string
		query               string
		requestIDSelector   string
		expectedReturnCount int
		expectedReqs        map[string]*storage.VulnerabilityRequest
	}{
		{
			name:                "get all pending deferrals",
			query:               "Request Status:PENDING+Requested Vulnerability State:DEFERRED",
			expectedReturnCount: len(pendingDeferrals),
			expectedReqs:        pendingDeferrals,
		},
		{
			name:                "get a specific pending deferral",
			query:               "Request Status:PENDING+Requested Vulnerability State:DEFERRED",
			requestIDSelector:   deferredReqIDs[0].String(),
			expectedReturnCount: 1,
			expectedReqs:        pendingDeferrals,
		},
		{
			name:                "get a multiple specific pending deferral",
			query:               "Request Status:PENDING+Requested Vulnerability State:DEFERRED",
			requestIDSelector:   fmt.Sprintf("%s,%s", deferredReqIDs[0].String(), deferredReqIDs[1].String()),
			expectedReturnCount: 2,
			expectedReqs:        pendingDeferrals,
		},
		{
			name:                "get a specific request given a search on all requests",
			query:               "Request Status:*",
			requestIDSelector:   deferredReqIDs[0].String(),
			expectedReturnCount: 1,
			expectedReqs:        pendingDeferrals,
		},
		{
			name:                "wrong request id for the result set",
			query:               "Request Status:PENDING+Requested Vulnerability State:FALSE_POSITIVE",
			requestIDSelector:   deferredReqIDs[0].String(),
			expectedReturnCount: 0,
			expectedReqs:        pendingDeferrals,
		},
	}
	for _, c := range cases {
		s.T().Run(c.name, func(t *testing.T) {
			response := s.schema.Exec(s.mockContext,
				query, "getReqs", map[string]interface{}{
					"query":             c.query,
					"requestIDSelector": c.requestIDSelector,
				})

			s.Len(response.Errors, 0)
			var resp struct {
				VulnerabilityRequests []vulnResponse `json:"vulnerabilityRequests"`
			}
			s.NoError(json.Unmarshal(response.Data, &resp))

			s.Len(resp.VulnerabilityRequests, c.expectedReturnCount)
			for _, r := range resp.VulnerabilityRequests {
				req, ok := c.expectedReqs[r.ID]
				if !ok {
					s.Fail("Got unexpected request given query")
				}
				s.validateReturnedRequest(r, req.GetCves().GetIds()[0], storage.VulnerabilityState_DEFERRED, req.GetStatus(), false, img)
				s.validateDeferralReq(r, false, req.GetDeferralReq().GetExpiry().GetExpiresOn())
			}
		})
	}

}

func getQueryWithVulnReqSelector(name, function string) string {
	return fmt.Sprintf(
		`%s {
			%s { 
				%s
			}}`,
		name,
		function,
		vulnerabilityRequestSelector,
	)
}
func getImageScopeVar(img *storage.ImageName) map[string]interface{} {
	return map[string]interface{}{
		"imageScope": map[string]interface{}{
			"registry": img.GetRegistry(),
			"remote":   img.GetRemote(),
			"tag":      img.GetTag(),
		},
	}
}

func (s *VulnRequestResolverTestSuite) getCategorizedRequests(img *storage.Image) (map[string]*storage.VulnerabilityRequest,
	map[string]*storage.VulnerabilityRequest, map[string]*storage.VulnerabilityRequest, map[string]*storage.VulnerabilityRequest) {
	pendingDeferrals, pendingFPs, approvedDeferrals, approvedFPs :=
		make(map[string]*storage.VulnerabilityRequest), make(map[string]*storage.VulnerabilityRequest), make(map[string]*storage.VulnerabilityRequest), make(map[string]*storage.VulnerabilityRequest)

	// grab some vulns and stick them into an appropriate category. Rest won't get a request
	var i = 0
	for _, comp := range img.GetScan().GetComponents() {
		for _, vuln := range comp.GetVulns() {
			var req *storage.VulnerabilityRequest
			cve := vuln.GetCve()
			if i < 17 {
				req = fixtures.GetImageScopeDeferralRequest(img.GetName().GetRegistry(), img.GetName().GetRemote(), img.GetName().GetTag(), cve)
				pendingDeferrals[req.GetId()] = req
			} else if i < 31 {
				req = fixtures.GetImageScopeFPRequest(img.GetName().GetRegistry(), img.GetName().GetRemote(), img.GetName().GetTag(), cve)
				pendingFPs[req.GetId()] = req
			} else if i < 36 {
				req = fixtures.GetApprovedDeferralRequest(img.GetName().GetRegistry(), img.GetName().GetRemote(), img.GetName().GetTag(), cve)
				approvedDeferrals[req.GetId()] = req
			} else if i < 42 {
				req = fixtures.GetImageScopeFPRequest(img.GetName().GetRegistry(), img.GetName().GetRemote(), img.GetName().GetTag(), cve)
				req.Status = storage.RequestStatus_APPROVED
				approvedFPs[req.GetId()] = req
			}
			if req != nil {
				s.NoError(s.vulnReqDataStore.AddRequest(allAllowedCtx, req))
			}
			i++
		}
	}
	return pendingDeferrals, pendingFPs, approvedDeferrals, approvedFPs
}

func (s *VulnRequestResolverTestSuite) verifyRequestInStore(id string, state storage.VulnerabilityState, status storage.RequestStatus, expired bool) *storage.VulnerabilityRequest {
	r, ok, err := s.vulnReqDataStore.Get(allAllowedCtx, id)
	s.NoError(err)
	s.True(ok)
	s.Equal(state, r.GetTargetState())
	s.Equal(status, r.GetStatus())
	s.Equal(expired, r.GetExpired())
	return r
}

func (s *VulnRequestResolverTestSuite) validateReturnedRequest(ret vulnResponse, cve string, state storage.VulnerabilityState, status storage.RequestStatus, expired bool, img *storage.Image) {
	s.NotNil(ret)
	s.Equal(ret.Cves.Ids, []string{cve})
	s.Equal(state.String(), ret.TargetState)
	s.Equal(status.String(), ret.Status)
	s.Equal(expired, ret.Expired)
	s.validateImageScope(ret, img.GetName(), false)
}

func (s *VulnRequestResolverTestSuite) validateDeferralReq(ret vulnResponse, expiresWhenFixed bool, expiresOn *types.Timestamp) {
	s.NotNil(ret.DeferralReq)
	s.validateExpiry(ret.DeferralReq, expiresWhenFixed, expiresOn)
}

func (s *VulnRequestResolverTestSuite) validateUpdatedDeferralReq(ret vulnResponse, updated, expiresWhenFixed bool, expiresOn *types.Timestamp) {
	if !updated {
		s.Nil(ret.UpdatedDeferralReq)
		return
	}
	s.NotNil(ret.UpdatedDeferralReq)
	s.validateExpiry(ret.UpdatedDeferralReq, expiresWhenFixed, expiresOn)
}

func (s *VulnRequestResolverTestSuite) validateExpiry(expiryReq *expiryResponse, expiresWhenFixed bool, expiresOn *types.Timestamp) {
	s.NotNil(expiryReq)
	s.Equal(expiresWhenFixed, expiryReq.ExpiresWhenFixed)
	if expiresOn != nil {
		s.Equal(expiresOn.String(), expiryReq.ExpiresOn.Format(time.RFC3339Nano))
	} else {
		s.Nil(expiryReq.ExpiresOn)
	}
}

func (s *VulnRequestResolverTestSuite) validateImageScope(ret vulnResponse, img *storage.ImageName, allTags bool) {
	s.Equal(img.GetRegistry(), ret.Scope.ImageScope.Registry)
	s.Equal(img.GetRemote(), ret.Scope.ImageScope.Remote)
	if allTags {
		s.Equal(".*", ret.Scope.ImageScope.Tag)
	} else {
		s.Equal(img.GetTag(), ret.Scope.ImageScope.Tag)
	}
}

func (s *VulnRequestResolverTestSuite) validateComments(ret vulnResponse, expectedComments []*storage.RequestComment) {
	for i, c := range ret.Comments {
		s.Equal(c.Message, expectedComments[i].GetMessage())
		s.Equal(c.User.Name, expectedComments[i].GetUser().GetName())
	}
}

func (s *VulnRequestResolverTestSuite) addImageToStore(img *storage.Image) {
	s.NoError(s.imageDataStore.UpsertImage(s.mockContext, img))
	s.forceIndexing()
}

func (s *VulnRequestResolverTestSuite) forceIndexing() {
	indexingDone := concurrency.NewSignal()
	s.indexQ.PushSignal(&indexingDone)
	indexingDone.Wait()
}

func boolPtr(s bool) *bool {
	return &s
}
