package imageintegrations

import (
	"context"
	"fmt"
	"strings"

	"github.com/pkg/errors"
	clusterDatastore "github.com/stackrox/rox/central/cluster/datastore"
	"github.com/stackrox/rox/central/enrichment"
	"github.com/stackrox/rox/central/imageintegration/datastore"
	countMetrics "github.com/stackrox/rox/central/metrics"
	"github.com/stackrox/rox/central/reprocessor"
	"github.com/stackrox/rox/central/sensor/service/common"
	"github.com/stackrox/rox/central/sensor/service/pipeline"
	"github.com/stackrox/rox/central/sensor/service/pipeline/reconciliation"
	v1 "github.com/stackrox/rox/generated/api/v1"
	"github.com/stackrox/rox/generated/internalapi/central"
	"github.com/stackrox/rox/generated/storage"
	"github.com/stackrox/rox/pkg/env"
	"github.com/stackrox/rox/pkg/logging"
	"github.com/stackrox/rox/pkg/metrics"
	"github.com/stackrox/rox/pkg/tlscheck"
	"github.com/stackrox/rox/pkg/urlfmt"
)

var (
	log = logging.LoggerForModule()

	autogeneratedRegistriesDisabled = env.AutogeneratedRegistriesDisabled.BooleanSetting()
)

// GetPipeline returns an instantiation of this particular pipeline
func GetPipeline() pipeline.Fragment {
	return NewPipeline(enrichment.ManagerSingleton(),
		datastore.Singleton(),
		clusterDatastore.Singleton(),
		reprocessor.Singleton())
}

// NewPipeline returns a new instance of Pipeline.
func NewPipeline(integrationManager enrichment.Manager,
	datastore datastore.DataStore,
	clusterDatastore clusterDatastore.DataStore,
	enrichAndDetectLoop reprocessor.Loop) pipeline.Fragment {
	return &pipelineImpl{
		integrationManager:  integrationManager,
		datastore:           datastore,
		clusterDatastore:    clusterDatastore,
		enrichAndDetectLoop: enrichAndDetectLoop,
	}
}

type pipelineImpl struct {
	integrationManager enrichment.Manager

	datastore           datastore.DataStore
	clusterDatastore    clusterDatastore.DataStore
	enrichAndDetectLoop reprocessor.Loop
}

func (s *pipelineImpl) Reconcile(_ context.Context, _ string, _ *reconciliation.StoreMap) error {
	// Nothing to reconcile for image integrations
	return nil
}

func (s *pipelineImpl) Match(msg *central.MsgFromSensor) bool {
	return msg.GetEvent().GetImageIntegration() != nil
}

func matchesAuth(ni, ei *storage.ImageIntegration) bool {
	return ni.GetDocker().GetUsername() == ei.GetDocker().GetUsername() &&
		ni.GetDocker().GetPassword() == ei.GetDocker().GetPassword()
}

// getMatchingImageIntegration returns the image integration that exists and should be updated
// the second return value
func (s *pipelineImpl) getMatchingImageIntegration(auto *storage.ImageIntegration, existingIntegrations []*storage.ImageIntegration) (*storage.ImageIntegration, bool) {
	for _, existing := range existingIntegrations {
		if auto.GetName() != existing.GetName() {
			continue
		}

		// If there exists a registry with an auto-generated name, and the user has somehow managed
		// (most likely through the API) to mark it as non-autogenerated, we do not want to overwrite
		// it.
		if !existing.GetAutogenerated() {
			return nil, false
		}

		// At this point, we just want to see if we already have an exact match
		// if so then we don't want to reprocess everything for no change.
		// The cluster ID can only be different if a cluster with this name was deleted
		// and then another with the same name was created. In this case, we do want to update
		// the integration, if only for the sake of updating the cluster ID.
		if matchesAuth(auto, existing) && auto.GetType() == existing.GetType() && auto.GetClusterId() == existing.GetClusterId() {
			return nil, false
		}
		return existing, true
	}

	return nil, true
}

func parseEndpointForURL(endpoint string) string {
	url := urlfmt.FormatURL(endpoint, urlfmt.HTTPS, urlfmt.NoTrailingSlash)

	server := urlfmt.GetServerFromURL(url)
	if strings.HasSuffix(server, "docker.io") || strings.HasSuffix(server, "docker.io:443") {
		return "https://registry-1.docker.io"
	}

	scheme := urlfmt.GetSchemeFromURL(url)
	defaultScheme := urlfmt.HTTPS
	if scheme == "http" {
		defaultScheme = urlfmt.InsecureHTTP
	}
	return urlfmt.FormatURL(server, defaultScheme, urlfmt.NoTrailingSlash)
}

// Run runs the pipeline template on the input and returns the output.
// Action is currently always update.
func (s *pipelineImpl) Run(ctx context.Context, clusterID string, msg *central.MsgFromSensor, _ common.MessageInjector) error {
	// Ignore autogenerated registries if they are disabled
	if autogeneratedRegistriesDisabled {
		return nil
	}

	defer countMetrics.IncrementResourceProcessedCounter(pipeline.ActionToOperation(msg.GetEvent().GetAction()), metrics.ImageIntegration)

	clusterName, exists, err := s.clusterDatastore.GetClusterName(ctx, clusterID)
	if err != nil {
		return errors.Wrapf(err, "error getting cluster name for cluster ID: %s", clusterID)
	}
	if !exists {
		return fmt.Errorf("cluster with id %q does not exist", clusterID)
	}

	imageIntegration := msg.GetEvent().GetImageIntegration()
	// Using GetDocker() because the config is within a `oneof` field.
	dockerIntegration := imageIntegration.GetDocker()
	if dockerIntegration == nil {
		return nil
	}

	validTLS, err := tlscheck.CheckTLS(ctx, dockerIntegration.GetEndpoint())
	if err != nil {
		return errors.Wrapf(err, "reaching out for TLS check to %s", imageIntegration.GetDocker().GetEndpoint())
	}
	dockerIntegration.Insecure = !validTLS
	dockerIntegration.Endpoint = parseEndpointForURL(dockerIntegration.GetEndpoint())

	imageIntegration.Name = fmt.Sprintf("Autogenerated %s for cluster %s", dockerIntegration.GetEndpoint(), clusterName)
	imageIntegration.ClusterId = clusterID

	existingIntegrations, err := s.datastore.GetImageIntegrations(ctx, &v1.GetImageIntegrationsRequest{})
	if err != nil {
		return errors.Wrap(err, "fetching all image integrations")
	}
	integrationToUpdate, shouldInsert := s.getMatchingImageIntegration(imageIntegration, existingIntegrations)
	if !shouldInsert {
		return nil
	}
	if integrationToUpdate == nil {
		if _, err := s.datastore.AddImageIntegration(ctx, imageIntegration); err != nil {
			return errors.Wrap(err, "adding integration")
		}
		if err := s.integrationManager.Upsert(imageIntegration); err != nil {
			return errors.Wrap(err, "notifying of image integration update")
		}
		// Only when adding the integration the first time do we need to run processing
		// Central receives many updates from OpenShift about the image integrations due to service accounts
		// So we can assume the other credentials were valid up to this point.
		// Also, they will eventually be picked up within an hour.
		s.enrichAndDetectLoop.ShortCircuit()
		return nil
	}

	imageIntegration.Id = integrationToUpdate.GetId()
	imageIntegration.Name = integrationToUpdate.GetName()
	if err := s.integrationManager.Upsert(imageIntegration); err != nil {
		return errors.Wrap(err, "notifying of image integration update")
	}
	if err := s.datastore.UpdateImageIntegration(ctx, imageIntegration); err != nil {
		return errors.Wrap(err, "updating integration")
	}
	return nil
}

func (s *pipelineImpl) OnFinish(_ string) {}
